;*******************************************************************************
; File: cobra.pro
;
; Description:
; ------------
; Container for procedures analyzing NPOI raw data HDS files.
;
; Block directory:
; ----------------
; Block 1: addbincounts,fringeframe,displayframe,rotateframe,
;          fringevis,fringevissq,fringenphot,fringemphot
; Block 2: natoffset,natsurface,onsource
; Block 3: visrotate,visdft,whitedelay,lambdawhite,
;          phaseunwrap,phasescan,phaseshift,
;	   timejitter,fdlstamp,fdldelay,
;	   groupdelay,trackdelay,fringedelay,
;	   groupdelays,trackdelays,fringedelays,fringephases
;	   coherentvis,coherentvissq
; Block 4: cohstrictor,constrictor3,constrictor,solve
; Block 5: plot_fdldelay,plot_fdlpower,plot_vispower,
;	   plot_fringepower,plot_fringeimage,
;	   plot_powerhist,plot_powerpeaks,plot_spectrumpeaks,
;	   plot_groupdelay,plot_fringephase,plot_dispsol,plot_coherence,
;	   plot_ratehist,plot_nathist,plot_benhist
; Block 6: packettype,readpacket,packetdir,rawlist
; Block 7: disptest,altair1/2/3ab/4/5
;
;************************************************************************Block 1
function addbincounts,c,n
;
; Fast addition specifically for bincounts with all three dimensions.
;
if n eq 1 then return,c
;
r=size(c)
if r[0] eq 2 then begin
	nb=n_elements(c[*,0])
	nf=n_elements(c[0,*])
	nr=nf/n
	r=intarr(nb,nr)
;
	for i=0L,nr-1 do r[*,i]=total(c[*,i*n:(i+1)*n-1],2)
endif else begin
	nb=n_elements(c[*,0,0])
	nc=n_elements(c[0,*,0])
	nf=n_elements(c[0,0,*])
	nr=nf/n
	r=intarr(nb,nc,nr)
;
	for i=0L,nr-1 do r[*,*,i]=total(c[*,*,i*n:(i+1)*n-1],3)
endelse
;
return,r
;
end
;-------------------------------------------------------------------------------
function fringeframe,a,g,l,v,n,poisson=poisson,glas=glas,phase=phase
;
; Make standard NPOI 8 or 64-bin frames.
;
; Compute fringe frames for airpath a[mu] and group delay g[mu]. Optionally
; include visibility amplitude and photonrate (arrays). Wavelength array l[mu].
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
n1=n_elements(a)
n2=n_elements(g)
n3=n_elements(l)
n4=n_elements(v)
n5=n_elements(n)
;
if not keyword_set(poisson) then poisson=0
if n_elements(glas) eq 0 then glas=0.0
if n_elements(phase) eq 0 then phase=0.0
;
if n_elements(GenConfig) ne 0 then num_bin=GenConfig.NumBin else num_bin=8
;
cb=bytarr(num_bin,n3,n1)
cf=fltarr(num_bin,n3,n1)
;
if n4 eq 0 then v=fltarr(n1)+0.6
if n5 eq 0 then n=fltarr(n1)+40
if n4 eq 1 then v=fltarr(n1)+v[0]
if n5 eq 1 then n=fltarr(n1)+n[0]
;
nbin=3
num_phs=num_bin*nbin
m=((findgen(num_phs)-nbin/2)*2*!pi/num_phs)#(fltarr(n3)+1)
twopi=fltarr(num_phs)+2*!pi
n_1=edlen(l)-1
ns_1=silica(l)-1
for i=0l,n1-1 do begin
	d=twopi#((n_1*a[i]+g[i]+ns_1*glas)/l)
;	c(*,*,i)=byte(((1.0+v(i)*cos(m+d))*n(i)))
;	y=(1.0+v(i)*cos(m+d))*((n(i)/num_bin)/nbin)
;	for k=0,num_bin-1 do cf(k,*,i)=total(y(k*nbin:(k+1)*nbin-1,*),1)
	y=cos(m+d-phase)
	for k=0,num_bin-1 do cf[k,*,i]=total(y[k*nbin:(k+1)*nbin-1,*],1)/nbin
	cf[*,*,i]=(cf[*,*,i]*v[i]+1)*(n[i]/num_bin)
;	cf(*,*,i)=(add(1.0+v(i)*cos(m+d),nbin,0)/nbin)*n(i)
;	cf(*,*,i)=(1.0+v(i)*cos(m+d+phase))*n(i)
endfor
;
k=n_elements(cf)
if poisson then begin
	s=linknload(!external_lib,'randpoisson',cb,cf,k)
endif
;
if poisson then return,cb else return,cf
;
end
;-------------------------------------------------------------------------------
function fringeframe3,n,k12,k23,k31
;
common LocalFringeFrame,seed
;
rad=180/!pi
i1=1.0
i2=1.0
i3=1.0
i=sqrt(complex(-1.0,0.0))
v12=0.486
p12=-129.1
v23=0.968
p23=0.0
v31=0.531
p31=115.6+40
v12=0.5
p12=0.0
v23=0.5
p23=0.0
v31=0.5
p31=0.0
numbin=64l
p=findgen(numbin)/numbin*2*!pi
print,'p12+p23+p31=',p12+p23+p31
print,'v12*v23*v31=',v12*v23*v31
k1=0
k2=2
k3=6
k1=1
k2=4
k3=5
k12=k2-k1
k23=k3-k2
k31=k1-k3
b1=2*sqrt(i1*i2)*v12*cos(p*k12-p12/rad)
b2=2*sqrt(i2*i3)*v23*cos(p*k23-p23/rad)
b3=2*sqrt(i3*i1)*v31*cos(p*k31-p31/rad)
b=i1+i2+i3+b1+b2+b3
;
b=b*0.5
;plot,b,psym=0
counts=bytarr(numbin*n)
p12=randomu(seed12,n)*2*!pi
p23=randomu(seed23,n)*2*!pi
p31=randomu(seed31,n)*2*!pi
p31=-p12-p23
for i=0L,n_elements(counts)-1 do begin
	b1=2*sqrt(i1*i2)*v12*cos(p[i mod numbin]*k12-p12[i/numbin])
	b2=2*sqrt(i2*i3)*v23*cos(p[i mod numbin]*k23-p23[i/numbin])
	b3=2*sqrt(i3*i1)*v31*cos(p[i mod numbin]*k31-p31[i/numbin])
	b=i1+i2+i3+b1+b2+b3
b=b*0.5
;	counts(i)=randomu(seed,poisson=b(i mod numbin))
	counts[i]=randomu(seed,poisson=b)
endfor
;
return,reform(counts,numbin,n)
;
end
;-------------------------------------------------------------------------------
pro closurebias3,numav=numav
;
forward_function fringevissq,fringenphot
;
if n_elements(numav) eq 0 then numav=500
;
c=fringeframe3(45000,k12,k23,k31)
;
n0=64
norm_factor12=4.0/sinc(float(k12)/n0)^2 
norm_factor23=4.0/sinc(float(k23)/n0)^2 
norm_factor31=4.0/sinc(float(k31)/n0)^2 
norm_factor=sqrt(norm_factor12)*sqrt(norm_factor23)*sqrt(norm_factor31)*(3.0/2.0)^3
;
n=fringenphot(c)
v1=fringevis[c,k12]
v2=fringevis[c,k23]
v3=fringevis[c,k31]
print,'Avg rate per frame: ',avg(n), $
        ', v1^2= ',avg(fringevissq(c,k12,numav,/normal)), $
        ', v2^2= ',avg(fringevissq(c,k23,numav,/normal)), $
        ', v3^2= ',avg(fringevissq(c,k31,numav,/normal))
t=v1*v2*v3
tb=t-abs(v1)^2-abs(v2)^2-abs(v3)^2+2.0*n
print,'3 on 1   biased: ',median(cphase(add(t,numav)))*57.3,(median(abs(add(t,numav)/numav))/avg(n)^3)*norm_factor, $
    ', 3 on 1 unbiased: ',median(cphase(add(tb,numav)))*57.3,(median(abs(add(tb,numav)/numav))/avg(n)^3)*norm_factor
;
cb=fringeframe3(45000)
nb=fringenphot(cb)
vb1=fringevis[cb,k12]
vb2=fringevis[cb,k23]
vb3=fringevis[cb,k31]
tb=v1*v2*vb3
tbb=tb-abs(vb3)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,numav)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,numav)))*57.3
tb=v1*vb2*v3
tbb=tb-abs(vb2)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,numav)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,numav)))*57.3
tb=vb1*v2*v3
tbb=tb-abs(vb1)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,numav)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,numav)))*57.3
;
tb=vb1*vb2*v3
tbb=tb-abs(v3)^2+nb
print,'3 on 2   biased: ',median(cphase(add(tb,numav)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,numav)))*57.3
tb=vb1*v2*vb3
tbb=tb-abs(v2)^2+nb
print,'3 on 2   biased: ',median(cphase(add(tb,numav)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,numav)))*57.3
tb=v1*vb2*vb3
tbb=tb-abs(v1)^2+nb
print,'3 on 2   biased: ',median(cphase(add(tb,numav)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,numav)))*57.3
;
cc=fringeframe3(45000)
nc=fringenphot(cc)
vc1=fringevis[cc,k12]
vc2=fringevis[cc,k23]
vc3=fringevis[cc,k31]
tc=v1*vb2*vc3
print,'3 on 3         : ',median(cphase(add(tc,numav)))*57.3,(median(abs(add(tc,numav)/numav))/avg(n)^3)*norm_factor
;
end
;-------------------------------------------------------------------------------
function fringeframe4,n,k12,k13,k14,k23,k42,k43
;
common LocalFringeFrame,seed
;
rad=180/!pi
i1=1.0
i2=1.0
i3=1.0
i4=1.0
i=sqrt(complex(-1.0,0.0))
v12=0.953
v13=0.324
v14=0.525
v23=0.186
v42=0.376
v43=0.719
p12=  -0.6
p13=-116.7
p14=  87.5
p23=-137.9
p42= -77.1
p43= 165.0
numbin=64l
print,'p12+p23-p13=',(p12+p23-p13) mod 360
print,'p13-p43-p14=',(p13-p43-p14) mod 360
print,'p14+p42-p12=',(p14+p42-p12) mod 360
print,'p23-p43+p42=',(p23-p43+p42) mod 360
p=findgen(numbin)/numbin*2*!pi
k1=-1
k2=+4
k3=-3
k4=+3
k12=k2-k1
k13=k3-k1
k14=k4-k1
k23=k3-k2
k42=k2-k4
k43=k3-k4
;help,k12,k13,k14,k23,k42,k43
b1=2*sqrt(i1*i2)*v12*cos(p*k12-p12/rad)
b2=2*sqrt(i1*i3)*v13*cos(p*k13-p13/rad)
b3=2*sqrt(i1*i4)*v14*cos(p*k14-p14/rad)
b4=2*sqrt(i2*i3)*v23*cos(p*k23-p23/rad)
b5=2*sqrt(i4*i2)*v42*cos(p*k42-p42/rad)
b6=2*sqrt(i4*i3)*v43*cos(p*k43-p43/rad)
b=i1+i2+i3+i4+b1+b2+b3+b4+b5+b6
;
b=b*0.5
;plot,b,psym=0
counts=bytarr(numbin*n)
for i=0L,n_elements(counts)-1 do counts[i]=randomu(seed,poisson=b[i mod numbin])
;
return,reform(counts,numbin,n)
;
end
;-------------------------------------------------------------------------------
pro closurebias4
;
forward_function fringevissq,fringenphot
;
c=fringeframe4(45000,k12,k13,k14,k23,k42,k43)
n=fringenphot(c)
v1=fringevis[c,k12]
v2=fringevis[c,k13]
v3=fringevis[c,k14]
v4=fringevis[c,k23]
v5=fringevis[c,k42]
v6=fringevis[c,k43]
print,'Avg rate per frame: ',avg(n)
print, $
          'v1^2= ',avg(fringevissq(c,k12,500,/normal)), $
        ', v2^2= ',avg(fringevissq(c,k13,500,/normal)), $
        ', v3^2= ',avg(fringevissq(c,k14,500,/normal)), $
        '  v4^2= ',avg(fringevissq(c,k23,500,/normal)), $
        ', v5^2= ',avg(fringevissq(c,k42,500,/normal)), $
        ', v6^2= ',avg(fringevissq(c,k43,500,/normal))
t=v1*v4*conj(v2)
tb=t-abs(v1)^2-abs(v4)^2-abs(v2)^2+2*n
print,'3 on 1   biased: ',median(cphase(add(t,500)))*57.3, $
    ', 3 on 1 unbiased: ',median(cphase(add(tb,500)))*57.3
t=v2*conj(v6)*conj(v3)
tb=t-abs(v2)^2-abs(v6)^2-abs(v3)^2+2*n
print,'3 on 1   biased: ',median(cphase(add(t,500)))*57.3, $
    ', 3 on 1 unbiased: ',median(cphase(add(tb,500)))*57.3
t=v3*v5*conj(v1)
tb=t-abs(v3)^2-abs(v5)^2-abs(v1)^2+2*n
print,'3 on 1   biased: ',median(cphase(add(t,500)))*57.3, $
    ', 3 on 1 unbiased: ',median(cphase(add(tb,500)))*57.3
t=v4*conj(v6)*v5
tb=t-abs(v4)^2-abs(v6)^2-abs(v5)^2+2*n
print,'3 on 1   biased: ',median(cphase(add(t,500)))*57.3, $
    ', 3 on 1 unbiased: ',median(cphase(add(tb,500)))*57.3
;
cb=fringeframe4(45000)
nb=fringenphot(cb)
vb1=fringevis[cb,k12]
vb2=fringevis[cb,k13]
vb3=fringevis[cb,k14]
vb4=fringevis[cb,k23]
vb5=fringevis[cb,k42]
vb6=fringevis[cb,k43]
;
tb=v1*v4*conj(vb2)
tbb=tb-abs(vb2)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=v1*vb4*conj(v2)
tbb=tb-abs(vb4)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=vb1*v4*conj(v2)
tbb=tb-abs(vb1)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
;
tb=v2*conj(v6)*conj(vb3)
tbb=tb-abs(vb3)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=v2*conj(vb6)*conj(v3)
tbb=tb-abs(vb6)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=vb2*conj(v6)*conj(v3)
tbb=tb-abs(vb2)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
;
tb=v3*v5*conj(vb1)
tbb=tb-abs(vb1)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=v3*vb5*conj(v1)
tbb=tb-abs(vb5)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=vb3*v5*conj(v1)
tbb=tb-abs(vb3)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
;
tb=v4*conj(v6)*vb5
tbb=tb-abs(vb5)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=v4*conj(vb6)*v5
tbb=tb-abs(vb6)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=vb4*conj(v6)*v5
tbb=tb-abs(vb4)^2+n
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
return
;
tb=vb1*vb4*conj(v2)
tbb=tb-abs(v2)^2+nb
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=vb1*v4*conj(vb2)
tbb=tb-abs(v4)^2+nb
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
tb=v1*vb4*conj(vb2)
tbb=tb-abs(v1)^2+nb
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 2 unbiased: ',median(cphase(add(tbb,500)))*57.3
return
;
cc=fringeframe3(45000)
nc=fringenphot(cc)
vc1=fringevis[cc,k12]
vc2=fringevis[cc,k23]
vc3=fringevis[cc,k31]
tc=v1*vb2*vc3
print,'3 on 2   biased: ',median(cphase(add(tb,500)))*57.3, $
    ', 3 on 3         : ',median(cphase(add(tc,500)))*57.3
;
end
;-------------------------------------------------------------------------------
function fringeframe4,n,k12,k13,k14,k23,k42,k43
;
common LocalFringeFrame,seed
;
rad=180/!pi
i1=0.5
i2=1.0
i3=1.0
i4=1.0
i=sqrt(complex(-1.0,0.0))
v12=1.000
v13=1.000
v14=1.000
v23=1.000
v42=1.000
v43=1.000
p12=0.0000
p13=0.0000
p14=0.0000
p23=0.0000
p42=0.0000
p43=0.0000
numbin=64l
print,'p12+p23-p13=',(p12+p23-p13) mod 360
print,'p13-p43-p14=',(p13-p43-p14) mod 360
print,'p14+p42-p12=',(p14+p42-p12) mod 360
print,'p23-p43+p42=',(p23-p43+p42) mod 360
p=findgen(numbin)/numbin*2*!pi
k1=-1
k2=+4
k3=-3
k4=+3
k12=k2-k1
k13=k3-k1
k14=k4-k1
k23=k3-k2
k42=k2-k4
k43=k3-k4
;help,k12,k13,k14,k23,k42,k43
b1=2*sqrt(i1*i2)*v12*cos(p*k12-p12/rad)
b2=2*sqrt(i1*i3)*v13*cos(p*k13-p13/rad)
b3=2*sqrt(i1*i4)*v14*cos(p*k14-p14/rad)
b4=2*sqrt(i2*i3)*v23*cos(p*k23-p23/rad)
b5=2*sqrt(i4*i2)*v42*cos(p*k42-p42/rad)
b6=2*sqrt(i4*i3)*v43*cos(p*k43-p43/rad)
b=i1+i2+i3+i4+b1+b2+b3+b4+b5+b6
;
b=b*0.5
;plot,b,psym=0
counts=bytarr(numbin*n)
for i=0L,n_elements(counts)-1 do counts[i]=randomu(seed,poisson=b[i mod numbin])
;
return,reform(counts,numbin,n)
;
end
;-------------------------------------------------------------------------------
pro displayframe,c
;
; Display NPOI fringe frames, with the bin axis up and the channels 
; to the right. If a single channel is selected, the frames are just
; associated with a virtual channel axis. The number of frames here
; is defined as the number of xy-pixel images which have to be displayed.
; It is equal to the real frame number if the frames are full bin/channel
; frames.
;
; This procedure is thus designed to produce meaningfull frame data
; displays under various selections for the bincounts passed to it.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(c) eq 0 then begin
	print,'***Error(DISPLAYFRAMES): frames undefined!'
	return
endif
b=float(reform(c))
r=size(b)
index=where(r[1:r[0]] eq genconfig.numbin,count) & index=index[0]+1
if count eq 0 then begin
	print,'***Error(DISPLAYFRAMES): unknown frame format!'
	return
endif
if index ge 2 then num_spec=r[1] else num_spec=1
if r[0] gt index+1 then num_frame=r[index+2] else num_frame=1
if r[0] eq 1 then num_chan=1 else num_chan=r[index+1]
;
ny=128
nx=max([128,num_chan])
if num_chan eq 1 then nx=ny/2
y=float(indgen(ny)/(ny/genconfig.numbin))
x=float(indgen(nx)/(nx/num_chan))
;b=b/max(b)*255
b=hist_equal(b)
b=reform(b,num_spec,genconfig.numbin,num_chan,num_frame)
;
for i=0,num_spec-1 do begin
;
if !d.window eq -1 or !d.x_size ne nx or !d.y_size ne ny or i ne 0 then begin
	if nx gt 640 then window_slide,xsize=nx,ysize=ny $
		     else window,xsize=nx,ysize=ny,/free
endif
;
for j=0,num_frame-1 do begin
tvscl,transpose(byte(interpolate(reform(b[i,*,*,j],genconfig.numbin,num_chan), $
                                 y,x,/grid)))
wait,0.1
endfor
;
endfor
;
end
;-------------------------------------------------------------------------------
function rotateframe,c,l,d,a
;
; l = wavelength [m]
; d = delay tracking offset [m]
; a = airpath [m]
;
; Rotate a frame by d and a. The airpath is translated into delay using
; the refractive index of air for wavelengths l.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
r=size(c)
numbin=r[1]
;
if n_elements(d) eq 1 then g=[d] else g=reform(d)
if n_elements(a) ne 0 then begin
	n=edlen(l*1e6)
	g=reform((l*0+1)#g+(n-1)#a,n_elements(l),n_elements(a))
endif
r=size(g)
if r[0] eq 1 then s=nint((numbin/l)#g) $
	     else begin
		  s=fix(g)
		  for j=0,n_elements(l)-1 do s[j,*]=nint(numbin/l[j]*g[j,*])
	          endelse
;
c_r=c
;
for i=0L,n_elements(d)-1 do begin
for j=0L,n_elements(l)-1 do begin
	c_r[*,j,i]=shift(c[*,j,i],s[j,i])
endfor
endfor
;
return,c_r
;
end
;-------------------------------------------------------------------------------
function fringevis,c
;
; Obsolete!
; Return complex visibility using old 8-bin CONSTRICTOR code.
;
scale1=cos(!pi/8)
scale2=sin(!pi/8)
;
real=(-float(c[*,0,*,*])+c[*,3,*,*]+c[*,4,*,*]-c[*,7,*,*])*scale1 $
    +(-float(c[*,1,*,*])+c[*,2,*,*]+c[*,5,*,*]-c[*,6,*,*])*scale2
imag=(-float(c[*,0,*,*])-c[*,3,*,*]+c[*,4,*,*]+c[*,7,*,*])*scale2 $
    +(-float(c[*,1,*,*])-c[*,2,*,*]+c[*,5,*,*]+c[*,6,*,*])*scale1
;
return,reform(complex(real,imag))
;
end
;-------------------------------------------------------------------------------
function fringevis,c
;
; Obsolete!
; Return complex visibility using DM's 8-bin formula.
;
real=(float(c[*,0,*,*])-c[*,4,*,*]) $
    +(float(c[*,1,*,*])-c[*,3,*,*]-c[*,5,*,*]+c[*,7,*,*])/sqrt(2)
imag=(float(c[*,2,*,*])-c[*,6,*,*]) $
    +(float(c[*,1,*,*])+c[*,3,*,*]-c[*,5,*,*]-c[*,7,*,*])/sqrt(2)
;
return,reform(complex(real,imag))
;
end
;-------------------------------------------------------------------------------
function fringevis8,c
;
; Obsolete!
; Return complex visibility using old 8-bin CONSTRICTOR code.
;
scale1=cos(!pi/8)
scale2=sin(!pi/8)
;
real=(-float(c[0,*,*])+c[3,*,*]+c[4,*,*]-c[7,*,*])*scale1 $
    +(-float(c[1,*,*])+c[2,*,*]+c[5,*,*]-c[6,*,*])*scale2
imag=(-float(c[0,*,*])-c[3,*,*]+c[4,*,*]+c[7,*,*])*scale2 $
    +(-float(c[1,*,*])-c[2,*,*]+c[5,*,*]+c[6,*,*])*scale1
;
return,reform(complex(real,imag))
;
end
;-------------------------------------------------------------------------------
function fringevis,c,k
;
; Compute the complex visibilities from fringe frames stored in c.
; The first index in the frames must correspond to the phase bins.
; Use k to select a baseline, if multiple baseline are present.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
c=reform(c)
if n_elements(k) eq 0 then k=1
;
n0=n_elements(c[*,0,0])
n1=n_elements(k)
n2=n_elements(c[0,*,0])
n3=n_elements(c[0,0,*])
;
vis=complexarr(n1,n2,n3)
;
for l=0,n1-1 do begin
for j=0,n0-1 do begin
	p=2*!pi*i_complex*j*k[l]/n0
	vis[l,*,*]=vis[l,*,*]+c[j,*,*]*exp(p)
endfor
endfor
;
return,reform(vis)
;
end
;-------------------------------------------------------------------------------
function fringenphot,c
;
; Return photonrate computed from fringe frames.
; The first index in the frames must correspond to the phase bins.
;
return,total(float(reform(c)),1)
;
end
;-------------------------------------------------------------------------------
function fringemphot,c
;
return,reform(float(c[0,*,*])-c[1,*,*]+c[2,*,*]-c[3,*,*] $
                   +c[4,*,*] -c[5,*,*]+c[6,*,*]-c[7,*,*])
;
end
;-------------------------------------------------------------------------------
function fringevissq,b,k,n,normal=normal,error=error, $
	delay=delay,lambda=lambda,numcoh=numcoh,bin8=bin8
;
; Return the squared normalized visibility amplitude. Zero if no counts.
; The first index in the frames must correspond to the phase bins.
; Use k to select a baseline, if multiple baseline are present.
; Average incoherently n number of samples.
;
; If given a delay [m] and a number of samples to integrate coherently,
; this function will ...
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(k) eq 0 then k=1
if n_elements(n) eq 0 then n=1l else n=long(n)
;
if n_elements(normal) eq 0 then normal=0 else normal=normal gt 0
if n_elements(numcoh) eq 0 then numcoh=1
if n_elements(bin8) eq 0 then bin8=0
;
b=reform(b)
s=size(b)
n0=s[1]	; Number of bins
;
; Do the coherent integration if requested
if numcoh gt 1 then c=add(rotateframe(fix(b),lambda,delay),numcoh,2) $
	       else c=fix(b)
r=fringenphot(b); *(2./3.)	; if there is incoherent flux 
os=replicate({n:transpose(r)},n0)
o=transpose(os.n)/n0
if bin8 then v=fringevis8(b) else v=fringevis(b-o,k)
if n_elements(delay) gt 0 then v=visrotate[v,lambda,delay]
if numcoh gt 1 then begin
	v=add(v,numcoh,1)
	r=add(r,numcoh,1)
endif
nominator=add(abs(v)^2-r,n,s[0]-2,sdev=sdev)/n
rate=add(r,n,s[0]-2)/n
denominator=rate^2
index=where(denominator eq 0,count)
if count gt 0 then denominator[index]=1
v=nominator/denominator
if count gt 0 then v[index]=0
error=(sdev/sqrt(n))/denominator
;
if normal then norm_factor=4.0/sinc(float(k)/n0)^2 $
	  else norm_factor=1.0
;
error=error*norm_factor
;
return,v*norm_factor
;
end
;-------------------------------------------------------------------------------
function fringetriple,c,k1,k2,n
;
; Compute the complex triple amplitude from fringe frames stored
; in c. The first index in the frames must correspond to the phase bins.
; Use k1 and k2 to select two baselines on the same detector, the third
; will be at -k1-k2 so that the sum k1+k2+k3=0. Average n number of samples.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
k3=-k1-k2
;
; Assume 3 telescopes on this detector
numsid=3
;
result=size(c)
n0=result[1]
norm_factor12=4.0/sinc(float(k1)/n0)^2 
norm_factor23=4.0/sinc(float(k2)/n0)^2 
norm_factor31=4.0/sinc(float(k3)/n0)^2 
norm_factor=sqrt(norm_factor12) $
	   *sqrt(norm_factor23) $
	   *sqrt(norm_factor31) $
	   *(numsid/2.0)^3
;
v1=fringevis(c,k1)
v2=fringevis(c,k2)
v3=fringevis(c,k3)
;
r=fringenphot(c)
;
; Triple product
t=v1*v2*v3
;
return,t
; Poisson bias correction
tb=t-abs(v1)^2-abs(v2)^2-abs(v3)^2+2.0*r
;
return,(add(tb,n)/n)*norm_factor/(add(r,n)/n)^3
;
end
;-------------------------------------------------------------------------------
function fringetriples,baseline1,baseline2,baseline3,beam1,beam2,beam3, $
	scanfile=scanfile,numav=numav
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scanfile) ne 0 then scanfiles=file_search(scanfile+'.??') $
			     else scanfiles=''
;
if n_elements(beam) eq 0 then beam=OutputBeam $
			 else if n_elements(scanfile) eq 0 $
			 	then get_bincounts,beam
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
;
end
;************************************************************************Block 2
function natoffset,c
;
; Return NAT offset in X and Y for quadcounts c for a specific siderostat.
;
cf=float(reform(c))
n=summe(cf,0)
index=where(n eq 0,count)
if count gt 0 then n[index]=1
x=reform((cf[0,*]+cf[3,*])-(cf[1,*]+cf[2,*]))/n
y=reform((cf[2,*]+cf[3,*])-(cf[0,*]+cf[1,*]))/n
if count gt 0 then begin
	x[index]=0
	y[index]=0
endif
;
return,[[x],[y]]
;
end
;-------------------------------------------------------------------------------
function natmfluxes,c
;
return,avg(total(c,2),1)
;
end
;-------------------------------------------------------------------------------
pro natsurface,nat
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
xy=natoffset(quadcounts[nat-1,*,*])
n=total(quadcounts[nat-1,*,*],2)
;
x=reform(xy[*,0])
y=reform(xy[*,1])
z=reform(n)
;
hh=fltarr(20,20)
for i=0,19 do begin
	j=where(y gt -0.5+i*0.05 and y lt -0.5+(i+1)*0.05)
	hh[i,*]=histogramm(x[j],z[j],binsize=0.05,min=-0.5,max=0.5)
endfor
;
!z.range=[0,max(hh)]
surface,hh,findgen(20)*0.05-0.5,findgen(20)*0.05-0.5,/lego
;
end
;-------------------------------------------------------------------------------
function onsource,baseline
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
limit=10
;
i=where(genconfig.stationid eq strmid(baseline,0,3))
j=where(genconfig.stationid eq strmid(baseline,4,3))
natcounts=total(float(quadcounts),2)
;
if median(natcounts[i,*]) lt limit $
or median(natcounts[j,*]) lt limit then return,0 else return,1
;
end
;************************************************************************Block 3
function visrotate,vis,l,d,air,glass
;
; Rotate complex visibility by group delay d and (optionally) air path a.
; The delay can be a vector, in which case the same delay applies to every
; wavelength l, or a 2-dimensional array, with the first dimension 
; corresponding to l.
;
; l = wavelength [m]
; d = group delay [m]
; air = airpath [m]
; glass = path through glass [m] (if defined, so must be air
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(d) eq 1 then g=[d] else g=reform(d)
if n_elements(air) eq 0 then air=g*0
n_air=edlen(l*1e6)
if n_elements(glass) eq 0 then glass=g*0
n_glass=silica(l*1e6)
g=reform((l*0+1)#g+(n_air-1)#air+(n_glass-1)#glass,n_elements(l),n_elements(d))
;
r=size(g)
if r[0] eq 1 then p=float(2*pi_circle*(1/l)#g) $
	     else begin
		  p=reform(float(vis))
		  for j=0,n_elements(l)-1 do p[j,*]=2*pi_circle*g[j,*]/l[j]
		  endelse
;
vis_r=reform(vis)*exp(i_complex*p)
;
return,vis_r
;
end
;-------------------------------------------------------------------------------
function visdft,vis,l,d
;
; Return complex Fourier transform of visibility wrt to wavenumber.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
n=(2*!pi)/reform(l)
f=complex(cos(d#n),sin(d#n))
v=reform(vis)
num1=n_elements(f[*,0])
num2=n_elements(v[0,*])
if num1 ne num2 then return,complex(cos(d#n),sin(d#n))#v
r=complexarr(num1)
for i=0l,num1-1 do r[i]=total(f[i,*]*v[*,i])
return,r
;
end
;-------------------------------------------------------------------------------
function whitedelay,v,l
;
; Return delay of white light fringe.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
w=total(v,1)
p=cphase(w)
d=(p/(2*pi_circle))*avg(l)
;
return,d
;
end
;-------------------------------------------------------------------------------
function lambdawhite,v,l
;
; Return visibility weighted mean of white light wavelength.
;
w=total(abs(v),2)
w=w/max(w)
w[*]=1	; Weighting is actually not recommended
return,total(w*l)/total(w)
;
end
;-------------------------------------------------------------------------------
function phaseunwrap,p
;
; Unwrap a vector of phases, in radians.
;
phase=float(p)
phaseerr=phase*0+0.1
;
NOB=1L
NCH=n_elements(phase[*,0])
NBL=1L
numbaseline=1L
numchannel=NCH
numpoint=n_elements(phase[0,*])
;
status=linknload(!external_lib,'unwrapphase',phase,phaseerr, $
        NOB,NBL,NCH,numbaseline,numchannel,numpoint)
;
return,phase
;
end
;-------------------------------------------------------------------------------
function phasescan,t,p,wrap=wrap
;
; For given times t [s] and phases p [rad], compute mean phase in tint second
; intervals, using histogram averaging in four bins.
;
common LocalPhaseScan,lastphase
;
if n_elements(lastphase) eq 0 then lastphase=0
if n_elements(wrap) eq 0 then wrap=0 else wrap=wrap gt 0
;
bin=2*!pi/4
pmin=-!pi
pmax=+!pi
;
ts=t-t[0]
tint=5.0
n=ceil(max(ts)/tint)
;
phase=fltarr(n)
time=findgen(n)*tint+tint/2
;
for i=0L,n-1 do begin
	index=where(ts gt i*tint and ts lt (i+1)*tint,count)
	if count gt 0 then begin
		r=histogram(p[index],binsize=bin,max=pmax,min=pmin)
		phase[i]=atan(r[3]-r[1],r[2]-r[0])+!pi/4
	endif else phase[i]=4
endfor
;
index=where(phase ne 4)
time=time[index]
phase=phase[index]
phase=phaseunwrap(phase)
if wrap then phase=phase-nint((phase[0]-lastphase)/(2*!pi))*(2*!pi)
wrap=1
lastphase=phase[n_elements(phase)-1]
;
case n_elements(time) of
	1: return,fltarr(n_elements(ts))+phase[0]
	2: return,interpol(phase,time,ts)
     else: return,spline(time,phase,ts)
endcase
;
end
;-------------------------------------------------------------------------------
function phaseshift,r,l
;
; Return the phase shift (group delay phase) between the central fringe
; and the peak of the power spectrum (group delay) for a given residual 
; delay r[m] and l=lambda [m]. Approximate.
;
; Write the refractive index n as: n=n0+a*s where s=1/l.
; Then the phase p after tracking is: p=2*Pi*s*A*(n0+a*s)-2*Pi*d*s,
; with A the differential air path, and d the delay compensation.
; If we use the group delay for d, i.e. d=A*n0, then the tracked fringe 
; phase will be: p=2*Pi*A*a*s^2. To return the white light phase shift,
; we have to sum this over all wavelengths.
;
n=edlen(l*1e6)-1
s=1/(l*1e6)
c=poly_fit(s,n,1)
return,total(2*!pi*(r*1e6)/c[0]#(c[1]*s^2),2)/n_elements(l)
;
; Old code using simulated fringe frames
n=20
a=findgen(n)*1e3
g=findgen(n)/3
frames=fringeframe(a,-g,l*1e6)
v=fringevis(frames)
d=gitter(41,-4e-6)
p=visdft(v,l,d)
gd=groupdelay[p,l,d]
gp=cphase(visdft(v,l,gd))
c=poly_fit(g,gp,1,yfit)
;plot,g,gp,psym=1 & oplot,g,yfit,psym=0
return,poly(r*1e6,c)
;
end
;-------------------------------------------------------------------------------
function timejitter,t,s,stroke
;
; Correct delay s (sidereal) for jitter in timestamps t.
; Return delays corresponding to nearest integer values
; of the timestamps. Corrections are applied both for 
; stroke and sidereal rate.
;
timeshift=median(t mod 1)
x=double(nint(t-timeshift))+timeshift
s_new=s
;
index=where(x ne t,count)
if count gt 0 then begin
	ddt=median(abs(t[index]-x[index]))
	print,'Jitter seems to be ',ddt, $
		', including timeshift of ',timeshift,'.'
	s_new[index]=s_new[index]-ddt*stroke
	c=perifit(t,s_new,86400000,y)
	s_new[index]=s_new[index] $
	            -(y[index]-interpol(y,t,x[index]))
endif
if timeshift ne 0 then begin
	c=perifit(t,s_new,86400000,y)
	s_new=s_new $
	     -(y-interpol(y,t,double(nint(x))))
endif
;
return,s_new
;
end
;-------------------------------------------------------------------------------
function fdlstamp,scan=scan
;
; Return time [ms] derived from time stamp as nearest integer time.
; This is the correct time to be used with the delays corrected for 
; time stamp jitter.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scan) eq 0 then scan=1 else scan=scan gt 0
;
if scan or n_elements(Raw0) eq 0 then begin
	rec0=0
	recn=n_elements(timestamp)-1
endif else begin
	rec0=raw0
	recn=rawn
endelse
;
return,nint(timestamp[rec0:recn])
;
end
;-------------------------------------------------------------------------------
function fdldelay,bl,ob,scan=scan,coeffs=coeffs
;
; Return residual delay on baseline 'ST1-ST2', defined as fdl(ST2)-fdl(ST1).
; Correct returned delay for time stamp jitter. If keyword "scan" is false,
; return only for current range of records loaded.
; Baseline bl can be specified either as a string, or the baseline index
; in conjunction with the output beam index ob. If ob is undefined, then
; it is derived from the currently loaded output beam data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scan) eq 0 then scan=1 else scan=scan gt 0
;
if scan or n_elements(Raw0) eq 0 then begin
	rec0=0
	recn=n_elements(timestamp)-1
endif else begin
	rec0=raw0
	recn=rawn
endelse
;
r=size(bl) & n=n_elements(r)
if n_elements(ob) eq 0 then if n_elements(outputbeam) ne 0 then ob=outputbeam
if r[n-2] ne 7 then baseline=genconfig.baselineid[bl-1,ob-1] $
	       else baseline=strupcase(bl)
;
i1=where(genconfig.stationid eq strmid(baseline,0,3)) & i1=i1[0]
i2=where(genconfig.stationid eq strmid(baseline,4,3)) & i2=i2[0]
ibr=genconfig.refstation-1
d1=laserpos[i1,rec0:recn]+laserpos[ibr,rec0:recn]*(i1 ne ibr)
d2=laserpos[i2,rec0:recn]+laserpos[ibr,rec0:recn]*(i2 ne ibr)
y=timejitter(timestamp[rec0:recn],reform(d2-d1),genconfig.stroke[i2] $
					       -genconfig.stroke[i1])
x=double(nint(timestamp[rec0:recn]))
period=86400000
; return,perires(x,y,period)
coeffs=perifit(x,y,period,yfit)
return,y-yfit
;
end
;-------------------------------------------------------------------------------
function groupdelay,p,l,d,gindex,plot=plot
;
; Obtain peak position of power spectrum of complex visibilities.
;
; p=power spectrum or complex dft of complex visibility
; l=wavelength array
; d=array of delays for power spectrum
;
n=(2*!pi)/reform(l)
a=abs(p)
;
num=n_elements(a[0,*])
index=indgen(n_elements(a[*,0]))
g=fltarr(num)
pv1=g
pv2=g
;
for i=0l,num-1 do begin
	j=where(max(a[*,i]) eq a[*,i]) & j=j[0]
	if j gt 1 and j lt n_elements(d)-2 then begin
		r=poly_fit([d[j-2],d[j-1],d[j],d[j+1],d[j+2]]*1e6, $
			   [a[j-2,i],a[j-1,i],a[j,i],a[j+1,i],a[j+2,i]],2)
		pv1[i]=max(a[*,i])
;		pv2(i)=max(a(*,i)-(poly(d*1e6,r) > 0))
		pv2[i]=max(a[where(abs(index-j)*(d[1]-d[0]) gt 3e-6),i])
		g[i]=(-r[1]/(2*r[2]))/1e6
	endif
endfor
;
if num eq 1 then return,g
;
index=where(pv1 gt 0 and pv2 gt 0)
pv1e=pv1[index]
pv2e=pv2[index]
maxv=2*median(pv2e)
bins=maxv/40
;
if keyword_set(plot) then begin
	title=!p.title
	set_screen
	!p.title=title
	!p.charsize=2
	case plot of
	      1:begin
		window,xsize=500,ysize=400,/free
		!x.title='Primary/secondary peak values'
		!y.range=[0,max([max(histogram(pv1e,binsize=bins*3)), $
				 max(histogram(pv2e,binsize=bins))])]
		histograph,pv1e,min=0,max=maxv*3,binsize=bins*3
		histograph,pv2e,min=0,max=maxv*3,binsize=bins,/oplot,color=tci(2)
		end
	      2:begin
		window,xsize=600,ysize=fix(600*(0.27/0.25)),/free
		!x.margin=[10,5]
		!y.margin=[6,3]
		!x.title='Primary peak value'
		!y.title='Secondary peak value'
		plot,pv1e,pv2e,psym=3,yrange=[0,max(pv1e)],xrange=[0,max(pv1e)]
		oplot,findgen(max(pv1e)),findgen(max(pv1e)),psym=0
		end
	endcase
	!x.title=''
	!y.title=''
endif
;
; Determine which samples are probably bad
method=2
;
case method of
1: 	begin
	h=histogram(pv2e,min=0,max=maxv,binsize=bins)
	hc=0.0
	hr=float(h)/total(h)
	i=0
	repeat begin
		hc=hc+hr[i]
		i=i+1
	endrep until hc gt 0.95
	threshold=bins*i
	ipv1=where(pv1 ge threshold)
	if n_params() eq 4 then $
	print,n_elements(ipv1),' out of ',n_elements(pv1), $
		' samples above threshold of ',threshold, $
		format='(i0,a,i0,a,f5.0)'
	end
2: 	begin
	h=histogram(pv2e,min=0,max=maxv,binsize=bins)
	i=where(h eq max(h))
	threshold_r=1.20
	threshold_a=i[0]*bins*threshold_r
	ipv1=where(pv1 gt pv2*threshold_r $
	       and pv1 gt threshold_a)
	if n_params() eq 4 then $
	print,n_elements(ipv1),' out of ',n_elements(pv1), $
		' samples above threshold of ',threshold_a, $
		format='(i0,a,i0,a,f5.0)'
	end
endcase
;
gindex=lonarr(num)
gindex[ipv1]=1
;
return,g
;
end
;-------------------------------------------------------------------------------
function trackdelay,v,l,f,p,t,gindex,wrap=wrap
;
; Compute the corrected group delay and return the corrected visibility
; phases. Think of the corrected group delay as the residual delay
; between the actual delay line tracking and an ideal delay line.
; The ideal delay line here is one using group delay tracking such that
; the mean white light fringe phase is close to zero.
;
; If parameter gindex is passed (to be set on return), use new algorithm
; to flag low-SNR groupdelays and smooth/interpolate those samples.
;
; v=complex visibility
; l=array of wavelengths [m]
; f=residual laser delay [m]
; t=time [s]
;
if n_elements(wrap) eq 0 then wrap=0 else wrap=wrap gt 0
;
p2d=lambdawhite(v,l)/(2*!pi)
;
d=gitter(121,-12e-6)
numav=10	; Average this many power spectra for group delay
;
if n_params() eq 6 then begin
	g=groupdelay(abs(visdft(v,l,d)),l,d,gindex)
	g=gsmooth(t[where(gindex gt 0)],g[where(gindex gt 0)],numav*0.001,t)
endif else begin
; 	Subtract any delay "vibrations" before averaging the power spectrum
;	s=f-smooth(f,100,/edge_truncate)
	s=f-median(f,100)
	g=s+groupdelay(box(abs(visdft(visrotate(v,l,s),l,d)),numav,1),l,d,index)
endelse
; f-g is the actual fringe position in case the delay line is off the fringe
g=g+phaseshift(median(f,100)-g,l)*p2d
p=cphase(total(visrotate(v,l,g),1))
c=phasescan(t,p,wrap=wrap)
g=g-c*p2d
p=cphase(total(visrotate(v,l,g),1))
;
return,g
;
end
;-------------------------------------------------------------------------------
function fringedelay,v,l,f,t,gindex,wrap=wrap
;
; This function returns a delay which is usable for coherent
; integration of visibilities. This involves white light phase
; tracking on the corrected fringe phases returned by function 
; "fringephase".
;
; l: lambda [m]
; f: residual laser delay [m]
; t: time [s]
;
if n_elements(wrap) eq 0 then wrap=0 else wrap=wrap gt 0
v=reform(temporary(v))
;
p2d=lambdawhite(v,l)/(2*!pi)
;
if n_params() eq 5 then d=trackdelay(v,l,f,p,t,gindex,wrap=wrap) $
		   else d=trackdelay(v,l,f,p,t,wrap=wrap)
return,d-p*p2d
;
end
;-------------------------------------------------------------------------------
function groupdelays,baseline,beam,scanfile, $
		channels=channels,numav=numav,gindex=gindex,plot=plot
;
; Compute and return groupdelay for specified baseline.
; Must have used get_rawdata first!
; If scanfile defined, compute and concatenate for all sub-scans.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scanfile) ne 0 then scanfiles=file_search(scanfile+'.??') $
			     else scanfiles=''
;
if n_elements(beam) eq 0 then beam=OutputBeam $
			 else if n_elements(scanfile) eq 0 $
			 	then get_bincounts,beam
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
;
if n_elements(numav) eq 0 then numav=1
;
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
f=fdldelay(genconfig.baselineid[bl,ob],scan=n_elements(scanfile))
delay=fltarr(n_elements(f))
gindex=intarr(n_elements(f))
;
for i=0,n_elements(scanfiles)-1 do begin
	if strlen(scanfiles[i]) gt 0 then begin
		hds_open,scanfiles[i]
		get_bincounts,beam
		hds_close
	endif
	v=fringevis(bincounts[*,ch,*],genconfig.fringemod[bl,ob])
	delay[Raw0:RawN]=groupdelay(box(abs(visdft(v,l,d)),numav,1),l,d,index, $
		plot=plot)
	gindex[Raw0:RawN]=index
endfor
;
return,delay
;
end
;-------------------------------------------------------------------------------
function trackdelays,baseline,beam,scanfile,channels=channels,gindex=gindex
;
; Compute and return tracking delays for specified baseline.
; Must have used get_rawdata first!
; If scanfile defined, compute and concatenate for all sub-scans.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scanfile) ne 0 then scanfiles=file_search(scanfile+'.??') $
			     else scanfiles=''
;
if n_elements(beam) eq 0 then beam=OutputBeam $
			 else if n_elements(scanfile) eq 0 $
			 	then get_bincounts,beam
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
if keyword_set(gindex) then classic=0 else classic=1
;
l=genconfig.wavelength[ch,ob]
;
phases=fltarr(n_elements(fdlstamp(scan=n_elements(scanfile))))
delays=phases
gindex=fix(phases*0)
;
for i=0,n_elements(scanfiles)-1 do begin
	if strlen(scanfiles[i]) gt 0 then begin
		hds_open,scanfiles[i]
		get_bincounts,beam
		hds_close
	endif
	v=fringevis(bincounts[*,ch,*],genconfig.fringemod[bl,ob])
	f=fdldelay(genconfig.baselineid[bl,ob],scan=0)
	t=fdlstamp()/1000.d0 & t=t-t[0]
	if classic then begin
			delays[raw0:rawn]=trackdelay(v,l,f,p,t)
			phases[raw0:rawn]=p
		   endif else begin
			delays[raw0:rawn]=trackdelay(v,l,f,p,t,index)
			phases[raw0:rawn]=p
			gindex[raw0:rawn]=index
		   endelse
endfor
;
return,delays
;
end
;-------------------------------------------------------------------------------
function fringedelays,baseline,beam,scanfile,channels=channels,gindex=gindex
;
; Compute and return fringe delays for specified baseline.
; (To compute total atmospheric delays, calculate fdl-fringedelay!)
; Must have used get_rawdata first!
; If scanfile defined, compute and concatenate for all sub-scans.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scanfile) ne 0 then scanfiles=file_search(scanfile+'.??') $
			     else scanfiles=''
;
if n_elements(beam) eq 0 then beam=OutputBeam $
			 else if n_elements(scanfile) eq 0 $
			 	then get_bincounts,beam
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
if keyword_set(gindex) then classic=0 else classic=1
;
f=fdldelay(genconfig.baselineid[bl,ob],scan=n_elements(scanfile))
delays=fltarr(n_elements(f))
gindex=delays*0
;
wrap=0
;
for i=0,n_elements(scanfiles)-1 do begin
	if strlen(scanfiles[i]) gt 0 then begin
		hds_open,scanfiles[i]
		get_bincounts,beam
		hds_close
		d=f[raw0:rawn]
	endif else d=f
	if classic then $
	delays[Raw0:RawN]=fringedelay(fringevis(bincounts[*,ch,*], $
				genconfig.fringemod[bl,ob]), $
				genconfig.wavelength[ch,ob], $
				d,fdlstamp(scan=0)/1000.d0,wrap=wrap) $
	else begin
	delays[Raw0:RawN]=fringedelay(fringevis(bincounts[*,ch,*], $
				genconfig.fringemod[bl,ob]), $
				genconfig.wavelength[ch,ob], $
				d,fdlstamp(scan=0)/1000.d0,index,wrap=wrap)
	gindex[raw0:rawn]=index
	endelse
endfor
;
return,delays
;
end
;-------------------------------------------------------------------------------
function fringephases,baseline,beam,scanfile,channels=channels,gindex=gindex
;
; Compute and return corrected fringe phases for specified baseline.
; Must have used get_rawdata first!
; If scanfile defined, compute and concatenate for all sub-scans.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scanfile) ne 0 then scanfiles=file_search(scanfile+'.??') $
			     else scanfiles=''
;
if n_elements(beam) eq 0 then beam=OutputBeam $
			 else if n_elements(scanfile) eq 0 $
			 	then get_bincounts,beam
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
if keyword_set(gindex) then classic=0 else classic=1
;
l=genconfig.wavelength[ch,ob]
;
phases=fltarr(n_elements(fdlstamp(scan=n_elements(scanfile))))
delays=phases
gindex=fix(phases*0)
;
for i=0,n_elements(scanfiles)-1 do begin
	if strlen(scanfiles[i]) gt 0 then begin
		hds_open,scanfiles[i]
		get_bincounts,beam
		hds_close
	endif
	v=fringevis(bincounts[*,ch,*],genconfig.fringemod[bl,ob])
	f=fdldelay(genconfig.baselineid[bl,ob],scan=0)
	t=fdlstamp()/1000.d0 & t=t-t[0]
	if classic then begin
			delays[raw0:rawn]=trackdelay(v,l,f,p,t)
			phases[raw0:rawn]=p
		   endif else begin
			delays[raw0:rawn]=trackdelay(v,l,f,p,t,index)
			phases[raw0:rawn]=p
			gindex[raw0:rawn]=index
		   endelse
endfor
;
return,phases
;
end
;-------------------------------------------------------------------------------
function coherentvis,baseline,beam,scanfile,channels=channels,cohint=cohint, $
						classic=classic,delay=delay
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(scanfile) ne 0 then scanfiles=file_search(scanfile+'.??') $
			     else scanfiles=''
;
if n_elements(beam) eq 0 then beam=OutputBeam $
			 else if n_elements(scanfile) eq 0 $
			 	then get_bincounts,beam
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
if n_elements(cohint) eq 0 then cohint=100
if n_elements(classic) eq 0 then classic=1
;
l=genconfig.wavelength[ch,ob]
k=genconfig.fringemod[bl,ob]
if scantable[0].starid eq 'FKV0000' and genconfig.beamcombinerid eq 3 $
				    then k=k*2
;
f=fdldelay(genconfig.baselineid[bl,ob],scan=n_elements(scanfile))
;
visibility=complexarr(n_elements(ch),n_elements(f)/cohint)
delay=fltarr(n_elements(f)/cohint)
;
n=0
wrap=0
;
for i=0,n_elements(scanfiles)-1 do begin
	if strlen(scanfiles[i]) gt 0 then begin
		hds_open,scanfiles[i]
		get_bincounts,beam
		hds_close
		d=f[raw0:rawn]
	endif else d=f
;
	v=fringevis(bincounts[*,ch,*],k)
	if classic then d=fringedelay(v,l,d,fdlstamp(scan=0)/1000.d0,wrap=wrap) $
		   else d=fringedelay(v,l,d,fdlstamp(scan=0)/1000.d0,gindex,wrap=wrap)
;
	n1=(rawn-raw0+1)/cohint
	delay[n:n+n1-1]=add(d,cohint)/cohint
	visibility[*,n:n+n1-1]=add(visrotate(v,l,d),cohint,1)
	n=n+n1
endfor
;
if scantable[0].starid eq 'FKV0000' then delay=delay/2
;
return,visibility
;
end
;************************************************************************Block 4
pro cohstrictor,cohfile,classic=classic,rawpath=rawpath
;
; Coherently integrates NPOI data for astrometry. Not to be used for any other
; purpose as yet. 6-way compatible version.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(classic) eq 0 then classic=1
;
hds_open,cohfile
get_sysconfig
get_scantable
fixearlydays
np=scantable.numpoint
scan_ids=scantable.scanid
numcoh=scantable.numcoh*scantable.numincoh
hds_close
;
if n_elements(rawpath) eq 0 then rp='' else rp=rawpath+'/'
stub=rp+checkdate()+'.raw.'
;
n=n_elements(scan_ids)
FOR i=0,n-1 DO BEGIN
;
rawfile=stub+string(scan_ids[i],format='(i3.3)')+'.01'
print,'Now processing ',rawfile,'...'
hds_open,rawfile,'READ',status
if status ne 0 then return
get_rawdata
hds_close
if n_elements(add(timestamp,numcoh[i])) ne np[i] then begin
	print,'***ERROR(COHSTRICTOR): bad # of samples!'
	return
endif
;
for j=0,genconfig.numoutbeam-1 do begin
;
	ch=channelindex[genconfig.spectrometerid[j]]-1
	l=genconfig.wavelength[ch,j]
;
	hds_open,rawfile,'READ',status
	get_bincounts,j+1
	hds_close
;
	v=complexarr(genconfig.numspecchan[j],genconfig.numbaseline[j],n_elements(timestamp))
	d=fltarr(genconfig.numbaseline[j],n_elements(timestamp))
	g=intarr(genconfig.numbaseline[j],n_elements(timestamp))
;
	vis=complexarr(genconfig.numspecchan[j],genconfig.numbaseline[j],np[i])
	err=vis
	del=fltarr(genconfig.numbaseline[j],np[i])
;
	for k=0,genconfig.numbaseline[j]-1 do begin
		i1=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],0,3))
		i2=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],4,3))
		i1=i1[0]
		i2=i2[0]
		if scantable[0].station(i1) ne 0 and scantable[0].station(i2) ne 0 then begin
		if genconfig.beamcombinerid eq 3 and scantable[0].starid eq 'FKV0000' $
			then kf=2 else kf=1
		v[*,k,*]=fringevis(bincounts,genconfig.fringemod[k,j]*kf)
		if classic then begin
			d[k,*]=fringedelay(v[ch,k,*],l,fdldelay(genconfig.baselineid[k,j]), $
						       fdlstamp()/1000.d0)
		endif else begin
			d[k,*]=fringedelay(v[ch,k,*],l,fdldelay(genconfig.baselineid[k,j]), $
						       fdlstamp()/1000.d0,gindex)
			g[k,*]=gindex
		endelse
		endif
	endfor
	for k=0,genconfig.numbaseline[j]-1 do begin
		i1=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],0,3))
		i2=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],4,3))
		i1=i1[0]
		i2=i2[0]
		if scantable[0].station(i1) ne 0 and scantable[0].station(i2) ne 0 then begin
		r=reform(visrotate(v[*,k,*],genconfig.wavelength[*,j],d[k,*]))
		if classic then begin
			vis[*,k,*]=add(r,numcoh[i],1)
			err[*,k,*]=complex(1,1)
			del[k,*]=add(reform(d[k,*]),numcoh[i])/numcoh[i]
		endif else begin
			for is=0,np[i]-1 do begin
				rg=r[*,is*numcoh[i]:(is+1)*numcoh[i]-1]
				gi=g[k,is*numcoh[i]:(is+1)*numcoh[i]-1]
				dg=d[k,is*numcoh[i]:(is+1)*numcoh[i]-1]
				index=where(gi gt 0,count)
				if count ge 2 then begin
					vis[*,k,is]=summe(rg[*,index],1)/count
					err[*,k,is]=complex(sqrt(avg(float(rg[*,index])^2,1) $
								-avg(float(rg[*,index]),1)^2), $
							    sqrt(avg(imaginary(rg[*,index])^2,1) $
								-avg(imaginary(rg[*,index]),1)^2))
					del[k,is]=summe(reform(dg[index]))/count
				endif else begin
					err[*,k,is]=complex(-1,-1)
					del[k,is]=summe(reform(dg))/numcoh[i]
				endelse
			endfor
		endelse
		endif
	endfor
;
	if scantable[0].starid eq 'FKV0000' then del=del/2
;
	hds_open,cohfile,'UPDATE',status
	if status ne 0 then return
	dat_find,'SCANDATA'
	dat_find,'POINTDATA'
	dat_cell,1,i+1
	dat_find,'OUTPUTBEAM'
;
	dat_cell,1,j+1

	ndim=4
	dims=[2,genconfig.numspecchan[j],genconfig.numbaseline[j],np[i]]
	data=fltarr(dims[0],dims[1],dims[2],dims[3],/nozero)
	data[0,*,*,*]=float(vis)
	data[1,*,*,*]=imaginary(vis)
	name='COMPLEXVIS'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
	name='COMPLEXVISERR'
	data[0,*,*,*]=float(err)
	data[1,*,*,*]=imaginary(err)
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
	ndim=2
	dims=[genconfig.numbaseline[j],np[i]]
	name='SOFTDELAY'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,del
	del=del*0+1
	name='SOFTDELAYERR'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
	hds_close
;
endfor
;
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro constrictor3,confile,stars=stars,numcoh=numcoh,code=code
;
; This coherent integration works with 3-way CEW data only 
; and is only kept here as a reference.
; This code was used for the coherent integration of the Gamma Sagittae data.
;
; No uncertainty estimates are computed.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(numcoh) eq 0 then numcoh=10L
if n_elements(code) eq 0 then code=[0,1]
;
hds_open,confile
get_scantable
get_sysconfig
fixearlydays
np=scantable.numpoint
scan_ids=scantable.scanid
numav=scantable[0].numcoh*scantable[0].numincoh
numincoh=numav/numcoh
sc_code=scantable.code
scantable.numcoh=numcoh
scantable.numincoh=numincoh
numcohs=scantable.numcoh
numincohs=scantable.numincoh
hds_close
;
if n_elements(stars) eq 0 then list_stars,stars
;
if n_elements(rawpath) eq 0 then rp='' else rp=rawpath+'/'
stub=rp+checkdate()+'.raw.'
;
channels=stringparse('1-10,13-20')
ch=channels-1
;
n=n_elements(scan_ids)
for i=0,n-1 do begin
	rawfile=stub+string(scan_ids[i],format='(i3.3)')+'.01'
	print,'Now processing ',rawfile,'...'
	hds_open,rawfile,'READ',status
	if status ne 0 then return
	get_rawdata
	if n_elements(add(timestamp,numcoh*numincoh)) ne np[i] then begin
		print,'***ERROR(CONSTRICTOR): bad # of samples!'
		return
	endif
;
	index=where(stars eq scantable[0].starid,count1)
	index=where(code eq sc_code[i],count2)
;
	IF count1 eq 1 and count2 eq 1 THEN BEGIN
;
	get_bincounts,3
	v3=fringevis(bincounts)
	if sc_code[i] eq 1 then begin
		d3=fringedelay(v3[ch,*],genconfig.wavelength[ch,2], $
			fdldelay(genconfig.baselineid[0,2]),fdlstamp()/1000.d0)
		v3=add(visrotate(v3,genconfig.wavelength[*,2],d3),numcoh,1)
		rf=float(rotateframe(bincounts,genconfig.wavelength[*,2],d3))
;		vsq3=add(abs(v3)^2-fringemphot(add(rf,numcoh,2))^2,numincoh,1) $
;		    /(numincoh*numcoh^2)
		vsq3=add(abs(v3)^2-fringenphot(add(float(bincounts),numcoh,2)),numincoh,1) $
		    /(numincoh*numcoh^2)
	endif else begin
		v3=add(v3,numcoh,1)
;		vsq3=add(abs(v3)^2-fringemphot(add(float(bincounts),numcoh,2))^2,numincoh,1) $
;		    /(numincoh*numcoh^2)
		vsq3=add(abs(v3)^2-fringenphot(add(float(bincounts),numcoh,2)),numincoh,1) $
		    /(numincoh*numcoh^2)
	endelse
;
	get_bincounts,2
	v2=fringevis(bincounts)
	if sc_code[i] eq 1 then begin
		d2=fringedelay(v2[ch,*],genconfig.wavelength[ch,1], $
			fdldelay(genconfig.baselineid[0,1]),fdlstamp()/1000.d0)
		v2=add(visrotate(v2,genconfig.wavelength[*,1],d2),numcoh,1)
		rf=float(rotateframe(bincounts,genconfig.wavelength[*,1],d2))
;		vsq2=add(abs(v2)^2-fringemphot(add(rf,numcoh,2))^2,numincoh,1) $
;		    /(numincoh*numcoh^2)
		vsq2=add(abs(v2)^2-fringenphot(add(float(bincounts),numcoh,2)),numincoh,1) $
		    /(numincoh*numcoh^2)
	endif else begin
		v2=add(v2,numcoh,1)
;		vsq2=add(abs(v2)^2-fringemphot(add(float(bincounts),numcoh,2))^2,numincoh,1) $
;		    /(numincoh*numcoh^2)
		vsq2=add(abs(v2)^2-fringenphot(add(float(bincounts),numcoh,2)),numincoh,1) $
		    /(numincoh*numcoh^2)
	endelse
;
	get_bincounts,1
	v1=fringevis(bincounts)
	if sc_code[i] eq 1 then begin
		d1=fringedelay(v1[ch,*],genconfig.wavelength[ch,0], $
			fdldelay(genconfig.baselineid[0,0]),fdlstamp()/1000.d0)
		v1=add(visrotate(v1,genconfig.wavelength[*,0],d2-d3),numcoh,1)
		rf=float(rotateframe(bincounts,genconfig.wavelength[*,0],d2-d3))
;		vsq1=add(abs(v1)^2-fringemphot(add(rf,numcoh,2))^2,numincoh,1) $
;		    /(numincoh*numcoh^2)
		vsq1=add(abs(v1)^2-fringenphot(add(float(bincounts),numcoh,2)),numincoh,1) $
		    /(numincoh*numcoh^2)
	endif else begin
		v1=add(v1,numcoh,1)
;		vsq1=add(abs(v1)^2-fringemphot(add(float(bincounts),numcoh,2))^2,numincoh,1) $
;		    /(numincoh*numcoh^2)
		vsq1=add(abs(v1)^2-fringenphot(add(float(bincounts),numcoh,2)),numincoh,1) $
		    /(numincoh*numcoh^2)
	endelse
;
	v123=add(v1*conj(v2)*v3,numincoh,1)/(numincoh*numcoh^3)
	hds_close
;
	hds_open,confile,'UPDATE',status
	if status ne 0 then return
	dat_find,'SCANDATA'
	dat_find,'POINTDATA'
	dat_cell,1,i+1

	dat_find,'OUTPUTBEAM'
;
	name='VISSQ'
	ndim=3
	type='_real'
	dat_cell,1,1
	dims=[genconfig.numspecchan[0],1,np[i]]
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,vsq1
	dat_annul
	dat_cell,1,2
	dims=[genconfig.numspecchan[1],1,np[i]]
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,vsq2
	dat_annul
	dat_cell,1,3
	dims=[genconfig.numspecchan[2],1,np[i]]
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,vsq3
	dat_annul
;
	dat_annul
;
	dat_find,'TRIPLE'
;
	name='COMPLTRIPLE'
	ndim=3
	type='_real'
	dat_cell,1,1
	dims=[2,genconfig.numspecchan[0],np[i]]
	data=fltarr(dims[0],dims[1],dims[2],/nozero)
	data[0,*,*]=float(v123)
	data[1,*,*]=imaginary(v123)
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
;
	hds_close
;
	ENDIF ELSE hds_close
;
endfor
;
hds_open,confile,'UPDATE',status
if status ne 0 then return
dat_find,'SCANDATA'
;
name='NUMCOH'
dat_erase,name
type='_integer'
ndim=1
dims[0]=n
dat_new,name,type,ndim,dims
cmp_put1i,name,numcohs
;
name='NUMINCOH'
dat_erase,name
type='_integer'
ndim=1
dims[0]=n
dat_new,name,type,ndim,dims
cmp_put1i,name,numincohs
;
hds_close
;
end
;-------------------------------------------------------------------------------
pro constrictor,confile,rawpath=rawpath,items=items,stars=stars, $
	classic=classic,numcoh=numcoh
;
; New 6-way compatible version. This procedure, with the same name as the 
; standard NPOI raw data averager, is used to replace certain variables with
; better estimates. New functionality can also be added here.
;
; Coherent integration of 6-way data using baseline boot strapping.
; Currently uses only baselines to the reference station.
;
; Available items for pointdata:
; FDLPOS, FDLPOSERR
; NATCOUNTS
; FRINGEJITTER
; COHERENT
; 
; Available items for scandata (and corresponding plot variable):
; NATJITTER		NATJITTER2
; SCINTILLATION		SCINTILLATION
; T0			T0
; DELAYRMS		DELAYRMS
; DELAYJITTER		DELAYJITTER2
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(rawpath) eq 0 then rp='' else rp=rawpath+'/'
if n_elements(items) eq 0 then begin
	print,'Warning(CONSTRICTOR): items undefined, setting to "SEEING"!'
	items=['NATJITTER','SCINTILLATION','T0', $
	       'DELAYRMS','DELAYJITTER','FRINGEJITTER']
endif
if n_elements(classic) eq 0 then classic=1
if n_elements(numcoh) eq 0 then numcoh=100L else numcoh=long(numcoh)
;
; Open existing output file to read some info
hds_close	; In case a file is open
hds_open,confile
get_scantable
get_sysconfig
fixearlydays
np=scantable.numpoint
scan_ids=scantable.scanid
numscan=n_elements(scan_ids)
numav=scantable.numcoh*scantable.numincoh
numincoh=numav/numcoh
hds_close
;
; Do all stars if none are specified
if n_elements(stars) eq 0 then list_stars,stars
;
stub=rp+checkdate()+'.raw.'
;
; Loop over all scans
FOR i=0,numscan-1 DO BEGIN
;
rawfile=stub+string(scan_ids[i],format='(i3.3)')+'.01'
print,'Now processing ',rawfile,'...'
hds_open,rawfile,'READ',status & if status ne 0 then return
get_rawdata
hds_close
;
dp=double(fdlstamp())/86400000*2*!pi
;
if n_elements(add(timestamp,numcoh*numincoh[i])) ne np[i] then begin
	print,'***ERROR(CONSTRICTOR): bad # of samples scan'+string(i+1)+'!'
	return
endif
;
index=where(stars eq scantable[0].starid,count)
;
IF count eq 1 THEN BEGIN
;
; INPUTBEAM section +++
;
; Point data section
;
; FDLPos
if strpos(strupcase(strjoin(items)),'FDLPOS') ne -1 then begin
for j=0,genconfig.numsid-1 do begin
;
fdlposerr=fltarr(np[i])
fdlpos=dblarr(np[i])
;
if j ne genconfig.refstation-1 then begin
	f=fdldelay(genconfig.stationid[j]+'-' $
	  +genconfig.stationid[genconfig.refstation-1],coeffs=coeffs)
	df=f-shift(f,1)
	for l=0,np[i]-1 do begin
		fdlpos[l]=-(median(f[l*numincoh[i]:(l+1)*numincoh[i]-1]) $
			 +perinom(avg(dp[l*numincoh[i]:(l+1)*numincoh[i]-1]),coeffs))
		fdlposerr[l]=abs(medianve(df[l*numincoh[i]:(l+1)*numincoh[i]-1]) $
				*numincoh[i])
		if fdlposerr[l] eq 0 then fdlposerr[l]=1e-6
	endfor
;
	hds_open,confile,'UPDATE',status & if status ne 0 then return
	dat_find,'SCANDATA'
	dat_find,'POINTDATA'
	dat_cell,1,i+1
	dat_find,'INPUTBEAM'
	dat_cell,1,j+1
;
	ndim=1
	dims=np[i]
	name='FDLPOSERR'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,fdlposerr
	name='FDLPOS'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_double'
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,fdlpos
	hds_close
endif
;
endfor
endif
;
; NATCounts
if strpos(strupcase(strjoin(items)),'NATCOUNTS') ne -1 then begin
for j=0,genconfig.numsid-1 do begin
;
	natcounts=add(total(reform(quadcounts[j,*,*]),1),numincoh[i])/numincoh[i]
;
	hds_open,confile,'UPDATE',status & if status ne 0 then return
	dat_find,'SCANDATA'
	dat_find,'POINTDATA'
	dat_cell,1,i+1
	dat_find,'INPUTBEAM'
	dat_cell,1,j+1
;
	ndim=1
	dims=np[i]
	name='NATCOUNTS'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,natcounts
	hds_close
;
endfor
endif
;
; Scan averaged data section
;
; Seeing index based on angle jitter
if strpos(strupcase(strjoin(items)),'NATJITTER') ne -1 then begin
;
if n_elements(natjitter) eq 0 then $
natjitter=fltarr(genconfig.numsid,2,numscan)
;
binsize=0.04
numbin=nint(2.0/binsize)+1
x=(findgen(numbin)-float(numbin-2)/2)*binsize
;
for j=0,genconfig.numsid-1 do begin
;
	if median(quadcounts[j,*,*]) gt 10 then begin
	xy=natoffset(quadcounts[j,*,*])
	y=lonarr(numbin,numbin)
	for k=0,numbin-1 do begin
		index=where(xy[*,1] ge x[k]-binsize/2 $
			and xy[*,1] lt x[k]+binsize/2,count)
		if count gt 0 then $
		y[*,k]=histogram(xy[index,0],min=-1,binsize=binsize,max=1)
	endfor
;
;	Measure width in X and Y (more stable than Gaussian fit)
	yx=total(y,2)
	yy=total(y,1)
	xindex=where(yx gt max(yx)/2)
	yindex=where(yy gt max(yy)/2)
	natjitter[j,0,i]=(max(xindex)-min(xindex))*binsize
	natjitter[j,1,i]=(max(yindex)-min(yindex))*binsize
;	Fit Gaussian profile to 2D histogram
;	aparm=fltarr(7)
;	aparm=[0.0,max(y),7.0,7.0,24.,24.,0.0]
;	r=gauss2dfit(y,aparm)
;	natjitter(j,0,i)=aparm(2)*binsize
;	natjitter(j,1,i)=aparm(3)*binsize
	endif
;
endfor
endif
;
; OUTPUTBEAM section +++
;
; Point data section
;
; FringeJitter (motion of fringe after tracking)
if strpos(strupcase(strjoin(items)),'FRINGEJITTER') ne -1 then begin
if scantable[0].code eq 1 then begin
;
for j=0,genconfig.numoutbeam-1 do begin
	hds_open,rawfile,'READ',status & if status ne 0 then return
	get_bincounts,j+1
;
	fringejitter=fltarr(genconfig.numbaseline[j],np[i])
;
	for k=0,genconfig.numbaseline[j]-1 do begin
		if onsource(genconfig.baselineid[k,j]) then begin
		g=fringedelays(k+1)
		for l=0,np[i]-1 do begin
			v=medianve(g[l*numincoh[i]:(l+1)*numincoh[i]-1],ve)
			fringejitter[k,l]=ve
		endfor
		endif
	endfor
	hds_close
;
	hds_open,confile,'UPDATE',status & if status ne 0 then return
	dat_find,'SCANDATA'
	dat_find,'POINTDATA'
	dat_cell,1,i+1
	dat_find,'OUTPUTBEAM'
	dat_cell,1,j+1
;
	ndim=2
	dims=[genconfig.numbaseline[j],np[i]]
	name='FRINGEJITTER'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,fringejitter
	hds_close
endfor
;
endif
endif
;
; Coherent integration of visibilities
if strpos(strupcase(strjoin(items)),'COHERENT') ne -1 then begin
;
; These arrays store the raw data
d=fltarr(genconfig.numoutbeam,max(genconfig.numbaseline),n_elements(timestamp))
g=intarr(genconfig.numoutbeam,max(genconfig.numbaseline),n_elements(timestamp))
v=complexarr(genconfig.numoutbeam,max(genconfig.numspecchan), $
				  max(genconfig.numbaseline),n_elements(timestamp))
;
; These arrays hold coherent data
vis=complexarr(genconfig.numoutbeam,max(genconfig.numspecchan), $
	max(genconfig.numbaseline),n_elements(timestamp)/numcoh)
var=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan), $
	n_elements(timestamp)/numcoh)
nph=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan), $
	n_elements(timestamp)/numcoh)
;
; These arrays hold the point data
vsq=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan),max(genconfig.numbaseline),np[i])
vse=vsq
t3v=complexarr(genconfig.numtriple,max(genconfig.numspecchan),np[i])
t3e=t3v
;
; Design matrix and right hand side
nrow=fix(total(genconfig.numbaseline))
dm=fltarr(nrow+1,genconfig.numsid)
rh=fltarr(nrow+1,n_elements(timestamp))
row=0
;
; Compute fringe delays and design matrix
for j=0,genconfig.numoutbeam-1 do begin
;
ch=channelindex[genconfig.spectrometerid[j]]-1
l=genconfig.wavelength[ch,j]
;
hds_open,rawfile,'READ',status
get_bincounts,j+1
hds_close
;
for k=0,genconfig.numbaseline[j]-1 do begin
i1=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],0,3))
i2=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],4,3))
i1=i1[0]
i2=i2[0]
if scantable[0].station(i1) ne 0 and scantable[0].station(i2) ne 0 then begin
	v[j,*,k,*]=fringevis(bincounts,genconfig.fringemod[k,j])
	if scantable.code eq 1 then begin
	if i1+1 eq genconfig.refstation $
	or i2+1 eq genconfig.refstation then begin
	if classic then begin
		d[j,k,*]=fringedelay(v[j,ch,k,*],l, $
				fdldelay(genconfig.baselineid[k,j]), $
			        fdlstamp()/1000.d0)
	endif else begin
		d[j,k,*]=fringedelay(v[j,ch,k,*],l, $
				fdldelay(genconfig.baselineid[k,j]), $
				fdlstamp()/1000.d0,gindex)
		g[j,k,*]=gindex
	endelse
	dm[row,i1]=+1
	dm[row,i2]=-1
	rh[row,*]=d[j,k,*]
	endif
	endif
endif
row=row+1
endfor
endfor
;
; Compute delay solutions
if scantable.code eq 1 then begin
dm[nrow,genconfig.refstation-1]=1
tm=transpose(dm)
nm=tm#dm
y=tm#rh
; s=invert(nm)#y
;
svd,nm,w_eigen,u_matrix,v_matrix
small=where(w_eigen lt max(w_eigen)*1e-6,count)
for j=0,count-1 do begin
	w_eigen[small[j]]=0
endfor
s=y
for n=0,n_elements(TimeStamp)-1 do begin
svbksb,u_matrix,w_eigen,v_matrix,y[*,n],sol
s[*,n]=sol
endfor
;
; for k=0l,n_elements(timestamp)-1 do begin
; 	y=r(*,k)
; 	tm=transpose(dm)
; 	nm=tm#dm
; 	r=tm#y
; 	s=invert(n)#r
; endfor
endif else s=fltarr(genconfig.numsid,n_elements(timestamp))
;
; Do the coherent integration
for j=0,genconfig.numoutbeam-1 do begin
;
hds_open,rawfile,'READ',status
get_bincounts,j+1
hds_close
;
; nph(j,*,*)=add(fringenphot(bincounts),numcoh,1)
nph[j,*,*]=fringenphot(add(float(bincounts),numcoh,2))
var[j,*,*]=nph[j,*,*]	; Poisson noise variance
;
for k=0,genconfig.numbaseline[j]-1 do begin
i1=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],0,3))
i2=where(genconfig.stationid eq strmid(genconfig.baselineid[k,j],4,3))
i1=i1[0]
i2=i2[0]
if scantable[0].station(i1) ne 0 and scantable[0].station(i2) ne 0 then begin
	d[j,k,*]=s[i1,*]-s[i2,*]
	r=reform(visrotate(v[j,*,k,*],genconfig.wavelength[*,j],d[j,k,*]))
	if classic then begin
		vis[j,*,k,*]=add(r,numcoh,1)
	endif else begin
		for ip=0,np[i]-1 do begin
			rg=r[*,ip*numcoh:(ip+1)*numcoh-1]
			gi=g[k,ip*numcoh:(ip+1)*numcoh-1]
			dg=d[k,ip*numcoh:(ip+1)*numcoh-1]
			index=where(gi gt 0,count)
			if count ge 2 then $
				vis[j,*,k,ip]=summe(rg[*,index],1)/count
		endfor
	endelse
endif
endfor
endfor
;
; Do the incoherent integration of the squared visibilities
for j=0,genconfig.numoutbeam-1 do begin
for k=0,genconfig.numbaseline[j]-1 do begin
	vsq[j,*,k,*]=add(abs(vis[j,*,k,*])^2-var[j,*,*],numincoh[i],3,sd=sd) $
		    /(numincoh[i]*numcoh^2)
	vse[j,*,k,*]=(sd/numcoh^2)/sqrt(numincoh[i])
endfor
endfor
;
; Do the coherent integration of the triple products
for j=0,genconfig.numtriple-1 do begin
;
fBaseMatrix=intarr(GenConfig.NumSid,3)
fBaseFactor=fltarr(3)+1
vis3=complexarr(3,genconfig.triplenumchan[j],n_elements(timestamp)/numcoh)
bias=0.
for l=0,2 do begin
        ob=GenConfig.TripleBeam[l,j]
        bl=GenConfig.TripleBase[l,j]
	ch=GenConfig.TripleChan[0:genconfig.triplenumchan[j]-1,l,j]
        j1=where(GenConfig.StationId eq strmid(GenConfig.BaselineId[bl,ob],0,3))
        j2=where(GenConfig.StationId eq strmid(GenConfig.BaselineId[bl,ob],4,3))
	j1=j1[0]
	j2=j2[0]
        fBaseMatrix[j1,l]=+1
        fBaseMatrix[j2,l]=-1
        if l gt 0 then fBaseFactor[l]=(-total(fBaseMatrix[*,0]*fBaseMatrix[*,l]) gt 0)*2-1.0
	vis3[l,*,*]=vis[ob,ch,bl,*]
	if fBaseFactor[l] lt 1 then vis3[l,*,*]=conj(vis3[l,*,*])
	bias=bias+abs(vis[ob,ch,bl,*])^2
endfor
bias=bias-2*nph[ob,ch,*]
tvis=vis3[0,*,*]*vis3[1,*,*]*vis3[2,*,*]
if genconfig.triplebeam[0,j] eq genconfig.triplebeam[1,j] and $
   genconfig.triplebeam[1,j] eq genconfig.triplebeam[2,j] then tvis=tvis-bias
t3v[j,*,*]=add(tvis,numincoh[i],2,sd=sd)/(numincoh[i]*numcoh^3)
; amperr=float(sd)/numcoh^3
; phserr=imaginary(sd)
; t3e(j,*,*)=complex(amperr,phserr)
t3e[j,*,*]=(sd/numcoh^3)/sqrt(numincoh[i])
;
endfor
;
; Write results to file
hds_open,confile,'UPDATE',status & if status ne 0 then return
dat_find,'SCANDATA'
dat_find,'POINTDATA'
dat_cell,1,i+1
;
dat_find,'OUTPUTBEAM'
for j=0,genconfig.numoutbeam-1 do begin
	dat_cell,1,j+1
	name='VISSQ'
	type='_real'
	ndim=3
	dims=[genconfig.numspecchan[j],genconfig.numbaseline[j],np[i]]
	data=reform(vsq[j,*,*,*])
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
	name='VISSQERR'
	data=reform(vse[j,*,*,*])
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
	dat_annul
endfor
dat_annul
;
dat_find,'TRIPLE'
for j=0,genconfig.numtriple-1 do begin
	dat_cell,1,j+1
	name='COMPLTRIPLE'
	ndim=3
	type='_real'
	dims=[2,genconfig.triplenumchan[j],np[i]]
	data=fltarr(dims[0],dims[1],dims[2],/nozero)
	data[0,*,*]=float(t3v[j,*,*])
	data[1,*,*]=imaginary(t3v[j,*,*])
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
	name='COMPLTRIPLEERR'
	data[0,*,*]=float(t3e[j,*,*])
	data[1,*,*]=imaginary(t3e[j,*,*])
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	dat_new,name,type,ndim,dims
	cmp_putnr,name,ndim,dims,data
	dat_annul
endfor
;
hds_close
;
endif
;
; Scan averaged data section
;
; Scintillation index
if strpos(strupcase(strjoin(items)),'SCINTILLATION') ne -1 then begin
;
if n_elements(scintillation) eq 0 then $
scintillation=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan),numscan)
;
for k=0,genconfig.numoutbeam-1 do begin
	hds_open,rawfile,'READ',status & if status ne 0 then return
	get_bincounts,k+1
	hds_close
	for j=0,genconfig.numspecchan[k]-1 do begin
		tc=summe(float(reform(bincounts[*,j,*])),0)
		binsize=1
		fr=histogram(tc,min=0,binsize=binsize)
		x=findgen(n_elements(fr))*binsize+binsize/2
		m=median(long(tc))
		y=eqp(m,long(x))*n_elements(tc)*binsize
		scintillation[k,j,i]=max(y)/max(fr)
	endfor
endfor
endif
;
; Coherence time t0
if strpos(strupcase(strjoin(items)),'T0') ne -1 then begin
if scantable[0].code eq 1 then begin
;
n=[1,2,3,4,6,8,10,15,20,25]
n=[1,2,4,8,15,25]
nn=n_elements(n)
dt=fltarr(nn)
;
if n_elements(t0) eq 0 then $
t0=fltarr(genconfig.numoutbeam,max(genconfig.numspecchan),max(genconfig.numbaseline),numscan)
;
for k=0,genconfig.numoutbeam-1 do begin
	hds_open,rawfile,'READ',status & if status ne 0 then return
	get_bincounts,k+1
	hds_close
;
;	t0=fltarr(genconfig.numspecchan(k),genconfig.numbaseline(k))
	v2=fltarr(nn,genconfig.numspecchan[k],genconfig.numbaseline[k])
;
	for m=0,nn-1 do begin
	dt[m]=n[m]*genconfig.instrcohint
;	c=add(fix(bincounts),n(m),2)
	c=addbincounts(fix(bincounts),n[m])
	p=fringenphot(c)
	z=fringemphot(c)
	num=n_elements(p[0,*])
	for l=0,genconfig.numbaseline[k]-1 do begin
	if onsource(genconfig.baselineid[l,k]) then begin
	norm_factor=4.0/sinc(float(genconfig.fringemod[l,k])/genconfig.numbin)^2
	v=fringevis(c,genconfig.fringemod[l,k])
	v2[m,*,l]=num*norm_factor*summe(abs(v)^2-z^2,1)/summe(p,1)^2
	endif
	endfor
	endfor

	for j=0,genconfig.numspecchan[k]-1 do begin
	for l=0,genconfig.numbaseline[k]*0+1-1 do begin
	if onsource(genconfig.baselineid[l,k]) then begin
	if v2[0,j,l] gt 0 and v2[0,j,l] lt 1 $
			  and v2[nn-1,j,l]/v2[0,j,l] lt 0.5 then begin
		a=[v2[0,j,l],2.0]
		v2fit=curvefit(dt,v2[*,j,l],v2[*,j,l]*0+1,a, $
			function_name='funct_coherence',/noderivat)
		t0[k,j,l,i]=a[1]
	endif
	endif
	endfor
	endfor
endfor
;
endif
endif
;
; Tracking delay RMS over whole scan
if strpos(strupcase(strjoin(items)),'DELAYRMS') ne -1 then begin
if scantable[0].code eq 1 then begin
;
if n_elements(delayrms) eq 0 then $
delayrms=fltarr(genconfig.numoutbeam,max(genconfig.numbaseline),numscan)
;
for k=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline[k]-1 do begin
	delayrms[k,l,i]=stddev(fdldelay(l+1,k+1))*1e6
endfor
endfor
;
endif
endif
;
; Tracking delay jitter (RMS of FDL-smoothed(FDL))
if strpos(strupcase(strjoin(items)),'DELAYJITTER') ne -1 then begin
if scantable[0].code eq 1 then begin
;
if n_elements(delayjitter) eq 0 then $
delayjitter=fltarr(genconfig.numoutbeam,max(genconfig.numbaseline),numscan)
;
for k=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline[k]-1 do begin
	d=fdldelay(l+1,k+1)
	delayjitter[k,l,i]=stddev(d-gsmooth(float(fdlstamp())/1000,d,1,/c8))*1e6
endfor
endfor
;
endif
endif
;
ENDIF
ENDFOR
;
; Update scantable
hds_open,confile,'UPDATE',status & if status ne 0 then return
dat_find,'SCANDATA'
;
if strpos(strupcase(strjoin(items)),'COHERENT') ne -1 then begin
	name='NUMCOH'
	dat_erase,name
	type='_integer'
	ndim=1
	dims=lonarr(7)
	dims[0]=numscan
	dat_new,name,type,ndim,dims
	cmp_put1i,name,lonarr(numscan)+numcoh
;
	name='NUMINCOH'
	dat_erase,name
	type='_integer'
	ndim=1
	dims[0]=numscan
	dat_new,name,type,ndim,dims
	cmp_put1i,name,numincoh
endif
;
if n_elements(natjitter) ne 0 then begin
	dat_find,'INPUTBEAM',status
	if status ne 0 then begin
		ndim=1L
		dims=lonarr(7)
		dims[0]=genconfig.numsid
		type='EXTCOLUMN'
		dat_new,'INPUTBEAM',type,ndim,dims
		dat_find,'INPUTBEAM'
	endif
	ndim=2
	dims=[2,numscan]
	data=fltarr(dims[0],dims[1],/nozero)
;
	for k=0,genconfig.numsid-1 do begin
	dat_cell,1,k+1
;
	name='NATJITTER'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	data[*]=natjitter[k,*,*]
	cmp_putnr,name,ndim,dims,data
	dat_annul
	endfor
	dat_annul
endif
;
if n_elements(scintillation) ne 0 then begin
	dat_find,'OUTPUTBEAM',status
	if status ne 0 then begin
		ndim=1L
		dims=lonarr(7)
		dims[0]=genconfig.numoutbeam
		type='EXTCOLUMN'
		dat_new,'OUTPUTBEAM',type,ndim,dims
		dat_find,'OUTPUTBEAM'
	endif
	ndim=2
	dims=[max(genconfig.numspecchan),numscan]
	data=fltarr(dims[0],dims[1],/nozero)
;
	for k=0,genconfig.numoutbeam-1 do begin
	dat_cell,1,k+1
;
	name='SCINTILLATION'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	data[*]=scintillation[k,*,*]
	cmp_putnr,name,ndim,dims,data
	dat_annul
	endfor
	dat_annul
endif
;
if n_elements(t0) ne 0 then begin
	dat_find,'OUTPUTBEAM',status
	if status ne 0 then begin
		ndim=1L
		dims=lonarr(7)
		dims[0]=genconfig.numoutbeam
		type='EXTCOLUMN'
		dat_new,'OUTPUTBEAM',type,ndim,dims
		dat_find,'OUTPUTBEAM'
	endif
	ndim=3
	dims=[max(genconfig.numspecchan),max(genconfig.numbaseline),numscan]
	data=fltarr(dims[0],dims[1],dims[2],/nozero)
;
	for k=0,genconfig.numoutbeam-1 do begin
	dat_cell,1,k+1
;
	name='T0'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	data[*]=t0[k,*,*,*]
	cmp_putnr,name,ndim,dims,data
	dat_annul
	endfor
	dat_annul
endif
;
if n_elements(delayrms) ne 0 then begin
	dat_find,'OUTPUTBEAM',status
	if status ne 0 then begin
		ndim=1L
		dims=lonarr(7)
		dims[0]=genconfig.numoutbeam
		type='EXTCOLUMN'
		dat_new,'OUTPUTBEAM',type,ndim,dims
		dat_find,'OUTPUTBEAM'
	endif
	ndim=2
	dims=[max(genconfig.numbaseline),numscan]
	data=fltarr(dims[0],dims[1],/nozero)
;
	for k=0,genconfig.numoutbeam-1 do begin
	dat_cell,1,k+1
;
	name='DELAYRMS'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	data[*]=delayrms[k,*,*]
	cmp_putnr,name,ndim,dims,data
	dat_annul
	endfor
	dat_annul
endif
;
if n_elements(delayjitter) ne 0 then begin
	dat_find,'OUTPUTBEAM',status
	if status ne 0 then begin
		ndim=1L
		dims=lonarr(7)
		dims[0]=genconfig.numoutbeam
		type='EXTCOLUMN'
		dat_new,'OUTPUTBEAM',type,ndim,dims
		dat_find,'OUTPUTBEAM'
	endif
	ndim=2
	dims=[max(genconfig.numbaseline),numscan]
	data=fltarr(dims[0],dims[1],/nozero)
;
	for k=0,genconfig.numoutbeam-1 do begin
	dat_cell,1,k+1
;
	name='DELAYJITTER'
	dat_there,name,reply & if reply eq 1 then dat_erase,name
	type='_real'
	dat_new,name,type,ndim,dims
	data[*]=delayjitter[k,*,*]
	cmp_putnr,name,ndim,dims,data
	dat_annul
	endfor
	dat_annul
endif
;
hds_close
;
end
;-------------------------------------------------------------------------------
pro solve,vis,lambda,soln,pt=pt,plot=plot
;
; lambda [m]
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
forward_function fringeid
;
if n_elements(pt) eq 0 then pt=0
if n_elements(plot) eq 0 then plot=0
;
phase=phaseunwrap(cphase(vis))/(2*!pi)
;
sigma=1.d0/(lambda*1.d6)
nrow=n_elements(sigma)+1
nump=n_elements(phase[0,*])
p=dblarr(nrow,nump)
p[0:nrow-2,*]=phase
;
a=a_disp[0]+a_disp[1]/(a_disp[2]-sigma^2)+a_disp[3]/(a_disp[4]-sigma^2)
a=a/10000.d0  ; We want coefficients of similar magnitude
b=b_disp[0]+b_disp[1]*sigma^2
;
; Standard solution
m=dblarr(n_elements(sigma),3)
m[*,0]=sigma
m[*,1]=sigma*a
m[*,2]=1
tm=transpose(m)
norm=tm#m
r=tm#phase
s=invert(norm)#r
soln=s
fidsol=s[2,*]
;
; Plot the fits
if plot then begin
	n=20
	x=findgen(n)/10
	m=dblarr(n_elements(x),3)
	m[*,0]=x
	m[*,1]=x $
	      *(a_disp[0]+a_disp[1]/(a_disp[2]-x^2) $
		+a_disp[3]/(a_disp[4]-x^2)) $
	      /10000.d0
	m[*,2]=1
	y=reform(m#s)
	r=size(y)
	if r[0] eq 2 then plot_slide,sigma,phase,y2=y,x2=x $
		     else begin
			  wset,0
			  plot,sigma,phase,psym=1,xrange=[0,2],yrange=[-1,1]
			  oplot,x,y,psym=0
			  endelse
endif
;
; Forced integer ID solution
s[2,*]=fringeid(s[2,*])
new_offset=min(s[2,*] mod 1)
if pt then s[2,*]=medianve(s[2,*])
m=dblarr(nrow,3)
m[0:nrow-2,0]=sigma
m[0:nrow-2,1]=sigma*a
m[0:nrow-2,2]=1
m[  nrow-1,2]=1
tm=transpose(m)
norm=tm#m
p[nrow-1,*]=s[2,*]
r=tm#p
s=invert(norm)#r
soln=s
soln[2,*]=fidsol
return
;
; Forced integer ID solutions
num_id=9
s=dblarr(num_id,3,nump)
rmsq=fltarr(num_id,nump)
for i=-num_id/2,num_id/2 do begin
	tm=transpose(m)
	norm=tm#m
	p[nrow-1,*]=i+new_offset
	r=tm#p
	s[i+num_id/2,*,*]=invert(norm)#r
	rmsq[i+num_id/2,*]=total((m#reform(s[i+num_id/2,*,*])-p)^2,1)
endfor
for i=0,nump-1 do begin
	x=min(rmsq[*,i],j)
	soln[*,i]=s[j,*,i]
endfor
;
; Contents of solution vector:
; vacuum delay, air path, fringe ID
; add vac to FDL delay 
;
end
;************************************************************************Block 5
pro plot_fdldelay,baseline,beam,slide=slide,scan=scan
;
; Compute and plot the groupdelay for the given baseline.
; Must have read raw data first.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_FDLDELAY): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_FDLDELAY): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
if n_elements(slide) eq 0 then slide=0
if n_elements(scan) eq 0 then scan=1
;
d=fdldelay(baseline,beam,scan=scan)
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
set_screen
if slide then window_slide,xsize=20000 else window,/free
plot,t,d*1e6,psym=3,charsize=2, $
	title=startable[0].starid $
	     +' ('+string(scantable[0].scanid,format='(i0)')+') ' $
	     +genconfig.baselineid[baseline-1,beam-1] $
	     +' ('+string(beam,format='(i1)')+') '+date, $
	ytitle='FDL delay [microns]', $
	xtitle='Time - '+hms(t0/3600)+' [s]'
;
; Recover from window_slide settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_fdlpower,baseline,beam,sf=sf
;
; timestamp [ms], no fractional values allowed!
; delay [m]
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if not keyword_set(sf) then sf=0 else sf=sf gt 0
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_FDLPOWER): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_FDLPOWER): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
;
d=fdldelay(baseline,beam)
; If the actual fringe position is needed, i.e. free from tracking errors
; g=fringedelays(baseline,beam) & d=d-g
t=fdlstamp()
;
series=powerpatch(t,d,limit=100000)*1d6
sampling_int=0.002
;
window,/free & set_screen
;
IF sf THEN BEGIN
;
; !y.title='Delay structure function [mu^2]'
strucfunc,series,sampling_int,intervals,values,accel=n_elements(series)/400
;
x=alog10(intervals)
y=alog10(values)
index=where(x gt -1.0 and x lt 0.0)
r=poly_fit(x[index],y[index],1)
t0=(10^(-(r[0]-alog10(1.44)-alog10(0.5^2))/(5.0/3.0))/7.5)*1000
t0=(10^(-(r[0]-alog10((0.5/(2*!pi))^2))/(5.0/3.0)))*1000
slope=r[1]
;
ENDIF ELSE BEGIN
;
series=poweroftwo(series)
print,'Length of series [s]: ',n_elements(series)*sampling_int
;
; !y.title='PSD [mu^2/Hz]'
tp=powerspectrum(series,sampling_int,f,p,f_avg,p_avg)
;
!x.ticks=0
!y.ticks=0
!p.charsize=1.5
;
; plot,f,alog10(p),xrange=[f(1),1/(2*sampling_int)],xstyle=1,ystyle=1,/xlog, $
;     xtitle='Frequency [Hz]',psym=3
;
plot,f_avg,alog10(p_avg),xrange=[f[1],1/(2*sampling_int)],xstyle=1,/xlog, $
     xtitle='Frequency [Hz]',psym=0, $
     title=!p.title, $
     xgridstyle=1,xticklen=1,ygridstyle=1,yticklen=1
;
x=alog10(f_avg)
y=alog10(p_avg)
index=where(x gt -1.0 and x lt 1.0)
r=poly_fit(x[index],y[index],1,yfit)
t0=10^(-(r[0]-alog10(5.68e-4)-alog10(0.5^2))/(5.0/3.0))*1000
slope=r[1]
xyouts,0.4,0.85,/normal,'t0 = '+string(t0,format='(f4.1)')+' [ms]' $
	              +', slope = '+string(slope,format='(f5.2)'), $
		charsize=2,charthick=2
;
ENDELSE
;
; Recover from plot settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_natpower,beam
;
; Plot power spectrum of NAT total quad counts for specified input beam.
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(beam) eq 0 then begin
	print,'***Error(PLOT_NATPOWER): you must specify beam ID!'
	return
endif
ib=beam-1
;
c=reform(total(quadcounts[ib,*,*],2))
t=fdlstamp()
n=avg(c)
c=c-n
;ts=(double(t)-t(0))/1000
;index=where(ts lt 30)
;c=c(index)
;t=t(index)
;
series=powerpatch(t,c,limit=100000)
sampling_int=0.002
;
window,/free & set_screen
;
series=poweroftwo(series)
print,'Length of series [s]: ',n_elements(series)*sampling_int
;
!y.title='Power'
tp=powerspectrum(series,sampling_int,f,p,f_avg,p_avg)
;
!x.ticks=0
!y.ticks=0
!p.charsize=1.5
;
plot,f_avg,alog10(p_avg),xrange=[f[1],1/(2*sampling_int)],xstyle=1,/xlog, $
     xtitle='Frequency [Hz]',psym=0, $
     title=!p.title, $
     xgridstyle=1,xticklen=1,ygridstyle=1,yticklen=1
;
x=alog10(f_avg)
y=alog10(p_avg)
index=where(x gt -1.0 and x lt 1.0)
r=poly_fit(x[index],y[index],1,yfit)
t0=10^(-(r[0]-alog10(5.68e-4)-alog10(0.5^2))/(5.0/3.0))*1000
slope=r[1]
xyouts,0.4,0.85,/normal,'<N> = '+string(long(n),format='(i4)') $
	              +', stddev = '+string(long(tp),format='(i4)'), $
		charsize=2,charthick=2
;
; Recover from plot settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_vispower,baseline,beam,numav=numav
;
; Compute and plot fringe power spectra.
; Must have read raw data first.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_FRINGEPOWER): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_FRINGEPOWER): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
;
if n_elements(numav) eq 0 then numav=1
;
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
get_bincounts,beam
v=fringevis(bincounts[*,ch,*],genconfig.fringemod[bl,ob])
fp=box(abs(visdft(v,l,d)),numav,1)
g=groupdelay(fp,l,d,gindex)
;
set_screen
!x.title='Delay [microns]'
plot_slide,d*1e6,fp,string(gindex)
;
; Recover from plot_slide settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_fringepower,beam,channels=channels,numav=numav
;
; Plot fringe power as a function of modulation k. 
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_FRINGEPOWER): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
get_bincounts,beam
;
if n_elements(channels) eq 0 then channels=1
if n_elements(numav) eq 0 then numav=10
;
ch=channels-1
c=reform(bincounts[*,ch,*])
;
r=size(c)
kmin=0.75
case r[1] of
	8:kmax=4+0.75
       64:kmax=10+0.75
     else:begin
	  print,'Error: unknown frame format!'
	  return
	  end
endcase
if r[0] eq 3 then nc=r[2] else begin
	nc=1
	c=reform(c,r[1],1,r[2])
	endelse
n=nint((kmax-kmin)/0.25)
x=findgen(n+1)/n*(kmax-kmin)+kmin
y=x
vdata=fltarr(n+1,n_elements(c[0,0,*])/numav)
for i=0,n do begin
	vdata[i,*]=fringevissq(c[*,0,*],x[i],numav,normal=1)
	y[i]=avg(vdata[i,*])
endfor
;
if !d.name ne 'PS' then begin window,/free & set_screen & endif
!p.charsize=1.5
!x.title='k' $
	+' (<N0> = '+string(long(avg(fringenphot(c[*,0,*]))),format='(i4)')+')'
!y.title='<X^2+Y^2-N>/<N>^2'
!y.title='<X^2+Y^2-N>/<N>^2, '+string(r[r[0]],format='(i5)') $
     +' samples, '+string(numav,format='(i3)')+' samp./exp.'
;
plot,x,y,psym=-1,xrange=[kmin,kmax],xstyle=1, $
	title=startable[0].starid $
	     +' ('+string(scantable[0].scanid,format='(i0)')+') ' $
	     +' ('+string(outputbeam,format='(i1)')+') '+date
;
for j=1,nc-1 do begin
for i=0,n do y[i]=avg(fringevissq(c[*,j,*],x[i],numav,normal=1))
oplot,x,y,psym=-1,color=getcolor(j+1)
endfor
;
; Recover from plot_slide settings
if !d.name eq 'PS' then device,/close
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_fringepower2,c,numav=numav
;
; Plot fringe power as a function of modulation k. 
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(numav) eq 0 then numav=10
;
r=size(c)
kmin=0.75
case r[1] of
	8:kmax=4+0.75
       64:kmax=10+0.75
     else:begin
	  print,'Error: unknown frame format!'
	  return
	  end
endcase
if r[0] eq 3 then nc=r[2] else begin
	nc=1
	c=reform(c,r[1],1,r[2])
	endelse
n=nint((kmax-kmin)/0.25)
x=findgen(n+1)/n*(kmax-kmin)+kmin
y=x
vdata=fltarr(n+1,n_elements(c[0,0,*])/numav)
for i=0,n do begin
	vdata[i,*]=fringevissq(c[*,0,*],x[i],numav,normal=1)
	y[i]=avg(vdata[i,*])
endfor
;
if !d.name ne 'PS' then begin window,/free & set_screen & endif
!p.charsize=1.5
!x.title='k' $
	+' (<N0> = '+string(long(avg(fringenphot(c[*,0,*]))),format='(i4)')+')'
!y.title='<X^2+Y^2-N>/<N>^2'
!y.title='<X^2+Y^2-N>/<N>^2, '+string(r[r[0]],format='(i5)') $
     +' samples, '+string(numav,format='(i3)')+' samp./exp.'
;
plot,x,y,psym=-1,xrange=[kmin,kmax],xstyle=1
;
for j=1,nc-1 do begin
for i=0,n do y[i]=avg(fringevissq(c[*,j,*],x[i],numav,normal=1))
oplot,x,y,psym=-1,color=getcolor(j+1)
endfor
;
; Recover from plot_slide settings
if !d.name eq 'PS' then device,/close
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_triplepower,c,numav=numav
;
; Plot fringe power as a function of modulation k. 
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(numav) eq 0 then numav=10
;
r=size(c)
kmin=0.75
case r[1] of
	8:kmax=4+0.75
       64:kmax=10+0.75
     else:begin
	  print,'Error: unknown frame format!'
	  return
	  end
endcase
if r[0] eq 3 then nc=r[2] else begin
	nc=1
	c=reform(c,r[1],1,r[2])
	endelse
n=nint((kmax-kmin)/0.25)
x=findgen(n+1)/n*(kmax-kmin)+kmin
y=x
vdata=fltarr(n+1,n_elements(c[0,0,*])/numav)
for i=0,n do begin
	vdata[i,*]=abs(fringetriple(c[*,0,*],1,x[i],numav))
	y[i]=avg(vdata[i,*])
endfor
;
if !d.name ne 'PS' then begin window,/free & set_screen & endif
!p.charsize=1.5
!x.title='k' $
	+' (<N0> = '+string(long(avg(fringenphot(c[*,0,*]))),format='(i4)')+')'
!y.title='<X^2+Y^2-N>/<N>^2'
!y.title='<X^2+Y^2-N>/<N>^2, '+string(r[r[0]],format='(i5)') $
     +' samples, '+string(numav,format='(i3)')+' samp./exp.'
;
plot,x,y,psym=-1,xrange=[kmin,kmax],xstyle=1
;
for j=1,nc-1 do begin
for i=0,n do y[i]=avg(fringevissq(c[*,j,*],x[i],numav,normal=1))
oplot,x,y,psym=-1,color=getcolor(j+1)
endfor
;
; Recover from plot_slide settings
if !d.name eq 'PS' then device,/close
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_fringepowers,beam,channels=channels,numav=numav,stars=stars
;
; Plot fringe power as a function of modulation k. 
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(stars) eq 0 then stars=''
;
files=file_search('????-??-??.raw.???.??')
first=1
for nf=0,n_elements(files)-1 do begin
;
hds_open,files[nf]
get_rawdata
;
index=where(stars eq scantable[0].starid,count)
if strlen(stars[0]) eq 0 then count=1
if scantable[0].code eq 1 and count gt 0 then begin
if n_elements(starids) eq 0 then starids=scantable.starid $
else begin
	index=where(starids eq scantable[0].starid,count)
	if count eq 0 then starids=[starids,scantable[0].starid]
endelse
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_FRINGEPOWER): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
get_bincounts,beam
hds_close
;
if n_elements(channels) eq 0 then channels=1
if n_elements(numav) eq 0 then numav=10
;
ch=channels-1
c=reform(bincounts[*,ch,*])
;
r=size(c)
kmin=0.75
case r[1] of
	8:kmax=4+0.75
       64:kmax=10+0.75
     else:begin
	  print,'Error: unknown frame format!'
	  return
	  end
endcase
if r[0] eq 3 then nc=r[2] else begin
	nc=1
	c=reform(c,r[1],1,r[2])
	endelse
n=nint((kmax-kmin)/0.25)
x=findgen(n+1)/n*(kmax-kmin)+kmin
y=x
vdata=fltarr(n+1,n_elements(c[0,0,*])/numav)
for i=0,n do begin
	vdata[i,*]=fringevissq(c[*,0,*],x[i],numav,normal=1)
	y[i]=avg(vdata[i,*])
endfor
;
!p.charsize=1.5
!x.title='k'
!y.title='<X^2+Y^2-N>/<N>^2'
;
!p.color=tci(where(starids eq scantable[0].starid)+1)
if first then $
plot,x,y,psym=0,xrange=[kmin,kmax],xstyle=1, $
	title=date+', beam='+string(outputbeam,format='(i1)') $
else oplot,x,y,psym=0
first=0
xyouts,100,!d.y_size-50-!p.color*(!d.y_ch_size+2),scantable[0].starid,/device
;
for j=1,nc-1 do begin
for i=0,n do y[i]=avg(fringevissq(c[*,j,*],x[i],numav,normal=1))
oplot,x,y,psym=-1,color=getcolor(j+1)
endfor
endif else hds_close
endfor
;
; Recover from plot_slide settings
if !d.name eq 'PS' then device,/close
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_fringeimage,channels=channels
;
; Display image of fringes (bincounts) as a function of time and either
; channel (wavelength) or delay (bin number). The first option is chosen
; if more than one channel is specified, the latter if only one channel
; has been specified. If no channels are undefined, produce a color image
; of three single channels.
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(bincounts) eq 0 then begin
	print,'***Error(PLOT_FRINGEIMAGE): bincounts undefined!'
	return
endif
;
if n_elements(channels) eq 0 then color=1 else color=0
if n_elements(channels) gt 1 then channeled=1 else channeled=0
;
; Define color table
if color then begin
	m=6
	n=m*m*m
	k=256/m
	r=(bindgen(n) mod m)*k+k
	g=((bindgen(n)/m) mod m)*k+k
	b=((bindgen(n)/(m*m)) mod m)*k+k
endif else begin
	r=indgen(255)
	g=indgen(255)
	b=indgen(255)
endelse
;
bc=float(reform(bincounts[*,*,0:10000]))
r=size(bc)
numbin=r[1]
if color then begin
	ch1=2
	ch2=7
	ch3=12
	b1=fix((bc[*,ch1,*]/max(bc[*,ch1,*])*255)-(k/2))/k
	b2=fix((bc[*,ch2,*]/max(bc[*,ch2,*])*255)-(k/2))/k
	b3=fix((bc[*,ch3,*]/max(bc[*,ch3,*])*255)-(k/2))/k
	image=reform(b1+b2*m+b3*m*m)
endif else if channeled then begin
	ch=channels-1
;	image=reform(bc(0,ch,*)/max(bc(0,ch,*))*255)
	image=reform(bc[0,ch,*])
	for j=0,n_elements(ch)-1 do $
		image[j,*]=(image[j,*]/max(bc[*,ch[j],*]))*255*2
endif else begin
	ch=channels-1
	image=reform(bc[*,ch,*]/max(bc[*,ch,*])*255*2)
	image[where(image gt 255)]=255
endelse
;
xsize=long(800)
if channeled then yoffs=n_elements(channels)+4  else yoffs=numbin+4
ysize=n_elements(image[0,*])*yoffs/xsize
window,xsize=xsize,ysize=ysize,colors=256,/free
tvlct,r,g,b
;
for i=0L,n_elements(image[0,*])/xsize-1 do begin
	tv,transpose(image[*,i*xsize:(i+1)*xsize-1]),0,i*yoffs
endfor
;
end
;-------------------------------------------------------------------------------
pro plot_powerhist,baseline,beam,scanfile,numav=numav
;
; Compute and plot histograms of primary and secondary peak heights
; in the fringe powerspectra.
; Must have read raw data first.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_POWERHIST): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_POWERHIST): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
if n_elements(numav) eq 0 then numav=1
;
g=groupdelays(baseline,beam,scanfile,numav=numav,plot=1)
;
; Recover from plot settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_powerpeaks,baseline,beam,scanfile,numav=numav
;
; Compute and plot scatter plots of primary and secondary peak heights
; in the fringe powerspectra.
; Must have read raw data first.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_POWERPEAKS): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_POWERPEAKS): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
if n_elements(numav) eq 0 then numav=1
;
g=groupdelays(baseline,beam,scanfile,numav=numav,plot=2)
;
; Recover from plot settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_spectrumpeaks,baseline,beam,channels=channels,numav=numav
;
; Compute fringe spectra of the squared visibility amplitude
; averaged incoherently over numav samples as a function of
; fringe frequency. Make a scatter plot of the peak amplitude
; at the k value corresponding to the selected baseline, versus
; the maximum peak amplitude over all the other fringe frequencies 
; which are not supposed to contain any signal.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_SPECTRUMPEAKS): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_SPECTRUMPEAKS): you must specify beam ID!'
		return
	endif else beam=OutputBeam
endif else get_bincounts,beam
ob=beam-1
bl=baseline-1
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[ob]]
ch=channels-1
if n_elements(numav) eq 0 then numav=10
;
kmin=0.75
case genconfig.numbin of
	8:kmax=4+0.75
       64:kmax=8+0.75
     else:begin
	  print,'Error: unknown frame format!'
	  return
	  end
endcase
nc=n_elements(channels)
n=nint((kmax-kmin)/0.25)
x=findgen(n+1)/n*(kmax-kmin)+kmin
y=x
;
for j=0,nc-1 do begin
;
window,xsize=600,ysize=fix(600*(0.27/0.25)),/free & set_screen
!p.charsize=2.0
!p.title=genconfig.baselineid[bl,ob]+' ('+string(beam,format='(i1)')+') ' $
	+startable.starid+' '+date
!p.subtitle='<X^2+Y^2-N>/<N>^2, ' $
	+string(n_elements(timestamp)/numav,format='(i0)')+' exp, ' $
	+string(numav,format='(i0)')+' samples/exp.'
!x.title='Primary peak amplitude at k='+string(genconfig.fringemod[bl,ob], $
					format='(i0)') $
	+', ch='+string(channels[j],format='(i0)')
!y.title='Maximum secondary peak amplitude'
!x.margin=[10,5]
!y.margin=[6,3]
;
pps=fringevissq(bincounts[*,j,*],genconfig.fringemod[bl,ob],numav,normal=1)
for k=1,fix(kmax) do begin
	index=where(genconfig.fringemod[*,ob] eq k,count)
	if count eq 0 then begin
		kps=fringevissq(bincounts[*,j,*],k,numav,normal=1)
		if n_elements(sps) eq 0 then sps=kps $
					else sps=kps > sps
	endif
endfor
plot,pps,sps,xrange=[0,max(pps)],yrange=[0,max(pps)],psym=3
;
endfor
;
; Recover standard plot settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_groupdelay,baseline,beam,scanfile,numav=numav,classic=classic, $
	slide=slide
;
; Compute and plot the groupdelay for the given baseline.
; Must have read raw data first.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_GROUPDELAY): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_GROUPDELAY): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
if n_elements(numav) eq 0 then numav=1
if n_elements(classic) eq 0 then classic=1
if n_elements(slide) eq 0 then slide=0
;
g=groupdelays(baseline,beam,scanfile,numav=numav,gindex=gindex) & g=g*1e6
t=fdlstamp(scan=n_elements(scanfile))/1000. & t0=t[0] & t=t-t[0]
;
set_screen
if slide then window_slide,xsize=20000 else window,/free
plot,t,g,psym=3,charsize=2, $
	title=startable[0].starid $
	     +' ('+string(scantable[0].scanid,format='(i0)')+') ' $
	     +genconfig.baselineid[baseline-1,beam-1] $
	     +' ('+string(beam,format='(i1)')+') '+date, $
	ytitle='Group delay [microns]', $
	xtitle='Time - '+hms(t0/3600)+' [s]'
oplot,t[where(gindex eq 0)],g[where(gindex eq 0)],psym=3,color=tci(2)
;
if classic eq 0 then begin
	if numav ne 1 then $
		g=groupdelays(baseline,beam,scanfile,gindex=gindex)*1e6
	s=gsmooth(t[where(gindex eq 1)],g[where(gindex eq 1)],0.01,t)
	oplot,t,s,psym=3,color=tci(3)
endif
;
; Recover from window_slide settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_fringedelay,baseline,beam,scanfile,numav=numav,classic=classic, $
	slide=slide
;
; Compute and plot the fringedelay for the given baseline.
; Must have read raw data first.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_GROUPDELAY): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_GROUPDELAY): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
if n_elements(numav) eq 0 then numav=1
if n_elements(classic) eq 0 then classic=1
if n_elements(slide) eq 0 then slide=0
if not classic then gindex=1
;
if classic then g=fringedelays(baseline,beam,scanfile) $
	   else g=fringedelays(baseline,beam,scanfile,gindex=gindex) 
g=g*1e6
t=fdlstamp(scan=n_elements(scanfile))/1000. & t0=t[0] & t=t-t[0]
if classic then gindex=lonarr(n_elements(t))+1
;
set_screen
if slide then window_slide,xsize=20000 else window,/free
plot,t,g,psym=3,charsize=2, $
	title=startable[0].starid $
	     +' ('+string(scantable[0].scanid,format='(i0)')+') ' $
	     +genconfig.baselineid[baseline-1,beam-1] $
	     +' ('+string(beam,format='(i1)')+') '+date, $
	ytitle='Fringe delay [microns]', $
	xtitle='Time - '+hms(t0/3600)+' [s]'
;
gd=groupdelays(baseline,beam,scanfile,numav=1,gindex=gindex)
oplot,t[where(gindex eq 0)],g[where(gindex eq 0)],psym=3,color=tci(0)
;
; Recover from window_slide settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_fringephase,baseline,beam,scanfile,classic=classic
;
; Compute and plot the corrected visibility phases.
; Must have read raw data first.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_FRINGEPHASES): you must specify baseline ID!'
	return
endif
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_FRINGREPHASES): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
endif
if n_elements(classic) eq 0 then classic=1
if not classic then gindex=1
;
if classic then p=fringephases(baseline,beam,scanfile) $
	   else p=fringephases(baseline,beam,scanfile,gindex=gindex)
t=fdlstamp(scan=n_elements(scanfile))/1000.& t0=t[0] & t=t-t0
if classic then gindex=lonarr(n_elements(t))+1
;
window,/free
set_screen
!p.charsize=1.5
!p.multi=[0,1,2]
!y.title='Residual phase [radians]'
!x.title='Time - '+hms(t0/3600)+' [s]'
!y.range=[-3,3]
!x.style=1
!y.style=1
num=n_elements(t)/20000L
if num eq 0 then num=1
index=lindgen(n_elements(t)/num)*num
plot,t[index],p[index],psym=3
index=where(gindex gt 0)
plot,findgen(n_elements(index))*0.002,p[index],psym=3, $
	title=startable[0].starid $
	     +' ('+string(scantable[0].scanid,format='(i0)')+') ' $
	     +genconfig.baselineid[baseline-1,beam-1] $
	     +' ('+string(beam,format='(i1)')+') '+date, $
	xtitle='Number of seconds'
print,'RMS of phase: ',stddev(p)
;
; Recover standard plot settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro plot_dispsol,baseline,beam,scanfile, $
	numav=numav,compute=compute,classic=classic
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(numav) eq 0 then numav=100
if n_elements(compute) eq 0 then compute=1
if n_elements(classic) eq 0 then classic=1
;
if n_elements(scanfile) ne 0 then $
savefile=scanfile+'.'+string(baseline,format='(i1)') $
                 +'.'+string(beam,format='(i1)')+'.xdr'
;
IF compute THEN BEGIN
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_DISPSOL): you must specify baseline ID!'
	return
endif
if n_elements(scanfile) ne 0 then begin
	hds_close
	get_rawdata,scanfile
	scan=1
endif else scan=0
if n_elements(beam) eq 0 then begin
	if n_elements(OutputBeam) eq 0 then begin
		print,'***Error(PLOT_DISPSOL): you must specify beam ID!'
		return
	endif
	beam=OutputBeam
	do_read=0
endif else do_read=1
if n_elements(scanfile) ne 0 then begin
	v=coherentvis(baseline,beam,scanfile,classic=classic,delay=d)
endif else begin
	if do_read then v=coherentvis(baseline,beam,classic=classic,delay=d) $
		   else v=coherentvis(baseline,classic=classic,delay=d)
endelse
f=add(fdldelay(genconfig.baselineid[baseline-1,beam-1],scan=scan),numav)*1d4
t=add(double(fdlstamp(scan=scan))/1000,numav)/numav & t0=t[0] & t=t-t[0]
d=d*1e6
if n_elements(channels) eq 0 then $
	channels=channelindex[genconfig.spectrometerid[beam-1]]
l=genconfig.wavelength[channels-1,beam-1]
solve,v,l,s,/pt,/plot
if n_elements(scanfile) ne 0 then begin
	save,t,f,s,filename=savefile
	print,'Saved file: '+savefile
endif
ENDIF ELSE restore,savefile
;
set_screen
!p.charsize=1.5
!p.multi=[0,2,2]
!x.title='Time - '+hms(t0/3600)+' [s]'
!p.psym=3
!y.range=0
!x.range=0
;
if n_elements(t) lt 50000l then window,/free $
	else if n_elements(t) lt 100000l then window_slide,xsize=1000 $
	else if n_elements(t) lt 1000000l then window_slide,xsize=5000 $
	else window,/free,xsize=1000
if scantable[0].starid eq 'FKV0000' then sf=10 else sf=1
plot,t,f,ytitle='FDL residual [microns]', $
	title=date
plot,t,polyres(t,f-d,1),ytitle='GRP residual [microns]', $
	yrange=!y.crange,ystyle=1, $
	title=scantable[0].starid+' ('+string(scantable[0].scanid,format='(i0)')+')'
plot,t,polyres(t,f-d+s[0,*],1),ytitle='Disp. corr. delay', $
	yrange=!y.crange*sf,ystyle=1, $
	title=genconfig.baselineid[baseline-1,beam-1] $
	     +' ('+string(beam,format='(i1)')+')'
plot,t,s[2,*],ytitle='Fringe ID',yrange=[-2,2]
!p.psym=1
;
; Recover standard plot settings
set_screen
!x.range=0
!y.range=0
;
end
;-------------------------------------------------------------------------------
function corrloss,tau
;
; Return the correlation loss factor (Davis & Tango 1996) for
; tau=dt/t0, t0 being the coherence time and dt the sampling time.
;
index=where(tau eq 0,count)
if count gt 0 then tau[index]=1
t=double(tau)
x=t^(5.d0/3.d0)
a1=3.d0/5.d0
a2=6.d0/5.d0
r=tau
for i=0,n_elements(tau)-1 do $
	r[i]=(a2/t[i])*(gamma(a1)*igamma(a1,x[i]) $
                       -gamma(a2)*igamma(a2,x[i])/t[i])
if count gt 0 then r[index]=1
return,r
;
end
;-------------------------------------------------------------------------------
pro funct_coherence,x,a,f
;
a=abs(a)
f=a[0]*corrloss(x/a[1])
;
end
;-------------------------------------------------------------------------------
pro plot_coherence,baseline,beam,channels,parms,untrack=untrack
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(untrack) eq 0 then untrack=0
;
if n_elements(baseline) eq 0 then begin
	print,'***Error(PLOT_COHERENCE): you must specify baseline ID!'
	return
endif
bl=baseline-1
if n_elements(beam) ne 0 then get_bincounts,beam
ob=OutputBeam-1
;
d=fdldelay(baseline,beam,scan=0)
;
if n_elements(channels) eq 0 then ch=indgen(GenConfig.NumSpecChan[ob])  $
			     else ch=channels-1
l=genconfig.wavelength[ch]
nch=n_elements(ch)
if nch gt 4 then begin
	!p.charsize=1.5 
	mf=2
endif else begin
	!p.charsize=1.0
	mf=1
endelse
;
; Set number of plots
ny=nint(sqrt(nch))
nx=nint((float(nch)/ny)+0.4999)
!p.multi=[0,nx,ny]
!p.color=tci(1)
!x.title='t [ms]'
!y.title='Squared visibility'
;
k=[1,2,3,4,6,8,10,15,20,25]
k=[1,2,4,8,15,25]
num=n_elements(k)
dt=fltarr(num)
v2=fltarr(num)
parms=fltarr(3,nch)
a=[0.4,4.0]
;
norm_factor=4.0/sinc(float(genconfig.fringemod[bl,ob])/genconfig.numbin)^2
for j=0,nch-1 do begin
for i=1,num do begin
	dt[i-1]=k[i-1]*genconfig.instrcohint
;v=add(visrotate(fringevis(bincounts(ob,*,ch(j),*)),l(j),-d),k(i-1))
;n=add(fringenphot(bincounts(ob,*,ch(j),*)),k(i-1))
;m=add(fringemphot(bincounts(ob,*,ch(j),*)),k(i-1))
	if untrack then c=rotateframe(bincounts[*,ch[j],*],l[ch[j]],d) $
		   else c=bincounts[*,ch[j],*]
;	c=add(fix(reform(c)),k(i-1),1)
	c=addbincounts(fix(reform(c)),k[i-1])
	v=fringevis(c,genconfig.fringemod[bl,ob])
	n=fringenphot(c)
	m=fringemphot(c)
	v2[i-1]=norm_factor*avg(abs(v)^2-m^2)/avg(n)^2
endfor
plot,dt,v2,psym=1,/xlog,/ylog, $
	title='ob '+string(OutputBeam,format='(i1)') $
	     +', ch '+string(ch[j]+1,format='(i2)')
if v2[0] gt 0 then begin
	a[0]=v2[0]
	v2fit=curvefit(dt,v2,v2*0+1,a, $
		function_name='funct_coherence',/noderivat)
	b=[a,v2[0]/a[0]]
	parms[*,j]=b
	x=dindgen(100)+1
	oplot,x,b[0]*corrloss(x/b[1]),psym=0
	xyouts,10^(!x.crange[0]+0.4),10^(!y.crange[0]+0.6), $
		't0 = '+string(b[1],format='(f4.1)'),charsize=!p.charsize/mf
	xyouts,10^(!x.crange[0]+0.4),10^(!y.crange[0]+0.5), $
		'C0 = '+string(b[0],format='(f4.2)'),charsize=!p.charsize/mf
	xyouts,10^(!x.crange[0]+0.4),10^(!y.crange[0]+0.4), $
		'Loss = '+string(b[2],format='(f4.2)'),charsize=!p.charsize/mf
endif
endfor
;
end
;-------------------------------------------------------------------------------
pro plot_ratehist,channels
;
; Plot histograms of the spectrometer photon count rate for the 
; selected channels.
;
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
ob=OutputBeam-1
;
if n_elements(channels) eq 0 then ch=indgen(GenConfig.NumSpecChan[ob])  $
			     else ch=channels-1
;
; Set number of plots
nch=n_elements(ch)
ny=nint(sqrt(nch))
nx=nint((float(nch)/ny)+0.4999)
!p.multi=[0,nx,ny]
!p.color=tci(1)
title='Rate histogram'
;
for j=0L,nch-1 do begin
	bc=float(reform(bincounts[*,ch[j],*]))
	tc=summe(bc,0)
;	tc=randomu(seed,n_elements(tc),poisson=median(long(tc)))
	binsize=1
	fr=histogram(tc,min=0,binsize=binsize)
	x=findgen(n_elements(fr))*binsize+binsize/2
;
	xtitle='Beam='+string(OutputBeam,format='(i1)')+ $
     	     ', channel='+string(ch[j]+1,format='(i2)')
	ytitle=' '
	plot,x,fr,psym=10,title=title,xtitle=xtitle,ytitle=ytitle
	oploterr,x,fr,sqrt(fr),3
;
	n=n_elements(fr)
;	x=dindgen(n)
	m=median(long(tc))
;	y=eqp(m,lindgen(n))*n_elements(tc)
	y=eqp(m,long(x))*n_elements(tc)*binsize
	oplot,x,y,psym=0,color=tci(5)
endfor
;
end
;-------------------------------------------------------------------------------
pro funct_nathist,x,a,f
;
f=a[0]*exp(-(x-a[1])^2/(a[2]/2)^2)
;
end
;-------------------------------------------------------------------------------
pro plot_nathist,sid
;
; Plot histograms of the X and Y offsets derived from the NAT quad counts.
;
common StarBase,StarTable,Notes
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(sid) eq 0 then sid=indgen(genconfig.numsid)+1
if n_elements(sid) gt 2 then mf=2 else mf=1
;
!p.multi=[0,n_elements(sid),2]
!p.multi[4]=1
!p.charsize=1.5
;
binsize=0.01
numbin=nint(2.0/binsize)+1
x=(findgen(numbin)-float(numbin-2)/2)*binsize
;
for i=0,n_elements(sid)-1 do begin
;
	xy=natoffset(quadcounts[sid[i]-1,*,*])
;
	y=histogram(xy[*,0],min=-1,binsize=0.01,max=1)
	plot,x,y,psym=10,ystyle=0,xrange=[-1,1], $
		xtitle='NAT normalized X offset',ytitle='N', $
		title=genconfig.stationid[sid[i]-1]+' ' $
	     	     +startable[0].starid+' ' $
	     	     +date+' ' $
	     	     +string(avg(timestamp)/3600000.,format='(f5.2)')+' UT'
	a=[max(y),0,0.5]
	nfit=curvefit(x,y,y*0+1,a,function_name='funct_nathist',/noderivat)
	xyouts,-0.9,a[0]*0.95,'FWHM = '+string(a[2],format='(f4.2)'), $
		charsize=!p.charsize/mf
	xyouts,-0.9,a[0]*0.85,'RMS  = '+string(stddev(xy[*,0]),format='(f4.2)'), $
		charsize=!p.charsize/mf
	oplot,x,nfit,psym=0,color=tci(3)
;
	y=histogram(xy[*,1],min=-1,binsize=0.01,max=1)
	plot,x,y,psym=10,ystyle=0,xrange=[-1,1], $
		xtitle='NAT normalized Y offset',ytitle='N', $
		title=''
	nfit=curvefit(x,y,y*0+1,a,function_name='funct_nathist',/noderivat)
	xyouts,-0.9,a[0]*0.95,'FWHM = '+string(a[2],format='(f4.2)'), $
		charsize=!p.charsize/mf
	xyouts,-0.9,a[0]*0.85,'RMS  = '+string(stddev(xy[*,1]),format='(f4.2)'), $
		charsize=!p.charsize/mf
	oplot,x,nfit,psym=0,color=tci(3)
;
endfor
;
end
;-------------------------------------------------------------------------------
pro plot_benhist,sid,counts=counts
;
common StarBase,StarTable,Notes
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(counts) gt 0 then natcounts=counts else natcounts=quadcounts
;
if n_elements(genconfig) ne 0 then begin
	if n_elements(sid) eq 0 then sid=indgen(genconfig.numsid)+1
	stations=genconfig.stationid
endif else begin
	if n_elements(sid) eq 0 then sid=indgen(n_elements(natcounts[*,0,0]))+1
	stations=strarr(n_elements(sid))
endelse
;
!p.charsize=1.5
loadct,0,/silent
;
f_rebin=3
binsize=0.04
numbin=nint(2.0/binsize)+1
x=(findgen(numbin)-float(numbin-2)/2)*binsize
;
ben_rms=[0.35,0.30,0.25,0.20,0.15,0.10,0.00]
ben_bno=['B0','B1','B2','B3','B4','B5','B6']
;
xsize=numbin*f_rebin*n_elements(sid)
ysize=numbin*f_rebin
if xsize ne !d.x_size or ysize ne !d.y_size then $
	window,xsize=xsize,ysize=ysize,/free
;
for i=0,n_elements(sid)-1 do begin
;
	xy=natoffset(natcounts[sid[i]-1,*,*])
;
	y=lonarr(numbin,numbin)
	for j=0,numbin-1 do begin
		index=where(xy[*,1] ge x[j]-binsize/2 $
			and xy[*,1] lt x[j]+binsize/2,count)
		if count gt 0 then $
		y[*,j]=histogram(xy[index,0],min=-1,binsize=binsize,max=1)
	endfor
;
	tvscl,rebin(y,numbin*f_rebin,numbin*f_rebin,/sample),i*numbin*f_rebin,0
	s=abs(ben_rms-sqrt(stddev(xy[*,0])*stddev(xy[*,1])))
	xyouts,(numbin*f_rebin/2)+i*numbin*f_rebin,1,/device, $
		ben_bno[where(s eq min(s))] $
		+' ('+string(sqrt(stddev(xy[*,0])*stddev(xy[*,1])),format='(f4.2)')+')',color=250
	xyouts,(numbin*f_rebin/2)+i*numbin*f_rebin,numbin*f_rebin-!d.y_ch_size-1,/device, $
		stations[sid[i]-1]
;
;	Fit Gaussian profile to 2D histogram
	aparm=fltarr(7)
	r=gauss2dfit(y,aparm)
	print,'Widths in X and Y directions: ',aparm[2],aparm[3]
;
endfor
;
end
;************************************************************************Block 6
function packettype,type
;
; Translate packet types and return their names.
;
result=strarr(n_elements(type))
;
types=[ $
['FILE_HEADER','0x00000000'], $
['PACKET_SYMBOL_TABLE','0x00010000'], $
['PACKET_DIRECTORY','0x00010001'], $
['PAD','0x00020000'], $
['END_OF_DISK','0x00030000'], $
['END_OF_SESSION','0x00030001'], $
['SYS_CONFIG','0x000a0000'], $
['SYS_LOG','0x000a0001'], $
['SCAN_START_VERSION_1','0x000b0000'], $
['STAR_ACQUIRED','0x000b0001'], $
['SCAN_START_VERSION_2','0x000b0002'], $
['SCAN_END','0x000b0003'], $
['SCAN_OPERATOR_COMMENT','0x000b0004'], $
['FRINGE_DATA_VERSION_1','0x000c0001'], $
['FRINGE_DATA_VERSION_2','0x000c0002'], $
['FRINGE_BG','0x000c0004'], $
['FRINGE_DARK','0x000c0005'], $
['FRINGECON_SERVO_PARAMS','0x000c0006'], $
['NAT_COUNTS','0x000d0001'], $
['NAT_PIEZO_SIGNAL','0x000d0002'], $
['NAT_BG','0x000d0003'], $
['NAT_DARK','0x000d0004'], $
['NAT_SERVO_PARAMS','0x000d0005'], $
['FDL_POSITION','0x000e0001'], $
['FDL_JITTER','0x000e0002'], $
['FDL_STATUS','0x000e0003'], $
['METRO_DATA','0x000f0001'], $
['METRO_CONFIG','0x000f0002'], $
['WASA_IMAGE','0x00110001'], $
['SID_MOTOR_COUNTS','0x00110002'], $
['SID_MODEL','0x00110003'], $
['SYNC','0x0e100f09'] $
]
;
for i=0,n_elements(types[0,*])-1 do begin
	status=execute("t='"+strmid(types[1,i],2,8)+"'X")
	index=where(type eq t,count)
	if count gt 0 then result[index]=types[0,i]
endfor
;
return,result
;
end
;-------------------------------------------------------------------------------
function readpacket,pdr
;
; Read and return packet specified with packet directory record 'pdr'.
;
common PacketData,pd_all,var_num,packet
;
packet={header:pdr}
;
case pdr.type of
;		FILE_HEADER
'00000000'X:	begin
		packet={header:pdr, $
			body:{ $
			Date:'                                 ' $
			}}
		date=packet.body.date
		date_b=byte(date)
		status=linknload(!external_lib,'readfileheader', $
			pdr.file,pdr.offset,pdr.length, $
			date,date_b)
		packet.body.date=date
		packet.body.date=string(date_b)
		end
;		SYS_CONFIG
'000a0000'X:	begin
		numsid=0l
		numout=0l
		maxwav=0l
		maxbas=0l
		command=0l
		status=linknload(!external_lib,'readsysconfig', $
			pdr.file,pdr.offset,pdr.length, $
			command,numsid,numout,maxwav,maxbas)
		maxtriple=(numsid-1)*(numsid-2)/2
		maxconfig=numsid 
		for i=2,numsid do maxconfig=maxconfig+combinations(numsid,i)
		packet={header:pdr, $
			body:{ $
			genconfig:allocGenConfig(numsid,numout,maxtriple,maxbas,maxwav,maxconfig), $
			geoparms:allocGeoParms(),systemid:'' $
			}}
		latitude=packet.body.geoparms.latitude
		longitude=packet.body.geoparms.longitude
		altitude=packet.body.geoparms.altitude
		earthradius=packet.body.geoparms.earthradius
		j2=packet.body.geoparms.j2
		tdtutc=packet.body.geoparms.tdt_tai
		ut1utc=0.d0
		instrcohint=packet.body.genconfig.instrcohint
		numbin=packet.body.genconfig.numbin
		beamcid=packet.body.genconfig.beamcombinerid
		stationcoord=packet.body.genconfig.stationcoord
		stationid=packet.body.genconfig.stationid & stationid[*]='123'
		stationid_b=byte(stationid)
		stroke=packet.body.genconfig.stroke
		fdltankid=packet.body.genconfig.delaylineid
		fringemod=packet.body.genconfig.fringemod
		biasmod=packet.body.genconfig.biasmod
		inputpair=lonarr(2,maxbas,numout)
		wavelength=packet.body.genconfig.wavelength
		wavelengtherr=packet.body.genconfig.wavelengtherr
		chanwidth=packet.body.genconfig.chanwidth
		chanwidtherr=packet.body.genconfig.chanwidtherr
		numbas=packet.body.genconfig.numbaseline
		numbias=packet.body.genconfig.numbiasfreq
		numwav=packet.body.genconfig.numspecchan
		command=1l
		status=linknload(!external_lib,'readsysconfig', $
			pdr.file,pdr.offset,pdr.length, $
			command,numsid,numout,maxwav,maxbas, $
			latitude,longitude,altitude,earthradius,j2, $
			tdtutc,ut1utc, $
			instrcohint,numbin,beamcid, $
			stationcoord,stationid,stationid_b,stroke,fdltankid, $
			fringemod,biasmod,inputpair, $
			wavelength,wavelengtherr,chanwidth,chanwidtherr, $
			numbas,numbias,numwav)
		packet.body.geoparms.latitude=latitude
		packet.body.geoparms.longitude=longitude
		packet.body.geoparms.altitude=altitude
		packet.body.geoparms.earthradius=earthradius
		packet.body.geoparms.j2=j2
		packet.body.geoparms.tai_utc=tdtutc
		packet.body.genconfig.numsid=numsid
		packet.body.genconfig.numoutbeam=numout
		packet.body.genconfig.numtriple=0
		packet.body.genconfig.numbaseline=numbas
		packet.body.genconfig.numbiasfreq=numbias
		packet.body.genconfig.numspecchan=numwav
		packet.body.genconfig.stationcoord=stationcoord
		packet.body.genconfig.delaylineid=fdltankid
		packet.body.genconfig.stationid=stationid
		packet.body.genconfig.fringemod=fringemod
		packet.body.genconfig.biasmod=biasmod
		packet.body.genconfig.wavelength=wavelength
		packet.body.genconfig.wavelengtherr=wavelengtherr
		packet.body.genconfig.chanwidth=chanwidth
		packet.body.genconfig.chanwidtherr=chanwidtherr
		packet.body.genconfig.numbaseline=numbas
		packet.body.genconfig.numspecchan=numwav
		packet.body.genconfig.baselineid=strjoin(stationid[inputpair],'-')
		packet.body.genconfig.refstation=1
		packet.body.systemid='NPOI'
		end
;		SCAN_START_VERSION_1
'000b0000'X:	begin
		packet={header:pdr, $
			body:{ $
			ScanId:0l,StarId:'                                 ' $
			}}
		scanid=packet.body.scanid
		starid=packet.body.starid
		starid_b=byte(starid)
		status=linknload(!external_lib,'readscanstartversion1', $
			pdr.file,pdr.offset,pdr.length, $
			scanid,starid,starid_b)
		packet.body.scanid=scanid
		packet.body.starid=starid
		end
;		FDL_POSITION
'000e0001'X:	begin
		packet={header:pdr, $
			body:{ $
			laserpath:dblarr(var_num.fdl,var_num.rec), $
			timestamp:lonarr(var_num.rec) $
			}}
		laserpath=packet.body.laserpath
		time=packet.body.timestamp
		status=linknload(!external_lib,'readfdlposition', $
			pdr.file,pdr.offset,pdr.length, $
			laserpath,time, $
			var_num.fdl,var_num.rec)
		packet.body.laserpath=laserpath
		packet.body.timestamp=time
		packet.header.time=time[var_num.rec-1]
		end
;		FRINGE_DATA_VERSION_1
'000c0001'X:	begin
		fcde=replicate({value:0,error:0},var_num.spec,var_num.rec)
		packet={header:pdr, $
			body:{ $
			bincounts:bytarr(var_num.spec,var_num.bin,var_num.chan,var_num.rec), $
			fcde:fcde $
			}}
		counts=packet.body.bincounts
		status=linknload(!external_lib,'readfringedataversion1', $
			pdr.file,pdr.offset,pdr.length, $
			counts, $
			var_num.spec,var_num.bin,var_num.chan,var_num.rec)
		packet.body.bincounts=counts
		end
;		FRINGE_BG
'000c0004'X:	begin
		packet={header:pdr, $
			body:{ $
			bincounts:bytarr(var_num.spec,var_num.bin,var_num.chan,var_num.rec), $
			timestamp:lonarr(var_num.rec) $
			}}
		counts=packet.body.bincounts
		time=packet.body.timestamp
		status=linknload(!external_lib,'readfringebg', $
			pdr.file,pdr.offset,pdr.length, $
			counts,time, $
			var_num.spec,var_num.bin,var_num.chan,var_num.rec)
		packet.body.bincounts=counts
		packet.body.timestamp=time
		packet.header.time=time[var_num.rec-1]
		end
;		NAT_COUNTS
'000d0001'X:	begin
		packet={header:pdr, $
			body:{ $
			quadcounts:intarr(var_num.sid,4,var_num.rec), $
			timestamp:lonarr(var_num.rec) $
			}}
		counts=packet.body.quadcounts
		time=packet.body.timestamp
		status=linknload(!external_lib,'readnatcounts', $
			pdr.file,pdr.offset,pdr.length, $
			counts,time, $
			var_num.sid,var_num.rec)
		packet.body.quadcounts=counts
		packet.body.timestamp=time
		packet.header.time=time[var_num.rec-1]
		end
;		FRINGE_DARK
'000c0005'X:	begin
		packet={header:pdr, $
			body:{ $
			rate:fltarr(var_num.spec,var_num.chan), $
			numint:0l,time:0l $
			}}
		rate=packet.body.rate
		time=packet.body.time
		status=linknload(!external_lib,'readfringedark', $
			pdr.file,pdr.offset,pdr.length, $
			rate,time, $
			var_num.spec,var_num.chan)
		packet.body.rate=rate
		packet.body.time=time
		end
	else:	if not !qiet then $
		print,'***Error(READPACKET): unknown packet type '+ $
			packettype(pdr.type)+'!'
endcase
;
return,packet
;
end
;-------------------------------------------------------------------------------
function packetdir,filename
;
; Return packet directory for NPOI raw file. 
; Also set the var_num fields which contain important packet
; dimensioning information.
;
common PacketData,pd_all,var_num,packet
;
rawfile=filename
;
maxdirent=40000l
;
pd_record={file:rawfile,day:0l,time:0l,type:0l,length:0l,offset:0l}
pd=replicate(pd_record,maxdirent)
;
day=pd.day
time=pd.time
type=pd.type
length=pd.length
offset=pd.offset
;
status=linknload(!external_lib,'scandisk', $
	rawfile,day,time,type,length,offset,maxdirent)
if status ne 0 then return,0
;
pd.day=day
pd.time=time
pd.type=type
pd.length=length
pd.offset=offset
;
pd=pd[where(pd.length gt 0)]
pd=pd[where(strlen(packettype(pd.type)) gt 0)]
openr,unit,rawfile,/get_lun
r=fstat(unit)
if long(total(pd.length)) gt r.size then pd=pd[0:n_elements(pd)-2]
free_lun,unit
;
; Set packet dimensioning info
index=where(pd.type eq '00000000'X,count)
if count gt 0 then begin
	p=readpacket(pd[index[0]])
	parsedate,p.body.date,y,m,d
	if julian(y,m,d) lt julian(2001,9,28) $
		then var_num={sid:6l,fdl:6l,spec:3l,bin:8l,chan:32l,rec:500l} $
		else var_num={sid:6l,fdl:6l,spec:2l,bin:64l,chan:16l,rec:500l}
;	These files have the wrong date
	if julian(y,m,d) ge 2001-01-08 and julian(y,m,d) le 2001-01-12 then $
		     var_num={sid:6l,fdl:6l,spec:2l,bin:64l,chan:16l,rec:500l}
endif
;
return,pd
;
end
;-------------------------------------------------------------------------------
pro rawlist,date
;
; Collect and analyse all raw files corresponding to date, write binary
; packet directories and parameter files for use with CONSTRICTOR.
;
if n_elements(date) eq 0 then begin
	jd=systime(/julian)
	caldat,systime(/julian),m,d,y
	date=constrictordate(y,m,d)
endif
;
parsedate,date,y,m,d
rdate=rawdate(y,m,d)
; Need to check for files from previous UT day 
yesterday=previousdate(date)
parsedate,yesterday,yy,my,dy
ydate=rawdate(yy,my,dy)

jd=julian(y,m,d)
doy=fix(jd-julian(y,1,1))+1
;
rawdir=''
if strupcase(getenv('HOST')) eq 'OCTANS'  then rawdir='/datacon11/'
;
rawfiles=file_search(rawdir+rdate+'*')
yesterdayFiles=findfile(rawdir+ydate+'*')
if strlen(rawfiles[0]) eq 0 then begin
	print,'***Error(RAWLIST): no files found for this date!'
	return
endif

; concatenate files with today's date and previous date 
; if required
if yesterdayFiles[0] ne '' then rawfiles=[yesterdayFiles,rawfiles]

index=where(strpos(rawfiles,'starLog') ge 0,count)
if count eq 1 then scnfile=rawfiles[index[0]] else scnfile=''
index=where(strpos(rawfiles,'log') ge 0,count)
if count eq 1 then obsfile=rawfiles[index[0]] else obsfile=''
;
index=where(strpos(rawfiles,'.gz') eq -1,numfiles)
if numfiles eq 0 then begin
	print,'***Error(RAWLIST): all files have .gz extension!'
	return
endif
rawfiles0=rawfiles[index]
;
i0=0
index=where(strpos(rawfiles0,'fringeData') ge 0,count)
if count gt 0 then rawfiles[i0:i0+count-1]=rawfiles0[index]
i0=i0+count
index=where(strpos(rawfiles0,'alignData') ge 0,count)
if count gt 0 then rawfiles[i0:i0+count-1]=rawfiles0[index]
i0=i0+count
index=where(strpos(rawfiles0,'sidData') ge 0,count)
if count gt 0 then rawfiles[i0:i0+count-1]=rawfiles0[index]
i0=i0+count
numfiles=i0
rawfiles=rawfiles[0:numfiles-1]
;
infiles=strarr(numfiles)
dirfiles=strarr(numfiles)
maxpacket=intarr(numfiles)
j=0
;
print,'Begin processing raw files...'
for i=0,numfiles-1 do begin
;	Get packet directory
	d=packetdir(rawfiles[i])
	n=n_elements(d)
;	Write packet list file
	openw,unit,date+'.'+strcompress(string(i+1),/remove)+'.lst',/get_lun
	for k=0l,n-1 do printf,unit,d[k].length,d[k].time,d[k].day,' ', $
		hms(double(d[k].time)/3600000l),packettype(d[k].type)
	free_lun,unit
;	Read all packets and update header time stamps from time stamps in body
	!qiet=1
	if jd ge julian(2001,9,25) then begin
		ht=d.time
		for k=0l,n-1 do begin
			p=readpacket(d[k])
			d[k].time=p.header.time
		endfor
		index=where(strpos(packettype(d.type),'SCAN_START') ge 0,count)
		for k=0,count-1 do begin
			kndex=where((strpos(packettype(d.type),'FRINGE_DATA') ge 0 $
				  or strpos(packettype(d.type),'FRINGE_BG') ge 0) $
				 and d.time lt d[index[k]].time)
			if kndex[0] ge 0 then begin
				d[where(d.time eq max(d[kndex].time))].time=0
			endif
		endfor
	endif
	d=d[where(d.time gt 0)]
	n=n_elements(d)
	print,rawfiles[i],': ',n,' valid packets'
	!qiet=0
;	Write binary packet directory
	openw,unit,date+'.'+strcompress(string(i+1),/remove)+'.dir',/get_lun
	for k=0l,n-1 do begin
		writeu,unit,long(i)
		writeu,unit,d[k].type
		writeu,unit,ulong(d[k].offset)
		writeu,unit,d[k].day
		writeu,unit,d[k].time
	endfor
	free_lun,unit
;	Check END_OF_DISK
	if packettype(d[n-1].type) ne 'END_OF_DISK' then begin
		print,'! No END_OF_DISK packet'
	endif
;	Check day numbers
	if n_elements(unique(d.day)) gt 1 then begin
		print,'! Several day numbers: ',unique(d.day)
	endif
	index=where(d.day ne doy,count)
	if count gt 0 then begin
		print,'! Unexpected day numbers: ',unique(d[index].day)
	endif
;
	infiles[j]=rawfiles[i]
	dirfiles[j]=date+'.'+strcompress(string(i+1),/remove)+'.dir'
	maxpacket[j]=n
	j=j+1
endfor
;
if j gt 0 then begin
	filestring=strjoin('"'+rawfiles[0:j-1]+'",')
	filestring=strmid(filestring,0,strlen(filestring)-1)
	dirstring=strjoin('"'+dirfiles[0:j-1]+'",')
	dirstring=strmid(dirstring,0,strlen(dirstring)-1)
	maxstring=strjoin(string(maxpacket)+',')
	maxstring=strmid(maxstring,0,strlen(maxstring)-1)
	openw,unit,date+'.par',/get_lun
	printf,unit,'InFiles = '+filestring
	printf,unit,'DirFiles = '+dirstring
	printf,unit,'MaxPacket ='+maxstring
	printf,unit,'DayNumber = '+string(doy)
	printf,unit,'StartTime = "00:00:00"'
	printf,unit,'StopTime = "24:00:00"'
	printf,unit,'LogFile = '+obsfile
	printf,unit,'ScnFile = '+scnfile
	printf,unit,'OutFile = '+date+'.con'
	printf,unit,'Raw = Off'
	printf,unit,'Lock = Off'
	printf,unit,'Triple = On'
	printf,unit,'NumAv = 500'
	printf,unit,'RefStation = 4'
	printf,unit,'SmartFDLAverage = Off'
	printf,unit,'Metrology = Off'
	printf,unit,'Nat = On'
	free_lun,unit
endif
;
end
;************************************************************************Block 7
pro disptest,baseline,cohint
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(cohint) eq 0 then cohint=100
;
f=fdldelay(baseline)
t=nint(timestamp-timestamp[0])/1000.
time=add(t,cohint)/cohint
fdl=add(f,cohint)*1e4
;
ch=[0,1,2,3,4,5,6,7,8,9,11,12,13,14,15,16,17,18,19]
l=genconfig.wavelength[ch,outputbeam-1]
v=fringevis(bincounts[*,ch,*])
a=total(abs(v),2) & a=a/max(a) & p2d=(total(a*l)/total(a))/(2*!pi)
d=gitter(41,-4e-6)
;
g=groupdelay(visdft(v,l,d),l,d)
r=visrotate(v,l,g)
vis=add(r,cohint,1)
solve,vis,l,sg
;
p=cphase(visdft(v,l,g))
s=f-smooth(f,100,/edge_truncate)
g1=s+groupdelay(box(abs(visdft(visrotate(v,l,s),l,d)),10,1),l,d) $
    +phaseshift(f,l)*p2d
;g1=s+groupdelay(box(abs(visdft(visrotate(v,l,s),l,d)),10,1),l,d) $
;    +phaseshift(f-groupshift(c1,l),l)*p2d
p1=cphase(total(visrotate(v,l,g1),1))
c1=phasescan(t,p1)
g2=g1-c1*p2d
p2=cphase(total(visrotate(v,l,g2),1))
;
!p.charsize=3
window_slide,xsize=20000,ysize=800
!p.multi=[0,1,3]
!p.psym=3
!y.range=[-4e-6,4e-6]
plot,t,g,ytitle='Group delay [m]'
plot,t,g-p*p2d,ytitle='Group + phase delay [m]'
plot,t,g-p*p2d,ytitle='Enh. group + phase delay [m]'
oplot,t,g2-p2*p2d,psym=3,color=tci(7) 
;
solve,add(visrotate(v,l,g),cohint,1),l,sg
solve,add(visrotate(v,l,g1),cohint,1),l,sg1
;v=fringevis(bincounts,1)
;del=fringedelay(v,f,channels=ch+1)
;solve,add(visrotate(v(ch,*),l,del),cohint,1),l,sp1,/pt
solve,add(visrotate(v,l,g2-p2*p2d),cohint,1),l,sp1,/pt
;
!p.charsize=2
window,/free,ysize=800
!x.ticks=0
!y.ticklen=0
!p.multi=[0,1,4]
!x.title='Time [s]'
!y.title='Residual corrected delay [mu]'
!y.range=[-40,40]
;
!p.title='One-color delay on '+baseline
y=polyres(time,fdl,2)
!p.charsize=2
plot,time,y
!p.charsize=1
xyouts,5,-15,'RMS = '+string(stddev(fdl),format='(f4.1)')
;
!p.title='Group delay tracking on '+baseline
y=polyres(time,fdl+sg[0,*],2)
index=where(abs(y) lt 40,count)
!p.charsize=2
plot,time[index],y[index]
!p.charsize=1
xyouts,5,-25,'RMS = '+string(stddev(y[index]),format='(f4.1)')
xyouts,40,-25,'Flagged '+string(n_elements(y)-count)+' samples'
;
!p.title='Enhanced group delay tracking on '+baseline
y=polyres(time,fdl+sg1[0,*],2)
index=where(abs(y) lt 40,count)
!p.charsize=2
plot,time[index],y[index]
!p.charsize=1
xyouts,5,-25,'RMS = '+string(stddev(y[index]),format='(f4.1)')
xyouts,40,-25,'Flagged '+string(n_elements(y)-count)+' samples'
;
!p.title='Phase tracking on '+baseline
y=polyres(time,fdl+sp1[0,*],2)
index=where(abs(y) lt 40,count)
!p.charsize=2
plot,time[index],y[index]
!p.charsize=1
xyouts,5,-25,'RMS = '+string(stddev(y[index]),format='(f4.1)')
xyouts,40,-25,'Flagged '+string(n_elements(y)-count)+' samples'
;...............................................................................
;
window,/free
!p.multi=[0,1,3]
!x.title='Interval'
!y.title='Fringe ID'
!y.range=[-4,4]
!p.title='Group delay tracking on '+baseline
plot,sg[2,*],psym=3
!p.title='Enhanced group delay tracking on '+baseline
plot,sg1[2,*],psym=3
!p.title='Phase tracking on '+baseline
plot,sp1[2,*],psym=3
;
;...............................................................................
;
window,/free
!p.multi=[0,1,2]
num=1
index=lindgen(n_elements(t)/num)*num
plot,t[index],p1[index],title='Residual phase on '+baseline,yrange=[-3,3], $
	xtitle='Time [s]',ytitle='[radians]'
plot,t[index],p2[index],title='Corrected phase on '+baseline,yrange=[-3,3], $
	xtitle='Time [s]',ytitle='[radians]'
;
end
;-------------------------------------------------------------------------------
pro altair1
;
; closure group delays
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
get_data,'2002-07-07.raw.062.01'
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
ob=1
ch=channelindex[genconfig.spectrometerid[ob]]-1
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
get_bincounts,2
hds_close
nphot=fringenphot(bincounts[*,ch,*])
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
!p.charsize=1.0
;
f1=fdldelay(genconfig.baselineid[0,ob])
v1=fringevis(bincounts[*,ch,*],genconfig.fringemod[0,ob])
g1b=groupdelay(box(abs(visdft(v1,l,d)),10,1),l,d,g1i)
p1=cphase(total(visrotate( $
	v1,l,g1b+phaseshift(f1-g1b,l)*lambdawhite(v1,l)/(2*!pi)),1))
plot,t,p1,psym=3,ystyle=1
f2=fdldelay(genconfig.baselineid[1,ob])
v2=fringevis(bincounts[*,ch,*],genconfig.fringemod[1,ob])
g2b=groupdelay(box(abs(visdft(v2,l,d)),10,1),l,d,g2i)
p2=cphase(total(visrotate( $
	v2,l,g2b+phaseshift(f2-g2b,l)*lambdawhite(v2,l)/(2*!pi)),1))
plot,t,p2,psym=3,ystyle=1
f3=fdldelay(genconfig.baselineid[2,ob])
v3=fringevis(bincounts[*,ch,*],genconfig.fringemod[2,ob])
g3b=groupdelay(box(abs(visdft(v3,l,d)),10,1),l,d,g3i)
p3=cphase(total(visrotate( $
	v3,l,g3b+phaseshift(f3-g3b,l)*lambdawhite(v3,l)/(2*!pi)),1))
plot,t,p3,psym=3,ystyle=1
f4=fdldelay(genconfig.baselineid[3,ob])
v4=fringevis(bincounts[*,ch,*],genconfig.fringemod[3,ob])
g4b=groupdelay(box(abs(visdft(v4,l,d)),10,1),l,d,g4i)
p4=cphase(total(visrotate( $
	v4,l,g4b+phaseshift(f4-g4b,l)*lambdawhite(v4,l)/(2*!pi)),1))
plot,t,p4,psym=3,ystyle=1
f5=fdldelay(genconfig.baselineid[4,ob])
v5=fringevis(bincounts[*,ch,*],genconfig.fringemod[4,ob])
g5b=groupdelay(box(abs(visdft(v5,l,d)),10,1),l,d,g5i)
p5=cphase(total(visrotate( $
	v5,l,g5b+phaseshift(f5-g5b,l)*lambdawhite(v5,l)/(2*!pi)),1))
plot,t,p5,psym=3,ystyle=1
f6=fdldelay(genconfig.baselineid[5,ob])
v6=fringevis(bincounts[*,ch,*],genconfig.fringemod[5,ob])
g6b=groupdelay(box(abs(visdft(v6,l,d)),10,1),l,d,g6i)
p6=cphase(total(visrotate( $
	v6,l,g6b+phaseshift(f6-g6b,l)*lambdawhite(v6,l)/(2*!pi)),1))
plot,t,p6,psym=3,ystyle=1
;
g1b=g1b*1e6
g2b=g2b*1e6
g3b=g3b*1e6
g4b=g4b*1e6
g5b=g5b*1e6
g6b=g6b*1e6
;
; g1b=groupdelays(1,2,numav=10,gindex=g1i)*1e6
; g2b=groupdelays(2,2,numav=10,gindex=g2i)*1e6
; g3b=groupdelays(3,2,numav=10,gindex=g3i)*1e6
; g4b=groupdelays(4,2,numav=10,gindex=g4i)*1e6
; g5b=groupdelays(5,2,numav=10,gindex=g5i)*1e6
; g6b=groupdelays(6,2,numav=10,gindex=g6i)*1e6
;
; Compute closure group delays
t1=g1b-g2b-g6b
t2=g2b+g3b-g4b
t3=g3b-g5b-g6b
;
; Flag bad data
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
; Plot closure group delays
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
x=findgen(n_elements(t))
plot,x,t1
oplot,x[t1ib],t1[t1ib],color=tci(2)
plot,x,t2
oplot,x[t2ib],t2[t2ib],color=tci(2)
plot,x,t3
oplot,x[t3ib],t3[t3ib],color=tci(2)
;
; Determine which data points are good
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
; g1b=g1b-t1m
; g4b=g4b+t2m
; g5b=g5b+t3m
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
index1=where(g1i eq 1 and g2i eq 1 and g6i eq 1 $
	and abs(t1-t1m) gt t1me)
index2=where(g2i eq 1 and g3i eq 1 and g4i eq 1 $
	and abs(t2-t2m) gt t2me)
index3=where(g3i eq 1 and g5i eq 1 and g6i eq 1 $
	and abs(t3-t3m) gt t3me)
g1i2=g1i
g2i2=g2i
g3i2=g3i
g4i2=g4i
g5i2=g5i
g6i2=g6i
g1i2[index1]=0 & g2i2[index1]=0 & g6i2[index1]=0
g2i2[index2]=0 & g3i2[index2]=0 & g4i2[index2]=0
g3i2[index3]=0 & g5i2[index3]=0 & g6i2[index3]=0
;
; Prepare weights
n=n_elements(t)	; the number of data points, i.e. time stamps
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
; These will hold solutions for E2, C, W, and W7
se2=fltarr(n)
sac=fltarr(n)
saw=fltarr(n)
sw7=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saci=intarr(n)+1
sawi=intarr(n)+1
sw7i=intarr(n)+1
solni=intarr(4,n)+1
;
; Design matrix, last line if for reference W
dm=transpose([[1,-1,0,0], $
	      [0,-1,1,0], $
	      [0,0,-1,1], $
	      [0,-1,0,1], $
	      [-1,0,0,1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
;
; For each time stamp, compute station based group delay solutions
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1b[i],g2b[i],g3b[i],g4b[i],g5b[i],g6b[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
; Extract solutions
se2=reform(solns[0,*])
sac=reform(solns[1,*])
saw=reform(solns[2,*])
sw7=reform(solns[3,*])
;
; Compute baseline solutions for the group delays
g1g=se2-sac
g2g=saw-sac
g3g=sw7-saw
g4g=sw7-sac
g5g=sw7-se2
g6g=se2-saw
;
; Add back median offsets
; g1g=g1g+t1m
; g4g=g4g-t2m
; g5g=g5g-t3m
;
g1g=g1g/1e6
g2g=g2g/1e6
g3g=g3g/1e6
g4g=g4g/1e6
g5g=g5g/1e6
g6g=g6g/1e6
;
; Plot phases rotated with the baseline solutions
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
;
g1gc=g1g+phaseshift(f1-g1g,l)*lambdawhite(v1,l)/(2*!pi)
p1=cphase(total(visrotate(v1,l,g1gc),1))
plot,t,p1,psym=3,ystyle=1
g2gc=g2g+phaseshift(f2-g2g,l)*lambdawhite(v2,l)/(2*!pi)
p2=cphase(total(visrotate(v2,l,g2gc),1))
plot,t,p2,psym=3,ystyle=1
g3gc=g3g+phaseshift(f3-g3g,l)*lambdawhite(v3,l)/(2*!pi)
p3=cphase(total(visrotate(v3,l,g3gc),1))
plot,t,p3,psym=3,ystyle=1
g4gc=g4g+phaseshift(f4-g4g,l)*lambdawhite(v4,l)/(2*!pi)
p4=cphase(total(visrotate(v4,l,g4gc),1))
plot,t,p4,psym=3,ystyle=1
g5gc=g5g+phaseshift(f5-g5g,l)*lambdawhite(v5,l)/(2*!pi)
p5=cphase(total(visrotate(v5,l,g5gc),1))
plot,t,p5,psym=3,ystyle=1
g6gc=g6g+phaseshift(f6-g6g,l)*lambdawhite(v6,l)/(2*!pi)
p6=cphase(total(visrotate(v6,l,g6gc),1))
plot,t,p6,psym=3,ystyle=1
;
; Use baseline solutions to coherently integrate visibilities
cohint=10
;
p2d=lambdawhite(v1,l)/(2*!pi)
c1=phasescan(t,p1)
g2=g1gc-c1*p2d
c2=cphase(total(visrotate(v1,l,g2),1))
vis1=add(visrotate(v1,l,g2-c2*p2d),cohint,1)
;
p2d=lambdawhite(v2,l)/(2*!pi)
c1=phasescan(t,p2)
g2=g2gc-c1*p2d
c2=cphase(total(visrotate(v2,l,g2),1))
vis2=add(visrotate(v2,l,g2-c2*p2d),cohint,1)
;
p2d=lambdawhite(v3,l)/(2*!pi)
c1=phasescan(t,p3)
g2=g3gc-c1*p2d
c2=cphase(total(visrotate(v3,l,g2),1))
vis3=add(visrotate(v3,l,g2-c2*p2d),cohint,1)
;
p2d=lambdawhite(v4,l)/(2*!pi)
c1=phasescan(t,p4)
g2=g4gc-c1*p2d
c2=cphase(total(visrotate(v4,l,g2),1))
vis4=add(visrotate(v4,l,g2-c2*p2d),cohint,1)
;
p2d=lambdawhite(v5,l)/(2*!pi)
c1=phasescan(t,p5)
g2=g5gc-c1*p2d
c2=cphase(total(visrotate(v5,l,g2),1))
vis5=add(visrotate(v5,l,g2-c2*p2d),cohint,1)
;
p2d=lambdawhite(v6,l)/(2*!pi)
c1=phasescan(t,p6)
g2=g6gc-c1*p2d
c2=cphase(total(visrotate(v6,l,g2),1))
vis6=add(visrotate(v6,l,g2-c2*p2d),cohint,1)
;
; Compute triples for unaveraged and coherently av. data
ch=11
tv1=reform(v1[ch,*]*conj(v2[ch,*])*conj(v6[ch,*])) $
   -reform(complex(abs(v1[ch,*])^2 $
		  +abs(v2[ch,*])^2 $
		  +abs(v6[ch,*])^2 $
	    	  -2*nphot[ch,*],0))*0
tv2=reform(v2[ch,*]*     v3[ch,*] *conj(v4[ch,*])) $
   -reform(complex(abs(v2[ch,*])^2 $
		  +abs(v3[ch,*])^2 $
		  +abs(v4[ch,*])^2 $
	    	  -2*nphot[ch,*],0))*0
tv3=reform(v3[ch,*]*conj(v5[ch,*])*conj(v6[ch,*])) $
   -reform(complex(abs(v3[ch,*])^2 $
		  +abs(v5[ch,*])^2 $
		  +abs(v6[ch,*])^2 $
	    	  -2*nphot[ch,*],0))*0
tv4=reform(v4[ch,*]*conj(v5[ch,*])*conj(v1[ch,*])) $
   -reform(complex(abs(v4[ch,*])^2 $
		  +abs(v5[ch,*])^2 $
		  +abs(v1[ch,*])^2 $
	    	  -2*nphot[ch,*],0))*0
;
tvis1=reform(vis1[ch,*]*conj(vis2[ch,*])*conj(vis6[ch,*]))
tvis2=reform(vis2[ch,*]*     vis3[ch,*] *conj(vis4[ch,*]))
tvis3=reform(vis3[ch,*]*conj(vis5[ch,*])*conj(vis6[ch,*]))
tvis4=reform(vis4[ch,*]*conj(vis5[ch,*])*conj(vis1[ch,*]))
;
window,/free
set_screen
!y.range=[-150,150]
!p.multi=[0,2,4]
!p.charsize=2.0
rad=180/!pi
;
; Further average to the same time interval
plot,rad*cphase(add(tv1,1000)),psym=1
plot,rad*cphase(add(tvis1,100)),psym=1
plot,rad*cphase(add(tv2,1000)),psym=1
plot,rad*cphase(add(tvis2,100)),psym=1
plot,rad*cphase(add(tv3,1000)),psym=1
plot,rad*cphase(add(tvis3,100)),psym=1
plot,rad*cphase(add(tv4,1000)),psym=1
plot,rad*cphase(add(tvis4,100)),psym=1
;
end
;-------------------------------------------------------------------------------
pro altair2
;
; closure track delays
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
get_rawdata,'2002-07-07.raw.062'
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
ob=1
ch=channelindex[genconfig.spectrometerid[ob]]-1
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
get_bincounts,2
hds_close
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
!p.charsize=1.0
;
f1=fdldelay(genconfig.baselineid[0,ob])
v1=fringevis(bincounts[*,ch,*],genconfig.fringemod[0,ob])
p2d1=lambdawhite(v1,l)/(2*!pi)
g1b=groupdelay(box(abs(visdft(v1,l,d)),10,1),l,d,g1i)
g1p=g1b+phaseshift(f1-g1b,l)*p2d1
p1=cphase(total(visrotate(v1,l,g1p),1))
c1=phasescan(t,p1)
g1pc=g1p-c1*p2d1
plot,t,p1,psym=3,ystyle=1
f2=fdldelay(genconfig.baselineid[1,ob])
v2=fringevis(bincounts[*,ch,*],genconfig.fringemod[1,ob])
p2d2=lambdawhite(v2,l)/(2*!pi)
g2b=groupdelay(box(abs(visdft(v2,l,d)),10,1),l,d,g2i)
g2p=g2b+phaseshift(f2-g2b,l)*p2d2
p2=cphase(total(visrotate(v2,l,g2p),1))
c2=phasescan(t,p2)
g2pc=g2p-c2*p2d2
plot,t,p2,psym=3,ystyle=1
f3=fdldelay(genconfig.baselineid[2,ob])
v3=fringevis(bincounts[*,ch,*],genconfig.fringemod[2,ob])
p2d3=lambdawhite(v3,l)/(2*!pi)
g3b=groupdelay(box(abs(visdft(v3,l,d)),10,1),l,d,g3i)
g3p=g3b+phaseshift(f3-g3b,l)*p2d3
p3=cphase(total(visrotate(v3,l,g3p),1))
c3=phasescan(t,p3)
g3pc=g3p-c3*p2d3
plot,t,p3,psym=3,ystyle=1
f4=fdldelay(genconfig.baselineid[3,ob])
v4=fringevis(bincounts[*,ch,*],genconfig.fringemod[3,ob])
p2d4=lambdawhite(v4,l)/(2*!pi)
g4b=groupdelay(box(abs(visdft(v4,l,d)),10,1),l,d,g4i)
g4p=g4b+phaseshift(f4-g4b,l)*p2d4
p4=cphase(total(visrotate(v4,l,g4p),1))
c4=phasescan(t,p4)
g4pc=g4p-c4*p2d4
plot,t,p4,psym=3,ystyle=1
f5=fdldelay(genconfig.baselineid[4,ob])
v5=fringevis(bincounts[*,ch,*],genconfig.fringemod[4,ob])
p2d5=lambdawhite(v5,l)/(2*!pi)
g5b=groupdelay(box(abs(visdft(v5,l,d)),10,1),l,d,g5i)
g5p=g5b+phaseshift(f5-g5b,l)*p2d5
p5=cphase(total(visrotate(v5,l,g5p),1))
c5=phasescan(t,p5)
g5pc=g5p-c5*p2d5
plot,t,p5,psym=3,ystyle=1
f6=fdldelay(genconfig.baselineid[5,ob])
v6=fringevis(bincounts[*,ch,*],genconfig.fringemod[5,ob])
p2d6=lambdawhite(v6,l)/(2*!pi)
g6b=groupdelay(box(abs(visdft(v6,l,d)),10,1),l,d,g6i)
g6p=g6b+phaseshift(f6-g6b,l)*p2d6
p6=cphase(total(visrotate(v6,l,g6p),1))
c6=phasescan(t,p6)
g6pc=g6p-c6*p2d6
plot,t,p6,psym=3,ystyle=1
;
g1pc=g1pc*1e6
g2pc=g2pc*1e6
g3pc=g3pc*1e6
g4pc=g4pc*1e6
g5pc=g5pc*1e6
g6pc=g6pc*1e6
;
t1=g1pc-g2pc-g6pc
t2=g2pc+g3pc-g4pc
t3=g3pc-g5pc-g6pc
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
; g1b=g1b-t1m
; g4b=g4b+t2m
; g5b=g5b+t3m
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
index1=where(g1i eq 1 and g2i eq 1 and g6i eq 1 $
	and abs(t1-t1m) gt t1me)
index2=where(g2i eq 1 and g3i eq 1 and g4i eq 1 $
	and abs(t2-t2m) gt t2me)
index3=where(g3i eq 1 and g5i eq 1 and g6i eq 1 $
	and abs(t3-t3m) gt t3me)
g1i2=g1i
g2i2=g2i
g3i2=g3i
g4i2=g4i
g5i2=g5i
g6i2=g6i
g1i2[index1]=0 & g2i2[index1]=0 & g6i2[index1]=0
g2i2[index2]=0 & g3i2[index2]=0 & g4i2[index2]=0
g3i2[index3]=0 & g5i2[index3]=0 & g6i2[index3]=0
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
se2=fltarr(n)
sac=fltarr(n)
saw=fltarr(n)
sw7=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saci=intarr(n)+1
sawi=intarr(n)+1
sw7i=intarr(n)+1
solni=intarr(4,n)+1
;
dm=transpose([[1,-1,0,0], $
	      [0,-1,1,0], $
	      [0,0,-1,1], $
	      [0,-1,0,1], $
	      [-1,0,0,1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
wf=[0.2,1.0,1.0,0.2,0.2,1.0,1.0]
; weight=weight*wf
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1pc[i],g2pc[i],g3pc[i],g4pc[i],g5pc[i],g6pc[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
index=where(solni[0,*] eq 1 and solni[1,*] eq 1 $
	and solni[2,*] eq 1 and solni[3,*] eq 1)
se2=reform(solns[0,index])
sac=reform(solns[1,index])
saw=reform(solns[2,index])
sw7=reform(solns[3,index])
;
t=t[index]
;
g1g=se2-sac
g2g=saw-sac
g3g=sw7-saw
g4g=sw7-sac
g5g=sw7-se2
g6g=se2-saw
;
; g1g=g1g+t1m
; g4g=g4g-t2m
; g5g=g5g-t3m
;
g1g=g1g/1e6
g2g=g2g/1e6
g3g=g3g/1e6
g4g=g4g/1e6
g5g=g5g/1e6
g6g=g6g/1e6
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
;
v1r=visrotate(v1[*,index],l,g1g)
v2r=visrotate(v2[*,index],l,g2g)
v3r=visrotate(v3[*,index],l,g3g)
v4r=visrotate(v4[*,index],l,g4g)
v5r=visrotate(v5[*,index],l,g5g)
v6r=visrotate(v6[*,index],l,g6g)
;
p1r=cphase(total(v1r,1))
p2r=cphase(total(v2r,1))
p3r=cphase(total(v3r,1))
p4r=cphase(total(v4r,1))
p5r=cphase(total(v5r,1))
p6r=cphase(total(v6r,1))
;
; plot,t,p1r,psym=3,ystyle=1
; plot,t,p2r,psym=3,ystyle=1
; plot,t,p3r,psym=3,ystyle=1
; plot,t,p4r,psym=3,ystyle=1
; plot,t,p5r,psym=3,ystyle=1
; plot,t,p6r,psym=3,ystyle=1
;
cohint=10
;
vis1=add(visrotate(v1[*,index],l,g1g-p1r*p2d1*1),cohint,1)
vis2=add(visrotate(v2[*,index],l,g2g-p2r*p2d2*1),cohint,1)
vis3=add(visrotate(v3[*,index],l,g3g-p3r*p2d3*1),cohint,1)
vis4=add(visrotate(v4[*,index],l,g4g-p4r*p2d4*1),cohint,1)
vis5=add(visrotate(v5[*,index],l,g5g-p5r*p2d5*1),cohint,1)
vis6=add(visrotate(v6[*,index],l,g6g-p6r*p2d6*1),cohint,1)
;
k=0
tv1=reform(v1[k,*]*conj(v2[k,*])*conj(v6[k,*]))
tv2=reform(v2[k,*]*v3[k,*]*conj(v4[k,*]))
tv3=reform(v3[k,*]*conj(v5[k,*])*conj(v6[k,*]))
tv4=reform(v4[k,*]*conj(v5[k,*])*conj(v1[k,*]))
;
tvis1=reform(vis1[k,*]*conj(vis2[k,*])*conj(vis6[k,*]))
tvis2=reform(vis2[k,*]*vis3[k,*]*conj(vis4[k,*]))
tvis3=reform(vis3[k,*]*conj(vis5[k,*])*conj(vis6[k,*]))
tvis4=reform(vis4[k,*]*conj(vis5[k,*])*conj(vis1[k,*]))
;
window,/free
set_screen
!y.range=[-90,90]
!p.multi=[0,2,4]
!p.charsize=2.0
rad=180/!pi
;
plot,rad*cphase(add(tv1,100)),psym=1
plot,rad*cphase(add(tvis1,10)),psym=1
plot,rad*cphase(add(tv2,100)),psym=1
plot,rad*cphase(add(tvis2,10)),psym=1
plot,rad*cphase(add(tv3,100)),psym=1
plot,rad*cphase(add(tvis3,10)),psym=1
plot,rad*cphase(add(tv4,100)),psym=1
plot,rad*cphase(add(tvis4,10)),psym=1
;
end
;-------------------------------------------------------------------------------
pro altair3a
;
; closure fringe delays
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
get_rawdata,'2002-07-07.raw.062'
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
ob=1
ch=channelindex[genconfig.spectrometerid[ob]]-1
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
get_bincounts,2
hds_close
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
!p.charsize=1.0
;
f1=fdldelay(genconfig.baselineid[0,ob])
v1=fringevis(bincounts[*,ch,*],genconfig.fringemod[0,ob])
p2d1=lambdawhite(v1,l)/(2*!pi)
g1b=groupdelay(box(abs(visdft(v1,l,d)),10,1),l,d,g1i)
g1p=g1b+phaseshift(f1-g1b,l)*p2d1
p1=cphase(total(visrotate(v1,l,g1p),1))
c1=phasescan(t,p1)
g1pca=g1p-c1*p2d1
plot,t,cphase(total(visrotate(v1,l,g1pca),1)),psym=3,ystyle=1
p1a=cphase(total(visrotate(v1,l,g1pca),1))
g1pc=g1pca-p1a*p2d1
;
f2=fdldelay(genconfig.baselineid[1,ob])
v2=fringevis(bincounts[*,ch,*],genconfig.fringemod[1,ob])
p2d2=lambdawhite(v2,l)/(2*!pi)
g2b=groupdelay(box(abs(visdft(v2,l,d)),10,1),l,d,g2i)
g2p=g2b+phaseshift(f2-g2b,l)*p2d2
p2=cphase(total(visrotate(v2,l,g2p),1))
c2=phasescan(t,p2)
g2pc=g2p-c2*p2d2
plot,t,cphase(total(visrotate(v2,l,g2pc),1)),psym=3,ystyle=1
p2=cphase(total(visrotate(v2,l,g2pc),1))
g2pc=g2pc-p2*p2d2
;
f3=fdldelay(genconfig.baselineid[2,ob])
v3=fringevis(bincounts[*,ch,*],genconfig.fringemod[2,ob])
p2d3=lambdawhite(v3,l)/(2*!pi)
g3b=groupdelay(box(abs(visdft(v3,l,d)),10,1),l,d,g3i)
g3p=g3b+phaseshift(f3-g3b,l)*p2d3
p3=cphase(total(visrotate(v3,l,g3p),1))
c3=phasescan(t,p3)
g3pc=g3p-c3*p2d3
plot,t,cphase(total(visrotate(v3,l,g3pc),1)),psym=3,ystyle=1
p3=cphase(total(visrotate(v3,l,g3pc),1))
g3pc=g3pc-p3*p2d3
;
f4=fdldelay(genconfig.baselineid[3,ob])
v4=fringevis(bincounts[*,ch,*],genconfig.fringemod[3,ob])
p2d4=lambdawhite(v4,l)/(2*!pi)
g4b=groupdelay(box(abs(visdft(v4,l,d)),10,1),l,d,g4i)
g4p=g4b+phaseshift(f4-g4b,l)*p2d4
p4=cphase(total(visrotate(v4,l,g4p),1))
c4=phasescan(t,p4)
g4pc=g4p-c4*p2d4
plot,t,cphase(total(visrotate(v4,l,g4pc),1)),psym=3,ystyle=1
p4=cphase(total(visrotate(v4,l,g4pc),1))
g4pc=g4pc-p4*p2d4
;
f5=fdldelay(genconfig.baselineid[4,ob])
v5=fringevis(bincounts[*,ch,*],genconfig.fringemod[4,ob])
p2d5=lambdawhite(v5,l)/(2*!pi)
g5b=groupdelay(box(abs(visdft(v5,l,d)),10,1),l,d,g5i)
g5p=g5b+phaseshift(f5-g5b,l)*p2d5
p5=cphase(total(visrotate(v5,l,g5p),1))
c5=phasescan(t,p5)
g5pc=g5p-c5*p2d5
plot,t,cphase(total(visrotate(v5,l,g5pc),1)),psym=3,ystyle=1
p5=cphase(total(visrotate(v5,l,g5pc),1))
g5pc=g5pc-p5*p2d5
;
f6=fdldelay(genconfig.baselineid[5,ob])
v6=fringevis(bincounts[*,ch,*],genconfig.fringemod[5,ob])
p2d6=lambdawhite(v6,l)/(2*!pi)
g6b=groupdelay(box(abs(visdft(v6,l,d)),10,1),l,d,g6i)
g6p=g6b+phaseshift(f6-g6b,l)*p2d6
p6=cphase(total(visrotate(v6,l,g6p),1))
c6=phasescan(t,p6)
g6pc=g6p-c6*p2d6
plot,t,cphase(total(visrotate(v6,l,g6pc),1)),psym=3,ystyle=1
p6=cphase(total(visrotate(v6,l,g6pc),1))
g6pc=g6pc-p6*p2d6
;
g1pc=g1pc*1e6
g2pc=g2pc*1e6
g3pc=g3pc*1e6
g4pc=g4pc*1e6
g5pc=g5pc*1e6
g6pc=g6pc*1e6
;
t1=g1pc-g2pc-g6pc
t2=g2pc+g3pc-g4pc
t3=g3pc-g5pc-g6pc
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
index1=where(g1i eq 1 and g2i eq 1 and g6i eq 1 $
	and abs(t1-t1m) gt t1me)
index2=where(g2i eq 1 and g3i eq 1 and g4i eq 1 $
	and abs(t2-t2m) gt t2me)
index3=where(g3i eq 1 and g5i eq 1 and g6i eq 1 $
	and abs(t3-t3m) gt t3me)
g1i2=g1i
g2i2=g2i
g3i2=g3i
g4i2=g4i
g5i2=g5i
g6i2=g6i
g1i2[index1]=0 & g2i2[index1]=0 & g6i2[index1]=0
g2i2[index2]=0 & g3i2[index2]=0 & g4i2[index2]=0
g3i2[index3]=0 & g5i2[index3]=0 & g6i2[index3]=0
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
se2=fltarr(n)
sac=fltarr(n)
saw=fltarr(n)
sw7=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saci=intarr(n)+1
sawi=intarr(n)+1
sw7i=intarr(n)+1
solni=intarr(4,n)+1
;
dm=transpose([[1,-1,0,0], $
	      [0,-1,1,0], $
	      [0,0,-1,1], $
	      [0,-1,0,1], $
	      [-1,0,0,1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
wf=[0.1,1.0,1.0,0.1,0.1,1.0,1.0]
weight=weight*wf
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1pc[i],g2pc[i],g3pc[i],g4pc[i],g5pc[i],g6pc[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
index=where(solni[0,*] eq 1 and solni[1,*] eq 1 $
	and solni[2,*] eq 1 and solni[3,*] eq 1)
se2=reform(solns[0,index])
sac=reform(solns[1,index])
saw=reform(solns[2,index])
sw7=reform(solns[3,index])
;
t=t[index]
;
g1g=se2-sac
g2g=saw-sac
g3g=sw7-saw
g4g=sw7-sac
g5g=sw7-se2
g6g=se2-saw
;
g1g=g1g/1e6
g2g=g2g/1e6
g3g=g3g/1e6
g4g=g4g/1e6
g5g=g5g/1e6
g6g=g6g/1e6
;
cohint=10
;
vis1=add(visrotate(v1[*,index],l,g1g),cohint,1)
vis2=add(visrotate(v2[*,index],l,g2g),cohint,1)
vis3=add(visrotate(v3[*,index],l,g3g),cohint,1)
vis4=add(visrotate(v4[*,index],l,g4g),cohint,1)
vis5=add(visrotate(v5[*,index],l,g5g),cohint,1)
vis6=add(visrotate(v6[*,index],l,g6g),cohint,1)
;
k=0
tv1=reform(v1[k,*]*conj(v2[k,*])*conj(v6[k,*]))
tv2=reform(v2[k,*]*v3[k,*]*conj(v4[k,*]))
tv3=reform(v3[k,*]*conj(v5[k,*])*conj(v6[k,*]))
tv4=reform(v4[k,*]*conj(v5[k,*])*conj(v1[k,*]))
;
tvis1=reform(vis1[k,*]*conj(vis2[k,*])*conj(vis6[k,*]))
tvis2=reform(vis2[k,*]*vis3[k,*]*conj(vis4[k,*]))
tvis3=reform(vis3[k,*]*conj(vis5[k,*])*conj(vis6[k,*]))
tvis4=reform(vis4[k,*]*conj(vis5[k,*])*conj(vis1[k,*]))
;
window,/free
set_screen
!y.range=[-180,180]
!p.multi=[0,2,4]
!p.charsize=2.0
rad=180/!pi
;
plot,rad*cphase(add(tv1,100)),psym=3
plot,rad*cphase(add(tvis1,10)),psym=3
plot,rad*cphase(add(tv2,100)),psym=3
plot,rad*cphase(add(tvis2,10)),psym=3
plot,rad*cphase(add(tv3,100)),psym=3
plot,rad*cphase(add(tvis3,10)),psym=3
plot,rad*cphase(add(tv4,100)),psym=3
plot,rad*cphase(add(tvis4,10)),psym=3
;
end
;-------------------------------------------------------------------------------
pro altair3b
;
; closure fringe delays
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
get_rawdata,'2002-08-14.raw.043'
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
ob=0
ch=channelindex[genconfig.spectrometerid[ob]]-1
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
get_bincounts,1
hds_close
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
!p.charsize=1.0
;
f1=fdldelay(genconfig.baselineid[0,ob])
v1=fringevis(bincounts[*,ch,*],genconfig.fringemod[0,ob])
p2d1=lambdawhite(v1,l)/(2*!pi)
g1b=groupdelay(box(abs(visdft(v1,l,d)),10,1),l,d,g1i)
g1p=g1b+phaseshift(f1-g1b,l)*p2d1
p1=cphase(total(visrotate(v1,l,g1p),1))
c1=phasescan(t,p1)
g1pca=g1p-c1*p2d1
plot,t,cphase(total(visrotate(v1,l,g1pca),1)),psym=3,ystyle=1
p1a=cphase(total(visrotate(v1,l,g1pca),1))
g1pc=g1pca-p1a*p2d1
;
f2=fdldelay(genconfig.baselineid[1,ob])
v2=fringevis(bincounts[*,ch,*],genconfig.fringemod[1,ob])
p2d2=lambdawhite(v2,l)/(2*!pi)
g2b=groupdelay(box(abs(visdft(v2,l,d)),10,1),l,d,g2i)
g2p=g2b+phaseshift(f2-g2b,l)*p2d2
p2=cphase(total(visrotate(v2,l,g2p),1))
c2=phasescan(t,p2)
g2pca=g2p-c2*p2d2
plot,t,cphase(total(visrotate(v2,l,g2pca),1)),psym=3,ystyle=1
p2a=cphase(total(visrotate(v2,l,g2pca),1))
g2pc=g2pca-p2a*p2d2
;
f3=fdldelay(genconfig.baselineid[2,ob])
v3=fringevis(bincounts[*,ch,*],genconfig.fringemod[2,ob])
p2d3=lambdawhite(v3,l)/(2*!pi)
g3b=groupdelay(box(abs(visdft(v3,l,d)),10,1),l,d,g3i)
g3p=g3b+phaseshift(f3-g3b,l)*p2d3
p3=cphase(total(visrotate(v3,l,g3p),1))
c3=phasescan(t,p3)
g3pca=g3p-c3*p2d3
plot,t,cphase(total(visrotate(v3,l,g3pca),1)),psym=3,ystyle=1
p3a=cphase(total(visrotate(v3,l,g3pca),1))
g3pc=g3pca-p3a*p2d3
;
f4=fdldelay(genconfig.baselineid[3,ob])
v4=fringevis(bincounts[*,ch,*],genconfig.fringemod[3,ob])
p2d4=lambdawhite(v4,l)/(2*!pi)
g4b=groupdelay(box(abs(visdft(v4,l,d)),10,1),l,d,g4i)
g4p=g4b+phaseshift(f4-g4b,l)*p2d4
p4=cphase(total(visrotate(v4,l,g4p),1))
c4=phasescan(t,p4)
g4pca=g4p-c4*p2d4
plot,t,cphase(total(visrotate(v4,l,g4pca),1)),psym=3,ystyle=1
p4a=cphase(total(visrotate(v4,l,g4pca),1))
g4pc=g4pca-p4a*p2d4
;
f5=fdldelay(genconfig.baselineid[4,ob])
v5=fringevis(bincounts[*,ch,*],genconfig.fringemod[4,ob])
p2d5=lambdawhite(v5,l)/(2*!pi)
g5b=groupdelay(box(abs(visdft(v5,l,d)),10,1),l,d,g5i)
g5p=g5b+phaseshift(f5-g5b,l)*p2d5
p5=cphase(total(visrotate(v5,l,g5p),1))
c5=phasescan(t,p5)
g5pca=g5p-c5*p2d5
plot,t,cphase(total(visrotate(v5,l,g5pca),1)),psym=3,ystyle=1
p5a=cphase(total(visrotate(v5,l,g5pca),1))
g5pc=g5pca-p5a*p2d5
;
f6=fdldelay(genconfig.baselineid[5,ob])
v6=fringevis(bincounts[*,ch,*],genconfig.fringemod[5,ob])
p2d6=lambdawhite(v6,l)/(2*!pi)
g6b=groupdelay(box(abs(visdft(v6,l,d)),10,1),l,d,g6i)
g6p=g6b+phaseshift(f6-g6b,l)*p2d6
p6=cphase(total(visrotate(v6,l,g6p),1))
c6=phasescan(t,p6)
g6pca=g6p-c6*p2d6
plot,t,cphase(total(visrotate(v6,l,g6pca),1)),psym=3,ystyle=1
p6a=cphase(total(visrotate(v6,l,g6pca),1))
g6pc=g6pca-p6a*p2d6
;
g1pc=g1pc*1e6
g2pc=g2pc*1e6
g3pc=g3pc*1e6
g4pc=g4pc*1e6
g5pc=g5pc*1e6
g6pc=g6pc*1e6
;
t1=g1pc-g2pc+g6pc
t2=g2pc-g3pc-g4pc
t3=g3pc+g5pc-g6pc
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
index1=where(g1i eq 1 and g2i eq 1 and g6i eq 1 $
	and abs(t1-t1m) gt t1me)
index2=where(g2i eq 1 and g3i eq 1 and g4i eq 1 $
	and abs(t2-t2m) gt t2me)
index3=where(g3i eq 1 and g5i eq 1 and g6i eq 1 $
	and abs(t3-t3m) gt t3me)
g1i2=g1i
g2i2=g2i
g3i2=g3i
g4i2=g4i
g5i2=g5i
g6i2=g6i
g1i2[index1]=0 & g2i2[index1]=0 & g6i2[index1]=0
g2i2[index2]=0 & g3i2[index2]=0 & g4i2[index2]=0
g3i2[index3]=0 & g5i2[index3]=0 & g6i2[index3]=0
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
se2=fltarr(n)
sae=fltarr(n)
saw=fltarr(n)
san=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saei=intarr(n)+1
sawi=intarr(n)+1
sani=intarr(n)+1
solni=intarr(4,n)+1
;
dm=transpose([[-1,1,0,0], $
	      [0,1,-1,0], $
	      [0,0,-1,1], $
	      [0,1,0,-1], $
	      [1,0,0,-1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
wf=[0.1,1.0,1.0,0.1,0.1,1.0,1.0]
weight=weight*wf
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1pc[i],g2pc[i],g3pc[i],g4pc[i],g5pc[i],g6pc[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
index=where(solni[0,*] eq 1 and solni[1,*] eq 1 $
	and solni[2,*] eq 1 and solni[3,*] eq 1)
se2=reform(solns[0,index])
sae=reform(solns[1,index])
saw=reform(solns[2,index])
san=reform(solns[3,index])
;
t=t[index]
;
g1g=sae-se2
g2g=sae-saw
g3g=san-saw
g4g=sae-san
g5g=se2-san
g6g=se2-saw
;
g1g=g1g/1e6
g2g=g2g/1e6
g3g=g3g/1e6
g4g=g4g/1e6
g5g=g5g/1e6
g6g=g6g/1e6
;
cohint=10
;
vis1=add(visrotate(v1[*,index],l,g1g),cohint,1)
vis2=add(visrotate(v2[*,index],l,g2g),cohint,1)
vis3=add(visrotate(v3[*,index],l,g3g),cohint,1)
vis4=add(visrotate(v4[*,index],l,g4g),cohint,1)
vis5=add(visrotate(v5[*,index],l,g5g),cohint,1)
vis6=add(visrotate(v6[*,index],l,g6g),cohint,1)
;
k=0
tv1=reform(v1[k,*]*conj(v2[k,*])*v6[k,*])
tv2=reform(v2[k,*]*conj(v3[k,*])*conj(v4[k,*]))
tv3=reform(v3[k,*]*v5[k,*]*conj(v6[k,*]))
tv4=reform(v4[k,*]*conj(v5[k,*])*conj(v1[k,*]))
;
tvis1=reform(vis1[k,*]*conj(vis2[k,*])*vis6[k,*])
tvis2=reform(vis2[k,*]*conj(vis3[k,*])*conj(vis4[k,*]))
tvis3=reform(vis3[k,*]*vis5[k,*]*conj(vis6[k,*]))
tvis4=reform(vis4[k,*]*conj(vis5[k,*])*conj(vis1[k,*]))
;
window,/free
set_screen
!y.range=[-180,180]
!p.multi=[0,2,4]
!p.charsize=2.0
rad=180/!pi
;
plot,rad*cphase(add(tv1,100)),psym=3
plot,rad*cphase(add(tvis1,10)),psym=3
plot,rad*cphase(add(tv2,100)),psym=3
plot,rad*cphase(add(tvis2,10)),psym=3
plot,rad*cphase(add(tv3,100)),psym=3
plot,rad*cphase(add(tvis3,10)),psym=3
plot,rad*cphase(add(tv4,100)),psym=3
plot,rad*cphase(add(tvis4,10)),psym=3
;
end
;-------------------------------------------------------------------------------
pro altair3
;
; closure fringe delays
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
ob=OutputBeam-1
ch=channelindex[genconfig.spectrometerid[ob]]-1
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
!p.charsize=1.0
;
f1=fdldelay(genconfig.baselineid[0,ob])
v1=fringevis(bincounts[*,ch,*],genconfig.fringemod[0,ob])
p2d1=lambdawhite(v1,l)/(2*!pi)
g1b=groupdelay(box(abs(visdft(v1,l,d)),10,1),l,d,g1i)
g1p=g1b+phaseshift(f1-g1b,l)*p2d1
p1=cphase(total(visrotate(v1,l,g1p),1))
c1=phasescan(t,p1)
g1pca=g1p-c1*p2d1
plot,t,cphase(total(visrotate(v1,l,g1pca),1)),psym=3,ystyle=1
p1a=cphase(total(visrotate(v1,l,g1pca),1))
g1pc=g1pca-p1a*p2d1
;
f2=fdldelay(genconfig.baselineid[1,ob])
v2=fringevis(bincounts[*,ch,*],genconfig.fringemod[1,ob])
p2d2=lambdawhite(v2,l)/(2*!pi)
g2b=groupdelay(box(abs(visdft(v2,l,d)),10,1),l,d,g2i)
g2p=g2b+phaseshift(f2-g2b,l)*p2d2
p2=cphase(total(visrotate(v2,l,g2p),1))
c2=phasescan(t,p2)
g2pca=g2p-c2*p2d2
plot,t,cphase(total(visrotate(v2,l,g2pca),1)),psym=3,ystyle=1
p2a=cphase(total(visrotate(v2,l,g2pca),1))
g2pc=g2pca-p2a*p2d2
;
f3=fdldelay(genconfig.baselineid[2,ob])
v3=fringevis(bincounts[*,ch,*],genconfig.fringemod[2,ob])
p2d3=lambdawhite(v3,l)/(2*!pi)
g3b=groupdelay(box(abs(visdft(v3,l,d)),10,1),l,d,g3i)
g3p=g3b+phaseshift(f3-g3b,l)*p2d3
p3=cphase(total(visrotate(v3,l,g3p),1))
c3=phasescan(t,p3)
g3pca=g3p-c3*p2d3
plot,t,cphase(total(visrotate(v3,l,g3pca),1)),psym=3,ystyle=1
p3a=cphase(total(visrotate(v3,l,g3pca),1))
g3pc=g3pca-p3a*p2d3
;
f4=fdldelay(genconfig.baselineid[3,ob])
v4=fringevis(bincounts[*,ch,*],genconfig.fringemod[3,ob])
p2d4=lambdawhite(v4,l)/(2*!pi)
g4b=groupdelay(box(abs(visdft(v4,l,d)),10,1),l,d,g4i)
g4p=g4b+phaseshift(f4-g4b,l)*p2d4
p4=cphase(total(visrotate(v4,l,g4p),1))
c4=phasescan(t,p4)
g4pca=g4p-c4*p2d4
plot,t,cphase(total(visrotate(v4,l,g4pca),1)),psym=3,ystyle=1
p4a=cphase(total(visrotate(v4,l,g4pca),1))
g4pc=g4pca-p4a*p2d4
;
f5=fdldelay(genconfig.baselineid[4,ob])
v5=fringevis(bincounts[*,ch,*],genconfig.fringemod[4,ob])
p2d5=lambdawhite(v5,l)/(2*!pi)
g5b=groupdelay(box(abs(visdft(v5,l,d)),10,1),l,d,g5i)
g5p=g5b+phaseshift(f5-g5b,l)*p2d5
p5=cphase(total(visrotate(v5,l,g5p),1))
c5=phasescan(t,p5)
g5pca=g5p-c5*p2d5
plot,t,cphase(total(visrotate(v5,l,g5pca),1)),psym=3,ystyle=1
p5a=cphase(total(visrotate(v5,l,g5pca),1))
g5pc=g5pca-p5a*p2d5
;
f6=fdldelay(genconfig.baselineid[5,ob])
v6=fringevis(bincounts[*,ch,*],genconfig.fringemod[5,ob])
p2d6=lambdawhite(v6,l)/(2*!pi)
g6b=groupdelay(box(abs(visdft(v6,l,d)),10,1),l,d,g6i)
g6p=g6b+phaseshift(f6-g6b,l)*p2d6
p6=cphase(total(visrotate(v6,l,g6p),1))
c6=phasescan(t,p6)
g6pca=g6p-c6*p2d6
plot,t,cphase(total(visrotate(v6,l,g6pca),1)),psym=3,ystyle=1
p6a=cphase(total(visrotate(v6,l,g6pca),1))
g6pc=g6pca-p6a*p2d6
;
g1pc=g1pc*1e6
g2pc=g2pc*1e6
g3pc=g3pc*1e6
g4pc=g4pc*1e6
g5pc=g5pc*1e6
g6pc=g6pc*1e6
;
if OutputBeam eq 1 then begin
	t1=g1pc-g2pc+g6pc
	t2=g2pc-g3pc-g4pc
	t3=g3pc+g5pc-g6pc
endif else begin
	t1=g1pc-g2pc-g6pc
	t2=g2pc+g3pc-g4pc
	t3=g3pc-g5pc-g6pc
endelse
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
index1=where(g1i eq 1 and g2i eq 1 and g6i eq 1 $
	and abs(t1-t1m) gt t1me)
index2=where(g2i eq 1 and g3i eq 1 and g4i eq 1 $
	and abs(t2-t2m) gt t2me)
index3=where(g3i eq 1 and g5i eq 1 and g6i eq 1 $
	and abs(t3-t3m) gt t3me)
g1i2=g1i
g2i2=g2i
g3i2=g3i
g4i2=g4i
g5i2=g5i
g6i2=g6i
g1i2[index1]=0 & g2i2[index1]=0 & g6i2[index1]=0
g2i2[index2]=0 & g3i2[index2]=0 & g4i2[index2]=0
g3i2[index3]=0 & g5i2[index3]=0 & g6i2[index3]=0
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
if OutputBeam eq 1 then begin
	se2=fltarr(n)
	sae=fltarr(n)
	saw=fltarr(n)
	san=fltarr(n)
;
	se2i=intarr(n)+1
	saei=intarr(n)+1
	sawi=intarr(n)+1
	sani=intarr(n)+1
;
	dm=transpose([[-1,1,0,0], $
		      [0,1,-1,0], $
		      [0,0,-1,1], $
		      [0,1,0,-1], $
		      [1,0,0,-1], $
		      [1,0,-1,0], $
		      [0,0,1,0]])
endif else begin
;
	se2=fltarr(n)
	sac=fltarr(n)
	saw=fltarr(n)
	sw7=fltarr(n)
;
	se2i=intarr(n)+1
	saci=intarr(n)+1
	sawi=intarr(n)+1
	sw7i=intarr(n)+1
;
	dm=transpose([[1,-1,0,0], $
		      [0,-1,1,0], $
		      [0,0,-1,1], $
		      [0,-1,0,1], $
		      [-1,0,0,1], $
		      [1,0,-1,0], $
		      [0,0,1,0]])
endelse
;
solns=fltarr(4,n)
solni=intarr(4,n)+1
;
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
wf=[0.1,1.0,1.0,0.1,0.1,1.0,1.0]
;weight=weight*wf
weight=weight^2
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1pc[i],g2pc[i],g3pc[i],g4pc[i],g5pc[i],g6pc[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
index=where(solni[0,*] eq 1 and solni[1,*] eq 1 $
	and solni[2,*] eq 1 and solni[3,*] eq 1)
t=t[index]
;
if OutputBeam eq 1 then begin
	se2=reform(solns[0,index])
	sae=reform(solns[1,index])
	saw=reform(solns[2,index])
	san=reform(solns[3,index])
;
	g1g=sae-se2
	g2g=sae-saw
	g3g=san-saw
	g4g=sae-san
	g5g=se2-san
	g6g=se2-saw
endif else begin
	se2=reform(solns[0,index])
	sac=reform(solns[1,index])
	saw=reform(solns[2,index])
	sw7=reform(solns[3,index])
;
	g1g=se2-sac
	g2g=saw-sac
	g3g=sw7-saw
	g4g=sw7-sac
	g5g=sw7-se2
	g6g=se2-saw
endelse
;
g1g=g1g/1e6
g2g=g2g/1e6
g3g=g3g/1e6
g4g=g4g/1e6
g5g=g5g/1e6
g6g=g6g/1e6
;
cohint=10
;
vis1=add(visrotate(v1[*,index],l,g1g),cohint,1)
vis2=add(visrotate(v2[*,index],l,g2g),cohint,1)
vis3=add(visrotate(v3[*,index],l,g3g),cohint,1)
vis4=add(visrotate(v4[*,index],l,g4g),cohint,1)
vis5=add(visrotate(v5[*,index],l,g5g),cohint,1)
vis6=add(visrotate(v6[*,index],l,g6g),cohint,1)
;
k=0
if OutputBeam eq 1 then begin
	tv1=reform(v1[k,*]*conj(v2[k,*])*v6[k,*])
	tv2=reform(v2[k,*]*conj(v3[k,*])*conj(v4[k,*]))
	tv3=reform(v3[k,*]*v5[k,*]*conj(v6[k,*]))
	tv4=reform(v4[k,*]*conj(v5[k,*])*conj(v1[k,*]))
;
	tvis1=reform(vis1[k,*]*conj(vis2[k,*])*vis6[k,*])
	tvis2=reform(vis2[k,*]*conj(vis3[k,*])*conj(vis4[k,*]))
	tvis3=reform(vis3[k,*]*vis5[k,*]*conj(vis6[k,*]))
	tvis4=reform(vis4[k,*]*conj(vis5[k,*])*conj(vis1[k,*]))
endif else begin
	tv1=reform(v1[k,*]*conj(v2[k,*])*conj(v6[k,*]))
	tv2=reform(v2[k,*]*v3[k,*]*conj(v4[k,*]))
	tv3=reform(v3[k,*]*conj(v5[k,*])*conj(v6[k,*]))
	tv4=reform(v4[k,*]*conj(v5[k,*])*conj(v1[k,*]))
;
	tvis1=reform(vis1[k,*]*conj(vis2[k,*])*conj(vis6[k,*]))
	tvis2=reform(vis2[k,*]*vis3[k,*]*conj(vis4[k,*]))
	tvis3=reform(vis3[k,*]*conj(vis5[k,*])*conj(vis6[k,*]))
	tvis4=reform(vis4[k,*]*conj(vis5[k,*])*conj(vis1[k,*]))
endelse
;
window,/free
set_screen
!y.range=[-180,180]
!p.multi=[0,2,4]
!p.charsize=2.0
rad=180/!pi
;
plot,rad*cphase(add(tv1,100)),psym=3
plot,rad*cphase(add(tvis1,10)),psym=3
plot,rad*cphase(add(tv2,100)),psym=3
plot,rad*cphase(add(tvis2,10)),psym=3
plot,rad*cphase(add(tv3,100)),psym=3
plot,rad*cphase(add(tvis3,10)),psym=3
plot,rad*cphase(add(tv4,100)),psym=3
plot,rad*cphase(add(tvis4,10)),psym=3
;
end
;-------------------------------------------------------------------------------
pro altair4
;
; closure fringe delay with previous solution for groupdelay
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
get_rawdata,'2002-07-07.raw.062'
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
ob=1
ch=channelindex[genconfig.spectrometerid[ob]]-1
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
get_bincounts,2
hds_close
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
!p.charsize=1.0
;
f1=fdldelay(genconfig.baselineid[0,ob])
v1=fringevis(bincounts[*,ch,*],genconfig.fringemod[0,ob])
g1b=groupdelay(box(abs(visdft(v1,l,d)),10,1),l,d,g1i)
f2=fdldelay(genconfig.baselineid[1,ob])
v2=fringevis(bincounts[*,ch,*],genconfig.fringemod[1,ob])
g2b=groupdelay(box(abs(visdft(v2,l,d)),10,1),l,d,g2i)
f3=fdldelay(genconfig.baselineid[2,ob])
v3=fringevis(bincounts[*,ch,*],genconfig.fringemod[2,ob])
g3b=groupdelay(box(abs(visdft(v3,l,d)),10,1),l,d,g3i)
f4=fdldelay(genconfig.baselineid[3,ob])
v4=fringevis(bincounts[*,ch,*],genconfig.fringemod[3,ob])
g4b=groupdelay(box(abs(visdft(v4,l,d)),10,1),l,d,g4i)
f5=fdldelay(genconfig.baselineid[4,ob])
v5=fringevis(bincounts[*,ch,*],genconfig.fringemod[4,ob])
g5b=groupdelay(box(abs(visdft(v5,l,d)),10,1),l,d,g5i)
f6=fdldelay(genconfig.baselineid[5,ob])
v6=fringevis(bincounts[*,ch,*],genconfig.fringemod[5,ob])
g6b=groupdelay(box(abs(visdft(v6,l,d)),10,1),l,d,g6i)
;
g1b=g1b*1e6
g2b=g2b*1e6
g3b=g3b*1e6
g4b=g4b*1e6
g5b=g5b*1e6
g6b=g6b*1e6
;
t1=g1b-g2b-g6b
t2=g2b+g3b-g4b
t3=g3b-g5b-g6b
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
g1b=g1b-t1m
g4b=g4b+t2m
g5b=g5b+t3m
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
index1=where(g1i eq 1 and g2i eq 1 and g6i eq 1 $
	and abs(t1-t1m) gt t1me)
index2=where(g2i eq 1 and g3i eq 1 and g4i eq 1 $
	and abs(t2-t2m) gt t2me)
index3=where(g3i eq 1 and g5i eq 1 and g6i eq 1 $
	and abs(t3-t3m) gt t3me)
g1i2=g1i
g2i2=g2i
g3i2=g3i
g4i2=g4i
g5i2=g5i
g6i2=g6i
g1i2[index1]=0 & g2i2[index1]=0 & g6i2[index1]=0
g2i2[index2]=0 & g3i2[index2]=0 & g4i2[index2]=0
g3i2[index3]=0 & g5i2[index3]=0 & g6i2[index3]=0
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
se2=fltarr(n)
sac=fltarr(n)
saw=fltarr(n)
sw7=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saci=intarr(n)+1
sawi=intarr(n)+1
sw7i=intarr(n)+1
solni=intarr(4,n)+1
;
dm=transpose([[1,-1,0,0], $
	      [0,-1,1,0], $
	      [0,0,-1,1], $
	      [0,-1,0,1], $
	      [-1,0,0,1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1b[i],g2b[i],g3b[i],g4b[i],g5b[i],g6b[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
se2=reform(solns[0,*])
sac=reform(solns[1,*])
saw=reform(solns[2,*])
sw7=reform(solns[3,*])
;
g1g=se2-sac
g2g=saw-sac
g3g=sw7-saw
g4g=sw7-sac
g5g=sw7-se2
g6g=se2-saw
;
g1g=g1g+t1m
g4g=g4g-t2m
g5g=g5g-t3m
;
g1b=g1g/1e6
g2b=g2g/1e6
g3b=g3g/1e6
g4b=g4g/1e6
g5b=g5g/1e6
g6b=g6g/1e6
;
p2d1=lambdawhite(v1,l)/(2*!pi)
g1p=g1b+phaseshift(f1-g1b,l)*p2d1
p1=cphase(total(visrotate(v1,l,g1p),1))
c1=phasescan(t,p1)
g1pca=g1p-c1*p2d1
; plot,t,cphase(total(visrotate(v1,l,g1pca),1)),psym=3,ystyle=1
p1a=cphase(total(visrotate(v1,l,g1pca),1))
g1pc=g1pca-p1a*p2d1
;
p2d2=lambdawhite(v2,l)/(2*!pi)
g2p=g2b+phaseshift(f2-g2b,l)*p2d2
p2=cphase(total(visrotate(v2,l,g2p),1))
c2=phasescan(t,p2)
g2pca=g2p-c2*p2d2
; plot,t,cphase(total(visrotate(v2,l,g2pca),1)),psym=3,ystyle=1
p2a=cphase(total(visrotate(v2,l,g2pca),1))
g2pc=g2pca-p2*p2d2
;
p2d3=lambdawhite(v3,l)/(2*!pi)
g3p=g3b+phaseshift(f3-g3b,l)*p2d3
p3=cphase(total(visrotate(v3,l,g3p),1))
c3=phasescan(t,p3)
g3pca=g3p-c3*p2d3
; plot,t,cphase(total(visrotate(v3,l,g3pca),1)),psym=3,ystyle=1
p3a=cphase(total(visrotate(v3,l,g3pca),1))
g3pc=g3pca-p3a*p2d3
;
p2d4=lambdawhite(v4,l)/(2*!pi)
g4p=g4b+phaseshift(f4-g4b,l)*p2d4
p4=cphase(total(visrotate(v4,l,g4p),1))
c4=phasescan(t,p4)
g4pca=g4p-c4*p2d4
; plot,t,cphase(total(visrotate(v4,l,g4pca),1)),psym=3,ystyle=1
p4a=cphase(total(visrotate(v4,l,g4pca),1))
g4pc=g4pca-p4a*p2d4
;
p2d5=lambdawhite(v5,l)/(2*!pi)
g5p=g5b+phaseshift(f5-g5b,l)*p2d5
p5=cphase(total(visrotate(v5,l,g5p),1))
c5=phasescan(t,p5)
g5pca=g5p-c5*p2d5
; plot,t,cphase(total(visrotate(v5,l,g5pca),1)),psym=3,ystyle=1
p5a=cphase(total(visrotate(v5,l,g5pca),1))
g5pc=g5pca-p5a*p2d5
;
p2d6=lambdawhite(v6,l)/(2*!pi)
g6p=g6b+phaseshift(f6-g6b,l)*p2d6
p6=cphase(total(visrotate(v6,l,g6p),1))
c6=phasescan(t,p6)
g6pca=g6p-c6*p2d6
; plot,t,cphase(total(visrotate(v6,l,g6pca),1)),psym=3,ystyle=1
p6a=cphase(total(visrotate(v6,l,g6pca),1))
g6pc=g6pca-p6*p2d6
;
g1pc=g1pc*1e6
g2pc=g2pc*1e6
g3pc=g3pc*1e6
g4pc=g4pc*1e6
g5pc=g5pc*1e6
g6pc=g6pc*1e6
;
t1=g1pc-g2pc-g6pc
t2=g2pc+g3pc-g4pc
t3=g3pc-g5pc-g6pc
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
se2=fltarr(n)
sac=fltarr(n)
saw=fltarr(n)
sw7=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saci=intarr(n)+1
sawi=intarr(n)+1
sw7i=intarr(n)+1
solni=intarr(4,n)+1
;
dm=transpose([[1,-1,0,0], $
	      [0,-1,1,0], $
	      [0,0,-1,1], $
	      [0,-1,0,1], $
	      [-1,0,0,1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
wf=[0.2,1.0,1.0,0.2,0.2,1.0,1.0]
; weight=weight*wf
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1pc[i],g2pc[i],g3pc[i],g4pc[i],g5pc[i],g6pc[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
index=where(solni[0,*] eq 1 and solni[1,*] eq 1 $
	and solni[2,*] eq 1 and solni[3,*] eq 1)
se2=reform(solns[0,index])
sac=reform(solns[1,index])
saw=reform(solns[2,index])
sw7=reform(solns[3,index])
;
t=t[index]
;
g1g=se2-sac
g2g=saw-sac
g3g=sw7-saw
g4g=sw7-sac
g5g=sw7-se2
g6g=se2-saw
;
g1g=g1g/1e6
g2g=g2g/1e6
g3g=g3g/1e6
g4g=g4g/1e6
g5g=g5g/1e6
g6g=g6g/1e6
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
;
v1r=visrotate(v1[*,index],l,g1g)
v2r=visrotate(v2[*,index],l,g2g)
v3r=visrotate(v3[*,index],l,g3g)
v4r=visrotate(v4[*,index],l,g4g)
v5r=visrotate(v5[*,index],l,g5g)
v6r=visrotate(v6[*,index],l,g6g)
;
p1r=cphase(total(v1r,1))
p2r=cphase(total(v2r,1))
p3r=cphase(total(v3r,1))
p4r=cphase(total(v4r,1))
p5r=cphase(total(v5r,1))
p6r=cphase(total(v6r,1))
;
cohint=10
;
vis1=add(visrotate(v1[*,index],l,g1g),cohint,1)
vis2=add(visrotate(v2[*,index],l,g2g),cohint,1)
vis3=add(visrotate(v3[*,index],l,g3g),cohint,1)
vis4=add(visrotate(v4[*,index],l,g4g),cohint,1)
vis5=add(visrotate(v5[*,index],l,g5g),cohint,1)
vis6=add(visrotate(v6[*,index],l,g6g),cohint,1)
;
k=0
tv1=reform(v1[k,*]*conj(v2[k,*])*conj(v6[k,*]))
tv2=reform(v2[k,*]*v3[k,*]*conj(v4[k,*]))
tv3=reform(v3[k,*]*conj(v5[k,*])*conj(v6[k,*]))
tv4=reform(v4[k,*]*conj(v5[k,*])*conj(v1[k,*]))
;
tvis1=reform(vis1[k,*]*conj(vis2[k,*])*conj(vis6[k,*]))
tvis2=reform(vis2[k,*]*vis3[k,*]*conj(vis4[k,*]))
tvis3=reform(vis3[k,*]*conj(vis5[k,*])*conj(vis6[k,*]))
tvis4=reform(vis4[k,*]*conj(vis5[k,*])*conj(vis1[k,*]))
;
window,/free
set_screen
!y.range=[-90,90]
!p.multi=[0,2,4]
!p.charsize=2.0
rad=180/!pi
;
plot,rad*cphase(add(tv1,100)),psym=3
plot,rad*cphase(add(tvis1,10)),psym=3
plot,rad*cphase(add(tv2,100)),psym=3
plot,rad*cphase(add(tvis2,10)),psym=3
plot,rad*cphase(add(tv3,100)),psym=3
plot,rad*cphase(add(tvis3,10)),psym=3
plot,rad*cphase(add(tv4,100)),psym=3
plot,rad*cphase(add(tvis4,10)),psym=3
;
end
;-------------------------------------------------------------------------------
pro altair5
;
; closure fringe delay with previous solution for trackdelay
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
;
get_rawdata,'2002-07-07.raw.062'
t=fdlstamp()/1000. & t0=t[0] & t=t-t[0]
;
ob=1
ch=channelindex[genconfig.spectrometerid[ob]]-1
l=genconfig.wavelength[ch,ob]
d=gitter(121,-12e-6)
;
get_bincounts,2
hds_close
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
!p.charsize=1.0
;
f1=fdldelay(genconfig.baselineid[0,ob])
v1=fringevis(bincounts[*,ch,*],genconfig.fringemod[0,ob])
p2d1=lambdawhite(v1,l)/(2*!pi)
g1b=groupdelay(box(abs(visdft(v1,l,d)),10,1),l,d,g1i)
g1p=g1b+phaseshift(f1-g1b,l)*p2d1
p1=cphase(total(visrotate(v1,l,g1p),1))
c1=phasescan(t,p1)
g1pca=g1p-c1*p2d1
;
f2=fdldelay(genconfig.baselineid[1,ob])
v2=fringevis(bincounts[*,ch,*],genconfig.fringemod[1,ob])
p2d2=lambdawhite(v2,l)/(2*!pi)
g2b=groupdelay(box(abs(visdft(v2,l,d)),10,1),l,d,g2i)
g2p=g2b+phaseshift(f2-g2b,l)*p2d2
p2=cphase(total(visrotate(v2,l,g2p),1))
c2=phasescan(t,p2)
g2pca=g2p-c2*p2d2
;
f3=fdldelay(genconfig.baselineid[2,ob])
v3=fringevis(bincounts[*,ch,*],genconfig.fringemod[2,ob])
p2d3=lambdawhite(v3,l)/(2*!pi)
g3b=groupdelay(box(abs(visdft(v3,l,d)),10,1),l,d,g3i)
g3p=g3b+phaseshift(f3-g3b,l)*p2d3
p3=cphase(total(visrotate(v3,l,g3p),1))
c3=phasescan(t,p3)
g3pca=g3p-c3*p2d3
;
f4=fdldelay(genconfig.baselineid[3,ob])
v4=fringevis(bincounts[*,ch,*],genconfig.fringemod[3,ob])
p2d4=lambdawhite(v4,l)/(2*!pi)
g4b=groupdelay(box(abs(visdft(v4,l,d)),10,1),l,d,g4i)
g4p=g4b+phaseshift(f4-g4b,l)*p2d4
p4=cphase(total(visrotate(v4,l,g4p),1))
c4=phasescan(t,p4)
g4pca=g4p-c4*p2d4
;
f5=fdldelay(genconfig.baselineid[4,ob])
v5=fringevis(bincounts[*,ch,*],genconfig.fringemod[4,ob])
p2d5=lambdawhite(v5,l)/(2*!pi)
g5b=groupdelay(box(abs(visdft(v5,l,d)),10,1),l,d,g5i)
g5p=g5b+phaseshift(f5-g5b,l)*p2d5
p5=cphase(total(visrotate(v5,l,g5p),1))
c5=phasescan(t,p5)
g5pca=g5p-c5*p2d5
;
f6=fdldelay(genconfig.baselineid[5,ob])
v6=fringevis(bincounts[*,ch,*],genconfig.fringemod[5,ob])
p2d6=lambdawhite(v6,l)/(2*!pi)
g6b=groupdelay(box(abs(visdft(v6,l,d)),10,1),l,d,g6i)
g6p=g6b+phaseshift(f6-g6b,l)*p2d6
p6=cphase(total(visrotate(v6,l,g6p),1))
c6=phasescan(t,p6)
g6pca=g6p-c6*p2d6
;
g1pca=g1pca*1e6
g2pca=g2pca*1e6
g3pca=g3pca*1e6
g4pca=g4pca*1e6
g5pca=g5pca*1e6
g6pca=g6pca*1e6
;
t1=g1pca-g2pca-g6pca
t2=g2pca+g3pca-g4pca
t3=g3pca-g5pca-g6pca
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
index1=where(g1i eq 1 and g2i eq 1 and g6i eq 1 $
	and abs(t1-t1m) gt t1me)
index2=where(g2i eq 1 and g3i eq 1 and g4i eq 1 $
	and abs(t2-t2m) gt t2me)
index3=where(g3i eq 1 and g5i eq 1 and g6i eq 1 $
	and abs(t3-t3m) gt t3me)
g1i2=g1i
g2i2=g2i
g3i2=g3i
g4i2=g4i
g5i2=g5i
g6i2=g6i
g1i2[index1]=0 & g2i2[index1]=0 & g6i2[index1]=0
g2i2[index2]=0 & g3i2[index2]=0 & g4i2[index2]=0
g3i2[index3]=0 & g5i2[index3]=0 & g6i2[index3]=0
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
se2=fltarr(n)
sac=fltarr(n)
saw=fltarr(n)
sw7=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saci=intarr(n)+1
sawi=intarr(n)+1
sw7i=intarr(n)+1
solni=intarr(4,n)+1
;
dm=transpose([[1,-1,0,0], $
	      [0,-1,1,0], $
	      [0,0,-1,1], $
	      [0,-1,0,1], $
	      [-1,0,0,1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1pca[i],g2pca[i],g3pca[i],g4pca[i],g5pca[i],g6pca[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
se2=reform(solns[0,*])
sac=reform(solns[1,*])
saw=reform(solns[2,*])
sw7=reform(solns[3,*])
;
g1g=se2-sac
g2g=saw-sac
g3g=sw7-saw
g4g=sw7-sac
g5g=sw7-se2
g6g=se2-saw
;
g1pca=g1g/1e6
g2pca=g2g/1e6
g3pca=g3g/1e6
g4pca=g4g/1e6
g5pca=g5g/1e6
g6pca=g6g/1e6
;
; plot,t,cphase(total(visrotate(v1,l,g1pca),1)),psym=3,ystyle=1
p1a=cphase(total(visrotate(v1,l,g1pca),1))
g1pc=g1pca-p1a*p2d1
;
; plot,t,cphase(total(visrotate(v2,l,g2pca),1)),psym=3,ystyle=1
p2a=cphase(total(visrotate(v2,l,g2pca),1))
g2pc=g2pca-p2*p2d2
;
; plot,t,cphase(total(visrotate(v3,l,g3pca),1)),psym=3,ystyle=1
p3a=cphase(total(visrotate(v3,l,g3pca),1))
g3pc=g3pca-p3a*p2d3
;
; plot,t,cphase(total(visrotate(v4,l,g4pca),1)),psym=3,ystyle=1
p4a=cphase(total(visrotate(v4,l,g4pca),1))
g4pc=g4pca-p4a*p2d4
;
; plot,t,cphase(total(visrotate(v5,l,g5pca),1)),psym=3,ystyle=1
p5a=cphase(total(visrotate(v5,l,g5pca),1))
g5pc=g5pca-p5a*p2d5
;
; plot,t,cphase(total(visrotate(v6,l,g6pca),1)),psym=3,ystyle=1
p6a=cphase(total(visrotate(v6,l,g6pca),1))
g6pc=g6pca-p6*p2d6
;
g1pc=g1pc*1e6
g2pc=g2pc*1e6
g3pc=g3pc*1e6
g4pc=g4pc*1e6
g5pc=g5pc*1e6
g6pc=g6pc*1e6
;
t1=g1pc-g2pc-g6pc
t2=g2pc+g3pc-g4pc
t3=g3pc-g5pc-g6pc
;
t1ib=where(g1i eq 0 or g2i eq 0 or g6i eq 0)
t2ib=where(g2i eq 0 or g3i eq 0 or g4i eq 0)
t3ib=where(g3i eq 0 or g5i eq 0 or g6i eq 0)
;
window,/free
set_screen
!p.charsize=2.0
!p.multi=[0,1,3]
!y.range=[-4,4]
!p.psym=3
plot,t,t1
oplot,t[t1ib],t1[t1ib],color=tci(2)
plot,t,t2
oplot,t[t2ib],t2[t2ib],color=tci(2)
plot,t,t3
oplot,t[t3ib],t3[t3ib],color=tci(2)
;
t1ig=where(g1i eq 1 and g2i eq 1 and g6i eq 1)
t2ig=where(g2i eq 1 and g3i eq 1 and g4i eq 1)
t3ig=where(g3i eq 1 and g5i eq 1 and g6i eq 1)
;
t1m=medianve(t1[t1ig],t1me)
t2m=medianve(t2[t2ig],t2me)
t3m=medianve(t3[t3ig],t3me)
;
t1me=t1me*4
t2me=t2me*4
t3me=t3me*4
;
n=n_elements(t)
wb1=total(g1i2)/n
wb2=total(g2i2)/n
wb3=total(g3i2)/n
wb4=total(g4i2)/n
wb5=total(g5i2)/n
wb6=total(g6i2)/n
;
se2=fltarr(n)
sac=fltarr(n)
saw=fltarr(n)
sw7=fltarr(n)
solns=fltarr(4,n)
;
se2i=intarr(n)+1
saci=intarr(n)+1
sawi=intarr(n)+1
sw7i=intarr(n)+1
solni=intarr(4,n)+1
;
dm=transpose([[1,-1,0,0], $
	      [0,-1,1,0], $
	      [0,0,-1,1], $
	      [0,-1,0,1], $
	      [-1,0,0,1], $
	      [1,0,-1,0], $
	      [0,0,1,0]])
dm=float(dm)
weight=[wb1,wb2,wb3,wb4,wb5,wb6,1]
wf=[0.2,1.0,1.0,0.2,0.2,1.0,1.0]
; weight=weight*wf
for j=0,n_elements(dm[0,*])-1 do dm[*,j]=dm[*,j]*sqrt(weight)
for i=0l,n-1 do begin
	gindex=[g1i2[i],g2i2[i],g3i2[i],g4i2[i],g5i2[i],g6i2[i],1]
	gright=[g1pc[i],g2pc[i],g3pc[i],g4pc[i],g5pc[i],g6pc[i],0]*sqrt(weight)
	index=where(gindex eq 1,count)
	if count gt 0 then begin
		m=dm[index,*]
		g=gright[index]
		tm=transpose(m)
		norm=tm#m
		r=tm#g
		svd,norm,w,u,v
		small=where(w lt max(w)*1e-6,count)
		for j=0,count-1 do begin
			w[small[j]]=0
			solni[*,i]=solni[*,i]*(1-nint(abs(v[*,small[j]])))
		endfor
		svbksb,u,w,v,r,s
		solns[*,i]=s
	endif	
endfor
;
index=where(solni[0,*] eq 1 and solni[1,*] eq 1 $
	and solni[2,*] eq 1 and solni[3,*] eq 1)
se2=reform(solns[0,index])
sac=reform(solns[1,index])
saw=reform(solns[2,index])
sw7=reform(solns[3,index])
;
t=t[index]
;
g1g=se2-sac
g2g=saw-sac
g3g=sw7-saw
g4g=sw7-sac
g5g=sw7-se2
g6g=se2-saw
;
g1g=g1g/1e6
g2g=g2g/1e6
g3g=g3g/1e6
g4g=g4g/1e6
g5g=g5g/1e6
g6g=g6g/1e6
;
window,/free
set_screen
!y.range=0
!p.multi=[0,2,3]
;
v1r=visrotate(v1[*,index],l,g1g)
v2r=visrotate(v2[*,index],l,g2g)
v3r=visrotate(v3[*,index],l,g3g)
v4r=visrotate(v4[*,index],l,g4g)
v5r=visrotate(v5[*,index],l,g5g)
v6r=visrotate(v6[*,index],l,g6g)
;
p1r=cphase(total(v1r,1))
p2r=cphase(total(v2r,1))
p3r=cphase(total(v3r,1))
p4r=cphase(total(v4r,1))
p5r=cphase(total(v5r,1))
p6r=cphase(total(v6r,1))
;
cohint=10
;
vis1=add(visrotate(v1[*,index],l,g1g),cohint,1)
vis2=add(visrotate(v2[*,index],l,g2g),cohint,1)
vis3=add(visrotate(v3[*,index],l,g3g),cohint,1)
vis4=add(visrotate(v4[*,index],l,g4g),cohint,1)
vis5=add(visrotate(v5[*,index],l,g5g),cohint,1)
vis6=add(visrotate(v6[*,index],l,g6g),cohint,1)
;
k=0
tv1=reform(v1[k,*]*conj(v2[k,*])*conj(v6[k,*]))
tv2=reform(v2[k,*]*v3[k,*]*conj(v4[k,*]))
tv3=reform(v3[k,*]*conj(v5[k,*])*conj(v6[k,*]))
tv4=reform(v4[k,*]*conj(v5[k,*])*conj(v1[k,*]))
;
tvis1=reform(vis1[k,*]*conj(vis2[k,*])*conj(vis6[k,*]))
tvis2=reform(vis2[k,*]*vis3[k,*]*conj(vis4[k,*]))
tvis3=reform(vis3[k,*]*conj(vis5[k,*])*conj(vis6[k,*]))
tvis4=reform(vis4[k,*]*conj(vis5[k,*])*conj(vis1[k,*]))
;
window,/free
set_screen
!y.range=[-90,90]
!p.multi=[0,2,4]
!p.charsize=2.0
rad=180/!pi
;
plot,rad*cphase(add(tv1,100)),psym=3
plot,rad*cphase(add(tvis1,10)),psym=3
plot,rad*cphase(add(tv2,100)),psym=3
plot,rad*cphase(add(tvis2,10)),psym=3
plot,rad*cphase(add(tv3,100)),psym=3
plot,rad*cphase(add(tvis3,10)),psym=3
plot,rad*cphase(add(tv4,100)),psym=3
plot,rad*cphase(add(tvis4,10)),psym=3
;
end
;-------------------------------------------------------------------------------
