;*******************************************************************************
; File: filter.pro
;
; Description:
; ------------
; Contains filter response/reflectivity/extinction functions.
;
; Block directory:
; ----------------
; Block 1: response
; Block 2: stroemgren_u,stroemgren_v,stroemgren_b,stroemgren_y,
;	   hipparcos_hp,tycho_bt,tycho_vt
;	   iras_f12,iras_12,iras_25,iras_60,iras_100,
;	   gaia2_g,gaia2_gbp,gaia2_grp,
;	   gaia3_g,gaia3_gbp,gaia3_grp,
;	   sloan_u,sloan_g,sloan_r,sloan_i,sloan_z,
;	   galex_fuv,galex_nuv,
;	   cousins_u,cousins_v,cousins_r,cousins_i,
;          johnson_u,johnson_b,johnson_v,johnson_r,johnson_i,
;	   johnson_j,johnson_h,johnson_k,
;	   johnson_ks,johnson_lp,johnson_m,johnson_n,
;	   vista_z,vista_y,vista_j,
;	   j_2mass,h_2mass,ks_2mass,
;	   akari_s9w,akari_l18w,
;	   wise_w1,wise_w2,wise_w3,wise_w4,
;	   mips_24,mips_70,mips_160,
;	   irac_i1,irac_i2,irac_i3,irac_i4,
;	   pacs_blue,pacs_green,pacs_red,
;	   skymapper_u,skymapper_v,skymapper_g,
;	   skymapper_r,skymapper_i,skymapper_z,
;	   pans_g,pans_i,pans_r,pans_w,pans_y,pans_z,
;	   xmm_uvw1,xmm_uvw2,xmm_uvm2,xmm_u,xmm_b,xmm_v,
;          mark3_800,mark3_550,mark3_500,mark3_450,
;          mark3_hab,mark3_han,mark3_500o,
;	   vinci_k,pti_k,pti_k1,pti_k2,pti_k3,pti_k4,pti_k5,
;	   pionier_free_12,pionier_free_13,pionier_free_14,
;	   pionier_free_23,pionier_free_24,pionier_free_34
; Block 3: filter_c,filter_cw,generic_c,vinci_c
; Block 4: pm1p21,pmt,apd,atmosphere,aluminium,silver
;
;************************************************************************Block 1
function response,l,t,lambda
;
; Return response for wavelengths lambda given a bandpass defined by l and t
;
n=n_elements(l)
num=n_elements(lambda)
tm=fltarr(num)
;
if max(lambda) lt min(l) then return,lambda*0
if min(lambda) gt max(l) then return,lambda*0
;
for i=0,num-1 do begin
	d=abs(lambda[i]-l)
	j=where(d eq min(d)) & j=j[0]
	if lambda[i] le l[j] then begin
		if j eq 0 then tm[i]=t[0] else $
			       tm[i]=t[j-1] $
			+((lambda[i]-l[j-1])/(l[j]-l[j-1]))*(t[j]-t[j-1])
	endif else begin
		if j eq n-1 then tm[i]=t[n-1] else $
				 tm[i]=t[j] $
			+((lambda[i]-l[j])/(l[j+1]-l[j]))*(t[j+1]-t[j])
	endelse
endfor
;
return,tm
;
end
;************************************************************************Block 2
function stroemgren_u,lambda
;
; lambda in nm.
;
n=28
l=findgen(n)*2.5+312.5
t=[0.00,0.01,0.03,0.06,0.11,0.17,0.23,0.29,0.33,0.36,0.38,0.39,0.39,0.39,0.38, $
   0.37,0.36,0.33,0.30,0.26,0.22,0.17,0.12,0.08,0.05,0.02,0.01,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function stroemgren_v,lambda
;
; lambda in nm.
;
n=35
l=findgen(n)*2.5+367.5
t=[0.00,0.01,0.01,0.01,0.01,0.01,0.01,0.02,0.02,0.03,0.05,0.07,0.13,0.21, $
   0.31,0.41,0.47,0.49,0.48,0.44,0.37,0.28,0.18,0.12,0.08,0.05,0.03,0.02, $
   0.02,0.01,0.01,0.01,0.01,0.01,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function stroemgren_b,lambda
;
; lambda in nm.
;
n=30
l=findgen(n)*2.5+427.5
t=[0.00,0.01,0.01,0.01,0.01,0.01,0.02,0.02,0.03,0.05,0.08,0.13,0.21,0.31, $
   0.41,0.46,0.48,0.46,0.39,0.29,0.19,0.11,0.07,0.04,0.03,0.02,0.01,0.01, $
   0.01,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function stroemgren_y,lambda
;
; lambda in nm.
;
n=30
l=findgen(n)*2.5+512.5
t=[0.00,0.01,0.01,0.02,0.03,0.04,0.06,0.10,0.15,0.23,0.33,0.41,0.46,0.48, $
   0.50,0.53,0.52,0.46,0.35,0.24,0.16,0.10,0.06,0.04,0.03,0.02,0.01,0.01, $
   0.01,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function hipparcos_hp,lambda
;
; lambda in nm.
;
n=113
l=findgen(n)*5+335
t=[0.000,0.006,0.023,0.047,0.078,0.114,0.154,0.198,0.248,0.305,0.369,0.442, $
   0.523,0.608,0.694,0.774,0.845,0.901,0.941,0.967,0.984,0.993,0.998,1.000, $
   1.000,0.998,0.993,0.987,0.979,0.969,0.958,0.946,0.933,0.919,0.903,0.888, $
   0.871,0.855,0.838,0.820,0.803,0.785,0.766,0.748,0.729,0.710,0.691,0.672, $
   0.653,0.634,0.615,0.596,0.577,0.558,0.539,0.520,0.502,0.483,0.465,0.447]
t=[t, $
  [0.429,0.412,0.395,0.378,0.361,0.345,0.329,0.314,0.298,0.283,0.269,0.254, $
   0.241,0.227,0.214,0.201,0.189,0.177,0.166,0.155,0.144,0.134,0.125,0.116, $
   0.108,0.100,0.092,0.085,0.079,0.073,0.067,0.062,0.057,0.053,0.049,0.045, $
   0.041,0.038,0.035,0.032,0.029,0.026,0.024,0.022,0.019,0.017,0.015,0.012, $
   0.010,0.007,0.005,0.002,0.000]]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function tycho_bt,lambda
;
; From: Bessell 2000 PASP 112, 961
; lambda in nm.
;
n=32
l=findgen(n)*5+350
t=[0.000,0.011,0.048,0.103,0.175,0.262,0.363,0.468,0.569,0.656,0.722,0.771, $
   0.810,0.846,0.886,0.929,0.968,1.000,0.985,0.879,0.707,0.510,0.334,0.215, $
   0.147,0.110,0.088,0.066,0.040,0.018,0.003,0.000]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function tycho_vt,lambda
;
; From: Bessell 2000 PASP 112, 961
; lambda in nm.
;
n=45
l=findgen(n)*5+455
t=[0.000,0.020,0.106,0.280,0.498,0.700,0.835,0.912,0.953,0.980,0.996,1.000, $
   0.994,0.977,0.952,0.921,0.886,0.849,0.808,0.766,0.723,0.678,0.632,0.584, $
   0.536,0.487,0.439,0.391,0.345,0.302,0.261,0.225,0.194,0.166,0.141,0.121, $
   0.103,0.087,0.074,0.060,0.049,0.037,0.024,0.011,0.000]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function iras_f12,lambda
;
; From: http://lambda.gsfc.nasa.gov/product/iras/spectral_resp.cfm.
; lambda in nm.
;
n=18
l=findgen(n)*500+7000
t=[0.000,0.008,0.535,0.689,0.735,0.815,0.900,0.904,0.834,0.816,0.793,0.854, $
   0.938,0.991,1.000,0.934,0.388,0.000]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function iras_12,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/iras/iras_12.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function iras_25,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/iras/iras_25.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function iras_60,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/iras/iras_60.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function iras_100,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/iras/iras_100.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function gaia_g,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
; print,'Warning: Gaia_g data not found, using data of Gaia2_g instead!'
;
restore,'~/oyster/svo/gaia/gaia2_g.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function gaia2_g,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/gaia/gaia2_g.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function gaia2_gbp,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/gaia/gaia2_gbp.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function gaia2_grp,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/gaia/gaia2_grp.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function gaia3_g,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/gaia/gaia3_g.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function gaia3_gbp,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/gaia/gaia3_gbp.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function gaia3_grp,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; lambda in nm.
;
restore,'~/oyster/svo/gaia/gaia3_grp.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function sloan_u,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/sloan
; r=dc_read_free('SLOAN_SDSS.u.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='sloan_u.xdr'
;
; lambda in nm.
;
restore,'~/oyster/svo/sloan/sloan_u.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function sloan_g,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/sloan
; r=dc_read_free('SLOAN_SDSS.g.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='sloan_g.xdr'
;
; lambda in nm.
;
restore,'~/oyster/svo/sloan/sloan_g.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function sloan_r,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/sloan
; r=dc_read_free('SLOAN_SDSS.r.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='sloan_r.xdr'
;
; lambda in nm.
;
restore,'~/oyster/svo/sloan/sloan_r.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function sloan_i,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/sloan
; r=dc_read_free('SLOAN_SDSS.i.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='sloan_i.xdr'
;
; lambda in nm.
;
restore,'~/oyster/svo/sloan/sloan_i.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function sloan_z,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/sloan
; r=dc_read_free('SLOAN_SDSS.z.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='sloan_z.xdr'
;
; lambda in nm.
;
restore,'~/oyster/svo/sloan/sloan_z.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function galex_fuv,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/galex
; r=dc_read_free('GALEX_FUV.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='GALEX_FUV.xdr'
;
; lambda in nm.
;
restore,'~/oyster/svo/galex/GALEX_FUV.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function galex_nuv,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/galex
; r=dc_read_free('GALEX_NUV.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='GALEX_NUV.xdr'
;
; lambda in nm.
;
restore,'~/oyster/svo/galex/GALEX_NUV.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function cousins_u,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Same as johnson_u?
; lambda in nm.
;
n=23
l=findgen(n)*5+305
t=[0.000,0.009,0.028,0.071,0.127,0.208,0.307,0.415,0.524,0.632,0.740,0.840, $
   0.929,0.995,0.990,0.830,0.613,0.406,0.212,0.090,0.033,0.014,0.000]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function cousins_v,lambda
;
; SAMI Kron-Cousins V filter
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/cousins/CTIO.SAMI.V.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function cousins_r,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/cousins/GCPD.Cousins.R.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function cousins_i,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/cousins/GCPD.Cousins.I.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_u,lambda
;
; lambda in nm.
;
n=23
l=findgen(n)*5+305
t=[0.000,0.009,0.028,0.071,0.127,0.208,0.307,0.415,0.524,0.632,0.740,0.840, $
   0.929,0.995,0.990,0.830,0.613,0.406,0.212,0.090,0.033,0.014,0.000]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_b,lambda
;
; lambda in nm.
;
n=40
l=findgen(n)*5+360
t=[0.000,0.006,0.030,0.060,0.134,0.302,0.567,0.841,0.959,0.983,0.996,1.000, $
   0.996,0.987,0.974,0.957,0.931,0.897,0.849,0.800,0.748,0.698,0.648,0.597, $
   0.545,0.497,0.447,0.397,0.345,0.297,0.252,0.207,0.166,0.129,0.095,0.069, $
   0.043,0.024,0.009,0.000]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_v,lambda
;
; lambda in nm.
;
n=54
l=findgen(n)*5+475
t=[0.000,0.030,0.084,0.163,0.301,0.458,0.630,0.780,0.895,0.967,0.997,1.000, $
   0.988,0.958,0.919,0.877,0.819,0.765,0.711,0.657,0.602,0.545,0.488,0.434, $
   0.386,0.331,0.289,0.250,0.214,0.181,0.151,0.120,0.093,0.069,0.051,0.036, $
   0.027,0.021,0.018,0.016,0.014,0.012,0.011,0.010,0.009,0.008,0.007,0.006, $
   0.005,0.004,0.003,0.002,0.001,0.000]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_r,lambda
;
; From: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
n=23
l=findgen(n)*20+520
t=[0.00,0.06,0.28,0.50,0.69,0.79,0.88,0.94,0.98,1.00,0.94,0.85,0.73,0.57, $
   0.42,0.31,0.17,0.11,0.06,0.04,0.02,0.01,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_i,lambda
;
; From: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
n=26
l=findgen(n)*20+680
t=[0.00,0.01,0.17,0.36,0.56,0.76,0.96,0.98,0.99,1.00,0.98,0.93,0.84,0.71, $
   0.58,0.47,0.36,0.28,0.20,0.15,0.10,0.08,0.05,0.03,0.02,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_j,lambda
;
; From: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
n=31
l=findgen(n)*20+960
t=[0.00,0.02,0.03,0.06,0.16,0.35,0.62,0.93,0.85,0.78,0.78,0.80,0.85,0.93, $
   0.75,0.64,0.63,0.63,0.66,0.68,0.70,0.70,0.66,0.60,0.46,0.27,0.14,0.09, $
   0.06,0.02,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_h,lambda
;
; From: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
; Preliminary, check reference!
;
n=17
l=findgen(n)*23+1460
t=[0.00,0.10,0.48,0.95,1.00,0.98,0.96,0.95,0.97,0.96,0.94,0.95,0.95,0.84, $
   0.46,0.08,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_k,lambda
;
; From: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
n=17
l=findgen(n)*50+1800
t=[0.00,0.10,0.48,0.95,1.00,0.98,0.96,0.95,0.97,0.96,0.94,0.95,0.95,0.84, $
   0.46,0.08,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_ks,lambda
;
; Here taken from NACO Ks-band filter curve.
; See also: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
n=29
l=findgen(n)*20+1880
t=[0.00,0.02,0.03,0.10,0.35,0.60,0.64,0.66,0.67,0.68,0.68,0.69,0.69,0.69, $
   0.69,0.70,0.69,0.68,0.68,0.66,0.64,0.63,0.50,0.19,0.05,0.02,0.01,0.01,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_lp,lambda
;
; L-Prime, which runs from about 3.4 - 4.2 microns, shifted redwards wrt L.
; Here taken from NACO L-band filter curve.
; See also: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
t=[0.00, 0.01, 0.02, 0.06, 0.23, 0.65, 0.82, 0.90, 0.92, 0.92, 0.94, 0.94, $
   0.95, 0.95, 0.94, 0.94, 0.94, 0.94, 0.93, 0.93, 0.93, 0.94, 0.95, 0.94, $
   0.93, 0.89, 0.65, 0.31, 0.09, 0.03, 0.01, 0.01, 0.00]


l=[3.38, 3.40, 3.42, 3.45, 3.47, 3.50, 3.52, 3.55, 3.57, 3.60, 3.62, 3.65, $
   3.68, 3.70, 3.73, 3.76, 3.79, 3.82, 3.85, 3.88, 3.91, 3.94, 3.97, 4.00, $
   4.03, 4.07, 4.10, 4.13, 4.17, 4.20, 4.24, 4.27, 4.31]*1000
n=n_elements(l)
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_m,lambda
;
; From: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
n=23
l=findgen(n)*100+4100
t=[0.00,0.04,0.25,0.48,1.00,0.93,0.92,0.89,0.84,0.78,0.76,0.79,0.74,0.68, $
   0.60,0.49,0.32,0.26,0.21,0.19,0.09,0.02,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function johnson_n,lambda
;
; From: Johnson, 1965, "The Absolute Calibration of the Arizona Photometry".
; lambda in nm.
;
n=35
l=findgen(n)*200+7000
t=[0.00,0.10,0.34,0.59,0.70,0.76,0.84,0.89,0.98,0.92,0.79,0.68,0.60,0.63, $
   0.69,0.73,0.72,0.65,0.60,0.60,0.61,0.64,0.65,0.68,0.71,0.76,0.76,0.76, $
   0.71,0.63,0.54,0.50,0.46,0.43,0.00]
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_z,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('VISTA.Z_filter.dat')
; save,template,file='VISTA_Z.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/VISTA_Z.tpl'
; data=read_ascii('~/oyster/svo/vista/VISTA.Z_filter.dat',template=template)
;
restore,'~/oyster/svo/vista/VISTA_Z.tpl'
data=read_ascii('~/oyster/svo/vista/VISTA.Z_filter.dat',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_y,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('VISTA.Y_filter.dat')
; save,template,file='VISTA_Y.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/VISTA_Y.tpl'
; data=read_ascii('~/oyster/svo/vista/VISTA.Y_filter.dat',template=template)
;
restore,'~/oyster/svo/vista/VISTA_Y.tpl'
data=read_ascii('~/oyster/svo/vista/VISTA.Y_filter.dat',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_j,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('VISTA.J_filter.dat')
; save,template,file='VISTA_J.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/VISTA_J.tpl'
; data=read_ascii('~/oyster/svo/vista/VISTA.J_filter.dat',template=template)
;
restore,'~/oyster/svo/vista/VISTA_J.tpl'
data=read_ascii('~/oyster/svo/vista/VISTA.J_filter.dat',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_z_full,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/vista/Paranal.VISTA.Z.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_y_full,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/vista/Paranal.VISTA.Y.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_j_full,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/vista/Paranal.VISTA.J.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_h_full,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/vista/Paranal.VISTA.H.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vista_ks_full,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/vista/Paranal.VISTA.Ks.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function j_2mass,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('J.txt')
; save,template,file='J.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/2mass/J.tpl'
; data=read_ascii('~/oyster/svo/2mass/J.txt',template=template)
;
restore,'~/oyster/svo/2mass/J.tpl'
data=read_ascii('~/oyster/svo/2mass/J.txt',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function h_2mass,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/2mass/H.txt')
; save,template,file='~/oyster/svo/2mass/H.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/2mass/H.tpl'
; data=read_ascii('~/oyster/svo/2mass/H.txt',template=template)
;
restore,'~/oyster/svo/2mass/H.tpl'
data=read_ascii('~/oyster/svo/2mass/H.txt',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function ks_2mass,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/2mass/Ks.txt')
; save,template,file='~/oyster/svo/2mass/Ks.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/2mass/Ks.tpl'
; data=read_ascii('~/oyster/svo/2mass/Ks.txt',template=template)
;
restore,'~/oyster/svo/2mass/Ks.tpl'
data=read_ascii('~/oyster/svo/2mass/Ks.txt',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function akari_s9w,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; http://svo2.cab.inta-csic.es/svo/theory/fps3/index.php?id=AKARI/IRC.S9W
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/akari/AKARI_IRC.S9W.dat')
; save,template,file='~/oyster/svo/akari/AKARI_IRC.S9W.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/akari/AKARI_IRC.S9W.tpl'
; data=read_ascii('~/oyster/svo/akari/AKARI_IRC.S9W.dat',template=template)
; 
restore,'~/oyster/svo/akari/AKARI_IRC.S9W.tpl'
data=read_ascii('~/oyster/svo/akari/AKARI_IRC.S9W.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function akari_l18w,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; http://svo2.cab.inta-csic.es/svo/theory/fps3/index.php?id=AKARI/IRC.L18W
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/akari/AKARI_IRC.L18W.dat')
; save,template,file='~/oyster/svo/akari/AKARI_IRC.L18W.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/akari/AKARI_IRC.L18W.tpl'
; data=read_ascii('~/oyster/svo/akari/AKARI_IRC.L18W.dat',template=template)
; 
restore,'~/oyster/svo/akari/AKARI_IRC.L18W.tpl'
data=read_ascii('~/oyster/svo/akari/AKARI_IRC.L18W.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w1,lambda
;
; Obsolete! See new version further below.
;
; https://www.astro.ucla.edu/~wright/WISE/index.html
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/wise/WISE1.txt')
; save,template,file='~/oyster/svo/wise/WISE1.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/wise/WISE1.tpl'
; data=read_ascii('~/oyster/svo/wise/WISE1.txt',template=template)
;
restore,'~/oyster/svo/wise/WISE1.tpl'
data=read_ascii('~/oyster/svo/wise/WISE1.txt',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w2,lambda
;
; Obsolete! See new version further below.
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/wise/WISE2.txt')
; save,template,file='~/oyster/svo/wise/WISE2.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/wise/WISE2.tpl'
; data=read_ascii('~/oyster/svo/wise/WISE2.txt',template=template)
;
restore,'~/oyster/svo/wise/WISE2.tpl'
data=read_ascii('~/oyster/svo/wise/WISE2.txt',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w3,lambda
;
; Obsolete! See new version further below.
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/wise/WISE1.txt')
; save,template,file='~/oyster/svo/wise/WISE1.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/wise/WISE1.tpl'
; data=read_ascii('~/oyster/svo/wise/WISE1.txt',template=template)
;
restore,'~/oyster/svo/wise/WISE3.tpl'
data=read_ascii('~/oyster/svo/wise/WISE3.txt',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w4,lambda
;
; Obsolete! See new version further below.
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/svo/wise/WISE1.txt')
; save,template,file='~/oyster/svo/wise/WISE1.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/wise/WISE1.tpl'
; data=read_ascii('~/oyster/svo/wise/WISE1.txt',template=template)
;
restore,'~/oyster/svo/wise/WISE4.tpl'
data=read_ascii('~/oyster/svo/wise/WISE4.txt',template=template)
l=data.field1*1000	; convert microns to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w1,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; This function uses the SVO data as the UCLA WISE data appear incorrect.
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/wise
; r=dc_read_free('WISE.W1.dat',l,t,/col,ignore=['!']) & l=l/10 ; lambda in nm.
; save,l,t,file='wise.w1.xdr'
;
restore,'~/oyster/svo/wise/wise.w1.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w2,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; This function uses the SVO data as the UCLA WISE data appear incorrect.
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/wise
; r=dc_read_free('WISE.W1.dat',l,t,/col,ignore=['!']) & l=l/10 ; lambda in nm.
; save,l,t,file='wise.w2.xdr'
;
restore,'~/oyster/svo/wise/wise.w2.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w3,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; This function uses the SVO data as the UCLA WISE data appear incorrect.
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/wise
; r=dc_read_free('WISE.W1.dat',l,t,/col,ignore=['!']) & l=l/10 ; lambda in nm.
; save,l,t,file='wise.w3.xdr'
;
restore,'~/oyster/svo/wise/wise.w3.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function wise_w4,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; This function uses the SVO data as the UCLA WISE data appear incorrect.
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/wise
; r=dc_read_free('WISE.W1.dat',l,t,/col,ignore=['!']) & l=l/10 ; lambda in nm.
; save,l,t,file='wise.w4.xdr'
;
restore,'~/oyster/svo/wise/wise.w4.xdr'
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mips_24,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
restore,!oyster_dir+'svo/spitzer/mips_24.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mips_70,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
restore,!oyster_dir+'svo/spitzer/mips_70.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mips_160,lambda
;
restore,!oyster_dir+'svo/spitzer/mips_160.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function irac_i1,lambda
;
; IRAC:3.6
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/spitzer
; r=dc_read_free('Spitzer_IRAC.I1.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='Spitzer_IRAC.I1.xdr'
;
; lambda in nm.
;
restore,!oyster_dir+'svo/spitzer/irac_i1.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function irac_i2,lambda
;
; IRAC:4.5
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/spitzer
; r=dc_read_free('Spitzer_IRAC.I2.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='Spitzer_IRAC.I2.xdr'
;
; lambda in nm.
;
restore,!oyster_dir+'svo/spitzer/irac_i2.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function irac_i3,lambda
;
; IRAC:5.7
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/spitzer
; r=dc_read_free('Spitzer_IRAC.I1.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='Spitzer_IRAC.I1.xdr'
;
; lambda in nm.
;
restore,!oyster_dir+'svo/spitzer/irac_i3.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function irac_i4,lambda
;
; IRAC:7.8
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use, prepare XDR file manually first:
; cd ~/oyster/svo/spitzer
; r=dc_read_free('Spitzer_IRAC.I1.dat',l,t,/col,ignore=['!']) & l=l/10
; save,l,t,file='Spitzer_IRAC.I1.xdr'
;
; lambda in nm.
;
restore,!oyster_dir+'svo/spitzer/irac_i4.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pacs_blue,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
restore,!oyster_dir+'svo/herschel/pacs_blue.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pacs_green,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
restore,!oyster_dir+'svo/herschel/pacs_green.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)

end
;-------------------------------------------------------------------------------
function pacs_red,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
restore,!oyster_dir+'svo/herschel/pacs_red.xdr'
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function ctiodecam_u,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/decam/CTIO.DECam.u.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function ctiodecam_g,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/decam/CTIO.DECam.g.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function ctiodecam_r,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/decam/CTIO.DECam.r.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function ctiodecam_i,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/decam/CTIO.DECam.i.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function ctiodecam_z,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/decam/CTIO.DECam.z.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function ctiodecam_y,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/decam/CTIO.DECam.y.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function skymapper_u,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/skymapper/SkyMapper.SkyMapper.u.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function skymapper_v,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/skymapper/SkyMapper.SkyMapper.v.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function skymapper_g,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/skymapper/SkyMapper.SkyMapper.g.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function skymapper_r,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/skymapper/SkyMapper.SkyMapper.r.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function skymapper_i,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/skymapper/SkyMapper.SkyMapper.i.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function skymapper_z,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/skymapper/SkyMapper.SkyMapper.z.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pans_g,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('PAN-STARRS.g.dat')
; save,template,file='PAN-STARRS.g.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/PAN-STARRS.g.tpl'
; data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.g.dat',template=template)
;
restore,'~/oyster/svo/panstarrs/PAN-STARRS.g.tpl'
data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.g.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pans_r,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('PAN-STARRS.r.dat')
; save,template,file='PAN-STARRS.r.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/PAN-STARRS.r.tpl'
; data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.r.dat',template=template)
;
restore,'~/oyster/svo/panstarrs/PAN-STARRS.r.tpl'
data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.r.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pans_w,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('PAN-STARRS.w.dat')
; save,template,file='PAN-STARRS.w.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/PAN-STARRS.w.tpl'
; data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.w.dat',template=template)
;
restore,'~/oyster/svo/panstarrs/PAN-STARRS.w.tpl'
data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.w.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pans_i,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('PAN-STARRS.i.dat')
; save,template,file='PAN-STARRS.i.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/PAN-STARRS.i.tpl'
; data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.i.dat',template=template)
;
restore,'~/oyster/svo/panstarrs/PAN-STARRS.i.tpl'
data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.i.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pans_z,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('PAN-STARRS.z.dat')
; save,template,file='PAN-STARRS.z.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/PAN-STARRS.z.tpl'
; data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.z.dat',template=template)
;
restore,'~/oyster/svo/panstarrs/PAN-STARRS.z.tpl'
data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.z.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pans_y,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?mode=browse
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('PAN-STARRS.y.dat')
; save,template,file='PAN-STARRS.y.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/svo/vista/PAN-STARRS.y.tpl'
; data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.y.dat',template=template)
;
restore,'~/oyster/svo/panstarrs/PAN-STARRS.y.tpl'
data=read_ascii('~/oyster/svo/panstarrs/PAN-STARRS.y.dat',template=template)
l=data.field1/10	; convert Angstroem to nm
t=data.field2
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function xmm_uvw1,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?id=XMM/OM.UVW2&&mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/xmm/XMM.OM.UVW1.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function xmm_uvw2,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?id=XMM/OM.UVW2&&mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/xmm/XMM.OM.UVW2.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function xmm_uvm2,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?id=XMM/OM.UVW2&&mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/xmm/XMM.OM.UVM2.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function xmm_u,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?id=XMM/OM.UVW2&&mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/xmm/XMM.OM.U.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function xmm_b,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?id=XMM/OM.UVW2&&mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/xmm/XMM.OM.B.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function xmm_v,lambda
;
; http://svo2.cab.inta-csic.es/theory/fps/index.php?id=XMM/OM.UVW2&&mode=browse
;
; Return filter transmission at the specified wavelengths (nm)
; Return central wavelength when lambda is not specified.
;
data=read_votable[!oyster_dir+'svo/xmm/XMM.OM.V.xml']
;
l=data.wavelength/10	; convert Angstroem to nm
t=data.transmission
;
if n_elements(lambda) eq 0 then begin
	lambda=l
; 	Return "central" wavelength
	return,total(l*t)/total(t)
endif
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_800,lambda
;
; Central wavelength of this filter: 801.3 nm.
; lambda in nm.
;
n=32
l=[786., 787., 788., 789., 790., 791., 792., 793., 794., 795., 796., 797., $
   798., 799., 800., 801., 802., 803., 804., 805., 806., 807., 808., 809., $
   810., 811., 812., 813., 814., 815., 816., 817.]
t=[00.0,13.3,20.4,29.6,41.1,53.6,64.7,71.3,75.6,76.4,76.3,76.1,76.0,75.9, $
   75.7,75.5,74.3,73.8,73.0,72.5,72.3,72.5,72.8,72.3,69.3,62.3,51.6,39.4, $
   27.8,19.0,12.3,00.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_550,lambda
;
; Central wavelength of this filter: 548.3 nm.
; lambda in nm.
;
n=38
l=[530., 531., 532., 533., 534., 535., 536., 537., 538., 539., 540., 541., $
   542., 543., 544., 545., 546., 547., 548., 549., 550., 551., 552., 553., $
   554., 555., 556., 557., 558., 559., 560., 561., 562., 563., 564., 565., $
   566., 567.]
t=[00.0, 10.0, 14.1, 19.6, 26.4, 33.8, 40.8, 46.6, 50.8, 54.0, 56.1, 58.1, $
   60.1, 62.4, 64.2, 65.7, 66.3, 66.5, 65.1, 63.7, 61.8, 59.2, 57.8, 56.7, $
   55.1, 54.3, 53.9, 53.2, 52.0, 49.4, 45.1, 38.7, 31.5, 24.4, 18.3, 13.6, $
   10.0, 00.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_500,lambda
;
; Central wavelength of this filter: 501.3 nm.
; lambda in nm.
;
n=32
l=[486., 487., 488., 489., 490., 491., 492., 493., 494., 495., 496., 497., $
   498., 499., 500., 501., 502., 503., 504., 505., 506., 507., 508., 509., $
   510., 511., 512., 513., 514., 515., 516., 517.]
t=[00.0,  8.9, 19.0, 37.5, 55.4, 60.0, 58.5, 56.1, 55.8, 56.8, 58.7, 60.6, $
   61.1, 62.6, 62.7, 62.5, 61.9, 61.0, 59.2, 56.8, 53.0, 51.3, 49.5, 49.1, $
   50.9, 54.3, 56.3, 49.5, 32.5, 17.5, 9.0, 00.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_450,lambda
;
; Central wavelength of this filter: 450.2 nm.
; lambda in nm.
;
n=32
l=[435., 436., 437., 438., 439., 440., 441., 442., 443., 444., 445., 446., $
   447., 448., 449., 450., 451., 452., 453., 454., 455., 456., 457., 458., $
   459., 460., 461., 462., 463., 464., 465., 466.]
t=[00.0, 13.3, 20.4, 29.6, 41.1, 53.6, 64.7, 71.3, 75.6, 76.4, 76.3, 76.1, $
   76.0, 75.9, 75.7, 75.5, 74.3, 73.8, 73.0, 72.5, 72.3, 72.5, 72.8, 72.3, $
   69.3, 62.3, 51.6, 39.4, 27.8, 19.0, 12.3, 00.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_656,lambda
;
; Central wavelength of this filter: 656.6 nm.
; lambda in nm.
;
n=22
l=[646.8, 647.8, 648.8, 649.8, 650.8, 651.8, 652.8, 653.8, 654.8, 655.8, $
   656.8, 657.8, 658.8, 659.8, 660.8, 661.8, 662.8, 663.8, 664.8, 665.8, $
   666.8, 667.8]
t=[ 0.0, 5.4, 8.0,12.1,19.0,29.8,45.0,61.0,73.2,79.2,79.0,78.8,75.6,68.5,56.4, $
   41.8,28.5,18.7,12.1, 8.1, 5.5, 0.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_hab,lambda
;
; Central wavelength of this filter: 656.6 nm.
; lambda in nm.
;
n=22
l=[646.8, 647.8, 648.8, 649.8, 650.8, 651.8, 652.8, 653.8, 654.8, 655.8, $
   656.8, 657.8, 658.8, 659.8, 660.8, 661.8, 662.8, 663.8, 664.8, 665.8, $
   666.8, 667.8]
t=[ 0.0, 5.4, 8.0,12.1,19.0,29.8,45.0,61.0,73.2,79.2,79.0,78.8,75.6,68.5,56.4, $
   41.8,28.5,18.7,12.1, 8.1, 5.5, 0.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_han,lambda
;
; Central wavelength of this filter: 657.0 nm.
; lambda in nm.
;
n=9
l=[654.8, 655.3, 655.8, 656.3, 656.8, 657.3, 657.8, 658.3, 658.8]
t=[ 0.0, 2.1, 9.4, 40.0, 70.2, 48.1, 12.8, 2.0, 0.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function mark3_500o,lambda
;
; Central wavelength of this filter: 499.5 nm.
; lambda in nm.
;
n=48
l=[477., 478., 479., 480., 481., 482., 483., 484., 485., 486., 487., 488., $
   489., 490., 491., 492., 493., 494., 495., 496., 497., 498., 499., 500., $
   501., 502., 503., 504., 505., 506., 507., 508., 509., 510., 511., 512., $
   513., 514., 515., 516., 517., 518., 519., 520., 521., 522., 523., 524.]
t=[00.0, 11.0, 19.0, 31.5, 46.6, 55.5, 55.2, 50.0, 44.9, 41.3, 39.1, 38.5, $
   38.8, 39.8, 41.6, 43.8, 46.5, 49.1, 51.8, 54.4, 56.3, 58.6, 60.3, 61.5, $
   62.6, 63.5, 64.1, 64.6, 65.0, 65.4, 65.3, 65.9, 65.8, 66.0, 66.1, 65.3, $
   65.0, 63.3, 58.0, 55.0, 48.2, 40.1, 31.8, 24.5, 18.5, 13.7, 10.1, 00.0]/100
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function vinci_k,lambda
;
; Based on code by M. Wittkowski, but modified for units of wavelength
; intervals rather than frequency intervals. The VINCI filter function,
; due to specific data reduction, was expressed over a grid of equidistant
; frequencies, but OYSTER in general uses grids equidistant in wavelength.
;
; This wavelength based version uses numbers from Pierre Kervella.
; lambda in nm.
;
n=62
l=1900+(findgen(n)*10.)
T=[0.00,0.01,0.00,0.04,0.06,0.16,0.25,0.52,0.76,1.12,0.95,1.00,1.24,1.64,$
   1.65,1.51,1.72,1.68,2.02,2.07,2.16,2.20,2.24,2.38,2.42,2.48,2.54,2.60,$
   2.57,2.57,2.45,2.59,2.63,2.67,2.73,2.75,2.79,2.80,2.90,2.77,2.86,$
   3.03,2.97,3.14,2.88,2.84,2.77,2.29,1.88,1.55,1.20,0.89,0.46,$
   0.49,0.39,0.27,0.21,0.13,0.07,0.03,0.04,0.00]
T=T/total(T)
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pti_k,lambda
;
; Provided by D. Peterson, from the PTI Collaboration
; lambda in nm.
;
l=[1.800,1.900,1.925,1.949,1.997,2.020,2.030,2.050,2.080,2.100,2.140,2.160, $
   2.200,2.210,2.220,2.240,2.260,2.300,2.310,2.320,2.360,2.380,2.400,2.420, $
   2.446,2.500,2.600]*1000
n=n_elements(l)
T=[0.000,0.005,0.010,0.060,0.410,0.700,0.725,0.715,0.750,0.780,0.800,0.820, $
   0.792,0.790,0.790,0.800,0.815,0.820,0.850,0.850,0.800,0.700,0.500,0.200, $
   0.100,0.010,0.000]
T=T/total(T)
;
if n_elements(lambda) eq 0 then begin
	lambda=l
	return,total(l*t)/total(t)
endif
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pti_k1,lambda
;
; lambda in nm.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
return,double(abs(lambda/1e9-GenConfig.Wavelength[0]) $
	   lt GenConfig.ChanWidth[0]/2)*pti_k(lambda)
;
end
;-------------------------------------------------------------------------------
function pti_k2,lambda
;
; lambda in nm.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
return,double(abs(lambda/1e9-GenConfig.Wavelength[1]) $
	   lt GenConfig.ChanWidth[1]/2)*pti_k(lambda)
;
end
;-------------------------------------------------------------------------------
function pti_k3,lambda
;
; lambda in nm.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
return,double(abs(lambda/1e9-GenConfig.Wavelength[2]) $
	   lt GenConfig.ChanWidth[2]/2)*pti_k(lambda)
;
end
;-------------------------------------------------------------------------------
function pti_k4,lambda
;
; lambda in nm.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
return,double(abs(lambda/1e9-GenConfig.Wavelength[3]) $
	   lt GenConfig.ChanWidth[3]/2)*pti_k(lambda)
;
end
;-------------------------------------------------------------------------------
function pti_k5,lambda
;
; lambda in nm.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
return,double(abs(lambda/1e9-GenConfig.Wavelength[4]) $
	   lt GenConfig.ChanWidth[4]/2)*pti_k(lambda)
;
end
;-------------------------------------------------------------------------------
function pionier_free_12,lambda
;
; Return response for PIONIER H-band for IP 12 for given lambda [mu].
;
; Restore bandpass data t(l)
restore,!oyster_dir+'vlti/pionier/12.xdr'
l=l*1000
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pionier_free_13,lambda
;
; Return response for PIONIER H-band for IP 13 for given lambda [mu].
;
; Restore bandpass data t(l)
restore,!oyster_dir+'vlti/pionier/13.xdr'
l=l*1000
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pionier_free_14,lambda
;
; Return response for PIONIER H-band for IP 14 for given lambda [mu].
;
; Restore bandpass data t(l)
restore,!oyster_dir+'vlti/pionier/14.xdr'
l=l*1000
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pionier_free_23,lambda
;
; Return response for PIONIER H-band for IP 23 for given lambda [mu].
;
; Restore bandpass data t(l)
restore,!oyster_dir+'vlti/pionier/23.xdr'
l=l*1000
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pionier_free_24,lambda
;
; Return response for PIONIER H-band for IP 24 for given lambda [mu].
;
; Restore bandpass data t(l)
restore,!oyster_dir+'vlti/pionier/24.xdr'
l=l*1000
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pionier_free_34,lambda
;
; Return response for PIONIER H-band for IP 34 for given lambda [mu].
;
; Restore bandpass data t(l)
restore,!oyster_dir+'vlti/pionier/34.xdr'
l=l*1000
;
if n_elements(lambda) eq 0 then return,total(l*t)/total(t)
return,response(l,t,lambda)
;
end
;************************************************************************Block 3
function filter_c,lambda,filter
;
; Return response for a filter by name, given a wavelength [nm] vector.
;
; Remove "/" to read filter ID 
words=nameparse(filter,'/')
filter=words[n_elements(words)-1]
;
case filter of
'U':    tm=johnson_u(lambda)
'B':    tm=johnson_b(lambda)
'V':    tm=johnson_v(lambda)
'R':    tm=johnson_r(lambda)
'I':    tm=johnson_i(lambda)
'J':    tm=johnson_j(lambda)
'H':    tm=johnson_h(lambda)
'K':    tm=johnson_k(lambda)
'L':    tm=johnson_lp(lambda)
'M':    tm=johnson_m(lambda)
'N':    tm=johnson_n(lambda)
'u':    tm=stroemgren_u(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
'v':    tm=stroemgren_v(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
'b':    tm=stroemgren_b(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
'y':    tm=stroemgren_y(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
'm800': tm=mark3_800(lambda)
'm550': tm=mark3_550(lambda)
'm500': tm=mark3_500(lambda)
'm450': tm=mark3_450(lambda)
'mhan': tm=mark3_han(lambda)
'mhab': tm=mark3_hab(lambda)
'm500o':tm=mark3_500o(lambda)
'Cousins:U':	tm=cousins_u(lambda)
'Cousins:V':	tm=cousins_v(lambda)
'Cousins:R':	tm=cousins_r(lambda)
'Cousins:I':	tm=cousins_i(lambda)
'Stroemgren:U': tm=stroemgren_u(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
'Stroemgren:V': tm=stroemgren_v(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
'Stroemgren:B': tm=stroemgren_b(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
'Stroemgren:Y': tm=stroemgren_y(lambda)*atmosphere[lambda]*aluminium[lambda]^2 $
          *pm1p21[lambda]
; Vizier filters
'Johnson:U':    tm=johnson_u(lambda)
'Johnson:B':    tm=johnson_b(lambda)
'Johnson:V':    tm=johnson_v(lambda)
'Johnson:R':    tm=johnson_r(lambda)
'Johnson:I':    tm=johnson_i(lambda)
'Johnson:J':    tm=johnson_j(lambda)
'Johnson:H':    tm=johnson_h(lambda)
'Johnson:K':    tm=johnson_k(lambda)
'Johnson:L':    tm=johnson_lp(lambda)
'Johnson:M':    tm=johnson_m(lambda)
'Johnson:N':    tm=johnson_n(lambda)
'VISTA:Z':	tm=vista_z_full(lambda)
'VISTA:Y':	tm=vista_y_full(lambda)
'VISTA:J':	tm=vista_j_full(lambda)
'VISTA:H':	tm=vista_h_full(lambda)
'VISTA:Ks':	tm=vista_ks_full(lambda)
'2MASS:J':	tm=j_2mass(lambda)
'2MASS:H':	tm=h_2mass(lambda)
'2MASS:Ks':	tm=ks_2mass(lambda)
'SDSS:g':	tm=sloan_g(lambda)
'SDSS:i':	tm=sloan_i(lambda)
'SDSS:r':	tm=sloan_r(lambda)
'SDSS:u':	tm=sloan_u(lambda)
'SDSS:z':	tm=sloan_z(lambda)
; Missions
'HIP:Hp':	tm=hipparcos_hp(lambda)
'HIP:VT':	tm=tycho_vt(lambda)
'HIP:BT':	tm=tycho_bt(lambda)
'Gaia:G':	tm=gaia_g(lambda)
'GAIA2:G':	tm=gaia2_g(lambda)
'GAIA2:Grp':	tm=gaia2_grp(lambda)
'GAIA2:Gbp':	tm=gaia2_gbp(lambda)
'GAIA3:G':	tm=gaia3_g(lambda)
'GAIA3:Grp':	tm=gaia3_grp(lambda)
'GAIA3:Gbp':	tm=gaia3_gbp(lambda)
'GALEX:FUV':	tm=galex_fuv(lambda)
'GALEX:NUV':	tm=galex_nuv(lambda)
'AKARI:S9W':	tm=akari_s9w(lambda)
'AKARI:L18W':	tm=akari_l18w(lambda)
'AKARI:L18':	tm=akari_l18w(lambda)
'WISE:W1':	tm=wise_w1(lambda)
'WISE:W2':	tm=wise_w2(lambda)
'WISE:W3':	tm=wise_w3(lambda)
'WISE:W4':	tm=wise_w4(lambda)
'IRAS:12':	tm=iras_12(lambda)
'IRAS:25':	tm=iras_25(lambda)
'IRAS:60':	tm=iras_60(lambda)
'IRAS:100':	tm=iras_100(lambda)
'IRAC:3.6':	tm=irac_i1(lambda)
'IRAC:4.5':	tm=irac_i2(lambda)
'IRAC:5.7':	tm=irac_i3(lambda)
'IRAC:7.9':	tm=irac_i4(lambda)
'MIPS:24':	tm=mips_24(lambda)	; Spitzer/MIPS:24
'MIPS:70':	tm=mips_70(lambda)	; Spitzer/MIPS:70
'MIPS:160':	tm=mips_160(lambda)	; Spitzer/MIPS:160
'PACS:blue':	tm=pacs_blue(lambda)	; Herschel/PACS.blue
'PACS:green':	tm=pacs_green(lambda)	; Herschel/PACS.green
'PACS:red':	tm=pacs_red(lambda)	; Herschel/PACS.red
'PAN-STARRS:g':	tm=pans_g(lambda)
'PAN-STARRS:r':	tm=pans_r(lambda)
'PAN-STARRS:w':	tm=pans_w(lambda)
'PAN-STARRS:i':	tm=pans_i(lambda)
'PAN-STARRS:z':	tm=pans_z(lambda)
'PAN-STARRS:y':	tm=pans_y(lambda)
'SkyMapper:u':	tm=skymapper_u(lambda)
'SkyMapper:v':	tm=skymapper_v(lambda)
'SkyMapper:g':	tm=skymapper_g(lambda)
'SkyMapper:r':	tm=skymapper_r(lambda)
'SkyMapper:i':	tm=skymapper_i(lambda)
'SkyMapper:z':	tm=skymapper_z(lambda)
'DECam:u':	tm=ctiodecam_u(lambda)
'DECam:g':	tm=ctiodecam_g(lambda)
'DECam:r':	tm=ctiodecam_r(lambda)
'DECam:i':	tm=ctiodecam_i(lambda)
'DECam:z':	tm=ctiodecam_z(lambda)
'DECam:Y':	tm=ctiodecam_y(lambda)
'XMM-OT:UVW1':	tm=xmm_uvw1(lambda)
'XMM-OT:UVW2':	tm=xmm_uvw2(lambda)
'XMM-OT:UVM2':	tm=xmm_uvm2(lambda)
'XMM_U':	tm=xmm_u(lambda)
'XMM_B':	tm=xmm_b(lambda)
'XMM_V':	tm=xmm_v(lambda)
endcase
;
return,tm
end
;-------------------------------------------------------------------------------
function filter_cw,filter
;
; Return central wavelength [nm] of a filter by name
;
case filter of
'U':    cw=johnson_u()
'B':    cw=johnson_b()
'V':    cw=johnson_v()
'R':    cw=johnson_r()
'I':    cw=johnson_i()
'J':    cw=johnson_j()
'H':    cw=johnson_h()
'K':    cw=johnson_k()
'L':    cw=johnson_lp()
'M':    cw=johnson_m()
'N':    cw=johnson_n()
'u':    begin
	tm=stroemgren_u(lambda)
	tm=tm*atmosphere[lambda]*aluminium[lambda]^2*pm1p21[lambda]
	end
'v':    begin
	tm=stroemgren_v(lambda)
	tm=tm*atmosphere[lambda]*aluminium[lambda]^2*pm1p21[lambda]
	end
'b':    begin
	tm=stroemgren_b(lambda)
	tm=tm*atmosphere[lambda]*aluminium[lambda]^2*pm1p21[lambda]
	end
'y':    begin
	tm=stroemgren_y(lambda)
	tm=tm*atmosphere[lambda]*aluminium[lambda]^2*pm1p21[lambda]
	end
'm800': cw=mark3_800()
'm550': cw=mark3_550()
'm500': cw=mark3_500()
'm450': cw=mark3_450()
'mhan': cw=mark3_han()
'mhab': cw=mark3_hab()
'm500o':cw=mark3_500o()
	else: cw=0
endcase
;
return,cw
end
;-------------------------------------------------------------------------------
function generic_c,lambda,ch,ob
;
; Return response for a generic rectangular OYSTER channel at model wavelengths
; lambda [nm].
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
return,double(abs(lambda/1e9-GenConfig.Wavelength[ch,ob]) $
	   lt GenConfig.ChanWidth[ch,ob]/2)
;
; num=n_elements(lambda)
; response=fltarr(num)
; index=where(abs(lambda/1e9-GenConfig.Wavelength(ch,ob)) $
; 	lt GenConfig.ChanWidth(ch,ob)/2,count)
; if count gt 0 then response(index)=1
; return,response
;
; num=n_elements(lambda)
; response=fltarr(num)
; for i=0,num-1 do begin
;	if abs(lambda(i)/1e9-GenConfig.Wavelength(ch,ob)) $
;		lt GenConfig.ChanWidth(ch,ob)/2 then response(i)=1 else $
;						   response(i)=0
; endfor
; return,response
;
end
;-------------------------------------------------------------------------------
function vinci_c,lambda
;
; Based on code by M. Wittkowski, but modified for units of wavelength
; intervals rather than frequency intervals. The VINCI filter function,
; due to specific data reduction, was expressed over a grid of equidistant
; frequencies, but OYSTER in general uses grids equidistant in wavelength.
; Therefore, we have to multiply the VINCI numbers with c/lambda^2.
;
; 2005-06-09: incorrect, since this is a response and not an energy,
; no such normalization is necessary: see commented ;/l^2 below!
;
; Superceded by vinci_k
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
c=2.9979E8
n=66
wn=(findgen(n)*20.)+3980.d
wn=wn*1e2*c
l=c/wn*1e9
T=[0.00,0.00038, 0.00032, 0.00072, 0.00126, 0.00213, 0.00385, 0.00493,$
0.00456, 0.00887, 0.01196, 0.01549, 0.02290, 0.02773, 0.02839, 0.02880,$
0.03139, 0.02972, 0.03028, 0.02856, 0.02770, 0.02896, 0.02801, 0.02792,$
0.02753, 0.02728, 0.02668, 0.02627, 0.02588, 0.02449, 0.02567, 0.02566,$
0.02604, 0.02544, 0.02478, 0.02422, 0.02379, 0.02241, 0.02200, 0.02200,$
0.02163, 0.02069, 0.02017, 0.01678, 0.01719, 0.01506, 0.01506, 0.01650,$
0.01638, 0.01240, 0.00997, 0.00950, 0.00950, 0.01117, 0.00763, 0.00520,$
0.00520, 0.00251, 0.00157, 0.00064, 0.00064, 0.00041, 0.00002, 0.00010,$
0.00010, 0.00];/l^2
T=T/total(T)
;
return,response(l,t,lambda)
;
end
;************************************************************************Block 4
function pm1p21,lambda
;
; Normalized sensitivity of 1P21 Photomultiplier (Kurucz.)
; lambda in nm.
;
n=33
l=findgen(n)*10+290
t=[0.00,0.08,0.22,0.45,0.68,0.81,0.90,0.95,0.97,0.99,1.00,1.00,0.99,0.98, $
   0.95,0.91,0.87,0.83,0.77,0.71,0.65,0.58,0.52,0.46,0.40,0.34,0.29,0.24, $
   0.20,0.16,0.13,0.10,0.00]
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function pmt,lambda
;
; Normalized sensitivity of GaAs RCA 31034 Photomultiplier Tube (Bessell 1979)
; lambda in nm.
;
n=8
l=findgen(n)*100+200
t=[0.00,0.36,0.40,0.52,0.71,0.91,0.99,0.00]*0.14
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function apd,lambda
;
; Normalized sensitivity of Si Avalanche Photodiode (Mozurkewich, priv. comm.)
; lambda in nm.
;
n=16
l=findgen(n)*50+350
t=[0.00,0.10,0.31,0.42,0.49,0.55,0.63,0.70, $
   0.77,0.79,0.71,0.56,0.30,0.11,0.03,0.00]
t=t/max(t)
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function atmosphere,lambda
;
; Transmission through atmosphere of airmass=1. 
; (Allen: Astrophysical Quantities, p. 126.)
; Lambda is in nm.
;
n=24
l=[280,300,320,340,360,380,400,450,500,550,600,650,700,800,900, $
   1000,1200,1400,1600,1800,2000,3000,5000,10000]
l=float(l)
t=[0.000,0.011,0.273,0.43,0.51,0.58,0.63,0.73,0.79,0.82,0.84,0.88,0.911, $
   0.939,0.953,0.962,0.972,0.979,0.983,0.985,0.987,0.992,0.994,0.995]
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function aluminium,lambda
;
; Reflectivity of aluminium coated mirror. (Allen: Astrophysical Quantities)
; lambda in nm.
;
n=22
l=[200,220,240,260,280,300,320,340,360,380,400,450,500,550,600,650,700,800, $
   1000,2000,5000,10000]
l=float(l)
t=[0.72,0.78,0.81,0.82,0.82,0.82,0.82,0.83,0.83,0.84,0.85,0.86,0.87,0.88, $ 
   0.89,0.88,0.87,0.85,0.93,0.96,0.97,0.98]
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
function silver,lambda
;
; Reflectivity of silver coated mirror. (Allen: Astrophysical Quantities)
; lambda in nm.
;
n=22
l=[200,220,240,260,280,300,320,340,360,380,400,450,500,550,600,650,700,800, $
   1000,2000,5000,10000]
l=float(l)
t=[0.20,0.25,0.27,0.27,0.23,0.12,0.07,0.63,0.77,0.82,0.85,0.90,0.91,0.92, $ 
   0.93,0.94,0.95,0.97,0.98,0.98,0.99,0.99]
;
return,response(l,t,lambda)
;
end
;-------------------------------------------------------------------------------
