;*******************************************************************************
; File: lists.pro
;
; Description:
; ------------
; Contains procedures to read external lists. These are ASCII/FITS tables
; with the catalog ID in the first column and data compiled from published
; catalogs. These data are used to /update the currently loaded StarTable
; or /init the startable.
;
; Block directory:
; ----------------
; Block 1: get_diameter,
;	   get_richichi,get_vanbelle,
;	   get_stecklum,get_verhoelst,get_merand,get_borde,
;	   get_calvin,get_jsdc
; Block 2: get_parallax,get_leeuwen,get_position,get_orbit,
;	   get_reandma
; Block 3: get_jhk_bsc,get_jhk_sky,get_jhk_hic,get_jhk_hdn,
;          get_jhk,get_n12,get_f12,get_i12,get_jp11,
;	   get_ubvri,get_ubv,get_uvbybeta,
;	   get_feh,get_logg,get_teff
; Block 4: get_tycho,get_mdfc,get_wise,get_2mass,get_iras
;
;************************************************************************Block 1
pro get_diameter,init=init,update=update
;
; Read diameters in starbase/diameter.bsc. 
; Note: data found will replace current values!
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update bsc.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/diameter.bsc',bsc,/col, $
		ignore=['!'])
	get_startable,'BSC'+string(bsc,format='(i4.4)')
	uindex=where(startable.bsc eq 0)
	startable.bsc=bsc
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.bsc ne 0,count)
if count gt 0 then begin
	diameter=startable[index].diameter
	zerospacing=startable[index].zerospacing
	a0=startable[index].a0
	a1=startable[index].a1
	a2=startable[index].a2
	bflag=startable[index].bflag
	bsc=startable[index].bsc
	n=0
	d=0.0
	z=0.0
	a_0=0.0
	a_1=0.0
	a_2=0.0
	c=''
	status=dc_read_free(!oyster_dir+'starbase/diameter.bsc', $
		n,d,z,a_0,a_1,a_2,c,/col,resize=[1,2,3,4,5,6,7],ignore=['!'])
;
	for i=0,count-1 do begin
		j=where(n eq bsc[i],j_count)
		if j_count gt 0 then begin
			diameter[i]=d[j[0]] 
			zerospacing[i]=z[j[0]]
			a0[i]=a_0[j[0]]
			a1[i]=a_1[j[0]]/1e3
			a2[i]=a_2[j[0]]/1e6
			bflag[i]=c[j[0]]
		endif
	endfor
	startable[index].diameter=diameter
	startable[index].zerospacing=zerospacing
	startable[index].a0=a0
	startable[index].a1=a1
	startable[index].a2=a2
	startable[index].bflag=bflag
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameters added: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_DIAMETER): No stars with BSC name found!'
;
end
;-------------------------------------------------------------------------------
pro get_richichi,init=init,update=update
;
; Read VINCI calibrator list courtesy of A. Richichi (et al. 2009)
; Diameters are LD.
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/richichi.hdn',hdn,/col, $
		ignore=['!'])
	get_startable,'HDN'+string(hdn,format='(i6.6)')
	uindex=where(startable.hdn eq 0)
	startable.hdn=hdn
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	sp=''
	vmag=''
	status=dc_read_free(!oyster_dir+'starbase/richichi.hdn', $
		hd,hr,ra,dec,pmra,pmdec,v,k,s,ld,lde, $
		/col,ignore=['!'])
;
	for i=0L,count-1 do begin
		j=where(hd eq hdn[i],j_count)
		if j_count eq 1 then begin
			startable[i].diameter=ld[j]
;			startable(i).diametere=lde(j)
			startable[i].spectrum=s[j]
			startable[i].mk=k[j]
			startable[i].bflag='C'
		endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameters added: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_RICHICHI): No stars with HDN name found!'
;
; Make sure integer spectral types and classes are up-to-date
spec_parse
;
; Update HDN.xdr
if n_elements(uindex) ne 0 then begin
	if uindex[0] ne -1 and keyword_set(update) then $
	update_cat,startable[uindex].starid
endif
;
end
;-------------------------------------------------------------------------------
pro get_vanbelle,init=init,update=update
;
; Read PTI calibrator list courtesy of Gerard van Belle (et al. 2008, ApJS, 176)
; Diameters are LD.
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/vanbelle.hdn',hdn,/col, $
		ignore=['!'])
	get_startable,'HDN'+string(hdn,format='(i6.6)')
	uindex=where(startable.hdn eq 0)
	startable.hdn=hdn
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	sp=''
	vmag=''
	status=dc_read_free(!oyster_dir+'starbase/vanbelle.hdn', $
		hd,rah,ram,ras,decd,decm,decs,pmra,pmdec,plx,d,e,spectra,vmag, $
		/col,ignore=['!'])
;
	for i=0L,count-1 do begin
	j=where(hd eq hdn[i],j_count)
	if j_count eq 1 then begin
		startable[i].ra=double(rah[j])+double(ram[j])/60+ras[j]/3600
		startable[i].dec=double(decd[j])+double(decm[j])/60+decs[j]/3600
		startable[i].pmra=((pmra[j]*100)/15)/cos(startable[i].dec*!pi/180)
		startable[i].pmdec=pmdec[j]*100
		startable[i].px=plx[j]
		startable[i].diameter=d[j]
;		startable(i).diametere=e(j)
		startable[i].spectrum=spectra[j]
		startable[i].mv=float(strmid(vmag[j],2,5))	
		startable[i].bflag='C'
	endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameters added: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_VANBELLE): No stars with HDN name found!'
;
; Make sure integer spectral types and classes are up-to-date
spec_parse
;
; Update HDN.xdr
if n_elements(uindex) ne 0 then begin
	if uindex[0] ne -1 and keyword_set(update) then $
	update_cat,startable[uindex].starid
endif
;
end
;-------------------------------------------------------------------------------
pro get_stecklum,init=init,update=update
;
; Read MIDI calibrator list, "stecklum.hdn", compiled by consortium.
; HDN.xdr can be updated (update=1) since not all stars in the Stecklum list are
; also in SKYCAT used by OYSTER for HDN starids.
; Diameters are LD.
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/stecklum.hdn',hdn,/col, $
		ignore=['!'])
	get_startable,'HDN'+string(hdn,format='(i6.6)')
	uindex=where(startable.hdn eq 0)
	startable.hdn=hdn
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	lines=''
	status=dc_read_fixed(!oyster_dir+'starbase/stecklum.hdn',lines, $
		/col,resize=[1],format='(a800)',ignore=['!'])
	n=n_elements(lines)
	hd=lonarr(n)
	ra=strarr(n)
	dc=strarr(n)
	pr=fltarr(n)
	pd=fltarr(n)
	sp=strarr(n)
	mv=fltarr(n)
	mh=fltarr(n)
	fi=fltarr(n)
	tf=fltarr(n)
	dm=fltarr(n)
	dme=fltarr(n)
	chisq=fltarr(n)
;
	for i=0,n-1 do begin
        hd[i]=long(strcompress(strmid(lines[i],3,6),/remove_all))
        ra[i]=strmid(lines[i],23,2)+':'+strmid(lines[i],26,2)+':'+strmid(lines[i],29,5)
        dc[i]=strmid(lines[i],49,3)+':'+strmid(lines[i],53,2)+':'+strmid(lines[i],56,5)
        sp[i]=strmid(lines[i],81,14)
        pr[i]=((float(strmid(lines[i],128,6))/15)*100)/cos(dms2d(dc[i])*!pi/180)
        pd[i]=float(strmid(lines[i],144,6))*100
        mv[i]=float(strmid(lines[i],160,6))
        mhs=strmid(lines[i],176,6)
        if strlen(strcompress(mhs,/remove_all)) gt 1 then mh[i]=float(mhs)
        fis=strmid(lines[i],192,7)
        f12=strmid(lines[i],224,6)
        if strlen(strcompress(fis,/remove_all)) ge 1 then fi[i]=float(fis) $
                                                     else fi[i]=float(f12)
       	tf[i]=float(strmid(lines[i],288,7))
       	dms=strmid(lines[i],320,9)
       	if strlen(strcompress(dms,/remove_all)) gt 1 then dm[i]=float(dms)
       	dmse=strmid(lines[i],336,9)
       	if strlen(strcompress(dmse,/remove_all)) gt 1 then dme[i]=float(dmse)
       	chi2=strmid(lines[i],368,9)
       	if strlen(strcompress(chi2,/remove_all)) gt 1 then chisq[i]=float(chi2)
	endfor
;
	for i=0L,count-1 do begin
		j=where(hd eq hdn[i] and dm gt 0,j_count)
		if j_count gt 0 then begin
			startable[i].ra=hms2h(ra[j])
			startable[i].dec=dms2d(dc[j])
			startable[i].spectrum=sp[j]
			startable[i].pmra=pr[j]
			startable[i].pmdec=pd[j]
			startable[i].diameter=dm[j]
;			startable(i).diametere=dme(j)*dm(j)/100
			startable[i].mh=mh[j]
			startable[i].f12=fi[j]
			startable[i].teff=tf[j]
			startable[i].sflag=string(chisq[j])
			startable[i].bflag='C'
		endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameter values found: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_STECKLUM): No stars with HDN name found!'
;
if init then startable=startable[where(startable.diameter gt 0)]
;
; Make sure integer spectral types and classes are up-to-date
spec_parse
;
; Update HDN.xdr
if n_elements(uindex) ne 0 then begin
	if uindex[0] ne -1 and keyword_set(update) then $
	update_cat,startable[uindex].starid
endif
;
end
;-------------------------------------------------------------------------------
pro get_verhoelst,init=init,update=update
;
; Read MIDI calibrator list, "verhoelst.hdn", compiled by Tijl Verhoelst 
; (Thesis). HDN.xdr can be updated (update=1) since not all stars in the 
; Verhoelst list are also in SKYCAT used by OYSTER for HDN starids.
;
; All quality flags except those related to TF are included and if any is set,
; slfag is set to 2, otherwise 1. TF accuracy criterion is relaxed to 5%.
;
; Diameters are LD.
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/verhoelst.hdn',hdn,/col, $
		ignore=['!'])
	get_startable,'HDN'+string(hdn,format='(i6.6)')
	uindex=where(startable.hdn eq 0)
	startable.hdn=hdn
	t=startable
	rename_starids,'hdn-hip'
	read_catalogs
	pmra=startable.pmra
	pmdec=startable.pmdec
	startable=t
	startable.pmra=pmra
	startable.pmdec=pmdec
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	lines=''
	status=dc_read_fixed(!oyster_dir+'starbase/verhoelst.hdn',lines, $
		/col,resize=[1],format='(a800)',ignore=['!'])
	n=n_elements(lines)
	hd=lonarr(n)
	ra=strarr(n)
	dc=strarr(n)
	pr=fltarr(n)
	pd=fltarr(n)
	px=fltarr(n)
	pe=fltarr(n)
	sp=strarr(n)
	mb=fltarr(n)
	mv=fltarr(n)
	mj=fltarr(n)
	mh=fltarr(n)
	mk=fltarr(n)
	fi=fltarr(n)
	tf=fltarr(n)
	dm=fltarr(n)
	dme=fltarr(n)
	chisq=fltarr(n)
	flag14=strarr(n)
	flag15=strarr(n)
	flag16=strarr(n)
	flag17=strarr(n)
	flag18=strarr(n)
	flag19a=strarr(n)
	flag19b=strarr(n)
	flag19c=strarr(n)
	flag19d=strarr(n)
	origin=strarr(n)
;
	for i=0,n-1 do begin
       	hd[i]=long(strcompress(strmid(lines[i],0,6),/remove_all))
       	ra[i]=strmid(lines[i],8,2)+':'+strmid(lines[i],11,2)+':'+strmid(lines[i],14,5)
       	dc[i]=strmid(lines[i],21,3)+':'+strmid(lines[i],25,2)+':'+strmid(lines[i],28,5)
       	pr[i]=((float(strmid(lines[i],34,6))/15)*100)/cos(dms2d(dc[i])*!pi/180)
       	pd[i]=float(strmid(lines[i],41,6))*100
	pxs=strmid(lines[i],67,6)
	if strlen(strcompress(pxs,/remove_all)) gt 1 then px[i]=float(pxs)/1000
	pes=strmid(lines[i],76,6)
	if strlen(strcompress(pes,/remove_all)) gt 1 then pe[i]=float(pes)/1000
       	sp[i]=strmid(lines[i],49,14)
       	mbs=strmid(lines[i],101,6)
       	if strlen(strcompress(mbs,/remove_all)) gt 1 then mb[i]=float(mbs)
       	mvs=strmid(lines[i],116,6)
       	if strlen(strcompress(mvs,/remove_all)) gt 1 then mv[i]=float(mvs)
       	mjs=strmid(lines[i],148,6)
       	if strlen(strcompress(mjs,/remove_all)) gt 1 then mj[i]=float(mjs)
       	mhs=strmid(lines[i],163,6)
       	if strlen(strcompress(mhs,/remove_all)) gt 1 then mh[i]=float(mhs)
       	mks=strmid(lines[i],178,6)
       	if strlen(strcompress(mks,/remove_all)) gt 1 then mk[i]=float(mks)
       	fis=strmid(lines[i],210,5)
       	if strlen(strcompress(fis,/remove_all)) ge 1 then fi[i]=float(fis)
	origin[i]=strmid(lines[i],224,1)
	if origin[i] eq 'M' then fi[i]=fi[i]*1.4	; MSX is color corrected
       	tf[i]=float(strmid(lines[i],231,5))
	dm[i]=float(strmid(lines[i],251,6))
	dme[i]=float(strmid(lines[i],260,5))
       	chisq[i]=strmid(lines[i],280,6)
	flag14[i]=strmid(lines[i],293,1)
	flag15[i]=strmid(lines[i],297,1)
	flag16[i]=strmid(lines[i],301,1)
	flag17[i]=strmid(lines[i],305,1)
	flag18[i]=strmid(lines[i],309,1)
	flag19a[i]=strmid(lines[i],315,1)
	flag19b[i]=strmid(lines[i],319,1)
	flag19c[i]=strmid(lines[i],323,1)
	flag19d[i]=strmid(lines[i],327,1)
	endfor
;
	for i=0L,count-1 do begin
	j=where(hd eq hdn[i],j_count)
	if j_count gt 0 then begin
		startable[i].ra=hms2h(ra[j])
		startable[i].dec=dms2d(dc[j])
		startable[i].spectrum=sp[j]
;		startable(i).pmra=pr(j)
;		startable(i).pmdec=pd(j)
		startable[i].px=px[j]
		startable[i].pxe=pe[j]
		startable[i].diameter=dm[j]
;		startable(i).diametere=dme(j)
		startable[i].mj=mj[j]
		startable[i].mh=mh[j]
		startable[i].mk=mk[j]
		startable[i].f12=fi[j]
		startable[i].teff=tf[j]
		startable[i].model='LDN'
		sflag=strjoin([flag14[j],flag15[j],flag16[j],flag17[j],flag18[j]])
		if strlen(strcompress(sflag,/remove_all)) eq 0 then sflag='1' $
							       else sflag='2'
		if viscalerror(dm[j],dme[j],bl=100.0,wl=10e-6) gt 5 then sflag='2' 
		startable[i].sflag=sflag
		startable[i].bflag='C'
;		startable(i).reference=origin(j)
	endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameter values found: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_VERHOELST): No stars with HDN name found!'
;
; Make sure integer spectral types and classes are up-to-date
spec_parse
;
; Update HDN.xdr
if n_elements(uindex) ne 0 then begin
	if uindex[0] ne -1 and keyword_set(update) then $
	update_cat,startable[uindex].starid
endif
;
end
;-------------------------------------------------------------------------------
pro get_merand,init=init,update=update
;
; Read AMBER calibrator list, "merand.hdn", compiled by Merand et al.
; HDN.xdr can be updated (update=1) since not all stars in the Merand list are
; also in SKYCAT used by OYSTER for HDN starids.
; Diameters are LD.
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/merand.hdn',hdn,/col, $
		ignore=['!'])
	get_startable,'HDN'+string(hdn,format='(i6.6)')
	uindex=where(startable.hdn eq 0)
	startable.hdn=hdn
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	lines=''
	status=dc_read_fixed(!oyster_dir+'starbase/merand.hdn',lines, $
		/col,resize=[1],format='(a800)',ignore=['!'])
	n=n_elements(lines)
	hd=lonarr(n)
	hip=lonarr(n)
	ra_h=intarr(n)
	ra_m=intarr(n)
	ra_s=fltarr(n)
	dec_d=intarr(n)
	dec_m=intarr(n)
	dec_s=fltarr(n)
	plx=fltarr(n)
	eplx=fltarr(n)
	spectrum=strarr(n)
	mb=fltarr(n)
	mv=fltarr(n)
	mj=fltarr(n)
	mh=fltarr(n)
	mk=fltarr(n)
	f12=fltarr(n)
	ld=fltarr(n)
	lde=fltarr(n)
	ud_j=fltarr(n)
	ud_h=fltarr(n)
	ud_k=fltarr(n)
;
	for i=0,n-1 do begin
		hd[i]=long(strmid(lines[i],0,6))
		hip[i]=long(strmid(lines[i],7,6))
		ra_h[i]=fix(strmid(lines[i],14,2))
		ra_m[i]=fix(strmid(lines[i],17,2))
		ra_s[i]=float(strmid(lines[i],20,4))
		dec_d[i]=fix(strcompress(strmid(lines[i],25,3),/remove_all))
		dec_m[i]=fix(strmid(lines[i],29,2))
		dec_s[i]=float(strmid(lines[i],32,4))
		plx[i]=float(strmid(lines[i],37,5))
		eplx[i]=float(strmid(lines[i],44,4))
		spectrum[i]=strmid(lines[i],49,13)
		mb[i]=float(strmid(lines[i],63,4))
		mv[i]=float(strmid(lines[i],68,4))
		mj[i]=float(strmid(lines[i],73,4))
		mh[i]=float(strmid(lines[i],78,4))
		mk[i]=float(strmid(lines[i],83,4))
		f12[i]=float(strmid(lines[i],88,4))
		ld[i]=float(strmid(lines[i],93,5))
		lde[i]=float(strmid(lines[i],117,5))
		ud_j[i]=float(strmid(lines[i],99,5))
		ud_h[i]=float(strmid(lines[i],105,5))
		ud_k[i]=float(strmid(lines[i],111,5))
	endfor
;
	for i=0L,count-1 do begin
		j=where(hd eq hdn[i],j_count)
		if j_count gt 0 then begin
			startable[i].diameter=ud_k[j]
;			startable(i).diametere=lde(j)
			startable[i].mv=mv[j]
			startable[i].mj=mj[j]
			startable[i].mh=mh[j]
			startable[i].mk=mk[j]
			startable[i].f12=f12[j]
			startable[i].bflag='C'
		endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameter values found: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_MERAND): No stars with HDN name found!'
;
; Make sure integer spectral types and classes are up-to-date
spec_parse
;
; Update HDN.xdr
if n_elements(uindex) ne 0 then begin
	if uindex[0] ne -1 and keyword_set(update) then $
	update_cat,startable[uindex].starid
endif
;
end
;-------------------------------------------------------------------------------
pro get_borde,init=init,update=update
;
; Read AMBER calibrator list, "borde.hdn", compiled by Borde et al.
; HDN.xdr can be updated (update=1) since not all stars in the Borde list are
; also in SKYCAT used by OYSTER for HDN starids.
; Diameters are LD.
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/borde.hdn',hdn,/col, $
		ignore=['!'])
	get_startable,'HDN'+string(hdn,format='(i6.6)')
	uindex=where(startable.hdn eq 0)
	startable.hdn=hdn
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	lines=''
	status=dc_read_fixed(!oyster_dir+'starbase/borde.hdn',lines, $
		/col,resize=[1],format='(a800)',ignore=['!'])
	n=n_elements(lines)
	hd=lonarr(n)
	hr=intarr(n)
	stars=strarr(n)
	ra=dblarr(n)
	dec=dblarr(n)
	pmra=dblarr(n)
	pmdec=dblarr(n)
	plx=fltarr(n)
	eplx=fltarr(n)
	spectrum=strarr(n)
	ldd=fltarr(n)
	ldde=fltarr(n)
	teff=fltarr(n)
	logg=fltarr(n)
	mb=fltarr(n)
	mv=fltarr(n)
	mj=fltarr(n)
	mh=fltarr(n)
	mk=fltarr(n)
;
	for i=0,n-1 do begin
		words=nameparse(lines[i],'|')
		hd[i]=long(words[0])
		hr[i]=fix(words[1]) > 0
		stars[i]=words[2]
		ra[i]=hms2h(strmid(words[3],1,12))
		d=strcompress(words[4],/remove_all)
		if strlen(d) eq 10 then d=' '+d
		dec[i]=abs(double(strmid(d,0,3)))+double(strmid(d,3,2))/60+double(strmid(d,5,6))/3600
		if float(strmid(d,0,3)) lt 0 then dec[i]=-dec[i]
		pmra[i]=((float(words[5])/3600)/15*100)/cos(dec[i]*!pi/180)
		pmdec[i]=(float(words[6])/3600)*100
		if strlen(strcompress(words[7],/remove_all)) gt 0 then begin
			plx[i]=float(words[7])
			eplx=float(words[8])
		endif
		spectrum[i]=words[9]
		ldd[i]=float(words[10])
		ldde[i]=float(words[11])
		teff[i]=float(words[12])
		logg[i]=float(words[13])
		mb[i]=float(words[25])
		mv[i]=float(words[26])
		mj[i]=float(words[27])
		mh[i]=float(words[29])
		mk[i]=float(words[31])
	endfor
;
	for i=0L,count-1 do begin
		j=where(hd eq hdn[i],j_count)
		if j_count gt 0 then begin
			startable[index[i]].ra=ra[j]
			startable[index[i]].dec=dec[j]
			startable[index[i]].pmra=pmra[j]
			startable[index[i]].pmdec=pmdec[j]
			startable[index[i]].px=plx[j]
			startable[index[i]].pxe=eplx[j]
			startable[index[i]].spectrum=spectrum[j]
			startable[index[i]].diameter=ldd[j]
;			startable(index(i)).diametere=ldde(j)
			startable[index[i]].teff=teff[j]
			startable[index[i]].logg=logg[j]
			startable[index[i]].mv=mv[j]
			startable[index[i]].bv=mb[j]-mv[j]
			startable[index[i]].mj=mj[j]
			startable[index[i]].mh=mh[j]
			startable[index[i]].mk=mk[j]
			startable[index[i]].bflag='C'
		endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameter values found: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_BORDE): No stars with HDN name found!'
;
; Make sure integer spectral types and classes are up-to-date
startable.spectrum=strcompress(startable.spectrum,/remove_all)
spec_parse
;
; Update HDN.xdr
if n_elements(uindex) ne 0 then begin
	if uindex[0] ne -1 and keyword_set(update) then $
	update_cat,startable[uindex].starid
endif
;
end
;-------------------------------------------------------------------------------
pro get_calvin,init=init
;
; Read CALVIN calibrator list, "calibrators.calvin", compiled by Richichi et al.
; These stars have name problems with coordinates: pi.02uma_B02/HD  73108b, 
; 45eri_B02, 45eri,epspsc, 19ari,nu2cma,30gem,v448car,6leo,v337car,51hya,110vir,
; 2lup,59sgr,kapgru.
;
; This file is reformatted and written to calvin.hdn to conform with the OYSTER
; format for star lists.
;
; Diameters are either LD for instr=MIDI, or UDK for VINCI and AMBER
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
calvinhdn=file_search(!oyster_dir+'starbase/calvin.hdn')
;
IF strlen(calvinhdn[0]) eq 0 THEN BEGIN
; We need to read the original file and reformat
name=''
sp=''
lc=''
model=''
instr=''
ref=''
h=0.
q=0.
pmra=0.d0
status=dc_read_free(!oyster_dir+'starbase/calibrators.calvin',name, $
	rah,ram,ras,si,dd,dm,ds,epoch, $
	v,r,j,h,k,l,m,n,q,sp,lc,flag,d,de,model, $
	teff,logg,mass,vsini,vrad,pi,epi,pmra,pmdec,z,instr,ref, $
	/col,ignore=['#'])
nn=n_elements(name)
index=where(finite(v) eq 0,count) & if count gt 0 then v[index]=100
index=where(finite(j) eq 0,count) & if count gt 0 then j[index]=100
index=where(finite(h) eq 0,count) & if count gt 0 then h[index]=100
index=where(finite(k) eq 0,count) & if count gt 0 then k[index]=100
index=where(finite(n) eq 0,count) & if count gt 0 then n[index]=100
index=where(abs(pmra) gt 1e6,count) & if count gt 0 then pmra[index]=0
index=where(abs(pmdec) gt 1e6,count) & if count gt 0 then pmdec[index]=0
index=where(abs(v) gt 1e6,count) & if count gt 0 then v[index]=100
index=where(abs(j) gt 1e6,count) & if count gt 0 then j[index]=100
index=where(abs(h) gt 1e6,count) & if count gt 0 then h[index]=100
index=where(abs(k) gt 1e6,count) & if count gt 0 then k[index]=100
index=where(abs(n) gt 1e6,count) & if count gt 0 then n[index]=100
index=where(teff lt 0,count) & if count gt 0 then teff[index]=0
index=where(logg lt 0,count) & if count gt 0 then logg[index]=0
index=where(pi lt 0,count) & if count gt 0 then pi[index]=0
index=where(epi lt 0,count) & if count gt 0 then epi[index]=0
;
ui=uniq(name)
nn=n_elements(ui)
name=name[ui]
;
starids=strarr(nn)
hdn=lonarr(nn)
;
for i=0,nn-1 do begin
	words=nameparse(name[i],'_')
	if strpos(words[0],'hd') ge 0 then begin
		hdn[i]=long(strmid(words[0],2,6))
		starids[i]='HDN'+string(hdn[i],format='(i6.6)')
	endif
	if strpos(words[0],'hr') ge 0 then begin
		hdn[i]=cri(long(strmid(words[0],2,6)),'bsc-hdn')
		if hdn[i] ne 0 then starids[i]='HDN'+string(hdn[i],format='(i6.6)')
	endif
endfor
;
; Stars without HD or HR number so far: lookup in SIMBAD
ra=float(rah)+float(ram)/60+float(ras)/3600
dec=float(si)*(float(dd)+float(dm)/60+float(ds)/3600)
objid='OBJ'+esoid(ra,dec) & objid=objid[ui]
;
for i=0,nn-1 do begin
	if hdn[i] eq 0 then begin
		star=cri_simbad(objid[i])
		print,'For name ',name[i],' found star ',star
		hd=cri_simbad(star,table)
		if abs(ra[ui[i]]-table.ra)*15.*3600. gt 0.2 or $
		   abs(dec[ui[i]]-table.dec)*3600. gt 0.2 then begin
			print,'***Error: misindentification!'
			return
		endif
		words=nameparse(hd)
		if words[0] eq 'HD' then begin
			hdn[i]=long(words[1])
			starids[i]='HDN'+string(hdn[i],format='(i6.6)')
		endif
	endif
endfor
openw,unit,!oyster_dir+'starbase/calvin.hdn',/get_lun
for i=0,nn-1 do begin
if hdn[i] ne 0 then begin
	printf,unit,hdn[i],ra[ui[i]],dec[ui[i]],pmra[ui[i]],pmdec[ui[i]], $
		pi[ui[i]],epi[ui[i]], $
		v[ui[i]],j[ui[i]],h[ui[i]],k[ui[i]],n[ui[i]],d[ui[i]],de[ui[i]], $
		sp[ui[i]],name[i],instr[ui[i]], $
		format='(i6,13(2x,f12.5),3(2x,a10))'
endif
endfor
free_lun,unit
ENDIF
;
spectrum=''
name=''
instr=''
status=dc_read_free(!oyster_dir+'starbase/calvin.hdn', $
	hd,ra,dec,pmra,pmdec,plx,eplx,mv,mj,mh,mk,mn,ldd,eldd, $
	spectrum,name,instr,/col, $
	ignore=['!'])
model=strarr(n_elements(instr))
index=where(instr eq 'VINCI' or instr eq 'AMBER_K',count)
if count gt 0 then model[index]='UDK'
index=where(instr eq 'MIDI',count)
if count gt 0 then model[index]='LD'
;
if n_elements(startable) eq 0 or init then begin
	create_startable,'HDN'+string(hd,format='(i6.6)')
	startable.hdn=hd
	startable.name=name
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	for i=0L,count-1 do begin
		j=where(hd eq hdn[i],j_count) & j=j[0]
		if j_count gt 0 then begin
			startable[index[i]].ra=ra[j]
			startable[index[i]].dec=dec[j]
			startable[index[i]].pmra=((pmra[j]/1000)/15)*100/cos(dec[j]*!pi/180)
			startable[index[i]].pmdec=pmdec[j]/10
			startable[index[i]].px=plx[j]
			startable[index[i]].pxe=eplx[j]
			startable[index[i]].spectrum=spectrum[j]
			startable[index[i]].diameter=ldd[j]
			startable[index[i]].mv=mv[j]
			startable[index[i]].mj=mj[j]
			startable[index[i]].mh=mh[j]
			startable[index[i]].mk=mk[j]
			startable[index[i]].f12=n2jy(mn[j])
			startable[index[i]].model=model[j]
			startable[index[i]].bflag='C'
		endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameter values found: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_CALVIN): No stars with HDN name found!'
;
; Make sure integer spectral types and classes are up-to-date
spec_parse
;
end
;-------------------------------------------------------------------------------
pro get_calvin,init=init,update=update
;
; Read the CALVIN calibrator list calvin.hdn and if not available, read the
; original list calibrators.calvin compiled by Richichi et al. and reformat.
; These stars have name problems with coordinates: pi.02uma_B02/HD  73108b, 
; 45eri_B02, 45eri,epspsc, 19ari,nu2cma,30gem,v448car,6leo,v337car,51hya,110vir,
; 2lup,59sgr,kapgru.
;
; Diameters are either LD for instr=MIDI, or UDK for VINCI and AMBER
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. If update=1, update hdn.xdr for HDN numbers not found.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
; The input file is derived from calibrators.calvin (export file)
calvinhdn=file_search(!oyster_dir+'starbase/calvin.hdn')
;
IF strlen(calvinhdn[0]) eq 0 THEN BEGIN
; We need to read the export file, reformat, and write to calvin.hdn
name=''
sp=''
lc=''
model=''
instr=''
ref=''
h=0.
q=0.
pmra=0.d0
status=dc_read_free(!oyster_dir+'starbase/calibrators.calvin',name, $
	rah,ram,ras,si,dd,dm,ds,epoch, $
	v,r,j,h,k,l,m,n,q,sp,lc,flag,d,de,model, $
	teff,logg,mass,vsini,vrad,pi,epi,pmra,pmdec,z,instr,ref, $
	/col,ignore=['#'])
nn=n_elements(name)
index=where(finite(v) eq 0,count) & if count gt 0 then v[index]=100
index=where(finite(j) eq 0,count) & if count gt 0 then j[index]=100
index=where(finite(h) eq 0,count) & if count gt 0 then h[index]=100
index=where(finite(k) eq 0,count) & if count gt 0 then k[index]=100
index=where(finite(n) eq 0,count) & if count gt 0 then n[index]=100
index=where(abs(pmra) gt 1e6,count) & if count gt 0 then pmra[index]=0
index=where(abs(pmdec) gt 1e6,count) & if count gt 0 then pmdec[index]=0
index=where(abs(v) gt 1e6,count) & if count gt 0 then v[index]=100
index=where(abs(j) gt 1e6,count) & if count gt 0 then j[index]=100
index=where(abs(h) gt 1e6,count) & if count gt 0 then h[index]=100
index=where(abs(k) gt 1e6,count) & if count gt 0 then k[index]=100
index=where(abs(n) gt 1e6,count) & if count gt 0 then n[index]=100
index=where(teff lt 0,count) & if count gt 0 then teff[index]=0
index=where(logg lt 0,count) & if count gt 0 then logg[index]=0
index=where(pi lt 0,count) & if count gt 0 then pi[index]=0
index=where(epi lt 0,count) & if count gt 0 then epi[index]=0
;
ui=uniq(name)
nn=n_elements(ui)
name=name[ui]
;
starids=strarr(nn)
hdn=lonarr(nn)
;
for i=0,nn-1 do begin
	words=nameparse(name[i],'_')
	if strpos(words[0],'hd') ge 0 then begin
		hdn[i]=long(strmid(words[0],2,6))
		starids[i]='HDN'+string(hdn[i],format='(i6.6)')
	endif
	if strpos(words[0],'hr') ge 0 then begin
		hdn[i]=cri(long(strmid(words[0],2,6)),'bsc-hdn')
		if hdn[i] ne 0 then starids[i]='HDN'+string(hdn[i],format='(i6.6)')
	endif
endfor
;
; Stars without HD or HR number so far: lookup in SIMBAD
ra=float(rah)+float(ram)/60+float(ras)/3600
dec=float(si)*(float(dd)+float(dm)/60+float(ds)/3600)
objid='OBJ'+esoid(ra,dec) & objid=objid[ui]
;
for i=0,nn-1 do begin
	if hdn[i] eq 0 then begin
		star=cri_simbad(objid[i])
		print,'For name ',name[i],' found star ',star
		hd=cri_simbad(star,table)
		if abs(ra[ui[i]]-table.ra)*15.*3600. gt 0.2 or $
		   abs(dec[ui[i]]-table.dec)*3600. gt 0.2 then begin
			print,'***Error: misindentification!'
			return
		endif
		words=nameparse(hd)
		if words[0] eq 'HD' then begin
			hdn[i]=long(words[1])
			starids[i]='HDN'+string(hdn[i],format='(i6.6)')
		endif
	endif
endfor
openw,unit,!oyster_dir+'starbase/calvin.hdn',/get_lun
for i=0,nn-1 do begin
if hdn[i] ne 0 then begin
	printf,unit,hdn[i],ra[ui[i]],dec[ui[i]],pmra[ui[i]],pmdec[ui[i]], $
		pi[ui[i]],epi[ui[i]], $
		v[ui[i]],j[ui[i]],h[ui[i]],k[ui[i]],n[ui[i]],d[ui[i]],de[ui[i]], $
		sp[ui[i]],name[i],instr[ui[i]], $
		format='(i6,13(2x,f12.5),3(2x,a10))'
endif
endfor
free_lun,unit
ENDIF
;
if n_elements(startable) eq 0 or init then begin
	status=dc_read_free(!oyster_dir+'starbase/calvin.hdn',hdn,/col, $
		ignore=['!'])
	get_startable,'HDN'+string(hdn,format='(i6.6)')
	uindex=where(startable.hdn eq 0)
	startable.hdn=hdn
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
	spectrum=''
	name=''
	instr=''
	status=dc_read_free(!oyster_dir+'starbase/calvin.hdn', $
		hd,ra,dec,pmra,pmdec,plx,eplx,mv,mj,mh,mk,mn,ldd,eldd, $
		spectrum,name,instr,/col, $
		ignore=['!'])
	model=strarr(n_elements(instr))
	index=where(instr eq 'VINCI' or instr eq 'AMBER_K',count)
	if count gt 0 then model[index]='UDK'
	index=where(instr eq 'MIDI',count)
	if count gt 0 then model[index]='LD'
;
	for i=0L,count-1 do begin
		j=where(hd eq hdn[i],j_count) & j=j[0]
		if j_count gt 0 then begin
			startable[index[i]].ra=ra[j]
			startable[index[i]].dec=dec[j]
			startable[index[i]].pmra=((pmra[j]/1000)/15)*100 $
				/cos(dec[j]*!pi/180)
			startable[index[i]].pmdec=pmdec[j]/10
			startable[index[i]].px=plx[j]
			startable[index[i]].pxe=eplx[j]
			startable[index[i]].spectrum=spectrum[j]
			startable[index[i]].diameter=ldd[j]
			startable[index[i]].mv=mv[j]
			startable[index[i]].mj=mj[j]
			startable[index[i]].mh=mh[j]
			startable[index[i]].mk=mk[j]
			startable[index[i]].f12=n2jy(mn[j])
			startable[index[i]].model=model[j]
			startable[index[i]].bflag='C'
		endif
	endfor
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameter values found: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_CALVIN): No stars with HDN name found!'
;
; Make sure integer spectral types and classes are up-to-date
spec_parse
;
; Update HDN.xdr
if n_elements(uindex) ne 0 then begin
	if uindex[0] ne -1 and keyword_set(update) then $
	update_cat,startable[uindex].starid
endif
;
end
;-------------------------------------------------------------------------------
function read_jsdc
;
; Get name of latest FITS file for JSDC, to be read by  get_jsdc
;
jsdc_fits=file_search(!oyster_dir+'starbase/JMMC/jsdc_????_??_??.fits')
if strlen(jsdc_fits[0]) eq 0 then begin
	print,'Error(GET_JSDC): FITS file not found!'
	return,-1
endif
; Use latest version
jsdc_fits=jsdc_fits[n_elements(jsdc_fits)-1]
;
d=mrdfits(jsdc_fits,1,header)
; We can only handle HD and HIP IDs, but must remove, e.g., TYC IDs
tags=tag_names(d)
index=where(strpos(tags,'MAINID_SIMBAD') ge 0)
index=index[0]
if index lt 0 then names=d.name else names=d.(index)
index=where(strmid(names,0,3) eq 'HIP' $
	 or strmid(names,0,2) eq 'HD',count)
d=d[index]
;
; Remove blanks and reformat star IDs to OYSTER format
d.name=strcompress(d.name,/remove_all)
index=where(strmid(d.name,0,3) eq 'HIP')
hip=long(strmid(d[index].name,3,max(strlen(d.name))-3))
d[index].name='HIP'+string(hip,format='(i6.6)')
index=where(strmid(d.name,0,2) eq 'HD')
hdn=long(strmid(d[index].name,2,max(strlen(d.name))-2))
d[index].name='HDN'+string(hdn,format='(i6.6)')
;
return,d
;
; Obsolete!
; Find as many HIP IDs as possible for HD IDs and rename
hip=cri(long(strmid(d[index].name,3,6)),'hdn-hic')
jndex=where(hip gt 0)
d[index[jndex]].name='HIP'+string(hip[jndex],format='(i6.6)')
; Find as many HD IDs as possible for HIP IDs and rename
hdn=cri(long(strmid(d[index].name,3,6)),'hic-hdn')
jndex=where(hdn gt 0)
d[index[jndex]].name='HDN'+string(hdn[jndex],format='(i6.6)')
;
return,d
;
end
;-------------------------------------------------------------------------------
pro get_jsdc,init=init,best=best,badcal=badcal
;
; Read latest JMMC calibrator catalog, jsdc_YYYY_MM_DD.fits.
; (currently at VizieR catalog II/346)
;
; Diameters are LD.
;
; II/346    JMMC Stellar Diameters Catalogue - JSDC. Version 2  (Bourges+, 2017)
; ==============================================================================
; An all-sky catalogue of computed star diameters.
;     Bourges L., Mella G., Lafrasse S., Duvert G., Chelli A., Le Bouquin J.-B.,
;     Delfosse X., Chesneau O.
;    <ASP Conference Series, Vol. 485, p.223 (2014)>
;    =2014ASPC..485..223B
;    =2017yCat.2346....0B
;
; If a StarTable is present (and init=0), update it with the information found.
; If no Startable is present or init=1, then read entire list and initialize
; a StarTable. Keyword update not available since JSDC is based on HIP IDs.
; If best=1, select only the highest grade calibrators.
;
; If keyword badcal is set, filter out stars in file badcal.xml.
;
common StarBase,startable,notes
;
if n_elements(init) eq 0 then init=0
;
; Read entire catalog if no startable exists or init=1
if n_elements(startable) eq 0 or init then begin
	init=1
	d=read_jsdc()
	create_startable,d.name
	index_hdn=where(strmid(startable.starid,0,3) eq 'HDN')
	index_hip=where(strmid(startable.starid,0,3) eq 'HIP')
	startable[index_hdn].hdn=long(strmid(startable[index_hdn].starid,3,6))
	startable[index_hip].hic=long(strmid(startable[index_hip].starid,3,6))
endif
;
index=where(startable.diameter ne 0,diameter_count1)
;
; Currently, we can only handle HDN and HIC IDs
index=where(startable.hic ne 0 or startable.hdn ne 0,count)
; index_hdn=where(startable.hdn ne 0,count_hdn)
; index_hic=where(startable.hic ne 0,count_hic)
;
if count gt 0 then begin
	hic=startable[index].hic
	hdn=startable[index].hdn
	startable[index].bflag='.'
;
;	Read FITS file
	d=read_jsdc() & n=n_elements(d)
	hdn_jsdc=lonarr(n)
	hip_jsdc=lonarr(n)
	i=where(strmid(d.name,0,2) eq 'HD')
	hdn_jsdc[i]=long(strmid(d[i].name,3,6))
	i=where(strmid(d.name,0,3) eq 'HIP')
	hip_jsdc[i]=long(strmid(d[i].name,3,6))
;
;	Check for tag names
	tags=tag_names(d)
	index=where(strpos(tags,'SPTYPE') ge 0)
	index=index[0]
	print,'Using SPTYPE from '+tags[index]
	sptype=d.(index)
;
;	Read required fields
;	sptype=d.sptype
	ldd_chi2=d.e_ldd*0+1
	av=d.vmag*0
	if keyword_set(best) then begin
		index_best=where(av lt 1 and ldd_chi2 lt 3)
		d=d[index_best]
		sptype=sptype[index_best]
		ldd_chi2=ldd_chi2[index_best]
		av=av[index_best]
	endif
	ra=d.raj2000/15
	dec=d.dej2000
;	pmra=d.pmra
;	pmdec=d.pmdec
;	plx=d.plx
;	e_plx=d.e_plx
	bmag=d.bmag
	vmag=d.vmag
	rmag=d.rmag
	imag=d.imag
	jmag=d.jmag
	hmag=d.hmag
	kmag=d.kmag
	ldd=d.ldd
	e_ldd=d.e_ldd
;	teff=d.teff_sptype
;	logg=d.logg_sptype
	ub=limbcoeff(d.ldd/d.uddb)
	uv=limbcoeff(d.ldd/d.uddv)
	ur=limbcoeff(d.ldd/d.uddr)
	ui=limbcoeff(d.ldd/d.uddi)
	uj=limbcoeff(d.ldd/d.uddj)
	uh=limbcoeff(d.ldd/d.uddh)
	uk=limbcoeff(d.ldd/d.uddk)
	un=limbcoeff(d.ldd/d.uddn)
;	http://en.wikipedia.org/wiki/Photometric_system
	wl=[0.445,0.551,0.658,0.806,1.220,1.630,2.190,9.67]
	tm=transpose([[ub],[uv],[ur],[ui],[uj],[uh],[uk],[un]])
	n=n_elements(d)
	a0=fltarr(n)
	a1=fltarr(n)
	a2=fltarr(n)
	for i=0L,n-1 do begin
		r=poly_fit(wl,tm[*,i],2)
		a0[i]=r[0]
		a1[i]=r[1]
		a2[i]=r[2]
	endfor
;
;	Transfer data to StarTable
	for i=0L,count-1 do begin
		if init then begin
			i=lindgen(n_elements(startable))
			j=lindgen(n_elements(d))
			j_count=1
		endif else begin
			j=where((hip_jsdc eq hic[i] and hic[i] ne 0) $
			     or (hdn_jsdc eq hdn[i] and hdn[i] ne 0),j_count)
			j=j[0]
		endelse
		if j_count ge 1 then begin
			startable[i].ra=ra[j]
			startable[i].dec=dec[j]
;			startable(i).pmra=((pmra(j)/10)/15) $
;					 /cos(dec(j)*!pi/180)
;			startable(i).pmdec=pmdec(j)/10
;			startable(i).px=plx(j)
;			startable(i).pxe=e_plx(j)
			startable[i].diameter=ldd[j]
;			startable(i).diametere=e_ldd(j)
			startable[i].a0=a0[j]
			startable[i].a1=a1[j]/1e3
			startable[i].a2=a2[j]/1e6
			startable[i].spectrum=sptype[j]
;			startable(i).logg=logg(j)
;			startable(i).teff=teff(j)
			startable[i].bv=bmag[j]-vmag[j]
			startable[i].mv=vmag[j]
			startable[i].ri=rmag[j]-imag[j]
			startable[i].mj=jmag[j]
			startable[i].mh=hmag[j]
			startable[i].mk=kmag[j]
			startable[i].bflag='C'
		endif
		if init then break
	endfor
	index=where(startable.hic ne 0,ic)
	if ic gt 0 then begin
	hdn=cri(startable[index].hic,'hic-hdn')
	jndex=where(hdn ge 0,jc)
	if jc gt 0 then $
	startable[index[jndex]].hdn=hdn[jndex]
	endif
;
	index=where(startable.diameter ne 0,diameter_count2)
	print,'Number of new diameters added: ', $
		diameter_count2-diameter_count1,'.', $
		format='(a,i6,a)'
;
endif else print,'Warning(GET_JSDC): No stars with HIP name found!'
;
; Make sure integer spectral types and classes are up-to-date
spec_parse,startable.spectrum
;
if n_elements(starids) ne 0 then startable.starid=starids
;
if keyword_set(badcal) then begin
	dec_rad=startable.dec*!pi/180
	ra_deg=startable.ra*15
;	badcal.xml: http://apps.jmmc.fr/badcal/show.jsp?type=all&display=simple
;	https://www.astro.umd.edu/~eshaya/PDS/PDS/read_votable.pro
	r=read_votable[!oyster_dir+'starbase/badcal.xml']
	bad_index=intarr(n_elements(startable))
	for i=0,n_elements(r.id)-1 do begin
		d=sqrt(((ra_deg-r.ra[i])*cos(dec_rad))^2 $
		       +(startable.dec-r.dec[i])^2)
		j=where(d eq min(d)) & j=j[0]
		if min(d) lt 0.5/3600 then begin
			bad_index[j]=1
		endif
	endfor
	startable=startable[where(bad_index eq 0)]
endif
;
end
;************************************************************************Block 2
pro get_parallax
;
; Read Yale (1991) parallaxes in starbase/parallax.hdn. 
; Note: data found will replace current values!
;
common StarBase,startable,notes
;
index=where(startable.px ne 0,p_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	px=startable[index].px
	pxe=startable[index].pxe
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/parallax.hdn', $
		n,p,pe,/col,ignore=['!'])
;
	for i=0,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			px[i]=p[j[0]]
			pxe[i]=pe[j[0]]
		endif
	endfor
	startable[index].px=px
	startable[index].pxe=pxe
;
	index=where(startable.px ne 0,p_count2)
	print,'Number of parallaxes increased by: ',p_count2-p_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_PARALLAX): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_leeuwen
;
; Read new Hipparcos reduction by Leeuwen
; http://cdsarc.u-strasbg.fr/viz-bin/Cat?I/311
;
common StarBase,startable,notes
;
index=where(startable.px ne 0,p_count1)
;
xdr_file=!oyster_dir+'catalogs/leeuwen/hip2.xdr'
f=file_search(xdr_file) & f=f[0]
;
if strlen(f) eq 0 then begin
;
	dat_file=!oyster_dir+'catalogs/leeuwen/hip2.dat'
	f=file_search(dat_file) & f=f[0]
	openr,unit,dat_file,/get_lun,error=status
	if status ne 0 then begin
		print,'***Error(GET_LEEUWEN): error opening file ',dat_file,'!'
		return
	endif
	n=117955
	hip=lonarr(n)
	px=fltarr(n)
	pxe=fltarr(n)
	for i=0l,n-1 do begin
		readf,unit,id,sn,so,nc,ra,dec,plx,pmra,pmdec,rae,dece,plxe
		hip[i]=id
		px[i]=plx
		pxe[i]=plxe
	endfor
	px=px/1000
	pxe=pxe/1000
	save,hip,px,pxe,file=xdr_file
;
endif else restore,xdr_file
;
index=where(startable.hic ne 0,count)
if count gt 0 then begin
	plx=startable[index].px
	plxe=startable[index].pxe
	hic=startable[index].hic
;
	for i=0,count-1 do begin
		j=where(hip eq hic[i],j_count)
		if j_count gt 0 then begin
			plx[i]=px[j[0]]
			plxe[i]=pxe[j[0]]
		endif
	endfor
	startable[index].px=plx
	startable[index].pxe=plxe
;
	index=where(startable.px ne 0,p_count2)
	print,'Number of parallaxes increased by: ',p_count2-p_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_LEEUWEN): No stars with HIC name found!'
;
end
;-------------------------------------------------------------------------------
pro get_position
;
; Read Yale (1991) positions in catalogs/npoi/position.hdn. 
; Note: data will replace current non-zero values!
;
common StarBase,startable,notes
;
index=where(startable.ra ne 0,p_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	ra=startable[index].ra
	dec=startable[index].dec
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/position.hdn', $
		n,r,d,/col,ignore=['!'])
;
	for i=0,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			ra[i]=r[j[0]]
			dec[i]=d[j[0]]
		endif
	endfor
	startable[index].ra=ra
	startable[index].dec=dec
;
	index=where(startable.ra ne 0,p_count2)
	print,'Number of positions increased by: ',p_count2-p_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_POSITION): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_orbit
;
common StarBase,startable,notes
;
stars=startable.starid
rename_starids,'fkv-wds'
rename_starids,'bsc-wds'
table=startable
get_wds
;
for i=0,n_elements(stars)-1 do begin
	if table[i].wds ne 0 then begin
		index=where(startable.wds eq table[i].wds,count)
		if count gt 0 then begin
			table[i].a=startable[index[0]].a
			table[i].e=startable[index[0]].e
			table[i].i=startable[index[0]].i
			table[i].o=startable[index[0]].o
			table[i].n=startable[index[0]].n
			table[i].p=startable[index[0]].p
			table[i].t=startable[index[0]].t
		endif
	endif
endfor
;
startable=table
startable.starid=stars
;
end
;-------------------------------------------------------------------------------
pro get_reandma
;
; For stars loaded in StarTable, find them in the catalog of HgMn stars by
; Renson and Manfroid (by position, precision=2").
;
common StarBase,startable,notes
;
; Read notes to catalog
notes=''
status=dc_read_fixed(!catalogs_dir+'proprietary/reandma/notes.dat',notes,/col, $
	format='(a80)')
n=n_elements(l)
note_ids=long(notes)

; Read catalog.xdr to startable_rm, if it exists
rmfile=!catalogs_dir+'proprietary/reandma/catalog.xdr'
result=file_search(rmfile)
IF strlen(result[0]) ne 0 THEN restore,rmfile $
			  ELSE BEGIN
;
; Make copy of startable
if n_elements(startable) ne 0 then startable_in=startable
;
; Read catalog data from Renson and Manfroid
l=''
status=dc_read_fixed(!catalogs_dir+'proprietary/reandma/catalog.dat',l,/col, $
	format='(a155)',ignore=['!'])
n=n_elements(l)
create_startable,strarr(n)
;
for i=0,n-1 do begin
	startable[i].reference=strmid(l[i],1,5)
	startable[i].bsc=fix(strcompress(strmid(l[i],20,4),/remove_all))
	startable[i].hdn=long(strcompress(strmid(l[i],7,5),/remove_all))
	startable[i].hic=long(strcompress(strmid(l[i],14,6),/remove_all))
	startable[i].name=strmid(l[i],49,11)
	rah=float(strmid(l[i],74,2))
	ram=float(strmid(l[i],76,2))
	ras=float(strmid(l[i],78,5))
	startable[i].ra=rah+ram/60+ras/3600
	decd=float(strcompress(strmid(l[i],83,3),/remove_all))
	decm=float(strmid(l[i],86,2))
	decs=float(strmid(l[i],88,4))
	startable[i].dec=abs(decd)+decm/60+decs/3600
	if decd lt 0 then startable[i].dec=-startable[i].dec
	startable[i].mv=float(strcompress(strmid(l[i],92,4),/remove_all))
	startable[i].bv=float(strcompress(strmid(l[i],96,5),/remove_all))
	startable[i].spectrum=strmid(l[i],130,14)
endfor
startable_rm=startable
save,startable_rm,filename=!catalogs_dir+'proprietary/reandma/catalog.xdr'
;
startable=startable_in
sep=fltarr(n_elements(startable))
;
if 0 then begin
for i=0,n_elements(startable)-1 do $
	sep[i]=min(winkel(startable_rm.ra,startable_rm.dec,$
		startable[i].ra,startable[i].dec)*3600)
	histograph,sep,binsize=0.05,min=0,max=4
endif
;
ENDELSE
;
for i=0,n_elements(startable)-1 do begin
;
sep=winkel(startable_rm.ra,startable_rm.dec,startable[i].ra,startable[i].dec)
index=where(sep*3600 lt 2,n)
if n eq 1 then begin
	print,startable[i].starid+' is in list of Renson and Manfroid!'
;	Retrieve notes
	jndex=where(note_ids eq long(startable_rm[index].reference))
	print,notes[jndex]
endif
;
endfor
;
end
;-------------------------------------------------------------------------------
pro get_standwb
;
; For stars loaded in StarTable, find them in the catalog of HgMn stars by
; Stickland and Weatherby (by position, precision=2").
;
common StarBase,startable,notes
;
; Read catalog.xdr to startable_sw, if it exists
rmfile=!catalogs_dir+'proprietary/standwb/SandW.xdr'
result=file_search(rmfile)
IF strlen(result[0]) ne 0 THEN restore,rmfile $
			  ELSE BEGIN
;
; Make copy of startable
if n_elements(startable) ne 0 then startable_in=startable
;
; Read catalog data from Renson and Manfroid
l=''
status=dc_read_fixed(!catalogs_dir+'proprietary/standwb/SandW.txt',l,/col, $
	format='(a166)',ignore=['!'])
n=n_elements(l)
create_startable,strarr(n)
;
for i=0,n-1 do begin
	words=nameparse(l[i],'|')
	startable[i].name=words[3]
	startable[i].reference=words[4]
	startable.mv=float(words[8])
	startable.bv=float(words[7])-float(words[8])
	startable[i].spectrum=words[11]
	coords=nameparse(words[5])
	rah=float(coords[0])
	ram=float(coords[1])
	ras=float(coords[2])
	decd=float(coords[3])
	decm=float(coords[4])
	decs=float(coords[5])
	startable[i].ra=rah+ram/60+ras/3600
	startable[i].dec=abs(decd)+decm/60+decs/3600
	if decd lt 0 then startable[i].dec=-startable[i].dec
endfor
startable_sw=startable
save,startable_sw,filename=!catalogs_dir+'proprietary/standwb/SandW.xdr'
;
startable=startable_in
sep=fltarr(n_elements(startable))
;
if 0 then begin
for i=0,n_elements(startable)-1 do $
	sep[i]=min(winkel(startable_sw.ra,startable_sw.dec,$
		startable[i].ra,startable[i].dec)*3600)
	histograph,sep,binsize=0.05,min=0,max=4
endif
;
ENDELSE
;
for i=0,n_elements(startable)-1 do begin
;
sep=winkel(startable_sw.ra,startable_sw.dec,startable[i].ra,startable[i].dec)
index=where(sep*3600 lt 2,n)
if n eq 1 then begin
	print,startable[i].starid+' is in list of Stickland and Weatherby!'
	print,startable[i].spectrum
endif
;
endfor
;
end
;************************************************************************Block 3
pro get_jhk_bsc
;
; Read JHK magnitudes from BSC based list of IRSA data
; Version using jhk_bsc.bsc, based on get_bsc.
;
; The data were obtained from http://irsa.ipac.caltech.edu/cgi-bin/Gator/nph-dd,
; using the following list (head only, all HIP which have HDN):
; \EQUINOX = 'J2000.0'
; |   ra           |   dec         |  major |
; |   double       |   double      | double |
;    0.00089920883       1.0890088     1.0
;     0.0042646079      -19.498840     1.0
;     0.0050243058       38.859279     1.0
;     0.0086292358      -51.893546     1.0
;     0.0099734379      -40.591202     1.0
;
; Coordinates used were from Hipparcos, available for all but 96 stars.
;
common StarBase,startable,notes
;
; cntr_u	Counter
; dist_x	Distance in arcseconds from target position
; pang_x	Position angle for offset from target position
; ra_u		Right ascension of target
; dec_u		Declination of target
; major_u	Major axis of target ellipse
; ra		Right ascension of IR point source
; dec		Declination of IR point source
; clon
; clat
; err_maj	Major axis of 1-sigma error ellipse
; err_min	Minor axis of 1-sigma error ellipse
; err_ang	Position angle of error ellipse
; designation	Sexagesimal coordinate designation of target
; j_m 		J band selected "default" magnitude
; j_cmsig	Corrected J band photometric uncertainty
; j_msigcom	Combined (total) J band photometric uncertainty
; j_snr		H band "scan" signal-to-noise ratio
; h_m 		H band selected "default" magnitude
; h_cmsig	Corrected H band photometric uncertainty
; h_msigcom	Combined (total) H band photometric uncertainty
; h_snr		H band "scan" signal-to-noise ratio
; k_m 		K band selected "default" magnitude
; k_cmsig	Corrected K band photometric uncertainty
; k_msigcom	Combined (total) K band photometric uncertainty
; k_snr		K band "scan" signal-to-noise ratio
; ph_qual	Flag indicating photometric quality of source
; rd_flg	Source of JHK "default"mags (AKA "read flag")
; bl_flg	Indicates # JHK components fit simultaneously to source
; cc_flg	Indicates JHK artifact contamination and/or confusion
; ndet		Number of >3-sig. ap. mag measurements, # possible (jjhhkk)
; gal_contam	Flag indicating if src is contaminated by extended source
; mp_flg	Src is positionally associated with an asteroid, comet, etc.
;
index=where(startable.mj ne +100,mj_count1)
;
index=where(startable.bsc ne 0,count)
if count gt 0 then begin
;
mj=startable[index].mj
mh=startable[index].mh
mk=startable[index].mk
bsc=startable[index].bsc
;
xdr_file=file_search(!oyster_dir+'starbase/jhk_bsc.xdr') & xdr_file=xdr_file[0]
if strlen(xdr_file) eq 0 then begin
status=dc_read_free(!oyster_dir+'starbase/jhk_bsc.bsc',n,/col,ignore=['!'])
designation=''
status=dc_read_free(!oyster_dir+'starbase/jhk_bsc.dat',cntr_u,dist_x,pang_x, $
	ra_u,dec_u,major_u,ra,dec,clon,clat,err_maj,err_min,err_ang,designation, $
	j_m,j_cmsig,j_msigcom,j_snr, $
	h_m,h_cmsig,h_msigcom,h_snr, $
	k_m,k_cmsig,k_msigcom,k_snr, $
	ignore=['!'],null=' 0  ',/col)
n=n[cntr_u-1]
save,n,h_m,j_m,k_m,filename=!oyster_dir+'starbase/jhk_bsc.xdr'
endif else restore,xdr_file
;
for i=0L,count-1 do begin
	j=where(n eq bsc[i],j_count)
	if j_count gt 0 then begin
		mj[i]=j_m[j[0]]
		mh[i]=h_m[j[0]]
		mk[i]=k_m[j[0]]
	endif
endfor
startable[index].mj=mj
startable[index].mh=mh
startable[index].mk=mk
;
index=where(startable.mj ne +100,mj_count2)
print,'Number of new JHK values found: ',mj_count2-mj_count1,'.', $
	format='(a,i5,a)'
;
endif else print,'Warning(GET_JHK): No stars with BSC name found!'
;
end
;-------------------------------------------------------------------------------
pro get_jhk_sky
;
; Read JHK magnitudes from BSC based list of IRSA data
; Version using jhk_sky.hdn, based on get_sky.
;
common StarBase,startable,notes
;
; cntr_u	Counter
; dist_x	Distance in arcseconds from target position
; pang_x	Position angle for offset from target position
; ra_u		Right ascension of target
; dec_u		Declination of target
; major_u	Major axis of target ellipse
; ra		Right ascension of IR point source
; dec		Declination of IR point source
; err_maj	Major axis of 1-sigma error ellipse
; err_min	Minor axis of 1-sigma error ellipse
; err_ang	Position angle of error ellipse
; designation	Sexagesimal coordinate designation of target
; j_m 		J band selected "default" magnitude
; j_cmsig	Corrected J band photometric uncertainty
; j_msigcom	Combined (total) J band photometric uncertainty
; j_snr		H band "scan" signal-to-noise ratio
; h_m 		H band selected "default" magnitude
; h_cmsig	Corrected H band photometric uncertainty
; h_msigcom	Combined (total) H band photometric uncertainty
; h_snr		H band "scan" signal-to-noise ratio
; k_m 		K band selected "default" magnitude
; k_cmsig	Corrected K band photometric uncertainty
; k_msigcom	Combined (total) K band photometric uncertainty
; k_snr		K band "scan" signal-to-noise ratio
; ph_qual	Flag indicating photometric quality of source
; rd_flg	Source of JHK "default"mags (AKA "read flag")
; bl_flg	Indicates # JHK components fit simultaneously to source
; cc_flg	Indicates JHK artifact contamination and/or confusion
; ndet		Number of >3-sig. ap. mag measurements, # possible (jjhhkk)
; gal_contam	Flag indicating if src is contaminated by extended source
; mp_flg	Src is positionally associated with an asteroid, comet, etc.
;
index=where(startable.mj ne +100,mj_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
;
mj=startable[index].mj
mh=startable[index].mh
mk=startable[index].mk
hdn=startable[index].hdn
;
xdr_file=file_search(!oyster_dir+'starbase/jhk_sky.xdr') & xdr_file=xdr_file[0]
if strlen(xdr_file) eq 0 then begin
status=dc_read_free(!oyster_dir+'starbase/jhk_sky.hdn',n,/col,ignore=['!'])
designation=''
status=dc_read_free(!oyster_dir+'starbase/jhk_sky.dat',cntr_u,dist_x,pang_x, $
	ra_u,dec_u,major_u,ra,dec, $
	j_m,j_cmsig,h_m,h_cmsig,k_m,k_cmsig, $
	rd_flg, $
	ignore=['!'],null=' 0  ',/col)
n=n[cntr_u-1]
save,n,h_m,j_m,k_m,filename=!oyster_dir+'starbase/jhk_sky.xdr'
endif else restore,xdr_file
;
	for i=0L,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			mj[i]=j_m[j[0]]
			mh[i]=h_m[j[0]]
			mk[i]=k_m[j[0]]
		endif
	endfor
	startable[index].mj=mj
	startable[index].mh=mh
	startable[index].mk=mk
;
	index=where(startable.mj ne +100,mj_count2)
	print,'Number of new JHK values found: ',mj_count2-mj_count1,'.', $
		format='(a,i5,a)'
;
endif else print,'Warning(GET_JHK): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_jhk_hic
;
; Read JHK magnitudes from HIP based list of IRSA data
; Version using jhk_hip.hic, based on get_hip.
;
; The data were obtained from http://irsa.ipac.caltech.edu/cgi-bin/Gator/nph-dd,
; using the following list (head only, all HIP which have HDN):
; \EQUINOX = 'J2000.0'
; |   ra           |   dec         |  major |
; |   double       |   double      | double |
;    0.00089920883       1.0890088     1.0
;     0.0042646079      -19.498840     1.0
;     0.0050243058       38.859279     1.0
;     0.0086292358      -51.893546     1.0
;     0.0099734379      -40.591202     1.0
;
common StarBase,startable,notes
;
; cntr_u	Counter
; dist_x	Distance in arcseconds from target position
; pang_x	Position angle for offset from target position
; ra_u		Right ascension of target
; dec_u		Declination of target
; major_u	Major axis of target ellipse
; ra		Right ascension of IR point source
; dec		Declination of IR point source
; clon
; clat
; err_maj	Major axis of 1-sigma error ellipse
; err_min	Minor axis of 1-sigma error ellipse
; err_ang	Position angle of error ellipse
; designation	Sexagesimal coordinate designation of target
; j_m 		J band selected "default" magnitude
; j_cmsig	Corrected J band photometric uncertainty
; j_msigcom	Combined (total) J band photometric uncertainty
; j_snr		H band "scan" signal-to-noise ratio
; h_m 		H band selected "default" magnitude
; h_cmsig	Corrected H band photometric uncertainty
; h_msigcom	Combined (total) H band photometric uncertainty
; h_snr		H band "scan" signal-to-noise ratio
; k_m 		K band selected "default" magnitude
; k_cmsig	Corrected K band photometric uncertainty
; k_msigcom	Combined (total) K band photometric uncertainty
; k_snr		K band "scan" signal-to-noise ratio
; ph_qual	Flag indicating photometric quality of source
; rd_flg	Source of JHK "default"mags (AKA "read flag")
; bl_flg	Indicates # JHK components fit simultaneously to source
; cc_flg	Indicates JHK artifact contamination and/or confusion
; ndet		Number of >3-sig. ap. mag measurements, # possible (jjhhkk)
; gal_contam	Flag indicating if src is contaminated by extended source
; mp_flg	Src is positionally associated with an asteroid, comet, etc.
;
index=where(startable.mj ne +100,mj_count1)
;
index=where(startable.hic ne 0,count)
if count gt 0 then begin
;
mj=startable[index].mj
mh=startable[index].mh
mk=startable[index].mk
hic=startable[index].hic
;
xdr_file=file_search(!oyster_dir+'starbase/jhk_hip.xdr') & xdr_file=xdr_file[0]
if strlen(xdr_file) eq 0 then begin
status=dc_read_free(!oyster_dir+'starbase/jhk_hip.hic',n,/col,ignore=['!'])
designation=''
status=dc_read_free(!oyster_dir+'starbase/jhk_hip.dat',cntr_u,dist_x,pang_x, $
	ra_u,dec_u,major_u,ra,dec,clon,clat,err_maj,err_min,err_ang,designation, $
	j_m,j_cmsig,j_msigcom,j_snr, $
	h_m,h_cmsig,h_msigcom,h_snr, $
	k_m,k_cmsig,k_msigcom,k_snr, $
	ignore=['!'],null=' 0  ',/col)
n=n[cntr_u-1]
save,n,h_m,j_m,k_m,filename=!oyster_dir+'starbase/jhk_hip.xdr'
endif else restore,xdr_file
;
for i=0L,count-1 do begin
	j=where(n eq hic[i],j_count)
	if j_count gt 0 then begin
		mj[i]=j_m[j[0]]
		mh[i]=h_m[j[0]]
		mk[i]=k_m[j[0]]
	endif
endfor
startable[index].mj=mj
startable[index].mh=mh
startable[index].mk=mk
;
index=where(startable.mj ne +100,mj_count2)
print,'Number of new JHK values found: ',mj_count2-mj_count1,'.', $
	format='(a,i5,a)'
;
endif else print,'Warning(GET_JHK): No stars with HIC name found!'
;
end
;-------------------------------------------------------------------------------
pro get_jhk_hdn
;
; Read JHK magnitudes from HIP based list of IRSA data
; Version using jhk_hdn.hdn, based on get_hdn.
;
common StarBase,startable,notes
;
; cntr_u	Counter
; dist_x	Distance in arcseconds from target position
; pang_x	Position angle for offset from target position
; ra_u		Right ascension of target
; dec_u		Declination of target
; major_u	Major axis of target ellipse
; ra		Right ascension of IR point source
; dec		Declination of IR point source
; err_maj	Major axis of 1-sigma error ellipse
; err_min	Minor axis of 1-sigma error ellipse
; err_ang	Position angle of error ellipse
; designation	Sexagesimal coordinate designation of target
; j_m 		J band selected "default" magnitude
; j_cmsig	Corrected J band photometric uncertainty
; j_msigcom	Combined (total) J band photometric uncertainty
; j_snr		H band "scan" signal-to-noise ratio
; h_m 		H band selected "default" magnitude
; h_cmsig	Corrected H band photometric uncertainty
; h_msigcom	Combined (total) H band photometric uncertainty
; h_snr		H band "scan" signal-to-noise ratio
; k_m 		K band selected "default" magnitude
; k_cmsig	Corrected K band photometric uncertainty
; k_msigcom	Combined (total) K band photometric uncertainty
; k_snr		K band "scan" signal-to-noise ratio
; ph_qual	Flag indicating photometric quality of source
; rd_flg	Source of JHK "default"mags (AKA "read flag")
; bl_flg	Indicates # JHK components fit simultaneously to source
; cc_flg	Indicates JHK artifact contamination and/or confusion
; ndet		Number of >3-sig. ap. mag measurements, # possible (jjhhkk)
; gal_contam	Flag indicating if src is contaminated by extended source
; mp_flg	Src is positionally associated with an asteroid, comet, etc.
;
index=where(startable.mj ne +100,mj_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
;
mj=startable[index].mj
mh=startable[index].mh
mk=startable[index].mk
hdn=startable[index].hdn
;
xdr_file=file_search(!oyster_dir+'starbase/jhk_hdn.xdr') & xdr_file=xdr_file[0]
if strlen(xdr_file) eq 0 then begin
status=dc_read_free(!oyster_dir+'starbase/jhk_hdn.hdn',n,/col,ignore=['!'])
designation=''
status=dc_read_free(!oyster_dir+'starbase/jhk_hdn.dat',cntr_u,dist_x,pang_x, $
	ra_u,dec_u,major_u,ra,dec,err_maj,err_min,err_ang,designation, $
	j_m,j_cmsig,j_msigcom,j_snr, $
	h_m,h_cmsig,h_msigcom,h_snr, $
	k_m,k_cmsig,k_msigcom,k_snr, $
	ignore=['!'],null=' 0  ',/col)
n=n[cntr_u-1]
save,n,h_m,j_m,k_m,filename=!oyster_dir+'starbase/jhk_hdn.xdr'
endif else restore,xdr_file
;
for i=0L,count-1 do begin
	j=where(n eq hdn[i],j_count)
	if j_count gt 0 then begin
		mj[i]=j_m[j[0]]
		mh[i]=h_m[j[0]]
		mk[i]=k_m[j[0]]
	endif
endfor
startable[index].mj=mj
startable[index].mh=mh
startable[index].mk=mk
;
index=where(startable.mj ne +100,mj_count2)
print,'Number of new JHK values found: ',mj_count2-mj_count1,'.', $
	format='(a,i5,a)'
;
endif else print,'Warning(GET_JHK): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_jhk
;
; Get JHK photometry from 2MASS and JP11
; This procedure replaces get_jhk_CCC.
;
common StarBase,startable,notes
;
starids=startable.starid
rename_starids,'bsc-hdn'
get_jp11,/quiet
get_jhk_hdn	; Overwrites the existing data (i.e. JP11)
startable.starid=starids
;
end
;-------------------------------------------------------------------------------
pro get_n12
;
; Read color-uncorrected IRAS 12 micron fluxes, which for normal stars are
; close to the actual fluxes in the N band.
;
common StarBase,startable,notes
;
files=file_search(!oyster_dir+'starbase/n.bsc') & files=files[0]
if strlen(files) eq 0 then begin
n=''
s=dc_read_fixed(!oyster_dir+'starbase/n.dat',n,/col,ignore=['!'],format='(a120)')
;
index=where(strmid(n,10,10) eq '----------',count)
bsc=intarr(count)
d=fltarr(count)
f12=fltarr(count)
;
dummy='   '
hr=0
r=0.
f=0.
for i=0,count-1 do begin
	reads,n[index[i]-3],dummy,hr,format='(35x,a3,i5)'
	bsc[i]=hr
	reads,n[index[i]-1],r,f,format='(8x,f5.3,62x,e8.2)'
	d[i]=r
	f12[i]=f
endfor
;
; index=where(d lt 0.1,count)
; bsc=bsc(index)
; f12=f12(index)
openw,unit,!oyster_dir+'starbase/n.bsc',/get_lun
printf,unit,'! IRAS 12 micron fluxes in Jansky'
printf,unit,'! Data are retrieved from VizieR-4,'
printf,unit,'! IRAS point sources within 6 arcseconds'
printf,unit,'! or target position.'
for i=0,count-1 do printf,unit,bsc[i],f12[i]
free_lun,unit
endif
;
index=where(startable.f12 ne 0,f12_count1)
;
index=where(startable.bsc ne 0,count)
if count gt 0 then begin
	f12=startable[index].f12
	bsc=startable[index].bsc
	status=dc_read_free(!oyster_dir+'starbase/n.bsc', $
		n,f_12,/col,ignore=['!'])
;
	for i=0,count-1 do begin
		j=where(n eq bsc[i],j_count)
		if j_count gt 0 then begin
			f12[i]=f_12[j[0]]
		endif
	endfor
	startable[index].f12=f12
;
	index=where(startable.f12 ne 0,f12_count2)
	print,'Number of new F12 values found: ',f12_count2-f12_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_F12): No stars with BSC name found!'
;
end
;-------------------------------------------------------------------------------
pro get_f12
;
; Read color-corrected 12 micron fluxes from MIDI consortium list.
;
common StarBase,startable,notes
;
index=where(startable.f12 ne 0,f12_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	f12=startable[index].f12
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/f12.hdn', $
		n,f_12,/col,ignore=['!'])
;
	for i=0l,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			f12[i]=f_12[j[0]]
		endif
	endfor
	startable[index].f12=f12
;
	index=where(startable.f12 ne 0,f12_count2)
	print,'Number of new F12 values found: ',f12_count2-f12_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_F12): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_i12
;
; Read un-corrected 12 micron IRAS fluxes.
;
common StarBase,startable,notes
;
index=where(startable.f12 ne 0,f12_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	f12=startable[index].f12
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/IRAS/iras.hdn', $
		n,f_12,/col,ignore=['!'])
;
	for i=0l,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			f12[i]=f_12[j[0]]
		endif
	endfor
	startable[index].f12=f12
;
	index=where(startable.f12 ne 0,f12_count2)
	print,'Number of new F12 values found: ',f12_count2-f12_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_I12): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_jp11,quiet=quiet
;
; Read Johnson (1966) UBVRIJKLMN from Vizier in catalogs/npoi/jp11.hdn. 
; If file does not exist, execute helper function to read jp11.dat.
; Note: data found will replace current values!
;
common StarBase,startable,notes
;
files=file_search(!oyster_dir+'starbase/jp11.hdn') & files=files[0]
if strlen(files) eq 0 then begin
	status=dc_read_free(!oyster_dir+'starbase/jp11.dat',/col,ignore=['!'], $
		rah,ram,ras,dd,dm,ds,lid,mu,mb,mv,mr,mi,mj,mk,ml,mm,mn,mh)
	ra=float(rah)+float(ram)/60+float(ras)/3600
	dec=signof(dd)*(float(abs(dd))+float(dm)/60+float(ds)/3600)
	index=where(ra lt 24 and mj lt 99 and mh lt 99 and mk lt 99,count)
	hdn=lonarr(count)
	get_sky
	jp11_ra=startable.ra
	jp11_dec=startable.dec
	for i=0L,count-1 do begin
		jp11_ra[*]=ra[index[i]]
		jp11_dec[*]=dec[index[i]]
		w=winkel(jp11_ra,jp11_dec,startable.ra,startable.dec)
		j=where(w eq min(w))
		if w[j] lt 2.0/(3600) then hdn[i]=startable[j].hdn
	endfor
	openw,unit,'jp11.hdn',/get_lun
	for i=0L,count-1 do if hdn[i] gt 0 then $
		printf,unit,hdn[i],mj[index[i]],mh[index[i]],mk[index[i]]
	free_lun,unit
endif
;
index=where(startable.mj ne +100,mj_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	mj=startable[index].mj
	mh=startable[index].mh
	mk=startable[index].mk
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/jp11.hdn', $
		n,m_j,m_h,m_k,/col,ignore=['!'])
;
	for i=0L,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			mj[i]=m_j[j[0]]
			mh[i]=m_h[j[0]]
			mk[i]=m_k[j[0]]
		endif
	endfor
	startable[index].mj=mj
	startable[index].mh=mh
	startable[index].mk=mk
;
	index=where(startable.mj ne +100,mj_count2)
	if not keyword_set(quiet) then $
	print,'Number of new JHK values found: ',mj_count2-mj_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_JP11): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_ubvri
;
; Read Nagy & Hill (1980) ubvri data in catalogs/npoi/ubvri.hdn. 
; Note: data found will replace current values!
;
common StarBase,startable,notes
;
index=where(startable.ri ne +100,ri_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	bv=startable[index].bv
	ub=startable[index].ub
	ri=startable[index].ri
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/ubvri.hdn', $
		n,b_v,u_b,r_i,/col,ignore=['!'])
;
	for i=0,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			bv[i]=b_v[j[0]]
			ub[i]=u_b[j[0]]
			ri[i]=r_i[j[0]]
		endif
	endfor
	startable[index].bv=bv
	startable[index].ub=ub
	startable[index].ri=ri
;
	index=where(startable.ri ne +100,ri_count2)
	print,'Number of new (R-I) values found: ',ri_count2-ri_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_UBVRI): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_ubv
;
; Read Mermilliod (1980) ubv data in catalogs/npoi/ubv.hdn. 
; Note: data found will replace current values!
;
common StarBase,startable,notes
;
index=where(startable.ub ne +100,ub_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	mv=startable[index].mv
	bv=startable[index].bv
	ub=startable[index].ub
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/ubv.hdn', $
		n,m_v,b_v,u_b,/col,ignore=['!'])
;
	for i=0,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			mv[i]=m_v[j[0]]
			bv[i]=b_v[j[0]]
			ub[i]=u_b[j[0]]
		endif
	endfor
	startable[index].mv=mv
	startable[index].bv=bv
	startable[index].ub=ub
;
	index=where(startable.ub ne +100,ub_count2)
	print,'Number of new (U-B) values found: ',ub_count2-ub_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_UBV): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_uvbybeta
;
; Read Mermilliod (1980) ubv data in catalogs/npoi/uvbybeta.hdn. 
; Note: data will replace current non-zero values!
;
common StarBase,startable,notes
;
index=where(startable.by ne +100,by_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	by=startable[index].by
	m1=startable[index].m1
	c1=startable[index].c1
	be=startable[index].beta
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/uvbybeta.hdn', $
		n,b_y,m_1,c_1,bet,/col,ignore=['!'])
;
	for i=0L,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			by[i]=b_y[j[0]]
			m1[i]=m_1[j[0]]
			c1[i]=c_1[j[0]]
			be[i]=bet[j[0]]
		endif
	endfor
	startable[index].by=by
	startable[index].m1=m1
	startable[index].c1=c1
	startable[index].beta=be
;
	index=where(startable.by ne +100,by_count2)
	print,'Number of new (b-y) values found: ',by_count2-by_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_UVBYBETA): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_feh
;
; Read Cayrel de Strobel (1985) [Fe/H] data in 
; catalogs/npoi/feh.hdn. 
; Note: data will replace current non-zero values!
;
common StarBase,startable,notes
;
index=where(startable.feh ne +100,feh_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	feh=startable[index].feh
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/feh.hdn', $
		n,fe_h,/col,ignore=['!'])
;
	for i=0L,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			feh[i]=fe_h[j[0]]
		endif
	endfor
	startable[index].feh=feh
;
	index=where(startable.feh ne +100,feh_count2)
	print,'Number of new [Fe/H] values found: ',feh_count2-feh_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_FEH): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_logg
;
; Read Cayrel de Strobel (1985) gravity data in 
; catalogs/npoi/logg.hdn.
; Note: data will replace current non-zero values!
;
common StarBase,startable,notes
;
index=where(startable.logg1 ne 0,logg_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	logg1=startable[index].logg1
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/logg.hdn', $
		n,logg,/col,ignore=['!'])
;
	for i=0L,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			logg1[i]=logg[j[0]]
		endif
	endfor
	startable[index].logg1=logg1
;
	index=where(startable.logg1 ne 0,logg_count2)
	print,'Number of new gravity values found: ',logg_count2-logg_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_LOGG): No stars with HDN name found!'
;
end
;-------------------------------------------------------------------------------
pro get_teff
;
; Read Cayrel de Strobel (1985) effective temperature data in 
; catalogs/npoi/teff.hdn.
; Note: data will replace current non-zero values!
;
common StarBase,startable,notes
;
index=where(startable.teff ne 0,teff_count1)
;
index=where(startable.hdn ne 0,count)
if count gt 0 then begin
	teff1=startable[index].teff
	hdn=startable[index].hdn
	status=dc_read_free(!oyster_dir+'starbase/teff.hdn', $
		n,teff,/col,ignore=['!'])
;
	for i=0L,count-1 do begin
		j=where(n eq hdn[i],j_count)
		if j_count gt 0 then begin
			teff1[i]=teff[j[0]]
		endif
	endfor
	startable[index].teff=teff1
;
	index=where(startable.teff ne 0,teff_count2)
	print,'Number of new temperature values found: ',teff_count2-teff_count1,'.', $
		format='(a,i4,a)'
;
endif else print,'Warning(GET_TEFF): No stars with HDN name found!'
;
end
;************************************************************************Block 4
pro get_tycho,init=init
;
; Read VT and BT from the Tycho catalog and compute V and (B-V).
; Update the startable, if it exists, or create new one.
; The StarIds are Hipparcos IDs!
;
common StarBase,startable,notes
common get_tycho_local,tycho_data
;
if n_elements(init) eq 0 then init=0
if n_elements(startable) eq 0 then init=1
;
if n_elements(startable) ne 0 then startable_inp=startable
;
; Read Tycho data (some of it...)
tycho_csv=!oyster_dir+'catalogs/tycho/tycho2.csv'
;
tycho_data=read_csv(tycho_csv)
hip=long(tycho_data.field24)
star_ids='HIP'+string(hip,format='(i6.6)')
create_startable,star_ids,/quiet
startable.hic=hip
;
comps=tycho_data.field24
index=where(isnumeric(comps) eq 0,count)
for i=0l,count-1 do begin
	len1=strlen(comps[index[i]])
	len2=strlen(strtrim(string(long(comps[index[i]])),1))
	startable[index[i]].name=strmid(comps[index[i]],len2,len1-len2)
endfor
;
startable.ra=tycho_data.field03/15
startable.dec=tycho_data.field04
startable.pmra=tycho_data.field05*(100./15d3)
startable.pmdec=tycho_data.field06*(100./1d3)
startable.pmrae=tycho_data.field08*(100./15d3)
startable.pmdece=tycho_data.field06*(100./1d3)
btmag=tycho_data.field18
ebtmag=tycho_data.field19
vtmag=tycho_data.field20
evtmag=tycho_data.field21
; http://www.aerith.net/astro/color_conversion.html
; V = VT - 0.090 * (BT - VT)
; B - V = 0.850 * (BT - VT) 
startable.mv=vtmag-0.090*(btmag-vtmag)
startable.bv=0.850*(btmag-vtmag)
;
if init then return
;
index=whereequal(startable_inp.hic,startable.hic)
for i=0l,n_elements(index)-1 do begin
	j=where(startable.hic eq startable_inp[index[i]].hic) & j=j[0]
	startable_inp[index[i]].mv=startable[j].mv
	startable_inp[index[i]].bv=startable[j].bv
endfor
;
; Try to find additional matches with the Tycho catalog if HIP ID is missing
;
; Adopted positional precision of catalog
min_sep=dblarr(n_elements(startable_inp))
precision=5.0	; ["]
;
for i=0l,n_elements(startable_inp)-1 do begin
	sep=winkel(startable.ra,startable.dec, $
		   startable_inp[i].ra,startable_inp[i].dec)*3600
	min_sep[i]=min(sep)
	if min_sep[i] le precision then begin
		j=where(sep eq min_sep[i]) & j=j[0]
		startable_inp[i].mv=startable[j].mv
		startable_inp[i].bv=startable[j].bv
	endif
endfor
; The histogramm shows that the precision is about 4"
histograph,min_sep,binsize=0.5,min=0,max=10
;
end
;-------------------------------------------------------------------------------
pro get_mdfc,init=init
;
; Read L-band and 12 micron fluxes from the MDFC catalog 
; (Cruzalebes et al. 2019)
;
common StarBase,startable,notes
common get_mdfc_local,mdfc_data
;
if n_elements(init) eq 0 then init=0
;
; Positional precision ["] of OYSTER HDN catalog
precision=1.0
;
; Find latest version and read it
if n_elements(mdfc_data) eq 0 then begin
; 	Get name of latest FITS file for MDFC
	mdfc_fits=file_search(!oyster_dir+'starbase/MDFC/mdfc-v??.fits')
	if strlen(mdfc_fits[0]) eq 0 then begin
		print,'Error(GET_MDFC): FITS file not found!'
		return
	endif
; 	Use latest version
	mdfc_fits=mdfc_fits[n_elements(mdfc_fits)-1]
;
	mdfc_data=mrdfits(mdfc_fits,1,header,/silent)
endif
;
; Read entire HDN catalog if no startable exists or init=1
if n_elements(startable) eq 0 or init then get_hdn
;
; Read entire HIP catalog if no startable exists or init=1
; if n_elements(startable) eq 0 or init then get_hip
;
ra=hms2h(mdfc_data.raj2000)
dec=dms2d(mdfc_data.dej2000)
;
; Try to find each star of startable in catalog by coordinates
for i=0L,n_elements(startable)-1 do begin
	angles=winkel(startable[i].ra,startable[i].dec,ra,dec)
	j=where(angles eq min(angles))
	if angles[j]*3600 lt precision then begin
		startable[i].f12=mdfc_data[j].med_nflux
		startable[i].flp=mdfc_data[j].med_lflux
	endif
endfor
;
; The histogramm shows that the precision is about 1" (not checked yet)
; histograph,min_sep,binsize=0.5,min=0,max=10
;
end
;-------------------------------------------------------------------------------
pro get_wise,k_max=k_max
;
; Read Band 1 (~=flp) and 4 (~=f12) magnitudes from WISE catalog (xdr format). 
; If file does not exist, read wise.dat and re-create xdr file.
; Note: data found will replace current values in StarTable!
;
; For applications which limit K, specifiying the limit here is faster.
;
; Superceded by following procedure!
;
common StarBase,startable,notes
;
if n_elements(startable) eq 0 then begin
	print,'Error: please create StarTable for your target first!'
	return
endif
;
file=!oyster_dir+'starbase/WISE/wise.xdr'
files=file_search(file) & file=files[0]
if strlen(files[0]) eq 0 then begin
	file=!oyster_dir+'starbase/WISE/wise.dat'
	status=dc_read_free(file,/col,ignore=['#'], $
		ra_deg,dec_deg,w1,w1e,w2,w2e,w3,w3e,w4,w4e,jmag,hmag,kmag)
	save,   ra_deg,dec_deg,w1,w1e,w2,w2e,w3,w3e,w4,w4e,jmag,hmag,kmag, $
		file=file
endif else restore,files[0]
;
if n_elements(k_max) eq 0 then k_max=max(kmag)
index=where(kmag le k_max,count)
if count eq 0 then return 
if count lt n_elements(ra) then begin
	ra_deg=ra_deg[index]
	dec_deg=dec_deg[index]
	w1=w1[index]
	w3=w3[index]
	jmag=jmag[index]
	hmag=hmag[index]
	kmag=kmag[index]
endif
ra_h=ra_deg/15	; -> RA in hours
dec_d=dec_deg
;
; Adopted positional precision of catalog
min_sep=dblarr(n_elements(startable))
precision=5.0		; ["]
precision=2000.0	; ["] for testing
;
index=where(startable.f12 gt 0,f12_count1)
;
for i=0,n_elements(startable)-1 do begin
	sep=winkel(ra_h,dec_d,startable[i].ra,startable[i].dec)*3600 ; arc-sec
;	Sort by separation
	si=sort(sep)
	sep=sep[si]
	ra_h=ra_h[si]
	dec_d=dec_d[si]
	min_sep[i]=min(sep)
	if min_sep[i] le precision then begin
		j=where(sep eq min_sep[i])
		startable[i].flp=w12jy(w1[j])
		startable[i].f12=w32jy(w3[j])
		startable[i].mj=jmag[j]
		startable[i].mh=hmag[j]
		startable[i].mk=kmag[j]
	endif
endfor
; The histogramm shows that the precision is about 4" (tbc)
; histograph,min_sep,binsize=0.5,min=0,max=10
;
index=where(startable.f12 gt 0,f12_count2)
print,'Number of new WISE values found: ',f12_count2-f12_count1,'.', $
	format='(a,i5,a)'
;
end
;-------------------------------------------------------------------------------
pro get_wise,k_max=ks_max
;
; Read W1 mag. (~=flp) from CatWise and W3 mag. (~=f12) from AllWISE.
; Note: data found will replace current values in StarTable!
;
; For applications which limit K, specifiying the limit here is faster.
; At this time, we equate K and Ks with one another.
;
; Example
; ra='07:32:09.8'
; dec='-16:58:12.654'
; get_startable,ra=ra,dec=dec
;
common StarBase,startable,notes
;
if n_elements(startable) eq 0 then begin
	print,'Error: please create StarTable for your target first!'
	return
endif
;
; Load the first survey data
file=!oyster_dir+'starbase/WISE/allwise/allwise.sam.xdr'
files=file_search(file) & file=files[0]
if strlen(files[0]) eq 0 then begin
	file=!oyster_dir+'starbase/WISE/allwise/allwise.sam'
	status=dc_read_fixed(file,rec,/col,ignore=['#'], $
		resize=[1],format='(a553)')
	allwise=strmid(rec,0,19)	; obj ID
	raj2000=double(strmid(rec,21,10))	; degrees
	dej2000=double(strmid(rec,33,10))	; degrees
	w1mag=float(strmid(rec,67,5))		; W1 magnitude (3.35 micron)
	w2mag=float(strmid(rec,74,5))		; W2 magnitude (4.6  micron)
	w3mag=float(strmid(rec,81,5))		; W3 magnitude (11.6 micron) *
	w4mag=float(strmid(rec,88,5))		; W4 magnitude (22.1 micron)
	jmag=fltarr(n_elements(rec))
	index=where(strlen(strcompress(strmid(rec,95,5),/remove_all)) gt 0)
	jmag[index]=float(strmid(rec[index],95,5))
	hmag=fltarr(n_elements(rec))
	index=where(strlen(strcompress(strmid(rec,102,5),/remove_all)) gt 0)
	hmag[index]=float(strmid(rec[index],102,5))
	ksmag=fltarr(n_elements(rec))
	index=where(strlen(strcompress(strmid(rec,109,5),/remove_all)) gt 0)
	ksmag[index]=float(strmid(rec[index],109,5))
	save,allwise,raj2000,dej2000,w1mag,w2mag,w3mag,w4mag, $
		jmag,hmag,ksmag,file=file+'.xdr'
endif else restore,files[0]
;
; Apply K-band limit, if specified
if n_elements(ks_max) eq 0 then ks_max=max(ksmag)
index=where(ksmag le ks_max,count)
if count eq 0 then begin
	print,'No targets found brighter than K-limit!'
	return 
endif
if count lt n_elements(ra) then begin
	raj2000=raj2000[index]
	dej2000=dej2000[index]
	w1mag=w1mag[index]
	w3mag=w3mag[index]
	jmag=jmag[index]
	hmag=hmag[index]
	ksmag=ksmag[index]
endif
;
; Update data in the StarTable with those found in AllWise
;
precision=5.0		; ["]
dh=360./24.
;
for i=0l,n_elements(startable)-1 do begin
	sep=winkel(raj2000*dh,dej2000,startable[i].ra,startable[i].dec)*3600
	index=where(sep eq min(sep))
	if sep[index] le precision then begin
		startable[i].flp=w12jy(w1[j])	; convert W1 mag to Jy
		startable[i].f12=w32jy(w3[j])	; convert W3 mag to Jy
		startable[i].mj=jmag[j]
		startable[i].mh=hmag[j]
		startable[i].mk=kmag[j]
	endif	
endfor
;
; Load the data from the second survey
;
file=!oyster_dir+'starbase/WISE/catwise/catwise.sam.xdr'
files=file_search(file) & file=files[0]
if strlen(files[0]) eq 0 then begin
	file=!oyster_dir+'starbase/WISE/catwise/catwise.sam'
	status=dc_read_fixed(file,rec,/col,ignore=['#'], $
		resize=[1],format='(a571)')
	catwise=strmid(rec,0,18)	; obj ID
	raj2000=double(strmid(rec,20,10))
	dej2000=double(strmid(rec,32,10))
	w1mag=float(strmid(rec,318,6))		; W1 magnitude (3.35 micron) *
	w2mag=float(strmid(rec,341,6))		; W2 magnitude (4.6  micron)
	save,catwise,raj2000,dej2000,w1mag,w2mag,file=file+'.xdr'
endif else restore,files[0]
;
; Update the data of allwise with those of catwise (W1 only, W3 if not av.)
;
precision=5.0		; ["]
;
for i=0l,n_elements(startable)-1 do begin
	sep=winkel(raj2000*dh,dej2000,startable[i].ra,startable[i].dec)*3600
	index=where(sep eq min(sep))
	if sep[index] le precision then begin
		startable[i].flp=w12jy(w1mag[j])	; convert W1 mag to Jy
		if startable[i].f12 eq 0 then $
		startable[i].f12=w32jy(w3mag[j])	; convert W3 mag to Jy
	endif	
endfor
;
ra_h=ra_deg/15	; -> RA in hours
dec_d=dec_deg
;
; Adopted a positional precision of the catalog coordinates
precision=5.0		; ["]
precision=2000.0	; ["] for testing
min_sep=dblarr(n_elements(startable))
;
index=where(startable.f12 gt 0,f12_count1)
;
for i=0,n_elements(startable)-1 do begin
	sep=winkel(ra_h,dec_d,startable[i].ra,startable[i].dec)*3600 ; arc-sec
;	Sort by separation
	si=sort(sep)
	sep=sep[si]
	ra_h=ra_h[si]
	dec_d=dec_d[si]
	min_sep[i]=min(sep)
	if min_sep[i] le precision then begin
		j=where(sep eq min_sep[i])
		startable[i].flp=w12jy(w1[j])	; convert W1 mag to Jy
		startable[i].f12=w32jy(w3[j])	; convert W3 mag to Jy
		startable[i].mj=jmag[j]
		startable[i].mh=hmag[j]
		startable[i].mk=kmag[j]
	endif
endfor
; The histogramm shows that the precision is about 4" (tbc)
; histograph,min_sep,binsize=0.5,min=0,max=10
;
index=where(startable.f12 gt 0,f12_count2)
print,'Number of new WISE values found: ',f12_count2-f12_count1,'.', $
	format='(a,i5,a)'
;
end
;-------------------------------------------------------------------------------
pro get_2mass,k_max=k_max
;
; Read JHK magnitudes from 2MASS catalog (xdr format). 
; If file does not exist, read 2mass_K6.fits and re-create xdr file.
; Note: data found will replace current values!
;
; For applications which limit K, specifiying the limit here is faster.
;
common StarBase,startable,notes
;
file=!oyster_dir+'starbase/2MASS/hip_2mass_K8.xdr'
files=file_search(file)
if strlen(files[0]) eq 0 then begin
	d=mrdfits(!oyster_dir+'starbase/2MASS/2mass_K6.fits',1,h)
	ra_deg=d.raj2000
	dec_deg=d.dej2000
	jmag=d.jmag
	hmag=d.hmag
	kmag=d.kmag
	save,ra_deg,dec_deg,jmag,hmag,kmag,file=file
endif else restore,file
ra_h=ra_deg/15
;
if n_elements(k_max) eq 0 then k_max=max(kmag)
index=where(kmag le k_max,count)
if count eq 0 then return 
if count lt n_elements(ra_h) then begin
	ra_h=ra_h[index]
	dec_deg=dec_deg[index]
	jmag=jmag[index]
	hmag=hmag[index]
	kmag=kmag[index]
endif
;
; Adopted positional precision of catalog
min_sep=dblarr(n_elements(startable))
precision=1.0	; ["]
;
index=where(startable.mk gt 0,kmag_count1)
;
for i=0,n_elements(startable)-1 do begin
	sep=winkel(ra_h,dec_deg,startable[i].ra,startable[i].dec)*3600
	min_sep[i]=min(sep)
	if min_sep[i] le precision then begin
		j=where(sep eq min_sep[i])
		startable[i].mj=jmag[j]
		startable[i].mh=hmag[j]
		startable[i].mk=kmag[j]
	endif
endfor
; The histogramm shows that the precision is about 1"
; histograph,min_sep,binsize=0.1,min=0,max=2
;
index=where(startable.mk gt 0,kmag_count2)
if kmag_count2 gt kmag_count1 then $
print,'Number of new 2MASS values found: ',kmag_count2-kmag_count1,'.', $
	format='(a,i7,a)'
;
; save,startable,file=!oyster_dir+'starbase/2MASS/hip_2mass.xdr'
;
;
end
;-------------------------------------------------------------------------------
pro get_iras
;
; Given an IRAS ID, look up the flux in a table extracted from the IRAS catalog
; and update the startable with all 12 micron fluxes found.
;
; The IRAS ID (iras_cat_id) has the form 12345-1234 or 12345+1234
; In StarTable, the prefix is 'IRAS'
;
common StarBase,startable,notes
;
; Look for IRAS XDR file, and if not found, create it for faster reading
file=!oyster_dir+'starbase/IRAS/iras.xdr'
files=file_search(file)
if strlen(files[0]) eq 0 then begin
;	Initialize variable type
	iras_cat_id=''
	f12=0.
	result=dc_read_free( $
		!oyster_dir+'starbase/IRAS/iras.iras',/col, $
		iras_cat_id,f12,ignore=['!'])
	save,iras_cat_id,f12,file=file
endif else restore,file
;
iras_id=irasid(startable.ra,startable.dec)
;
; Setup for counting new values found
index=where(startable.f12 gt 0,f12_count1)
;
for i=0,n_elements(startable)-1 do begin
	j=where(iras_cat_id eq iras_id[i] $
	       or iras_cat_id eq startable[i].name,count)
	if count eq 1 then startable[i].f12=f12[j]
endfor	
;
index=where(startable.f12 gt 0,f12_count2)
print,'Number of new F12 values found: ',f12_count2-f12_count1,'.', $
	format='(a,i7,a)'
;
end
;-------------------------------------------------------------------------------
