;*******************************************************************************
; File: model.pro
;
; Description:
; ------------
; This file contains various functions for the AMOEBA software.
; Please note that all Julian Day arguments (epochs) are JD-2440000,
; only exceptions are functions called at user level (e.g. binarypos).
;
; Block directory:
; ----------------
; Block 1: num_binary,num_star,
;          componentparse,modelparse,wmcc,componentmass,
;	   topbincomp,topcomp,systemcomp,checkcomponents,checkmodel,
;
; Block 2  true2app,true2vel,vel2gamma,
;	   mag2flux,modelfluxes,stellarfluxes,
;	   modelpx,modelm,modelk,modelf,modeldm,wilson_wd,wilson
;	   setwdparms,setlcparms,setrocheparms,
;	   wdmap,wdrav,lcmap,imgmap,edstrip,gdmap,chmap,
;	   modelpos,modelvel,binarypos,binaryvel
;	   apodize_f,apodize_m,
;	   stripvis,mapvis,imgvis,provis,rochevis,
;          componentvis,modelvis,componentflux,componentfluxes,modelflux,
;	   modelchisq,cleanup_componentvis
;
;************************************************************************Block 1
function num_binary
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return,0
if n_elements(binary_model) eq 0 then return,0
index=where(strlen(binary_model.component) gt 0,num)
return,num
;
end
;-------------------------------------------------------------------------------
function num_star
;
; Counts the number of stars in the model. If a resolved flux component 
; is defined in the model, it needs to be the last component defined!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return,0
index=where(strlen(star_model.component) gt 0 and star_model.type gt 0,num)
return,num
;
end
;-------------------------------------------------------------------------------
function componentparse,component,comp_1,comp_2
;
; For a given component, extract the two subcomponents.
; Example: A-BC -> A,BC.
; Return status: -1: error, 0: normal completion.
;
len=strlen(component)
pos=strpos(component,'-')
comp_1=strmid(component,0,pos)
comp_2=strmid(component,pos+1,len-pos-1)
if strpos(comp_2,'-') ne -1 then begin
	print,'***Error(COMPONENTPARSE): invalid component name!'
	return,-1
endif
return,0
;
end
;-------------------------------------------------------------------------------
function modelparse,component,comp_1,comp_2
;
; For a given component, extract the two subcomponents and replace with
; stars and/or corresponding binary components as defined in model.
; Example: A-BC -> A,B-C.
; Return status: -1: error, 0: normal completion.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if componentparse(component,comp_1,comp_2) ne 0 then return,-1
;
for i=0,num_binary()-1 do begin
	if componentparse(binary_model[i].component,c1,c2) ne 0 then return,-1
	if c1+c2 eq comp_1 then comp_1=binary_model[i].component
	if c1+c2 eq comp_2 then comp_2=binary_model[i].component
endfor
;
flag=0
if strlen(comp_1) gt 1 and strpos(comp_1,'-') eq -1 then flag=1
if strlen(comp_2) gt 1 and strpos(comp_2,'-') eq -1 then flag=2
comp_12=[comp_1,comp_2]
if flag ne 0 then begin
	comp_12=comp_12[flag-1]
	print,'***Error(MODELPARSE): missing binary component for '+comp_12+'!'
	return,-1
endif
;
return,0
;
end
;-------------------------------------------------------------------------------
function wmcc,component
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return,component
comp1=''
comp2=''
if strpos(component,'-') ge 0 then begin
	if componentparse(component,comp_1,comp_2) ne 0 then return,component
endif else begin
	comp_1=component
	comp_2=''
	comp2=' '
endelse
;
for i=0,strlen(comp_1)-1 do begin
	index=where(star_model.component eq strmid(comp_1,i,1),count)
	if count gt 0 then comp1=comp1+star_model[index].wmc
endfor
chars=unique(byte(comp1))
if n_elements(chars) lt strlen(comp1) then comp1=string(chars[0])
;
for i=0,strlen(comp_2)-1 do begin
	index=where(star_model.component eq strmid(comp_2,i,1),count)
	if count gt 0 then comp2=comp2+star_model[index].wmc
endfor
chars=unique(byte(comp2))
if n_elements(chars) lt strlen(comp2) then comp2=string(chars[0])
;
if strlen(comp_2) eq 0 then wmccd=comp1 else wmccd=comp1+'-'+comp2
if strlen(comp1) eq 0 or strlen(comp2) eq 0 then wmccd=component
;
return,wmccd
;
end
;-------------------------------------------------------------------------------
function componentmass,comp_1,comp_2,num_body,index_comp
;
; Compute the component masses (=sum of stellar masses)
; Components specified as returned by componentparse, e.g. 'AB', 'C'
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
components=[comp_1,comp_2]
num_body=[strlen(comp_1),strlen(comp_2)]
index_comp=intarr(max(num_body),2)
mass_sum=dblarr(2)
for k=0,1 do begin
	for j=0,num_body[k]-1 do begin
		index=where(strmid(components[k],j,1) eq star_model.component)
		mass_sum[k]=mass_sum[k]+star_model[index[0]].mass
		index_comp[j,k]=index[0]
	endfor
endfor
;
return,mass_sum
;
end
;-------------------------------------------------------------------------------
function topbincomp
;
; Return the highest ranking binary component, i.e. the component which
; contains all stars comprising the system.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
levels=strlen(binary_model.component)
index=where(max(levels) eq levels)
return,binary_model[index[0]].component
;
end
;-------------------------------------------------------------------------------
function topcomp
;
; Return the highest ranking model component, binary or single (1st).
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if strlen(topbincomp()) ne 0 then return,topbincomp()
;
return,star_model[0].component
;
end
;-------------------------------------------------------------------------------
function systemcomp
;
; Return the sum of all stellar components of a system.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
c2=''
if strlen(topbincomp()) ne 0 then begin
	if componentparse(topbincomp(),c1,c2) ne 0 then return,-1
endif else c1=topcomp()
;
return,'System '+c1+c2
;
end
;-------------------------------------------------------------------------------
function checkcomponents,component
;
; This stub of function CHECKCOMPONENTS is included here so that PV-WAVE, by 
; way of compiling it, stores the function address, which is apparently needed 
; when compiling the real thing (see the following function). Otherwise, PV-WAVE
; would complain about an unknown function CHECKCOMPONENTS when trying to 
; execute it. This does not happen, however, with recursive procedures.
;
end
;-------------------------------------------------------------------------------
function checkcomponents,component
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if strlen(component) gt 1 then begin
	index=where(binary_model.component eq component,count)
	if count eq 0 then begin
	   print,'***(CHECKCOMPONENTS): undefined component! ',component
	   return,-1
	endif
	if count gt 1 then begin
	   print,'***Error(CHECKCOMPONENTS): component defined more than once!'
	   return,-1
	endif
endif
;
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
components=[comp_1,comp_2]
for i=0,1 do begin
	if strlen(components[i]) eq 1 then begin
		index=where(star_model.component eq components[i],count)
		if count ne 1 then begin
			print,'***Error(CHECKCOMPONENTS): '+ $
				'component undefined or non-unique!'
			return,-1
		endif
	endif else if checkcomponents(components[i]) ne 0 then return,-1
endfor
;
return,0
;
end
;-------------------------------------------------------------------------------
function checkmodel
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then begin
	print,'***Error(CHECKMODEL): gen_model undefined!'
	return,-1
endif
if total(strlen(star_model.component)) eq 0 then begin
	print,'***Error(CHECKMODEL): star_model undefined!'
	return,-1
endif
if total(strlen(star_model.component)) ne n_elements(star_model) then begin
	print,'***Error(CHECKMODEL): some star components composit!'
	return,-1
endif
;
; Make sure a resolved flux component is always the last "star" component
; otherwise the model checking and fitting routines get "confused"!
index=where(star_model.type eq 0,count)
if count gt 0 and index[0] lt n_elements(star_model)-1 then begin
	print,'***Error(CHECKMODEL): resolved flux component must be last one!'
	return,-1
endif
;
if num_binary() ge 1 then if checkcomponents(topbincomp()) eq -1 then return,-1
;
return,0
;
end
;************************************************************************Block 2
function true2app,mjd,o_parms,rho,theta,ra,dec,z
;
; Compute and return right ascension and declination offset of secondary
; relative to primary  for given epoch (mod. JD!) and orbital elements.
; Originally written by J. Armstrong, based on W. Heintz, Double Stars, p. 34
;
; Note: The RV of whichever component is in the vicinity of the ascending
; node is positive (away from observer) with respect to that of the other 
; component, or wrt its own RV in other parts of the orbit.
;
; Note: if the inclination is < 90, the orbit is prograde, i.e. the 
; PA increases. If i>90, the orbit is retrograde, i.e. PA decreases.
;
; Note: the periastron angle is the one of the primary. It is counted from
; the ascending node in the direction of motion of that component.
; It is changed from the primary to the secondary by adding 180 deg as the
; formulae from Heintz and Green are all assuming the secondary elements.
;
; The following statement was true until Version 6.05:
; Currently, the angle of the ascending node is 180 degrees off.
; Therefore, currently the primary is moving away from the
; observer if the secondary is at the "ascending" node.
;
; Position angle verifications: (this paragraph started in Nov 2008)
; Gamma Per: A-comp. rel. to G-comp. PA=244 deg (most of the time)
; This is according to Pourbaix 1999, and Prieur et al. 2002 (Speckle)
; and Couteau (1987, but he mentions that it is in the 4th quadrant?).
; However, the McAlister measurements give the PA as 60 deg, why?
; Mason et al. 2001 even lists PA=64 deg! Zavala confirmed 244 deg is
; consistent with NPOI observations reduced with OYSTER.
;
; Another check star is 1 Geminorum, which was measured by Hipparcos
; at rho=185 mas, theta=227 deg.
;
; Theta is returned in [rad].
;
; Hint: if you happen to get a negative eccentricity, add 180 to the
; periastron angle and half a period to the epoch to change the sign.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
a=o_parms[0]
e=o_parms[1]
i=o_parms[2]
w=o_parms[3]
s=o_parms[4]
if n_elements(dec) ne 0 $
	then n=precess_pa(o_parms[5],jd2jy(mjd+2440000L),ra,dec,1) $
	else n=o_parms[5]
w=(w+!pi) mod (2*!pi)	; change from primary to secondary definition
;
p=o_parms[6]
t=o_parms[7]
;
M_anom=2*pi_circle*((mjd-t) mod p)/p
E_anom=M_anom+e*sin(M_anom)+e^2/2*sin(2*M_anom)
for k=0,4 do E_anom=E_anom+(M_anom-E_anom+e*sin(E_anom))/(1-e*cos(E_anom))
T_anom=2*atan(sqrt((1+e)/(1-e))*sin(E_anom/2),cos(E_anom/2))
;
; Apply apsidal motion
ws=w+(mjd-t)/s_year*s
;
alpha=T_anom+ws
;
; From R.M. Green, Spherical Astronomy, p. 470
r=((a*(1-e^2))/(1+e*cos(T_anom)))
x=r*(cos(alpha)*sin(n)+sin(alpha)*cos(n)*cos(i))
y=r*(cos(alpha)*cos(n)-sin(alpha)*sin(n)*cos(i))
z=r*sin(alpha)*sin(i)
;
; From Heintz (results in same [x,y] values)
theta=atan(sin(alpha)*cos(i),cos(alpha))+n
rho=((a*(1-e^2))/(1+e*cos(T_anom)))*sqrt(cos(alpha)^2+sin(alpha)^2*cos(i)^2)
index=where(rho lt 0,count)
if count gt 0 then begin
	theta[index]=theta[index]+pi_circle
	rho[index]=abs(rho[index])
endif
index=where(theta lt 0,count)
if count gt 0 then theta[index]=theta[index]+2*pi_circle
;
; return,[[rho*sin(theta)],[rho*cos(theta)]]
return,[[x],[y]]
;
end
;-------------------------------------------------------------------------------
function true2vel,jd,o_parms
;
; Note that if radial velocities are available for the primary only,
; one should fit M2 only.
;
; Argument of periastron (w) taken to be of the primary component.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
e=o_parms[0]
i=o_parms[1]
w=o_parms[2]
s=o_parms[3]	; apsidal motion
p=o_parms[4]	; period
t=o_parms[5]	; epoch
m1=o_parms[6]
m2=o_parms[7]
;
M_anom=2*pi_circle*((jd-t) mod p)/p
E_anom=M_anom+e*sin(M_anom)+e^2/2*sin(2*M_anom)
for k=0,4 do E_anom=E_anom+(M_anom-E_anom+e*sin(E_anom))/(1-e*cos(E_anom))
T_anom=2*atan(sqrt((1+e)/(1-e))*sin(E_anom/2),cos(E_anom/2))
;
; K1, K2 [km/s]
p=p/s_year
k1=2*pi_circle*sin(i)/sqrt(1-e^2)*p^(-1.0/3.0)*m2*(m1+m2)^(-2.0/3.0) $
	*(a_unit/s_year)/86400000L
k2=k1*(m1/m2)
;
; Apply apsidal motion
ws=w+(jd-t)/s_year*s
v1=+k1*(cos(T_anom+ws)+e*cos(ws))
v2=-k2*(cos(T_anom+ws)+e*cos(ws))
return,[[v1],[v2]]
;
end
;-------------------------------------------------------------------------------
function vel2gamma,velocities,mass_a,mass_b
;
; Given velocities (ameeba structure!), primary and secondary mass, compute 
; gamma velocities for matching pairs of measurement.
;
ujds=unique(velocities.jd)
n_ujds=n_elements(ujds)
v_gamma=fltarr(n_ujds)
j_gamma=fltarr(n_ujds)
for i=0,n_ujds-1 do begin
	i_a=where(velocities.jd eq ujds[i] and velocities.component eq 'A',n_a)
	i_b=where(velocities.jd eq ujds[i] and velocities.component eq 'B',n_b)
	if n_a eq 1 and n_b eq 1 then begin
		v_gamma[i]=(mass_a*velocities[i_a].value $
			   +mass_b*velocities[i_b].value)/(mass_a+mass_b)
		j_gamma[i]=ujds[i]
	endif
endfor
;
index=where(v_gamma ne 0,count)
if count eq 0 then begin
	print,'Error: no matching RV observations!'
	return,0
endif else begin
	jd=j_gamma[index]+2440000.d0
	rv=v_gamma[index]
	for i=0,count-1 do $
	print,'AB   '+string(jd[i],format='(f11.3)')+' '+string(rv[i])+' 1.0'
	return,{rv:rv,jd:jd}
endelse
;
end
;-------------------------------------------------------------------------------
function mag2flux,magnitudes
;
return,10^(magnitudes/(-2.5))
;
end
;-------------------------------------------------------------------------------
function modelfluxes,model,lamda
;
; Compute the flux scaling factor based on the magnitude of a component.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Corrections with magnitudes
if n_elements(gen_model.wavelengths) eq 1 then begin
	c1=lamda*0+mag2flux(model.magnitudes)
	c1=c1[0]
endif else if n_elements(gen_model.wavelengths) eq 2 then begin
	c1=mag2flux(polynom(lamda*1d6, $
			polyfit(gen_model.wavelengths,model.magnitudes,1)))
;print,'----------------------------'
;print,'Lambda',lamda*1e6,'Magnitudes:',model.magnitudes,'Flux',c1
;if n_elements(lamda) gt 2 then stop
endif else begin
	c1=mag2flux(spline(gen_model.wavelengths,model.magnitudes,lamda*1d6))
endelse
;
return,c1
;
end
;-------------------------------------------------------------------------------
function stellarfluxes,model,lamda,ld_coeffs
;
; For a given stellar model, return the fluxes for given wavelengths lamda[m]
; as well as limb darkening coefficients according to the model type.
; Note that all stellar disk fluxes are scaled with the squared diameter
; if the effective temperature (T) is non-zero, except if T=-5555, which
; is the value of T assigned by readmodel if T=0 on input (req. by PEARL).
;
; Flat flux distribution, no limb darkening (if applicable)
if model.teff eq 0 then begin
	ld_coeffs=dblarr(n_elements(lamda))
	fluxes=dblarr(n_elements(lamda))+1
;
; Black body (standard, negative teff)
endif else if model.teff lt 0 then begin
	ld_coeffs=dblarr(n_elements(lamda))
	fluxes=blackbody(abs(model.teff),lamda)
;
; Limb-darkening coefficients from Kurucz model atmospheres
endif else if abs(model.type) ge 5 and abs(model.type) le 7 then begin
	ld_coeffs=kurucoeffs(model,lamda*1d9,fluxes)
;
; Limb-darkening coefficients from Aufdenberg model atmospheres
endif else if abs(model.type) eq 8 then begin
	ld_coeffs=jasoncoeffs(model,lamda*1d9,fluxes)
;
; Limb-darkening coefficients and fluxes from Van Hamme
endif else begin
	ld_coeffs=limblinear(model.teff,model.logg,lamda*1d9,fluxes)
endelse
;
; Scale fluxes with square of diameter for physical models
if model.teff ne 0 and model.teff ne -5555 then begin
	if abs(model.type) ge 1 and abs(model.type) le 11 then begin
		fluxes=fluxes*model.diameter^2*model.ratio
	endif
endif
;
return,fluxes
;
end
;
;-------------------------------------------------------------------------------
function modelpx,component,error
;
; Compute the parallax [mas] from data for any input binary component.
; The error estimate is somewhat conservative in that the 
; masses are usually correlated, but this function just adds
; all stellar mass errors in quadrature.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(component) eq 0 then begin
	print,'Component not specified, assume A-B'
	component='A-B'
endif
;
if componentparse(strupcase(component),comp_1,comp_2) ne 0 then return,-1
mass_sum=total(componentmass(comp_1,comp_2))
i=where(binary_model.component eq strupcase(component)) & i=i[0]
parallax=binary_model[i].semimajoraxis $
	/(mass_sum*(binary_model[i].period/365.25)^2)^(1./3)
error=parallax*sqrt( $
	(binary_error[i].semimajoraxis/binary_model[i].semimajoraxis)^2 $
       +(total((star_error.mass/star_model.mass)^2)/9) $
       +(binary_error[i].period/binary_model[i].period)^2*(4./9))
return,parallax
;
end
;-------------------------------------------------------------------------------
function modelm,component,error
;
; Compute total mass of a binary component with Kepler's 3rd law with parallax.
; Also provides an error estimate if parameter errors been computed before.
;
common AuxData,parallaxes,k1,k2,vsini
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; if componentparse(strupcase(component),comp_1,comp_2) ne 0 then return,-1
; mass_sum=total(componentmass(comp_1,comp_2))
i=where(binary_model.component eq strupcase(component)) & i=i[0]
mass_sum=(binary_model[i].semimajoraxis/parallaxes.value[0])^3 $
	/(binary_model[i].period/365.25)^2
error=mass_sum*sqrt( $
	(3*binary_error[i].semimajoraxis/binary_model[i].semimajoraxis)^2+ $
       	(3*binary_error[i].period/binary_model[i].period)^2+ $
	(2*parallaxes.error[0]/parallaxes.value[0])^2)
return,mass_sum
;
end
;-------------------------------------------------------------------------------
function modelk,component
;
; Compute and return velocity semi-amplitude of requested stellar component.
; Note that for systems of three or more components, the amplitude is in
; rest frame of the corresponding binary component.
; Return -1 if failed for any reason.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180/pi_circle
;
if n_elements(binary_model) eq 0 then return,-1
pos=strpos(binary_model.component,strupcase(component))
index=where(pos ge 0,count)
if count eq 0 then return,-1
models=binary_model[index]
model=models[where(strlen(binary_model.component) eq $
	       min(strlen(binary_model.component)))]
model=model[0]
pos=strpos(model.component,strupcase(component))
if componentparse(model.component,comp_1,comp_2) ne 0 then return,-1
masses=componentmass(comp_1,comp_2)
m1=masses[0]
m2=masses[1]
i=model.inclination/RAD
e=model.eccentricity
p=model.period/s_year
k1=2*pi_circle*sin(i)/sqrt(1-e^2)*p^(-1.0/3.0)*m2*(m1+m2)^(-2.0/3.0) $
	*(a_unit/s_year)/86400000L
k2=k1*(m1/m2)
;
if pos eq 0 then return,k1 else return,k2
;
end
;-------------------------------------------------------------------------------
function modelf,component
;
; Compute and return mass function.
; Return -1 if failed for any reason.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180/pi_circle
;
if componentparse(strupcase(component),comp_1,comp_2) ne 0 then return,-1
i=where(binary_model.component eq strupcase(component)) & i=i[0]
masses=componentmass(comp_1,comp_2)
m1=masses[0]
m2=masses[1]
i=binary_model[i].inclination/RAD
;
fm=(m2^3*sin(i)^3)/(m1+m2)^2
;
return,fm
;
end
;-------------------------------------------------------------------------------
function modeldm,component,filter_in
;
; Compute broad band filter magnitude differences given a physical model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Filter should be one of UBVRI, or full function name as in filter.pro
if strlen(filter_in) eq 1 then filter_in=strupcase(filter_in)
;
teff1=star_model[0].teff
teff2=star_model[1].teff
logg1=star_model[0].logg
logg2=star_model[1].logg
d1=star_model[0].diameter
d2=star_model[1].diameter
;
case filter_in of
	'U':filter='johnson_u'
	'B':filter='johnson_b'
	'V':filter='johnson_v'
	'R':filter='johnson_r'
	'I':filter='johnson_i'
	'J':filter='johnson_j'
	'K':filter='johnson_k'
endcase
;
; Primary A
r=limbgrid(teff1,logg1,lamda,limbdu,fluxes)
f1=total(call_function(filter,lamda)*fluxes) $
    /total(call_function(filter,lamda))
;
; Secondary B
r=limbgrid(teff2,logg2,lamda,limbdu,fluxes)
f2=total(call_function(filter,lamda)*fluxes) $
    /total(call_function(filter,lamda))
;
; Radii
r1=d1/2
r2=d2/2
;
; Magnitudes
mag1=-2.5*alog10(f1*r1^2)
mag2=-2.5*alog10(f2*r2^2)
;
; Return predicted A-B magnitude differences
return,mag2-mag1
;
end
;-------------------------------------------------------------------------------
function modellum,component
;
; Given a binary_component, compute for all single-star sub-components
; the luminosity using Stefan-Boltzmann law from radius and eff. temperature.
; Must have read physical binary model before (Teff, diameter, masses, etc.).
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(component) eq 0 then component=topbincomp()
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
components=[comp_1,comp_2]
index=where(strlen(components) eq 1,n_comp)
components=components[index]
lum=fltarr(n_comp)
;
px=modelpx(component)
dm=5*alog10(1./(px/1000))-5
mas_to_rsun=(215./2)/px ; to convert angular diameters to radii in solar units
;
for i=0,n_comp-1 do begin
	k=where(star_model.component eq components[i])
	lum[i]=lum_teff[star_model[k].teff,star_model[k].diameter*mas_to_rsun]
	mv=4.74-2.5*alog10(lum[i])-bc_teff[star_model[k].teff]+dm
	if i eq 0 then tmv=mv else tmv=cmag(tmv,mv)
endfor
;
; Make sure total luminosity is normalized to observed (total) V magnitude
j=where(abs(gen_model.wavelengths-0.55) lt 0.02,count)
if count eq 0 then begin
	print,'Error: model lacks visual magnitude!'
	return,-1
endif
f=10^(-(star_model[0].magnitudes(j[0])-tmv)/2.5)
;
return,lum*f
;
end
;-------------------------------------------------------------------------------
function modellogg,component
;
; Given a binary_component, compute for all single-star sub-components
; the luminosity using Stefan-Boltzmann law from radius and eff. temperature.
; Must have read physical binary model before (Teff, diameter, masses, etc.).
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(component) eq 0 then component=topbincomp()
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
components=[comp_1,comp_2]
index=where(strlen(components) eq 1,n_comp)
components=components[index]
logg=fltarr(n_comp)
;
px=modelpx(component)
mas_to_rsun=(215./2)/px ; to convert angular diameters to radii in solar units
;
for i=0,n_comp-1 do begin
	k=where(star_model.component eq components[i])
	logg[i]=logg_mass[star_model[k].mass,star_model[k].diameter*mas_to_rsun]
endfor
;
return,logg
;
end
;-------------------------------------------------------------------------------
function wilson_wd,model
;
;     In IMLC, phase = 0 when eclipse occurs and arg_periastron = 90.
;     Therefore, we borrowed the following code from MODLOG in FNL
;     to calculate phase of the periastron, which is our zero point.
;
;     Function returns orbital phases for periastron, sup., and inferior conj.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(model) eq 0 then model=binary_model[0]
if n_elements(phzero) eq 0 then phzero=0
;
e=binary_model[0].eccentricity
p=((binary_model[0].periastron+180) mod 360)*!pi/180	; secondary periastron
;
perr=p
if e eq 0 then perr=!pi/2
;
trc=!pi/2-perr
while trc lt 0 do trc=trc+2*!pi
while trc ge 2*!pi do trc=trc-2*!pi
htrc=0.5*trc
if abs(!pi/2-htrc) ge 7.0e-6 and abs(4.712389-htrc) ge 7.0e-6 then $
	ecan=2.0*atan(sqrt((1.0-e)/(1.0+e))*tan(htrc)) else ecan=!pi
xmc=ecan-e*sin(ecan)
if xmc lt 0 then xmc=xmc+2*!pi
phper=1.0-xmc/(2*!pi)
; conjph15 subroutine
ecfac=sqrt((1.d0-e)/(1.d0+e))
trsc=!pi/2-p
tric=3.d0*!pi/2-p
econsc=2.d0*atan(ecfac*tan(trsc/2))
econic=2.d0*atan(ecfac*tan(tric/2))
xmsc=econsc-e*sin(econsc)
xmic=econic-e*sin(econic)
pconsc=(xmsc+p)/(2*!pi)-0.25+phzero
pconic=(xmic+p)/(2*!pi)-0.25+phzero
;
while pconsc ge 1.0 do pconsc=pconsc-1.0
while pconsc lt 0.0 do pconsc=pconsc+1.0
while pconic ge 1.0 do pconic=pconic-1.0
while pconic lt 0.0 do pconic=pconic+1.0
phperi=phper+pconsc	; phase of periastron
;
return,[phperi,pconsc,pconic]
;
; John Southword code (jktebop)
pi=pi_circle
ecc=e
argper=p/(180/pi_circle)
pih=pi/2
pi32=1.5*pi
twopi=2*pi
ecfac=sqrt((1.d0-ecc)/(1.d0+ecc))
;
trsc=pih-argper                                                   
tric=pi32-argper                                                  
econsc=2.d0*atan(ecfac*tan(.5d0*trsc))                          
econic=2.d0*atan(ecfac*tan(.5d0*tric))                          
xmsc=econsc-ecc*sin(econsc)                                      
xmic=econic-ecc*sin(econic)                                      
pconsc=(xmsc+argper)/twopi-.25d0+phzero                           
pconic=(xmic+argper)/twopi-.25d0+phzero                           
;
return,[pconsc,pconic]
;
end
;-------------------------------------------------------------------------------
function wilson,model
;
; Iteratively compute the orbital phase for the inferior conjunction of
; the secondary when it eclipses the primary. Primary eclipse is defined
; in lcmap to be at phase zero (modulo the phase correction parameter).
; By convention the primary eclipse is deeper than the secondary eclipse
; and star A is the star which is eclipsed during primary eclipse.
;
; The orbital phase for epoch JD is defined as (JD-EPOCH(PERIASTRON))/PERIOD
;
; The ascending node is read taken from the model, without verification.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(model) eq 0 then model=binary_model[0]
;
RAD=180/!pi
nsteps=45L
;
phases=findgen(nsteps)/nsteps
epochs=phases*model.period+model.epoch
radii=fltarr(nsteps)
angle=fltarr(nsteps)
;
o_parms=fltarr(8)
o_parms[0]=model[0].semimajoraxis
o_parms[1]=model[0].eccentricity
o_parms[2]=model[0].inclination/RAD
o_parms[3]=model[0].periastron/RAD
o_parms[4]=model[0].apsidalmotion/RAD
o_parms[5]=model[0].ascendingnode/RAD
o_parms[6]=model[0].period
o_parms[7]=model[0].epoch
;
for i=0L,nsteps-1 do begin
	xy=true2app(epochs[i],o_parms,rho,theta,gen_model.ra,gen_model.dec)
	radii[i]=rho
	angle[i]=theta*RAD
endfor
;
; Identify ascending node
if min(abs(angle-model[0].ascendingnode)) gt 5 then $
	print,'Warning: phase grid too coarse in function WILSON!'
index=where(abs((angle-model[0].ascendingnode)) eq $
	 min(abs(angle-model[0].ascendingnode)))
asc_phase=phases[index[0]]
if asc_phase lt 0 then asc_phase=asc_phase+1
;
; Look around orbit for smallest separation
i=where(radii eq min(radii)) & i=i[0]
p0=phases[0]
if i eq 0 	 then phases=(findgen(nsteps)-nsteps/2)/nsteps
if i eq nsteps-1 then phases=(findgen(nsteps)+nsteps/2)/nsteps
if phases[0] ne p0 then begin
	epochs=phases*model.period+model.epoch
	for i=0L,nsteps-1 do begin
	    xy=true2app(epochs[i],o_parms,rho,theta,gen_model.ra,gen_model.dec)
	    radii[i]=rho
	    angle[i]=theta
	endfor
endif
i=where(radii eq min(radii)) & i=i[0]
;
; Make sure it is the inferior conjunction
; Star A is, by definition, the one at superior conjunction near phase zero
con_phase=phases[i]
if con_phase gt asc_phase then radii[i]=max(radii)
;
; Iteratively home into closest approach of components
repeat begin
	i=where(radii eq min(radii)) & i=i[0]
	dphase=phases[i+1]-phases[i-1]
	phases=dphase*findgen(nsteps)/nsteps+phases[i-1]
	epochs=phases*model.period+model.epoch
	for i=0L,nsteps-1 do begin
	    xy=true2app(epochs[i],o_parms,rho,theta,gen_model.ra,gen_model.dec)
	    radii[i]=rho
	    angle[i]=theta
	endfor
endrep until dphase lt 0.0001
i=where(radii eq min(radii)) & i=i[0]
con_phase=phases[i]
;
return,con_phase
;
end
;-------------------------------------------------------------------------------
pro rochelobes
;
; Given a binary model, compute the Roche lobe radii in mas.
; Modern Astrophyics, Carroll and Ostlie, p.687
; https://en.wikipedia.org/wiki/Roche_lobe
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; print,'Roche lobe radii in solar radii:'
; a=((binary_model(0).period/365.25)^2*total(star_model.mass))^(1./3)
;
print,'Roche lobe radii in mas:'
a=binary_model[0].semimajoraxis
;
l1=a*(0.5-0.227*alog10(star_model[1].mass/star_model[0].mass))
l2=a*(0.5+0.227*alog10(star_model[1].mass/star_model[0].mass))
;
; Wikipedia
l1=a*(0.38+0.2*alog10(star_model[0].mass/star_model[1].mass))
l2=a*(0.38+0.2*alog10(star_model[1].mass/star_model[0].mass))
;
print,'Star 1:',l1
print,'Star 2:',l2
;
end
;-------------------------------------------------------------------------------
pro surfpot
;
; Read imlcout.xdr written by setlcparms with results on adjusting the 
; modified surface potentials.
;
restore,'imlcout.xdr'
print,'Initial  surface potential star 1:',poth
print,'Adjusted surface potential star 1:',poth_adj
print,'Initial  surface potential star 2:',potc
print,'Adjusted surface potential star 2:',potc_adj
;
end
;-------------------------------------------------------------------------------
function imlcparse,r,p
;
; Read output imlcout.dat of lc15 and extract r_side values and derivatives.
; Also return radii in solar units and periastron and conjunction phases (p).
;
f=file_search('imlcout.dat')
if strlen(f[0]) eq 0 then return,fltarr(2,2)
;
l=''
status=dc_read_fixed('imlcout.dat',l,/col,format='(a140)')
;
index=where(strpos(l,'(Mean Radius)/Rsun') ge 0,count) & index=index[0]
i=1
while strlen(strcompress(l[index+i],/remove_all)) eq 0 do i=i+1
r=fltarr(2)
words=nameparse(l[index+i])
r[0]=float(words[2])
words=nameparse(l[index+i+1])
r[1]=float(words[2])
;
index=where(strpos(l,'inferior') ge 0,count) & index=index[0]
i=1
while strlen(strcompress(l[index+i],/remove_all)) eq 0 do i=i+1
p=fltarr(3)
words=nameparse(l[index+i+2])
p[0]=float(words[6])
p[1]=float(words[7])
p[2]=float(words[8])
;
index=where(strpos(l,'r side') ge 0,count) & index=index[0]
i=1
while strlen(strcompress(l[index+i],/remove_all)) eq 0 do i=i+1
s=fltarr(2,2)
words=nameparse(l[index+i])
s[*,0]=float(words[5:6])
words=nameparse(l[index+i+1])
s[*,1]=float(words[5:6])
;
return,s
;
end
;-------------------------------------------------------------------------------
function opotentials,r,q,f,the=the,phi=phi
;
; Compute modified surface potentials according to Eq.1 of Wilson, 1979
; r is radius/semi-major axis, hence r <= 1
;
if n_elements(f) eq 0 then f=1.
if n_elements(q) eq 0 then q=1.
if n_elements(r) eq 0 then r=1.
;
if n_elements(the) eq 0 then the=0
if n_elements(phi) eq 0 then phi=0
;
d=1.
l=cos(the*!pi/180)
n=cos(phi*!pi/180)
;
return,1/r+q*((d^2+r^2-2*r*l*d^2)+0.5*f^2*(1+q)*r^2*(1-n^2))
;
end
;-------------------------------------------------------------------------------
function lcpotentials,parameter
;
; Given a binary model for LCDC, return the potentials approximating best
; the given stellar diameters. Also return the corresponding diameters. 
;
forward_function setlcparms
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
index=where(binary_model.method eq 4)
model=binary_model[index[0]]
if componentparse(model.component,comp_1,comp_2) ne 0 then return,-1
star1=where(star_model.component eq comp_1) & star1=star1[0]
star2=where(star_model.component eq comp_2) & star2=star2[0]
model1=star_model[star1]
model2=star_model[star2]
max_spots=2
max_stars=2
spotparms=fltarr(4,max_spots,max_stars)
;
if n_elements(parameter) gt 2 $
then lcparms=parameter $
else lcparms=setlcparms(model,binary_model.epoch,spotparms)
lcparms[0]=1
poth=lcparms[34]
potc=lcparms[35]
;
; Adjust surface potentials to match diameters
maxep=200L		; maximum number of epochs, must be same in lc15.f
icc=1l
ccc=1l
xcc=0.d0
ycc=0.d0
fcc=0.d0
phas=fltarr(maxep)
hkm1=fltarr(maxep)
hkm2=fltarr(maxep)
hflx=fltarr(maxep)
phas0p5=double(0.5)
n_epoch=1L
lcpath=blanks(80)
strput,lcpath,!oyster_dir+'source/lcdc2015'
repeat begin
status=linknload(!external_lib,'lc2015',lcparms,spotparms, $
	icc,ccc,xcc,ycc,fcc, $
	n_epoch,phas0p5,hkm1,hkm2,hflx,lcpath)
;
s=imlcparse(r)*model.semimajoraxis
if total(s) ne 0 then begin
ds=[(model1.diameter/2)-s[0,0],(model2.diameter/2)-s[0,1]]
d_omega1=ds[0]/s[1,0]
d_omega2=ds[1]/s[1,1]
lcparms[34]=lcparms[34]+d_omega1
lcparms[35]=lcparms[35]+d_omega2/10
endif else begin
	print,'Could not read imlcout.dat!'
	d_omega1=0
	d_omega2=0
endelse
endrep until abs(ds[0]/s[0,0]) lt 0.1 and abs(ds[1]/s[0,1]) lt 0.1
poth_adj=lcparms[34]
potc_adj=lcparms[35]
save,poth,potc,poth_adj,potc_adj,filename='imlcout.xdr'
;
au2mas=modelpx(binary_model[0].component)/215
d=s[0,*]*2	; use r_side value
d=r*au2mas*2
if n_elements(parameter) le 2 then parameter=d
;
return,lcparms[34:35]
;
end
;-------------------------------------------------------------------------------
function setwdparms,model,jd,lamda,spotparms
;
; Initialize parameters for the Wilson-Devinney code (WD, 1992).
; jd is the Julian day epochs (-2440000); lamda [m] the wavelength.
; Return the parameters in form of an array, also return spotparms.
;
; Primary: increasing spot longitude rotates star against orbital sense
;		longitude zero spot faces secondary
; Secondary: increasing longitude rotates star against orbital sense
;		longitude zero spot faces primary
; Spot latitude increases from North to South.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k1,k2,vsini
;
RAD=180/pi_circle
;
if componentparse(model.component,comp_1,comp_2) ne 0 then return,-1
star1=where(star_model.component eq comp_1) & star1=star1[0]
star2=where(star_model.component eq comp_2) & star2=star2[0]
model1=star_model[star1]
model2=star_model[star2]
;
num_parms=54
wdparms=fltarr(num_parms)
max_spots=2
max_stars=2
spotparms=fltarr(4,max_spots,max_stars)
;
; spotparms(0,0,0)=180. 	; Latitude [deg], 0 - 180 
; spotparms(1,0,0)=0.		; Longitude [deg], 0 - 360 
; spotparms(2,0,0)=20.		; Radius [deg]
; spotparms(3,0,0)=0.8		; Temperature factor
;
;
ifrad=0			; controls output format (unused)
nref=1			; number of reflections if mref=2
mref=1			; simple treatment of reflection effect
ifsmv1=0 		; star spot control #1 (0/1, 0=stationary spot)
ifsmv2=0 		; star spot control #2 (0/1, 1=spot moves in long.)
icor1=1			; unknown at present
icor2=1			; unknown at present
ld=1			; linear cosine law for limb darkening
modeb=model.wdtype 	; -1: X-ray binaries
			; 0: no constraints
			; 2: detached binary
			; 3: overcontact binaries
			; 4: semi-detached binaries
			; 5: Algol-types
			; 6: double-contact
ipb=1  			; 0: compute L2 from T2; 1: decouple L from T
ifat1=0 		; 0 = Blackbody radiation, 1 = stellar atmosphere
ifat2=0 		; see ifat1
n1=10 			; number of latitude rows per hemisphere
n2=10 			; see n1, longitude elements scale with sine of latitude
bperiod=model.period	; orbital period in days
the=0. 			; semi-duration of primary eclipse (unused)
vunit=1 		; unit for radial velocity input and output (km/s)
e=model.eccentricity	; eccentricity
per=model.periastron 	; in degrees
; 			K1, K2 [km/s]
k1_m=2*pi_circle*sin(model.inclination/RAD)/sqrt(1-model.eccentricity^2) $
    *(model.period/s_year)^(-1.0/3.0) $
    *model2.mass*(model1.mass+model2.mass)^(-2.0/3.0) $
    *a_unit/s_year/86400/1000
k2_m=k1_m*(model1.mass/model2.mass)
a1sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k1_m*model.period
a2sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k2_m*model.period
rsun=6.96e5 		; radius of sun [km]
asemima=(a1sini+a2sini)/sin(model.inclination/RAD)/rsun
f1=model1.omega 	; ratio of axial rotation rate to orbital rate
f2=model2.omega
vga=gen_model.rv/vunit	; systemic velocity (in units of vunit)
pshift=0. 		; constant phase shift
xincl=model.inclination	; orbital inclination in degrees
xincl=(90-xincl)+90	; Bug in WD?: prograde and retrograde exchanged!
gr1=model1.gr 		; exponent in gravity darkening law
gr2=model2.gr 		; see gr1, convective envelopes: 0.3, radiative: 1.0
tavh=abs(model1.teff)/10000 	; effective temperature in 10000 K
tavc=abs(model2.teff)/10000	; see tavh, make sure tavh/tavc are positive
alb1=model1.albedo	; bolometric albedo
alb2=model2.albedo	; see alb1, convective envelopes: 0.5-1.0, radiative:1.0
poth=model.semimajoraxis/(model1.diameter/2)
			; omega potential = semi-major axis / radius
potc=model.semimajoraxis/(model2.diameter/2)
			; see poth, for second star
rm=model2.mass/model1.mass
			; mass ratio
xbol1=0.4       	; coefficient in bolometric limb darkening law
xbol2=0.4		; see xbol1
ybol1=0.0		; coefficient for logarithmic term
ybol2=0.0		; see ybol1
wl=lamda*1e6		; wavelength in microns
; model1.teff=abs(model1.teff)	; make sure stellarfluxes uses blackbody
; model2.teff=abs(model2.teff)	; make sure stellarfluxes uses blackbody
; hlum=poly(lamda*1e6,model1.flux_fitcoeffs)*4*pi_circle
hlum=stellarfluxes(model1,lamda,xh)
			; monochromatic luminosity
; clum=poly(lamda*1e6,model2.flux_fitcoeffs)*4*pi_circle
clum=stellarfluxes(model2,lamda,xc)
; we need clum/hlum=0.12
; clum=hlum*0.15
			; see hlum
tlum=hlum+clum
hlum=hlum/tlum
clum=clum/tlum
; xh=poly(lamda*1e6,model1.ld_fitcoeffs)
			; limb-darkening coefficient
; xc=poly(lamda*1e6,model2.ld_fitcoeffs)
			; see xh, for second star
yh=0.0			; see xh coefficient for logarithmic term
yc=0.0			; see yh, (unused if ld=1)
el3=0.0			; third light (unused)
zero=0.0 		; zero point for output magnitudes
factor=1.0		; scaling factor for normalized light column
phperi=wilson_wd(0)
phn=(jd-model.epoch)/bperiod mod 1	; phases of binary
if e eq 0 then phn=phn+(per-90.)/360. else phn=phn+phperi
phstrt=phn 		; start phase
phstop=phn 		; stop phase (do only phase phn)
phin=0.01  		; phase increment (we do only phn, however)
;
; spotparms(0,0,0)=180. 	; Latitude [deg], 0 - 180 
; spotparms(1,0,0)=0.		; Longitude [deg], 0 - 360 
; spotparms(2,0,0)=20.		; Radius [deg]
; spotparms(3,0,0)=0.8		; Temperature factor
;
spotparms=star_model.spotparms
nsp1=fix(total(total(abs(model1.spotparms),1) gt 0)); Number of spots on star 1
nsp2=fix(total(total(abs(model2.spotparms),1) gt 0)); Number of spots on star 2
;
wdparms[0] = ifrad
wdparms[1] = nref
wdparms[2] = mref
wdparms[3] = ifsmv1
wdparms[4] = ifsmv2
wdparms[5] = icor1
wdparms[6] = icor2
wdparms[7] = ld
wdparms[8] = MODEB
wdparms[9] = IPB
wdparms[10]= IFAT1
wdparms[11]= IFAT2
wdparms[12]= N1
wdparms[13]= N2
wdparms[14]= BPERIOD
wdparms[15]= THE
wdparms[16]= VUNIT
wdparms[17]= PHN
wdparms[18]= PHSTRT
wdparms[19]= PHSTOP
wdparms[20]= PHIN
wdparms[21]= E
wdparms[22]= PER
wdparms[23]= ASEMIMA
wdparms[24]= F1
wdparms[25]= F2
wdparms[26]= VGA
wdparms[27]= PSHIFT
wdparms[28]= XINCL
wdparms[29]= GR1
wdparms[30]= GR2
wdparms[31]= tavh
wdparms[32]= tavc
wdparms[33]= alb1
wdparms[34]= alb2
wdparms[35]= poth
wdparms[36]= potc
wdparms[37]= rm
wdparms[38]= xbol1
wdparms[39]= xbol2
wdparms[40]= ybol1
wdparms[41]= ybol2
wdparms[42]= WL
wdparms[43]= hlum
wdparms[44]= clum
wdparms[45]= xh
wdparms[46]= xc
wdparms[47]= yh
wdparms[48]= yc
wdparms[49]= EL3
wdparms[50]= ZERO
wdparms[51]= FACTOR
wdparms[52]= NSP1
wdparms[53]= NSP2
;
return,wdparms
;
end
;-------------------------------------------------------------------------------
function setlcparms,model,jds,spotparms
;
; Initialize parameters for the Wilson-Devinney code (LCDC, 2015).
; jds are the Julian day epochs (-2440000).
; Return the parameters in form of an array, also return spotparms.
;
; Primary: increasing spot longitude rotates star against orbital sense
;		longitude zero spot faces secondary
; Secondary: increasing longitude rotates star against orbital sense
;		longitude zero spot faces primary
; Spot latitude increases from North to South.
;
; For light curves (MPAGE=1), star 1 is, by definition, the one at superior
; conjunction near phase zero when parameter PSHIFT is entered as zero. It
; will be the one eclipsed near phase zero if there are eclipses.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k1,k2,vsini
;
RAD=180/pi_circle
;
; Restore float array "effwvls" and string array "filters" for WD bands
restore,!oyster_dir+'source/lcdc2015/ibands.xdr'
;
if componentparse(model.component,comp_1,comp_2) ne 0 then return,-1
star1=where(star_model.component eq comp_1) & star1=star1[0]
star2=where(star_model.component eq comp_2) & star2=star2[0]
model1=star_model[star1]
model2=star_model[star2]
;
num_parms=85
lcparms=dblarr(num_parms)
max_spots=2
max_stars=2
spotparms=dblarr(4,max_spots,max_stars)
;
; spotparms(0,0,0)=180. 	; Latitude [deg], 0 - 180 
; spotparms(1,0,0)=0.		; Longitude [deg], 0 - 360 
; spotparms(2,0,0)=20.		; Radius [deg]
; spotparms(3,0,0)=0.8		; Temperature factor
;
for j=0,max_spots-1 do begin
	spotparms[*,j,0]=model1.spotparms[*,j]
	spotparms[*,j,1]=model2.spotparms[*,j]
endfor
spotparms[0:2,*,*]=spotparms[0:2,*,*]/RAD
nsp1=fix(total(total(abs(model1.spotparms),1) gt 0)); Number of spots on star 1
nsp2=fix(total(total(abs(model2.spotparms),1) gt 0)); Number of spots on star 2
;
; Parameters (in order read by kmlc): implicit real*8(a-h,o-z)
mpage=5			; output: 1=light curves, 2=RVs, 5=images
nref=1			; number of reflections if mref=2
mref=1			; simple treatment of reflection effect
ifsmv1=0 		; star spot control #1 (0/1, 0=stationary spot)
ifsmv2=0 		; star spot control #2 (0/1, 1=spot moves in long.)
icor1=1			; proximity and eclipse effects star 1 if 1
icor2=1			; proximity and eclipse effects star 2 if 1
if3b=0			; light time effect due to 3rd body
ld1=1			; linear cosine law for limb darkening od star 1
ld2=1			; linear cosine law for limb darkening od star 2
kspev=0			; whether spots can evolve
kspot=1			; whether old spot model applies (1) or new one (2)
nomax=1			; spot growth trapezoidal
ifcgs=0			; physical units for light curves
ktstep=0		; only for mpage=6
jdphs=2			; independent variable time (1) or phase (2)
hjd0=model.epoch	; ephemeris reference time (for jdphs=1)
pzero=model.period	; orbital period in days (at hjd0)
phases=(jds-model.epoch)$
      /pzero mod 1	; phases of binary, superior conjunction of primary at 0
dpdt=0.			; rate of period change
pshift=0. 		; constant phase shift
delph=0.		;
nga=1			; NGA=1 there is no phase smearing
stdev=0.		;
noise=0.		;
seed=0.			;
hjdst=min(jds)		; start time for output points (jdphs=1)
hjdsp=max(jds)		; stop  time for output points (jdphs=1)
hjdin=1			; increment of time for output points (jdphs=1)
phn=0			; Phase of normalization for normalized light/magnitude
phstrt=min(phases)	; start phase, for jdphs=2
phstop=max(phases)	; stop phase, for jdphs=2
phin=0.1  		; phase increment, for jdphs=2
phobs=0.		; phase of tobs datum
lsp=1			; whether TOBS and TAVSP are for star 1 or star 2
tobs=abs(model1.teff)/10000  ; Spectroscopic temperature at phase PHOBS
tavsp=abs(model1.teff)/10000 ; Computed flux-weighted mean surface temperature
mode=model.wdtype 	; -1: X-ray binaries
			; 0: no constraints
			; 2: detached binary
			; 3: overcontact binaries
			; 4: semi-detached binaries
			; 5: Algol-types
			; 6: double-contact
ipb=1  			; 0: compute L2 from T2; 1: decouple L from T
ifat1=0 		; 0 = Blackbody radiation, 1 = stellar atmosphere
ifat2=0 		; see ifat1
n1=20 			; number of latitude rows per hemisphere
n2=20 			; see n1, longitude elements scale with sine of latitude
perr0=(model.periastron+180)/RAD 	; in radians, +180 as of Feb 22, 2019!
dperdt=0.		; the first time derivative of ω
the=0. 			; semi-duration of primary eclipse
vunit=1 		; unit for radial velocity input and output (km/s)
e=model.eccentricity	; eccentricity
k1_m=modelk(model1.component)
k2_m=modelk(model2.component)
a1sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k1_m*model.period
a2sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k2_m*model.period
rsun=6.96e5 		; radius of sun [km]
a=((a1sini+a2sini)/sin(model.inclination/RAD))/rsun
f1=model1.omega 	; ratio of axial rotation rate to orbital rate
f2=model2.omega
vga=gen_model.rv/vunit	; systemic velocity (in units of vunit)
vga=0			; set to zero, this is added by modelvel
xincl=model.inclination	; orbital inclination in degrees
xincl=(90-xincl)+90	; Bug in WD?: prograde and retrograde exchanged!
gr1=model1.gr 		; exponent in gravity darkening law
gr2=model2.gr 		; see gr1, convective envelopes: 0.3, radiative: 1.0
abunin=0.		; [M/H] as defined logarithmically, relative to the Sun
fspot1=0.		; spot angular drift rates in longitude
fspot2=0.		; rate 1.0000 means drift matches mean orbital rate
tavh=abs(model1.teff)/10000 	; effective temperature in 10000 K
tavc=abs(model2.teff)/10000	; see tavh, make sure tavh/tavc are positive
alb1=model1.albedo	; bolometric albedo
alb2=model2.albedo	; see alb1, convective envelopes: 0.5-1.0, radiative:1.0
rm=model2.mass/model1.mass
			; mass ratio
poth=model.semimajoraxis/(model1.diameter/2)
			; omega potential estimate = semi-major axis / radius
potc=model.semimajoraxis/(model2.diameter/2)
			; see poth, for second star
xbol1=0.4       	; coefficient in bolometric limb darkening law
xbol2=0.4		; see xbol1
ybol1=0.0		; coefficient for logarithmic term
ybol2=0.0		; see ybol1
dpclog=alog10(1000./modelpx(model.component))
			; logarithm (base 10) of distance (d) in parsecs
;
iband=model.wdband	; photometric band offered by lcdc, e.g. 7 for V-band
; hl,cl			; narrowband (5%) luminosities
l0=effwvls[iband-1]*1e-9; lamda [m]
lamda_min=l0/1.025
lamda_max=l0*1.025
nl=100
lamda=lamda_min+(lamda_max-lamda_min)*findgen(nl)/(nl-1)
hl=mean(stellarfluxes(model1,lamda,x1)) & x1=mean(x1)
cl=mean(stellarfluxes(model2,lamda,x2)) & x2=mean(x2)
if ipb eq 1 then begin
	tl=hl+cl
	cl=cl/hl
	hl=1.0
;	hl=hl/tl
;	cl=cl/tl
endif else begin
	hl=1.
	cl=1.
endelse
;
; Tests with parameters from Binary Stars, A Pictorial Atlas, Terrell et al.
; TU Cam, page 199, mode/wdtype=2
if 0 then begin
	poth=3.548
	potc=5.701
	rm=0.47
	pzero=2.9333
	xincl=77.6
; Beta Per, page 239, mode/wdtype=5 (Algols, secondary star fills roche lobe)
	poth=5.151
	potc=2.299
	rm=0.227
	pzero=2.867
	xincl=82.31
endif
;
y1=0.0			; see xh coefficient for logarithmic term
y2=0.0			; see yh, (unused if ld=1)
el3=0.0			; third light
opsf=0.			; opacity in a specific band, related to ISM
zero=0.0 		; zero point for output magnitudes
factor=1.0		; scaling factor for normalized light column
wl=l0*1e6		; wavelength in microns, only used for mpage=3
aextinc=0.		; ISM extinction in band
calib=1.		; flux calibration constant (cgs) for star of mag=0
;
lcparms[0] = mpage
lcparms[1] = nref
lcparms[2] = mref
lcparms[3] = ifsmv1
lcparms[4] = ifsmv2
lcparms[5] = icor1
lcparms[6] = icor2
lcparms[7] = ld1
lcparms[8] = mode
lcparms[9] = ipb
lcparms[10]= ifat1
lcparms[11]= ifat2
lcparms[12]= n1
lcparms[13]= n2
lcparms[14]= perr0
lcparms[15]= the
lcparms[16]= vunit
lcparms[17]= phn
lcparms[18]= phstrt
lcparms[19]= phstop
lcparms[20]= phin
lcparms[21]= e
lcparms[22]= a
lcparms[23]= f1
lcparms[24]= f2
lcparms[25]= vga
lcparms[26]= pshift
lcparms[27]= xincl
lcparms[28]= gr1
lcparms[29]= gr2
lcparms[30]= tavh
lcparms[31]= tavc
lcparms[32]= alb1
lcparms[33]= alb2
lcparms[34]= poth
lcparms[35]= potc
lcparms[36]= rm
lcparms[37]= xbol1
lcparms[38]= xbol2
lcparms[39]= ybol1
lcparms[40]= ybol2
lcparms[41]= wl
lcparms[42]= hl
lcparms[43]= cl
lcparms[44]= x1
lcparms[45]= x2
lcparms[46]= y1
lcparms[47]= y2
lcparms[48]= el3
lcparms[49]= zero
lcparms[50]= factor
lcparms[51]= nsp1
lcparms[52]= nsp2
lcparms[53]= if3b
lcparms[54]= ld2
lcparms[55]= kspev
lcparms[56]= kspot
lcparms[57]= nomax
lcparms[58]= ifcgs
lcparms[59]= ktstep
lcparms[60]= jdphs
lcparms[61]= hjd0
lcparms[62]= pzero
lcparms[63]= dpdt
lcparms[64]= delph
lcparms[65]= nga
lcparms[66]= stdev
lcparms[67]= noise
lcparms[68]= seed
lcparms[69]= hjdst
lcparms[70]= hjdsp
lcparms[71]= hjdin
lcparms[72]= phobs
lcparms[73]= lsp
lcparms[74]= tobs
lcparms[75]= tavsp
lcparms[76]= dperdt
lcparms[77]= abunin
lcparms[78]= fspot1
lcparms[79]= fspot2
lcparms[80]= dpclog
lcparms[81]= iband
lcparms[82]= opsf
lcparms[83]= aextinc
lcparms[84]= calib
;
; Use LCDC to compute Omega potentials which match the diameters.
; Diameters of components constrained by the model (e.g. Algols)
; are not checked as the input diameter is ignored in these cases.
pothc=lcpotentials(lcparms)
lcparms[34]= pothc[0]
lcparms[35]= pothc[1]
;lcparms(34)= 5.033
;lcparms(35)= 4.281
;
return,lcparms
;
end
;-------------------------------------------------------------------------------
function setrocheparms,model,lamda,u,v
;
; Return array of parameters needed by the Roche code to compute model.
; This function is for the code V-1.4
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k1,k2,vsini
;
RAD=180/pi_circle
;
rocheparms=fltarr(7)
;
rocheparms[0]=model.omega	; Rotational velocity ( < 1) in units of
				; break-up velocity
rocheparms[1]=((model.diameter/2)/parallaxes.value[0])*(a_unit/r_sun)
				; Stellar radius [Rsun] for zero rotation
rocheparms[2]=model.teff	; Effective temperature at pole for omega=1
rocheparms[3]=model.mass	; Stellar mass, needed to compute gravity
rocheparms[4]=model.tilt/RAD	; tilt of the rotation axis to LOS in degrees
				; Positive tilt moves pole southwards
rocheparms[5]=model.pa		; Pos. angle of the rotation  axis on sky,
				; from N over E
rocheparms[6]=model.diameter	; Polar angular diameter at breakup, 
				; which is 0.8913 of the diameter of the
				; non-rotating star. But acc. to more recent
				; theories, is almost (1-2%) the same, so
				; we just use the angular diameter.
;
return,rocheparms
;
end
;-------------------------------------------------------------------------------
function setrocheparms13,model,lamda,u,v
;
; Return array of parameters needed by the Roche code to compute model.
; This function is for the code V-1.3
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k1,k2,vsini
;
RAD=180/pi_circle
;
rocheparms=fltarr(9)
;
rocheparms[0]=model.omega	; rotational velocity ( < 1)  
				; in units of break-up vel.
rocheparms[1]=lamda*1e9	; wavelength in nm
rocheparms[2]=model.tilt	; tilt of the rotation axis to LOS in degrees
				; Positive tilt moves pole southwards
if model.tilt gt 90 then rocheparms[2]=-(180-model.tilt)
rocheparms[3]=atan(u,v)*180/!pi+model.pa
rocheparms[3]=atan(u,v)*180/!pi-model.pa
				; angle of baseline to rotational axis
rocheparms[4]=model.mass	; stellar mass
rocheparms[5]=((model.diameter/2)/parallaxes.value[0])*(a_unit/r_sun)
;				; stellar radius in units of solar radius
rocheparms[6]=model.teff	; effective temperature
rocheparms[7]=model.diameter	; Angular diameter
rocheparms[8]=model.pa	 	; PA of the maj. axis, from N over E
;
return,rocheparms
;
end
;-------------------------------------------------------------------------------
function wdmap,model,epoch,lamda
;
; Setup a call to the Wilson-Devinney (1992) code to get a map. Model is the 
; binary model. Epoch is JD-2440000 and lamda in [m].
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
wdparms=setwdparms(model,epoch,lamda,spotparms)
;
icc=0L
maxcc=1000
ccc=lonarr(maxcc)
xcc=dblarr(maxcc)
ycc=dblarr(maxcc)
fcc=dblarr(maxcc)
;
status=linknload(!external_lib,'wilson',wdparms,spotparms,icc,ccc,xcc,ycc,fcc)
i=where(finite(fcc) eq 0,count)
;
ccc=ccc[0:icc-1]
xcc=xcc[0:icc-1]
ycc=ycc[0:icc-1]
fcc=fcc[0:icc-1]
angle=(90-model.ascendingnode)/RAD
rxcc=(-ycc*sin(angle)+xcc*cos(angle))*model.semimajoraxis
rycc=(+xcc*sin(angle)+ycc*cos(angle))*model.semimajoraxis
map=alloc_map(icc,ccc,rxcc,rycc,fcc)
;
return,map
;
end
;-------------------------------------------------------------------------------
function wdrav,model,epochs,lamda
;
; Setup a call to the Wilson-Devinney (1992) code to get radial velocities. 
; Model is the binary model. Epochs are JD-2440000 and lamda in [m].
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
wdparms=setwdparms(model,epochs[0],lamda,spotparms)
;
icc=0L
maxcc=1000
ccc=lonarr(maxcc)
xcc=dblarr(maxcc)
ycc=dblarr(maxcc)
fcc=dblarr(maxcc)
;
n_epochs=n_elements(epochs)
phas=fltarr(maxcc)
hkm1=fltarr(maxcc)
hkm2=fltarr(maxcc)
;
phas[0:n_epochs-1]=((epochs-model.epoch)/model.period) mod 1
phas=phas-0.255
;
status=linknload(!external_lib,'devinney',wdparms,spotparms, $
	icc,ccc,xcc,ycc,fcc, $
	n_epochs,phas,hkm1,hkm2)
;
return,[[hkm1],[hkm2]]
;
end
;-------------------------------------------------------------------------------
function lcmap,model,epochs,mpage
;
; Setup a call to the Wilson-Devinney (2015) code. 
; Model is the binary model. Epochs are JD-2440000.
; Repeat the computations only if any of the parameters changed.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common LocalLcMap1,lcparms,lcparms_p,lcepoch,lcepoch_p,hflx,hkm1,hkm2
common LocalLcMap2,icc,ccc,xcc,ycc,fcc,maps
;
RAD=180/pi_circle
if n_elements(lcparms_p) eq 0 then lcparms_p=''
if n_elements(lcepoch_p) eq 0 then lcepoch_p=''
;
lcparms=setlcparms(model,epochs[0],spotparms)
lcparms[0]=mpage
lcepoch=epochs
n_epochs=n_elements(epochs)
;
; Use this string to decide whether to call lc15 again
lcparms_s=strjoin(string(lcparms))
lcepoch_s=strjoin(string(lcepoch))
;
; Information needed for flux normalization
restore,!oyster_dir+'source/lcdc2015/ibands.xdr'
l0=effwvls[model.wdband-1]*1e-9; lamda [m]
;
IF lcparms_s NE lcparms_p OR lcepoch_s NE lcepoch_p THEN BEGIN
;
lcparms_p=lcparms_s
lcepoch_p=lcepoch_s
;
maxep=200L		; maximum number of epochs, must be same in lc15.f
icc=lonarr(maxep)	; number of fluxes for each map
maxcc=maxep*1000L	; maximum number of elements in following arrays
ccc=lonarr(maxcc)
xcc=dblarr(maxcc)
ycc=dblarr(maxcc)
fcc=dblarr(maxcc)
;
if n_epochs gt maxep then begin
	print,'***ERROR(LC2015): number of epochs larger than MAXEP!'
	return,-1
endif
phas=dblarr(maxep)
hkm1=dblarr(maxep)
hkm2=dblarr(maxep)
hflx=dblarr(maxep)
;
; Correct orbital phases for eclipse phase (binary phase=0)
; The next line in effect until February 21, 2019 (UX Ari, zero eccentricity)
phas[0:n_epochs-1]=((epochs-model.epoch)/model.period mod 1)-wilson(model)
; New code from February 22, 2019, onwards (Alpha Draconis, non-zero ecc.!)
phases=wilson_wd(model)
phas[0:n_epochs-1]=((epochs-model.epoch)/model.period mod 1)+phases[0]
if median(phas) lt 0 then phas=phas+1
;
; Compute orbital phase of secondary's inferior conjunction
phase_p=phases[0]	; corrected periastron (model + 90 degrees)
phase_s=phases[1]	; superior conjunction
phase_i=phases[2]	; inferior conjunction
if model.eccentricity eq 0 then phase_p=0.5
print,'Inferior conjunction at orbital phase:',(phase_i-phase_p) mod 1
;
; Call LC without spots and outside eclipse to normalize flux ratio
;
lcparms[0]=1
phas0p5=double(0.5)
spotparms0=spotparms*0
n_epoch=n_elements(phas0p5)
lcpath=blanks(80)
strput,lcpath,!oyster_dir+'source/lcdc2015'
status=linknload(!external_lib,'lc2015',lcparms,spotparms0, $
	icc,ccc,xcc,ycc,fcc, $
	n_epoch,phas0p5,hkm1,hkm2,hflx,lcpath)
hflx0=hflx[0]
;
; Compute stellar fluxes in erg/s/cm^3 for equivalent physical model
df=1.25	; ratio type 3 diameter to WD diameter
d1=df*2*model.semimajoraxis/lcparms[34]	; mas
d2=df*2*model.semimajoraxis/lcparms[35]	; mas
r1=lcparms[22]/lcparms[34]
r2=lcparms[22]/lcparms[35]
pi=model.semimajoraxis/(lcparms[22]/215)
tm=(model.semimajoraxis/modelpx(model.component))^3/(model.period/365.25)^2
rm=lcparms[36]
m1=tm/(1+rm)
m2=m1*rm
l1=alog10(m1)-2*alog10(r1)+4.44
l2=alog10(m2)-2*alog10(r2)+4.44
f1=stellarfluxes( $
	{teff:lcparms[30]*1e4,logg:l1,diameter:d1,ratio:1.0,type:3}, $
	 l0,ld_coeffs)
f2=stellarfluxes( $
	{teff:lcparms[31]*1e4,logg:l2,diameter:d2,ratio:1.0,type:3}, $
	 l0,ld_coeffs)
;
; Regular LC call
lcparms[0]=mpage
lcpath=blanks(80)
strput,lcpath,!oyster_dir+'source/lcdc2015'
status=linknload(!external_lib,'lc2015',lcparms,spotparms, $
	icc,ccc,xcc,ycc,fcc, $
	n_epochs,phas,hkm1,hkm2,hflx,lcpath)
;
; Convert fluxes to physical
fcc=fcc*(f1+f2)/hflx0
;
if mpage eq 5 then begin
	n=total(icc)
	ccc=ccc[0:n-1]
	xcc=xcc[0:n-1]
	ycc=ycc[0:n-1]
	fcc=fcc[0:n-1]
endif
;
ENDIF
;
case lcparms[0] of
1: 	return,hflx[0:n_epochs-1]
2: 	return,[[hkm1[0:n_epochs-1]],[hkm2[0:n_epochs-1]]]
5: 	begin
	angle=(90-model.ascendingnode)/RAD
	rxcc=(-ycc*sin(angle)+xcc*cos(angle))*model.semimajoraxis
	rycc=(+xcc*sin(angle)+ycc*cos(angle))*model.semimajoraxis
	maps=alloc_maps(icc,ccc,rxcc,rycc,fcc)
	return,maps
	end
endcase
;
end
;-------------------------------------------------------------------------------
function imgmap,map,uv_cell=uv_cell_in,uv_taper=uv_taper
;
; Creates an image from a map using the fast Fourier transform.
; Map coordinates are in mas. map.x corresponds to RA and increases towards
; East, i.e. left. Lambda [m].
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; Determine mean resolution and maximum size
x_size=max(map.xcc)-min(map.xcc)
y_size=max(map.ycc)-min(map.ycc)
c_size=sqrt((x_size*y_size)/n_elements(map.fcc))
i_size=sqrt(x_size*y_size)
;
fringespacing_min=c_size/2	; c_size*4 (pearldata), but we need larger image
max_uvr=(180/!pi)*3600000/fringespacing_min
imsize=[201,201]	; must be odd numbers
uv_cell=max_uvr/imsize[0]
if n_elements(uv_cell_in) ne 0 then $
	if uv_cell_in eq 0 then print,'uv_cell=',uv_cell else uv_cell=uv_cell_in
if n_elements(uv_taper) eq 0 then uv_taper=100.
;
uvmap=set_uvc(imsize,uv_cell,1)
cvmap=complexarr(n_elements(uvmap.u))
;
; Image has to have N top and E to the left
u=-uvmap.u
v=-uvmap.v
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
for i=0,map.icc[0]-1 do begin
	arg=2*pi_circle*(u*map.xcc[i]+v*map.ycc[i])*MAS/RAD
	cvmap=cvmap+map.fcc[i]*complex(cos(arg),sin(arg))
endfor
;
cvmap=reform(cvmap,imsize)/total(map.fcc)
;
; Apply taper (equals convolution resulting image with Gaussian beam)
u=reform(u,201,201)
v=reform(v,201,201)
r=sqrt(u^2+v^2)/uv_cell
y=(-(r/uv_taper)^2)
result=machar() & tiny=result.xmin*1e6 & toosmall=alog(tiny)
index=where(y lt toosmall,count)
if count gt 0 then y[index]=1
cvmap=cvmap*exp(y)
if count gt 0 then cvmap[index]=0
; tvscl,float(cvmap)
image=shift(float(fft(cvmap,/inverse,/center)),imsize[0]/2,imsize[1]/2)
;
return,image
;
end
;-------------------------------------------------------------------------------
function edstrip,model,lamda,u,v
;
; Return strip brightness distribution for elliptical components.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180/pi_circle
;
n=299	; Number of strip points
;
case abs(model.type) of 
	 9:	begin
		amajor=model.diameter/2
		bminor=amajor*model.ratio
		t=atan(u,v)-model.pa/RAD
		a=(amajor^2*cos(t)^2+bminor^2*sin(t)^2)/(amajor^2*bminor^2)
		b=(2*sin(t)*cos(t)*(bminor^2-amajor^2))/(amajor^2*bminor^2)
		c=(amajor^2*sin(t)^2+bminor^2*cos(t)^2)/(amajor^2*bminor^2)
		y=(findgen(n)-n/2)/n*2*sqrt(amajor^2*cos(t)^2+bminor^2*sin(t)^2)
		b=sqrt((b*y)^2-4*a*(c*y^2-1))
		strip=alloc_strip(n,y,b,t)
		end
endcase
;
return,strip
;
end
;-------------------------------------------------------------------------------
function gdmap,model,lamda,m_maps,ndim=ndim
;
; Obsolete
;
; Setup a call to the 1.3 Roche code to get a gravity darkened
; brightness distribution. Returns a map.
; lamda [m]
;
; This function will store equally sized maps in case
; one is called for with parameters already computed
; before.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common LocalDeane,rochemaps,n_maps,n_dim,rocheparms
;
RAD=180/pi_circle
;
if n_elements(n_dim) eq 0 then n_dim=0
if n_elements(ndim) eq 0 then ndim=128L else ndim=long(ndim)
if ndim ne n_dim then n_maps=0
n_dim=ndim
;
n_maps=0
;
parms=setrocheparms13(model,lamda,0,0)
parmstring=strjoin(string(parms))
;
; If we have one or more maps stored, check whether the one is here
if n_elements(n_maps) eq 0 then n_maps=0
if n_maps gt 0 then begin
	index=where(parmstring eq rocheparms,count)
	if count eq 1 then return,rochemaps[index[0]]
endif
;
rpb=0.0
;
; x=RA runs from -ndim/2 to ndim/2-1
x=float((lindgen(ndim*ndim) mod ndim) - (ndim/2))
x=-x	; RA positive towards East
; y=Dec runs from -ndim/2 to ndim/2-1
y=float((lindgen(ndim*ndim) / ndim) - (ndim/2))
f=reform(fltarr(ndim,ndim),ndim*ndim)
xs=0.0
is=0L
;
status=linknload(!external_lib,'peterson',parms,ndim,x,y,f,xs,is,rpb)
x=reform(x,ndim,ndim)
y=reform(y,ndim,ndim)
f=reform(f,ndim,ndim)
save,x,y,f
;
f=reform(transpose(reform(f,ndim,ndim)),ndim*ndim)
;
; Pixel scale in units of RpB
pfactor=xs/abs(ndim/2-is)
;
; Scale in mas/RpB
rfactor=(rpb/parms[5])*(model.diameter/2)
;
; PA is here the position angle of the major axis, from N over E
angle=(-parms[8]+90)/RAD
rx=(-y*sin(angle)+x*cos(angle))*pfactor*rfactor
ry=(+x*sin(angle)+y*cos(angle))*pfactor*rfactor
map=alloc_map(ndim*ndim,intarr(ndim*ndim),rx,ry,f)
;
; Store the map
if n_elements(m_maps) eq 0 then m_maps=2
if n_maps eq 0 then begin
	rochemaps=replicate(map,m_maps)
	rocheparms=strarr(m_maps)
endif
if n_maps ge m_maps then n_maps=0
rochemaps[n_maps]=map
rocheparms[n_maps]=parmstring
n_maps=n_maps+1
;
return,map
;
end
;-------------------------------------------------------------------------------
function chmap,model,lamda,threshold=threshold
;
; Given an image cube, interpolate image at wavelength lamda [m].
; Returns a map. Sets all pixels to zero which are less than a threshold.
;
; Map coordinates are in mas.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
RAD=180/pi_circle
if n_elements(threshold) eq 0 then threshold=0.001
;
wavelengths=channel_wavelengths
;
if total(channel_images) eq 0 then retall
;
IF n_elements(channel_wavelengths) EQ 1 THEN BEGIN
;
image=channel_images
;
ENDIF ELSE BEGIN
;
i=where(wavelengths ge lamda,count)
if count eq 0 then begin
	image=channel_images[*,*,n_elements(wavelengths)-1]
endif else if count eq n_elements(wavelengths) then begin
	image=channel_images[*,*,0]
endif else begin
	dll=(lamda-wavelengths[i[0]-1])/(wavelengths[i[0]]-wavelengths[i[0]-1])
	dlh=(wavelengths[i[0]]-lamda)  /(wavelengths[i[0]]-wavelengths[i[0]-1])
;
; 	Linearily interpolate images to wavelength desired
	image=channel_images[*,*,i[0]-1]*dlh+channel_images[*,*,i[0]]*dll
endelse
;
ENDELSE
;
; Remove DC component
; image=image-sockel(image)
;
r=size(image)
imsize=r[1:2]
x=(lindgen(imsize[0]*imsize[1]) mod imsize[0]) - (imsize[0]/2)
y=(lindgen(imsize[0]*imsize[1]) / imsize[0]) - (imsize[1]/2)
x=-x*images_cellsize   ; RA increases to the left (East)
y=+y*images_cellsize
;
if max(image) ne 0 $
	then index=where(image/max(image) ge threshold,count) $
	else index=where(image eq 0,count)
f=reform(image[index],count)
x=x[index]
y=y[index]
icc=lindgen(count)
ccc=icc	; ccc is not used here
map=alloc_map(icc,ccc,x,y,f,imsize[0],imsize[1])
;
return,map
;
end
;-------------------------------------------------------------------------------
function modelpos,epoch,vellist
;
; Compute the locations of stars in a multiple system by summing up
; positional offsets generated by the binary pairs. Each binary is centered
; on the location of its center of mass. The positions are right ascension
; and declination offsets.
;
; Also return the velocities in RA and Dec [mas/day].
;
; Note: epoch = JD - 2440000 !
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
num_epoch=n_elements(epoch)
;
o_parms=dblarr(8)
poslist=dblarr(num_epoch,num_star(),2)
poslist_h=poslist
poslist_l=poslist
vellist=dblarr(num_epoch,num_star(),2)
complist=strarr(num_star())
;
epoch_l=epoch-min(binary_model.period)/1000
epoch_h=epoch+min(binary_model.period)/1000
;
FOR i=0,num_binary()-1 DO BEGIN
;
if componentparse(binary_model[i].component,comp_1,comp_2) eq -1 then return,-1
;
; Compute the component masses (=sum of stellar masses)
masses=componentmass(comp_1,comp_2,num_body,index_comp)
;
case binary_model[i].method of
;
;	Normal detached binary, positions derived from orbit
      1:begin
	o_parms[0]=binary_model[i].semimajoraxis
	o_parms[1]=binary_model[i].eccentricity
	o_parms[2]=binary_model[i].inclination/RAD
	o_parms[3]=binary_model[i].periastron/RAD
	o_parms[4]=binary_model[i].apsidalmotion/RAD
	o_parms[5]=binary_model[i].ascendingnode/RAD
	o_parms[6]=binary_model[i].period
	o_parms[7]=binary_model[i].epoch
	xy=true2app(epoch,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	xy_h=true2app(epoch_h,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	xy_l=true2app(epoch_l,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	end
;
;	Normal detached binary, position from model values (rho,theta)
      2:begin
	xy=dblarr(n_elements(epoch),2)
	xy[*,0]=binary_model[i].rho*sin(binary_model[i].theta/RAD)
	xy[*,1]=binary_model[i].rho*cos(binary_model[i].theta/RAD)
	xy_l=xy
	xy_h=xy
	end
;
;	Here this is the same as method 2, i.e. ignore orbit for astrometry
      3:begin
	xy=dblarr(n_elements(epoch),2)
	xy[*,0]=binary_model[i].rho*sin(binary_model[i].theta/RAD)
	xy[*,1]=binary_model[i].rho*cos(binary_model[i].theta/RAD)
	xy_l=xy
	xy_h=xy
	end
;
;	Here this is the same as method 1
      4:begin
	o_parms[0]=binary_model[i].semimajoraxis
	o_parms[1]=binary_model[i].eccentricity
	o_parms[2]=binary_model[i].inclination/RAD
	o_parms[3]=binary_model[i].periastron/RAD
	o_parms[4]=binary_model[i].apsidalmotion/RAD
	o_parms[5]=binary_model[i].ascendingnode/RAD
	o_parms[6]=binary_model[i].period
	o_parms[7]=binary_model[i].epoch
	xy=true2app(epoch,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	xy_h=true2app(epoch_h,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	xy_l=true2app(epoch_l,o_parms,rho,theta,gen_model.ra,gen_model.dec)
	end
;
endcase
;
factor=[-masses[1]/total(masses),+masses[0]/total(masses)]
for k=0,1 do begin
	for j=0,num_body[k]-1 do begin
	    poslist[*,index_comp[j,k],*]=poslist[*,index_comp[j,k],*] $
					+factor[k]*xy
	endfor
endfor
;
; Accumulate positions for epoch_h, epoch_l
for k=0,1 do begin
	for j=0,num_body[k]-1 do begin
	    poslist_h[*,index_comp[j,k],*]=poslist_h[*,index_comp[j,k],*] $
					+factor[k]*xy_h
	endfor
endfor
for k=0,1 do begin
	for j=0,num_body[k]-1 do begin
	    poslist_l[*,index_comp[j,k],*]=poslist_l[*,index_comp[j,k],*] $
					+factor[k]*xy_l
	endfor
endfor
;
ENDFOR
;
; Compute derivatives
for k=0,1 do begin
for j=0,num_star()-1 do begin
	if epoch_h[0]-epoch_l[0] gt 0 then $
	vellist[*,j,k]=(poslist_h[*,j,k]-poslist_l[*,j,k])/(epoch_h-epoch_l)
endfor
endfor
;
return,poslist
;
end
;-------------------------------------------------------------------------------
function modelvel,epochs,components
;
; Compute radial velocity of stars in a multiple system by adding up all the
; contributions from the binary pairs specified in components. If they are
; not passed, then all components are added, yielding the total velocities
; for each star. If, on the other hand, only the top binary component is
; passed, then the velocity of any star is just its motion around the
; common center of mass in the top binary component.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
mas2rad=pi_circle/(180L*3600L*1000L)
RAD=180/pi_circle
;
num_epoch=n_elements(epochs)
if n_elements(components) eq 0 then components=binary_model.component
;
o_parms=dblarr(8)
vellist=dblarr(num_epoch,num_star())+gen_model.rv
complist=strarr(num_star())
;
FOR ic=0,n_elements(components)-1 DO BEGIN
;
i=where(binary_model.component eq components[ic]) & i=i[0]
if i eq -1 then return,-1
if componentparse(binary_model[i].component,comp_1,comp_2) eq -1 then return,-1
;
; Compute the component masses (=sum of stellar masses)
masses=componentmass(comp_1,comp_2,num_body,index_comp)
;
o_parms[0]=binary_model[i].eccentricity
o_parms[1]=binary_model[i].inclination/RAD
o_parms[2]=binary_model[i].periastron/RAD
o_parms[3]=binary_model[i].apsidalmotion/RAD
o_parms[4]=binary_model[i].period
o_parms[5]=binary_model[i].epoch
o_parms[6]=masses[0]
o_parms[7]=masses[1]
;
if binary_model[i].method eq 4 then begin
	iband=binary_model[i].wdband	; save wdband
	binary_model[i].wdband=4	; RV measurements are taken in Y band
	binary_model[i].wdband=25	; RV measurements are taken in Hp-band
	binary_model[i].wdband=7	; RV measurements are taken in V-band
	binary_model[i].wdband=6	; RV measurements are taken in B-band
	v=lcmap(binary_model[i],epochs,2)
	if v[0] eq -1 then return,0
	binary_model[i].wdband=iband	; restore wdband
;	lamda=0.55e-6			; RV measurements are taken in V-band
;	v=wdrav(binary_model(i),epochs,lamda)-gen_model.rv
endif else begin
	v=true2vel(epochs,o_parms)
endelse
;
for k=0,1 do begin
	for j=0,num_body[k]-1 do begin
	    vellist[*,index_comp[j,k]]=vellist[*,index_comp[j,k]]+v[*,k]
	endfor
endfor
;
ENDFOR
;
return,vellist
;
end
;-------------------------------------------------------------------------------
function binarypos,epoch,component,lamda=lamda,abs=abs
;
; Return separation and position angle for the specified binary
; component. Return absolute positions for each component if abs=1.
; Assume visual band is the requested wavelength if not specified.
; Epoch is the *full* Julian day number; if not specified, use
; today's date and the top level binary component.
;
; Please note: uses !cop system variable to select center-of-light or
; center-of-mass (use latter when computing nominal positions for *.psn
; model files! Otherwise, the center of light correction is applied twice!)
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(binary_model) eq 0 then begin
	print,'***Error(BINARYPOS): no model!'
	return,-1
endif
;
if n_elements(epoch) eq 0 then begin
	print,"Using today's date..."
        parseidldate,systime(),y,m,d
	epoch=julian(y,m,d)
endif
mjd=epoch-2440000L
;
if n_elements(component) eq 0 then component=topbincomp() $
			      else component=strupcase(component)
;
; Assume visual (speckle) band if not specified
if n_elements(lamda) eq 0 then lamda=550e-9
;
; Assume relative position, not absolute
if n_elements(abs) eq 0 then abs=0
;
rad=180/pi_circle
;
; Compute list of positions relative to center-of-mass
poslist=modelpos(mjd)
;
; Compute weights from masses and fluxes
if componentparse(component,c1,c2) ne 0 then return,-1
c1=nameparse(c1,'') & nc1=n_elements(c1)
c2=nameparse(c2,'') & nc2=n_elements(c2)
index1=intarr(nc1)
index2=intarr(nc2)
flux1=fltarr(nc1)
flux2=fltarr(nc2)
mass1=fltarr(nc1)
mass2=fltarr(nc2)
for i=0,nc1-1 do begin
	index1[i]=where(star_model.component eq c1[i])
	flux1[i]=stellarfluxes(star_model[index1[i]],lamda) $
		*modelfluxes(star_model[index1[i]],lamda)
	mass1[i]=star_model[index1[i]].mass
endfor
for i=0,nc2-1 do begin
	index2[i]=where(star_model.component eq c2[i])
	flux2[i]=stellarfluxes(star_model[index2[i]],lamda) $
		*modelfluxes(star_model[index2[i]],lamda)
	mass2[i]=star_model[index2[i]].mass
endfor
;
!cop=strupcase(!cop)
;
if !cop eq 'COM' then begin
;	Center of mass
	weight1=mass1
	weight2=mass2
endif else if !cop eq 'COL' then begin
;	Center of light
	weight1=flux1
	weight2=flux2
endif else if !cop eq 'COP' then begin
;	Primary component
	weight1=fltarr(nc1) & weight1[0]=1
	weight2=fltarr(nc1) & weight2[0]=1
endif
x1=total(poslist[0,index1,0]*weight1)/total(weight1)
x2=total(poslist[0,index2,0]*weight2)/total(weight2)
y1=total(poslist[0,index1,1]*weight1)/total(weight1)
y2=total(poslist[0,index2,1]*weight2)/total(weight2)
if abs then begin
	x=[x1,x2]
	y=[y1,y2]
endif else begin
	x=x2-x1
	y=y2-y1
endelse
rho=sqrt(x^2+y^2)
theta=atan(x,y)*rad
index=where(theta lt 0,count)
if count gt 0 then theta[index]=theta[index]+360
;
return,[[rho],[theta]]
;
end
;-------------------------------------------------------------------------------
function binaryvel,epoch,component
;
; Epoch is the full Julian day number.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(binary_model) eq 0 then begin
	print,'***Error(BINARYVEL): no model!'
	return,-1
endif
;
if n_elements(epoch) eq 0 then begin
        parseidldate,systime(),y,m,d
	epoch=julian(y,m,d)
endif
;
vels=modelvel(epoch-2440000.d0)
if n_elements(component) eq 0 then component='A' $
			      else component=strupcase(component)
;
return,vels[where(star_model.component eq component)]
;
end
;-------------------------------------------------------------------------------
function apodize_f,radius,lamda
;
; Should be called when computing visibilities from maps in order to exclude
; emission outside the field of view. Takes radius [mas] and lamda[m],
; and uses the telescope diameter [m] stored in GenConfig to return
; apodization factor.
;
; Either radius or lamda can be arrays, but not both.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if genconfig.diameter[0] eq 0 then return,1.0
;
rad2mas=180l*3600l*1000l/!pi
; Compute Airy's disk
airy=(lamda/genconfig.diameter[0])*rad2mas
return,exp(-(2*radius/airy)^2)	; Gaussian approximation (no sidelobes)
return,sinc(radius/airy)^2	; Airy pattern itself
;
end
;-------------------------------------------------------------------------------
function apodize_m,r0,am
;
forward_function componentflux
;
; For multiple stellar systems, compute photocenter convolved with telescope
; PSF and return magnitude differences corresponding to the apodisation. 
;
; r0 is the zenithal seeing at 500 nm (Fried's parameter), am the airmass.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
rad2mas=180l*3600l*1000l/!pi
lamda=gen_model.wavelengths*1e-6
;
if num_binary() eq 0 then return,0
;
; r0 is the zenith seeing (Fried's parameter) at 550 nm in arcseconds
if n_elements(r0) eq 0 then r0=1.0 else r0=float(r0)
;
; Compute component positions xy relative to center of mass
parsedate,date,y,m,d
epoch=julian(y,m,d)-2440000.d0
xy=reform(modelpos(epoch))
;
n=n_elements(star_model[where(star_model.type gt 0)])
n_stars=n
m=n_elements(lamda)
fl=fltarr(n,m)
dm=fltarr(n,m)
;
; Make backup copy of star_model
star_model_0=star_model
;
; Iterate
teff=star_model.teff
index=where(star_model.teff eq -5555,count)
if count ge 1 then star_model[index].teff=0
index=where(star_model.type eq 15,count)
if count ge 1 then star_model[index].teff=-star_model[index].teff
for l=0,5 do begin
;
; 	Weigh xy with component fluxes
	for i=0,n-1 do $
		fl[i,*]=componentflux(star_model[i].component,epoch,lamda)
	tfl=total(fl,1)
	xyfl=fltarr(n,m,2)
	for j=0,m-1 do begin
	for k=0,1 do begin
		xyfl[*,j,k]=(xy[*,k]*fl[*,j])/tfl[j]
	endfor
	endfor
;	Coordinates of photometric center (for each wavelength)
	pcl=total(xyfl,1)
;	White light photometric center (tracking is common for all wavel.)
	pc=total(pcl,1)/m
;	Shift xy center of mass to photometric center
	xyl=fltarr(n,m,2)
	for j=0,m-1 do begin
	for k=0,1 do begin
;		xyl(*,j,k)=xy(*,k)-pcl(j,k)
		xyl[*,j,k]=xy[*,k]-pc[k]
	endfor
	endfor
;	Compute distances of components from photometric center
	radius=sqrt(total(xyl^2,3))
; 	Compute PSF as a function of wavelength (first approximation)
	airy=(lamda/genconfig.diameter[0])*rad2mas
	psf=sqrt(((r0*1000)*(lamda/550e-9)^(-0.2))^2+airy^2)
;	Compute IQ (image quality) from atmosphere and telescope TF instead...
;	http://www.eso.org/observing/etc/bin/simu/hawki
	psf=iq_total(lamda,am,abs(r0),genconfig.diameter[0])*1000	; [mas]
;	Compute flux corrections
	for i=0,n-1 do dm[i,*]=-2.5*alog10(exp(-(2*radius[i,*]/psf[*])^2))
;	Apply corrections for next iteration
;	for i=0,n_elements(star_model)-1 do $
	for i=0,n_stars-1 do $
	star_model(i).magnitudes(*)=star_model_0(i).magnitudes(*) $
				   +reform(dm(i,*))
;
endfor
;
; Restore input star_model
star_model=star_model_0
;
return,transpose(dm)
;
end
;-------------------------------------------------------------------------------
function stripvis,strip,u,v
;
; Compute the visibility for a strip at a single coordinate u and v[lamda]. 
; strip positions are in mas.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
uvr=sqrt(u^2+v^2)
arg=2*pi_circle*(uvr*strip.x)*MAS/RAD
;
return,total(strip.b*complex(cos(arg),sin(arg)))
;
end
;-------------------------------------------------------------------------------
function mapvis,map,u,v
;
; Compute the visibility for a map at a single coordinate u and v[lamda]. 
; Map coordinates are in mas. map.x corresponds to RA and increases towards
; East, i.e. left.
;
; Obsolete! See next procedure.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
image=reform(map.fcc,map.nx,map.ny)
dx=map.xcc[0]-map.xcc[1] & dx=dx*MAS/RAD
dy=map.ycc[map.nx]-map.ycc[map.nx-1] & dy=dy*MAS/RAD
du=1./(max(map.xcc)*MAS/RAD)
dv=1./(max(map.ycc)*MAS/RAD)
vf=fft(image)
i0=map.nx/2
j0=map.ny/2
vfs=shift(vf,i0,j0)
;
; Transform u,v coordinates into locations
ul=u/du+i0
vl=v/dv+j0
;
return,interpolate(vfs,ul,vl)
;
arg=2*pi_circle*(u*map.xcc+v*map.ycc)*MAS/RAD
;
return,total(map.fcc*complex(cos(arg),sin(arg)))
;
end
;-------------------------------------------------------------------------------
function mapvis,map,u,v
;
; Compute the visibility for a map at a single coordinate u and v[lamda]. 
; Map coordinates are in mas. map.x corresponds to RA and increases towards
; East, i.e. left.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
arg=2*pi_circle*(u*map.xcc+v*map.ycc)*MAS/RAD
;
return,total(map.fcc*complex(cos(arg),sin(arg)))
;
end
;-------------------------------------------------------------------------------
function imgvis,u,v,lamda
;
; Computes visibilities for OYSTER/PEARL "effective temperature" images.
;
; u[lamda],v[lamda],lamda[m]
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
n=n_elements(u)
nl=n_elements(lamda)
;
; nr is the number of non-zero effective temperature regions
ir=where(rt ne 0,nr)
sf=fltarr(nl,nr)
;
for i=0,nr-1 do begin
;	This was the old call for stellar atmospheres
;	ld_coeffs=limblinear(rt(i),rg(i),lamda*1d9,fluxes)
;	Now we call a more general function for SEDs
	fluxes=stellarfluxes( $
		{teff:-rt[i],logg:rg[i], $
		 diameter:1.0,ratio:1.0, $
		 type:13}, $
		 lamda,ld_coeffs)
;	sf(*,i)=fluxes/max(fluxes)	; for Pearl Version 7.06
	sf[*,i]=fluxes
endfor
;
return,mft(cm,complexarr(n),reform(u,n),reform(v,n),(indgen(n) mod nl)+1, $
	rt,sf,/normalize)
;
; Call in Version 7.06
return,mft(cm,complexarr(n),reform(u,n),reform(v,n),rt,sf, $
	(indgen(n) mod nl)+1,/normalize)
;
end
;-------------------------------------------------------------------------------
function provis_f1,x
;
common LocalProvis,r,a
;
return,poly(x,r)
;
end
;-------------------------------------------------------------------------------
function provis_f2,x
;
common LocalProvis,r,a
;
return,beselj(a*sqrt(1-x^2),0)*provis_f1(x)
;
end
;-------------------------------------------------------------------------------
function provis,profile,mu,arg
;
; Compute the visibility of a stellar atmosphere specified with a profile
; as a function of mu=cos(theta), mu=1 at the center (see Wittkowski 2003):
; F_LD(lamda)=Int_[0,1] I_lamda(mu) J0(2pi*r_LD*B/lamda*sqrt(1-mu^2)) mu dmu
; To normalize, divide by Int_[0,1] I_lamda(mu) mu dmu.
; sqrt(1-mu^2)=sin(theta), the projected radius
;
common LocalProvis,r,a
;
f1=(mu*profile)/profile[n_elements(mu)-1]
f2=beselj(arg*sqrt(1-mu^2),0)*f1
;
return,int_tabulated(mu,f2)/int_tabulated(mu,f1)
;
; Obsolete
a=arg
r=poly_fit(mu,f1,3)
return,qromb('provis_f2',0,1,/double)/qromb('provis_f1',0,1,/double)
;
end
;-------------------------------------------------------------------------------
function rochevis,model,lamda,u,v,fluxes,init=init
;
; Prepare a call to rochevis, which computes visibilities for a Roche sphere.
; Lambda[m], u, and v should be vectors.
;
common LimbBase,limb_data
common LocalRocheVis,LDinit
common AuxData,parallaxes,k1,k2,vsini
;
if n_elements(limb_data) eq 0 then $
restore,!atmospheres_dir+'vanhamme/limbdata.xdr'
;
rocheparms=setrocheparms(model,lamda,u,v)
;
lam=float(lamda)
nl=n_elements(lam)
vr=fltarr(nl)
vi=vr
fluxes=fltarr(nl)
um=float(u)*lam		; convert u,v coordinates in meters
vm=float(v)*lam
vsi=0.0
;
lam=lam*1e9		; convert to nm
;
; Prepare LD data
if n_elements(LDinit) eq 0 then LDinit=1
if LDinit ne 0 then begin
	case model.model of
		'Log' :LDinit=2l
		'Sqrt':LDinit=3l
	 	else  :LDinit=1L
	endcase
endif
if n_elements(init) ne 0 then LDinit=long(init)
if LDinit gt 0 then begin
	afTeff=float(unique(limb_data.t))
	iTeffMax=n_elements(afTeff)
	afLogG=reverse(float(unique(limb_data.g)))
	iLogGMax=n_elements(afLogG)
	afWaves=float(unique(limb_data.w))
	iWavesMax=n_elements(afWaves)
	afLD=fltarr(3,iTeffMax,iLogGMax,iWavesMax)
	for i=0,n_elements(limb_data.m)-1 do begin
		index=where(limb_data.n eq limb_data.m[i])
		iT=where(afTeff eq limb_data.t[i]) & iT=iT[0]
		iG=where(afLogG eq limb_data.g[i]) & iG=iG[0]
		case LDinit of
		1: afLD[0,iT,iG,*]=limb_data.u[index]
		2: begin
		   afLD[0,iT,iG,*]=limb_data.x[index]
		   afLD[1,iT,iG,*]=limb_data.y[index]
		   end
		3: begin
		   afLD[0,iT,iG,*]=limb_data.a[index]
		   afLD[1,iT,iG,*]=limb_data.b[index]
		   end
		endcase
		afLD[2,iT,iG,*]=(limb_data.f[index]/1e0) $
			       /blackbody(afTeff[iT],afWaves/1e9)
	endfor
	aiLogG=lonarr(iTeffMax)
	for i=0,iTeffMax-1 do $
		aiLogG[i]=n_elements(where(limb_data.t eq afTeff[i]))
endif else begin
	afTeff=0.
	iTeffMax=0l
	afLogG=0.
	iLogGMax=0l
	afWaves=0.
	iWavesMax=0l
	afLD=0.
	aiLogG=0l
endelse
;
status=linknload(!external_lib,'peterson',rocheparms,nl,lam,vr,vi,fluxes, $
		um,vm, $
		LDinit,iWavesMax,iTeffMax,iLogGMax,aiLogG, $
		afLD,afLogG,afWaves,afTeff,vsi)
LDinit=0l
;
if n_elements(vsini) ne 0 then vsini.valuem=vsi
;
return,complex(vr,vi)
;
end
;-------------------------------------------------------------------------------
function componentvis,component,time,lamda,fluxes,u,v
;
; Compute the complex visibility of a component. If double, add sub-components.
; Sub-components may themselves be binary, single stars, or both. The mode 
; parameter determines which astrophysical scenario/method is to be used for 
; the visibility computation of the components/sub-components.
;
; Input parameters:
; component: 		A, B, C,..., A-B, AB-C, AB-CD,...
; time [s]: 		dblarr(num_scan)
; lamda grid [m]: 	dblarr(num_wave)
; u,v [lamda]: 	dblarr(num_wave,num_scan)
;
; Output parameter:
; fluxes:		dblarr(numwave)
;
; The model can also provide a file name from which to restore an SED.
; This is then used to interpolate the fluxes.
;
; If only one value for time and wavelength is passed, assume u and v
; are arrays corresponding to an arbitrary coverage.
;
; Note: this function actually returns a correlated flux, as complex number so
; that components can be added with the proper phase term. After addition of all
; correlated fluxes, the visibility is obtained by normalizing with the total
; flux. That is also why this function returns the total flux contribution.
;
; Types renamed with version 6.12 (2 Aug 2007)
; 10->8; 11->9; 12->10; 14->11; 8->12; 16->13; 13->obsolete; 9->14; 17->16
;
; Sign for type introduced with version 8: if negative and the component is
; part of a multiple system, do not include it in the total correlated flux.
;
forward_function componentvis
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common RandomModel,RandomSeed
common AuxData,parallaxes,k1,k2,vsini
common LocalModelVis,comp_fluxes,md5seed
;
; Common block needed to include md5sum of image
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
; Initialize constants
mas2rad=pi_circle/(180L*3600L*1000L)
;
num_scan=n_elements(time)
num_wave=n_elements(lamda)
;
; Get length of component to find out whether component is single or double
len=strlen(component)
;
; Single star...................................................................
if len eq 1 then begin
;
; Find applicable model
index=where(star_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): non-unique component definition!'
	return,-1
endif
model=star_model[index]
;
; Create directory for temporary products
if model.type eq 12 then begin
	tmp_products_dir='oyster.componentvis'
	if not file_test(tmp_products_dir,/directory) then $
		file_mkdir,tmp_products_dir
endif
;
; Allow rotation and stretch of all components: PA > 0 rotates counter-clockwise
; Ratio < 1 compresses E-W size of object
vr=v*cos(model.pa/RAD)+u*sin(model.pa/RAD)
ur=model.ratio*(u*cos(model.pa/RAD)-v*sin(model.pa/RAD))
;
; Get SED; all astrophysical types 12 and up compute their own fluxes
if abs(model.type) le 11 then fluxes=stellarfluxes(model,lamda,ld_coeffs) $
			 else fluxes=fltarr(num_wave)+1
;
; If an SED file (XDR format) has been specified, it overrides
if strlen(model.sed) gt 0 then begin
;	Read spectrum from XDR file: l[m], f
	restore,model.sed
	fb=f
;	Reduce resolution to half (Nyquist!) of that of the lamda array
	if n_elements(lamda) ge 2 then begin
		r=4*fix(abs(lamda[1]-lamda[0])/(l[1]-l[0]))
		if r gt 1 then fb=box(f,r)
	endif
	si=sort(lamda)
	fluxes=spline(l,fb,lamda[si])
	fluxes[si]=fluxes
endif
;
; Apply correction to fluxes specified by magnitudes
fluxes=fluxes*modelfluxes(model,lamda)
;
; Replicate fluxes to same dimensions as (u,v)
flux=dblarr(num_wave,num_scan)
for j=0,num_scan-1 do flux[*,j]=fluxes 
if n_elements(flux) eq 1 then flux=flux[0]
;
; Compute contribution due to a star spot
if model.spot[3] ne 0 then begin
	print,'Warning: spot computation possible only with LCDC2015!'
	spot_amp=0
	spot_flux=0
	spot_phase=0
	spot_model=model
	spot_model.teff=model.spot[0]*model.teff
	spot_model.diameter=model.spot[3]
	spot_fluxes=stellarfluxes(spot_model,lamda) $
		   -fluxes*(spot_model.diameter/model.diameter)^2
	xy=dblarr(2)
	xy[0]=model.spot[1]*sin(model.spot[2]/RAD)*MAS/RAD
	xy[1]=model.spot[1]*cos(model.spot[2]/RAD)*MAS/RAD
	spot_phase=complexarr(num_wave,num_scan)
	if n_elements(spot_phase) eq 1 then begin
		spot_phase=exp(2*pi_circle*i_complex*(u*xy[0]+v*xy[1]))
	endif else begin
		for j=0,num_wave-1 do $
   	    	spot_phase[j,*]=exp(2*pi_circle*i_complex $
				*(u[j,*]*xy[0]+v[j,*]*xy[1]))
	endelse
	UD=dblarr(num_wave,num_scan)
	UD[*,*]=model.spot[3]
	if n_elements(UD) eq 1 then UD=UD[0]
	arg=pi_circle*mas2rad*sqrt(u^2+v^2)*UD
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
	spot_amp=2*beselj(arg,1)/arg
	if count gt 0 then spot_amp[index]=1
	spot_flux=flux
	for j=0,num_scan-1 do spot_flux[*,j]=spot_fluxes 
endif else begin
	spot_amp=0
	spot_flux=0
	spot_phase=0
endelse
;
; Derive component phase from RA Dec offsets
phase=exp(2*pi_circle*i_complex*(u*model.xoff+v*model.yoff)*mas2rad)
;
case abs(model.type) of
;
;	Uncorrelated flux component
      0:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
	endif
;	For fiber-fed BCs, the FOV depends on lambda, so also the flux
;	GRAVITY: the FOV decreases towards the blue end of the spectrum!
	if strpos(genconfig.configid,'GRAVITY') eq 0 then begin
		airy=(lamda/genconfig.diameter[0])/mas2rad
		radius=airy[0]/2; distance from center in field of view [mas]
		fluxes=fluxes*exp(-(2*radius/airy)^2)
	endif
	return,modvis
	end
;
;	Uniform (elliptical) disk independent of wavelength, input diameter UD
      1:begin
	UD=dblarr(num_wave,num_scan)
	UD[*,*]=model.diameter
	if n_elements(UD) eq 1 then UD=UD[0]
	arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
	visamp=2*beselj(arg,1)/arg
	if count gt 0 then visamp[index]=1
;	save,lamda,fluxes,filename='fluxes_comp_1.xdr'
;	save,lamda,spot_fluxes,filename='fluxes_spot_1.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Uniform (elliptical) disk scales with wavelengths, input diameter LD
      2:begin
	UD=dblarr(num_wave,num_scan)
	UD1=sqrt((1-7*ld_coeffs/15)/(1-ld_coeffs/3))*model.diameter
	for j=0,num_wave-1 do UD[j,*]=UD1[j]
	if n_elements(UD) eq 1 then UD=UD[0]
	arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
	visamp=2*beselj(arg,1)/arg
	if count gt 0 then visamp[index]=1
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, linear law, input diameter LD
;	From: R. Hanbury Brown et al. 1974 MNRAS 167, 475
      3:begin
	alpha=dblarr(num_wave,num_scan)
	beeta=dblarr(num_wave,num_scan)
	for i=0,num_scan-1 do begin
		alpha[*,i]=1-ld_coeffs
		beeta[*,i]=ld_coeffs
	endfor
	if n_elements(alpha) eq 1 then alpha=alpha[0]
	if n_elements(beeta) eq 1 then beeta=beeta[0]
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
;       visamp=(alpha*beselj(arg,1)/arg+beeta*sqrt(pi_circle/2)* $
;               sqrt(2/(pi_circle*arg))*(sin(arg)/arg-cos(arg))/ $
;               sqrt(arg*arg*arg))/(alpha/2+beeta/3)
;	Use simplified form
	visamp=(alpha*beselj(arg,1)/arg+beeta*(sin(arg)/arg-cos(arg))/arg^2) $
	       /(alpha/2+beeta/3)

	if count gt 0 then visamp[index]=1
;	save,lamda,fluxes,filename='fluxes_comp_3.xdr'
;	save,lamda,spot_fluxes,filename='fluxes_spot_3.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Limb-darkened disk, logarithmic law, input diameter LD
      4:begin
	modvis=complexarr(num_wave,num_scan)
	maps=limbmaps(model,lamda*1d9)
	for j=0,num_wave-1 do begin
		fcc_total=total(maps[j].fcc)
		for i=0,num_scan-1 do begin
			modvis[j,i]=mapvis(maps[j],ur[j,i],vr[j,i])/fcc_total
		endfor
	endfor
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
;	Limb-darkened disk, linear law, input diameter LD, from Kurucz
;	Note: type number hard coded in kurugrid (limb.pro)
      5:begin
	alpha=dblarr(num_wave,num_scan)
	beeta=dblarr(num_wave,num_scan)
	for i=0,num_scan-1 do begin
		alpha[*,i]=1-ld_coeffs
		beeta[*,i]=ld_coeffs
	endfor
	if n_elements(alpha) eq 1 then alpha=alpha[0]
	if n_elements(beeta) eq 1 then beeta=beeta[0]
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	visamp=(alpha*bessjy8(arg,1)/arg $
	        +beeta*sqrt(pi_circle/2)*bessjy8(arg,1.5)/arg^1.5) $
	       /(alpha/2+beeta/3)
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, Hestroffer law, input diameter LD, from Kurucz
;	Note: type number hard coded in kurugrid (limb.pro)
      6:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	arg_nu=1+(ld_coeffs/2)
	gaama=dblarr(num_wave,num_scan)
	bessl=dblarr(num_wave,num_scan)
	delta=dblarr(num_wave,num_scan)
	for j=0,num_wave-1 do begin
		gaama[j,*]=gamma(arg_nu[j]+1)
		bessl[j,*]=bessjy8(arg[j,*],arg_nu[j])
		delta[j,*]=(arg[j,*]/2)^arg_nu[j]
	endfor
	if n_elements(gaama) eq 1 then gaama=gaama[0]
	if n_elements(bessl) eq 1 then bessl=bessl[0]
	if n_elements(delta) eq 1 then delta=delta[0]
	visamp=gaama*(bessl/delta)
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, visibility from stellar atmosphere profile
;	Kurucz models
      7:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	profiles=kurufluxes(model,lamda,mu)
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		visamp=fltarr(num_uv)
		for i=0L,num_uv-1 do begin
			visamp[i]=provis(profiles,mu,arg[i])
		endfor
	endif else begin
		for j=0,num_wave-1 do begin
		for i=0,num_scan-1 do begin
			visamp[j,i]=provis(profiles[*,j],mu,arg[j,i])
		endfor
		endfor
	endelse
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, visibility from stellar atmosphere profile
;	Models from Jason Aufdenberg
      8:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	profiles=jasonfluxes(model,lamda,mu)
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		visamp=fltarr(num_uv)
		for i=0L,num_uv-1 do begin
			visamp[i]=provis(profiles,mu,arg[i])
		endfor
	endif else begin
		for j=0,num_wave-1 do begin
		for i=0,num_scan-1 do begin
			visamp[j,i]=provis(profiles[*,j],mu,arg[j,i])
		endfor
		endfor
	endelse
	return,complex(visamp)*flux+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Elliptical uniform disk
;	This is a numerical version of the analytical code for mode=1
      9:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		modvis=complexarr(num_uv)
		for i=0L,num_uv-1 do begin
			strip=edstrip(model,lamda,ur[i],vr[i])
			fcc_total=total(strip.b)
			modvis[i]=stripvis(strip,ur[i],vr[i])/fcc_total
		endfor
	endif else begin
		for i=0,num_scan-1 do begin
		for j=0,num_wave-1 do begin
			strip=edstrip(model,lamda[j],ur[j,i],vr[j,i])
			fcc_total=total(strip.b)
			modvis[j,i]=stripvis(strip,ur[j,i],vr[j,i])/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase
        end
;
;	Gaussian elliptical disk
;	Reference: "difmap" software by M. Shepherd.
     10:begin
	GD=dblarr(num_wave,num_scan)
	GD[*,*]=model.diameter
	if n_elements(GD) eq 1 then GD=GD[0]
	uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
	index=where(uvr gt 5,count)
	if count gt 0 then uvr[index]=5
;	visamp=exp(-0.3606737602*uvr*uvr)
	visamp=exp(-(1./(4.*alog(2)))*uvr*uvr)
	if count gt 0 then visamp[index]=0
; 	save,lamda,fluxes,filename='fluxes_comp_10.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Pearson disk (FFT of which is exponential)
     11:begin
	modvis=complexarr(num_wave,num_scan)
;
;	Create image with Pearson radial profile
	num=41
	map=indgen(num*num)
	xcc=float((map mod num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
	ycc=float((map  /  num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
;	Make clumpy
	nc=20
	cr=fltarr(nc)+0.1	; clump radii
	xc=randomu(RandomSeed,nc)*2-1
	yc=randomu(RandomSeed,nc)*2-1
	xc=[xc,0]
	yc=[yc,0]
	cr=[cr,0.1]	; add central clump
	nc=nc+1
	index=intarr(n_elements(map))
	for k=0,nc-1 do begin
		jndex=where(sqrt((xcc-xc[k])^2+(ycc-yc[k])^2) lt cr[k])
		index[jndex]=1
	endfor
	xcc=xcc[where(index eq 1)]
	ycc=ycc[where(index eq 1)]
; 	Compute radius squared
	r=sqrt(xcc^2+ycc^2)
; 	Use only circular region with r<1
	index=where(r lt 1)
	xcc=xcc[index]
	ycc=ycc[index]
	r=r[index]
	n=n_elements(r)
;	Create array of to hold maps
	map=alloc_map(n,lonarr(n),fltarr(n),fltarr(n),fltarr(n))
	maps=replicate(map,num_wave)
;	Pearson width = 1.5 (1 = Lorentzian, inf. = Gaussian)
	a=[1.0,1.5,20.0]
	for j=0,num_wave-1 do begin
		fcc=fluxes[j]*pearson_funct(r*a[2]*10,a)
		maps[j].xcc=xcc*model.diameter/2
		maps[j].ycc=ycc*model.diameter/2
		maps[j].fcc=fcc
	endfor
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
		fcc_total=total(maps[0].fcc)
                for i=0L,num_uv-1 do begin
		modvis[i]=mapvis(maps[0],ur[i],vr[i])/fcc_total
		endfor
;	For model
	endif else begin 
		for j=0,num_wave-1 do begin
		fcc_total=total(maps[j].fcc)
		for i=0,num_scan-1 do begin
		modvis[j,i]=mapvis(maps[j],ur[j,i],vr[j,i])/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
; 	Pearson disk (FFT of which is exponential)
;	(Place holder for new model type, still to be defined)
    111:begin
	modvis=complexarr(num_wave,num_scan)
;
;	Create image with Pearson radial profile
	num=41
	map=indgen(num*num)
	xcc=float((map mod num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
	ycc=float((map  /  num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
;	Make clumpy
	nc=20
	cr=fltarr(nc)+0.1	; clump radii
	if RandomSeed[0] eq 0 then do_plot=1 else do_plot=0
	xc=randomu(RandomSeed,nc)*2-1
	yc=randomu(RandomSeed,nc)*2-1
	xc=[xc,0]
	yc=[yc,0]
	cr=[cr,0.1]	; add central clump
	nc=nc+1
	index=intarr(n_elements(map))
	for k=0,nc-1 do begin
		jndex=where(sqrt((xcc-xc[k])^2+(ycc-yc[k])^2) lt cr[k])
		index[jndex]=1
	endfor
	xcc=xcc[where(index eq 1)]
	ycc=ycc[where(index eq 1)]
; 	Compute radius squared
	r=sqrt(xcc^2+ycc^2)
; 	Use only circular region with r<1
	index=where(r lt 1)
	xcc=xcc[index]
	ycc=ycc[index]
	r=r[index]
	n=n_elements(r)
	if do_plot then plot,xcc,ycc,psym=3
;	Create array of to hold maps
	map=alloc_map(n,lonarr(n),fltarr(n),fltarr(n),fltarr(n))
	maps=replicate(map,num_wave)
;	Pearson width = 1.5 (1 = Lorentzian, inf. = Gaussian)
	a=[1.0,1.5,20.0]
	for j=0,num_wave-1 do begin
		fcc=fluxes[j]*pearson_funct(r*a[2]*10,a)
		maps[j].xcc=xcc*model.diameter/2
		maps[j].ycc=ycc*model.diameter/2
		maps[j].fcc=fcc
	endfor
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
		fcc_total=total(maps[0].fcc)
                for i=0L,num_uv-1 do begin
		modvis[i]=mapvis(maps[0],ur[i],vr[i])/fcc_total
		endfor
;	For model
	endif else begin 
		for j=0,num_wave-1 do begin
		fcc_total=total(maps[j].fcc)
		for i=0,num_scan-1 do begin
		modvis[j,i]=mapvis(maps[j],ur[j,i],vr[j,i])/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
; 	Image cube (third axis: wavelength)
     12:begin
;	If diameter is specified, it is the clean beam
	if model.diameter gt 0 then begin
		ur=u
		vr=v
	endif
	cellsize=model.cellsize
	if strlen(model.model) gt 0 then $
		if model.cellsize ne 0 then $
			readimages,model.model,/quiet,cellsize=cellsize else $
			readimages,model.model,/quiet
;	Obtain md5sum of input for visibility computation
	md5file=tmp_products_dir+'/'+specname(model.model)+'_' $
		+strtrim(string(randomu(md5seed)*6),2)+'_md5sum.dat'
	openw,unit,md5file,/get_lun
	writeu,unit,ur,vr,lamda,model, $
		channel_images,channel_wavelengths,images_cellsize
	free_lun,unit
	spawn,'md5sum '+md5file,md5sum
	spawn,'rm -f '+md5file
	words=nameparse(md5sum)
	xdrfile=file_search(tmp_products_dir+'/vis_comp_12.'+words[0]+'.xdr')
;
	IF strlen(xdrfile) ne 0 THEN BEGIN
;		print,'Restoring previously saved vis_comp_12 data...'
		restore,xdrfile
	ENDIF ELSE BEGIN
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		map=chmap(model,lamda)
		radius=sqrt(map.xcc^2+map.ycc^2)
		map.fcc=map.fcc*apodize_f(radius,lamda[0]) $
			       *modelfluxes(model,lamda[0])
		num_uv=n_elements(u)
		modvis=complexarr(num_uv)
		fcc_total=total(map.fcc)
		for i=0L,num_uv-1 do begin
			modvis[i]=mapvis(map,ur[i],vr[i])/fcc_total
		endfor
		if strlen(model.sed) eq 0 then begin
			fluxes=fcc_total*modelfluxes(model,lamda)
			flux=fluxes
		endif
	endif else begin
;		For model
		for i=0,num_scan-1 do begin
		for j=0,num_wave-1 do begin
			map=chmap(model,lamda[j])
			radius=sqrt(map.xcc^2+map.ycc^2)
			map.fcc=map.fcc*apodize_f(radius,lamda[j]) $
				       *modelfluxes(model,lamda[j])
			fcc_total=total(map.fcc)
			if fcc_total gt 0 then $
			modvis[j,i]=mapvis(map,ur[j,i],vr[j,i])/fcc_total
			if strlen(model.sed) eq 0 then begin
				fluxes[j]=fcc_total*modelfluxes(model,lamda[j])
				flux[j,i]=fluxes[j]
			endif
		endfor
		endfor
	endelse
;	Obtain md5sum for input to this computation
	openw,unit,md5file,/get_lun
	writeu,unit,ur,vr,lamda,model, $
		channel_images,channel_wavelengths,images_cellsize
	free_lun,unit
	spawn,'md5sum '+md5file,md5sum
	words=nameparse(md5sum)
	xdrfile=tmp_products_dir+'/vis_comp_12.'+words[0]+'.xdr'
	save,fluxes,flux,modvis,phase,filename=xdrfile
	spawn,'rm -f '+md5file
	ENDELSE
;
;	Correct for restoring beam effect if diameter is specified
	if model.diameter gt 0 then begin
		GD=dblarr(num_wave,num_scan)
		GD[*,*]=model.diameter
		vr=v*cos(model.pa/RAD)+u*sin(model.pa/RAD)
		ur=model.ratio*(u*cos(model.pa/RAD)-v*sin(model.pa/RAD))
		if n_elements(GD) eq 1 then GD=GD[0]
		uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
		index=where(uvr gt 5,count)
		if count gt 0 then uvr[index]=5
		visamp=exp(-0.3606737602*uvr*uvr)
		if count gt 0 then visamp[index]=0.01
		modvis=modvis/visamp
	endif
;
	return,modvis*flux*phase
        end
;
; 	"Effective temperature" PEARL image
     13:begin
;	If diameter is specified, it is the clean beam
	if model.diameter gt 0 then begin
		ur=u
		vr=v
	endif
	if strlen(model.model) gt 0 then $
		if model.cellsize ne 0 then $
			readimage,model.model,cellsize=cellsize else $
			readimage,model.model
	fluxes=fltarr(num_wave)+1
	flux=dblarr(num_wave,num_scan)+1
	modvis=complexarr(num_wave,num_scan)
	modvis[*]=imgvis(ur,vr,lamda)
;
;	Correct for restoring beam effect
	if model.diameter gt 0 then begin
		GD=dblarr(num_wave,num_scan)
		GD[*,*]=model.diameter
		if n_elements(GD) eq 1 then GD=GD[0]
		uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
		index=where(uvr gt 5,count)
		if count gt 0 then uvr[index]=5
		visamp=exp(-0.3606737602*uvr*uvr)
		if count gt 0 then visamp[index]=0.01
		modvis=modvis/visamp
	endif
;
	return,modvis*phase
        end
;
; 	Rotating star, using Roche code V-1.4
     14:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
		num_uv=n_elements(ur)
		modvis=complexarr(num_uv)
		modvis[*]=rochevis(model,fltarr(num_uv)+lamda,ur,vr,fluxes)
		flux[0]=fluxes[0]
	endif else begin
		for i=0,num_scan-1 do begin
		modvis[*,i]=rochevis(model,lamda,ur[*,i],vr[*,i],fluxes)
		flux[*,i]=fluxes 
		endfor
	endelse
	fluxes=fluxes*modelfluxes(model,lamda)
	flux=flux*modelfluxes(model,lamda)
	return,modvis*flux*phase
        end
;
;	Ring or disk with hole, and temperature profile (Hillenbrand 1992),
;	also used by Malbet et al. (2005, "temperature gradient model").
;	Diameter refers to the inner diameter (2*r_min), width is r_max-r_min. 
;	For the effect of tilt on flux, see Berger & Segransan (2007).
;	T  ~ (radius/r_min)^alpha, alpha should be -3/4.
     15:begin
	num_ring=100
	w_ring=(model.diameter/2)*model.width/num_ring
	r_ring=model.diameter/2+w_ring/2+findgen(num_ring)*w_ring
	alpha=model.alpha
	teff0=model.teff
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
		num_uv=n_elements(u)
		visamp=complexarr(num_uv)
	endif
	fluxes=fltarr(num_wave)
	for k=0,num_ring-1 do begin
		arg=pi_circle*r_ring[k]*2*mas2rad*sqrt(vr^2+ur^2)
		model.teff=-teff0*(r_ring[k]/r_ring[0])^alpha
;		Compute black body fluxes of each ring and sum up
		f=stellarfluxes(model,lamda,ld_coeffs)*2*!pi*r_ring[k]*w_ring $
		 *cos(model.tilt/RAD)*modelfluxes(model,lamda); <-- scaling
		fluxes=fluxes+f
		flux=dblarr(num_wave,num_scan)
		for j=0,num_scan-1 do flux[*,j]=f
		if n_elements(flux) eq 1 then flux=flux[0]
;		Compute visiblity of ringlet
		visamp=visamp+flux*beselj(arg,0)
	endfor
;	save,lamda,fluxes,filename='fluxes_comp_15.xdr'
	model.teff=teff0
	return,complex(visamp)*phase
	end
;
;	Result file (*.i###) from DUSTY or other shell simulation
     16:begin
	arg=pi_circle*2*mas2rad*sqrt(vr^2+ur^2)
;	Read and decode file, last # line before data must contain wavelengths
	readshell,model.model,l_shell,r_shell,f_shell
	nl=n_elements(l_shell)
	nr=n_elements(r_shell)
	r_shell=r_shell*abs(model.diameter/2)	; Scale with diameter/2
;	Extract rows corresponding to central star
;	We assume that these have the highest intensities!
	tf=total(f_shell,1)
	index_s=where(tf/max(tf) gt 0.99,count_s)
	num_ring=nr-count_s
	visamp_shell=fltarr(num_wave,num_scan)
	if n_elements(visamp_shell) eq 1 then begin
		num_uv=n_elements(u)
		visamp_shell=complexarr(num_uv)
	endif
	fluxes=fltarr(num_wave)
	flux=dblarr(num_wave,num_scan)
;
;	V(q)/F(q)/F(0); F(q)=2pi Int(I*J0(2pi*q*x)*x*dx)
	x0=r_shell[count_s:nr-1]
	f0=dblarr(num_wave,nr)
;	Interpolate fluxes onto lamda grid
	for k=0,nr-1 do begin
		index=where(lamda ge l_shell[0] and lamda le l_shell[nl-1])
		f0[index,k]=interpol(f_shell[*,k],l_shell,lamda[index])
;		Extrapolate to smaller lamda as a constant
		index=where(lamda lt l_shell[0],count)
		if count gt 0 then f0[index,k]=f_shell[0,k]
;		Extrapolate to longer lamda as a constant
		index=where(lamda gt l_shell[nl-1],count)
		if count gt 0 then f0[index,k]=f_shell[nl-1,k]
	endfor
;	Integrate rings other than those of the central star
	f=f0[*,count_s:nr-1]
	for j=0,num_wave-1 do begin
		y0=f[j,*]*2*pi_circle*x0 & y0=reform(y0)
;		y0=gsmooth(x0,y0,15)
		y0=apodize_f(x0,lamda[j])*y0*modelfluxes(model,lamda[j])
		maxrad=max(x0-x0[0]) < 1000	; Maximum radius 1"
		x1=findgen(1000)/1000*maxrad+x0[0]
		y1=interpol(y0,x0,x1)
;		y1=spline(x0,y0,x1)
;		fluxes(j)=int_tabulated(x0,y0,/double)
		fluxes[j]=int_tabulated(x1,y1,/double)
		for i=0,num_scan-1 do begin
;			y=beselj(x0*arg(j,i),0)*y0
			y=beselj(x1*arg[j,i],0)*y1
;			visamp_shell(j,i)=int_tabulated(x0,y,/double)
			visamp_shell[j,i]=int_tabulated(x1,y,/double)
			flux[j,i]=fluxes[j]
		endfor
	endfor
;
;	Add back transform of central star
	if model.diameter gt 0 then begin
		UD=dblarr(num_wave,num_scan)
		UD[*,*]=r_shell[count_s]*2
		if n_elements(UD) eq 1 then UD=UD[0]
		arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
		index=where(arg eq 0,count)
		if count gt 0 then arg[index]=1
		visamp_star=2*beselj(arg,1)/arg
		if count gt 0 then visamp_star[index]=1
;		f=total(f_shell(*,0:count_s-1),2)/count_s
;		f=interpol(f,l_shell,lamda)*!pi*(UD(0)/2)^2
		f=f0[*,0]*!pi*(UD[0]/2)^2
		f=f*modelfluxes(model,lamda)
		fluxes=fluxes+f
		flux=dblarr(num_wave,num_scan)
		for j=0,num_scan-1 do flux[*,j]=f
		if n_elements(flux) eq 1 then flux=flux[0]
		return,(complex(visamp_shell)+complex(visamp_star)*flux)*phase
	endif else begin
		return,complex(visamp_shell)*phase
	endelse
	end
;
endcase
endif
;
; Double star...................................................................
;
; Find applicable model
index=where(binary_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): no or non-unique component definition!'
	return,-1
endif
model=binary_model[index]
;
parsedate,date,y,m,d
midnight=system_config(systemid,'MIDNIGHT')
jd=(julian(y,m,d)-2440000)+time/86400
jd0=(julian(y,m,d)-2440000)+(time*0+midnight/24)
;
; Do the astrometry (i.e. compute xy) for binary components
case model.method of
;
;	Normal detached binary, positions derived from orbit
      1:begin
	o_parms=dblarr(8)
	o_parms[0]=model.semimajoraxis*MAS/RAD
	o_parms[1]=model.eccentricity
	o_parms[2]=model.inclination/RAD
	o_parms[3]=model.periastron/RAD
	o_parms[4]=model.apsidalmotion/RAD
	o_parms[5]=model.ascendingnode/RAD
	o_parms[6]=model.period
	o_parms[7]=model.epoch
	xy=true2app(jd,o_parms,rho,theta,gen_model.ra,gen_model.dec,z)
	x=xy[*,0] & if n_elements(jd) eq 1 then x=x[0]
	y=xy[*,1] & if n_elements(jd) eq 1 then y=y[0]
	end
;
;	Normal detached binary, position from model
      2:begin
	xy=dblarr(n_elements(time),2)
	xy[*,0]=model.rho*sin(model.theta/RAD)*MAS/RAD
	xy[*,1]=model.rho*cos(model.theta/RAD)*MAS/RAD
	end
;
; 	Normal detached binary, positions from model w/orbital motion
;	Positions calculated for MIDNIGHT need to be derived for actual epochs
      3:begin
	o_parms=dblarr(8)
	o_parms[0]=model.semimajoraxis*MAS/RAD
	o_parms[1]=model.eccentricity
	o_parms[2]=model.inclination/RAD
	o_parms[3]=model.periastron/RAD
	o_parms[4]=model.apsidalmotion/RAD
	o_parms[5]=model.ascendingnode/RAD
	o_parms[6]=model.period
	o_parms[7]=model.epoch
	xyO=true2app(jd, o_parms,rhoO,thetaO,gen_model.ra,gen_model.dec)
	xy0=true2app(jd0,o_parms,rho0,theta0,gen_model.ra,gen_model.dec)
	dxy=xyO-xy0
	xy=dblarr(n_elements(time),2)
	xy[*,0]=model.rho*sin(model.theta/RAD)*MAS/RAD+dxy[*,0]
	xy[*,1]=model.rho*cos(model.theta/RAD)*MAS/RAD+dxy[*,1]
	end
;
; 	Interacting binary, with LC/WD code
      4:begin
	modvis=complexarr(num_wave,num_scan)
	fluxes=fltarr(num_wave)+1
	maps=lcmap(model,jd,5)
	for i=0,num_scan-1 do begin
	fcc_total=total(maps[i].fcc)
	for j=0,num_wave-1 do begin
	modvis[j,i]=mapvis(maps[i],u[j,i],v[j,i]);/fcc_total
	fluxes[j]=fcc_total
	endfor
	endfor
	return,modvis	; Special case: this code returns the visibility
;
;	Old WD code
	modvis=complexarr(num_wave,num_scan)
	fluxes=fltarr(num_wave)+1
	for i=0,num_scan-1 do begin
	for j=0,num_wave-1 do begin
	map=wdmap(model,jd[i],lamda[j])
	fcc_total=total(map.fcc)
	modvis[j,i]=mapvis(map,u[j,i],v[j,i])/fcc_total
	endfor
	endfor
	return,modvis	; Special case: this code returns the visibility
        end
;
endcase
;
; Compute component visibilities (must follow astrometry because of mode 4)
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
modvis_1=componentvis(comp_1,time,lamda,fluxes1,u,v)
modvis_2=componentvis(comp_2,time,lamda,fluxes2,u,v)
;
; Process negative model type: contribution to be ignored
if strlen(comp_1) eq 1 then begin
	index=where(star_model.component eq comp_1)
	comp_fluxes[*,index]=fluxes1
	if star_model[index].type lt 0 then modvis_1=0
endif
if strlen(comp_2) eq 1 then begin
	index=where(star_model.component eq comp_2)
	comp_fluxes[*,index]=fluxes2
	if star_model[index].type lt 0 then modvis_2=0
endif
;
; Compute total flux to be returned
fluxes=fluxes1+fluxes2
;
; Compute masses for center-of-mass referencing
if componentparse(component,comp_1,comp_2) eq -1 then return,-1
masses=componentmass(comp_1,comp_2,num_body,index_comp)
;
; Establish phase center for bandwidth smearing computation
if model.component eq topbincomp() then begin
	!cop=strupcase(!cop)
	if !cop eq 'COM' then begin
;		Common center of mass
		factor1=-(1-masses[0]/total(masses))
		factor2=+(1-masses[1]/total(masses))
	endif else if !cop eq 'COP' then begin
;		Move phase center to most massive component
		factor1=0.
		factor2=1.
	endif else begin
;		Photometric center (COL)
		factor1=-(1-total(fluxes1)/total(fluxes))
		factor2=+(1-total(fluxes2)/total(fluxes))
	endelse
	index=where(star_model.type lt 0,count)
	for i=0,count-1 do fluxes=fluxes-comp_fluxes[*,index[i]]
endif else begin
;	For lower level binary components, reference to center of mass
	factor1=-(1-masses[0]/total(masses))
	factor2=+(1-masses[1]/total(masses))
endelse
;
; Allocate phase offsets and compute
phase1=complexarr(num_wave,num_scan) 
if n_elements(phase1) eq 1 then phase1=complexarr(n_elements(u))
phase2=phase1
;
if num_wave gt 1 then begin
	for j=0,num_wave-1 do begin
	   phase1[j,*]=exp(2*pi_circle*i_complex*factor1 $
		*(u[j,*]*xy[*,0]+v[j,*]*xy[*,1]))
	   phase2[j,*]=exp(2*pi_circle*i_complex*factor2 $
		*(u[j,*]*xy[*,0]+v[j,*]*xy[*,1]))
	endfor
endif else begin
	   phase1[*]=exp(2*pi_circle*i_complex*factor1 $
		*(u[*]*xy[0,0]+v[*]*xy[0,1]))
	   phase2[*]=exp(2*pi_circle*i_complex*factor2 $
		*(u[*]*xy[0,0]+v[*]*xy[0,1]))
endelse
;
return,modvis_1*phase1+modvis_2*phase2
;
end
;-------------------------------------------------------------------------------
function modelvis,time,lamda,fluxes,u,v
;
; Computes complex visibility for a multiple star model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common LocalModelVis,comp_fluxes,md5seed
;
; The number of elements in the binary model corresponds to the number of
; binary components (e.g. components like A-B, A-BC, AB-CD, etc.). The
; number of elements of the star model must correspond to the number of
; real stars involved in the entire model of the (multiple) stellar system.
;
if num_binary() gt 0 then begin
;	Hierarchical model
	comp_fluxes=fltarr(n_elements(lamda),num_star())
	levels=strlen(binary_model.component)
	i=where(max(levels) eq levels)
	modvis=componentvis(binary_model[i].component,time,lamda,fluxes,u,v)
; 	Check if there is a "dangling" uncorrelated flux component
	i=where(star_model.type eq 0,count)
	if count eq 1 then begin
		if strpos(strjoin(binary_model.component), $
			star_model[i].component) lt 0 then begin
		modvis_2= $
		componentvis(star_model[i].component,time,lamda,fluxes1,u,v)
		fluxes=fluxes+fluxes1
		endif
	endif
endif else if num_star() ge 1 then begin
;	Distribution of single components
	modvis=complexarr(n_elements(time),n_elements(lamda))
	fluxes=fltarr(n_elements(lamda))
;	For uvimage, which calls with num_scan=num_wave=1
	if n_elements(modvis) eq 1 then begin
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
	endif
	for i=0,num_star()-1 do begin
	modvis= $
	modvis+componentvis(star_model[i].component,time,lamda,fluxes1,u,v)
	fluxes=fluxes+fluxes1
	endfor
endif else begin
	print,'***Error(MODELVIS): model incomplete!'
	return,-1
endelse
;
return,modvis
end
;-------------------------------------------------------------------------------
function componentflux,component,epochs,lamda
;
; Input parameters:
; component: 		A, B, C,..., A-B, AB-C, AB-CD,...
; epochs: 		dblarr(num_scan), Julian day (-2440000)
; lamda [m]: 		dblarr(num_wave)
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
forward_function componentflux
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
; Initialize constants
factor=(pi_circle^2)/(180L*3600L*1000L)
mas2rad=pi_circle/(180L*3600L*1000L)
;
num_scan=n_elements(epochs)
num_wave=n_elements(lamda)
;
; Replicate fluxes to same dimensions as [epochs,lamda]
flux=dblarr(num_wave,num_scan)
;
; Get length of component to find out whether component is single or double
len=strlen(component)
;
; Single star...................................................................
if len eq 1 then begin
;
; Find applicable model
index=where(star_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTFLUX): non-unique component definition!'
	return,-1
endif
model=star_model[index]
;
; This initializes the limb darkening database
fluxes=stellarfluxes(model,lamda,ld_coeffs)
fluxes=fluxes*modelfluxes(model,lamda)
for i=0,num_scan-1 do flux[*,i]=fluxes 
if n_elements(flux) eq 1 then flux=flux[0]	; default model flux
;
case abs(model.type) of
;	Image cube
     12:begin
	nl=n_elements(channel_wavelengths)
	fluxes=fltarr(nl)
	for j=0,nl-1 do fluxes[j]=total(channel_images[*,*,j])
	flux=interpol(fluxes,channel_wavelengths,lamda)
	if n_elements(flux) eq 1 then flux=flux[0]
	return,flux
	end
; 	Rotating star, using Roche code V-1.4
     14:begin
	vis=rochevis(model,lamda,0,0,fluxes)
	for i=0,num_scan-1 do flux[*,i]=fluxes
	if n_elements(flux) eq 1 then flux=flux[0]
;	3.24 mas is the diameter of Vega, whose colors we use for normalization
	return,flux*(star_model[0].diameter/3.24)^2
	end
;
;	Ring or disk with hole, and temperature profile (Hillenbrand 1992),
;	also used by Malbet et al. (2005, "temperature gradient model").
;	T  ~ (radius/r_min)^-alpha, alpha should be -3/4
;	For the effect of tilt on flux, see Berger & Segransan (2007).
;	T  ~ (radius/r_min)^alpha, alpha should be -3/4.
     15:begin
	num_ring=100
	w_ring=(model.diameter/2)*model.width/num_ring
	r_ring=model.diameter/2+w_ring/2+findgen(num_ring)*w_ring
	alpha=model.alpha
	teff0=model.teff
	fluxes=fltarr(num_wave)
	for k=0,num_ring-1 do begin
		model.teff=-teff0*(r_ring[k]/r_ring[0])^alpha
		f=stellarfluxes(model,lamda,ld_coeffs)*2*!pi*r_ring[k]*w_ring $
		 *modelfluxes(model,lamda)*cos(model.tilt/RAD)
		fluxes=fluxes+f
	endfor
	model.teff=teff0
	for i=0,num_scan-1 do flux[*,i]=fluxes 
	if n_elements(flux) eq 1 then flux=flux[0]
;	save,lamda,fluxes,filename='fluxes_comp_15.xdr'
	return,flux
	end
;
   else:begin
;	Return default flux computed by stellarfluxes
	return,flux
	end
;
endcase
;
endif
;
; Double star...................................................................
;
; Set applicable model
index=where(binary_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): non-unique component definition!'
	return,-1
endif
model=binary_model[index]
;
case model.method of
;
; 	Interacting binary, with LC/WD code
      4:begin
	f=lcmap(model,epochs,1)
	for i=0,num_scan-1 do begin
;	for j=0,num_wave-1 do begin
	flux[*,i]=f[i]
;	endfor
	endfor
	return,flux
;
; 	Old WD code
	for i=0,num_scan-1 do begin
	for j=0,num_wave-1 do begin
	map=wdmap(model,epochs[i],lamda[j])
	flux[j,i]=total(map.fcc)
	endfor
	endfor
	return,flux
        end
;
   else:
;
endcase
;
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
return,componentflux(comp_1,epochs,lamda)+componentflux(comp_2,epochs,lamda)
;
end
;-------------------------------------------------------------------------------
function componentfluxes,epoch,lamda
;
; Convenience function to compute the fluxes of the individual components of
; a model. Returns a string array
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Initialize parameters for convenience use
if n_elements(epoch) eq 0 then epoch=systime(/julian)-2440000.d0
if n_elements(lamda) eq 0 then lamda=mean(gen_model.wavelengths)*1e-6
;
components=star_model.component
fluxes=strarr(n_elements(components))
;
for i=0,n_elements(components)-1 do begin
	fluxes[i]=string(componentflux(components[i],epoch,lamda))
	components[i]=blanks(strlen(fluxes[i])-1)+components[i]
endfor
;
return,[[components],[fluxes]]
;
end
;-------------------------------------------------------------------------------
function modelflux,epochs,lamda
;
; Computes total fluxes for a multiple star model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if num_binary() gt 0 then begin
; 	Find the toplevel component
	levels=strlen(binary_model.component)
	index=where(max(levels) eq levels)
	modflux=componentflux(binary_model[index].component,epochs,lamda)
endif else if num_star() ge 1 then begin
	num_scan=n_elements(epochs)
	num_wave=n_elements(lamda)
	modflux=dblarr(num_wave,num_scan)
	for i=0,num_star()-1 do begin
	modflux= $
	modflux+componentflux(star_model[i].component,epochs,lamda)
	endfor
endif else begin
	print,'***Error(MODELFLUX): model incomplete!'
	return,-1
endelse
;
return,modflux
end
;-------------------------------------------------------------------------------
function modelchisq,n_deg_free,n_parms
;
; Compute and return reduced Chi^2 of model.
; Make sure you computed the model value before calling this function!
;
if n_elements(n_parms) eq 0 then n_parms=0
;
; marquardtdata uses also buffered data
marquardtdata,y,ysig,ymod
if y[0] eq 0 and ysig[0] eq 0 then return,-1
;
n_deg_free=n_elements(y)-n_parms
return,total(((y-ymod)/ysig)^2)/n_deg_free
;
end
;-------------------------------------------------------------------------------
pro cleanup_componentvis
;
; Removes all oyster.componentvis directories accumulated over time.
;
answer=''
read,answer,prompt='Did you run updatedb already? (y/n) '
answer=strupcase(answer)
if answer eq 'N' then begin
	print,'Please run updatedb (as root) first!'
	return
endif
;
spawn,'locate oyster.componentvis',r
index=where(strpos(r,'xdr') lt 0,count)
if count gt 0 then r=r[index] else begin
	print,'No files found.'
	return
endelse
;
print,'Directories with oyster.componentvis files (sizes in MB):'
spawn,'du -s '+strjoin(r,' ')
;
answer=''
read,answer,prompt='Delete all? (y/n) '
answer=strupcase(answer)
if answer eq 'Y' then begin
	for i=0,count-1 do spawn,'rm -rf '+r[i]
	print,'Please run updatedb now as root!'
endif
;
end
;-------------------------------------------------------------------------------
