;*******************************************************************************
; File: starbase.pro
;
; Description:
; ------------
; Implements a stellar database and contains the procedures manipulating it.
;
; Block directory:
; ----------------
; Block 1: extract_npoi,merge_npoi,merge_startable,vizier_starid,oyster_starid,
;	   create_startable,read_catalogs,get_startable
; Block 2: get_hic,get_hip,get_sao,get_bsc,get_fln,get_sky,get_fkv,
;	   get_bat,get_cat,get_wbs,
;	   read_notes,read_startable,read_obslist,read_hicorbits
; Block 3: get_sb9,get_wds,put_wds,datarequest,get_bcr
; Block 4: hdn_startable,usr_startable,edit_startable,add_startable,
;	   valid_cat,update_cat,assign_ids
; Block 5: write_aspro,write_calvin,write_latex,write_npoi,write_at,
;	   write_finito,write_fov,write_iris,write_rdb,
;	   list_star,list_note,list_notes,list_category,list_keyword
; Block 6: rename_starids,rename_bsc,
;	   get_crossindex,get_wdsbsc,get_wdshdn,get_wdssao,
;	   create_cri,edit_cri,update_cri,
;	   hdn_cri,bsc_cri,fkv_cri
; Block 7: spec_parse,set_defclass,
;	   diameter_ri,diameter_bv,diameter_vk
;	   bad_cals
; Block 8: avm_stars,d_stars,
;	   i_binaries,a_binaries,asini_binaries,a_sini,a12sini_binaries,
;	   k_binaries,m_binaries,mf_binaries
; Block 9: compile_sb2,compile_sb1,compile_sb0,compile_speckle,
;	   compile_bbc,compile_bbcsouth,compile_wdssouth,
;	   compile_diameter,compile_calmaster,compile_cal,compile_bcr,
;	   compile_technical
;
;************************************************************************Block 1
function extract_npoi
;
; Return reduced column startable for NPOI star catalogs.
;
common StarBase,StarTable,Notes
;
return,build_npoitable(StarTable.starid,StarTable.name, $
		       StarTable.ra,StarTable.dec, $
		       StarTable.mv,StarTable.spectrum, $
		       StarTable.pmra,StarTable.pmdec, $
		       StarTable.rv,StarTable.px,StarTable.diameter)
;
end
;-------------------------------------------------------------------------------
function merge_npoi,table1,table2
;
; Merge two NPOI catalogs. Does not check for duplicity!
;
starid=[table1.starid,table2.starid]
name=[table1.name,table2.name]
ra=[table1.ra,table2.ra]
dec=[table1.dec,table2.dec]
mv=[table1.mv,table2.mv]
spectrum=[table1.spectrum,table2.spectrum]
pmra=[table1.pmra,table2.pmra]
pmdec=[table1.pmdec,table2.pmdec]
rv=[table1.rv,table2.rv]
px=[table1.px,table2.px]
diameter=[table1.diameter,table2.diameter]
;
t=build_npoitable(starid,name,ra,dec,mv,spectrum,pmra,pmdec,rv,px,diameter)
;
starid=''
name=''
ra=0
dec=0
mv=0
spectrum=''
pmra=0
pmdec=0
rv=0
px=0
diameter=0
;
return,t
;
end
;-------------------------------------------------------------------------------
function merge_startable,table1,table2
;
; Merge two star tables and return unique result. 
; (The presently loaded startable is not affected.)
;
; Note that table1 entries take precedence if duplicate.
;
common StarBase,StarTable,Notes
;
if n_elements(StarTable) ne 0 then table=StarTable
;
create_startable,[table1.starid,table2.starid],/quiet
new_table=StarTable
if n_elements(table) ne 0 then StarTable=table
;
sortkey=[table1.starid+'2',table2.starid+'1']
for i=0,n_tags(new_table)-1 do begin
	if n_elements(size(new_table.(i),/dim)) eq 1 then $
		new_table.(i)=[table1.(i),table2.(i)]
	if n_elements(size(new_table.(i),/dim)) eq 2 then $
		new_table.(i)=[[table1.(i)],[table2.(i)]]
endfor
index=uniq(new_table.starid,sort(sortkey))
return,new_table[index]
;
end
;-------------------------------------------------------------------------------
function vizier_starid,starids
;
; This function is called to enable Vizier access for OYSTER StarIds which
; do not conform to the SimBad naming convention, e.g. HDN123456 -> HD123456
;
; Remove white space
for i=0l,n_elements(starids)-1 do begin
	ok=1
	case strmid(starids[i],0,3) of
	'HDN': ok=0
	'HIP': ok=0
	'HIC': ok=0
	'BSC': ok=0
	'FKV': ok=0
	else:
	endcase
	if not ok then begin
	ID=string(strmid(strcompress(starids,/remove_all),3,6),format='(i6.6)')
	case strmid(starids[i],0,3) of 
;	'HDN': starids(i)='HD'+string(strmid(starids(i),3,6),format='(i6.6)')
;	'HIP': starids(i)='HI'+string(strmid(starids(i),3,6),format='(i6.6)')
;	'BSC': starids(i)='HR'+string(strmid(starids(i),3,4),format='(i4.4)')
;	'FKV': starids(i)='FK5'+string(strmid(starids(i),3,4),format='(i4.4)')
	'HDN': starids[i]='HD'+ID
	'HIP': starids[i]='HI'+ID
	'BSC': starids[i]='HR'+ID
	'FKV': starids[i]='FK5'+ID
	endcase
	endif
endfor
;
return,starids
;
end
;-------------------------------------------------------------------------------
function oyster_starid,starids,quiet=quiet
;
; This function is called to allow the creation of the OYSTER StarTable with
; StarIDs not following the OYSTER standard, e.g., HD 123456 -> HDN123456
;
for i=0l,n_elements(starids)-1 do begin
	starids[i]=strcompress(starids[i],/remove_all)
	ok=0
	case strmid(starids[i],0,3) of
	'HDN': ok=1
	'HIP': ok=1
	'BSC': ok=1
	'FKV': ok=1
	'USR': ok=1
	 else:
	endcase
	if not ok then begin
	case strmid(starids[i],0,2) of 
	'HD': starids[i]='HDN'+string(strmid(starids[i],2,6),format='(i6.6)')
	'HI': starids[i]='HIP'+string(strmid(starids[i],2,6),format='(i6.6)')
	'HR': starids[i]='BSC'+string(strmid(starids[i],2,4),format='(i4.4)')
	'FK': starids[i]='FKV'+string(strmid(starids[i],3,4),format='(i4.4)')
	else: if not keyword_set(quiet) then print,'Unknown StarID '+starids[i] $
		+'! Please add in oyster/starbase/vlti.hdn.'
	endcase
	endif
endfor
;
return,starids
;
end
;-------------------------------------------------------------------------------
pro create_startable,star_ids_in,quiet=quiet,ra_in=ra,dec_in=dec
;
; In call, use ra_in=ra with, e.g., ra='07:32:12.1' to set ra
; In call, use dec_in=dec with, e.g., dec='-16:58:24.64' to set dec
;
; Create and initialize StarTable.
;
common StarBase,StarTable,Notes
;
if strlen(star_ids_in[0]) gt 0 then begin
	star_ids=oyster_starid(star_ids_in,quiet=quiet)
endif else begin
;	Define star just by coordinates
	star_ids=''
	if n_elements(ra) eq 0 or n_elements(dec) eq 0 then begin
		print,'Error: if StarId undefined, please specify RA & Dec!'
		return
	endif
	if n_elements(ra) ne n_elements(dec) then begin
		print,'Error: RA has not the same size as DEC!'
		return
	endif
	star_ids=strarr(n_elements(ra))
	ra_in=ra
	dec_in=dec
endelse
;
numstars=n_elements(star_ids)
;
maxwaves=100
;
starid=strarr(numstars)		; StarId (e.g. BSC1708)
starid[*]=star_ids
;
; Allocate arrays
name=strarr(numstars)		; Star name (20 chars)
; name(*)='                    '
var=strarr(numstars)		; Variable name (10 chars)
; var(*)='          '
;
; toe=strarr(numstars)		; Proper name (40 chars)
; toe(*)='                                        '
;
ra=dblarr(numstars)		; Right Ascension [h]
rae=fltarr(numstars)		; Right Ascension error [h]
dec=dblarr(numstars)		; Declination [deg]
dece=fltarr(numstars)		; Declination error [deg]
;
bat=intarr(numstars)		; Batten catalogue number
fkv=intarr(numstars)		; FK5 number
bsc=intarr(numstars)		; HR number
fln=intarr(numstars)		; Finding list number
ads=intarr(numstars)		; ADS number
hdn=lonarr(numstars)		; HD number
hic=lonarr(numstars)		; HIPPARCOS input catalogue number
sao=lonarr(numstars)		; SAO number
wds=lonarr(numstars)		; WDS number
;
mv=fltarr(numstars)+100		; Johnson V, combined if double
av=fltarr(numstars)		; Visual extinction
mj=fltarr(numstars)+100		; Johnson J, combined if double
mh=fltarr(numstars)+100		; Johnson H, combined if double
mk=fltarr(numstars)+100		; Johnson K, combined if double
flp=fltarr(numstars)		; 3.8 micron Lp flux [Jansky]
f12=fltarr(numstars)		; 12 micron N-band flux [Jansky]
bv=fltarr(numstars)+100		; (B-V), combined if double
ub=fltarr(numstars)+100		; (U-B), combined if double
ri=fltarr(numstars)+100		; (R-I), combined if double
dmv=fltarr(numstars)+100	; V magnitude difference
amv=fltarr(numstars)+100	; Absolute V magnitude, combined if double
;
by=fltarr(numstars)+100		; Stroemgren (b-y)
m1=fltarr(numstars)+100		; Stroemgren m1
c1=fltarr(numstars)+100		; Stroemgren c1
beta=fltarr(numstars)+100	; Stroemgren Beta index
feh=fltarr(numstars)+100	; [Fe/H]
;
pmra=fltarr(numstars)		; Cent. Proper motion in RA [s]
pmrae=fltarr(numstars)		; Cent. Proper motion error in RA [s]
pmdec=fltarr(numstars)		; Cent. Proper motion in DEC ["]
pmdece=fltarr(numstars)		; Cent. Proper motion error in DEC ["]
rv=fltarr(numstars)		; mean radial velocity [km/s]
vsini=fltarr(numstars)		; rotational velocity [km/s]
px=fltarr(numstars)		; Parallax ["]
pxe=fltarr(numstars)+0.01	; Parallax error ["]; default to 10 mas
d=fltarr(numstars)		; Distance [pc]
;
spectrum=strarr(numstars)	; Spectral class [MK]
; spectrum(*)='                              '
type=fltarr(numstars)		; Spectral type (primary)
type2=fltarr(numstars)		; Spectral type (secondary)
class=fltarr(numstars)		; Luminosity class (primary)
class2=fltarr(numstars)		; Luminosity class (secondary)
sed=fltarr(maxwaves,numstars)	; Flux
wvl=fltarr(maxwaves,numstars)	; Wavelengths [microns]
teff=fltarr(numstars)		; Effective temperature [K] (primary)
teff2=fltarr(numstars)		; Effective temperature [K] (secondary)
logg=fltarr(numstars)		; log(g) [cgs] (primary)
logg2=fltarr(numstars)		; log(g) [cgs] (secondary)
;
p=fltarr(numstars)		; Orbital period [d]
t=dblarr(numstars)		; Epoch [JD]
o=fltarr(numstars)		; Argument of periastron [deg]
e=fltarr(numstars)		; Eccentricity
i=fltarr(numstars)		; Inclination [deg]
n=fltarr(numstars)		; Argument of ascending node [deg]
a=fltarr(numstars)		; Semi-major axis ["]
k1=fltarr(numstars)		; [km/s]
k2=fltarr(numstars)		; [km/s]
mf=fltarr(numstars)		; Mass function for single-lined binaries
m1sin3i=fltarr(numstars)
m2sin3i=fltarr(numstars)
a1sini=fltarr(numstars)		; [km/s]
a2sini=fltarr(numstars)		; [km/s]
;
diameter=fltarr(numstars)	; [mas]; polar diameter for rotating stars
diameter2=fltarr(numstars)	; [mas]
omega=fltarr(numstars)		; Rotational angular rate in units of breakup
tilt=fltarr(numstars)		; tilt of rotation axis, 90 = orthogonal
rapa=fltarr(numstars)		; PA of rotation axis on sky
zexp=fltarr(numstars)+0.25	; van Zeipel exponent, default=0.25
zerospacing=fltarr(numstars)	; Zero spacing visibility
a0=fltarr(numstars)		; Linear limb darkening fit coefficient
a1=fltarr(numstars)		; Linear limb darkening fit coefficient
a2=fltarr(numstars)		; Linear limb darkening fit coefficient
mass=fltarr(numstars)		; [M_sun]
mass2=fltarr(numstars)		; [M_sun]
;
bflag=strarr(numstars)+'.'	; Binary/calibrator flag, '.','B','C'
hflag=strarr(numstars)+' '	; Hipparcos double/multiple systems flag:
;    C : solutions for the components
;    G : acceleration or higher order terms
;    O : orbital solutions
;    V : variability-induced movers (apparent motion arises from variability)
;    X : stochastic solution (probably astrometric binaries with short period)
vflag=strarr(numstars)+' '	; Hipparcos variability flag: 
;    C : no variability detected ("constant")
;    D : duplicity-induced variability
;    M : possibly micro-variable (amplitude < 0.03mag)
;    P : periodic variable
;    R : V-I colour index was revised due to variability analysis
;    U : unsolved variable which does not fall in the other categories
sflag=strarr(numstars)+'OK'	; Status/consistency flag; '!' if bad
;
model=strarr(numstars)		; e.g. UDK, LD, etc...
reference=strarr(numstars)	; Reference
;
StarTable=build_startable(starid,name,var,ra,rae,dec,dece, $
	mv,av,mj,mh,mk,flp,f12,bv,ub,ri,dmv,amv,by,m1,c1,beta, $
	diameter,diameter2,omega,tilt,rapa,zexp, $
	zerospacing,a0,a1,a2,mass,mass2,feh, $
	pmra,pmrae,pmdec,pmdece,rv,vsini,px,pxe,d, $
	spectrum,type,type2,class,class2,sed,wvl, $
	teff,teff2,logg,logg2, $
	p,t,o,e,i,n,a,k1,k2,mf,m1sin3i,m2sin3i,a1sini,a2sini, $
	bflag,hflag,vflag,sflag,model,reference) 
;
if not keyword_set(quiet) then $
print,'StarTable created; number of entries = ',numstars,'.',format='(a,i6,a)'
;
; If call used coordinates, update these in the StarTable
if strlen(starid[0]) eq 0 then begin
	StarTable.ra=hms2h(ra_in)
	StarTable.dec=dms2d(dec_in)
;	Restore inputs
	ra=ra_in
	dec=dec_in
endif
;
end
;-------------------------------------------------------------------------------
pro read_catalogs
;
; Read catalogs according to star names.
; Implemented catalogs:
; BSC Bright Star Catalogue, Hoffleit & Jaschek 1982
; HIC HIPPARCOS Input Catalogue
; HIP HIPPARCOS Output Catalogue
; HDN Sky Catalogue 2000 (redirected to HIP due to copyright)
; SAO Catalogue
; FKV Fundamental Katalog Nr. 5
; BAT Eighth Catalogue of the orbital elements of spectroscopic binary systems
; FLN A Finding List for Observers of Interacting Binary Stars, 5th Edition
; WDS Washington Double Star Catalog (read with get_wds,stars)
;
common StarBase,StarTable,Notes
;
numstars=n_elements(StarTable)
;
starid=StarTable.starid
name=strarr(numstars)
for i=0L,numstars-1 do begin
	starname='                    '
	strput,starname,StarTable[i].name
	name[i]=starname
endfor
var=strarr(numstars)
for i=0L,numstars-1 do begin
	starname='          '
	strput,starname,StarTable[i].var
	var[i]=starname
endfor
;
; Redirect HDN to HIP if SkyCat not available
hdn_count=0
if getenv('USER') ne !owner then begin
	hdn_index=where(strmid(startable.starid,0,3) eq 'HDN',hdn_count)
	if hdn_count gt 0 then begin
		print,'Redirecting HDN to HIP...'
		hdn_starids=startable[hdn_index].starid
		hdn_ids=long(strmid(hdn_starids,3,6))
		hip=cri(hdn_ids,'hdn-hic') > 0
		startable[hdn_index].starid='HIP'+stringl(hip,format='(i6.6)')
		starid[hdn_index]=startable[hdn_index].starid
	endif
endif
;
ra=StarTable.ra
dec=StarTable.dec
bat=StarTable.bat
hdn=StarTable.hdn
bsc=StarTable.bsc
fkv=StarTable.fkv
hic=StarTable.hic
sao=StarTable.sao
fln=StarTable.fln
ads=StarTable.ads
mv=StarTable.mv
bv=StarTable.bv
ub=StarTable.ub
ri=StarTable.ri
dmv=StarTable.dmv
pmra=StarTable.pmra
pmdec=StarTable.pmdec
rv=StarTable.rv
vsini=StarTable.vsini
px=StarTable.px
pxe=StarTable.pxe
spectrum=StarTable.spectrum & spectrum[*]='                              '
type=StarTable.type
type2=StarTable.type2
class=StarTable.class
class2=StarTable.class2
a=StarTable.a
p=StarTable.p
t=StarTable.t
o=StarTable.o
e=StarTable.e
k1=StarTable.k1
k2=StarTable.k2
mf=StarTable.mf
m1sin3i=StarTable.m1sin3i
m2sin3i=StarTable.m2sin3i
a1sini=StarTable.a1sini
a2sini=StarTable.a2sini
hflag=StarTable.hflag
vflag=StarTable.vflag
;
name_b=byte(name) & name_b_num=n_elements(name_b[*,0])
var_b=byte(var) & var_b_num=n_elements(var_b[*,0])
spectrum_b=byte(spectrum) & spectrum_b_num=n_elements(spectrum_b[*,0])
hflag_b=byte(hflag) & hflag_b_num=n_elements(hflag_b[*,0])
vflag_b=byte(vflag) & vflag_b_num=n_elements(vflag_b[*,0])
;
status=linknload(!external_lib,'catalog',numstars,starid,name,var,ra,dec, $
	bat,hdn,bsc,fkv,hic,sao,fln,ads, $
	mv,bv,ub,ri,dmv,pmra,pmdec,rv,vsini,px,pxe, $
	spectrum,type,type2,class,class2, $
	p,t,o,e,k1,k2,mf,m1sin3i,m2sin3i,a1sini,a2sini,a,hflag,vflag, $
	name_b,var_b,spectrum_b,hflag_b,vflag_b, $
	name_b_num,var_b_num,spectrum_b_num,hflag_b_num,vflag_b_num)
if status ne 0 then begin
	print,'***Error(READ_CATALOGS): error reading catalogs!'
	return
endif
;
if numstars eq 1 then begin
	name=name[0]
	var=var[0]
	spectrum=spectrum[0]
endif
;
StarTable.starid=starid
StarTable.name=strtrim(name,0)
StarTable.var=var
StarTable.ra=ra
StarTable.dec=dec
StarTable.bat=bat
StarTable.hdn=hdn
StarTable.bsc=bsc
StarTable.fkv=fkv
StarTable.hic=hic
StarTable.sao=sao
StarTable.fln=fln
StarTable.ads=ads
StarTable.mv=mv
StarTable.bv=bv
StarTable.ub=ub
StarTable.ri=ri
StarTable.dmv=dmv
StarTable.pmra=pmra
StarTable.pmdec=pmdec
StarTable.rv=rv
StarTable.vsini=vsini
StarTable.px=px
StarTable.pxe=pxe
StarTable.spectrum=strcompress(spectrum,/remove_all)
StarTable.type=type
StarTable.type2=type2
StarTable.class=class
StarTable.class2=class2
StarTable.a=a
StarTable.p=p
StarTable.t=t
StarTable.o=o
StarTable.e=e
StarTable.k1=k1
StarTable.k2=k2
StarTable.mf=mf
StarTable.m1sin3i=m1sin3i
StarTable.m2sin3i=m2sin3i
StarTable.a1sini=a1sini
StarTable.a2sini=a2sini
StarTable.hflag=strtrim(hflag)
StarTable.vflag=strtrim(vflag)
;
StarTable.name=strtrim(string(name_b),0)
StarTable.var=string(var_b)
StarTable.spectrum=strcompress(string(spectrum_b),/remove_all)
StarTable.hflag=strtrim(string(hflag_b))
StarTable.vflag=strtrim(string(vflag_b))
;
; If HDN was redirected to HIP, restore original star ID
if hdn_count gt 0 then begin
	startable[hdn_index].starid=hdn_starids
	startable[hdn_index].hdn=hdn_ids
endif
;
; Non-standard star lists
; Look here first as we use cri_vlti
index=where(strmid(startable.starid,0,3) eq 'OBJ',count)
if count gt 0 then begin
;	Stars without known catalog ID
	radec=esopos(strmid(startable[index].starid,3,16))
	startable[index].ra=radec[*,0]
	startable[index].dec=radec[*,1]
	for i=0,n_elements(index)-1 do $
		startable[index[i]].starid=cri_vlti(startable[index[i]].starid, $
		startable[index[i]].ra,startable[index[i]].dec)
endif
index=where(strmid(startable.starid,0,3) eq 'WDS',count)
if count gt 0 then begin
;	WDS stars
	n=0l
	table=startable
	get_wds,table[index].starid
	for i=0L,n_elements(startable)-1 do begin
		index=where(table.starid eq startable[i].starid,l)
		if l gt 0 then begin
		n=n+1
		for j=0,n_tags(startable[i])-1 do $
			table[index].(j)=startable[i].(j)
		endif
	endfor
	print,'Entries found in WDS.xdr: ',n
	startable=table
endif
index=where(strmid(startable.starid,0,3) eq 'USR',count)
if count gt 0 then begin
;	USR stars
	n=0l
	table=startable
	table_tags=tag_names(table)
	get_cat,'USR'
	startable_tags=tag_names(startable)
	for i=0L,n_elements(startable)-1 do begin
		jndex=where(long(strmid(table[index].starid,3,6)) $
			 eq long(strmid(startable[i].starid,3,6)),l)
		if l gt 0 then begin
		n=n+1
		for j=0,n_tags(startable[i])-1 do begin
			k=where(table_tags eq startable_tags[j],l)
			if l eq 1 then table[index[jndex]].(k)=startable[i].(j)
		endfor
		endif
	endfor
	print,'Entries found in USR.xdr: ',n
	startable=table
endif
index=where(strmid(startable.starid,0,3) eq 'HDN' and startable.hdn eq 0,count)
if count gt 0 then begin
;	Get info for stars not found in Skycat from dynamic HDN catalog
	table=startable
	zero_ids=long(strmid(table[index].starid,3,6))
	get_cat,'HDN' ; load from dynamic HDN catalog
;
	hdn_ids=long(strmid(startable.starid,3,6))
	j=whereequal(zero_ids,hdn_ids)
	if j[0] ne -1 then count=n_elements(j) else count=0
	n=0L
	for i=0L,count-1 do begin
		jndex=where(hdn_ids eq zero_ids[j[i]],l)
		if l eq 1 then begin
			n=n+1
			t=table[index[j[i]]]
			struct_assign,startable[jndex],t
			table[index[j[i]]]=t
		endif
	endfor	
	print,'Entries found in HDN.xdr: ',n
	index=where(strmid(table.starid,0,3) eq 'HDN' $
		       and table.hdn eq 0,count)
	if count gt 0 then begin
		print,'WARNING: unresolved HDN entries: ',count
		print,'Use hdn_startable,HD_ID to add star.'
	endif
	startable=table
endif
;
print,'Finished reading catalogs.'
;
end
;-------------------------------------------------------------------------------
pro get_startable,starids,names=names,ra_in=ra,dec_in=dec,name_in=name, $
	quiet=quiet
;
; Compound procedure to create startable and read catalogs.
; If stars are not specified, assume CHAMELEON is calling this
; procedure and list the stars in the scantable, and also read
; the diameters.
;
; The procedure expects starids like CAT1234, e.g., HDN123456
;
; If names keyword set, store star names.
; If keyword name_in is set, use it to set the name in startable.
; If RA and DEC are specified, just create a single entry:
; 	get_startable,ra='02h33m09.668',dec='+42d53m46.265'
;
common StarBase,StarTable,Notes
;
if keyword_set(ra) and keyword_set(dec) then begin
	print,'Please note: RA [hms or deg] and DEC [dms or deg]'
;	Both h/d and d/d input formats are now allowed.
	ra_in=ra
	dec_in=dec
	if isnumeric(ra) then ra_d=hms(ra_in) else ra_d=ra_in
	if isnumeric(dec) then dec_d=dms(dec_in) else dec_d=dec_in
	create_startable,'',ra_in=ra_d,dec_in=dec_d
	if keyword_set(name) then startable.name=name[0]
;	startable.ra=ra/15	; StarTable RA must be in hours
;	startable.dec=dec
	return
endif
;
; If called without star list, this is a call within data reduction context
if n_params() eq 0 then begin
	list_stars,starids,names
	if starids[0] eq '' then return
endif
;
; Make backup copy
; starids_bck=starids
;
starids=strupcase(starids)
create_startable,starids,/quiet
read_catalogs
;
; Restore input starids for those targets not found
; index=where(startable.ra eq 0 and startable.dec eq 0,count)
; if count gt 0 then startable(index).starid=starids_bck(index)
;
; Read additional info if called within data reduction context without star list
if n_params() eq 0 then begin
;	Get better photometry from Bright Star Catalog
	rename_starids,'fkv-bsc'
	read_catalogs
	rename_bsc
	get_diameter
	get_ubvri
;	Obtain improved astrometry from Hipparcos catalog
	table=StarTable
	rename_starids,'fkv-hic'
	rename_starids,'bsc-hic'
	rename_starids,'hic-hip'
	read_catalogs
	table.ra=StarTable.ra
	table.rae=StarTable.rae
	table.dec=StarTable.dec
	table.dece=StarTable.dece
	table.pmra=StarTable.pmra
	table.pmrae=StarTable.pmrae
	table.pmdec=StarTable.pmdec
	table.pmdece=StarTable.pmdece
	table.px=StarTable.px
	table.pxe=StarTable.pxe
	table.hic=StarTable.hic
	table.starid=starids
	StarTable=table
endif
if keyword_set(names) then StarTable.name=names
;
end
;************************************************************************Block 2
pro get_hic
;
; Read entire Hipparcos Input Catalog into reduced NPOI StarTable. 
; Note, that we cannot load the complete table due to limited memory!
;
common StarBase,StarTable,Notes
;
; With this n, you need 7 catalog reads to fill in the reduced table.
n=16887L
;
for i=0,6 do begin
	get_startable,'HIC'+stringl(lindgen(n)+1+i*n,format='(i6.6)')
	table_n=extract_npoi()
	if n_elements(table) ne 0 then begin
		table=merge_npoi(table,table_n)
	endif else table=table_n
endfor
index= $
  where(strlen(strcompress(strmid(table.starid,0,3),/remove_all)) eq 0,count)
if count gt 0 then table=table[index]
;
StarTable=table
;
end
;-------------------------------------------------------------------------------
pro get_hip
;
; Superceded by following procedure!
;
; Read entire Hipparcos Output Catalog into reduced NPOI StarTable. Note that
; we cannot load the complete table due to limited memory!
;
; These stars need to be checked:
; HD 89484
; HD 100713
; HD 101666
; HD 110380
; HD 151051
; HD 163755
;
common StarBase,StarTable,Notes
;
; With this n, you need 6 catalog reads to fill in the reduced table.
n=19703L
;
for i=0,5 do begin
	get_startable,'HIP'+stringl(lindgen(n)+1+i*n,format='(i6.6)')
	table_n=extract_npoi()
	if n_elements(table) ne 0 then begin
		table=merge_npoi(table,table_n)
	endif else table=table_n
endfor
index= $
  where(strlen(strcompress(strmid(table.starid,0,3),/remove_all)) ne 0,count)
if count gt 0 then table=table(index)
;
StarTable=table
;
end
;-------------------------------------------------------------------------------
pro get_hip,gaia=gaia
;
; Read entire Hipparcos Output Catalog, 120416 entries.
; Optionally, replace HIP parallaxes with GAIA parallaxes (81867 avail.)
;
common StarBase,StarTable,Notes
;
hipfile=!catalogs_dir+'hipparcos/hip_main.xdr'
if keyword_set(gaia) then hipfile=!catalogs_dir+'hipparcos/hip_gaia.xdr'
result=file_search(hipfile)
IF strlen(result[0]) ne 0 THEN restore,hipfile $
ELSE BEGIN
;
get_startable,'HIP'+stringl(lindgen(120416)+1,format='(i6.6)') 
StarTable.starid='HIP'+stringl(StarTable.hic,format='(i6.6)')
;
if keyword_set(gaia) then begin
	d=read_csv(!catalogs_dir+'gaia/hip-gaia.csv',record_start=1)
	hip=d.field1
	hip_plx=d.field2/1d3	; convert mas to arcsec
	hip_plxe=d.field3/1d3
	gaia_plx=d.field4/1d3
	gaia_plxe=d.field5/1d3
	sep_gaia_hip=d.field6
	med_sep_gaia_hip=median(sep_gaia_hip)
	if max(sep_gaia_hip) gt 3*med_sep_gaia_hip then begin
		print,'Warning: possible misidentification(s):'
		i=where(sep_gaia_hip gt 3*med_sep_gaia_hip)
		print,hip[i]
	endif
	index=whereequal(startable.hic,hip)
	for i=0l,n_elements(index)-1 do begin
		j=where(hip eq startable[index[i]].hic) & j=j[0]
		startable[index[i]].hic=hip[j]
		startable[index[i]].px=gaia_plx[j]
		startable[index[i]].pxe=gaia_plxe[j]
	endfor
endif
;
save,StarTable,file=hipfile
;
ENDELSE
;
end
;-------------------------------------------------------------------------------
pro get_sao
;
; Read entire Smithsonian Astrophysical Catalog into reduced NPOI StarTable. 
; Note that we cannot load the complete table due to limited memory!
;
common StarBase,StarTable,Notes
;
; With this n, you need 14 catalog reads to fill in the reduced table.
n=18496L
;
for i=0,13 do begin
	get_startable,'SAO'+stringl(lindgen(n)+1+i*n,format='(i6.6)')
	table_n=extract_npoi()
	if n_elements(table) ne 0 then begin
		table=merge_npoi(table,table_n)
	endif else table=table_n
endfor
;
StarTable=table
;
end
;-------------------------------------------------------------------------------
pro get_bsc
;
; Read entire Bright Star Catalogue.
;
common StarBase,StarTable,Notes
;
get_startable,'BSC'+stringl(indgen(9096)+1,format='(i4.4)') 
StarTable.starid='BSC'+stringl(StarTable.bsc,format='(i4.4)')
rename_bsc
;
end
;-------------------------------------------------------------------------------
pro get_fln
;
; Read the entire Finding List
;
common StarBase,StarTable,Notes
;
get_startable,'FLN'+stringl(indgen(3564)+1,format='(i4.4)')
StarTable.starid='FLN'+stringl(StarTable.fln,format='(i4.4)')
;
end
;-------------------------------------------------------------------------------
pro get_sky
;
; Read entire Sky Catalogue 2000.0. This catalog is copyright, and unless 
; user !owner uses the full version, it will read from the Hipparcos
; catalog instead those HD stars with a HIP ID.
;
common StarBase,StarTable,Notes
;
if getenv('USER') ne !owner then begin
;	This file just has the HDN IDs of the Sky Catalogue 2000
	restore,!catalogs_dir+'skycat/hdn.xdr'
	starids='HDN'+stringl(hdn,format='(i6.6)')
 	create_startable,starids
 	rename_starids,'hdn-hip'
 	index=where(startable.hic ne 0)
 	starids=starids[index]
 	get_startable,startable[index].starid
endif else begin
	hdn=0L
	hdn=lindgen(49418)+1	; Trigger catalog.c to read all records
	starids='HDN'+stringl(hdn,format='(i6.6)')
	get_startable,starids
	hdn=startable.hdn
	save,hdn,filename=!catalogs_dir+'skycat/hdn.xdr'
;	Restore StarId due to sequential read
	startable.starid='HDN'+stringl(startable.hdn,format='(i6.6)')
endelse
;
end
;-------------------------------------------------------------------------------
pro get_hdn
;
; Read "dynamic" HD catalog, i.e. Sky Catalogue 2000.0 + HDN.xdr list
;
common StarBase,StarTable,Notes
;
get_sky
table=startable
; restore,!catalogs_dir+'hdn/HDN.xdr'
get_cat,'HDN'
startable=merge_startable(table,startable)
;
end
;-------------------------------------------------------------------------------
pro get_fkv
;
; Read entire Fifth Fundamental Catalogue.
;
common StarBase,StarTable,Notes
;
get_startable,'FKV'+stringl(indgen(4652)+1,format='(i4.4)') 
StarTable.starid='FKV'+stringl(StarTable.fkv,format='(i4.4)')
;
end
;-------------------------------------------------------------------------------
pro get_bat
;
; Read entire catalogue of Batten et al. (SB8)
;
common StarBase,StarTable,Notes
;
get_startable,'BAT'+stringl(indgen(1513)+1,format='(i4.4)') 
StarTable.starid='BAT'+stringl(StarTable.bat,format='(i4.4)')
;
end
;-------------------------------------------------------------------------------
pro get_cat,cat
;
; Restore external dynamical catalogs saved in XDR format. In these secondary
; catalogs, just as in the primary catalogs, the starid has a unique catalog
; identifier, e.g. 'USR' or 'CAL'.
;
common StarBase,StarTable,Notes
;
case cat of 
	'USR':catfile='usr/USR.xdr'
	'BBC':catfile='bbc/BBC.xdr'
	'HDN':catfile='hdn/HDN.xdr'
	'CAL':catfile='cal/CAL.xdr'
	else:
endcase
;
; Restore catfile to StarTable
if n_elements(catfile) eq 0 then print,'***Error(GET_CAT): unknown catalog!' $
			    else restore,!catalogs_dir+catfile
;
; Make sure the restored table is compatible with the startable format
table=startable
create_startable,table.starid
for i=0,n_elements(table)-1 do begin
	t=startable[i]
	struct_assign,table[i],t
	startable[i]=t
endfor
;
end
;-------------------------------------------------------------------------------
pro get_wbs,init=init,tycho=tycho
;
; Read entire WDS bible into startable. If init=1, read text file and save as
; xdr file. Only in this case, if tycho=1, match coordinates with Tycho catalog
; and extract V and (B-V) values.
;
common StarBase,StarTable,Notes
;
RAD=180/!pi
;
wbs_file='wds/wdsweb_summ.txt'
xdr_file='wds/wdsweb_summ.xdr'
;
if keyword_set(tycho) then begin
	tycho_csv=!oyster_dir+'catalogs/tycho/tycho2.csv'
	tycho_data=read_csv(tycho_csv)
	tycho_hip=long(tycho_data.field24)
	tycho_ra=tycho_data.field03/15
	tycho_dec=tycho_data.field04
	tycho_mv=tycho_data.field20 $
		-0.090*(tycho_data.field18-tycho_data.field20)
	tycho_bv=0.850*(tycho_data.field18-tycho_data.field20)
	precision=2.0	; ["]
endif
;
file=file_search(!catalogs_dir+xdr_file)
;
if strlen(file) eq 0 or keyword_set(init) then begin
;
l=''
status=dc_read_fixed(!catalogs_dir+wbs_file,l,/col,format='(a130)')
;
; Remove the 2 header lines
l=l[2:n_elements(l)-1]
;
; Extract data lines
coord_info=strmid(l,118,1)+strmid(l,121,1)+strmid(l,128,1)
index=where(coord_info eq '.-.' or coord_info eq '.+.',n)
l=l[index]
names=strarr(n)
ra=dblarr(n)
dec=dblarr(n)
a=fltarr(n)
mv=fltarr(n)+100
bv=fltarr(n)+100
dm=fltarr(n)+100
s=strarr(n)
starid=strarr(n)
wds=lonarr(n)
hip=lonarr(n)
;
min_sep=dblarr(n)+1e9
bad_entries=intarr(n)
;
for i=0l,n-1 do begin
	wds[i]=long(strmid(l[i],5,1)+strmid(l[i],0,5)+strmid(l[i],6,4))
	names[i]=strcompress(strmid(l[i],10,9),/remove_all)
	bad_entries[i]=1
	if strlen(strcompress(strmid(l[i],112,6),/remove_all)) ne 0 and $
	   strlen(strcompress(strmid(l[i],126,4),/remove_all)) eq 4 then begin
		bad_entries[i]=0
		ra[i]=double(strmid(l[i],112,2)) $
		     +double(strmid(l[i],114,2))/60 $
		     +double(strmid(l[i],116,5))/3600
		dec[i]=double(strmid(l[i],122,2)) $
		      +double(strmid(l[i],124,2))/60 $
		      +double(strmid(l[i],126,4))/3600
		if strmid(l[i],121,1) eq '-' then sign=-1 else sign=1
		dec[i]=sign*dec[i]
	endif
	starid[i]='WDS'+strmid(l[i],5,1)+strmid(l[i],0,5)+strmid(l[i],6,4)
	starid[i]='WDS'+strmid(l[i],0,10)
	if strlen(strcompress(strmid(l[i],52,5),/remove_all)) gt 1 then $
	a[i]=float(strmid(l[i],52,5))
	if strlen(strcompress(strmid(l[i],58,5),/remove_all)) gt 1 then $
	mv[i]=float(strmid(l[i],58,5))
	if strlen(strcompress(strmid(l[i],64,5),/remove_all)) gt 1 then begin
		mb=float(strmid(l[i],64,5))
;		If both mags available, compute combined and magn. difference
		if mv[i] ne 100 then begin
			dm[i]=mb-mv[i]
			mv[i]=cmag(mv[i],mb)
		endif
	endif
	s[i]=strmid(l[i],70,8)
;	Tycho
	if keyword_set(tycho) and bad_entries[i] eq 0 then begin
		sep=winkel(tycho_ra,tycho_dec,ra[i],dec[i])*3600
		min_sep[i]=min(sep)
		if min_sep[i] le precision then begin
			j=where(sep eq min_sep[i]) & j=j[0]
			hip[i]=tycho_hip[j]
			mv[i]=tycho_mv[j]
			bv[i]=tycho_bv[j]
		endif
	endif
endfor
;
; The histogramm shows that the precision is about 2" 
; minsep=min_sep(where(min_sep lt 1e9))
; histograph,min_sep,binsize=0.05,min=0,max=2
;
create_startable,starid
startable.name=names
startable.ra=ra
startable.dec=dec
startable.mv=mv
startable.bv=bv
startable.dmv=dm
startable.spectrum=s
startable.a=a
startable.wds=wds
startable.hic=hip
;
startable=startable[where(bad_entries eq 0)]
;
; Remove some "suspicious" entries
index=where(startable.a gt 0,count)
if count gt 0 then startable=startable[index]
index=where(startable.a ne 999.9,count)
if count gt 0 then startable=startable[index]
;
save,startable,file=!catalogs_dir+xdr_file
;
endif else begin
;
restore,!catalogs_dir+xdr_file
;
endelse
;
end
;-------------------------------------------------------------------------------
pro read_notes,file
;
common StarBase,StarTable,Notes
;
if n_elements(file) eq 0 then file=!oyster_dir+'starbase/stars.notes'
;
notes=''
status=dc_read_fixed(file,notes,/col,resize=[1],format='(a80)',ignore=['!'])
;
end
;-------------------------------------------------------------------------------
pro read_startable
;
; Create StarTable from the list of stars for which notes are available.
;
common StarBase,StarTable,Notes
;
list_notes,starids
get_startable,starids
;
end
;-------------------------------------------------------------------------------
pro read_obslist,obslist
;
common StarBase,StarTable,Notes
;
if n_elements(obslist) eq 0 then begin
	print,'***Error(READ_OBSLIST): file undefined!'
	return
endif
if strupcase(obslist) eq 'ASTROMETRY' then begin
	status=dc_read_free(!oyster_dir+'npoi/obslist.don',fkv,/col)
	get_startable,'FKV'+string(fkv,format='(i4.4)')
	index=where(startable.mv lt 4.0)
	startable=startable[index]
	return
endif
result=file_search(obslist,count=fcount)
if fcount eq 0 then begin
	print,'***Error(READ_OBSLIST): file not found!'
	return
endif
stars=''
status=dc_read_free(obslist,stars,/col,ignore=['!'])
get_startable,stars
;
end
;-------------------------------------------------------------------------------
pro read_hicorbits
;
common StarBase,StarTable,Notes
;
file=!catalogs_dir+'hipparcos/hip_dm_o.dat'
records=''
status=dc_read_fixed(file,records,resize=[1],format='(a119)',/col)
get_startable,'HIC'+strmid(records,0,6)
;
p=float(strmid(records,7,10))
t=float(strmid(records,18,11))
a=float(strmid(records,30,8))
e=float(strmid(records,39,6))
w=float(strmid(records,46,6))
i=float(strmid(records,53,6))
n=float(strmid(records,60,6))
;
startable.p=p
startable.t=t
startable.a=a
startable.e=e
startable.w=w
startable.i=i
startable.n=n
;
end
;************************************************************************Block 3
pro get_sb9,starids
;
; Create StarTable for starids and read from the SB9 data base by D. Pourbaix
; http://sb9.astro.ulb.ac.be/SB9public.tar.gz
; Remove StarTable entries for which no data was found.
;
common StarBase,StarTable,Notes
;
RAD=180/!pi
;
; Read sb9.xdr, if it exists
sb9file=!catalogs_dir+'sb9/sb9.xdr'
result=file_search(sb9file)
IF strlen(result[0]) ne 0 THEN restore,sb9file $
			  ELSE BEGIN
;
; Read Hipparcos catalog to be able identfy SB9 stars by coordinates
get_hip
t_hip=startable
get_hdn
t_hdn=startable
;
sb9main=!catalogs_dir+'sb9/Main.dta'
sb9orbits=!catalogs_dir+'sb9/Orbits.dta'
lm=''
s=dc_read_free(sb9main,lm,/col)
lo=''
s=dc_read_free(sb9orbits,lo,/col)
;
nm=n_elements(lm)
bat_m=lonarr(nm)
hip=lonarr(nm)
hdn=lonarr(nm)
ras=dblarr(nm)
dec=dblarr(nm)
mv1=fltarr(nm)+100
mv2=fltarr(nm)+100
mf1=strarr(nm)
mf2=strarr(nm)
sp1=strarr(nm)
sp2=strarr(nm)
;
; Read main data for mv, hip, and hdn IDs
for i=0,nm-1 do begin
	words=nameparse(lm[i],'|')
	bat_m[i]=fix(words[0])
	pos=words[2]
	ipos=max([strpos(pos,'+'),strpos(pos,'-')])
	ras[i]=hms2h(strmid(pos,0,ipos),/sb9)
	dec[i]=dms2d(strmid(pos,ipos),/sb9)
;	Find HIP from closest target
	dw=winkel(ras[i],dec[i],t_hip.ra,t_hip.dec)*3600; convert to "
	index=where(dw eq min(dw))
	if min(dw[index]) le 1.0 then hip[i]=t_hip[index].hic
;	Find HDN from closest target
	dw=winkel(ras[i],dec[i],t_hdn.ra,t_hdn.dec)*3600; convert to "
	index=where(dw eq min(dw))
	if min(dw[index]) le 1.0 then hdn[i]=t_hdn[index].hdn
;	Extract parameters from SB9
	if strlen(words[4]) ne 0 then mv1[i]=float(words[4])
	mf1[i]=words[5]
	if strlen(words[6]) ne 0 then mv2[i]=float(words[6])
	mf2[i]=words[7]
	sp1[i]=words[8]
	if n_elements(words) eq 10 then sp2[i]=words[9]
endfor
;
no=n_elements(lo)
bat_o=lonarr(no)
orb=lonarr(no)
who=strarr(no)
p=fltarr(no)
t=fltarr(no)
e=fltarr(no)
o=fltarr(no)
k1=fltarr(no)
k1e=fltarr(no)
k2=fltarr(no)
k2e=fltarr(no)
rv=fltarr(no)
;
; Read orbit data (contains more orbital solutions than main data)
for i=0,no-1 do begin
	words=nameparse(lo[i],'|')
	bat_o[i]=fix(words[0])
	orb[i]=fix(words[1])
	p[i]=float(words[2])
	t[i]=2440000.d0+float(words[4])
	e[i]=float(words[7])
	o[i]=float(words[9])
	if words[11] ne '-' then $
	k1[i]=float(words[11])
	if words[12] ne '-' and words[12] ne '>' and words[12] ne 'Fixed' then $
	k1e[i]=float(strcompress(words[12],/remove_all))
	if words[13] ne '-' then $
	k2[i]=float(words[13])
	if words[14] ne '-' and words[12] ne '>' and words[14] ne 'Fixed' then $
	k2e[i]=float(strcompress(words[14],/remove_all))
	if words[15] ne '-' then $
	rv[i]=float(strcompress(words[15],/remove_all))
	who[i]=words[23]
endfor
print,'Median error K1:',median(k1e)
print,'Median error K2:',median(k2e[where(k2e ne 0)])
;
create_startable,'HIP'+string(hip,format='(i6.6)')
StarTable.hic=hip
StarTable.hdn=hdn
index=where(startable.hic eq 0 and startable.hdn ne 0,count)
if count gt 0 then $
	startable[index].starid='HDN'+string(hdn[index],format='(i6.6)')
; Next line added 2021, based on improved crossindex hic-hdn
rename_starids,'hic-hdn'
StarTable.bat=bat_m
StarTable.mv=mv1
startable.dmv=100
index=where(mv2 ne 100,count)
if count gt 0 then begin
	StarTable[index].dmv=mv2[index]-mv1[index]
	StarTable[index].mv=cmag(mv1[index],mv2[index])	; mv1 is total magnitude
endif
StarTable.bflag='B'
StarTable.ra=ras
StarTable.dec=dec
;
; Assign the latest orbital elements
for i=0,nm-1 do begin
	index=where(bat_o eq bat_m[i])
	j=where(who[index] eq 'PBX',count)
	if count gt 1 then j=j[count-1]
	if count eq 0 then j=n_elements(index)-1
	StarTable[i].p=p[index[j]]
	StarTable[i].t=t[index[j]]
	StarTable[i].e=e[index[j]]
	StarTable[i].o=o[index[j]]
	StarTable[i].k1=k1[index[j]]
	StarTable[i].k2=k2[index[j]]
	StarTable[i].rv=rv[index[j]]
endfor
;
; Save this catalog to an XDR file
save,StarTable,filename=sb9file
;
ENDELSE
;
n=n_elements(starids)
if n ne 0 then begin
	index=lonarr(n)-1
	m=0
	for i=0,n-1 do begin
		k=where(startable.starid eq starids[i],l)
		if l gt 1 then index=[index,-1]
		if l gt 0 then index[i+m:i+m+l-1]=k
		if l gt 1 then m=m+1
	endfor
	jndex=where(index eq -1,count)
	for i=0,count-1 do print,'Star not found: ',starids[jndex[i]]
	jndex=where(index ge 0,count)
	if count gt 0 then begin
		index=index[jndex]
		startable=startable[index]
	endif
endif
;
end
;-------------------------------------------------------------------------------
pro get_wds,starids
;
; Note: this procedure accesses the WDS orbit catalog with the new SQL format.
;
; Allocate startable and read orbital element data from WDS orbit solution file.
; Read the orbital element data from the "Sixth Catalog of Orbits of
; Visual Binary Stars" by W.I. Hartkopf and B.D. Mason, USNO.
; The file is !catalogs_dir/wds/orb6orbits.txt, and can by downloaded from
; http://www.usno.navy.mil/USNO/astrometry/optical-IR-prod/wds/orb6
; or directly at http://ad.usno.navy.mil/wds/orb6/orb6orbits.txt
; Delete file orb6orbits.xdr to have this procedure re-create it.
; 
; This procedure uses or prepares an XDR file for faster access.
; Set startable with data from requested stars. 
;
common StarBase,StarTable,Notes
;
RAD=180/!pi
;
; Read wdsorb.xdr, if it exists
; This file contains all orbits calculcated
; orbfile=!catalogs_dir+'wds/orb6master.xdr'
; This file contains only the "official" orbits
orbfile=!catalogs_dir+'wds/orb6orbits.xdr'
result=file_search(orbfile)
IF strlen(result[0]) ne 0 THEN restore,orbfile $
			  ELSE BEGIN
;
IF 0 THEN BEGIN
; This file no longer needed, EQNX is in orb6master.txt
; Read orb6ephem.txt
;
;000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011111111111111111111111111111111
;000000000011111111112222222222333333333344444444445555555555666666666677777777778888888888999999999900000000001111111111222222222233
;012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901
;00014+3937  HLD  60       4  Hei1963a  2000 1961 1960-85    173.9  1.191   173.5  1.197   173.1  1.203   172.8  1.208   172.4  1.214
;
rec=''
status=dc_read_fixed(!catalogs_dir+'wds/orb6ephem.txt',rec,/col, $
        resize=[1],format='(a142)',ignore=['!'])
num_ephem=n_elements(rec)
wds1=strmid(rec,0,10)
equinx_string=strmid(rec,223,4)
index=where(equinx_string eq '    ',count)
if count gt 0 then equinx_string[index]='   0'
equinx=double(equinx_string)
index=where(equinx eq 0,count)
if count gt 0 then equinx[index]=2000
ENDIF
;
; Read orb6orbits.txt
;
; New 2013 format
;00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111122222222222222222222222222222222222222222222222222222222222222222
;00000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999000000000011111111112222222222333333333344444444445555555555666666666677777777778888888888999999999900000000001111111111222222222233333333334444444444555555555566666
;23456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234
; RA,Dec (J2000).. WDS....... DD............ ADS.. HD.... HIP...  V1.11* V2.22*  PPPP.PPPPPP* eee.eeeeee AAA.AAAAA* ee.eeeee III.IIII eee.eeee NNN.NNNN* eee.eeee TTTTT.TTTTTT* eee.eeeeee E.EEEEEE e.eeeeee OOO.OOOO eee.eeee EQNX LAST G N REF..... PNGFILE...........
;
rec=''
status=dc_read_fixed(!catalogs_dir+'wds/orb6orbits.txt',rec,/col, $
	resize=[1],format='(a264)')
num_orbit=n_elements(rec)
IF 0 THEN BEGIN
if num_orbit ne num_ephem then begin
	print,'***Error(GET_WDS): files corrupt?'
	return
endif
ENDIF
;
; Remove blank lines
index=where(strlen(strcompress(rec,/remove_all)) ne 0) & rec=rec[index]
; Extract data lines
coord_info=strmid(rec,6,1)+strmid(rec,9,1)+strmid(rec,16,1)
index=where(coord_info eq '.-.' or coord_info eq '.+.',num_orbit) & rec=rec[index]
; Remove records with missing data (.) in various fields
index=where(strcompress(strmid(rec,58,6),/remove_all) ne '.',num_orbit) & rec=rec[index]
index=where(strcompress(strmid(rec,81,11),/remove_all) ne '.',num_orbit) & rec=rec[index]
index=where(strcompress(strmid(rec,205,8),/remove_all) ne '.',num_orbit) & rec=rec[index]
index=where(strcompress(strmid(rec,125,8),/remove_all) ne '.',num_orbit) & rec=rec[index]
index=where(strcompress(strmid(rec,105,9),/remove_all) ne '.',num_orbit) & rec=rec[index]
index=where(strcompress(strmid(rec,143,8),/remove_all) ne '.',num_orbit) & rec=rec[index]
index=where(strcompress(strmid(rec,162,12),/remove_all) ne '.',num_orbit) & rec=rec[index]
index=where(strcompress(strmid(rec,187,8),/remove_all) ne '.',num_orbit) & rec=rec[index]
;
; Decode the data
;
wds2=strmid(rec,19,10)
wds_string='WDS'+wds2
wds=long(strmid(rec,19,5)+strmid(rec,25,4))
index=where(strmid(rec,24,1) eq '-')
sign=strarr(n_elements(rec))
sign[*]='+'
sign[index]='-'
wdsstarids='WDS'+sign+stringl(wds,format='(i9.9)')
wds[index]=-wds[index]
;
hic_string=strcompress(strmid(rec,58,6),/remove_all)
index=where(hic_string eq '.',count)
if count gt 0 then hic_string[index]='0'
hic=long(hic_string)
hdn_string=strcompress(strmid(rec,51,6),/remove_all)
index=where(hdn_string eq '.',count)
if count gt 0 then hdn_string[index]='0'
hdn=long(hdn_string)
; hdn=cri(hic,'hic-hdn') > 0
;
rah=double(strmid(rec,0,2))
ram=double(strmid(rec,2,2))
ras=double(strmid(rec,4,5))
ra=rah+ram/60+ras/3600
;
decd=double(strmid(rec,10,2))
decm=double(strmid(rec,12,2))
decs_string=strmid(rec,14,4)
; index=where(strlen(strcompress(decs_string,/remove_all)) eq 0,count)
; if count gt 0 then decs_string(index)='0'
decs=double(decs_string)
dec=decd+decm/60+decs/3600
index=where(strmid(rec,9,1) eq '-',count)
if count gt 0 then dec[index]=-dec[index]
;
ads_string=strcompress(strmid(rec,45,5),/remove_all)
; index=where(strlen(ads_string) eq 0,count)
index=where(ads_string eq '.',count)
if count gt 0 then ads_string[index]='0'
ads=long(ads_string)
;
mag_string=strcompress(strmid(rec,66,5),/remove_all)
; index=where(strlen(mag_string) eq 0,count)
index=where(mag_string eq '.',count)
if count gt 0 then mag_string[index]='+100'
index=where(strpos(mag_string,'var') ne -1,count)
if count gt 0 then mag_string[index]='+100'
ma=double(mag_string)
;
mag_string=strcompress(strmid(rec,73,5),/remove_all)
; index=where(strlen(mag_string) eq 0,count)
index=where(mag_string eq '.',count)
if count gt 0 then mag_string[index]='+100'
index=where(strpos(mag_string,'var') ne -1,count)
if count gt 0 then mag_string[index]='+100'
mb=double(mag_string)
;
dmv=mb-ma
index=where(ma gt 99 or mb gt 99,count)
if count gt 0 then dmv[index]=100
;
; Period [d]
period=double(strmid(rec,80,12))
u=strmid(rec,92,1)
index=where(u eq 'y',count)
if count gt 0 then period[index]=period[index]*365.25
index=where(u eq 'c',count)
if count gt 0 then period[index]=period[index]*365.25*100.
; Period error [d]
err_string=strcompress(strmid(rec,94,10),/remove_all)
index=where(err_string eq '.',count)
if count gt 0 then err_string[index]='0'
period_err=double(err_string)
index=where(u eq 'y',count)
if count gt 0 then period_err[index]=period_err[index]*365.25
index=where(u eq 'c',count)
if count gt 0 then period_err[index]=period_err[index]*365.25*100.
; Semimajor axis ["]
amajor=double(strmid(rec,105,9))			; arc seconds
u=strmid(rec,114,1)
index=where(u eq 'm',count)
if count gt 0 then amajor[index]=amajor[index]/1000.	; mas -> arcsec
; Semi-major axis error [d]
err_string=strcompress(strmid(rec,116,8),/remove_all)
index=where(err_string eq '.',count)
if count gt 0 then err_string[index]='0'
smaxis_err=double(err_string)
index=where(u eq 'm',count)
if count gt 0 then smaxis_err[index]=smaxis_err[index]/1000.
; Inclination [deg]
inclin=double(strmid(rec,125,8))		; degrees
; Ascending node [deg]
ascnod=double(strmid(rec,143,8)) & ascnod=precess_pa(ascnod/RAD,equinx,ra,dec,1)*RAD
; Epoch
epochp=double(strmid(rec,162,12))
u=strmid(rec,174,1)
index=where(u eq 'y',count)
if count gt 0 then epochp[index]=by2jd(epochp[index])
index=where(u eq 'd',count)
if count gt 0 then epochp[index]=epochp[index]+2400000.0
; epoch_jd=lonarr(num_orbit)
; for i=0,num_orbit-1 do epoch_jd(i)=julian(long(epochp(i)),1,1)
; years=intarr(num_orbit)+365
; index=where(((long(epochp) mod 4) eq 0) and ((long(epochp) mod 400) ne 0),count)
; if count gt 0 then years(index)=366
; epoch_jd=epoch_jd+(epochp mod 1)*years
eccent=double(strmid(rec,187,8))
argper=double(strmid(rec,205,8))
equinx_string=strmid(rec,223,4)
index=where(equinx_string eq '    ',count)
if count gt 0 then equinx_string[index]='   0'
equinx=double(equinx_string)
index=where(equinx eq 0,count)
if count gt 0 then equinx[index]=2000
name=strmid(rec,30,11)
;
; Grade
grade=strmid(rec,233,1)
; Reference
ref=strmid(rec,237,7)
;
create_startable,'HIP'+string(hic,format='(i6.6)')
read_catalogs
StarTable.starid=wds_string
StarTable.name=strcompress(name)
StarTable.bflag='B'
StarTable.ra=ra
StarTable.dec=dec
StarTable.ads=ads
StarTable.wds=wds
StarTable.hic=hic
StarTable.hdn=hdn
;
StarTable.mv=cmag(ma,mb)
StarTable.dmv=dmv
StarTable.p=period
StarTable.a=amajor
StarTable.i=inclin
StarTable.n=ascnod
StarTable.t=epochp
StarTable.e=eccent
StarTable.o=argper
;
StarTable.sflag=grade
StarTable.reference=ref
;
; Plot for GAIA 2017
;index=where(period_err ne 0)
;p=period(index)
;pe=period_err(index)
;!x.style=1
;!x.title='log(period/d)'
;!y.title='Fractional period error'
;!p.title='Orbits in the WDS'
;!p.charsize=1.5
;plot,alog10(p),pe/p,psym=1,yrange=[0,0.3],xrange=[1,7]
;
; index=where(smaxis_err ne 0)
; a=amajor(index)*1000
; ae=smaxis_err(index)
; !x.style=1
; !x.title='log(semi-major axis/mas)'
; !y.title='Fractional error'
; !p.title='Orbits in the WDS'
; !p.charsize=1.5
; plot,alog10(a),ae/a,psym=1,yrange=[0,0.5],xrange=[-3,1]
; !y.title='Error [mas]'
; plot,alog10(a),ae*1000,psym=1,yrange=[0,30],xrange=[0,4]
;
; Remove entries corresponding to photocenter orbits
index=where(strcompress(StarTable.reference,/remove_all) ne 'Ren2013')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Jnc2005')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'HIP1997')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Pbx2000')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Pbx2013')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Trr2011')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Gln2007')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'SaJ2013')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'AST2011')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'AST2016')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'AST2007')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'HaI2002')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Ald1939')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Ald1925')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'WaX2015')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Hor2015')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Fek2015')
StarTable=StarTable[index]
index=where(strcompress(StarTable.reference,/remove_all) ne 'Dru1995')
StarTable=StarTable[index]
;
; Save this catalog to an XDR file
; save,StarTable,filename=!catalogs_dir+'wds/orb6master.xdr'
save,StarTable,filename=!catalogs_dir+'wds/orb6orbits.xdr'
;
ENDELSE
;
n=n_elements(starids)
if n ne 0 then begin
	index=lonarr(n)-1
	m=0
	for i=0,n-1 do begin
		k=where(startable.starid eq starids[i],l)
		if l gt 1 then index=[index,-1]
		if l gt 0 then index[i+m:i+m+l-1]=k
		if l gt 1 then m=m+1
	endfor
	jndex=where(index eq -1,count)
	for i=0,count-1 do print,'Star not found: ',starids[jndex[i]]
	jndex=where(index ge 0,count)
	if count gt 0 then begin
		index=index[jndex]
		startable=startable[index]
	endif
endif
;
end
;-------------------------------------------------------------------------------
pro put_wds,starid,datarequest=wdsfile
;
; If starid is specified, determine its WDS name and lookup in WDS data base.
; Retrieve the data using B. Mason's datarequest procedure, and save to disk.
;
; starid must be CCCNNNN/CCCNNNNNN or hhmmf+ddmm.
;
; If we have the data file from USNO, pass it as wdsfile.
;
common StarBase,startable,notes
;
rad=180/!pi
;
; If we got the file from USNO, go directly to DATAREQUEST 
if n_elements(wdsfile) ne 0 then begin
	f=file_search(wdsfile)
	if n_elements(f) gt 1 then f='' else f=f[0]
	if strlen(f) eq 0 then begin
		printf,'File not found.'
		return
	endif
	words=nameparse(f,'.')
	wdspos=words[0]
	comp=''
	cmpidx=0
	goto,DATAREQUEST
endif
;
if n_elements(starid) eq 0 then begin
	starid=''
	if n_elements(startable) eq 1 then starid=startable.starid
endif
if n_elements(startable) eq 0 then begin
	if valid_cat[starid] then get_startable,starid
endif
;
count=0
starid=strupcase(starid[0])
if n_elements(startable) ne 0 then i=where(startable.starid eq starid,count)
if count gt 1 then print,'Warning(PLOT_WDS): more than one star!'
if count eq 1 then begin
;	Found the star in startable
	i=i[0]
	ra=startable[i].ra
	dec=startable[i].dec
	name=hms(ra,/wds)+dms(dec,/wds)
;
	if 0 then begin
; 	Obtain search string from (J2000.0) coordinates
; 	RA:
	rah=fix(ra)
	ram=string(fix((ra mod 1)*60),format='(i2.2)')
;	if ram eq '60' then begin
;		ram='00'
;		rah=rah+1
;	endif
	ras=string(rah,format='(i2.2)')+ram
	ras=hms(ra,/wds)
;
; 	DEC:
	decd=abs(fix(dec))
	decm1=string(nint(abs(dec mod 1)*60),format='(i2.2)')
	decm2=string(fix(abs(dec mod 1)*60),format='(i2.2)')
;	if decm eq '60' then begin
;		decm='00'
;		if dec ge 0 then decd=decd+1 else decd=decd-1
;	endif
	decs1=string(decd,format='(i3.2)')+decm1
	decs2=string(decd,format='(i3.2)')+decm2
;	This code added Sep 16, 2015
	if dec gt 0 then strput,decs1,'+' else strput,decs1,'-'
	if dec gt 0 then strput,decs2,'+' else strput,decs2,'-'
	endif
endif else begin
;	...or the star might be of form 'WDS+hhmmfddmm' or 'hhmmf+ddmm'
	if strlen(starid) ne 10 and strlen(starid) ne 13 then begin
		print,'Star ID not the right format!'
		print,'Must be: hhmmf+ddmm or WDS+hhmmfddmm.'
		return
	endif
	name=starid
	if strmid(starid,0,3) eq 'WDS' then $
	name=strmid(starid,4,5)+strmid(starid,3,1)+strmid(starid,9,4)
;	ra=float(strmid(name,0,2))+float(strmid(name,2,2))/60 $
;				  +float(strmid(name,4,1))/600
;	dec=float(strmid(name,6,2))+float(strmid(name,8,2))/60
;	if fix(strmid(name,5,5)) lt 0 then dec=-dec
;	name=name
;
;	if strlen(starid) eq 10 then pos=starid
;	if strlen(starid) eq 13 then pos=strmid(starid,4,5) $
;					+strmid(starid,3,1) $
;					+strmid(starid,9,4)
;	ras=strmid(pos,0,4)
;	decs1=strmid(pos,6,4)
;	decs2=decs1
endelse
;
; Check for the entry in the bible (outdated, see file "update")
; wget http://ad.usno.navy.mil/wds/Webtextfiles/wdsweb_summ2.txt
; wget http://ad.usno.navy.mil/wds/Webtextfiles/wdsnewnotes_main.txt
; wget http://ad.usno.navy.mil/wds/Webtextfiles/wdsnewref.txt
; wget http://ad.usno.navy.mil/wds/orb6/orb6orbits.txt
; wget http://ad.usno.navy.mil/wds/orb6/orb6ephem.txt
print,'hhmmf ddmmDSCNNNNComp  YYYY YYYY   #O PAf PAl Sep_f Sep_l Mag_A Mag_B Spectrum '
wds_bible=' '+!catalogs_dir+'wds/wdsnewframe.txt'
wds_bible=' '+!catalogs_dir+'wds/wdsweb_summ.txt'
; This code commented (4 lines) Sep 16, 2015
; wds_bible=' '+!catalogs_dir+'wds/wdsids_summ.txt'
; spawn,'grep '+ras+wds_bible+' | grep '+decs1,entries
; if strlen(strcompress(entries(0),/remove_all)) eq 0 then $
; spawn,'grep '+ras+wds_bible+' | grep '+decs2,entries
; This code replaces the prvious 4 lines
; spawn,'grep '+ras+decs1+wds_bible,entries
; if strlen(strcompress(entries(0),/remove_all)) eq 0 then  $
; spawn,'grep '+ras+decs2+wds_bible,entries
;
spawn,'grep '+name+wds_bible,entries
if strlen(strcompress(entries[0],/remove_all)) eq 0 then begin
	print,'Star not found! Please check coordinates!'
	return
endif
; index=where(strpos(entries,ras) eq 0)
index=where(strpos(entries,name) eq 0)
entries=entries[index]
for i=0,n_elements(entries)-1 do print,strmid(entries[i],0,78)
;
; Retrieve WDS data locally, else ask to submit request form
path=''
path=!catalogs_dir
if strlen(path) eq 0 or total(strlen(file_search(path))) eq 0 then begin
   print,'Warning(PLOT_WDS): cannot access WDS, please submit data request to:'
   print,'                    http://ad.usno.navy.mil/ad/wds/wds_request.html'
   return
endif
comp=''
read,'Component to select (hit RETURN to quit, SPACE for blank component): ',comp
if strlen(comp) eq 0 then return
comps=strmid(entries,17,5)
index=where(strpos(comps,comp) ne -1,count)
comps=strcompress(comps,/remove_all)
if total(strlen(comps)) eq 0 then comps=comp
index=where(comps eq comp,count)
if count eq 0 then begin
	print,'***Error(PLOT_WDS): component not found!'
	return
endif
entry=strmid(entries[index[0]],0,17)
pos=strmid(entry,0,10)
;
; As of 2012, we use the sequence number to identify the selected data
; cmpidx=where(strpos(entries,comp) ge 0)
; cmpidx=cmpidx(0)
;
; Submit request
infile='datarequest.in'
openw,unit,infile,/get_lun
printf,unit,pos
free_lun,unit
spawn,!catalogs_dir+'wds/datarequest'
spawn,'rm -f '+infile
wdspos='wds'+pos
;
; Read the extracted data
DATAREQUEST:
data=''
status=dc_read_fixed(wdspos+'.txt',data,/col,resize=[1],format='(a120)')
if status ne 0 then return
;
; Locate the section which lists the observations, cmpidx locates by sequence #
index=where(strpos(data,'MEASURES') ne -1)
; added code 2015...
jndex=where(strpos(data,'--------') ne -1)
ipos1=index[0]+2
ipos2=jndex[where(jndex gt index[0])] & ipos2=ipos2[0]-1
cmpidx=where(strlen(strtrim(data[ipos1:ipos2])) eq strlen(strtrim(data[ipos1])))
if strlen(comp) eq 0 then cmpsel=indgen(n_elements(cmpidx)) else $
cmpsel=where(strtrim(strmid(data[cmpidx+ipos1],7,5)) eq comp)
ipos1=cmpidx[cmpsel]+ipos1+1
; ipos1=index(cmpidx)+3
; ...end of update
ipos2=ipos1
while strlen(strcompress(data[ipos2],/remove_all)) ne 0 do ipos2=ipos2+1
data=data[ipos1:ipos2-1]
;
; Extract data for this component
; The following line with formats copied from datarequest.pro for reference
; printf,3,datlin,newref,aper,meth,code1,code2,format='(a40,3x,a8,3x,a3,8x,a1,7x,a1,1x,a1)'
;
num_obs=n_elements(data)
pa=fltarr(num_obs)
pae=fltarr(num_obs)
sep=fltarr(num_obs)
sepe=fltarr(num_obs)
epa=fltarr(num_obs)
emaj=fltarr(num_obs)+5
emin=fltarr(num_obs)+5
date=dblarr(num_obs)
for i=0,num_obs-1 do begin
	date[i]=float(strmid(data[i],7,8))
	pa_string=strmid(data[i],19,5)
	sep_string=strmid(data[i],37,6)
	if isnumeric(pa_string) and $
	   isnumeric(sep_string) then begin
		pa[i]=float(pa_string)
		sep[i]=float(sep_string)*1000
		pae_string=strmid(data[i],28,3)
		sepe_string=strmid(data[i],46,6)
		if isnumeric(pae_string) and $
		   isnumeric(sepe_string) then begin
			pae[i]=float(pae_string)/180*!pi
			sepe[i]=float(sepe_string)*1000
			if sin(pae[i])*sep[i] gt sepe[i] then begin
				epa[i]=(pa[i]+90) mod 180
				emin[i]=sepe[i]
				emaj[i]=sin(pae[i])*sep[i]
			endif else begin
				epa[i]=pa[i]
				emaj[i]=sepe[i]
				emin[i]=sin(pae[i])*sep[i]
			endelse
		endif
	endif
;	sep_unit_code=strmid(data(i),73,1)
;	if sep_unit_code eq 'A' then sep(i)=sep(i)/1000
endfor
;
index=where(sep gt 0,count)
if count eq 0 then return
pa=pa[index]
sep=sep[index]
epa=epa[index]
emaj=emaj[index]
emin=emin[index]
date=date[index]
year=jd2jy(by2jd(date))
pap=precess_pa(pa/rad,year,ra,dec,1)
;
; Write data in AMOEBA format
filename=wdspos+'.psn'
openw,unit,filename,/get_lun
print,'Opened file for output: ',filename
num_obs=n_elements(date)
bin_comp='A-B'
for i=0,num_obs-1 do begin
	       printf,unit,bin_comp,year[i],sep[i],pa[i], $
				emaj[i],emin[i],epa[i], $
 		format='(a,2x,f9.4,3x,f7.1,2x,f6.1,2x,f7.1,1x,f7.1,1x,f6.1)'
endfor
;
free_lun,unit
;
; Remove useless pngfiles file
command='rm -f '+wdspos+'.pngfiles'
if safe(command) then spawn,command
;
end
;-------------------------------------------------------------------------------
PRO datarequest,dummy,path=path
;
; Obsolete! Functionality implemented by datarequest.f in WDS directory.
; 
; this program is written to fill wds data requests. The input is the 10 digit
; WDS coordinate. This will grab all multiple systems at a specified coordinate.
;
; output 1a ---> from WDS, each data line
; output 1b ---> explanation of codes from above
; output 1c ---> references from WDS
; output 1d ---> note from wdsnot.memo file
; output 1e ---> gif of measures                    *** NOT IMPLEMENTED YET ***
; output 2a ---> orbit parameters from orbit file
; output 2c ---> references from orbit file
; output 2d ---> note from orbit note file          
; output 2e ---> gif of measures with orbit(s)      *** NOT IMPLEMENTED YET ***
; output 2f ---> ephemerides for next 10 years      
; output 3a ---> delta-m measurements               
; output 3b ---> delta-m codes                      
; output 3c ---> delta-m references                 
; output 4c ---> reference from note file           *** NOT IMPLEMENTED YET ***
;
; To Do : How do you handle astrometric binaries? They are not in the WDS, so
; don't triger anything. Should you do a strict coordinate search or add a
; summary line (with blanks for the data) and the appropriate codes? (N, O, D)?
;
; the routine is written in IDL so that adding the features in 1e and 2e can be
; done with the plotorb routines. Also, outputs 1e, 2d, 2e, 2f, 3a, 3b, 3c, 3d,
; and 4c can be implemented as available as they are not included in the old
; (v 1.0) data request output (which is non-functional due to the new dragon).
;
; set up arrays for checking references (algorithm by wih)
;
; Version 1.0  is the old hp-unix written by Geoff Douglass
; Version 2.0  is beta test IDL version (gives as much information as v 1.0)
; Version 2.1  gives Delta-m catalog information
; Version 2.2  reads from 5th Orbit Catalog
; Version 2.3  reads from 5th Orbit Catalog ephemerides file
; Version 2.4  reads correctly from the 5th Orbit Catalog ephemerides file
;              (Julian North <jnorth@suphys.physics.usyd.edu.au> pointed out
;              error in ephemerides of Alpha Cen - Not enough digits for sep
; Version 2.5  reads from 5th Orbit Catalog notes file
; Version 2.51 fixed error in orbit and note reading.
; Version 2.52 changed to reflect reformatting of delta-m file.
; Version 2.53 changed to reflect reformatting of reference file.
; Version 2.54 changed to 6th Orbit Catalog, etc. files
; Version 2.6  output includes precise position, when available.
; Version 2.7  works with new format wds data lines and single refcode.
; Version 2.71 output includes secondary proper motion, when available.
; Version 2.8  uses local copies of data files. Fake last lines are added to
;              these to avoid EOF errors.
; Version 2.9  reads "M" same as an "O" in column 80. This will eventually give
;              way to reading it as an "O" and a "L" once the "L" code is ready
;              for data requests.
; Version 3.0  Reads new format of orbit ephemeris file (when did it change? I
;              don't know). Correctly interpret dates in summary line.

print, ' '
print, ' DATAREQUEST.PRO: WDS (and associated catalogs) data request software'
print, '      Version 3.0  (03/29/04)'
print, ' '

;
; set up arrays for checking references (algorithm by wih)
;

refflag   = strarr(20000)
refcode2  = strarr(20000)
ref1      = strarr(20000)
ref2      = strarr(20000)
tmpcode1  = ' '
tmpcode2  = ' '
tmpref1   = ' '
tmpref2   = ' '
for i=0,19999 do begin
   refflag[i] = '0'
   endfor

iref = -1
openr,4,path+'/wds/wdsref.memo'

while (not EOF(4)) do begin
   readf,4,tmpcode2,tmpref1,tmpref2,format='(a8,9x,a37,a60)'
   if (tmpcode1 ne '        ') then begin
      iref = iref+1
      refcode2[iref] = tmpcode2
      ref1[iref]     = tmpref1
      ref2[iref]     = tmpref2
      endif
   endwhile
close,4

; initialize some variables

wdscoord = string(10)
coord    = string(10)
dd       = string(7)
stuff    = string(86)

; read,' Enter the WDS coordinate of the system : ',wdscoord    ; commented by CAH
wdscoord=dummy                                                  ; added by CAH

; determine which WDS data file to open

     hour      = strmid(wdscoord,0,2)
     halfhr    = strmid(wdscoord,2,1)
     halfhrchk = fix(strmid(wdscoord,2,1))
     
     if (halfhrchk ge 3) then halfhr = '5'
     if (halfhrchk lt 3) then halfhr = '0'
 
filein = path+'/wds/wds'+hour+halfhr+'.data'
fileout1 = 'junk'+wdscoord+'.delete_me'
fileout2 = 'wds'+wdscoord+'.measures'
     
; read the wds data into the file "'junk'+wdscoord+'delete_me'"

openr,1,filein
openw,2,fileout1

while (not EOF(1)) do begin
   readf,1,coord,dd,stuff,format='(a10,a7,a86)'
   if (wdscoord eq coord) then printf,2,coord,dd,stuff, $
        format='(a10,a7,a86)'
endwhile

close,1
close,2

; read and format the measures into the file "'wds'+wdscoord+'.measures'"

openr,2,fileout1
openw,3,fileout2

printf,3,' '
printf,3,wdscoord,format='("WDS Star No. ",a10)'
printf,3,'======================='
printf,3,' '

; define a bunch of variables

coord    = string(10)    &     dd       = string(7)     &     comp     = string(5)
;date1    = integer(4)    &     date2    = integer(4)    &     
n1       = string(1)
n2       = string(1)     &     pa1      = string(3)     &     pa2      = string(3)
sep1     = string(3)     &     n3       = string(1)     &     n4       = string(2)
n5       = string(1)     &     n6       = string(1)     &     maga1    = string(2)
maga2    = string(2)     &     magb1    = string(2)     &     magb2    = string(2)
n7       = string(1)     &     n8       = string(5)     &     n9       = string(2)
n10      = string(1)     &     n11      = string(1)     &     n12      = string(1)
n13      = string(2)     &     n14      = string(4)     &     dm       = string(8)
n15      = string(1)     &     n16      = string(1)     &     n17      = string(1)
notcoord = string(10)    &     dmmeth   = string(1)     &     notdd    = string(7)
notnot   = string(62)    &     orbcoord = string(10)    &     orbdata1 = string(67)
grade    = fix(1)        &     orbnot   = string(1)     &     orbref   = string(8)
notstat  = string(3)     &     orbstat  = string(3)     &     dmstat   = string(3)
dmcoord  = string(10)    &     name     = string(12)    &     dm2      = string(5)
numdm    = string(2)     &     dmref    = string(8)     &     dmnote   = string(24)
notstat  = 'no '         &     orbstat  = 'no '         &     dmstat   = 'no '
n15      = ' '           &     n16      = ' '           &     n17      = ' '
note     = '0'           &     orbit    = '0'           &     deltam   = '0'
spacer   = ' '           &     fakeread = string(10)    &     cgrade   = string(3)
e0       = string(5)     &     e1       = string(6)     &     e2       = string(5)
e3       = string(6)     &     e4       = string(5)     &     e5       = string(6)
e6       = string(5)     &     e7       = string(6)     &     e8       = string(5)
e9       = string(6)     &     orbinfo  = string(1)     &     notnot2  = string(70)
n18      = string(2)     &     n19      = string(2)     &     n20      = string(2)
n21      = string(1)     &     n22      = string(3)     &     n23      = string(2)
n24      = string(2)     &     n18      = '  '          &     n19      = '  '
n20      = '  '          &     n21      = ' '           &     n22      = '   '
n23      = '  '          &     n24      = '  '          &     notref   = string(8)
datlin   = string(40)    &     aper     = string(3)     &     meth     = string(1)
code1    = string(1)     &     code2    = string(1)     &     newref   = string(8)
n25      = string(4)     &     n26      = string(4)     &     n25      = '    '
n26      = '    '    
;
;sample of BIG read
;
;000000000111111111122222222223333333333444444444455555555556666666666777777777788
;123456789.123456789.123456789.123456789.123456789.123456789.123456789.123456789.1
;09123+1500FIN 347Aa    95999898156301  01  01 72  72 G8V      -526+246+15 2003NOD0912176+145946
;09123+1500FIN 347Aa    95999887156301  01  01 72  72 G8V      -526+246+15 2003NOD
;09123+1500       Aa   19840609 1427    0165          1MSN0558S8 5               
;09123+1500       Aa    1984.0609 142.7    0.165   .     .    1 Msn1996a158S  5  
;<-coord--><-dd--><com><d1><d)12<p><p><s>3<>56<><><><>7<no8><>012<><14><--dm-->56788990012223344
;

; add summary and measures to .measures file
 
while (not EOF(2)) do begin
   readf,2,coord,dd,comp,date1,date2,n1,n2,pa1,pa2,sep1,n3,n4,n5,n6,maga1, $
      maga2,magb1,magb2,n7,n8,n9,n10,n11,n12,n13,n14,dm,n15,n16,n17,n18,   $
      n19,n20,n21,n22,n23,n24,n25,n26,datlin,newref,aper,meth,code1,code2, $
      format='(a10,a7,a5,i4,i3,a1,a1,a3,a3,a3,a1,a2,a1,a1,a2,a2,a2,a2,a1,a5,a2,a1,a1,a1,a2,a4,a8,a1,a1,a1,a2,a2,a2,a1,a3,a2,a2,a4,a4,t24,a40,a8,a3,a1,a1,a1)'
   if (dd ne '       ') then begin
      if (n15 eq 'N') then note = '1'
      if (n16 eq 'O') then orbit = '1'
      if (n16 eq 'M') then orbit = '1'
      if (n17 eq 'D') then deltam = '1'
      notstat = 'no '
      orbstat = 'no '
      dmstat = 'no '
      if (n15 eq 'N') then notstat = 'yes'
      if (n16 eq 'O') then orbstat = 'yes'
      if (n16 eq 'M') then orbstat = 'yes'
      if (n17 eq 'D') then dmstat = 'yes'
      if (date1 lt 50) then date1 = date1+2000
      if (date1 le 999) then date1 = date1+1000
      if (date2 lt 50) then date2 = date2+2000
      if (date2 le 999) then date2 = date2+1000
      spacer =' '
      if (n1 eq '9') and (n2 eq '9') then spacer ='>'
      printf,3,' '
      printf,3,' '
      printf,3,' RA & DEC  Disc.   Comp   No.  Magnitudes   Spec.    PM/1000 yr  DM No.' 
      printf,3,'  (2000)   Number         Obs.   A     B    Type      RA   DEC'
      printf,3,coord,dd,comp,spacer,n1,n2,maga1,maga2,magb1,magb2,n7,n8,n9,n11,n12,n13,n14,dm, $
         format='(a10,1x,a7,1x,a5,2x,a1,a1,a1,2x,a2,".",a2,1x,a2,".",a2,2x,a1,a5,a2,1x,a1,a1,a2,1x,a4,3x,a8)'
      printf,3,' '
      if (n18 ne '  ') then printf,3,'Precise Position of Primary of System or Subsystem (when available)'
      if (n18 ne '  ') then printf,3,n18,n19,n20,n21,n22,n23,n24, $
         format='("       RA = ",a2," ",a2," ",a2,".",a1,"       Dec = ",a3," ",a2," ",a2," ")'
      if (n18 ne '  ') then printf,3,' '
      if (n25 ne '   ') then printf,3,'Proper Motion (PM/1000 yr) of Secondary of Subsystem (when available)'
      if (n25 ne '   ') then printf,3,n25,n26, $
         format='("       RA = ",a4,"             Dec = ",a4," ")'
      if (n25 ne '   ') then printf,3,' '
      printf,3,notstat,orbstat,dmstat,format='("Note? = ",a3,3x,"Orbit? = ",a3,3x,"DM? = ",a3)'
      printf,3,' '
      printf,3,'          Date Observed     Position Angle     Separation' 
      printf,3,date1,pa1,sep1,n3,format='("First",9x,i4,t15,"1",19x,a3,12x,a3,".",a1)'
      printf,3,date2,pa2,n4,n5,n6,format='("Last",10x,i4,16x,a3,12x,a2,a1,".",a1)'
      printf,3,' '
      printf,3,'Observations'
      printf,3,'============'
      printf,3,'Date       P.A.     Sep. Mag-a Mag-b  #    RefCode  Aperture  Method  Codes'
      endif else begin
       printf,3,datlin,newref,aper,meth,code1,code2,format='(a40,3x,a8,3x,a3,8x,a1,7x,a1,1x,a1)'
         for i = 0,iref do begin
            if (newref eq refcode2[i]) then refflag[i] = '1'
            endfor
      endelse
   endwhile

; add notes to .measures

if (note eq '1') then begin
   printf,3,' '
   printf,3,'WDS Index Catalog Notes'
   printf,3,'======================='
   openr,5,path+'/wds/wdsnot.memo'
   while (not EOF(5)) do begin
      readf,5,notcoord,notdd,notnot,format='(a10,a7,1x,a62)'
      if (notcoord eq coord) then begin
         printf,3,notnot,format='(a62)'
NEXTREAD:
         readf,5,notcoord,notdd,notnot,format='(a10,a7,1x,a62)'
         if (notcoord ne coord) and (notcoord ne '          ') then goto, OUT
            printf,3,notnot,format='(a62)'
            goto, NEXTREAD
         endif
      endwhile
OUT:
      close,5
   endif

; add orbits to .measures

if (orbit eq '1') then begin
   printf,3,' '
   printf,3,'6th Orbit Catalog Orbits'
   printf,3,'========================'
   openr,6,path+'/wds/orb6.master'
   readf,6,fakeread,format='(a10)'
   readf,6,fakeread,format='(a10)'
   readf,6,fakeread,format='(a10)'
   readf,6,fakeread,format='(a10)'
   while (not EOF(6)) do begin
      readf,6,orbcoord,orbdata1,cgrade,orbinfo,orbnot,orbref,format='(19x,a10,51x,a67,21x,a3,1x,a1,1x,a1,1x,a8)'
      if (orbcoord eq coord) and (orbnot eq 'y') then begin
         printf,3,'     P        a          i      Node        T       e      omega   G Refcode'
ORBREAD1:
         grade = fix(cgrade)
         printf,3,orbdata1,grade,orbref,format='(a67,i1,1x,a8)'
         for i = 0,iref do begin
            if (orbref eq refcode2[i]) then refflag[i] = '1'
            endfor
ORBREAD2:
         readf,6,orbcoord,orbdata1,cgrade,orbinfo,orbnot,orbref,format='(19x,a10,51x,a67,21x,a3,1x,a1,1x,a1,1x,a8)'
         if (orbcoord eq coord) and (orbnot eq 'y') then goto, ORBREAD1
         if (orbcoord eq coord) and (orbnot eq 'n') then goto, ORBREAD2
         if (orbcoord eq '          ') and (orbnot eq ' ') then goto, ORBREAD2
         endif
      endwhile
      close,6
   endif

; add ephemerides to file

if (orbit eq '1') then begin
   printf,3,' '
   printf,3,'6th Orbit Catalog Ephemerides'
   printf,3,'-----------------------------'
   openr,8,path+'/wds/orb6.ephem'
   readf,8,fakeread,format='(a10)'
   readf,8,fakeread,format='(a10)'
   readf,8,fakeread,format='(a10)'
   readf,8,fakeread,format='(a10)'
   while (not EOF(8)) do begin
      readf,8,orbcoord,orbref,e0,e1,e2,e3,e4,e5,e6,e7,e8,e9, $
         format='(a10,15x,a8,4x,a5,1x,a6,3x,a5,1x,a6,3x,a5,1x,a6,3x,a5,1x,a6,3x,a5,1x,a6)'
      if (orbcoord eq coord) then begin
         printf,3,'Refcode   Theta  Rho    Theta  Rho    Theta  Rho    Theta  Rho    Theta  Rho'
         printf,3,'              2004          2005          2006          2007          2008'
FEMREAD:
         printf,3,orbref,e0,e1,e2,e3,e4,e5,e6,e7,e8,e9, $
            format='(a8,2x,a5,1x,a6,2x,a5,1x,a6,2x,a5,1x,a6,2x,a5,1x,a6,2x,a5,1x,a6)'
         readf,8,orbcoord,orbref,e0,e1,e2,e3,e4,e5,e6,e7,e8,e9, $
            format='(a10,13x,a8,4x,a5,1x,a6,3x,a5,1x,a6,3x,a5,1x,a6,3x,a5,1x,a6,3x,a5,1x,a6)'
         if (orbcoord eq coord) then goto, FEMREAD
         endif
      endwhile
      close,8
   endif

; add orbit catalog notes to file

if (orbit eq '1') then begin
   openr,6,path+'/wds/orb6.master'
   readf,6,fakeread,format='(a10)'
   readf,6,fakeread,format='(a10)'
   readf,6,fakeread,format='(a10)'
   readf,6,fakeread,format='(a10)'
   while (not EOF(6)) do begin
      readf,6,orbcoord,orbdata1,cgrade,orbnot,orbinfo,orbref,format='(19x,a10,51x,a67,21x,a3,1x,a1,1x,a1,1x,a8)'
      if (orbcoord eq coord) and (orbinfo eq 'n') then begin
         openr,9,path+'/wds/orb6.notes'
         readf,9,fakeread,format='(a10)'
         readf,9,fakeread,format='(a10)'
         while (not EOF(9)) do begin
            readf,9,notcoord,notnot2,notref,format='(a10,10x,a70,a8)'
            if (notcoord eq coord) then begin
               printf,3,' '
               printf,3,'6th Orbit Catalog Notes'
               printf,3,'-----------------------'
               printf,3,notnot2,format='(a70)'
NEXTREAD2:
               readf,9,notcoord,notnot2,notref,format='(a10,10x,a70,a8)'
               if (notcoord ne coord) and (notcoord ne '          ') then goto, OUT2
               printf,3,notnot2,format='(a70)'
               if (notref eq '        ') then goto, NEXTREAD2   
                  for i = 0,iref do begin
                     if (notref eq refcode2[i]) then refflag[i] = '1'
                     endfor
               goto, NEXTREAD2
              endif
           endwhile        
         endif
         close,9
      endwhile
OUT2:
   close,6
   endif

; add delta-m to .measures

if (deltam eq '1') then begin
   printf,3,' '
   printf,3,'Delta-M Catalog Measures'
   printf,3,'========================'
   openr,7,path+'/wds/deltam.txt'
   while (not EOF(7)) do begin
      readf,7,dmcoord,name,dm2,numdm,dmref,dmmeth,dmnote,format='(a10,1x,a12,a5,2x,a2,1x,a8,1x,a1,1x,a24)'
      if (dmcoord eq coord) then begin
      if (numdm eq '  ') then numdm = ' ?'
         printf,3,'Name & Comp     DM    Method   N    Notes (e.g. filter, error)       Refcode'
DMREAD:
         if (numdm eq '  ') then numdm = ' ?'
         printf,3,name,dm2,dmmeth,numdm,dmnote,dmref,format='(a12,3x,a5,4x,a1,6x,a2,3x,a24,9x,a8)'
         for i = 0,iref do begin
            if (dmref eq refcode2[i]) then refflag[i] = '1'
            endfor
         readf,7,dmcoord,name,dm2,numdm,dmref,dmmeth,dmnote,format='(a10,1x,a12,a5,2x,a2,1x,a8,1x,a1,1x,a24)'
         if (dmcoord eq coord) then goto, DMREAD
         endif
      endwhile
      close,7
   endif

; add references to .measures

printf,3,' '
printf,3,'WDS References'
printf,3,'reference code       Author(s)/Reference'
printf,3,'========================================'
for i=0,iref do begin
   if (refflag[i] eq '1') then printf,3,refcode2[i],ref1[i],ref2[i], $
      format='(a8,13x,a37,/,21x,a60)'
   endfor

   close,2
   close,3
   return
   end
;-------------------------------------------------------------------------------
pro get_bcr,starids
;
; Obsolete!
;
; Read bad calibrator registry list (CSV format), calibrators.bcr.
;
common StarBase,StarTable,Notes
;
s=dc_read_fixed(!oyster_dir+'starbase/calibrators.bcr',l,/col, $
	format='(a300)')
;
n=n_elements(l)-1
l=l[1:n]	; remove header line
hdn=lonarr(n)
model=strarr(n)
;
for i=0,n-1 do begin
	v=nameparse(l[i],',')
	hdn[i]=cri(v[0],'hdn')
	model[i]=system_id(v[6])
	case v[7] of
	'MIDI': model[i]=model[i]+'/MIDI-N-PRISM'
	  else:
	endcase
endfor
;
get_startable,'HDN'+string(hdn,format='(i6.6)')
;
startable.model=model
;
end
;************************************************************************Block 4
pro hdn_startable,targets
;
; OYSTER uses the SkyCat 2.0 (Sky Publishing) for HDN star IDs, but HD numbers
; not found here (often the case with MIDI calibrators) have to be added
; manually to HDN.xdr. This procedure gets the basic information from simbad
; and stores it into a startable. Use edit_startable to check results, and 
; update_cat to add the entries to HDN.xdr. Targets are resolved by SimBad.
;
common StarBase,StarTable,Notes
;
n=n_elements(targets)
if n eq 0 then return
;
; Make sure the ID comes with the catalog name
index=where(strpos(targets,'HD') eq -1,count)
if count gt 0 then begin
	print,'Please include catalog name in target names, e.g., HDN123456!'
	return
endif
;
create_startable,targets
hdn=intarr(n)
stars=strarr(n)
;
print,'Number of targets to send to SimBad: ',n
;
success=0
;
for i=0,n-1 do begin
	print,'Retrieving info for target #',i+1
;	print,format='(%"Retrieving info for target: %s\r")',targets(i)
	star=cri_simbad(targets[i],t)
;	starid=cri_simbad(startable(i).name,t)
	if strlen(star) ne 0 then begin
		table=startable[i]
		struct_assign,t,table
		startable[i]=table
		startable[i].name=star
		success=1
	endif else begin
		print,'Warning (hdn_startable): not found:',targets[i]
	endelse
;	if strpos(starid,'HD') ge 0 then begin
;		startable(i).hdn=t.hdn
;		startable(i).starid='HDN'+string(startable(i).hdn,format='(i6.6)')
;		table=startable(i)
;		struct_assign,t,table
;		startable(i)=table
;		words=nameparse(targets(i))
;		startable(i).name=strjoin(words,'_')
;	endif
endfor
;
if success then begin
	print,'Use edit_startable to check/correct results for each star.'
	print,'If no HD number is listed, the object was not found!'
	print,'If not found, run procedure usr_startable instead.'
	print,'If found, use update_cat to add star to catalogs/hdn/HDN.xdr'
endif
;
end
;-------------------------------------------------------------------------------
pro usr_startable,targets,names
;
; This procedure gets the basic information for targets from simbad and
; stores it into the star table along with their common names, if found. 
; Use edit_startable to check and update_cat to add/update the entries 
; to/in USR.xdr.
;
; e.g:
; targets='Cl* Westerlund 1 W 9'
; name='W9'
; e.g.:
; targets='V* T CrA'
; name='TCrA'
;
; As of 2018-07-25: (USR 0001 to 0015)
; As of 2024-09-30: (USR 0001 to 0016)
;
; Simbad identifier		2MASS identifier
;------------------------------------------------------------------------
; [NS2003] 9A:						(NGC 3603, IRS 9A)
; RU Lup:			J15564230-3749154
; V* V334 Sgr:			J17523269-1741080 	(Sakurai's object)
; IRC +10420:			J19264809+1121167
; RY Cet:			J02160009-2031108
; V* V1974 Cyg:			J20303161+5237513	(Nova Cyg 1992)
; V* V339 Del:						(Nova Del 2013)
; Cl* Westerlund 1 W 9: 	J16470414-4550312
; Cl* Westerlund 1 W 20:	J16470468-4551238
; Cl* Westerlund 1 W 26:	J16470540-4550367
; Cl* Westerlund 1 W 237:	J16470309-4552189
; UCAC4_410-067619:		J16493775-0809334
; TYC_5058-927-1:		J16470868-0535237
; V* V1349 Cen:			J14262809-6002109
; Herschel 36:			J18034033-2422427
; V* S CrA A:			J19010860-3657200	
; V* S CrA B:			J19010860-3657200	
; WR 48a			J1312398 -624255
; WR 70-16:			J16005047-5142449	(Beauty Contest 2020)
; V921 Sco:			J16590677-4242083
; V*TCrA:			J19015878-3657498
; V*UYSct:			J18273652-1227589
; BD+11 2724 (WIP)
;
; Please also see/update oyster/starbase/vlti.usr
;
common StarBase,StarTable,Notes
;
; Read already existing USR.xdr to avoid assigning existing IDs
get_cat,'USR'
startable_usr=startable
ra0=startable_usr.ra*15*(!pi/180)	; convert to radians
dec0=startable_usr.dec*(!pi/180)
id0_max=max(fix(strmid(startable_usr.starid,3,4)))
;
n=n_elements(targets)
m=n_elements(names)
if n eq 0 then return
if m eq 0 then names=targets
if n_elements(names) ne n then begin
	print,'Error: number of targets and their names must be equal!'
	return
endif
get_startable,targets,/quiet
stars=strarr(n)
;
k=0
success=0
;
for i=0,n-1 do begin
	star=cri_simbad(strcompress(targets[i],/remove_all),t)
	if strlen(star) eq 0 then star=cri_simbad(names[i],t)
	if strlen(t.name) ne 0 then begin
		if strlen(names[i]) ne 0 then t.name=names[i]
		table=startable[i]
		struct_assign,t,table
		startable[i]=table
		ra1=startable[i].ra*15*(!pi/180)	; convert to radians
		dec1=startable[i].dec*(!pi/180)
		dist=sqrt(((ra0-ra1)*cos(dec0))^2 $
			  +(dec0-dec1)^2)*(180/!pi)*3600
		j=where(dist eq min(dist))
		if min(dist) lt 1 then begin
			startable[i].starid=startable_usr[j].starid 
		endif else begin
			k=k+1
			id=id0_max+k
			startable[i].starid='USR'+string(id,format='(i4.4)')
		endelse
		print,'Star ',star,' found in SimBad'
		success=1
	endif
endfor
;
if success then begin
	print,'Use edit_startable to check/correct results for each star.'
	print,'Use update_cat to add star to catalogs/usr/USR.xdr'
	print,'Add StarID to starbase/vlti.usr'
endif else begin
	print,'No target found with this name in SimBad!'
endelse
;
end
;-------------------------------------------------------------------------------
pro edit_startable,starid
;
; Display widget for updating individual fields for specified
; star. If new, then add this star and display widget.
;
common StarBase,StarTable,Notes
;
print,'Modify cell contents and click OK to save, else Cancel'
print,'Important: Star ID and catalog ID must agree!'
;
if n_elements(starid) eq 0 and n_elements(startable) eq 1 then begin
	starid=startable[0].starid
endif else begin
	add_startable,starid	; Will create or add star to startable
endelse
index=where(startable.starid eq starid,count)
;
t=startable[index]
t=t[0]
;
desc=strarr(n_tags(t)+3)
desc=strarr(100)
desc[0]='0,LABEL,Startable['+string(index[0],format='(i4)')+'],CENTER'
desc[1]='1,BASE,,ROW,FRAME'
fields=tag_names(t)
k=2
imod=16
for i=0,n_tags(t)-1 do begin
	depth=string(0,format='(i1.1)')
	result=size(t.(i))
	case result[n_elements(result)-2] of
		2:item='INTEGER'
		3:item='INTEGER'
		4:item='FLOAT'
		5:item='FLOAT'
		7:item='TEXT'
	endcase
	if n_elements(t.(i)) eq 1 then begin
	if i mod imod eq 0 then begin
		desc[k]='1,BASE,,COLUMN,FRAME'
		k=k+1
		desc[k]='0,'+item+','+string(t.(i))+ $
			',LABEL_LEFT='+fields[i]+',WIDTH=12,TAG='+fields[i]
		k=k+1
	endif else begin
		if (i+1) mod imod eq 0 then depth=string(2,format='(i1.1)')
		desc[k]=depth+','+item+','+string(t.(i))+ $
			',LABEL_LEFT='+fields[i]+',WIDTH=12,TAG='+fields[i]
		k=k+1
	endelse
	endif
endfor
desc[k]='1,base,,row'
k=k+1
desc[k]='0,button,OK,QUIT,tag=OK'
k=k+1
desc[k]='2,BUTTON,Cancel,QUIT'
desc=desc[0:k]
;
t=CW_FORM(desc,/column)
;
if t.ok then begin
	k=0
	for i=0,n_tags(startable)-1 do begin
	if n_elements(startable[index[0]].(i)) eq 1 then begin
		startable[index].(i)=t[0].(k)
		k=k+1
	endif
	endfor
endif
;
end
;-------------------------------------------------------------------------------
pro add_startable,starids
;
; Non-destructive add of starids to startable. If some starids exist, they
; will be left untouched. If no startable exists yet, it will be created.
;
common StarBase,StarTable,Notes
;
if n_elements(starids) eq 0 then return
;
if n_elements(startable) ne 0 then table0=startable
get_startable,starids
if n_elements(table0) eq 0 then return
table1=startable
;
; Create combined startable
create_startable,unique([table0.starid,table1.starid])
;
; Fill info, giving priority to existing entries
for i=0,n_elements(startable)-1 do begin
	index0=where(table0.starid eq startable[i].starid,count0)
	index1=where(table1.starid eq startable[i].starid,count1)
	if count0 eq 1 then t=table0[index0] else t=table1[index1]
	for j=0,n_tags(startable)-1 do startable[i].(j)=t[0].(j)
endfor
;
end
;-------------------------------------------------------------------------------
function valid_cat,cat_id_in
;
; Check if a catalog or starID corresponds to a valid primary catalog.
;
cat_id=strmid(cat_id_in,0,3)
;
case cat_id of
	'FKV':valid=1
	'BSC':valid=1
	'HDN':valid=1
	'HIP':valid=1
	'HIC':valid=1
	'SAO':valid=1
	'FLN':valid=1
	'USR':valid=1	; added March 2022, needs checking!
	else :valid=0
endcase
;
return,valid
;
end
;
;-------------------------------------------------------------------------------
pro update_cat,starids
;
; Update a secondary catalog file (only one at a time is allowed) with 
; information found in the startable for the specified starids. If a starid
; is not found in the catalog, it is added, otherwise it is updated.
; If starids are not defined, the entire startable is used for the update.
;
; Note: the catalog, e.g. USR, is extracted from the first three letters
; of the StarID, and therefore the only valid catalogs for this function
; are currently: USR, HDN, and CAL.
;
common StarBase,StarTable,Notes
;
allowed_cats=['USR','HDN','CAL']
;
if n_elements(startable) eq 0 then begin
	print,'***Error(UPDATE_CAT): startable does not exist!'
	return
endif
if n_elements(starids) eq 0 then starids=startable.starid
catids=unique(strmid(starids,0,3))
if n_elements(catids) gt 1 then begin
	print,'***Error(UPDATE_CAT): cannot update more than one catalog!'
	return
endif
catid=catids[0]
index=where(allowed_cats eq catid,n)
if n ne 1 then begin
	print,'***Error(UPDATE_CAT): catalog not allowed: '+catid
	return
endif
;
table0=startable				; startable with new star IDs
get_cat,catid					; load requested catalog
startable=merge_startable(startable,table0)	; merge the 2 tables
;
; In case of binaries with composite spectral classes, attempt to split them
spec_parse
;
; Save copy of current StarTable (xdr format)
f_stub=!catalogs_dir+strlowcase(catid)+'/'+catid
file_copy,f_stub+'.xdr',f_stub+'_'+jd2date(systime(/julian))+'.xdr',/overwrite
;
; Save new StarTable
save,startable,filename=f_stub+'.xdr'
;
print,'Catalog '+catid+' updated. Number of entries:' $
	+string(n_elements(startable))
print,'Backup of previous version made.'
;
end
;-------------------------------------------------------------------------------
pro assign_ids,starids
;
; For the specified starids, which must all refer to a single catalog and must
; be found in the startable, check whether they are already present in the
; catalog file and if not, assign unused starids to them.
;
common StarBase,StarTable,Notes
;
catids=unique(strmid(starids,0,3))
if n_elements(catids) gt 1 then begin
	print,'***Error(CHECK_CAT): cannot check more than one catalog!'
	return
endif
table0=startable	; this table is not updated
table1=startable	; this table is updated
get_cat,catids[0]
cat_ids=startable.starid
cat_nms=startable.name
;
for i=0,n_elements(starids)-1 do begin
	index0=where(table0.starid eq starids[i],count0)
	if count0 eq 0 then begin
		print,'***Error(ASSIGN_IDS): star not found in startable!'
		startable=table0
		return
	endif
	index=where(cat_ids eq starids[i],count)
	if count eq 1 then begin
		if cat_nms[index] ne table0[index0].name then begin
;			Same ID, but different names => find new ID
			new_id=max(long(strmid(cat_ids,3,4)))+1
			starids[i]=strmid(starids[i],0,3)+string(new_id,format='(i4.4)')
			table1[index0].starid=starids[i]
			cat_ids=[cat_ids,starids[i]]
			cat_nms=[cat_nms,table1[index0].name]
		endif
;		Same ID and same name => same star, no need for new ID
	endif 	
	index=where(cat_nms eq table0[index0].name,count)
	if count eq 1 then begin
		starids[i]=cat_ids[index]
		table1[index0].starid=starids[i]
	endif
endfor
;
end
;************************************************************************Block 5
pro write_aspro
;
; Write star table to file conforming to ASPRO object list format
;
common StarBase,StarTable,Notes
;
openw,unit,'oipt_sources.sou',/get_lun
;
printf,unit,'!---------------------------------------------------------------------'
printf,unit,'!  File oipt_sources.sou'
printf,unit,'!---------------------------------------------------------------------'
;
for i=0,n_elements(startable)-1 do begin
	source='            '
	if strlen(strcompress(startable[i].name,/remove_all)) gt 0 $
		then strput,source,strcompress(startable[i].name) $
		else strput,source,startable[i].starid
	words=nameparse(source)
	for j=1,n_elements(words)-1 do words[0]=words[0]+'_'+words[j]
	source='            '
	strput,source,words[0]
	printf,unit,source+' EQ  '+'2000.000'+ $
		hms(startable[i].ra,/aspro)+dms(startable[i].dec,/aspro)+ $
		'  MV  '+string(startable[i].mv,format='(f5.2)')
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_calvin
;
; Write startable to file conforming to CalVin data base format
;
common StarBase,StarTable,Notes
;
n=n_elements(startable)
;
names=strarr(n)
for i=0,n-1 do begin
	cat=strmid(startable[i].starid,0,3)
	if cat eq 'HDN' $
	or cat eq 'CAL' then starid='HD'+strmid(startable[i].starid,3,6) $
			else starid=startable[i].starid
	name='          '
	strput,name,starid
	names[i]=name
endfor
ra=hms(startable.ra,/aspro)
hhr=strmid(ra,0,2)
mmr=strmid(ra,3,2)
ssr=strmid(ra,6,5)
dec=dms(startable.dec,/aspro)
si=strmid(dec,0,1)+'1'
ddd=strmid(dec,1,2)
mmd=strmid(dec,4,2)
ssd=strmid(dec,7,4)
epoch='2000'
; V
mv=string(startable.mv,format='(f6.2)')
i=where(startable.mv eq 100,c)
if c gt 0 then mv[i]='-99e99'
; R
mr=strarr(n)+'-99e99'
; J
mj=string(startable.mj,format='(f6.2)')
i=where(startable.mj eq 100,c)
if c gt 0 then mj[i]='-99e99'
; H
mh=string(startable.mh,format='(f6.2)')
i=where(startable.mh eq 100,c)
if c gt 0 then mh[i]='-99e99'
; H
mk=string(startable.mk,format='(f6.2)')
i=where(startable.mk eq 100,c)
if c gt 0 then mk[i]='-99e99'
; L
ml=strarr(n)+'-99e99'
; M
mm=strarr(n)+'-99e99'
; N
mn=string(jy2n(startable.f12),format='(f6.2)')
i=where(startable.f12 eq 0,c)
if c gt 0 then mn[i]='-99e99'
; Q
mq=strarr(n)+'-99e99'
; Spectrum
spectra=strarr(n)
for i=0,n-1 do begin
	spectrum='          '
	strput,spectrum,startable[i].spectrum
	spectra[i]=spectrum
endfor
spec_parse,spectra,types,classes
lc=strarr(n)+'V     '
i=where(classes eq 1,c)
if c gt 0 then lc[i]='I     '
i=where(classes eq 1.25,c)
if c gt 0 then lc[i]='Iab   '
i=where(classes eq 1.5,c)
if c gt 0 then lc[i]='Ib    '
i=where(classes eq 1.75,c)
if c gt 0 then lc[i]='I/II  '
i=where(classes ge 2 and classes lt 3,c)
if c gt 0 then lc[i]='II    '
i=where(classes eq 2.5,c)
if c gt 0 then lc[i]='II/III'
i=where(classes ge 3 and classes lt 4,c)
if c gt 0 then lc[i]='III   '
i=where(classes eq 3.5,c)
if c gt 0 then lc[i]='III/IV'
i=where(classes ge 4 and classes lt 5,c)
if c gt 0 then lc[i]='IV    '
i=where(classes eq 4.5,c)
if c gt 0 then lc[i]='IV/V  '
i=where(classes ge 5,c)
if c gt 0 then lc[i]='V     '
; F (quality flag), use the status flag
f=startable.sflag
; Diameter
theta=string(startable.diameter,format='(f5.2)')
; etheta=string(startable.diametere,format='(f5.2)')
etheta=0	; Nov 3, 2015, diametere no longer in startable
; Model and instrument
model=strarr(n)+blanks(max(strlen(startable.model)))
instrument=strarr(n)
for i=0,n-1 do begin
	m=model[i]
	strput,m,startable[i].model
	model[i]=m
	case strtrim(model[i],2) of
	'UDN':instrument[i]='MIDI   '
	'LDN':instrument[i]='MIDI   '
	'UDK':instrument[i]='AMBER'
	'LDK':instrument[i]='AMBER'
	 'LD':instrument[i]='VLTI   '
	else: instrument[i]='VLTI   '
	endcase
endfor
; Teff
teff=string(fix(startable.teff),format='(i6.6)')
i=where(startable.teff eq 0,c)
if c gt 0 then teff[i]='-99e99'
; log(g)
logg=string(startable.logg,format='(f6.2)')
i=where(startable.logg eq 0,c)
if c gt 0 then logg[i]='-99e99'
; Mass
mass=string(startable.mass,format='(f6.2)')
i=where(startable.mass eq 0,c)
if c gt 0 then mass[i]='-99e99'
; V sin(i)
vsini=string(startable.vsini,format='(f6.2)')
i=where(startable.vsini eq 0,c)
if c gt 0 then vsini[i]='-99e99'
; Radial velocity
vrad=string(startable.rv,format='(f6.2)')
i=where(startable.rv eq 0,c)
if c gt 0 then vrad[i]='-99e99'
; Parallax
pi=string(startable.px*1000,format='(f7.2)')
i=where(startable.px eq 0,c)
if c gt 0 then pi[i]=' -99e99'
; Parallax error
epi=string(startable.pxe*1000,format='(f6.2)')
i=where(startable.pxe eq 0,c)
if c gt 0 then epi[i]='-99e99'
; Proper motion in RA
pmra=string(startable.pmra*150*cos(startable.dec/(180/!pi)),format='(f7.1)')
i=where(startable.pmra eq 0,c)
if c gt 0 then pmra[i]=' -99e99'
; Proper motion in Dec
pmdec=string(startable.pmdec*10,format='(f7.1)')
i=where(startable.pmdec eq 0,c)
if c gt 0 then pmdec[i]=' -99e99'
; Z
z=strarr(n)+'-99e99'
; Reference
ref=startable.reference
;
; Open file; extension indicates that it is CALVIN format
openw,unit,'list.calvin',/get_lun
s=' '
;
models=['UDJ','UDH','UDK']
instruments=['AMBER_J','AMBER_H','AMBER_K']
wavelengths=[1210.0,1650.0,2175.0]
x=alog10(wavelengths)
;
for i=0,n-1 do begin
;
for j=0,n_elements(wavelengths)-1 do begin
u=10^(startable[i].a0+startable[i].a1*x[j]+startable[i].a2*x[j]^2)
if total([startable[i].a0,startable[i].a1,startable[i].a2]) eq 0 then u=0.0
UD=startable[i].diameter/sqrt(((1-u/3)/(1-7*u/15)))
if strpos(instrument[i],'AMBER') ge 0 then begin
	model[i]=models[j]
	instrument[i]=instruments[j]
endif
theta[i]=string(UD,format='(f5.2)')
entry=names[i]+s $
	+hhr[i]+s+mmr[i]+s+ssr[i]+s+si[i]+s+ddd[i]+s+mmd[i]+s+ssd[i]+s+epoch+s $
	+mv[i]+s+mr[i]+s+mj[i]+s+mh[i]+s+mk[i]+s+ml[i]+s+mm[i]+s+mn[i]+s+mq[i]+s $
	+spectra[i]+s+lc[i]+s+f[i]+s+theta[i]+s+etheta[i]+s+model[i]+s $
	+teff[i]+s+logg[i]+s+mass[i]+s+vsini[i]+s+vrad[i]+s+pi[i]+s+epi[i]+s $
	+pmra[i]+s+pmdec[i]+s+z[i]+s+instrument[i]+s+ref[i]
if strpos(instrument[i],'AMBER') ge 0 or j eq 0 then printf,unit,entry
;
endfor
;
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_latex
;
; Write file in LATEX format containing the star list.
;
common StarBase,StarTable,Notes
;
openw,unit,'stars.tex',/get_lun
;
header='  Star      Name       SB  BAT  FK5   '+ $
	'HR     HD   mV   mK (B-V)   D         Spectrum      '+ $
	'P [d]  ecc.  incl. a [mas] Flag'
;
printf,unit,'\documentstyle[11pt]{report}'
printf,unit,'\topmargin -3cm'
printf,unit,'\oddsidemargin -2cm'
printf,unit,'\textheight 26cm'
printf,unit,'\textwidth 21cm'
printf,unit,'\pagestyle{empty}'
printf,unit,'\begin{document}'
printf,unit,'{\tt\scriptsize\begin{verbatim}'
;
for i=0,n_elements(StarTable)-1 do begin
	if i mod 76 eq 0 then printf,unit,header
	if StarTable[i].mf ne 0 then sb=1 else sb=2
	printf,unit, $
	StarTable[i].starid,StarTable[i].name,sb, $
	StarTable[i].bat,StarTable[i].fkv, $
	StarTable[i].bsc,StarTable[i].hdn, $
	StarTable[i].mv,Startable[i].mk,StarTable[i].bv,StarTable[i].diameter, $
	strcompress(StarTable[i].spectrum,/remove_all), $
	StarTable[i].p,StarTable[i].e,StarTable[i].i,StarTable[i].a*1000., $
	StarTable[i].sflag, $
	format='(a9,1x,a12,1x,i1,2x,i4,1x,i4,1x,i4,1x,i6,1x,'+ $
		'f4.1,1x,f4.1,1x,f5.2,1x,f4.1,1x,a17,1x,f8.1,2x,f4.2,2x,f4.0,1x,f7.1,3x,a2)'
endfor
;
printf,unit,'\end{verbatim}}'
printf,unit,'\end{document}'
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_npoi
;
; Write reduced column NPOI startable to catalog file.
;
common StarBase,StarTable,Notes
;
openw,unit,'stars_npoi.txt',/get_lun
;
for i=0L,n_elements(StarTable)-1L do begin
	printf,unit,long(strmid(StarTable[i].starid,3,6)),StarTable[i].ra,StarTable[i].dec, $
		StarTable[i].mv,strcompress(StarTable[i].spectrum,/remove_all), $
		StarTable[i].pmra,StarTable[i].pmdec,StarTable[i].rv,StarTable[i].px, $
		StarTable[i].diameter,StarTable[i].name, $
		format='(i6.6,1x,f11.8,1x,f11.7,1x,'+ $
			'f4.1,1x,a14,1x,f7.3,1x,f7.2,1x,f6.1,1x,f6.3,1x,f5.1,1x,a13)'
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_at
;
; Write star table to text file with info useful for VLTI-AT observations.
;
common StarBase,StarTable,Notes
;
openw,unit,'stars_at.txt',/get_lun
;
t=startable
;
printf,unit,'!  HD   HR                RA (2000)        Dec (2000)           V    J    H   K     '
for i=0L,n_elements(t)-1L do begin
	printf,unit,t[i].hdn,t[i].bsc,t[i].name,hms(t[i].ra),dms(t[i].dec), $
		t[i].spectrum,t[i].mv,t[i].mj,t[i].mh,t[i].mk, $
		format='(i6,1x,i4,2x,a9,1x,a14,1x,a14,2x,a8,4(f4.1,1x))'
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_finito
;
; Write star table to text file with info useful for FINITO observations.
;
common StarBase,StarTable,Notes
;
openw,unit,'stars_finito.txt',/get_lun
;
t=startable
;
printf,unit,'!  HD   HR                RA (2000)        Dec (2000)           V    H    K  D [mas]'
for i=0L,n_elements(t)-1L do begin
	printf,unit,t[i].hdn,t[i].bsc,t[i].name,hms(t[i].ra),dms(t[i].dec), $
		t[i].spectrum,t[i].mv,t[i].mh,t[i].mk,t[i].diameter, $
		format='(i6,1x,i4,2x,a9,1x,a14,1x,a14,2x,a8,3(f4.1,1x),1x,f3.1)'
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_fov
;
; Write star table to text file with info useful for FINITO observations.
;
common StarBase,StarTable,Notes
;
openw,unit,'stars_fov.txt',/get_lun
;
t=startable
;
printf,unit,'!  HD   Rho [mas]         RA (2000)        Dec (2000) Spectrum  V    H    K  D [mas]'
for i=0L,n_elements(t)-1L do begin
	printf,unit,t[i].hdn,fix(t[i].a*1000),t[i].name,hms(t[i].ra),dms(t[i].dec,/dec), $
		t[i].spectrum,t[i].mv,t[i].mh,t[i].mk,t[i].diameter, $
		format='(i6,1x,i3,3x,a9,1x,a14,1x,a14,2x,a8,3(f4.1,1x),1x,f3.1)'
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_iris
;
; Write star table to text file with info useful for IRIS observations.
;
common StarBase,StarTable,Notes
;
openw,unit,'stars_iris.txt',/get_lun
;
t=startable
;
printf,unit,'!  HIP  HR                RA (2000)        Dec (2000)           V    H    K  A [as] DmV [mag]'
for i=0L,n_elements(t)-1L do begin
	printf,unit,t[i].hic,t[i].bsc,t[i].name,hms(t[i].ra),dms(t[i].dec), $
		t[i].spectrum,t[i].mv,t[i].mh,t[i].mk,t[i].a,t[i].dmv, $
		format='(i6,1x,i4,2x,a9,1x,a14,1x,a14,2x,a8,3(f4.1,1x),2(1x,f5.1))'
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro write_rdb
;
; Write star table in rdb format for use in Apes.
;
common StarBase,StarTable,Notes
;
openw,unit,'stars.rdb',/get_lun
;
t=startable
n=n_elements(t)
;
stype=strarr(n) & stype[*]='T    '
star_id=strarr(n) & star_id[*]='HD '+string(t.hdn,format='(i6.6)')
alpha=hms(t.ra,/aspro)
delta=dms(t.dec,/aspro)
coord_syst=strarr(n) & coord_syst[*]='ICRS         '
epoch=strarr(n) & epoch[*]='J2000  '
equinox=strarr(n) & equinox[*]='2000.00'
mualpha=string(fix(((t.pmra)/15)*100/cos(t.dec*!pi/180)),format='(f7.1)')
mudelta=string(fix(t.pmdec*100),format='(f7.1)')
sp_type=strarr(n)
for i=0,n-1 do begin
sp_type_s=blanks(10)
strput,sp_type_s,t[i].spectrum
sp_type[i]=sp_type_s
endfor
magk=string(t.mk,format='(f6.1)')
magv=string(t.mv,format='(f6.1)')
magh=string(t.mh,format='(f6.1)')
;
stype_s=blanks(strlen(stype[0])) & strput,stype_s,'stype'
star_id_s=blanks(strlen(star_id[0])) & strput,star_id_s,'star_id'
alpha_s=blanks(strlen(alpha[0])) & strput,alpha_s,'alpha'
delta_s=blanks(strlen(delta[0])) & strput,delta_s,'delta'
coord_syst_s=blanks(strlen(coord_syst[0])) & strput,coord_syst_s,'coord_syst'
epoch_s=blanks(strlen(epoch[0])) & strput,epoch_s,'epoch'
equinox_s=blanks(strlen(equinox[0])) & strput,equinox_s,'equinox'
mualpha_s=blanks(strlen(mualpha[0])) & strput,mualpha_s,'mualpha'
mudelta_s=blanks(strlen(mudelta[0])) & strput,mudelta_s,'mudelta'
sp_type_s=blanks(strlen(sp_type[0])) & strput,sp_type_s,'SP_type'
magK_s=blanks(strlen(magK[0])) & strput,magK_s,'magK'
magV_s=blanks(strlen(magV[0])) & strput,magV_s,'magV'
magH_s=blanks(strlen(magH[0])) & strput,magH_s,'magH'
;
stype_d=dashes(strlen(stype[0]))
star_id_d=dashes(strlen(star_id[0]))
alpha_d=dashes(strlen(alpha[0]))
delta_d=dashes(strlen(delta[0]))
coord_syst_d=dashes(strlen(coord_syst[0]))
epoch_d=dashes(strlen(epoch[0]))
equinox_d=dashes(strlen(equinox[0]))
mualpha_d=dashes(strlen(mualpha[0]))
mudelta_d=dashes(strlen(mudelta[0]))
sp_type_d=dashes(strlen(sp_type[0]))
magK_d=dashes(strlen(magK[0]))
magV_d=dashes(strlen(magV[0]))
magH_d=dashes(strlen(magH[0]))
;
printf,unit,stype_s,star_id_s,alpha_s,delta_s,coord_syst_s,epoch_s,equinox_s, $
	mualpha_s,mudelta_s,sp_type_s,magk_s,magv_s,magh_s, $
	format='(12(a,%"\t"),a)'
printf,unit,stype_d,star_id_d,alpha_d,delta_d,coord_syst_d,epoch_d,equinox_d, $
	mualpha_d,mudelta_d,sp_type_d,magk_d,magv_d,magh_d, $
	format='(12(a,%"\t"),a)'
for i=0,n-1 do printf,unit,stype[i],star_id[i], $
	alpha[i],delta[i],coord_syst[i],epoch[i], $
	equinox[i],mualpha[i],mudelta[i], $
	sp_type[i],magk[i],magv[i],magh[i], $
	format='(12(a,%"\t"),a)'
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro list_star,starids
;
; Print available information on specific star to screen. Star is identified
; by ID or name. Minimum match is implemented for names out of StarTable.name, 
; var, and toe.
;
common StarBase,StarTable,Notes
;
if n_elements(StarTable) eq 0 then begin
	print,'***Error(LIST_STAR): StarTable does not exist!'
	return
endif
if n_elements(starids) eq 0 then begin
	if n_elements(StarTable) eq 1 then begin
		list_star,startable[0].starid
	endif else begin
		print,'***Error(LIST_STARID): stars not specified!'
		return
	endelse
endif
;
FOR k=0L,n_elements(starids)-1 DO BEGIN
;
starid=strupcase(starids[k])
;
found=0
;
; Check StarIDs first
if found eq 0 then begin
	i=where(StarTable.starid eq starid[0],found)
endif
;
; Check other indices
if found eq 0 then begin
	id=strmid(starid,3,6)
	if isnumeric(id) then id=long(id) else id=0
	case strmid(starid,0,3) of
	'FKV': ids=StarTable.fkv
	'BSC': ids=StarTable.bsc
	'HDN': ids=StarTable.hdn
	'FLN': ids=StarTable.fln
	'SAO': ids=StarTable.sao
	'HIC': ids=StarTable.hic
	'HIP': ids=StarTable.hic
	else : ids=-1
	endcase
	i=where(ids eq id,found)
endif
;
; Then check names
if found eq 0 then begin
	words=strupcase(nameparse(starid))
	names=strupcase(StarTable.name+StarTable.var)
	i=lindgen(n_elements(StarTable))
	for j=n_elements(words)-1,0,-1 do begin
		index=where(strpos(names,words[j]) ne -1,count)
		if count gt 0 then begin
			names=names[index]
			i=i[index]
			found=count
			stars=names[index]
		endif
	endfor
endif
;
; Check proper names
if found eq 0 then begin
	id_bsc=cri(starid,'bsc') & id_bsc=id_bsc[0]
	if id_bsc gt 0 then i=where(StarTable.bsc eq id_bsc,found)
	if found eq 0 then begin
		id_fkv=cri(starid,'fkv') & id_fkv=id_fkv[0]
		if id_fkv gt 0 then i=where(StarTable.fkv eq id_fkv,found)
	endif
endif
;
if found eq 0 then begin
	print,'***Error(LIST_STAR): star not found!'
	return
endif
;
if n_elements(stars) eq 0 then stars=StarTable.starid
j=0
if found gt 1 then begin
	print,'Ambiguous: '
	print,strcompress(stars[i]),format='(a,T20,a,T40,a)'
	read,'Enter index (first = 1): ',j
	if j eq 0 then j=1
	if j gt found then j=found
	i=i[j-1]
	found=1
endif
;
for j=0,found-1 do begin
	print,'_____________________________________________'
	print,'Star: ',strcompress(StarTable[i[j]].starid), $
		' Name: ',strcompress(StarTable[i[j]].name), $
;		' (#',i(j),StarTable(i(j)).sflag,')', $
		' (#',i(j),')', $
		format='(a,a,a,a,a,i6.6,1x,a,a)'
	print,'Var: ',StarTable[i[j]].var,format='(a,a10)'
	print,'HDN',StarTable[i[j]].hdn,' SAO',StarTable[i[j]].sao, $
		' HIC',StarTable[i[j]].hic,' WDS',StarTable[i[j]].wds, $
		format='(a,i6.6,a,i6.6,a,i6.6,a,i10.9)'
	print,'BSC',StarTable[i[j]].bsc,' FKV',StarTable[i[j]].fkv, $
		' BAT',StarTable[i[j]].bat,' FLN',StarTable[i[j]].fln, $
		' ADS',StarTable[i[j]].ads, $
		format='(a,i4.4,a,i4.4,a,i4.4,a,i4.4,a,i5.5)'
	print,'Right ascension (2000.0) = ',hms(StarTable[i[j]].ra)
	print,'Declination     (2000.0) = ',dms(StarTable[i[j]].dec)
	print,'Centennial proper motion in  RA [s] = ', $
		StarTable[i[j]].pmra,format='(a,f7.3)'
	print,'Centennial proper motion in Dec ["] = ', $
		StarTable[i[j]].pmdec,format='(a,f7.2)'
	print,'RV [km/s] = ',StarTable[i[j]].rv, $
		', PX ["] = ',StarTable[i[j]].px,' +/- ',StarTable[i[j]].pxe, $
		format='(a,f5.1,a,f6.3,a,f6.3)'
	print,'m_V = ',StarTable[i[j]].mv,', M_V = ',StarTable[i[j]].amv, $
		', Dm_V = ',StarTable[i[j]].dmv,format='(a,f5.2,a,f5.2,a,f5.2)'
	print,'(B-V) = ',StarTable[i[j]].bv,', (U-B) = ',StarTable[i[j]].ub, $
		', (R-I) = ',StarTable[i[j]].ri,format='(a,f5.2,a,f5.2,a,f5.2)'
	print,'b-y = ',StarTable[i[j]].by,', m1 = ',StarTable[i[j]].m1, $
		', c1 = ',StarTable[i[j]].c1,', b = ',StarTable[i[j]].beta, $
		format='(a,f5.2,a,f5.2,a,f5.2,a,f4.2)'
	print,'Spectrum = ',strcompress(StarTable[i[j]].spectrum), $
		', [Fe/H] = ',StarTable[i[j]].feh,format='(a,a,a,f5.2)'
	print,'Primary   = ',StarTable[i[j]].type,StarTable[i[j]].class, $
		', Secondary = ',StarTable[i[j]].type2,StarTable[i[j]].class2, $
		format='(a,f4.1,1x,f3.1,a,f4.1,1x,f3.1)'
	print,'Diameter = ',StarTable[i[j]].diameter, $
	    ', T_eff = ',long(StarTable[i[j]].teff), $
	    ', log(g) = ',StarTable[i[j]].logg, $
		format='(a,f6.3,a,i5,a,f3.1)'
	if StarTable[i[j]].bflag eq 'B' then begin
	print,'Period = ',StarTable[i[j]].p,', Epoch = ',StarTable[i[j]].t, $
		format='(a,f11.4,a,f12.4)'
	print,'Arg.periastron = ',StarTable[i[j]].o, $
		', Eccentricity = ',StarTable[i[j]].e, $
		format='(a,f5.1,a,f4.2)'
	print,'K1 = ',StarTable[i[j]].k1,', K2 = ',StarTable[i[j]].k2, $
		format='(a,f5.1,a,f5.1)'
	print,'M1sin3i = ',StarTable[i[j]].m1sin3i,', M2sin3i = ', $
		StarTable[i[j]].m2sin3i, $
		format='(a,e7.1,a,e7.1)'
	print,'a1sini = ',StarTable[i[j]].a1sini,', a2sini = ', $
		StarTable[i[j]].a2sini,format='(a,e8.2,a,e8.2)'
	print,'I = ',StarTable[i[j]].i, $
		', W = ',StarTable[i[j]].n, $
		', A [mas] = ',1000*StarTable[i[j]].a, $
		format='(a,f4.0,a,f5.1,a,f7.1)'
	endif
	print,'____________________***______________________'
	if j lt found-1 then $ 
		hak,mesg='More than one star: hit return to continue...'
endfor
;
if k lt n_elements(starids)-1 then $
	hak,mesg='Hit return to continue...'
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro list_note,starid,note
;
; Print notes for starid to screen or return as note.
;
common StarBase,StarTable,Notes
;
if n_elements(notes) eq 0 then begin
	print,'***Error(LIST_NOTE): notes not available!'
	return
endif
;
if n_elements(starid) eq 0 then begin
	print,'***Error(LIST_NOTE): stars undefined!'
	return
endif
;
if n_params() eq 1 then do_print=1 else do_print=0
;
note=''
separator='-----------------------------'
for i=0,n_elements(starid)-1 do begin
	index=where(strpos(notes,starid[i]) ne -1 $
		and strpos(shift(notes,1),separator) ne -1,count)
	if count ne 0 then begin
		if do_print then print,'_____________________________________________'
		j=index[0]
		while strpos(notes[j],separator) eq -1 do begin
			if do_print then print,notes[j]
			j=j+1
		endwhile
		if do_print then print,'____________________***______________________'
		note=[note,notes[index[0]:j-1]]
	endif
endfor
;
if n_elements(note) gt 1 then note=note[1:n_elements(note)-1]
;
end
;-------------------------------------------------------------------------------
pro list_notes,starids
;
; Return a list of stars for which notes are available.
;
common StarBase,StarTable,Notes
;
if n_elements(notes) eq 0 then begin
	print,'***Error(LIST_NOTES): notes not available!'
	return
endif
;
index=where(strpos(notes,'---------------------------') ne -1,count)
if count eq 0 then begin
	print,'***Error(LIST_NOTES): file corrupt!'
	return
endif
index=index+1
jndex=where(index lt n_elements(notes),count)
if count eq 0 then begin
	print,'***Error(LIST_NOTES): file corrupt!'
	return
endif
index=index[jndex]
starids=strarr(count)
for i=0,count-1 do begin
	pos=strpos(notes[index[i]],' ')
	if pos eq -1 then pos=strlen(notes[index[i]])
	starids[i]=strmid(notes[index[i]],0,pos)
endfor
;
if n_params() eq 0 then print,starids
;
end
;-------------------------------------------------------------------------------
pro list_category,category,starids
;
; Return list of stars for which a note was found with in the given category
;
list_notes,starids
;
num_star=n_elements(starids)
if num_star eq 0 then return
index=intarr(num_star)
;
for i=0,n_elements(starids)-1 do begin
	list_note,starids[i],note
	list_keyword,note,'CATEGORY',value
	if value[0] eq category then index[i]=1
endfor
;
if total(index) ne 0 then starids=starids[where(index ne 0)]
;
if n_params() eq 1 then print,starids
;
end
;-------------------------------------------------------------------------------
pro list_keyword,note,keyword,value
;
; Given a note, derive the value of the keyword
;
index=where(strpos(note,keyword) ne -1,count)
if count gt 0 then begin
	value=strarr(count)
	for i=0,count-1 do begin
		pos1=strpos(note[index[i]],'(')
		pos2=strpos(note[index[i]],')')
		value[i]=strmid(note[index[i]],pos1+1,pos2-pos1-1)
	endfor
endif
;
if n_params() eq 2 then print,value
;
end
;************************************************************************Block 6
pro rename_starids,request
;
; Rename stars to new catalogue system. If given a list of stars, rename
; starids in startable to match the names using existing ID numbers.
;
common StarBase,StarTable,Notes
;
starids=StarTable.starid
starids_bck=starids
;
; If request is an array, it is a list of stars to be homogenized
if n_elements(request) gt 1 then begin
   ustars=unique(request) 
   ustids=unique(startable.starid)
;  FKV0000 is "alpha lab"
   index=where(ustars ne 'FKV0000',count)
   if count gt 0 then begin
      ustars=ustars[index]
      ustids=ustids[index]
      for i=0,n_elements(ustars)-1 do begin
         if ustars[i] ne ustids[i] then begin
	   case strmid(ustars[i],0,3) of
	   'FKV':StarTable[where(StarTable.fkv $
		   eq long(strmid(ustars[i],3,6)))].starid=ustars[i]
	   'BSC':StarTable[where(StarTable.bsc $
		   eq long(strmid(ustars[i],3,6)))].starid=ustars[i]
	   'HDN':StarTable[where(StarTable.hdn $
		   eq long(strmid(ustars[i],3,6)))].starid=ustars[i]
	   'HIP':StarTable[where(StarTable.hic $
		   eq long(strmid(ustars[i],3,6)))].starid=ustars[i]
	   else: 
	   endcase
	endif
      endfor
   endif
   return
endif
;
if n_elements(request) eq 0 then begin
	print,"ERROR(RENAME_STARIDS): please specify request, e.g., 'fkv-bsc'"
	return
endif
;
if request eq 'hic-hip' then begin
	i=where(StarTable.hic ne 0,count)
	if count gt 0 then begin
		starids='HIP'+stringl(StarTable[i].hic,format='(i6.6)')
		StarTable[i].starid=starids
	endif
	return
endif
if request eq 'hip-hic' then begin
	i=where(StarTable.hic ne 0,count)
	if count gt 0 then begin
		starids='HIC'+stringl(StarTable[i].hic,format='(i6.6)')
		StarTable[i].starid=starids
	endif
	return
endif
if strpos(request,'hip-') ne -1 then begin
	strput,request,'hic-',0
endif
if strpos(request,'-hip') ne -1 then begin
	strput,request,'-hic',3
	do_hip=1
endif else do_hip=0
;
if strpos(request,'-hdn') ne -1 then begin
	i=where(StarTable.hdn ne 0,count)
	if count gt 0 then $
		starids[i]='HDN'+stringl(StarTable[i].hdn,format='(i6.6)')
endif
if strpos(request,'-bat') ne -1 then begin
	i=where(StarTable.bat ne 0,count)
	if count gt 0 then $
		starids[i]='BAT'+stringl(StarTable[i].bat,format='(i4.4)')
endif
if strpos(request,'-bsc') ne -1 then begin
	i=where(StarTable.bsc ne 0,count)
	if count gt 0 then $
		starids[i]='BSC'+stringl(StarTable[i].bsc,format='(i4.4)')
endif
if strpos(request,'-wds') ne -1 then begin
	i=where(StarTable.wds ne 0,count)
	if count gt 0 then $
		starids[i]='WDS'+stringl(StarTable[i].wds,format='(i10.9)')
endif
if strpos(request,'-sao') ne -1 then begin
	i=where(StarTable.sao ne 0,count)
	if count gt 0 then $
		starids[i]='SAO'+stringl(StarTable[i].sao,format='(i6.6)')
endif
if strpos(request,'-hic') ne -1 then begin
	i=where(StarTable.hic ne 0,count)
	if count gt 0 then $
		starids[i]='HIC'+stringl(StarTable[i].hic,format='(i6.6)')
endif
if strpos(request,'-fkv') ne -1 then begin
	i=where(StarTable.fkv ne 0,count)
	if count gt 0 then $
		starids[i]='FKV'+stringl(StarTable[i].fkv,format='(i4.4)')
endif
if strpos(request,'-fln') ne -1 then begin
	i=where(StarTable.fln ne 0,count)
	if count gt 0 then $
		starids[i]='FLN'+stringl(StarTable[i].fln,format='(i4.4)')
endif
;
case request of
	'fkv-hic':begin
		  i=where(strmid(starids,0,3) eq 'FKV',count)
		  if count gt 0 then begin
		  	hic_ids=cri(long(strmid(starids[i],3,4)),'fkv-hic')
		  	starids[i]='HIC'+stringl(hic_ids,format='(i6.6)')
			StarTable[i].hic=hic_ids
		  endif
		  i=where(StarTable.hic eq -1,count)
		  if count gt 0 then StarTable[i].hic=0
		  end
	'hic-fkv':begin
		  i=where(strmid(starids,0,3) eq 'HIC' or $
			  strmid(starids,0,3) eq 'HIP',count)
		  if count gt 0 then begin
		  	fkv_ids=cri(long(strmid(starids[i],3,6)),'hic-fkv')
		  	starids[i]='FKV'+stringl(fkv_ids,format='(i4.4)')
			StarTable[i].fkv=fkv_ids
		  endif
		  i=where(StarTable.fkv eq -1,count)
		  if count gt 0 then StarTable[i].fkv=0
		  end
	'fkv-bsc':begin
		  i=where(strmid(starids,0,3) eq 'FKV',count)
		  if count gt 0 then begin
			bsc_ids=cri(long(strmid(starids[i],3,4)),'fkv-bsc')
			starids[i]='BSC'+stringl(bsc_ids,format='(i4.4)')
			StarTable[i].bsc=bsc_ids
		  endif
		  i=where(StarTable.bsc eq -1,count)
		  if count gt 0 then StarTable[i].bsc=0
		  end
	'bsc-fkv':begin
		  i=where(strmid(starids,0,3) eq 'BSC',count)
		  if count gt 0 then begin
			fkv_ids=cri(long(strmid(starids[i],3,4)),'bsc-fkv')
			starids[i]='FKV'+stringl(fkv_ids,format='(i4.4)')
			StarTable[i].fkv=fkv_ids
		  endif
		  i=where(StarTable.fkv eq -1,count)
		  if count gt 0 then StarTable[i].fkv=0
		  end
	'fkv-sao':begin
		  i=where(strmid(starids,0,3) eq 'FKV',count)
		  if count gt 0 then begin
			sao_ids=cri(cri(long(strmid(starids[i],3,4)),'fkv-hdn'),'hdn-sao')
			starids[i]='SAO'+stringl(sao_ids,format='(i4.4)')
			StarTable[i].sao=sao_ids
		  endif
		  i=where(StarTable.sao eq -1,count)
		  if count gt 0 then StarTable[i].sao=0
		  end
	'sao-fkv':begin
		  i=where(strmid(starids,0,3) eq 'SAO',count)
		  if count gt 0 then begin
			fkv_ids=cri(cri(long(strmid(starids[i],3,6)),'sao-hdn'),'hdn-fkv')
			starids[i]='FKV'+stringl(fkv_ids,format='(i4.4)')
			StarTable[i].fkv=fkv_ids
		  endif
		  i=where(StarTable.fkv eq -1,count)
		  if count gt 0 then StarTable[i].fkv=0
		  end
	'fkv-hdn':begin
		  i=where(strmid(starids,0,3) eq 'FKV',count)
		  if count gt 0 then begin
			hdn_ids=cri(long(strmid(starids[i],3,4)),'fkv-hdn')
			starids[i]='HDN'+stringl(hdn_ids,format='(i6.6)')
			StarTable[i].hdn=hdn_ids
		  endif
		  i=where(StarTable.hdn eq -1,count)
		  if count gt 0 then StarTable[i].hdn=0
		  end
	'hdn-fkv':begin
		  i=where(strmid(starids,0,3) eq 'HDN',count)
		  if count gt 0 then begin
			fkv_ids=cri(long(strmid(starids[i],3,6)),'hdn-fkv')
			starids[i]='FKV'+stringl(fkv_ids,format='(i4.4)')
			StarTable[i].fkv=fkv_ids
		  endif
		  i=where(StarTable.fkv eq -1,count)
		  if count gt 0 then StarTable[i].fkv=0
		  end
	'bat-hdn':begin
		  i=where(strmid(starids,0,3) eq 'BAT',count)
		  if count gt 0 then begin
			hdn_ids=cri(long(strmid(starids[i],3,4)),'bat-hdn')
			starids[i]='HDN'+stringl(hdn_ids,format='(i6.6)')
			StarTable[i].hdn=hdn_ids
		  endif
		  i=where(StarTable.hdn eq -1,count)
		  if count gt 0 then StarTable[i].hdn=0
		  end
	'hdn-bat':begin
		  i=where(strmid(starids,0,3) eq 'HDN',count)
		  if count gt 0 then begin
			bat_ids=cri(long(strmid(starids[i],3,6)),'hdn-bat')
			starids[i]='BAT'+stringl(bat_ids,format='(i4.4)')
			StarTable[i].bat=bat_ids
		  endif
		  i=where(StarTable.bat eq -1,count)
		  if count gt 0 then StarTable[i].bat=0
		  end
	'fkv-bat':begin
		  i=where(strmid(starids,0,3) eq 'FKV',count)
		  if count gt 0 then begin
			bat_ids=cri(cri(long(strmid(starids[i],3,4)),'fkv-hdn'),'hdn-bat')
			starids[i]='BAT'+stringl(bat_ids,format='(i4.4)')
			StarTable[i].bat=bat_ids
		  endif
		  i=where(StarTable.bat eq -1,count)
		  if count gt 0 then StarTable[i].bat=0
		  end
	'bat-fkv':begin
		  i=where(strmid(starids,0,3) eq 'BAT',count)
		  if count gt 0 then begin
			fkv_ids=cri(cri(long(strmid(starids[i],3,4)),'bat-hdn'),'hdn-fkv')
			starids[i]='FKV'+stringl(fkv_ids,format='(i4.4)')
			StarTable[i].fkv=fkv_ids
		  endif
		  i=where(StarTable.fkv eq -1,count)
		  if count gt 0 then StarTable[i].fkv=0
		  end
	'bat-bsc':begin
		  i=where(strmid(starids,0,3) eq 'BAT',count)
		  if count gt 0 then begin
			bsc_ids=cri(cri(long(strmid(starids[i],3,4)),'bat-hdn'),'hdn-bsc')
			starids[i]='BSC'+stringl(bsc_ids,format='(i4.4)')
			StarTable[i].bsc=bsc_ids
		  endif
		  i=where(StarTable.bsc eq -1,count)
		  if count gt 0 then StarTable[i].bsc=0
		  end
	'hic-bsc':begin
		  i=where(strmid(starids,0,3) eq 'HIC' or $
			  strmid(starids,0,3) eq 'HIP',count)
		  if count gt 0 then begin
			bsc_ids=cri(cri(long(strmid(starids[i],3,6)),'hic-hdn'),'hdn-bsc')
			starids[i]='BSC'+stringl(bsc_ids,format='(i4.4)')
			StarTable[i].bsc=bsc_ids
		  endif
		  i=where(StarTable.bsc eq -1,count)
		  if count gt 0 then StarTable[i].bsc=0
		  end
	'bsc-bat':begin
		  i=where(strmid(starids,0,3) eq 'BSC',count)
		  if count gt 0 then begin
			bat_ids=cri(cri(long(strmid(starids[i],3,4)),'bsc-hdn'),'hdn-bat')
			starids[i]='BAT'+stringl(bat_ids,format='(i4.4)')
			StarTable[i].bat=bat_ids
		  endif
		  i=where(StarTable.bat eq -1,count)
		  if count gt 0 then StarTable[i].bat=0
		  end
	'bsc-hic':begin
		  i=where(strmid(starids,0,3) eq 'BSC',count)
		  if count gt 0 then begin
			hic_ids=cri(cri(long(strmid(starids[i],3,4)),'bsc-hdn'),'hdn-hic')
			starids[i]='HIC'+stringl(hic_ids,format='(i6.6)')
			StarTable[i].hic=hic_ids
		  endif
		  i=where(StarTable.hic eq -1,count)
		  if count gt 0 then StarTable[i].hic=0
		  end
	'bat-hic':begin
		  i=where(strmid(starids,0,3) eq 'BAT',count)
		  if count gt 0 then begin
			hic_ids=cri(cri(long(strmid(starids[i],3,4)),'bat-hdn'),'hdn-hic')
			starids[i]='HIC'+stringl(hic_ids,format='(i6.6)')
			StarTable[i].hic=hic_ids
		  endif
		  i=where(StarTable.hic eq -1,count)
		  if count gt 0 then StarTable[i].hic=0
		  end
	'hic-bat':begin
		  i=where(strmid(starids,0,3) eq 'HIC' or $
			  strmid(starids,0,3) eq 'HIP',count)
		  if count gt 0 then begin
			bat_ids=cri(cri(long(strmid(starids[i],3,6)),'hic-hdn'),'hdn-bat')
			starids[i]='BAT'+stringl(bat_ids,format='(i4.4)')
			StarTable[i].bat=bat_ids
		  endif
		  i=where(StarTable.bat eq -1,count)
		  if count gt 0 then StarTable[i].bat=0
		  end
	'bat-sao':begin
		  i=where(strmid(starids,0,3) eq 'BAT',count)
		  if count gt 0 then begin
			sao_ids=cri(cri(long(strmid(starids[i],3,4)),'bat-hdn'),'hdn-sao')
			starids[i]='SAO'+stringl(sao_ids,format='(i6.6)')
			StarTable[i].sao=sao_ids
		  endif
		  i=where(StarTable.sao eq -1,count)
		  if count gt 0 then StarTable[i].sao=0
		  end
	'sao-bat':begin
		  i=where(strmid(starids,0,3) eq 'SAO',count)
		  if count gt 0 then begin
			bat_ids=cri(cri(long(strmid(starids[i],3,6)),'sao-hdn'),'hdn-bat')
			starids[i]='BAT'+stringl(bat_ids,format='(i4.4)')
			StarTable[i].bat=bat_ids
		  endif
		  i=where(StarTable.bat eq -1,count)
		  if count gt 0 then StarTable[i].bat=0
		  end
	'bat-fln':begin
		  i=where(strmid(starids,0,3) eq 'BAT',count)
		  if count gt 0 then begin
			fln_ids=cri(cri(long(strmid(starids[i],3,4)),'bat-hdn'),'hdn-fln')
			starids[i]='FLN'+stringl(fln_ids,format='(i4.4)')
			StarTable[i].fln=fln_ids
		  endif
		  i=where(StarTable.fln eq -1,count)
		  if count gt 0 then StarTable[i].fln=0
		  end
	'fln-bat':begin
		  i=where(strmid(starids,0,3) eq 'FLN',count)
		  if count gt 0 then begin
			bat_ids=cri(cri(long(strmid(starids[i],3,4)),'fln-hdn'),'hdn-bat')
			starids[i]='BAT'+stringl(bat_ids,format='(i4.4)')
			StarTable[i].bat=bat_ids
		  endif
		  i=where(StarTable.bat eq -1,count)
		  if count gt 0 then StarTable[i].bat=0
		  end
	'fln-bsc':begin
		  i=where(strmid(starids,0,3) eq 'FLN',count)
		  if count gt 0 then begin
			bsc_ids=cri(cri(long(strmid(starids[i],3,4)),'fln-hdn'),'hdn-bsc')
			starids[i]='BSC'+stringl(bsc_ids,format='(i4.4)')
			StarTable[i].bsc=bsc_ids
		  endif
		  i=where(StarTable.bsc eq -1,count)
		  if count gt 0 then StarTable[i].bsc=0
		  end
	'bsc-fln':begin
		  i=where(strmid(starids,0,3) eq 'BSC',count)
		  if count gt 0 then begin
			fln_ids=cri(cri(long(strmid(starids[i],3,4)),'bsc-hdn'),'hdn-fln')
			starids[i]='FLN'+stringl(fln_ids,format='(i4.4)')
			StarTable[i].fln=fln_ids
		  endif
		  i=where(StarTable.fln eq -1,count)
		  if count gt 0 then StarTable[i].fln=0
		  end
	'fln-hdn':begin
		  i=where(strmid(starids,0,3) eq 'FLN',count)
		  if count gt 0 then begin
			hdn_ids=cri(long(strmid(starids[i],3,4)),'fln-hdn')
			starids[i]='HDN'+stringl(hdn_ids,format='(i4.4)')
			StarTable[i].hdn=hdn_ids
		  endif
		  i=where(StarTable.hdn eq -1,count)
		  if count gt 0 then StarTable[i].hdn=0
		  end
	'hdn-fln':begin
		  i=where(strmid(starids,0,3) eq 'HDN',count)
		  if count gt 0 then begin
			fln_ids=cri(long(strmid(starids[i],3,4)),'hdn-fln')
			starids[i]='FLN'+stringl(fln_ids,format='(i4.4)')
			StarTable[i].fln=fln_ids
		  endif
		  i=where(StarTable.fln eq -1,count)
		  if count gt 0 then StarTable[i].fln=0
		  end
	'bat-wds':begin
		  i=where(strmid(starids,0,3) eq 'BAT',count)
		  if count gt 0 then begin
			wds_ids=cri(cri(long(strmid(starids[i],3,4)),'bat-hdn'),'hdn-wds')
			starids[i]='WDS'+stringl(wds_ids,format='(i10.9)')
			StarTable[i].wds=wds_ids
		  endif
		  i=where(StarTable.wds eq -1,count)
		  if count gt 0 then StarTable[i].wds=0
		  end
	'wds-bat':begin
		  i=where(strmid(starids,0,3) eq 'WDS',count)
		  if count gt 0 then begin
			bat_ids=cri(cri(long(strmid(starids[i],3,10)),'wds-hdn'),'hdn-bat')
			starids[i]='BAT'+stringl(bat_ids,format='(i4.4)')
			StarTable[i].bat=bat_ids
		  endif
		  i=where(StarTable.bat eq -1,count)
		  if count gt 0 then StarTable[i].bat=0
		  end
	'wds-bsc':begin
		  i=where(strmid(starids,0,3) eq 'WDS',count)
		  if count gt 0 then begin
			bsc_ids=cri(long(strmid(starids[i],3,10)),'wds-bsc')
			starids[i]='BSC'+stringl(bsc_ids,format='(i4.4)')
			StarTable[i].bsc=bsc_ids
		  endif
		  i=where(StarTable.bsc eq -1,count)
		  if count gt 0 then StarTable[i].bsc=0
		  end
	'bsc-wds':begin
		  i=where(strmid(starids,0,3) eq 'BSC',count)
		  if count gt 0 then begin
			wds_ids=cri(long(strmid(starids[i],3,4)),'bsc-wds')
			starids[i]='WDS'+stringl(wds_ids,format='(i10.9)')
			StarTable[i].wds=wds_ids
		  endif
		  i=where(StarTable.wds eq -1,count)
		  if count gt 0 then StarTable[i].wds=0
		  end
	'fkv-wds':begin
		  i=where(strmid(starids,0,3) eq 'FKV',count)
		  if count gt 0 then begin
			wds_ids=cri(cri(long(strmid(starids[i],3,4)),'fkv-hdn'),'hdn-wds')
			starids[i]='WDS'+stringl(wds_ids,format='(i10.9)')
			StarTable[i].wds=wds_ids
		  endif
		  i=where(StarTable.wds eq -1,count)
		  if count gt 0 then StarTable[i].wds=0
		  end
	'wds-fkv':begin
		  i=where(strmid(starids,0,3) eq 'WDS',count)
		  if count gt 0 then begin
			fkv_ids=cri(cri(long(strmid(starids[i],3,10)),'wds-hdn'),'hdn-fkv')
			starids[i]='FKV'+stringl(fkv_ids,format='(i4.4)')
			StarTable[i].fkv=fkv_ids
		  endif
		  i=where(StarTable.fkv eq -1,count)
		  if count gt 0 then StarTable[i].fkv=0
		  end
	'bsc-hdn':begin
		  i=where(strmid(starids,0,3) eq 'BSC',count)
		  if count gt 0 then begin
			hdn_ids=cri(long(strmid(starids[i],3,4)),'bsc-hdn')
			starids[i]='HDN'+stringl(hdn_ids,format='(i6.6)')
			StarTable[i].hdn=hdn_ids
		  endif
		  i=where(StarTable.hdn eq -1,count)
		  if count gt 0 then StarTable[i].hdn=0
		  end
	'bsc-sao':begin
		  i=where(strmid(starids,0,3) eq 'BSC',count)
		  if count gt 0 then begin
			sao_ids=cri(cri(long(strmid(starids[i],3,4)),'bsc-hdn'),'hdn-sao')
			starids[i]='SAO'+stringl(sao_ids,format='(i6.6)')
			StarTable[i].sao=sao_ids
		  endif
		  i=where(StarTable.sao eq -1,count)
		  if count gt 0 then StarTable[i].sao=0
		  end
	'sao-bsc':begin
		  i=where(strmid(starids,0,3) eq 'SAO',count)
		  if count gt 0 then begin
			bsc_ids=cri(cri(long(strmid(starids[i],3,6)),'sao-hdn'),'hdn-bsc')
			starids[i]='BSC'+stringl(bsc_ids,format='(i4.4)')
			StarTable[i].bsc=bsc_ids
		  endif
		  i=where(StarTable.bsc eq -1,count)
		  if count gt 0 then StarTable[i].bsc=0
		  end
	'hdn-bsc':begin
		  i=where(strmid(starids,0,3) eq 'HDN',count)
		  if count gt 0 then begin
			bsc_ids=cri(long(strmid(starids[i],3,6)),'hdn-bsc')
			starids[i]='BSC'+stringl(bsc_ids,format='(i4.4)')
			StarTable[i].bsc=bsc_ids
		  endif
		  i=where(StarTable.bsc eq -1,count)
		  if count gt 0 then StarTable[i].bsc=0
		  end
	'hdn-hic':begin
		  i=where(strmid(starids,0,3) eq 'HDN',count)
		  if count gt 0 then begin
			hic_ids=cri(long(strmid(starids[i],3,6)),'hdn-hic')
			starids[i]='HIC'+stringl(hic_ids,format='(i6.6)')
			StarTable[i].hic=hic_ids
		  endif
		  i=where(StarTable.hic eq -1,count)
		  if count gt 0 then StarTable[i].hic=0
		  end
	'hic-hdn':begin
		  i=where(strmid(starids,0,3) eq 'HIC' or $
			  strmid(starids,0,3) eq 'HIP',count)
		  if count gt 0 then begin
			hdn_ids=cri(long(strmid(starids[i],3,6)),'hic-hdn')
			starids[i]='HDN'+stringl(hdn_ids,format='(i6.6)')
			StarTable[i].hdn=hdn_ids
	    	  endif
		  i=where(StarTable.hdn eq -1,count)
		  if count gt 0 then StarTable[i].hdn=0
		  end
	'hdn-sao':begin
		  i=where(strmid(starids,0,3) eq 'HDN',count)
		  if count gt 0 then begin
			sao_ids=cri(long(strmid(starids[i],3,6)),'hdn-sao')
			starids[i]='SAO'+stringl(sao_ids,format='(i6.6)')
			StarTable[i].sao=sao_ids
		  endif
		  i=where(StarTable.sao eq -1,count)
		  if count gt 0 then StarTable[i].sao=0
		  end
	'sao-hdn':begin
		  i=where(strmid(starids,0,3) eq 'SAO',count)
		  if count gt 0 then begin
			hdn_ids=cri(long(strmid(starids[i],3,6)),'hdn-sao')
			starids[i]='HDN'+stringl(hdn_ids,format='(i6.6)')
			StarTable[i].hdn=hdn_ids
		  endif
		  i=where(StarTable.hdn eq -1,count)
		  if count gt 0 then StarTable[i].hdn=0
		  end
	'wds-sao':begin
		  i=where(strmid(starids,0,3) eq 'WDS',count)
		  if count gt 0 then begin
			sao_ids=cri(long(strmid(starids[i],3,10)),'wds-sao')
			starids[i]='SAO'+stringl(sao_ids,format='(i6.6)')
			StarTable[i].sao=sao_ids
		  endif
		  i=where(StarTable.sao eq -1,count)
		  if count gt 0 then StarTable[i].sao=0
		  end
	'sao-wds':begin
		  i=where(strmid(starids,0,3) eq 'SAO',count)
		  if count gt 0 then begin
			wds_ids=cri(long(strmid(starids[i],3,6)),'sao-wds')
			starids[i]='WDS'+stringl(wds_ids,format='(i10.9)')
			StarTable[i].wds=wds_ids
		  endif
		  i=where(StarTable.wds eq -1,count)
		  if count gt 0 then StarTable[i].wds=0
		  end
	'wds-hdn':begin
		  i=where(strmid(starids,0,3) eq 'WDS',count)
		  if count gt 0 then begin
			hdn_ids=cri(long(wdsid(id2=starids[i])),'wds-hdn')
			starids[i]='HDN'+stringl(hdn_ids,format='(i6.6)')
			StarTable[i].hdn=hdn_ids
		  endif
		  i=where(StarTable.hdn eq -1,count)
		  if count gt 0 then StarTable[i].hdn=0
		  end
	'hdn-wds':begin
		  i=where(strmid(starids,0,3) eq 'HDN',count)
		  if count gt 0 then begin
			wds_ids=cri(long(strmid(starids[i],3,6)),'hdn-wds')
			starids[i]='WDS'+stringl(wds_ids,format='(i10.9)')
			StarTable[i].wds=wds_ids
		  endif
		  i=where(StarTable.wds eq -1,count)
		  if count gt 0 then StarTable[i].wds=0
		  end
             else:begin
		  print,'***Error(RENAME_STARIDS): request not valid!'
		  return
		  end
endcase
;
index=where(strmid(starids,3,4) eq '****' $
	or strmid(starids,3,10) eq '-000000001' ,count)
if count gt 0 then begin
	starids[index]=starids_bck[index]
	print,'Unresolved requests of type ',request,':',count,'.', $
		format='(a,a,a,i6,a)'
endif
;
if strpos(request,'-wds') ne -1 then begin
	index=where(strmid(starids,0,3) eq 'WDS',count)
	if count gt 0 then begin
		sign=strmid(starids[index],3,1)
		i=where(sign eq ' ',count)
		if count gt 0 then sign[i]='+'
		starids[index]='WDS'+sign+strmid(starids[index],4,9)
	endif
endif
if do_hip then begin
	index=where(strmid(starids,0,3) eq 'HIC',count)
	if count gt 0 then $
		starids[index]='HIP'+stringl(StarTable[index].hic,format='(i6.6)')
endif
;
StarTable.starid=starids 
;
end
;-------------------------------------------------------------------------------
pro rename_bsc
;
; Replace Flamsteed+Bayer IDs from the HR catalog in the star names with 
; Greek letters if the Bayer ID is available, otherwise just keep the 
; Flamsteed ID. Example: '4201 COM' -> 'ALP COM', '15   LYN' -> '15 LYN'
;
common StarBase,StarTable,Notes
;
greek_alphabet=['   ','ALP','BET','GAM','DEL','EPS','ZET','ETA','THE','IOT', $
		'KAP','LAM',' MU',' NU',' XI','OMI',' PI','RHO','SIG','TAU', $
		'UPS','PHI','CHI','PSI','OME']
;
index_bsc=where(strmid(StarTable.starid,0,3) eq 'BSC',count_bsc)
index_bsc=where(StarTable.bsc ne 0,count_bsc)
on_ioerror,skip
if count_bsc gt 0 then begin
	flamsteed=strmid(StarTable[index_bsc].name,0,3)
	bayer=strmid(StarTable[index_bsc].name,3,2)
	bayer_no=intarr(count_bsc)
	jndex=where(strlen(strcompress(bayer,/remove_all)) gt 0,count_bayer)
	if count_bayer gt 0 then bayer_no[jndex]=fix(bayer[jndex])
	number=strmid(StarTable[index_bsc].name,5,2)
	constellation=strmid(StarTable[index_bsc].name,7,3)
	index=where(bayer_no gt 0,count)
	if count gt 0 then flamsteed[index]=''
	name=flamsteed+' '+greek_alphabet[bayer_no]+number+' '+constellation
	name=strtrim(strcompress(name),2)
	if count_bsc eq 1 then StarTable[index_bsc].name=name[0] else $
	if count_bsc ne 0 then StarTable[index_bsc].name=name
endif
;
skip:
;
end
;-------------------------------------------------------------------------------
pro get_crossindex,level
;
; From the Hipparcos input catalog, the FKV and BSC catalogs, compile and save
; crossindex tables.
; Also prepare non-direct and derived cross indices.
;
; Level 0: do only derived
; Level 1: do full set
;
common StarBase,StarTable,Notes
;
if n_elements(level) eq 0 then level=0
if level eq 0 then goto,derived
;
status=linknload(!external_lib,'crossindex')
if status ne 0 then return
;
hic=0L
fkv=0
bsc=0
hdn=0L
sao=0L
status=dc_read_free('crossindex.dat',hic,fkv,bsc,hdn,sao,/col)
spawn,'rm -f crossindex.dat'
hic=long(hic)
fkv=long(fkv)
bsc=long(bsc)
hdn=long(hdn)
sao=long(sao)
;
; Create "direct" cross indices
create_cri,'fkv-hic',fkv,hic
create_cri,'hdn-fkv',hdn,fkv
create_cri,'hdn-bsc',hdn,bsc
create_cri,'sao-hic',sao,hic
create_cri,'hdn-hic',hdn,hic
create_cri,'hdn-sao',hdn,sao
create_cri,'fkv-sao',fkv,sao
;
; Create "non-direct" cross indices
id1=fkv[where(fkv ne 0)]
id2=cri(cri(id1,'fkv-hdn'),'hdn-bsc')
index=where(id2 gt 0)
create_cri,'fkv-bsc',id1[index],id2[index]
id1=bsc[where(bsc ne 0)]
id2=cri(cri(id1,'bsc-hdn'),'hdn-hic')
index=where(id2 gt 0)
create_cri,'bsc-hic',id1[index],id2[index]
;
; Create derived cross indices
derived:
; get_bat 
; Now replaced by SB9 (The ninth catalogue of spectroscopic binary orbits",
; Pourbaix D., Tokovinin A.A., Batten A.H., Fekel F.C., Hartkopf W.I.,
; Levato H., Morrell N.I., Torres G., Udry S., 2004, Astronomy and
; Astrophysics, 424, 727-732.
get_sb9
index=where(startable.bat ne 0 and startable.hdn ne 0)
bat=startable[index].bat
hdn=startable[index].hdn
create_cri,'bat-hdn',bat,hdn
bsc=cri(hdn,'hdn-bsc')
index=where(bsc gt 0)
create_cri,'bat-bsc',bat[index],bsc[index]
fkv=cri(hdn,'hdn-fkv')
index=where(fkv gt 0)
create_cri,'bat-fkv',bat[index],fkv[index]
;
get_fln
index=where(startable.fln ne 0 and startable.hdn ne 0)
fln=startable[index].fln
hdn=startable[index].hdn
create_cri,'fln-hdn',fln,hdn
;
end
;-------------------------------------------------------------------------------
pro get_wdsbsc
;
; Prepare crossindex tables for WDS numbers by comparing the position
; extracted from these numbers to the HR catalog position.
; A separation of less than 100" is considered a match.
;
common StarBase,StarTable,Notes
;
get_bsc
n=StarTable.starid
r=StarTable.ra
d=StarTable.dec
;
get_wds
;
for i=0,n_elements(StarTable)-1 do begin
	sep=sqrt(((StarTable[i].ra-r)*3600*15)^2+((StarTable[i].dec-d)*3600)^2)
	j=where(sep lt 100,count)
	if count gt 0 then begin
		j=where(min(sep) eq sep)
		StarTable[i].bsc=long(strmid(n[j[0]],3,4))
	endif
endfor
;
index=where((StarTable.bsc ne 0) and (StarTable.wds ne 0),count)
wds=StarTable[index].wds
bsc=StarTable[index].bsc
create_cri,'wds-bsc',wds,bsc
;
end
;-------------------------------------------------------------------------------
pro get_wdshdn
;
; Prepare crossindex tables for WDS numbers by comparing the position
; extracted from these numbers to the HD catalog position (obtained from Yale 
; Parallax catalog). A separation of less than 100" is considered a match.
;
; Obsolete! See next procedure.
;
common StarBase,StarTable,Notes
;
get_sky
n=startable.hdn
r=startable.ra
d=startable.dec
;
; get_wds
get_wbs		; read the bible
;
; n=0l
; r=0.0d
; d=0.0d
; status=dc_read_free(!oyster_dir+'starbase/position.hdn', $
; 	n,r,d,/col,resize=[1,2,3],ignore=['!'])
;
for i=0l,n_elements(StarTable)-1 do begin
	sep=winkel(r,d,startable[i].ra,startable[i].dec)*3600
	min_sep=min(sep)
	if min_sep lt 5 then begin
		j=where(sep eq min_sep)
		StarTable[i].hdn=n(j[0])
	endif
endfor
;
index=where((StarTable.hdn ne 0) and (StarTable.wds ne 0),count)
wds=StarTable[index].wds
hdn=StarTable[index].hdn
create_cri,'wds-hdn',wds,hdn
;
end
;-------------------------------------------------------------------------------
pro get_wdshdn
;
; Prepare crossindex tables for WDS numbers by comparing the position
; extracted from these numbers to the HD catalog position (obtained from Yale 
; Parallax catalog). A separation of less than 100" is considered a match.
;
common StarBase,StarTable,Notes
;
; Read Vizier-prepared SAO/HD catalog with J2000 positions
sao=0L
hdn=0L
rah=0d
ram=0d
ras=0d
decd=0d
decm=0d
decs=0d
status=dc_read_free(!catalogs_dir+'hdn/catalog.dat', $
	sao,hdn,rah,ram,ras,decd,decm,decs,/col,resize=[1,2,3,4,5,6,7,8], $
	ignore=['#'])
; Decode sign of declination
l=blanks(39)
status=dc_read_fixed(!catalogs_dir+'hdn/catalog.dat',l,/col,format='(a39)', $
	ignore=['#'])
sign=strmid(l,27,1)
index=where(sign eq '-')
decm[index]=-decm[index]
decs[index]=-decs[index]
;
ra=rah+ram/60+ras/3600
dec=decd+decm/60+decs/3600
;
get_wbs		; read the bible
;
; Adopted positional precision of catalog
min_sep=dblarr(n_elements(StarTable))
precision=5 ; ["]
;
for i=0l,n_elements(StarTable)-1 do begin
	sep=winkel(ra,dec,startable[i].ra,startable[i].dec)*3600
	min_sep[i]=min(sep)
	if min_sep[i] le precision then begin
		j=where(sep eq min_sep[i])
		StarTable[i].hdn=hdn[j]
	endif
endfor
; The histogramm shows that the precision is between 5"-6" 
; histograph,min_sep,binsize=0.5,min=0,max=10
;
index=where((StarTable.hdn ne 0) and (StarTable.wds ne 0),count)
wds=StarTable[index].wds
hdn=StarTable[index].hdn
create_cri,'wds-hdn',wds,hdn
;
end
;-------------------------------------------------------------------------------
pro get_wdssao
;
; Prepare crossindex tables for WDS numbers by comparing the position
; extracted from these numbers to the SAO catalog position. A separation of
; less than 100" is considered a match.
;
; Obsolete! See next procedure.
;
common StarBase,StarTable,Notes
;
get_wds
;
n=''
r=0.0d
d=0.0d
status=dc_read_free(!catalogs_dir+'sao/position.dat', $
	n,r,d,/col,resize=[1,2,3],ignore=['!'])
;
for i=0,n_elements(StarTable)-1 do begin
	sep=sqrt(((StarTable[i].ra-r)*3600*15)^2+((StarTable[i].dec-d)*3600)^2)
	j=where(sep lt 100,count)
	if count gt 0 then begin
		j=where(min(sep) eq sep)
		StarTable[i].sao=long(strmid(n(j[0]),3,6))
	endif
endfor
;
index=where((StarTable.sao ne 0) and (StarTable.wds ne 0),count)
wds=StarTable[index].wds
sao=StarTable[index].sao
create_cri,'wds-sao',wds,sao
;
end
;-------------------------------------------------------------------------------
pro get_wdssao
;
; Prepare crossindex tables for WDS numbers by comparing the position
; extracted from these numbers to the SAO catalog position (obtained from Yale 
; Parallax catalog). A separation of less than 5" is considered a match.
;
common StarBase,StarTable,Notes
;
; Read Vizier-prepared SAO/HD catalog with J2000 positions
sao=0L
hdn=0L
rah=0d
ram=0d
ras=0d
decd=0d
decm=0d
decs=0d
status=dc_read_free(!catalogs_dir+'hdn/catalog.dat', $
	sao,hdn,rah,ram,ras,decd,decm,decs,/col,resize=[1,2,3,4,5,6,7,8], $
	ignore=['#'])
; Decode sign of declination
l=blanks(39)
status=dc_read_fixed(!catalogs_dir+'hdn/catalog.dat',l,/col,format='(a39)', $
	ignore=['#'])
sign=strmid(l,27,1)
index=where(sign eq '-')
decm[index]=-decm[index]
decs[index]=-decs[index]
;
ra=rah+ram/60+ras/3600
dec=decd+decm/60+decs/3600
;
get_wbs		; read the bible
;
; Adopted positional precision of catalog
min_sep=dblarr(n_elements(StarTable))
precision=5 ; ["]
;
for i=0l,n_elements(StarTable)-1 do begin
	sep=winkel(ra,dec,startable[i].ra,startable[i].dec)*3600
	min_sep[i]=min(sep)
	if min_sep[i] le precision then begin
		j=where(sep eq min_sep[i])
		StarTable[i].sao=sao[j]
	endif
endfor
; The histrogramm shows that the precision is between 5"-6" 
; histograph,min_sep,binsize=0.5,min=0,max=10
;
index=where((StarTable.sao ne 0) and (StarTable.wds ne 0),count)
wds=StarTable[index].wds
sao=StarTable[index].sao
create_cri,'wds-sao',wds,sao
;
end
;-------------------------------------------------------------------------------
pro create_cri,request,i1,i2
;
; Given arrays of corresponding catalog numbers (may contain zeroes), create
; crossindex tables for request in both directions and save to disk files.
;
;
flag=0
if n_elements(request) eq 0 then flag=1
ni1=n_elements(i1)
ni2=n_elements(i2)
if ni1 ne ni2 or ni1 eq 0 then flag=1
if flag then begin
	print,'***Error(CREATE_CRI): insufficient input!'
	return
endif
index=where(i1 ne 0 and i2 ne 0,count)
if count eq 0 then begin
	print,'***Error(CREATE_CRI): indices empty!'
	return
endif
t=build_critable(i1[index],i2[index])
;
if criparse(request,cat_1,cat_2) ne -1 then begin
	index=sort(t.id1) & t=t[index]
	index=uniq(t.id1) & t=t[index]
	file=!catalogs_dir+'crossindex/'+cat_1+'_'+cat_2+'.cri'
	save,t,filename=file
	id1=t.id2 & id2=t.id1 & t.id1=id1 & t.id2=id2
	index=sort(t.id1) & t=t[index]
	index=uniq(t.id1) & t=t[index]
	file=!catalogs_dir+'crossindex/'+cat_2+'_'+cat_1+'.cri'
	save,t,filename=file
endif
;
; Changed print below by Bob to output cross-index created and increase 
; the integer size printed to accomodate the large number of 
; entries in the SAO catalog.
;
print,'Saved crossindex; ',request,' number of entries=',n_elements(t),'.', $
	format='(a,a,a,i6,a)'
;
end
;-------------------------------------------------------------------------------
pro edit_cri,request,i1,i2
;
; Used to update crossindex table.
;
;
if n_params() ne 3 then begin
	print,'***Error(EDIT_CRI): wrong number of parameters!'
	print,'***Should be: request,i1,i2.'
	return
endif
;
i1=long(i1)
i2=long(i2)
;
if criparse(request,cat_1,cat_2) ne -1 then begin
	file=!catalogs_dir+'crossindex/'+cat_1+'_'+cat_2+'.cri'
        result=file_search(file,count=fcount)
        if fcount ne 0 then restore,filename=file else begin
		print,'***Error(EDIT_CRI): file not found: ',file,'!'
		return
	endelse
endif else return
;
id1=t.id1
id2=t.id2
;
index=where(id1 eq i1,count)
if count eq 1 then begin
	id2[index]=i2
endif else if count eq 0 then begin
	id1n=[id1,i1] & id2n=[id2,i2]
	s=sort(id1n)
	id1=id1n[s] & id2=id2n[s]
endif else begin
	print,'***Error(EDIT_CRI): crossindex corrupt; not saved!'
	return
endelse
;
create_cri,request,id1,id2
;
print,'Updated crossindex; number of entries=',n_elements(t)
;
end
;-------------------------------------------------------------------------------
pro update_cri
;
; Enter here which updates were applied to crossindex tables.
;
edit_cri,'wds-hdn',+234114613,222516
edit_cri,'wds-hdn',-002624217,  2261
edit_cri,'wds-hdn',-002832020,  2475
edit_cri,'wds-hdn',-003166258,  2885
edit_cri,'wds-hdn',+010835455,  6582
edit_cri,'wds-hdn',-011586853,  7788
edit_cri,'wds-hdn',-012206943,  8519
edit_cri,'wds-hdn',-012594754,  8821
edit_cri,'bat-hdn',157,19356
;
end
;-------------------------------------------------------------------------------
pro hdn_cri
;
; For all HDN stars, list ids in higher positional accuracy catalogs.
;
common StarBase,StarTable,Notes
;
get_sky
rename_starids,'hdn-hic'
rename_starids,'hdn-fkv'
;
hdnstars='HDN'+stringl(StarTable.hdn,format='(i6.6)')
openw,unit,'hdn.cri',/get_lun
for i=0L,n_elements(StarTable)-1 do $
	printf,unit,StarTable[i].hdn,StarTable[i].starid,format='(i6.6,1x,a9)'
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro bsc_cri
;
; For all BSC stars, list ids in higher positional accuracy catalogs.
;
common StarBase,StarTable,Notes
;
get_bsc
rename_starids,'bsc-hic'
rename_starids,'bsc-fkv'
;
bscstars='BSC'+stringl(StarTable.bsc,format='(i6.6)')
openw,unit,'bsc.cri',/get_lun
for i=0L,n_elements(StarTable)-1 do $
	printf,unit,StarTable[i].bsc,StarTable[i].starid,format='(i6.6,1x,a9)'
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro fkv_cri
;
; For all FKV stars, list ids in higher positional accuracy catalogs.
;
common StarBase,StarTable,Notes
;
get_fkv
rename_starids,'fkv-hic'
openw,unit,'fkv.cri',/get_lun
for i=0L,n_elements(StarTable)-1 do $
	printf,unit,StarTable[i].fkv,StarTable[i].starid,format='(i6.6,1x,a9)'
free_lun,unit
;
end
;************************************************************************Block 7
function spec_array,type,low=low
;
; Return spectral types in string format for an integer type array.
;
types=['O','B','A','F','G','K','M']
;
if keyword_set(low) then types=strlowcase(types)
;
return,types[type/10]+string(type mod 10,format='(i1)')
;
end
;-------------------------------------------------------------------------------
function class_array,class,low=low
;
; Return luminosity classes in string format for an integer class array.
;
classes=['I','II','III','IV','V']
;
if keyword_set(low) then classes=strlowcase(classes)
;
return,classes[class-1]
;
end
;-------------------------------------------------------------------------------
pro spec_parse,spectra,type,class,type2,class2
;
; Call the C function specparse which parses spectra strings into integer
; types and classes. If called without arguments, initialize types and
; classes in the startable according to the spectral classifications.
;
common StarBase,StarTable,Notes
;
; Save a copy of the input spectra, if any
if n_elements(spectra) gt 0 then spectra_in=spectra
;
if n_elements(spectra) eq 0 then spectra=StarTable.spectrum $
			    else spectra=strupcase(spectra)
index=where(strlen(spectra) eq 0,count)
if count gt 0 then spectra[index]=' '
index=where(strpos(startable.spectrum,'[WC') ge 0,count)
for i=0,count-1 do begin
	spectrum=startable[index[i]].spectrum
	startable[index[i]].spectrum= $
		strsplit(strsplit(spectrum,'[',/extract),']',/extract)
endfor
;
n=n_elements(spectra)
type=fltarr(n)		; 0=O-type, 1=B-type, 2=A-type, 34=F4
type2=fltarr(n)		; Same for companion
class=fltarr(n)		; 5=V=dwarf,...
class2=fltarr(n)	; Same for companion
;
status=linknload(!external_lib,'specparse',n,spectra,type,type2,class,class2)
; for j=0,n_elements(spectra)-1 do begin
;	print,spectra(j),type(j),type2(j),class(j),class2(j)
; endfor
;
; Process WR stars separately
index_wr=where(strmid(spectra,0,2) eq 'WC',n_wr) ; The WR is always the 1st star
for i=0,n_wr-1 do begin
	j=index_wr[i]		; Index into spectra for this WR
	type[j]=-type[j]	; negative type for WRs
;	Example spectra: WC WC+... WC... WC4 WC4+O WC4+O5-6 WC4+O9 WC8+O7.5e
	spectrum=nameparse(spectra[j],['+','...'])	; e.g.: 'WC...'
	if n_elements(spectrum) eq 2 then begin
		if strlen(spectrum[1]) eq 0 then spectrum=spectrum[0]
	endif
	if n_elements(spectrum) eq 2 then begin
		class[j]=float(strmid(spectra[j],2,1))
		class2[j]=float(strmid(spectra[j],2,1))
	endif else begin
		class[j]=float(strmid(spectra[j],2,1))
	endelse
endfor
; 
if n_params() eq 0 then begin
	StarTable.spectrum=spectra
	StarTable.type=type
	StarTable.type2=type2
	StarTable.class=class
	StarTable.class2=class2
endif
;
; Restore the input spectra
if n_elements(spectra_in) ne 0 then spectra=spectra_in
;
end
;-------------------------------------------------------------------------------
pro set_defclass
;
; Set default luminosity class to dwarf where class=0.
;
common StarBase,StarTable,Notes
;
index=where(fix(StarTable.class) eq 0,count)
if count gt 0 then StarTable[index].class=5
;
index=where(fix(StarTable.class2) eq 0,count)
if count gt 0 then StarTable[index].class2=5
;
end
;-------------------------------------------------------------------------------
pro diameter_ri,force=force
;
; Compute apparent UD diameter of star at 800 nm in mas from (R-I) color index 
; and m_V. Do this only for non-supergiants.
;
common StarBase,StarTable,Notes
;
if n_elements(force) eq 0 then force=0
;
; Old polynomial coefficients
; r=[0.518,1.299,-0.308]
;
; New coefficients with Dave's Mark III diameter data
r=[0.50017342,1.5095543,-0.62435937,0.13392176]
;
startable.diameter=0
index=where(StarTable.ri ne +100 $
	and StarTable.mv ne +100 $
	and StarTable.bflag ne 'B' $
	and StarTable.class ge 2,count)
if force then $
index=where(StarTable.ri ne +100 $
	and StarTable.mv ne +100,count)
if count gt 0 then begin
	logud=poly(StarTable[index].ri,r)
	StarTable[index].diameter=10^(logud-0.2*StarTable[index].mv)
endif
;
end
;-------------------------------------------------------------------------------
pro diameter_bv,force=force
;
; Compute apparent diameter of star in mas from (B-V) color index and m_V.
; Do this only for single stars earlier than M and dwarf through giant.
; Note: probably not very accurate, and not tested well.
;
common StarBase,StarTable,Notes
;
if n_elements(force) eq 0 then force=0
;
r=[0.496,0.975,-0.661,0.269]
;
startable.diameter=0
index=where(StarTable.bv lt 1.6 $
	and StarTable.mv ne +100 $
	and StarTable.bflag ne 'B' $
	and StarTable.class ge 3 $
	and StarTable.type lt 70,count)
if force then $
index=where(StarTable.bv ne +100 $
	and StarTable.mv ne +100,count)
if count gt 0 then begin
	logud=poly(StarTable[index].bv,r)
	StarTable[index].diameter=10^(logud-0.2*StarTable[index].mv)
endif
;
end
;-------------------------------------------------------------------------------
pro diameter_vk,mv,mk,d_vk,force=force
;
; Compute apparent LD diameter of star in mas from (V-K) color index and m_V.
; Use coefficients fit to data from van Belle et al. 2009 for supergiants
; and brightgiants (ptisgbg.pro), unless forced. "Surface-brightness method"
;
; Nov 3, 2015: added computation of secondary diameter, diameter2.
; Jan 31, 2018: added parameters mv,mk for on-the-spot computations (MS only)
;
common StarBase,StarTable,Notes
;
; New coefficients with Dave's Mark III diameter data
r=[2.658,1.385,-0.021]
;
; Option to print or return estimated diameters
if n_params() ge 2 then begin
	vk=mv-mk
	logud=poly(vk,r)
	d_vk=10^(0.2*(logud-mv))
	if n_params() eq 2 then print,d_vk
	return
endif
;
if n_elements(force) eq 0 then force=0
;
startable.diameter=0
index=where(StarTable.mv ne +100 $
	and StarTable.mk ne +100 $
	and StarTable.bflag ne 'B' $
	and StarTable.class ge 3,count)
if force then $
index=where(StarTable.mv ne +100 $
	and StarTable.mk ne +100,count)
if count gt 0 then begin
	vk=startable[index].mv-startable[index].mk
	logud=poly(vk,r)
;	Also estimate diameters of secondaries
	mv12=dmag(StarTable[index].mv,StarTable[index].dmv)
	j=where(StarTable[index].dmv eq 100,nj)
	StarTable[index].diameter=10^(0.2*(logud-mv12[0,*]))
	StarTable[index].diameter2=10^(0.2*(logud-mv12[1,*]))
	if nj gt 0 then StarTable[index[j]].diameter2=0
endif
if force then return
;
; Using coefficients fit to data in van Belle et al. 2009 for brightgiants
r=[2.869,1.292,-0.018]
index=where(StarTable.mv ne +100 $
	and StarTable.mk ne +100 $
	and StarTable.bflag ne 'B' $
	and StarTable.class eq 2,count)
if count gt 0 then begin
	vk=startable[index].mv-startable[index].mk
	logud=poly(vk,r)
	StarTable[index].diameter=10^(0.2*(logud-StarTable[index].mv))
endif
;
; Using coefficients fit to data in van Belle et al. 2009 for supergiants
r=[2.633,1.407,-0.032]
index=where(StarTable.mv ne +100 $
	and StarTable.mk ne +100 $
	and StarTable.bflag ne 'B' $
	and StarTable.class lt 2,count)
if count gt 0 then begin
	vk=startable[index].mv-startable[index].mk
	logud=poly(vk,r)
	StarTable[index].diameter=10^(0.2*(logud-StarTable[index].mv))
endif
;
end
;-------------------------------------------------------------------------------
pro bad_cals
;
; Read bad calibrator registry list (tab format), calibrators.bcr,
; using the ascii template calibrators.tpl. Check if there are matches
; with stars in StarTable with bflag='C', and remove these (set to '.')
; Notify of bad calibrators!
;
; Create input calibrators.bcr using copy/paste of the table at 
; http://www.jmmc.fr/badcal/show.jsp?type=all&display=complet
;
; To use this procedure, one has to create the ASCI template as follows:
; template=ascii_template('~/oyster/starbase/calibrators.bcr')
; (Except for a few strings, e.g., coordinates, all other are recognized)
; save,template,file='~/oyster/starbase/calibrators.tpl'
; The template can then be used to read the data:
; restore,'~/oyster/starbase/calibrators.tpl'
; data=read_ascii('calibrators.bcr',template=template)
;
common StarBase,StarTable,Notes
;
restore,!oyster_dir+'starbase/calibrators.tpl'
bc=read_ascii(!oyster_dir+'starbase/calibrators.bcr',template=template)
;
num_badcal=0
index=where(startable.bflag eq 'C',count)
for i=0,count-1 do begin
	w=winkel(startable[index[i]].ra*15,startable[index[i]].dec, $
		bc.field02,bc.field03)*3600	; ["]
	j=where(w lt 60,n)
	if n eq 1 then begin
		print,'BadCal found: '+startable[index[i]].starid
		startable[index[i]].bflag='.'
		num_badcal=num_badcal+1
	endif else if n gt 1 then begin
		print,'More than 1 bad CAL near SCI! (' $
			+strjoin(bc.field06[j],' ')+')'
		startable[index[i]].bflag='.'
	endif
endfor
if num_badcal eq 0 then print,'No bad calibrators known.'
;
end
;************************************************************************Block 8
pro avm_stars
;
; Derive absolute V-magnitude M_V from trigonometric parallax. 
;
common StarBase,StarTable,Notes
;
index=where((StarTable.px gt 0) and (StarTable.mv ne +100.0),count)
;
dmod=-5*alog10(1./StarTable[index].px)+5
StarTable[index].amv=StarTable[index].mv+dmod
;
end
;-------------------------------------------------------------------------------
pro d_stars
;
; Compute distance d[pc] from parallax, absolute magnitude,
; or weighted average of both, if available.
;
common StarBase,StarTable,Notes
;
; Parallax only
index=where((StarTable.px gt 0) $
	and (StarTable.amv eq +100),count)
if count gt 0 then StarTable[index].d=1./StarTable[index].px
;
; Absolute magnitude only
index=where((StarTable.px le 0) $
	and (StarTable.amv ne +100) $
	and (StarTable.mv ne +100),count)
if count gt 0 then $
	StarTable[index].d= $
		10^(((StarTable[index].amv-StarTable[index].mv)-5)/(-5))
;
; Weighted average of parallax and absolute magnitude (error assumed +/- 1 mag)
index=where((StarTable.px gt 0) $
	and (StarTable.amv ne +100) $
	and (StarTable.mv ne +100),count)
if count gt 0 then begin
	d_px=1./StarTable[index].px
	d_pxe=StarTable[index].pxe*d_px^2
	d_ph=10^(((StarTable[index].amv-StarTable[index].mv)-5)/(-5))
	d_phe=abs(d_ph)/5/alog(10)*1.0
	d_pxw=1/d_pxe^2
	d_phw=1/d_phe^2
	StarTable[index].d=(d_px*d_pxw+d_ph*d_phw)/(d_pxw+d_phw)
endif
;
end
;-------------------------------------------------------------------------------
pro i_binaries
;
; Estimate orbital inclination for double-classification binaries. Do
; consistency checks. Requires massfunction, mass1/2, p[d], d[pc].
; The Batten catalog has f(m) values for SB1, and m*sin(i)^3 for SB2's.
;
common StarBase,StarTable,Notes
;
index=where((StarTable.mass ne 0) $
	and (StarTable.mass2 ne 0) $
	and (StarTable.mf ne 0),count)
print,'Number of single-lined double-classification binaries =',count
if count gt 0 then begin
	flags=StarTable[index].sflag
	mf=StarTable[index].mf
	m1=StarTable[index].mass
	m2=StarTable[index].mass2
	i1=(mf*(m1+m2)^2/m2^3)^(1./3)
	bad_index=where(i1 gt 1.1,bad_count)
	if bad_count gt 0 then flags[bad_index]='!'
	bad_index=where(i1 gt 1.0,bad_count)
	if bad_count gt 0 then i1[bad_index]=1.0
	StarTable[index].i=asin(i1)/!pi*180
	StarTable[index].sflag=flags
endif
;
index=where((StarTable.mass ne 0) $
	and (StarTable.mass2 ne 0) $
	and (StarTable.m1sin3i ne 0) $
	and (StarTable.m2sin3i ne 0),count)
print,'Number of double-lined    s/d-classification binaries =',count
if count gt 0 then begin
	i1=(StarTable[index].m1sin3i/StarTable[index].mass)^(1./3)
	i2=(StarTable[index].m2sin3i/StarTable[index].mass2)^(1./3)
;
;	Flag inconsistent results
	flags=StarTable[index].sflag
	bad_index=where((i1 gt 1.1) or (i2 gt 1.1),bad_count)
	if bad_count gt 0 then flags[bad_index]='!'
	i_ratio=i1/i2
	bad_index=where((i_ratio lt 0.9) or (i_ratio gt 1.1),bad_count)
	if bad_count gt 0 then flags[bad_index]='!'
;
;	Compute inclination
	bad_index=where((i1 gt 1.0) or (i2 gt 1.0),bad_count)
	if bad_count gt 0 then begin
		i1[bad_index]=1.0
		i2[bad_index]=1.0
	endif
	StarTable[index].i=asin(sqrt(i1*i2))*180/!pi
	StarTable[index].sflag=flags
endif
;
end
;-------------------------------------------------------------------------------
pro a_binaries
;
; Estimate semi-major axis [arcsec] of binaries using Kepler's third law.
; Requires mass, mass2, p[d], d[pc].
;
common StarBase,StarTable,Notes
;
index=where((StarTable.mass ne 0) $
	and (StarTable.mass2 ne 0) $
	and (StarTable.p ne 0) $
	and (StarTable.d ne 0),count)
if count gt 0 then begin
	StarTable[index].a= $
		(StarTable[index].mass+StarTable[index].mass2)^(1./3) $
		*(StarTable[index].p/365.25)^(2./3)/StarTable[index].d
endif
;
end
;-------------------------------------------------------------------------------
pro asini_binaries
;
; Estimate sin(i)*semi-major axis [arcsec] of SB2 using Kepler's third law.
; Requires mass1/2, parallax ["], d[pc].
;
common StarBase,StarTable,Notes
;
index=where((StarTable.mass ne 0) $
	and (StarTable.mass2 ne 0) $
	and (StarTable.p ne 0) $
	and (StarTable.d ne 0),count)
if count gt 0 then begin
	StarTable[index].a= $
		(StarTable[index].mass+StarTable[index].mass2)^(1./3) $
		*(StarTable[index].p/365.25)^(2./3)/StarTable[index].d
endif
;
end
;-------------------------------------------------------------------------------
function a_sini,p,e,k1,k2,px=px
;
; Return a*sin(i) [km] for a given binary component or the sum (k1,k2).
; If keyword px is set to the parallax ["], return a_sini [mas].
;
; Period p [d], eccentricity e, semi-amplitudes k1, k2 (optional) [km/s]
; Example for HD 22128: print,a_sini(5.08,0,68.4,73.7,px=0.0058)
; Inputs can be arrays
;
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180/!pi
; d[pc]=1/px["]
pc=(a_unit/sin((1./3600)/RAD))/1000	; [km]
semi_axis2=0
;
if keyword_set(px) then d=1/px	; [PC] 
;
semi_axis=k1*sqrt(1.0-e^2)*p*13751.0	; [km]
if n_elements(k2) gt 0 then $
semi_axis2=k2*sqrt(1.0-e^2)*p*13751.0	; [km]
if n_elements(k2) gt 0 then semi_axis=semi_axis+semi_axis2
if keyword_set(px) then begin
	semi_axis=atan((semi_axis/(d*pc))*RAD)*3600.*1000. ; [mas]
endif
;
return,semi_axis
;
end
;-------------------------------------------------------------------------------
pro a12sini_binaries
;
; Given e and p[d], compute a1sini from k1 [km] and a2sini [km] from k2 [km/s]
; See Batten, Fletcher and MacCarthy, 8th catalogue, page 2.
;
common StarBase,StarTable,Notes
;
index=where(StarTable.e ge 0,count)
;
StarTable[index].a1sini= $
	StarTable[index].k1 $
	*(sqrt(1.0-StarTable[index].e^2)*StarTable[index].p*13751.0)
;
StarTable[index].a2sini= $
	StarTable[index].k2 $
	*(sqrt(1.0-StarTable[index].e^2)*StarTable[index].p*13751.0)
;
end
;-------------------------------------------------------------------------------
pro k_binaries
;
; Estimate K1 and K2 for SB1 with orbits and distances.
;
common StarBase,StarTable,Notes
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
index=where((StarTable.a gt 0) $
	and (StarTable.i gt 0) $
	and (StarTable.d gt 0) $
	and (StarTable.e lt 1) $
	and (StarTable.p gt 0),count)
RAD=180.0/!pi
sini=sin(StarTable[index].i/RAD)
if count gt 0 then begin
	asini=StarTable[index].a*StarTable[index].d*a_unit*0.001*sini	; [km]
	Startable[index].a2sini=asini-StarTable[index].a1sini
	StarTable[index].k1= $
		StarTable[index].a1sini $
		/(sqrt(1.0-StarTable[index].e^2)*StarTable[index].p*13751.0)
	StarTable[index].k2= $
		StarTable[index].a2sini $
		/(sqrt(1.0-StarTable[index].e^2)*StarTable[index].p*13751.0)
endif
index=where(StarTable.i eq 0,count)
if count gt 0 then begin
	StarTable[index].k1=0
	StarTable[index].k2=0
endif
;
end
;-------------------------------------------------------------------------------
pro m_binaries
;
; Obtain mass estimate for secondary in a binary if it is not classified but
; binary is double lined. Requires primary mass, and m1/2sin(i)^3.
;
common StarBase,startable,notes
;
index=where((startable.mass ne 0) $
	and (startable.m1sin3i ne 0) $
	and (startable.m2sin3i ne 0),count)
print,'Number of double-lined single-classification binaries =',count
if count gt 0 then startable[index].mass2= $
	startable[index].mass*startable[index].m2sin3i $
			      /startable[index].m1sin3i
;
index=where((startable.mass ne 0) $
	and (startable.mf ne 0) $
	and (startable.type2 eq 0),count)
print,'Number of single-lined single-classification binaries =',count
if count gt 0 then begin
	c0=-startable[index].mass^2
	c1=-2*startable[index].mass
	c2=fltarr(count)-1
	c3=sin(!pi/4)^3/startable[index].mf
	for i=0,count-1 do begin
		s=cuberoot[[c0[i],c1[i],c2[i],c3[i]]]
		startable[index[i]].mass2=s[0]
	endfor
endif
;
end
;-------------------------------------------------------------------------------
pro mf_binaries,p,k1,e
;
; Compute mass function from measured quantities P, K, and e of single lined
; binaries.
;
common StarBase,startable,notes
;
G=6.673e-11
m_Sun=2.0e30
;
if n_params() eq 0 then begin
	index=where(startable.mass ne 0 and startable.k1 ne 0,count)
	index=where(startable.p ne 0 and startable.k1 ne 0,count)
	if count gt 0 then begin
		p_SI=startable.p*86400.
		k_SI=startable.k1*1e3
		e=startable.e
		startable[index].mf=(p_SI*k_SI^3*(1-e^2)^(3./2)/(2*!pi*G))/M_Sun
	endif
	return
endif
;
p_SI=p*86400.
k_SI=k1*1e3
;
if n_params() eq 3 then print,(p_SI*k_SI^3*(1-e^2)^(3./2)/(2*!pi*G))/M_Sun
;
end
;-------------------------------------------------------------------------------
pro teff_binaries
;
common StarBase,startable,notes
;
end
;************************************************************************Block 9
pro compile_sb2
;
common StarBase,StarTable,Notes
;
; Get Batten's catalog and make copy starbase
get_bat
starbase=StarTable
bat_starids=StarTable.starid
;
; Read Sky Catalogue 2000.0
rename_starids,'bat-hdn'
read_catalogs
;
; Save parallaxes
px=StarTable.px
;
; Read parallaxes and photometry from auxilliary catalogs
get_parallax
get_ubvri
get_ubv
;
; Replace the parallaxes from the Sky Catalog
index=where(px lt 0.01,count)
if count gt 0 then begin
	StarTable[index].px=px[index]
	StarTable[index].pxe=0.01
endif
;
; Update starbase color indices, radial velocities, and parallaxes
starbase.ub=StarTable.ub
starbase.bv=StarTable.bv
starbase.ri=StarTable.ri
starbase.rv=StarTable.rv
starbase.px=StarTable.px
starbase.pxe=StarTable.pxe
starbase.hdn=StarTable.hdn
starbase.sao=StarTable.sao
starbase.ads=StarTable.ads
;
; Get positions from SAO catalog
rename_starids,'hdn-sao'
read_catalogs
;
; Get WDS number
rename_starids,'sao-wds'
starbase.wds=StarTable.wds
;
; Update positions with FK5 catalog and copy to data base
StarTable.starid=bat_starids
rename_starids,'bat-fkv'
read_catalogs
starbase.fkv=StarTable.fkv
starbase.ra=StarTable.ra
starbase.dec=StarTable.dec
starbase.pmra=StarTable.pmra
starbase.pmdec=StarTable.pmdec
;
; Get FLN
StarTable.starid=bat_starids
rename_starids,'bat-fln'
starbase.fln=StarTable.fln
;
; Get star names from Bright Star Catalog
StarTable.starid=bat_starids
rename_starids,'bat-bsc'
read_catalogs
starbase.bsc=StarTable.bsc
;
; Replace StarTable with starbase
StarTable=starbase
;
; Do the math
amv_stars
d_stars
mass_stars
m_binaries
i_binaries
a_binaries
;
index=where(StarTable.a ne 0,count)
print,'Number of non-zero semi-major axis estimates :',count
;
end
;-------------------------------------------------------------------------------
pro compile_sb1,plots=plots
;
common StarBase,StarTable,Notes
;
get_wds & get_leeuwen
st_wds=startable[where(startable.hic ne 0)]; two stars with HIC=0(?)
get_sb9 
st_sb9=startable
;
; Assign K1, K2, and RV to WDS stars by matching the period
index=where(st_wds.hic ne 0)
jndex=whereequal(st_wds[index].hic,st_sb9.hic)
for i=0,n_elements(jndex)-1 do begin
	k=where(st_sb9.hic eq st_wds[index[jndex[i]]].hic)
	for l=0,n_elements(k)-1 do begin
	if abs(st_sb9[k[l]].p/st_wds[index[jndex[i]]].p-1) lt 0.05 then begin
	st_wds[index[jndex[i]]].k1=st_sb9[k[0]].k1
	st_wds[index[jndex[i]]].k2=st_sb9[k[0]].k2
	st_wds[index[jndex[i]]].rv=st_sb9[k[0]].rv
	endif
	endfor
endfor
;
index=where(st_wds.hdn ne 0)
jndex=whereequal(st_wds[index].hdn,st_sb9.hdn)
for i=0,n_elements(jndex)-1 do begin
	k=where(st_sb9.hdn eq st_wds[index[jndex[i]]].hdn)
	for l=0,n_elements(k)-1 do begin
	if abs(st_sb9[k[l]].p/st_wds[index[jndex[i]]].p-1) lt 0.05 then begin
	st_wds[index[jndex[i]]].k1=st_sb9[k[0]].k1
	st_wds[index[jndex[i]]].k2=st_sb9[k[0]].k2
	st_wds[index[jndex[i]]].rv=st_sb9[k[0]].rv
	endif
	endfor
endfor
;
; Fix some issues
index=where(st_wds.hic eq 2912)
st_wds[index].k2=0	; Pi And: secondary curve bad
;
index=where(st_wds.k1 ne 0 and st_wds.k2 eq 0)
st_wds=st_wds[index]
;
ustars=unique(st_wds.starid,ui)
for i=0,n_elements(ustars)-1 do begin
	index=where(st_wds.starid eq ustars[i],count)
	if count ge 2 then begin
		a=medianve(st_wds[index].a,ae)
		p=medianve(st_wds[index].p,pe)
		if ae/a le 0.05 and pe/p le 0.05 then begin
			st_wds[index].a=a
			st_wds[index].p=p
		endif else begin
			st_wds[index].k1=0
			if ae/a gt 0.05 or pe/p gt 0.05 then begin
				print,ustars[i],st_wds[index].a*1000
				print,ustars[i],st_wds[index].p
			endif
		endelse
	endif
endfor
index=where(st_wds.k1 ne 0)
st_wds=st_wds[index]
ustars=unique(st_wds.starid,ui)
startable=st_wds[ui]
print,'Total number of binaries:',n_elements(startable)
;
; Fix some catalog errors
i=where(startable.hic eq 11767)
startable[i].spectrum='F8II+F0V'
i=where(startable.hic eq 21476)
startable[i].spectrum='G8III'
i=where(startable.hic eq 105269)
startable[i].spectrum='F1II+B7V'
spec_parse,startable.spectrum,type,class,type2,class2
startable.type=type
startable.class=class
startable.type2=type2
startable.class2=class2
;
if n_elements(plots) eq 0 then plots=0
if not plots then return
!p.charthick=2.0
!p.charsize=2.0
!p.thick=2.0
!x.thick=2.0
!y.thick=2.0
if !d.name ne 'PS' then !p.multi=[0,2,2] else !p.multi=0
if !d.name eq 'PS' then device,filename='parallax.eps'
!x.title='Hipparcos parallax [mas]'
histograph,startable.px*1000,bin=5,max=100
if !d.name eq 'PS' then device,/close
if !d.name eq 'PS' then device,filename='period.eps'
!x.title='log(Period/d)'
histograph,alog10(startable.p),binsize=0.2
vmed=medianve(alog10(startable.p),verr)
print,'Median period and error:',vmed,verr
if !d.name eq 'PS' then device,/close
if !d.name eq 'PS' then device,filename='semiaxis.eps'
!x.title='Semi-major axis [mas]'
histograph,startable.a*1000,min=0,max=50,bin=5
if !d.name eq 'PS' then device,/close
if !d.name eq 'PS' then device,filename='magnitude.eps'
!x.title='V magnitude'
histograph,startable.mv,min=-2,max=10,bin=0.5
vmed=medianve(startable.mv,verr)
print,'Median magnitude and error:',vmed,verr
if !d.name eq 'PS' then device,/close
;
end
;-------------------------------------------------------------------------------
pro compile_sb0
;
; Compile single star list with V<6, px>10mas, luminosity class non-zero
;
common StarBase,StarTable,Notes
;
get_jsdc
i=where(startable.mv le 6)
startable=startable[i]
i=where(startable.px gt 10)
startable=startable[i]
t=startable
rename_starids,'hip-hdn'
read_catalogs
t.type=startable.type
t.class=startable.class
t.spectrum=startable.spectrum
i=where(t.class ne 0,n)
t=t[i]
startable=t
;
mv=startable.mv
bv=startable.bv
px=startable.px
amv=mv-(-5+5*alog10(1000./px))
teff=teff_bv(bv)
mbol=amv+bc_teff(teff)
llum=-(mbol-4.74)/2.5
ltff=alog10(teff)
;
restore,'tracks.xdr'
startable.teff=teff
startable.amv=amv
;
for i=0,n-1 do begin
	d=sqrt((logl-llum[i])^2+(logt-ltff[i])^2)
	index=where(d eq min(d))
	startable[i].mass=mass[index]
endfor
;
end
;-------------------------------------------------------------------------------
pro compile_speckle,date,rho,theta
;
; With orbital elements stored in the startable, compute separations for given 
; date and select all those double stars with separations less than 200 mas.
;
common StarBase,StarTable,Notes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
;
; get_wds
;
if n_elements(StarTable) eq 0 then begin
	print,'***Error(COMPILE_SPECKLE): no StarTable!'
	return
endif
if n_elements(date) eq 0 then begin
	print,'***Error(COMPILE_SPECKLE): no date specified!'
	return
endif
n=n_elements(StarTable)
if n eq 0 then return
o_parms=dblarr(8,n)
rho=dblarr(n)
theta=rho
;
o_parms[0,*]=StarTable.a
o_parms[1,*]=StarTable.e
o_parms[2,*]=StarTable.i/rad
o_parms[2,*]=StarTable.i/rad
o_parms[3,*]=StarTable.o/rad
o_parms[5,*]=StarTable.n/rad
o_parms[6,*]=StarTable.p
o_parms[7,*]=StarTable.t
o_parms[7,*]=StarTable.t
;
parsedate,date,y,m,d
jd=julian(y,m,d)
;
for i=0,n-1 do begin
	xy=true2app(jd,o_parms[*,i],r,t,StarTable[i].ra,StarTable[i].dec)
	rho[i]=r
	theta[i]=t
endfor
;
index=where(rho lt 0.150,count)
if count gt 0 then begin
	StarTable=StarTable[index]
	rho=rho[index]
	theta=theta[index]
endif else print,'No stars found!'
;
end
;-------------------------------------------------------------------------------
pro compile_bbc
;
common StarBase,StarTable,Notes
;
; Get Batten's catalog and make copy starbase
get_bat
starbase=StarTable
bat_starids=StarTable.starid
;
; Update positions and parallaxes with Hipparcos data
StarTable.starid=bat_starids
rename_starids,'bat-hip'
read_catalogs
starbase.hic=StarTable.hic
starbase.ra=StarTable.ra
starbase.dec=StarTable.dec
starbase.pmra=StarTable.pmra
starbase.pmdec=StarTable.pmdec
starbase.px=StarTable.px
starbase.pxe=StarTable.pxe
;
; Update with data from Sky Catalogue 2000.0
; rename_starids,'bat-hdn'
; read_catalogs
;
; Save parallaxes
; px=StarTable.px
;
; Read parallaxes and photometry from auxilliary HDN indexed catalogs
; get_parallax
; get_ubvri
; get_ubv
;
; Reinstate small parallaxes with values from the Sky Catalog
; index=where(px lt 0.01,count)
; if count gt 0 then begin
; 	StarTable(index).px=px(index)
; 	StarTable(index).pxe=0.01
; endif
;
; Update starbase color indices, radial velocities, and parallaxes
; starbase.ub=StarTable.ub
; starbase.bv=StarTable.bv
; starbase.ri=StarTable.ri
; starbase.rv=StarTable.rv
; starbase.px=StarTable.px
; starbase.pxe=StarTable.pxe
; starbase.hdn=StarTable.hdn
; starbase.sao=StarTable.sao
; starbase.ads=StarTable.ads
;
; Get HDN numbers
StarTable.starid=bat_starids
rename_starids,'bat-hdn'
starbase.hdn=StarTable.hdn
;
; Get WDS numbers
StarTable.starid=bat_starids
rename_starids,'bat-wds'
starbase.wds=StarTable.wds
;
; Get FLN numbers
StarTable.starid=bat_starids
rename_starids,'bat-fln'
starbase.fln=StarTable.fln
;
; Replace StarTable with starbase
StarTable=starbase
;
; Fill in unknown data with estimates
; index=where(StarTable.type2 eq 0,count)
; if count gt 0 then StarTable(index).type2=StarTable(index).type
; if count gt 0 then StarTable(index).class2=StarTable(index).class
;
; Where class/class2=0 set to default (5=dwarf)
set_defclass
;
; Estimate semi-major axes and magnitude differences
d_stars		; Compute distance in pc from parallax
mass_stars	; Estimate primary mass from spectral type
m_binaries	; Estimate secondary mass for SB2s
mass_stars,1	; Estimate sec. mass from spectral classification
		; Deal with SB1s w/out secondary classification
a_binaries	; Estimate semi-major axis with Kepler's 3rd
; i_binaries
;
amv_stars	; Estimate abs. V from spectral types
; For unclassified secondaries, derive MV from mass assuming MS
index=where(startable.dmv eq 100 and startable.mass2 ne 0)
lum2=lum_mass(startable[index].mass2)
teff2=teff_mass(startable[index].mass2)
mv2=-2.5*alog10(lum2)+4.74-bc_teff(teff2)
startable[index].dmv=mv2-startable[index].amv
startable[index].amv=cmag(startable[index].amv,mv2)
;
; Select stars observable by NPOI
t=StarTable
i=where(t.a gt 0.001 and t.a lt 0.2 and t.dec gt -20 and t.mv lt 6.0)
StarTable=t[i]
;
; Eliminate some notorious stars
;
; Recurrent novae RS Oph
i=where(StarTable.bat ne 996)
StarTable=StarTable[i]
;
print,'Number of elements in BBC :',n_elements(StarTable)
;
end
;-------------------------------------------------------------------------------
pro compile_bbcsouth
;
common StarBase,StarTable,Notes
;
; Get Batten's catalog and make copy starbase
get_bat
starbase=StarTable
bat_starids=StarTable.starid
;
; Read Sky Catalogue 2000.0
rename_starids,'bat-hdn'
read_catalogs
;
; Save parallaxes
px=StarTable.px
;
; Read parallaxes and photometry from auxilliary catalogs
get_parallax
get_ubvri
get_ubv
;
; Replace the parallaxes from the Sky Catalog
index=where(px lt 0.01,count)
if count gt 0 then begin
	StarTable[index].px=px[index]
	StarTable[index].pxe=0.01
endif
;
; Update starbase color indices, radial velocities, and parallaxes
starbase.ub=StarTable.ub
starbase.bv=StarTable.bv
starbase.ri=StarTable.ri
starbase.rv=StarTable.rv
starbase.px=StarTable.px
starbase.pxe=StarTable.pxe
starbase.hdn=StarTable.hdn
starbase.sao=StarTable.sao
starbase.ads=StarTable.ads
;
; Get WDS number
rename_starids,'hdn-wds'
starbase.wds=StarTable.wds
;
; Update positions with FK5 catalog and copy to data base
StarTable.starid=bat_starids
rename_starids,'bat-fkv'
read_catalogs
starbase.fkv=StarTable.fkv
starbase.ra=StarTable.ra
starbase.dec=StarTable.dec
starbase.pmra=StarTable.pmra
starbase.pmdec=StarTable.pmdec
;
; Get FLN
StarTable.starid=bat_starids
rename_starids,'bat-fln'
starbase.fln=StarTable.fln
;
; Get star names from Bright Star Catalog
StarTable.starid=bat_starids
rename_starids,'bat-bsc'
read_catalogs
get_jhk_bsc
starbase.bsc=StarTable.bsc
starbase.mj=StarTable.mj
starbase.mh=StarTable.mh
starbase.mk=StarTable.mk
;
; Replace StarTable with starbase
StarTable=starbase
;
; Fill in unknown data with estimates
index=where(StarTable.type2 eq 0,count)
if count gt 0 then StarTable[index].type2=StarTable[index].type
if count gt 0 then StarTable[index].class2=StarTable[index].class
set_defclass
;
; Do the math
amv_stars
d_stars
mass_stars
m_binaries
i_binaries
a_binaries
;
t=StarTable
i=where(t.a gt 0.001 and t.a lt 0.2 and t.dec lt 20 and t.mk lt 4.0)
StarTable=t[i]
;
; Eliminate some notorious stars
;
; Recurrent novae RS Oph
i=where(StarTable.bat ne 996)
StarTable=StarTable[i]
;
print,'Number of elements in BBC (south) :',n_elements(StarTable)
;
end
;-------------------------------------------------------------------------------
pro compile_wdssouth,rho,bible=bible,simbad=simbad
;
; Compile list of binaries with orbits from WDS in the southern hemisphere.
; If keyword "bible" is set, use the bible of all binary detections.
; If keyword "simbad" is set, use the WDS entries in SimBad (yields mags).
; For known orbits, compute separation rho ["] at the current date.
;
common StarBase,StarTable,Notes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(bible) eq 0 then bible=0
if n_elements(simbad) eq 0 then simbad=0
if simbad and bible then begin
	print,'Please choose only one option: bible or simbad!'
	return
endif
;
orbit_options=alloc_orbit_options()
orbit_options.component='A-B'
;
IF bible THEN BEGIN
	get_wbs
	rho=startable.a
ENDIF ELSE IF simbad THEN BEGIN
	restore,!catalogs_dir+'/simbad/wdssim.xdr'
ENDIF ELSE BEGIN
	get_wds
	nb=n_elements(startable)
;
	b_parms=dblarr(7,nb)
	b_parms[0,*]=startable.a
	b_parms[1,*]=startable.e
	b_parms[2,*]=startable.i*!pi/180
	b_parms[3,*]=((startable.o+180) mod 360)*!pi/180
	b_parms[4,*]=startable.n*!pi/180
	b_parms[5,*]=startable.p
	b_parms[6,*]=startable.t-2440000.d0
;
	rho=fltarr(nb)
;
	parseidldate,systime(),y,m,d
	jd=julian(y,m,d)
;
	!quiet=1
	for ib=0,nb-1 do begin
		o_parms=b_parms[*,ib]
		setmodel
		rhotheta=binarypos(jd)
		rho[ib]=rhotheta[0]
	endfor
	!quiet=0
ENDELSE
;
; Select southern targets for VLTI
index=where(startable.dec lt 20,count)
if count ge 1 then begin
	startable=startable[index]
	rho=rho[index]
endif
;
; Process spectrum information
spec_parse,startable.spectrum,type,class,type2,class2
startable.type=type
startable.class=class
startable.type2=type2
startable.class2=class2
;
end
;-------------------------------------------------------------------------------
pro compile_diameter
;
; Compile file diameter.bsc, which contains all BSC stars for which diameter
; information could be obtained. Also obtain information on duplicity, making
; this file useful for selecting calibrators for optical interferometers.
;
common StarBase,StarTable,Notes
;
; Get StarTable based on Bright Star Catalogue
get_bsc
index=where(startable.mv lt 5.5,num_bsc)
startable=startable[index]
;
; Initialize comments array
comments=strarr(num_bsc)
;
; Obtain Hipparcos parallaxes, and also update spectral types
t=startable
rename_starids,'bsc-hip'
read_catalogs
t.spectrum=startable.spectrum
t.class=startable.class
t.type=startable.type
t.px=startable.px
startable=t
;
; Assume that still unclassified stars are giants
index=where(startable.class eq 0)
startable[index].class=3
;
; Compute T_eff and log(g) for limb darkening
teff_stars
logg_stars
;
; Compute limb darkening coefficients
limb_stars
;
; For unclassified stars, set coefficients so that d_550/d_800=0.967
; Also assume they are giants, which is OK for estimated diameters here
startable[index].a0=1.28
startable[index].a1=-1.6/1e3
startable[index].a2=0
;
; Compute (B-V)0 and from that visual extinction Av
bv=startable.bv
bv_stars
av=3.1*((bv-startable.bv)>0)
;
; Obtain Av from extinction maps and correct V for extinction
print,'Estimating extinction...'
av_stars,/drimmel
index_av=where(bv eq 100 or abs(startable.av-av) gt 0.5,count_av)
if count_av gt 0 then startable[index_av].av=0
startable.mv=startable.mv-startable.av
index=where(startable.av ne 0,count)
if count gt 0 then comments[index]='Av '
;
; Compute estimated diameter from photometry for all stars (binary/single)
print,'Estimating diameters...'
startable.bflag='.'
; diameter_ri	; This is the UD diameter
; We now use V-K calibration because 2MASS gives us all the IR photometry
get_jhk_bsc
diameter_vk	; This is the LD diameter
predicted=startable.diameter
;
; Read various lists; each one supercedes the previous one!
; Read Infrared Flux Method diameters
print,'Reading IRFM diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.irfm',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='irfm '
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i];*sqrt((1-7*u/15)/(1-u/3))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 1st publication
print,'Reading DM1 diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm1',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm1 '
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt(((1-u/3)/(1-7*u/15)))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 2nd publication
print,'Reading DM2 diameters...'
bsc=0
mv=0.0
ri=0.0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm2',bsc,mv,ri,d,/col, $
	resize=[1,2,3,4],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm2 '
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt(((1-u/3)/(1-7*u/15)))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Tyler Nordgren's NPOI diameters
print,'Reading TN diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.tn',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='tn '
		u=poly(750.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt(((1-u/3)/(1-7*u/15)))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Brian's (Kloppenborg?) single star list
print,'Reading single star list...'
bsc=0
s=dc_read_free(!oyster_dir+'starbase/calibrators.brian',bsc,/col, $
	resize=[1],ignore=['!'])
for i=0,n_elements(bsc)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]=comments[j]+'S '
	endif
endfor
;
; Read JPL calibrator list
print,'Reading JPL calibrator list...'
n='                                                                            '
n=n+n
s=dc_read_fixed(!oyster_dir+'starbase/calibrators.jpl',n,/col, $
	resize=[1],ignore=['#'],format='(a132)')
hdn=''
s=dc_read_free(!oyster_dir+'starbase/calibrators.jpl',hdn,/col, $
	resize=[1],ignore=['#'])
bsc=cri(long(strmid(hdn,3,6)),'hdn-bsc')
for i=0,n_elements(bsc)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		if strpos(n[i],'Dbl') eq -1 and strpos(n[i],'SB') eq -1 then $
		comments[j]=comments[j]+'JPL '
	endif
endfor
;
; Potential calibrators have d < 1.1 mas
index=where(startable.diameter lt 1.1)
startable[index].bflag='C'
; Stars with unreliable colors should not be used as calibrators
; unless they have really small apparent diameters (< 0.5 mas)
if count_av gt 0 then begin
	comments[index_av]=comments[index_av]+'(B-V)! '
	startable[index_av].bflag='.'
	index=where(startable[index_av].diameter le 0.5,count)
	if count gt 0 then startable[index_av[index]].bflag='C'
endif
;
; Check with Batten's catalogue for known spectroscopic binaries
bat=cri(startable.bsc,'bsc-bat')
index=where(bat ne -1)
startable[index].bflag='B'
startable.zerospacing=1
;
; Check with Hipparcos catalog for multiplicity information
table=startable
rename_starids,'bsc-hip'
read_catalogs
table[where(startable.hflag eq 'C' and startable.a gt 0 $
				   and startable.a lt 5)].bflag='B'
table[where(startable.hflag eq 'O')].bflag='B'
table.a=startable.a
table.dmv=startable.dmv
table.hflag=startable.hflag
startable=table
;
; Check against BCR
table=startable	; startable will be modified
; get_bcr	; outdated
bad_cals	; new, bad cal. will have bflag='.' instead of 'C'
; index=whereequal(table.hdn,startable.hdn)
; startable=table
; if index(0) ge 0 then begin
	jndex=where(table.bflag eq 'C' and startable.bflag eq '.',count)
	if count gt 0 then comments[jndex]=comments[jndex]+'BCR '
; endif
;
; These are known binaries missed (so far) by the algorithm
; Tau Ori: listed as double in Simbad, poor results in SigOri reductions
j=where(startable.starid eq 'BSC1735',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Suspicious'
endif
; HR 6554 has a close companion which makes acquisition difficult
; (Information from R. Zavala May 6, 2013)
j=where(startable.starid eq 'BSC6554',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Physical'
endif
; HR 5774 has a close companion which makes acquisition difficult
; (Information from R. Zavala April 18, 2013)
j=where(startable.starid eq 'BSC5774',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Physical'
endif
; Upsilon Tauri (occultation double)
j=where(startable.starid eq 'BSC1392',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Occultation'
endif
;
; Theta 1 Tauri (WDS)
j=where(startable.starid eq 'BSC1411',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Hyades'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC1427',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=200 mas'
endif
;
; Zeta Orionis (NPOI)
j=where(startable.starid eq 'BSC1948',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=42 mas'
endif
;
; Nu Geminorum (WDS)
j=where(startable.starid eq 'BSC2343',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=80 mas'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC5733',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=50-100 mas'
endif
;
; Sigma Herculis (WDS)
j=where(startable.starid eq 'BSC6168',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=67 mas'
endif
;
; Zeta Draconis (WDS)
j=where(startable.starid eq 'BSC6396',count) & j=j[0]
if count eq 1 then begin
        startable[j].bflag='B'
        comments[j]=comments[j]+'WDS'
endif                                                                           
;
; Zeta Sagittae (WDS)
j=where(startable.starid eq 'BSC7546',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=146 mas'
endif
;
; FKV0782
j=where(startable.starid eq 'BSC7955',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Griffin SB'
endif
;
; FKV0131
j=where(startable.starid eq 'BSC1122',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'NPOI'
endif
;
; FKV0622 (BSC6175)
j=where(startable.starid eq 'BSC6175',count) & j=j[0]
if count eq 1 then begin
	startable[j].diameter=0.510
	comments[j]=comments[j]+'Gordon ea. 2018'
endif
;
; Update some comments
j=where(startable.starid eq 'BSC2421',count) & j=j[0]
if count eq 1 then begin
	comments[j]=comments[j]+'hb 1.32 mas @ 443 nm'
endif
j=where(startable.starid eq 'BSC5340',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='.'
	comments[j]=comments[j]+'aq=19.86 mas'
endif
;
; Write diameters to file
num_stars=n_elements(startable)
if n_elements(comments) ne num_stars then comments=strarr(num_stars)
openw,unit,'diameter.bsc',/get_lun
printf,unit, $
'! NOTE: THIS FILE IS GENERATED BY OYSTER. ANY MODIFICATIONS'
printf,unit, $
'! SHOULD BY MADE TO COMPILE_DIAMETER.'
printf,unit, $
'!'
printf,unit, $
'!This file is read by OYSTER to obtain limb-darkened diameters of'
printf,unit, $
'!calibrator stars. These were either taken from observations listed below,'
printf,unit, $
'!or were estimated based on (V-K) and V  according to Mozurkewich et'
printf,unit, $
'!al. 2003 and van Belle et al. 2009. Estimates are +/- 5%, except possibly'
printf,unit, $
'!for supergiants.  Also given are coefficients of a quadratic fit to the'
printf,unit, $
'!linear limb darkening coefficients for the interval [450 nm,850 nm]:'
printf,unit, $
'!u_l=a0 + a1 * lambda + a2 * lambda^2, lambda = wavelength in nm.  Stars'
printf,unit, $
'!without luminosity class were assumed to be giants, and limb darkening'
printf,unit, $
'!fit coefficients will give u_l=0.0 at 800 nm, u_l=0.4 at 550 nm.  ZERO is'
printf,unit, $
'!the zero spacing visibility (unused). There is a flag to indicate type of'
printf,unit, $
'!star: B/C/. = binary or multiple/calibrator/everything else.  A calibrator'
printf,unit, $
'!is meant here to be a single star less than 1 mas in diameter.'
printf,unit, $
'!'
printf,unit, $
'!Stars found in the Batten catalog of spectroscopic binary orbits, as'
printf,unit, $
'!well as stars having C and O solutions in the Hipparcos catalog were'
printf,unit, $
'!labeled as multiple. There is a number of stars labeled as multiple'
printf,unit, $
'!based on previous Mark III and NPOI observations.'
printf,unit,'!'
printf,unit, $
'! Measurements:'
printf,unit, $
'! irfm  Infrared Flux Method (Blackwell & Shallis 1977)'
printf,unit, $
'! dm1:  Mozurkewich et al. 1991, first MarkIII diameter paper'
printf,unit, $
'! dm2:  Mozurkewich et al. 2003, second MarkIII diameter paper'
printf,unit, $
'! tn :  Nordgren et al. 1999, NPOI diameter paper'
printf,unit, $
'!'
printf,unit, $
'! Qualifier:'
printf,unit, $
'! Av: Non-zero Av correction applied to V based on E(B-V)'
printf,unit, $
'! (B-V)!: Unexpected (B-V) color, should not be used as calibrator'
printf,unit, $
'! S:    Brian Mason single star list'
printf,unit, $
'! JPL:  JPL calibrator candidate'
printf,unit, $
'! WDS:  Washington Double Star catalog'
printf,unit, $
'! BCR:  Bad Calibrator Registry'
printf,unit, $
'!       (http://www.eso.org/sci/observing/tools/bcr/bcr.html)'
printf,unit,'!'
printf,unit,'! BSC LD[mas]  ZERO    a0     a1     a2  cflag Comment'
printf,unit,'!'
for i=0,num_stars-1 do begin
	if startable[i].diameter ne 0 and startable[i].diameter lt 99 then begin
	printf,unit,startable[i].bsc,startable[i].diameter, $
		startable[i].zerospacing,startable[i].a0, $
		startable[i].a1*1000,startable[i].a2*1000000, $
		startable[i].bflag,startable[i].name,' '+comments[i], $
		format='(i4.4,2x,f6.3,2x,f4.2,2x,3(f6.3,1x),2x,a1,2x,a,a)'
	endif
endfor
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro compile_calmaster
;
; Compile file diameter.hdn, which contains all HDN stars for which diameter
; information could be obtained. Also obtain information on duplicity, making
; this file useful for selecting calibrators for optical interferometers.
;
; This procedure will make use of the Borde, Merand, Verhoelst, and Van Belle
; calibrator lists.
;
common StarBase,StarTable,Notes
;
; Get StarTable based on extended HDN catalogue
get_hdn
num_hdn=n_elements(startable)
;
; Estimate diameters based on surface brightness
get_jhk_hdn
diameter_vk
startable.model='LD'
startable.bflag='.'
startable.sflag='5'
predicted=startable.diameter
;
; Compute T_eff and log(g) for limb darkening
teff_stars	; Teff will be zero for stars w/out full classification
logg_stars
;
; Compute limb darkening coefficients
limb_stars,[450,2500],/logarithmic
;
table=startable
;
; Read various calibrator catalogs
get_borde,/init
;
; Initialize comments array
comments=strarr(num_hdn)
;
; Read various lists; each one supercedes the previous one!
; Read Infrared Flux Method diameters
print,'Reading IRFM diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.irfm',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='irfm'
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt((1-7*u/15)/(1-u/3))
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 1st publication
print,'Reading DM1 diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm1',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm1'
		diam=d[i]
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 2nd publication
print,'Reading DM2 diameters...'
bsc=0
mv=0.0
ri=0.0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm2',bsc,mv,ri,d,/col, $
	resize=[1,2,3,4],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm2'
		diam=d[i]
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Tyler's NPOI diameters
print,'Reading TN diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.tn',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='tn'
		diam=d[i]
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read bad calibrator CSV list from 
; http://www.astro.lsa.umich.edu/~pmuirhea/calib.html
print,'Reading bad calibrators...'
bsc=0
d=0.0
s=dc_read_fixed(!oyster_dir+'starbase/diameter.bad',l,/col, $
	ignore=['!'],format='(a300)')
for i=0,n_elements(l)-1 do begin
	v=nameparse(l[i],',')
	hdn=cri(v[0],'hdn')
	j=where(startable.hdn eq hdn,count) & j=j[0]
	if count eq 1 then begin
		comments[j]='bad'
	endif
endfor
;
; Potential calibrators have d < 1 mas
index=where(startable.diameter lt 1)
startable[index].bflag='C'
;
; Check with Batten's catalogue for known spectroscopic binaries
bat=cri(startable.bsc,'bsc-bat')
index=where(bat ne -1)
startable[index].bflag='B'
startable.zerospacing=1
;
; Check with Hipparcos catalog for multiplicity information
table=startable
rename_starids,'bsc-hip'
read_catalogs
table[where(startable.hflag eq 'C' and startable.a gt 0 $
				   and startable.a lt 5)].bflag='B'
table[where(startable.hflag eq 'O')].bflag='B'
table.a=startable.a
table.dmv=startable.dmv
table.hflag=startable.hflag
startable=table
;
; These are known binaries missed (so far) by the algorithm
;
; Upsilon Tauri (occultation double)
j=where(startable.starid eq 'BSC1392',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Occultation'
endif
;
; Theta 1 Tauri (WDS)
j=where(startable.starid eq 'BSC1411',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Hyades'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC1427',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=200 mas'
endif
;
; Zeta Orionis (NPOI)
j=where(startable.starid eq 'BSC1948',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=42 mas'
endif
;
; Nu Geminorum (WDS)
j=where(startable.starid eq 'BSC2343',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=80 mas'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC5733',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=50-100 mas'
endif
;
; Sigma Herculis (WDS)
j=where(startable.starid eq 'BSC6168',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=67 mas'
endif
;
; Zeta Draconis (WDS)
j=where(startable.starid eq 'BSC6396',count) & j=j[0]
if count eq 1 then begin
        startable[j].bflag='B'
        comments[j]=comments[j]+'WDS'
endif                                                                           
;
; Zeta Sagittae (WDS)
j=where(startable.starid eq 'BSC7546',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=146 mas'
endif
;
; FKV0782
j=where(startable.starid eq 'BSC7955',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Griffin SB'
endif
;
; FKV0131
j=where(startable.starid eq 'BSC1122',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'NPOI'
endif
;
; Update some comments
j=where(startable.starid eq 'BSC2421',count) & j=j[0]
if count eq 1 then begin
	comments[j]=comments[j]+'hb 1.32 mas @ 443 nm'
endif
j=where(startable.starid eq 'BSC5340',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='.'
	comments[j]=comments[j]+', aq=19.86 mas'
endif
;
; Write diameters to file
; put_diameter,comments
;
end
;-------------------------------------------------------------------------------
pro compile_cal,starids
;
; Compiles astrophysical data for calibrators. Creates startable from
; starids or uses existing startable. The base catalog ID should be HDN,
; but is converted to CAL.
;
; The diameter are LD computed from (V-K) and V. A 'K' or 'N' is added
; to indicate whether the calibrator is for AMBER or MIDI. This, for now,
; is decided on whether or not the star has a significant 12 micron flux.
;
common StarBase,StarTable,Notes
;
if n_elements(starids) ne 0 then get_startable,starids
;
; Get program IDs from notes, if available
read_notes
for i=0,n_elements(startable)-1 do begin
	list_note,startable[i].starid,note
	list_keyword,note,'PROGRAM',value
	startable[i].reference=value
endfor
;
index=where(startable.class lt 2,count)
if count gt 0 then startable[index].class=5
;
get_jhk_hdn
diameter_vk
startable.model='LDK'
;
teff_stars
logg_stars
;
get_f12
index=where(startable.f12 gt 0,count)
if count gt 0 then startable[index].model='LDN'
;
; Compute a0, a1, and a2 fit coefficients to linear limb dark. law coeff. u
limb_stars,[450,2500],/logarithmic
;
; Replace HDN catalog identifier with CAL
s=startable.starid
strput,s,'CAL'
startable.starid=s
;
end
;-------------------------------------------------------------------------------
pro compile_bcr
;
; Read bad calibrator registry list calibrators.bcr (tabs), and replace
; coordinates with Simbad values as these are used by the perl scripts
; in /var/www/cgi-bin. Output file is bcr.csv for use in /var/www/html.
;
; Make sure the local file is a copy of the original /var/www/html/bcr.csv!
; (a copy of all tables and scripts exists in Public/public_html/bcr)
;
; IAU Bad Calibrator page:
; http://www.eso.org/sci/observing/tools/bcr.html
; On usg:  ssh -X w4; cd /home/web/eso/docs/2007/sci/bin/bcr/
;
common StarBase,StarTable,Notes
;
s=dc_read_fixed(!oyster_dir+'starbase/calibrators.bcr',l,/col, $
	format='(a300)')
header=l[0]
;
n=n_elements(l)
l=l[1:n-1]
n=n-1
hdn=lonarr(n)
model=strarr(n)
starid=strarr(n)
;
openw,unit,'bcr.csv',/get_lun
printf,unit,header
;
for i=0,n-1 do begin
	v=nameparse(l[i],',')
	for j=0,n_elements(v)-1 do v[j]=strtrim(v[j],2)
;	Call by coordinate
;	star=cri_simbad(v(1)+' '+v(2),t)
;	Call by HD name
	star=cri_simbad(v[0],t)
;	v(0)=star	; we need HD not HDN names on the web page
	v[1]=hms(t.ra,/simbad)
	v[2]=dms(t.dec,/simbad)
	printf,unit,strjoin(v,',')
endfor
;
free_lun,unit
;
print,'Saved to bcr.csv.'
;
end
;-------------------------------------------------------------------------------
pro compile_technical,rho,rho_min=rho_min,rho_max=rho_max, $
	ra_min=ra_min,ra_max=ra_max, $
	v_max=v_max,k_max=k_max, $
	flp_min=flp_min,f12_min=f12_min
;
; Compile list of binaries for technical tests at VLTI. Rho["]
; compile_technical,rho,rho_min=0.02,rho_max=0.1,flp_min=1,f12_min=30
; rho_min=0.05,rho_max=0.2,ra_min=10,ra_max=20,flp_min=20,f12_min=80
;
common StarBase,StarTable,Notes
;
; Compile list of southern binaries with orbits
compile_wdssouth,rho
;
; If RA range was not provided, initialize for current date
if n_elements(ra_min) eq 0 or n_elements(ra_max) eq 0 then begin
	words=nameparse(systime())
	jd0=julian(fix(words[4]),1,1)
	doy=systime(/julian)-jd0
;	Original version
	ra_min=12.+(doy-90)/7.5-2.0
	ra_max=ra_min+11
;	Rising to setting
	ra_min=(doy/365.)*24-4.0
	ra_max=ra_min+21
;	Culminating
	ra_min=(doy/365.)*24
	ra_max=(ra_min+12) mod 24
	print,'Selecting RA range: ',ra_min,ra_max
endif
if ra_max lt ra_min then begin
	ra_max=ra_max+24
	index=where(startable.ra lt ra_min,count)
	if count gt 0 then startable[index].ra=startable[index].ra+24
endif
index=where(startable.ra gt ra_min and startable.ra lt ra_max)
startable=startable[index]
startable.ra=startable.ra mod 24
rho=rho[index]
t1=startable
r1=rho
;
; Compile list of all binaries
compile_wdssouth,rho,/bible
compile_wdssouth,rho,/simbad
t2=startable
r2=rho
index=where(t2.ra gt ra_min and t2.ra lt ra_max)
t2=t2[index]
t2.ra=t2.ra mod 24
r2=r2[index]
; Merge the two lists, keep orbital binaries
index=whereequal(t2.starid,t1.starid)
t2[index].starid=''
index=where(strlen(t2.starid) ne 0)
startable=[t2[index],t1]
rho=[r2[index],r1]
;
; Select all binaries with separation greater than limit
if n_elements(rho_min) eq 0 then rho_min=min(rho)
index=where(rho ge rho_min)
startable=startable[index] & rho=rho[index]
;
; Select all binaries with separation less than limit
if n_elements(rho_max) eq 0 then begin
	rho_max=0.2
	print,'Limiting maximum separation ["] to:',rho_max
endif
index=where(rho le rho_max)
startable=startable[index] & rho=rho[index]
;
; Select on visual magnitude
if n_elements(v_max) eq 0 then begin
	v_max=12
	print,'Limiting faintest visual mag. to:',v_max
endif
index=where(startable.mv le v_max)
startable=startable[index] & rho=rho[index]
;
; Get more HDN IDs from crossindex file
st=startable
rename_starids,'wds-hdn'
st.hdn=startable.hdn
startable=st
;
; For stars with HD number, look for MID-IR fluxes in MDFC
get_mdfc
;
; Select on K magnitude
if n_elements(k_max) eq 0 then begin
	k_max=9
	print,'Limiting faintest K magnitude to:',k_max
endif
; Get JHK magnitudes from hip_2mass extension and select for k < k_max
get_2mass,k_max=k_max
;
; get_wise,k_max=k_max
; index=where(startable.mk ne 100)
; startable=startable(index)
; rho=rho(index)
;
; Select only stars with (B-V) value for Teff estimation
index=where(startable.bv ne 100)
startable=startable[index] & rho=rho[index]
;
; For the estimation of the 1pc fluxes, we need Teff and lum. class
; If the luminosity class is not known, we adopt class=5 (dwarfs)
index=where(startable.class eq 0,count)
if count gt 0 then startable[index].class=5
;
; Estimate L and N-band fluxes for those stars not found in MDFC
i=where(startable.flp eq 0 or startable.f12 eq 0)
; Restore calibration tables: rlg,rld,rng,rnd (see mdfc_cal in mymatissegui.pro)
restore,!oyster_dir+'starbase/MDFC/mdfc_ln_coeffs.xdr'
;
j=where(startable[i].class ge 4.5 and startable[i].class le 5)
startable[i[j]].flp=poly(teff_bv(startable[i[j]].bv),rld)*startable[i[j]].px^2
startable[i[j]].f12=poly(teff_bv(startable[i[j]].bv),rnd)*startable[i[j]].px^2
;
j=where(startable[i].class ge 2.5 and startable[i].class le 3.5)
startable[i[j]].flp=poly(teff_bv(startable[i[j]].bv),rlg)*startable[i[j]].px^2
startable[i[j]].f12=poly(teff_bv(startable[i[j]].bv),rng)*startable[i[j]].px^2
;
index=where(startable.flp gt 0 and startable.f12 gt 0)
startable=startable[index] & rho=rho[index]
;
; Select on L and N fluxes (Jy)
if n_elements(flp_min) eq 0 then flp_min=min(startable.flp)
if n_elements(f12_min) eq 0 then f12_min=min(startable.f12)
index=where(startable.flp gt flp_min and startable.f12 gt f12_min,count)
if count eq 0 then begin
	print,'No stars found within the specified limits!'
	return
endif
if count ge 1 then begin
	startable=startable[index] & rho=rho[index]
endif
;
; Find HD or HIC numbers
IF 0 THEN BEGIN
rename_starids,'wds-hdn'
;
index=where(startable.hdn eq 0,n)
for i=0,n-1 do begin
  star=cri_simbad('OBJ'+esoid(startable[index[i]].ra,startable[index[i]].dec),t)
  if strpos(star,'HD') ge 0 then begin
	words=nameparse(star)
	if n_elements(words) eq 2 then $
	startable[index[i]].hdn=long(words[1])
  endif
  if strpos(star,'HIC') ge 0 then begin
	words=nameparse(star)
	if n_elements(words) eq 2 then $
	startable[index[i]].hic=long(words[1])
  endif
endfor
ENDIF
;
; Estimate primary diameters from V and (V-K)
t=startable
index=where(startable.dmv eq 100.0)
startable[index].dmv=2
m1m2v=dmag(startable.mv,startable.dmv)
m1m2k=dmag(startable.mk,startable.dmv)
startable.mv=reform(m1m2v[0,*])
startable.mk=reform(m1m2k[0,*])
diameter_vk,/force
diameter=startable.diameter
startable=t
startable.diameter=diameter
index=where(startable.diameter lt 15)
startable=startable[index] & rho=rho[index]
;
; Store rho in semimajor axis and write list of binaries
startable.a=rho
;
; Return here, code following is no longer used
return
;
; Select maximum magnitude difference
index=where(startable.dmv lt 3 and startable.dmv ne 100.0)
startable=startable[index]
rho=rho[index]
;
; Get Greek letter names
startable.starid='HDN'+string(startable.hdn,format='(i6.6)')
t=startable
rename_starids,'hdn-bsc'
read_catalogs
rename_bsc
t.name=startable.name
t.spectrum=startable.spectrum
startable=t
;
; Get astrometry from Hipparcos
t=startable
rename_starids,'hdn-hip'
read_catalogs
t.ra=startable.ra
t.dec=startable.dec
t.pmra=startable.pmra
t.pmdec=startable.pmdec
startable=t
;
end
;-------------------------------------------------------------------------------
