;*******************************************************************************
; File: starplot.pro
;
; Description:
; ------------
; Contains STARBASE plotting procedures.
;
; Block directory:
; ----------------
; Block 1: plot_dmoddwarfs,plot_dmodstars,plot_colormagnitude,
;          plot_sky
;
; Block 2: query_vizier,vizier_phot,analyze_vizier_phot
;
;************************************************************************Block 1
pro plot_dmoddwarfs
;
; Plot photometric distance modulus vs trigonometric distance modulus 
; for dwarfs.
;
common StarBase,startable,notes
;
index=where((startable.class1 eq 5) $
	and (startable.mv ne +100.0) $
	and (startable.amv ne +100.0) $
	and (startable.px gt 0),count)
if count gt 0 then begin
	dmod_p=startable[index].amv-startable[index].mv
	dmod=-5*alog10(1./startable[index].px)+5
	plot,dmod,dmod_p,psym=1,title='Dwarfs', $
		xtitle='Parallax distance modulus', $
		ytitle='Photometric distance modulus'
	oplot,findgen(40)-20,findgen(40)-20
endif else print,'Insufficient data!'
;
end
;-------------------------------------------------------------------------------
pro plot_dmodstars
;
; Plot photometric distance modulus vs trigonometric distance modulus 
; for all stars.
;
common StarBase,startable,notes
;
index=where((startable.mv ne +100) $
	and (startable.amv ne +100) $
	and (startable.px gt 0),count)
if count gt 0 then begin
	dmod_p=startable[index].amv-startable[index].mv
	dmod=-5*alog10(1./startable[index].px)+5
	!p.charsize=1.5
	plot,dmod,dmod_p,psym=1,title='All stars', $
		xtitle='Parallax distance modulus', $
		ytitle='Photometric distance modulus'
	oplot,findgen(40)-20,findgen(40)-20
endif else print,'Insufficient data!'
;
end
;-------------------------------------------------------------------------------
pro plot_colormagnitude
;
; Plot HR diagram using all stars.
;
common StarBase,startable,notes
;
index=where((startable.amv ne +100) and (startable.bv ne +100),count)
!p.charsize=1.5
if count gt 0 then plot,startable[index].bv,startable[index].amv,psym=1, $
	xtitle='(B-V)',ytitle='Absolute magnitude', $
	xrange=[-0.5,2],yrange=[10,-8]
;
end
;-------------------------------------------------------------------------------
pro plot_sky,center_ra,center_dec,center=star
;
; Plot the stars of the startable on a hemisphere.
;
common StarBase,startable,notes
;
n=n_elements(startable)
if n eq 0 then begin
	print,'***Error(PLOT_SKY): startable undefined!'
	return
endif
;
if n_elements(center_ra) eq 0 or n_elements(center_dec) eq 0 then begin
	x=sin(startable[where(startable.starid ne 'FKV0000')].ra/12*!pi)
	z=cos(startable[where(startable.starid ne 'FKV0000')].ra/12*!pi)
	xmid=(min(x)+max(x))/2
	zmid=(min(z)+max(z))/2
	center_ra=atan(xmid,zmid)*12/!pi
	center_dec=(min(startable.dec)+max(startable.dec))/2
	if n_elements(star) ne 0 then begin
		index=where(startable.starid eq star,count)
		if count eq 0 then $
		index=where(strpos(startable.starid,star) ge 0,count)
		if count eq 0 then $
		index=where(strpos(startable.starid,strmid(star,2,6)) ge 0,count)
		if count eq 0 then $
		index=where(strpos(startable.starid,strmid(star,3,6)) ge 0,count)
		if count ne 0 then begin
			center_ra=startable[index].ra
			center_dec=startable[index].dec
		endif
	endif
endif
;
; Add lines of constant dec to data
num_ra=144
dec_int=10
num_lat=170/dec_int
num_pts_dec=num_ra*num_lat
dec_g=fltarr(num_pts_dec)
ra_g=fltarr(num_pts_dec)
for i=1,num_lat do begin
	dec_g[(i-1)*num_ra:i*num_ra-1]=i*dec_int-90.0
	ra_g[(i-1)*num_ra:i*num_ra-1]=findgen(num_ra)/num_ra*24.0
endfor
f_density=2	; Densify the equator
ra_g=[ra_g,findgen(num_ra*f_density)/(num_ra*f_density)*24]
dec_g=[dec_g,fltarr(num_ra*f_density)]
num_pts_dec=num_pts_dec+num_ra*f_density	; Update num_pts_dec
ra=[ra_g,startable.ra]
dec=[dec_g,startable.dec]
;
; Add lines of constant ra to data
num_dec=180
ra_int=1
num_lon=24/ra_int
num_pts_ra=num_dec*num_lon
dec_g=fltarr(num_pts_ra)
ra_g=fltarr(num_pts_ra)
for i=1,num_lon do begin
	ra_g[(i-1)*num_dec:i*num_dec-1]=(i-1)*ra_int
	dec_g[(i-1)*num_dec:i*num_dec-1]=findgen(num_dec)/num_dec*180.0-90.0
endfor
ra=[ra_g,ra]
dec=[dec_g,dec]
;
; Convert to radians
ra=ra/12*!pi
dec=dec/180*!pi
ra0=float(center_ra)/12*!pi
dec0=float(center_dec)/180*!pi
;
; Get x,y,z coordinates of stars
; z towards observer, y up (north), x right (west)
x=-cos(dec)*sin(ra)
y=+sin(dec)
z=+cos(dec)*cos(ra)
;
; Get x,y,z coordinates of center, call this the new z' axis
xz0=-cos(dec0)*sin(ra0)
yz0=+sin(dec0)
zz0=+cos(dec0)*cos(ra0)
;
; Get new x' axis with x' = y  X  z'
xx0=zz0
yx0=0.0
zx0=-xz0
length=sqrt(xx0^2+yx0^2+zx0^2)
xx0=xx0/length
yx0=yx0/length
zx0=zx0/length
;
; Get new y' axis with y' = z' X  x'
xy0=yz0*zx0-zz0*yx0
yy0=zz0*xx0-xz0*zx0
zy0=xz0*yx0-yz0*xx0
length=sqrt(xy0^2+yy0^2+zy0^2)
xy0=xy0/length
yy0=yy0/length
zy0=zy0/length
;
; Get new x' coordinates for all stars
xp=x*xx0+y*yx0+z*zx0
yp=x*xy0+y*yy0+z*zy0
zp=x*xz0+y*yz0+z*zz0
;
; Extract the grid lines
xp_g_ra=xp[0:num_pts_ra-1]
yp_g_ra=yp[0:num_pts_ra-1]
zp_g_ra=zp[0:num_pts_ra-1]
xp_g_dec=xp[num_pts_ra:num_pts_ra+num_pts_dec-1]
yp_g_dec=yp[num_pts_ra:num_pts_ra+num_pts_dec-1]
zp_g_dec=zp[num_pts_ra:num_pts_ra+num_pts_dec-1]
;
xp=xp[num_pts_ra+num_pts_dec:num_pts_ra+num_pts_dec+n-1]
yp=yp[num_pts_ra+num_pts_dec:num_pts_ra+num_pts_dec+n-1]
zp=zp[num_pts_ra+num_pts_dec:num_pts_ra+num_pts_dec+n-1]
;
; These are the visible stars
visible=where(zp ge 0,n)
xpv=xp[visible]
ypv=yp[visible]
stars=startable[visible].starid
;
; Plot them
nummax=65
if n lt nummax then psym=2 else psym=3
!p.multi=0
if !d.name eq 'X' then begin
	window,xsize=!xsize,ysize=!xsize,/free
	lc=1
	fc=4
	bc=5
endif else if !d.name eq 'PS' then begin 
	width=17.78
        factor=width/!xsize
        device,xsize=factor*!xsize,ysize=factor*!ysize, $
                xoffset=1,yoffset=1,color=tci(1)
	lc=0
	fc=5
	bc=4
endif else begin
        print,'***Error(SKY_PLOT): invalid device name!'
        return
endelse                                                                         
plot,xpv,ypv,psym=psym,xstyle=4,ystyle=4,xrange=[-1,1],yrange=[-1,1],color=tci(lc)
;
!p.charsize=1.0
if n lt 20 then !p.charsize=1.5
if n lt nummax then begin
	for i=0,n-1 do xyouts,xpv[i],ypv[i],stars[i],color=tci(lc)
	invisible=where(zp lt 0,in)
	if in gt 0 then begin
		xpiv=xp[invisible]
		ypiv=yp[invisible]
		stars=startable[invisible].starid
		for i=0,in-1 do xyouts,xpiv[i],ypiv[i],stars[i],color=tci(fc)
	endif
endif
;
; Plot the grid lines data
if n lt nummax then begin
	invisible=where(zp_g_ra lt 0,in)
	xp_g_ra_nv=xp_g_ra[invisible]
	yp_g_ra_nv=yp_g_ra[invisible]
	oplot,xp_g_ra_nv,yp_g_ra_nv,psym=3,color=tci(fc)
	invisible=where(zp_g_dec lt 0,in)
	xp_g_dec_nv=xp_g_dec[invisible]
	yp_g_dec_nv=yp_g_dec[invisible]
	oplot,xp_g_dec_nv,yp_g_dec_nv,psym=3,color=tci(fc)
endif
if n lt nummax then color=bc else color=fc
visible=where(zp_g_ra ge 0)
xp_g_ra_v=xp_g_ra[visible]
yp_g_ra_v=yp_g_ra[visible]
oplot,xp_g_ra_v,yp_g_ra_v,psym=3,color=tci(color)
visible=where(zp_g_dec ge 0)
xp_g_dec_v=xp_g_dec[visible]
yp_g_dec_v=yp_g_dec[visible]
oplot,xp_g_dec_v,yp_g_dec_v,psym=3,color=tci(color)
;
if !d.name eq 'PS' then device,/close
;
end
;************************************************************************Block 2
function Queryvizier, catalog, target, dis, VERBOSE=verbose, CFA=CFA,  $
               CONSTRAINT = constraint, ALLCOLUMNS=allcolumns, SILENT=silent
;
; Downloaded from:
; https://idlastro.gsfc.nasa.gov/ftp/pro/sockets/queryvizier.pro
;+
; NAME: 
;   QUERYVIZIER
;
; PURPOSE: 
;   Query any catalog in the Vizier database by position
; 
; EXPLANATION:
;   Uses the IDLnetURL object to provide a positional query of any catalog 
;   in the the Vizier (http://vizier.u-strasbg.fr/) database over the Web and
;   return results in an IDL structure.    
; 
;    
; CALLING SEQUENCE: 
;     info = QueryVizier(catalog, targetname_or_coords, [ dis
;                        /ALLCOLUMNS, /CFA, CONSTRAINT= ,/VERBOSE ])
;
; INPUTS: 
;      CATALOG - Scalar string giving the name of the VIZIER catalog to be
;            searched.    The complete list of catalog names is available at
;            http://vizier.u-strasbg.fr/vizier/cats/U.htx . 
;
;            Popular VIZIER catalogs include 
;            'II/328'- AllWISE Data Release (Cutri+ 2013)
;            'V/139' - Sloan SDSS photometric catalog Release 9 (2012)
;            '2MASS-PSC' - 2MASS point source catalog (2003)
;            'GSC2.3' - Version 2.3.2 of the HST Guide Star Catalog (2006)
;            'USNO-B1' - Verson B1 of the US Naval Observatory catalog (2003)
;            'UCAC5'  - 5th U.S. Naval Observatory CCD Astrograph Catalog (2017)
;            'B/DENIS/DENIS' - 2nd Deep Near Infrared Survey of southern Sky (2005)
;            'I/337/gaia' - Gaia DR1 Data Release 1 (2016)
;            'I/311/HIP2' - Hipparcos main catalog, new reduction (2007)
;
;          Note that some names will prompt a search of multiple catalogs
;          and QUERYVIZIER will only return the result of the first search.
;          Thus, setting catalog to "HIP2" will search all catalogs 
;          associated with the Hipparcos mission, and return results for the
;          first catalog found.    To specifically search the Hipparcos or
;          Tycho main catalogs use the VIZIER catalog names listed above
;                             
;      TARGETNAME_OR_COORDS - Either a scalar string giving a target name, 
;          (with J2000 coordinates determined by SIMBAD), or a 2-element
;          numeric vector giving the J2000 right ascension in *degrees* and 
;          the target declination in degrees.
;          If the targetname is set to 'NONE' then QUERYVIZIER will perform
;          an all-sky search using the constraints given in the CONSTRAINT
;          keyword.   
; OPTIONAL INPUT:
;    dis - scalar or 2-element vector.   If one value is supplied then this
;          is the search radius in arcminutes.     If two values are supplied
;          then this is the width (i.e., in longitude direction) and height
;          of the search box.   Default is a radius search with radius of
;          5 arcminutes
;
; OUTPUTS: 
;   info - Anonymous IDL structure containing information on the catalog  
;          sources within the specified distance of the specified center.  The 
;          structure tag names are identical with the VIZIER catalog column 
;          names, with the exception of an occasional underscore
;          addition, if necessary to convert the column name to a valid 
;          structure tag.    The VIZIER Web  page should consulted for the 
;          column names and their meaning for each particular catalog..
;           
;          If the tagname is numeric and the catalog field is blank then either
;          NaN  (if floating) or -1 (if integer) is placed in the tag.
;
;          If no sources are found within the specified radius, or an
;          error occurs in the query then -1 is returned. 
; OPTIONAL KEYWORDS:
;          /ALLCOLUMNS - if set, then all columns for the catalog are returned
;                 The default is to return a smaller VIZIER default set. 
;
;          /CFA - By default, the query is sent to the main VIZIER site in
;            Strasbourg, France.   If /CFA is set then the VIZIER site
;            at the Harvard Center for Astrophysics (CFA) is used instead.
;            Note that not all Vizier sites have the option to return
;            tab-separated values (TSV) which is required by this program.
;   
;          CONSTRAINT - string giving additional nonpositional numeric 
;            constraints on the entries to be selected.     For example, when 
;            in the GSC2.3  catalog, to only select sources with Rmag < 16 set 
;            Constraint = 'Rmag<16'.    Multiple constraints can be 
;            separated by commas.    Use '!=' for "not equal", '<=' for smaller
;            or equal, ">=" for greater than or equal.  See the complete list
;            of operators at  
;                 http://vizier.u-strasbg.fr/doc/asu.html#AnnexQual
;            For this keyword only, **THE COLUMN NAME IS CASE SENSITIVE** and 
;            must be written exactly as displayed on the VIZIER Web page.  
;            Thus for the GSC2.3 catalog one must use 'Rmag' and not 'rmag' or
;            'RMAG'.    In addition, *DO NOT INCLUDE ANY BLANK SPACE* unless it 
;            is a necessary part of the query.
;         
;           /SILENT - If set, then no message will be displayed if no sources
;                are found.    Error messages are still displayed.
;           /VERBOSE - If set then the query sent to the VIZIER site is
;               displayed, along with the returned title(s) of found catalog(s)
; EXAMPLES: 
;          (1) Plot a histogram of the J magnitudes of all 2MASS point sources 
;          stars within 10 arcminutes of the center of the globular cluster M13 
;
;          IDL>  info = queryvizier('2MASS-PSC','m13',10)
;          IDL> plothist,info.jmag,xran=[10,20]
;
;          (2)  Find the brightest J mag GSC2.3 source within 3' of the 
;               J2000 position ra = 10:12:34, dec = -23:34:35
;          
;          IDL> str = queryvizier('GSC2.3',[ten(10,12,34)*15,ten(-23,34,35)],3)
;          IDL> print,min(str.jmag,/NAN)
;
;          (3) Find sources with V < 19 in the Magellanic Clouds Photometric 
;              Survey (Zaritsky+, 2002) within 5 arc minutes of  the position 
;              00:47:34 -73:06:27
;
;              Checking the VIZIER Web page we find that this catalog is
;          IDL>  catname =  'J/AJ/123/855/table1'
;          IDL>  ra = ten(0,47,34)*15 & dec = ten(-73,6,27)
;          IDL> str = queryvizier(catname, [ra,dec], 5, constra='Vmag<19')
;
;          (4) Perform an all-sky search of the Tycho-2 catalog for stars with
;              BTmag = 13+/-0.1
;
;         IDL> str = queryvizier('I/259/TYC2','NONE',constrain='BTmag=13+/-0.1')
;
; PROCEDURES USED:
;          GETTOK(), REMCHAR, REPSTR(), STRCOMPRESS2(), ZPARCHECK
; TO DO:
;       (1) Allow specification of output sorting
; MODIFICATION HISTORY: 
;         Written by W. Landsman  SSAI  October 2003
;         Added /SILENT keyword  W.L.  Jan 2009
;         Avoid error if output columns but not data returned W.L. Mar 2010
;         Ignore vector tags (e.g. SED spectra) W.L.   April 2011
;         Better checking when more than one catalog returned W.L. June 2012
;         Assume since IDL V6.4 W.L. Aug 2013
;         Update HTTP syntax for /CANADA    W. L.  Feb 2014
;         Add CFA keyword, remove /CANADA keyword  W.L. Oct 2014
;         Use IDLnetURL instead of Socket   W.L.    October 2014
;         Add Catch, fix problem with /AllColumns W.L. September 2016
;         Update Strasbourg Web address  W.L. April 2017
;         Handle multiple tables, don't remove leading blanks W.L. Feb 2018
;-

  compile_opt idl2
  if N_params() LT 2 then begin
       print,'Syntax - info = QueryVizier(catalog, targetname_or_coord, dis,'
       print,'         [/ALLCOLUMNS, /SILENT, /VERBOSE, /CFA, CONSTRAINT= ]'
       print,'                       '
       print,'  Coordinates (if supplied) should be J2000 RA (degrees) and Dec'
       print,'  dis -- search radius or box in arcminutes'
       if N_elements(info) GT 0 then return,info else return, -1
  endif

 Catch, theError
 IF theError NE 0 THEN BEGIN
       Catch,/CANCEL
      void = cgErrorMsg(/Quiet)
      return, -1
      ENDIF   
 
 if keyword_set(cfa) then host = "vizier.cfa.harvard.edu" $
                     else host = "vizier.u-strasbg.fr" 
 silent = keyword_set(silent)
 
  if N_elements(catalog) EQ 0 then $
            message,'ERROR - A catalog name must be supplied as a keyword'
  zparcheck,'QUERYVIZIER',catalog,1,7,0,'Catalog Name'         
  targname = 0b
 if N_elements(dis) EQ 0 then dis = 5
 if min(dis) LE 0 then $
     message,'ERROR - Search distances must be greater than zero'
 
 nopoint = 0b
 if N_elements(dis) EQ 2 then $
    search = "&-c.bm=" + strtrim(dis[0],2) + '/' + strtrim(dis[1],2) else $
    search = "&-c.rm=" + strtrim(dis,2) 
    if N_elements(target) EQ 2 then begin
      ra = float(target[0])
      dec = float(target[1])
   endif else begin
       nopoint = strupcase( strtrim(target,2) ) EQ 'NONE' 
       object = repstr(target,'+','%2B')
        object = repstr(strcompress(object),' ','+')
       targname = 1b 
  endelse

; Add any additional constraints to the search. Convert any URL special 
; special characters in the constraint string.

 if N_elements(constraint) EQ 0 then constraint = '' 
 if strlen(constraint) GT 0 then begin
     urlconstrain = strtrim(constraint,2)
     urlconstrain = strcompress2(constraint,['<','>','='])
;Note that one cannot uses the URLENCODE method of IDLnetURL
;because of the "=" needed when encoding "<" and ">" characters.
;I am not sure why this is so.  ---WL    
      urlconstrain = repstr(urlconstrain, ',','&')
     urlconstrain = repstr(urlconstrain, '<','=%3C')
     urlconstrain = repstr(urlconstrain, '>','=%3E')
     urlconstrain = repstr(urlconstrain, '+','%2B')
     urlconstrain = repstr(urlconstrain, '/','%2F')
     urlconstrain = repstr(urlconstrain, '!','=!')
     if nopoint then search = urlconstrain else $
                     search = search + '&' + urlconstrain
 endif
 ;
 path = 'viz-bin/asu-tsv'
 if nopoint then $
  Query = "-source=" + catalog + '&' + $
              search + '&-out.max=unlimited' else $
 if targname then $
  Query = $
          "-source=" + catalog + $
     "&-c=" + object + search + '&-out.max=unlimited' else $
  query = $
          "-source=" + catalog + $
       "&-c.ra=" + strtrim(ra,2) + '&-c.dec=' + strtrim(dec,2) + $
       search + '&-out.max=unlimited'

 if keyword_set(allcolumns) then query += '&-out.all=1'
 if keyword_set(verbose) then begin
      message,'http://' + host + '/' + path,/inf
      message,query,/inf
 endif     
  
  oURL = obj_new('IDLnetURL')
  oURL -> SetProperty, URL_Scheme='http',URL_host=host,URL_query=query, $
                    URL_PATH = path
  result = oURL -> GET(/STRING_ARRAY)
; 
  t = strtrim(result)        ;Feb 2018 don't remove leading blanks
  keyword = strtrim(strmid(t,0,7),2)
  N = N_elements(t)

  if strmid(keyword[n-1],0,5) EQ '#INFO' then begin      ;Error finding catalog?
      message,/INF,t[n-1]
      return, -1
  endif    

  linecon = where(keyword EQ '#---Lis', Ncon)
  if Ncon GT 0 then remove,linecon, t, keyword
 
; Check to see if more than one catalog has been searched
; Use only the first catalog found

  rcol = where(keyword Eq '#RESOUR', Nfound) 
  if N_elements(rcol) GT 1 then begin 
       if keyword_set(verbose) then $
        message,/inf,'Warning - more than one catalog found -- only returning first one'
       t = t[0:rcol[1]-1 ]
       keyword = keyword[0:rcol[1]-1]
  endif   
  
  tcol = where(keyword Eq '#Table', Nfound) 
  if N_elements(tcol) GT 1 then begin 
       if keyword_set(verbose) then $
        message,/inf,'Warning - more than table found in catalog-- only returning first one'
       t = t[0:tcol[1]-1 ]
       keyword = keyword[0:tcol[1]-1]
  endif   
    
  lcol = where(keyword EQ "#Column", Nfound)
  if Nfound EQ 0 then begin
       if max(strpos(strlowcase(t),'errors')) GE 0 then begin 
            message,'ERROR - Unsuccessful VIZIER query',/CON 
            print,t
       endif else if ~silent then $
            message,'No sources found within specified radius',/INF
       return,-1
  endif
  

  if keyword_set(verbose) then begin
    titcol = where(keyword EQ '#Title:', Ntit)
        if Ntit GT 0 then message,/inform, $
        strtrim(strmid(t[titcol[0]],8),2)
  endif
;Check if any Warnings or fatal errors in the VIZIER output
   badflag = strmid(keyword,0,5)
   warn = where(badflag EQ '#++++', Nwarn)
   if Nwarn GT 0 then for i=0,Nwarn-1 do $
        message,'Warning: ' + strtrim(t[warn[i]],2),/info
   
   fatal = where(badflag EQ '#****', Nfatal)
   if Nfatal GT 0 then for i=0,Nfatal-1 do $
        message,'Error: ' + strtrim(t[fatal[i]],2),/info


  trow = t[lcol]
  dum = gettok(trow,' ')
  colname = gettok(trow,' ')
  fmt = gettok(trow,' ')

  remchar,fmt,'('
  remchar,fmt,')' 
  remchar,colname,')'
  remchar,colname,'<'
  remchar,colname,'>'
  colname = IDL_VALIDNAME(colname,/convert_all)
 
; Find the vector tags (Format begins with a number) and remove them 

 bad = where(stregex(fmt,'^[0-9]') GE 0, Nbad)
 if Nbad GT 0 then remove,bad,fmt,colname 
 
 ntag = N_elements(colname)
 fmt = strupcase(fmt)
 val = fix(strmid(fmt,1,4))
 
 for i=0,Ntag-1 do begin

 case strmid(fmt[i],0,1) of 
 
  'A': cval = ' '
  'I': cval = (val[i] LE 4) ? 0 : 0L         ;16 bit integer if 4 chars or less
  'F': cval = (val[i] LE 7) ? 0. : 0.0d      ;floating point if 7 chars or less
  'E': cval = (val[i] LE 7) ? 0. : 0.0d 
  'D': cval = (val[i] LE 7) ? 0. : 0.0d 
   else: message,'ERROR - unrecognized format ' + fmt[i]
 
  endcase

   if i EQ 0 then   info = create_struct(colname[0], cval) else begin
	   ; If you set the /ALLCOLUMNS flag, in some cases (2MASS) you
	   ; get a duplicate column name. Check for this and avoid it by appending
	   ; an extra bit to the duplicate name
	   if where(tag_names(info) eq strupcase(colname[i])) ge 0 then $
	      colname[i] = colname[i] + '_2'
   info =  create_struct(temporary(info), colname[i],cval)
   endelse
 endfor
 
  i0 = max(lcol) + 4  
  if i0 GT (N_elements(t)-1) then begin 
       message,'No sources found within specified radius',/INF
       return, -1
  endif
  
  iend = where( t[i0:*] EQ '', Nend)
  if Nend EQ  0  then iend = N_elements(t) else iend = iend[0] + i0
  nstar = iend - i0 
  info = replicate(info, nstar)

; Find positions of tab characters 
  t = t[i0:iend-1]

  for j=0,Ntag-1 do begin

      x = strtrim( gettok(t,string(9b),/exact ),2)
       dtype = size(info[0].(j),/type)
       if (dtype NE 7) then begin
             bad = where(~strlen(x), Nbad)
             if (Nbad GT 0) then $
             if (dtype EQ 4) || (dtype EQ 5) then x[bad] = 'NaN' $
                                            else x[bad] = -1
      endif
      info.(j) = x 
   endfor
 return,info
END 
;-------------------------------------------------------------------------------
function vizier_phot_xml,xml_file
;
; Read SED data as VOT table downloaded from:
; http://vizier.cds.unistra.fr/vizier/sed/
;
if n_elements(xml_file) eq 0 then xml_file='HD217014.vot'
;
phot_struct_xml=read_votable[xml_file]
;
; help,phot_struct_xml
;* Structure <27ea898>, 10 tags, length=62784, data length=62784, refs=1:
;  _RAJ2000        DOUBLE    Array[654]
;  _DEJ2000        DOUBLE    Array[654]
;  _TABNAME        STRING    Array[654]
;  _ID             STRING    Array[654]
;  _TIME           DOUBLE    Array[654]
;  _ETIME          DOUBLE    Array[654]
;  _SED_FREQ       DOUBLE    Array[654]
;  _SED_FLUX       FLOAT     Array[654]
;  _SED_EFLUX      FLOAT     Array[654]
;  _SED_FILTER     STRING    Array[654]
;
; Extract data
unid=phot_struct_xml._ID
wave=3.0e5/phot_struct_xml._SED_FREQ
flux=phot_struct_xml._SED_FLUX
fnan=finite(phot_struct_xml._SED_EFLUX)
ferr=phot_struct_xml._SED_EFLUX
filt=phot_struct_xml._SED_FILTER
;
index=where(fnan eq 0)
fnan[index]=1
; Remove these entries
index=where(fnan ne 0)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
; Use higher quality measurements
index=where(ferr gt 0 and abs(ferr) lt 1e-7,count)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
; Many records are duplicates
index=uniq(unid,sort(unid))
l=l[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
; Write SED data
sed_file=star+'.txt'
openw,unit,sed_file,/get_lun
for i=0,n_elements(l)-1 do printf,unit,l[i]
free_lun,unit
;
index=sort(wave)
return,{wave:wave,flux:flux,ferr:ferr,filt:filt}
;
end
;-------------------------------------------------------------------------------
function vizier_phot,star_in
;
; Prepare and execute Python script to download photometry from Vizier.
;
; Example: r=vizier_phot('HD217014')
; Vizier returns wavelength in GHz and flux in Jy
;
; Return: [wave (microns),flux [CGS: erg s^-1 cm^-2 mum^-1],err,filter]
; See also Boden 2007, New Astronomy Reviews 51 (2007) 617–622
;	"Calibrating optical/IR interferometry visibility data"
;
; Conversion Jy to CGS system: 3.27e3 Jy = 3.32e-12 erg/(s*cm^2*um)
; => CGS = Jy * 1.0153e-15
;
; Example (wavelength in microns):
; r=vizier_phot('HD27376')
;
; On machine PTI Ubuntu, go to:
; https://irsa.ipac.caltech.edu/cgi-bin/Radar/nph-discovery
; and add Star ID (e.g., "HD 27376" in box at bottom and "Search IRSA"
;
common StarBase,startable,notes
;
if n_elements(star_in) eq 0 then begin
	print,'Error: you must supply a star ID (e.g., HD 123456)!'
	return,0
endif else begin
	star=vizier_starid(strcompress(star_in,/remove_all))
endelse
;
IF 1 THEN BEGIN
;
; Get data from VIZIER
com_file=star+'.inp'
sed_file=star+'.out'
openw,unit,com_file,/get_lun
printf,unit,"import warnings"
printf,unit,"import astropy"
printf,unit,"from astropy.table import Table"
printf,unit,'warnings.filterwarnings("ignore")'
radius='5'
printf,unit, $
'sed=Table.read("https://vizier.cds.unistra.fr/viz-bin/sed?-c={'+star+'}&-c.rs={'+radius+'}")'
;sed=Table.read("https://vizier.cds.unistra.fr/viz-bin/sed?-c={HD2737}&-c.rs={5}
printf,unit,"import tabulate"
printf,unit,"from tabulate import tabulate"
printf,unit,"open('"+sed_file+"','w').write(tabulate(sed))"
free_lun,unit
;
ENDIF ELSE BEGIN
;
; Read VOT table (not working yet, only as command to web browser)
command='/usr/bin/wget https://vizier.cds.unistra.fr/viz-bin/sed?-c="'+star+'"'
spawn,command
;
sed_file=star+'.vot'
com_file=star+'.inp'
openw,unit,com_file,/get_lun
printf,unit,"import astropy"
printf,unit,"from astropy.table import Table"
printf,unit,'radius="5"'
printf,unit,'target="'+star+'"'
printf,unit, $
  'sed=Table.read("https://vizier.cds.unistra.fr/viz-bin/sed?-c={"+target+"}&-c.rs={"+radius+"}")'
printf,unit,"import tabulate"
printf,unit,"from tabulate import tabulate"
printf,unit,"open('"+sed_file+"','w').write(tabulate(sed))"
free_lun,unit
;
ENDELSE
;
; Run Python script
print,'Excuting Python script '+com_file+' to access VIZIER...'
spawn,"/usr/bin/python < "+com_file
print,'Output SED [Jy vs frequency] saved to: '+sed_file
;
status=dc_read_fixed(sed_file,l,/col,format='(a192)')
num_line=n_elements(l)
;
; Decode first line dashes for formatting and perform some checks
words=nameparse(l[0],' ')
if n_elements(words) ne 10 then begin
	print,'Error: expected number (10) of columns not found!'
endif
column_l=strlen(words)
;
; Put | in each row where there is a blank in the first line between the dashes
for i=0,n_elements(column_l)-1 do begin
	for j=0,num_line-1 do strput,l,'|',total(column_l[0:i])+i*2
endfor
;
; Remove first and last line (these should look like "------ ------")
l=l[1:num_line-1]
;
; Vizier output is in flux[Jy] and freq.[GHz].
; Extract photometry as flux [erg 1/cm^2 1/s 1/micron] from input in Jy
; Convert freq [GHz] to wavelength [micron]: lambda [mu] = 3.0e5 / freq
; Example: 1.5 Jy @ 3.48x10^4 GHz l=8.62 mu: F=6.06e-11 erg/(s*cm^2*mu) 
; flux(i)=1.0153e-15*float(words(7))	; erg/cm^2/s/micron
;
; Conversions: 
; (Eureka Scientific,  Inc., 2452 Delmer Street, Suite 100, Oakland, CA 94602)
; Adapted for wavelength measured in microns (original: Angstroem)
; F_lambda = 3.0x10^(-1) S(Jy) / lambda^2 
;	with [lambda] = micron (mu): 1 mu = 10000 A
;
; Verification using "Magnitude to Flux Density converter" at 
; irsa.ipac.caltech.edu/data/SPITZER/docs/dataanalysistools/tools/pet/magtojy/
; HD 34029: flux = 3230 Jy at 136890 GHz corresponds to K = -1.71 (SimBad: K = -1.78)
;
; Columns (epoch 2000):
;RA2000|DE2000|Catalog|Cat.ID|tbl_86|tbl_87|SED_freq|SED_flux|SED_eflux|filter
;  deg |  deg |       |      |  d   |      |  GHz   |   Jy   |    Jy   |     
;
; Write raw SED data
sed_file=star+'.sed'
print,'Writing SED Flux [erg/cm^2 mu] vs wavelength to '+sed_file
openw,unit,sed_file,/get_lun
for i=0,n_elements(l)-1 do printf,unit,l[i]
free_lun,unit
;
; Extract data line by line
cite=strarr(n_elements(l))	; citation, e.g., J/MNRAS/495/1943/table3
unid=strarr(n_elements(l))	; unique ID created for each record
flux=dblarr(n_elements(l))	; Jy
ferr=dblarr(n_elements(l))	; Jy
wave=flux
fnan=strarr(n_elements(l))	; char, NaN or blank
filt=strarr(n_elements(l))	; char, filter name
;
for i=0,n_elements(l)-1 do begin
	words=nameparse(l[i],'|')
	cite[i]=strcompress(words[2],/remove_all)
	unid[i]=strcompress(words[6]+words[7]+words[8]+words[9])
	wave[i]=3.0e5/float(words[6])		; microns
	flux[i]=3.0e-9*float(words[7])/wave[i]^2; erg/cm^2/s/micron
	fnan[i]=strtrim(words[8],2)
	if isnumeric(fnan[i]) then $	; can be NaN or a value
	ferr[i]=3.0e-9*((float(words[7])+float(fnan[i])) $
		       -(float(words[7])-float(fnan[i])))/(2*wave[i]^2)
;	Extract filters and and fix some names
	filt[i]=strcompress(words[9],/remove_all)
endfor
;
; Fix some filter names to enforce name consistency w/filter_c (filter.pro)
index=where(strmid(filt,0,4) eq 'SDSS',count)	; has ' at end of name
for j=0,count-1 do filt[index[j]]=strmid(filt[index[j]],0,6)
index=where(strmid(filt,0,9) eq 'Johnson:L',count); has ' at end of name
for j=0,count-1 do filt[index[j]]=strmid(filt[index[j]],0,9)
index=where(strmid(filt,0,3) eq 'PAN' and strlen(filt) eq 16,count)
for j=0,count-1 do begin
	words=strsplit(filt[index[j]],'/',/extract)
	strgs=strsplit(words[1],':',/extract)
	filt[index[j]]=words[0]+':'+strgs[1]
endfor
;
; Spitzer detectors: IRAC I1/2/3/4; IRS.Blue/Red, MIPS.24/70/160
;
; Spitzer/IRAC
index=where(strpos(filt,'Spitzer/IRAC:3.6') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:3.6'
index=where(strpos(filt,'Spitzer/IRAC:4.5') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:4.5'
index=where(strpos(filt,'Spitzer/IRAC:5.7') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:5.7'
index=where(strpos(filt,'Spitzer/IRAC:7.9') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:7.9'
; Spitzer/MIPS 24mu, 70um, 160mu; J/ApJ/785/33/table2
; https://cdsarc.u-strasbg.fr/ftp/pub/J/other/ApJ/785/33/
index=where(strpos(filt,'Spitzer/MIPS_24') ge 0,count)
for k=0,count -1 do filt[index[k]]='MIPS:24'
index=where(strpos(filt,'Spitzer/MIPS_70') ge 0,count)
for k=0,count -1 do filt[index[k]]='MIPS:70'
index=where(strpos(filt,'Spitzer/MIPS_160') ge 0,count)
for k=0,count -1 do filt[index[k]]='MIPS:160'
;
; http://vizier.cds.unistra.fr/viz-bin/VizieR?-source=J/PASP/119/994/table5
index=where(cite eq 'J/PASP/119/994/table5' and filt eq ':=12um',count)
if count gt 0 then filt[index]='IRAS:12'
index=where(cite eq 'J/PASP/119/994/table5' and filt eq ':=24um',count)
if count gt 0 then filt[index]='MIPS:24'
index=where(cite eq 'J/PASP/119/994/table5' and filt eq ':=25um',count)
if count gt 0 then filt[index]='MIPS:24' ; We don't have the transmission
; Photometry in the MIPS 70µm, PACS 70µm, and PACS 100µm bands
; http://vizier.cds.unistra.fr/viz-bin/VizieR?-source=J/MNRAS/495/1943/table3
index=where(cite eq 'J/MNRAS/495/1943/table3' and filt eq ':=70um',count)
if count gt 0 then filt[index]='PACS:blue'
index=where(cite eq 'J/MNRAS/495/1943/table3' and filt eq ':=100um',count)
if count gt 0 then filt[index]='PACS:green'
; https://vizier.cds.unistra.fr/viz-bin/VizieR-3?-source=J/MNRAS/488/3588
index=where(cite eq 'J/MNRAS/488/3588/table2' and filt eq ':=70um',count)
if count gt 0 then filt[index]='Spitzer/MIPS:70'
index=where(cite eq 'J/MNRAS/488/3588/table2' and filt eq ':=100um',count)
if count gt 0 then filt[index]='PACS:green'
;
; CTIO
index=where(strpos(filt,'CTIO/DECam:u') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:u'
index=where(strpos(filt,'CTIO/DECam:g') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:g'
index=where(strpos(filt,'CTIO/DECam:r') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:r'
index=where(strpos(filt,'CTIO/DECam:i') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:i'
index=where(strpos(filt,'CTIO/DECam:z') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:z'
index=where(strpos(filt,'CTIO/DECam:Y') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:Y'
;
; GAIA
index=where(strpos(filt,'GAIA/') ge 0,count)
for k=0,count -1 do begin
	words=nameparse(filt[index[k]],'/')
	filt[index[k]]=words[1]
endfor
;
stop
; Remove NaN entries and those without error bar
index=where(fnan eq 'nan',nan)
print,'Number of NaN entries to be removed:',nan
index=where(fnan ne 'nan',nan)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
index=where(ferr eq 0,count)
print,'Number of entries without error bar:',count
index=where(ferr gt 0,count)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
; Use higher quality measurements
if 0 then begin
index=where(abs(ferr) lt 1e-7,count)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
endif
;
; Many records are duplicates
index=uniq(unid,sort(unid))
l=l[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
; Write SED data
sed_file=star+'.viz'
openw,unit,sed_file,/get_lun
for i=0,n_elements(l)-1 do printf,unit,l[i]
free_lun,unit
;
; Print data for filters under review...
if 0 then begin
for i=0,n_elements(l)-1 do begin
    if filt[i] eq 'Gaia:G' then print,wave[i],flux[i],ferr[i]
    if filt[i] eq 'GAIA3:G' then print,wave[i],flux[i],ferr[i],'  '+filt[i]
    if filt[i] eq 'GAIA3:Grp' then print,wave[i],flux[i],ferr[i],'  '+filt[i]
    if filt[i] eq 'GAIA3:Gbp' then print,wave[i],flux[i],ferr[i],'  '+filt[i]
endfor
endif
;
index=sort(wave)
return,{wave:wave,flux:flux,ferr:ferr,filt:filt}
;
end
;-------------------------------------------------------------------------------
pro groom_vizier_phot,named_list=named_list
;
; Procedure to remove photometry given list of filters or a named list.
;
common LocalVizier,starid,phot,ferr,filt
return
; 
; Initialize list of filters if a named list has been specified
if keyword_set(named_list) then begin
	r=dc_read_free(!oyster_dir+'source/common/'+'filter.txt',/col,f)
	case named_list of
	'gerard':begin	; select only data for these filters, if available
		g=['Johnson:U','Johnson:B','Johnson:V','Johnson:R', $
		   'Johnson:I',$ 'Johnson:J','Johnson:H','Johnson:K', $
		   'Stroemgren:U','Stroemgren:B', $
		   'Stroemgren:V','Stroemgren:Y', $
		   'Gaia:G', $
		   '2MASS:J','2MASS:H','2MASS:K','2MASS:Ks']
		end
	endcase
;	Make sure you don't select filters not known to OYSTER
	index=whereequal(filt,g)
	filt=filt[index]
	ferr=ferr[index]
	phot=phot[index,*]
;	Sort by wavelength
	si=sort(phot[*,0])
	phot=phot[si,*]
	filt=filt[si]
	ferr=ferr[si]
endif
index=where(ferr gt 0,count)
phot=phot[index,*]
filt=filt[index]
ferr=ferr[index]
;
return	; ---------------------------------------------------------------------
;
; For each VIZIER filter, check if photometry is available for it
;for i=0,n_elements(filters)-1 do begin
;	index=where(strpos(filt,filters(i)) lt 0,nfilt)
;	if nfilt gt 0 then filt(index)=''
;	if nfilt ge 1 then begin
;		phot=phot(index,*)
;		ferr=ferr(index)
;		filt=filt(index)
;	endif
;endfor
;
; Filter not included: do not use
index=where(strpos(filt,'GALEX:FUV') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'GALEX:NUV') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(filt ne 'SDSS:g',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SDSS:r',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SDSS:i',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SDSS:u',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SDSS:z',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'SDSS') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'HIP:Hp') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'HIP:BT',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'HIP:VT',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Gaia:G',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'GAIA/GAIA3:G') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'GAIA/GAIA2:G') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'GAIA/GAIA2:Grp') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'GAIA/GAIA3:Grp') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:U',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:B',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:V',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:R',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:I',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:J',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:H',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:K',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:L',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:M',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Johnson:N',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Cousins:V',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Cousins:B',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Cousins:I',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Cousins:R',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'IRAS') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'VISTA:J') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(filt ne 'VISTA:Ks',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'2MASS:J') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'2MASS:H') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'2MASS:Ks') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'AKARI:S9W') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'AKARI:L18W') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(filt ne 'Spitzer',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'Spitzer/MIPS',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(strpos(filt,'Subaru') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'XMM-OT') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'PAN-STARRS') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(strpos(filt,'WISE') lt 0,nfilt)
phot=phot[index,*]
ferr=ferr[index]
filt=filt[index]
index=where(filt ne 'SkyMapper/SkyMapper:g',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SkyMapper/SkyMapper:i',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SkyMapper/SkyMapper:u',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SkyMapper/SkyMapper:r',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'SkyMapper/SkyMapper:z',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'MIPS:24',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'CTIO/DECam:i',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'CTIO/DECam:r',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'CTIO/DECam:g',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'CTIO/DECam:z',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
index=where(filt ne 'CTIO/DECam:Y',nfilt)
phot=phot[index,*]
ferr=ferr[index,*]
filt=filt[index]
;
end
;-------------------------------------------------------------------------------
pro analyze_vizier_phot,template,star=star,spec_class=spec_class, $
	t1=t1,t2=t2,dm=dm,l_dm=l_dm,binary=binary
;
; Example call: analyze_vizier_phot,template,star='HD27377',spec_class='B9V'
;
; For a given star, compile the SED with data from Vizier and save to disk.
; If the star is not defined, use the SED from the previous call. 
; Additional parameters for a binary star: eff. temperatures t1 and t2,
; magnitude difference dm at wavelength l_dm [microns]
;
; Optional Pickles template on output: wave=template(*,0), flux=template(*,1)
; spec_class: e.g., 'B9V' to find closest template, or 'b9v' for specific file.
;
; Binary option: equal type and class only!
; 
; Example: analyze_vizier_phot,template,star='HD217014',spec_class='g2iv'
; The keyword value will be assigned to spec_class in function pickles.
;
; plot,template(*,0),template(*,1)*1e6,psym=0, $
; 	xtitle='Wavelength ('+greek('mu')+'m]', $
; 	ytitle='F!D'+greek('lambda')+'!N * 10!E6!N ', $
; 	'(erg cm!E-2!N s!E-1!N'+greek('mu')+'m!E-1!N)'
; oplot,r(*,0),r(*,1)*1e6,psym=5
;
; The function to download data from vizier
forward_function vizier_phot
;
common LocalVizier,starid,phot,ferr,filt
common StarBase,startable,notes
;
if not keyword_set(star) then goto,ok_photometry else starid=oyster_starid(star)
;
if n_elements(startable) ne 0 then startable_bck=startable
get_startable,starid
;
phot_struct=vizier_phot(star) ; flux in Jy, wavelength in microns
if not isastruct(phot_struct) then return
if keyword_set(binary) then phot_struct.flux=phot_struct.flux/2
;
phot=[[phot_struct.wave],[phot_struct.flux],[phot_struct.ferr]]
filt=phot_struct.filt
ferr=phot_struct.ferr
nfilt=n_elements(filt)
;
; Remove data from some filters
if 1 then groom_vizier_phot,named_list='gerard'
nfilt=n_elements(filt)
;
; Remove outliers (not easy!)
ufilt=unique(filt)
for k=0,n_elements(ufilt)-1 do begin
	index=where(filt eq ufilt[k],n)
	if n ge 3 then begin
		mflux=median(phot[index,1])
		jndex=where(abs(phot[index,1]-mflux) gt 1e-7,m)
		if m gt 0 then ferr[index[jndex]]=-1
	endif
endfor
;
print,'Valid photometry found in '+strtrim(string(nfilt),2)+' filters.'
;
; Jump here if photometry has already been retrieved
ok_photometry:
;
if n_elements(starid) eq 0 then begin
	print,'Please specify star in call to this procedure!'
	return
endif
;
nfilt=n_elements(filt)
;
; Sort ascending to get good plots
index=sort(phot[*,0])
phot[*,0]=phot[index,0]
phot[*,1]=phot[index,1]
ferr=ferr[index]
filt=filt[index]
;
; Plot the photometry
!x.range=[0,2.5]
!y.range=[0,6e-6]
if startable.starid eq 'HDN217014' then !y.range=[0,3e-7]
if startable.starid eq 'HDN191110' then !y.range=[0,1e-6]
!x.margin=[12,3]
!p.charsize=2
plot,phot[*,0],phot[*,1],psym=3,title=starid, $
 	xtitle='Wavelength ('+greek('mu')+'m)', $
 	ytitle='F!D'+greek('lambda')+'!N '+ $
 	'(erg cm!E-2!N s!E-1!N'+greek('mu')+'m!E-1!N)';,/xlog,/ylog
errplot,phot[*,0],phot[*,1]-ferr/2,phot[*,1]+ferr/2,width=0.01
!x.margin=10
;
; Determine wavelength of SED maximum and estimate BB temperature
i=where(phot[*,1] eq max(phot[*,1])) & i=i[0]
print,'Wavelength [mu] of SED maximum:',float(phot[i,0])
teff_wien=wien(phot[i,0]*1e-6) & teff_wien=teff_wien[0]
print,'Temperature [K] of BB peaking at SED maximum:   '+ $
	strcompress(string(float(teff_wien)),/remove_all)
;
; Overplot blackbody (BB) curves
; Solid: 0, Dotted: 1, Dashed: 2, Dash-Dot: 3, Dashed-Long: 5
if startable.class eq 0 then begin
	print,'Warning: unknown class, assuming class V for ',star,'!'
	startable.class=5
endif
startable.teff=5689		; Teff from Boden et al. 2007 for HD 217014
startable.teff=11700		; Teff from Hummel et al. 2007 for HD 27376
startable.teff=teff_wien	; Teff from SED maximum
if strlen(startable.spectrum) gt 0 then begin  
	startable.teff=teff_star(startable.spectrum); Teff from spectral type
	print,'Use BB temperature derived from stellar type: ', $
							startable.teff
endif else begin
	print,'Use temperature of BB peaking at SED maximum:    ', $
							float(teff_wien)
endelse	
print,'Plot spectrum for BB with this temperature (dotted line)'
if startable.teff ne 0 then begin
	teff=startable.teff
	l_bb=phot[*,0]				; microns
	f_bb=blackbody(teff,l_bb*1e-6)
	idx=where(l_bb ge 0.3 and l_bb le 0.7)	; range to normalize
	f_rt=mean(f_bb[idx]/phot[idx,1])	; normalization ratio
	f_rt=median(f_bb[idx]/phot[idx,1])	; fits better at high flux
	l_bb=0.1+findgen(30)*0.1
	oplot,l_bb,blackbody(teff,l_bb*1e-6)/f_rt,linestyle=1
;	Re-compute spectrum 
	f_bb=blackbody(teff,phot[*,0]*1e-6)/f_rt
endif
;
; Inform about outliers
h_index=where(alog(phot[*,1])/alog(f_bb) gt 1.1)
l_index=where(alog(phot[*,1])/alog(f_bb) lt 0.9)
print,'-------------------------------'
print,'Filters with outliers too high:'
for f=0,n_elements(h_index)-1 do print,filt[h_index[f]]
print,'Filters with outliers too low:'
for f=0,n_elements(l_index)-1 do print,filt[l_index[f]]
print,'-------------------------------'
;
; Fit Teff of blackbody spectrum
l=phot[*,0]*1e-6 	; [m]
y=phot[*,1]
w=fltarr(nfilt)+1	; weights
a=[startable.teff,1]
blackbody_pro,l,a,f	; get scaling factor f
a[1]=median(y/f)	; correct?
a[1]=mean(y/f)	; correct?
f=curvefit(l,y,w,a,fita=[1,0],function_name='blackbody_pro',/noderiv)
teff_bb=a[0]
print,'Temperature [K] of best-fit BB (dashed line): '+ $
	strcompress(string(float(teff_bb)),/remove_all)
l_bb=0.1+findgen(30)*0.1
oplot,l_bb,blackbody(teff_bb,l_bb*1e-6)*a[1],linestyle=2
; Re-compute spectrum at measured photometry
f_bb=blackbody(teff,phot[*,0]*1e-6)/f_rt
f_bb=blackbody(teff,phot[*,0]*1e-6)*a[1]
;
; Fit Teff of blackbody spectrum in log/log space
IF 0 THEN BEGIN
l=alog10(phot[*,0]*1e-6) 	; [m]
y=alog10(phot[*,1])
w=fltarr(nfilt)+1	; weights
a=[startable.teff,1]
blackbody_log_pro,l,a,f	; get scaling factor f
a[1]=median(10^y/10^f)
r=curvefit(l,y,w,a,fita=[1,0],function_name='blackbody_log_pro',/noderiv)
; oplot,1e6*10^l,10^r 	; check that the model fits the data
teff_bb=a[0]
f_bb=blackbody(teff_bb,phot[*,0]*1e-6)
idx=where(phot[*,0] ge 0.3 and phot[*,0] le 0.8)	; range to normalize
f_rt=mean(f_bb[idx]/phot[idx,1])		; normalize to SED
f_bb=blackbody(teff_bb,l_bb*1e-6)/f_rt
;oplot,l_bb,f_bb,linestyle=2
print,'Temperature [K] of best-fit BB: '+ $
	strcompress(string(float(teff_bb)),/remove_all)
ENDIF
;
; Fit Pickles template spectrum equal/nearest to the requested spectral class
if keyword_set(spec_class) then begin
;	spec_class=spec_class
;	Lower-case spec_class to select a specific template, e.g., g2iv
	if spec_class ne strupcase(spec_class) then begin
		template=pickles(star,template=spec_class)
		if template[0] eq 0 then begin
;			print,'Template not available'
			return
		endif
;	Upper-case spec_class to select nearest available template
	endif else begin
		template=pickles(startable.starid)
	endelse
	wave_p=template[*,0]*1e6	; [microns]
	flux_p=template[*,1]
	if total(flux_p) eq 0 then return ; No matching template was found
;	flux_p=flux_p*(max(f_bb)/max(flux_p)); Normalize template to BB
;	oplot,wave_p,flux_p,linestyle=0
;
;	Integrate (scaled) Pickles template fluxes over each filter.
;	Note that the template does not extend over all filter bandpasses!
;	Make copies of the variables in LOCALVIZIER
	phot_tpl=phot
	ferr_tpl=ferr
	filt_tpl=filt
	for i=0,n_elements(filt_tpl)-1 do begin
;		filter_c requires wl. in nm
		transmission=total(filter_c(wave_p*1e3,filt_tpl[i]))
		if transmission gt 0 then begin
		phot_tpl[i,1]=total(filter_c(wave_p*1e3,filt_tpl[i])*flux_p) $
			    /transmission
	    	endif else begin
;			print,'No transmission: '+filt_tpl(i)
			phot_tpl[i,1]=0
		endelse
	endfor
;	Use Pickles filter values to compute median ratio f
	index=where(phot_tpl[*,1] gt 0)
	f=median(phot[index,1]/phot_tpl[index,1])
	f=mean(phot[index,1]/phot_tpl[index,1]) ; more weight at higher fluxes
;	Scale template photometry by computed ratio
	phot_tpl[*,1]=phot_tpl[*,1]*f
	flux_tpl=phot_tpl[index,1]
	wave_tpl=phot_tpl[index,0]
	ferr_tpl=ferr_tpl[index]
	filt_tpl=filt_tpl[index]
	index=uniq(wave_tpl,sort(wave_tpl))
;	Plot photometry template
;	oplot,wave_tpl(index),flux_tpl(index),psym=-1	; filter values
	print,'Pickles template spectrum (solid line)'
	oplot,wave_p,f*flux_p,psym=0		; scaled Pickles temp.
;
;	Scale Pickles template as well (already done above)
	flux_p=flux_p*f
;
;	Integrate BB flux (the wavelength bin is 0.1 micron => normalize by 10)
	sigma=5.67e-5	; Boltzmann constant erg 1/cm^3 1/K^4
	wl_interval= $
		median(l_bb[1:n_elements(l_bb)-1]-l_bb[0:n_elements(l_bb)-2])
	f_bol_bb=total(f_bb)*wl_interval
;	Derive apparent diameter (/home/pti/nexsci/dev/sed/pieces/angularSize.m)
	theta=sqrt(4*f_bol_bb/(sigma*startable.teff^4))*2.062648e8
	print,'Derived BB angular diameter [mas]: ',theta
;
;	Integrate template flux (the bin is 0.0005 micron => normalize by 2000)
	wl_interval=median(wave_p[1:n_elements(wave_p)-1] $
			  -wave_p[0:n_elements(wave_p)-2])
	f_bol_flux_p=total(flux_p)*wl_interval
;	Derive apparent diameter
	theta=sqrt(4*f_bol_flux_p/(sigma*startable.teff^4))*2.062648e8
	print,'Derived TP angular diameter [mas]: ',theta
;
; (/home/pti/nexsci/dev/sed/pieces/angularSize.m)
; ## Angular size uncertainty calculation
; t1 = 2 * theta * sigmaTeff / Teff;  ## Teff error contribution
; t2 = 1/2 * theta * sigmaFbol / fBol; ## fBol error contribution
; sigmaTheta = sqrt(t1.^2.0 + t2.^2.0);  ## final error is both in quadrature

endif
;
; Restore pre-existing startable
if n_elements(startable_bck) ne 0 then startable=startable_bck
;
end
;-------------------------------------------------------------------------------
pro blackbody_pro,l,a,f
;
; l=wavelength [m]
; a[0]=teff, a[1]=scaling factor
; f=flux on output
;
if a[0] lt 100 then f=l*0 $
	       else f=((1.191d-22/l^5)/(exp(1.439d-2/(l*a[0]))-1))*a[1]
;
end
;-------------------------------------------------------------------------------
pro blackbody_log_pro,l_log,a,f_log
;
; l=alog10(wavelength) [m]
; a[0]=teff, a[1]=scaling factor
; f_log=alog10(flux) on output
;
l=10^l_log
;
if a[0] lt 100 then f=l*0 $
	       else f=((1.191d-22/l^5)/(exp(1.439d-2/(l*a[0]))-1))*a[1]
f_log=alog10(f)
;
end
;--------------------------------------------------------------------
pro atmtests
;
spec_g0v=pickles(template='g0v')
spec_g2v=pickles(template='g2v')
spec_g5v=pickles(template='g5v')
spec_g8v=pickles(template='g8v')
;
end
;-------------------------------------------------------------------------------
