;*******************************************************************************
; File: starwheel.pro
;
; Description:
; ------------
; Container of IDL scripts for NPOI observation planning.
;
; Block directory:
; ----------------
; Block 1: fakedata,mockdata
; Block 2: uvimage,xyimage
; Block 3: addstar,addcal,removestar,compileobslist,bglocation,
;	   writeobslist,reduceobslist,vignet,
;	   plotuptime,plotfov,selectup,selectnew,selectdone
; Block 4: biasbase,
;	   obsbase,obsstars,obsnights,obsdates,obssummary
;
;************************************************************************Block 1
pro fakedata,stations,stars,dutycycle=dutycycle,ldl=ldl,npoi=npoi, $
	starids=starids,times=times
;
; Create fake scan data for list of stations and stars. Use defaults for date
; and SystemId if not provided. One needs to specify the stations; if stars
; are not defined, they will be taken from the startable (or keyword starids).
; Units of dutycycle are hours, and corresponds to interval between successive
; scans on the same star. The schedule will cycle through all available
; stars within one hour, whether realistic or not.
;
; Using keywords starids *and* times [s], a specific schedule can be enforced.
;
; NPOI: ldl=1 (i.e. LDL in use) if not specified differently.
;
; Example:
; date='2023-12-16'
; systemid='VLTI/GRAVITY_LR'
; get_startable,gen_model.starid
; get_sysconfig
; fakedata,['AA0','AB5','AJ2','AJ6']
; readmodel,'delori.model'
; calcmodel
; mockdata
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common ModelFit,parameters,ds_options
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
if n_elements(ldl) eq 0 then ldl=1 else ldl=ldl gt 0
if n_elements(npoi) eq 0 then npoi=0 else npoi=1
;
; Shortcut to simulate NPOI data
if npoi then begin
	systemid='NPOI/6way'
	stations=['E07','AC0','AE0','AW0','W07','AN0']
	if n_elements(date) eq 0 then begin
		idldate=systime()
		parseidldate,idldate,y,m,d
		date=nextdate(constrictordate(y,m,d))
	endif
endif
;
if n_elements(stations) eq 0 then stations=genconfig.stationid
if n_elements(stars) eq 0 then begin
	if n_elements(startable) ne 0 then stars=startable.starid $
	else begin
		print,'Error: no stars defined and no StarTable present!'
		return
	endelse
endif
;
RAD=180/!pi
;
; Check inputs
flag=0
NumSid=n_elements(stations)
if NumSid gt 0 then begin
	sid_index=where(strlen(strcompress(stations,/remove)) gt 0,NumSid)
	if NumSid eq 0 then flag=1
endif else flag=1
if flag then begin
	print,'***Error(FAKEDATA): no stations specified!'
	return
endif
if n_elements(starids) ne 0 then stars=unique(starids)
if n_elements(stars) eq 0 then begin
	if n_elements(startable) eq 0 then begin
		print,'***Error(FAKEDATA): no stars specified!'
		return
	endif
	stars=startable.starid
endif else get_startable,stars
;
get_sysconfig,stations=stations
genconfig.configid=instrument_id(systemid)
if checkdata([7,8]) ne 0 then return
;
IF n_elements(times) eq 0 or n_elements(times) ne n_elements(starids) $
	THEN BEGIN
; Compute the visibility range
riseset,uptime,nighttime,ldl=ldl,/force
if uptime[0] gt uptime[1] then uptime[0]=uptime[0]-86400
;
; Create time stamps: believe it or not, but CALIB crashes if time stamps
; fall on full hours!
if n_elements(dutycycle) eq 0 then dutycycle=1.0 ; repeat after this many hours
seed=0
ti=randomu(seed,n_elements(stars))*3600*dutycycle
if instrument_id(systemid) eq 'PRIMA' then ti[*]=0
t0=findgen(nint(24/dutycycle))*3600*dutycycle $
	 -(geoparms.longitude/15+12)*3600
for i=0,n_elements(stars)-1 do begin
;	if min(t) lt 0 then t=t+86400
	t=t0+ti[i]
	ids=strarr(n_elements(t)) & ids[*]=stars[i]
	index=where(t gt uptime[0,i] and t lt uptime[1,i] $
		and t gt nighttime[0] and t lt nighttime[1],n)
;	index=where(t gt uptime(0,i) and t lt uptime(1,i),n)
	if n eq 0 then begin
		print,'Warning(FAKEDATA): ',stars[i],' not visible!'
	endif else begin 
		if n_elements(times) eq 0 then begin
			times=t[index]
			starids=ids[index]
		endif else begin
			times=[times,t[index]]
			starids=[starids,ids[index]]
		endelse
	endelse
endfor
if n_elements(times) eq 0 then return
si=sort(times)
times=times[si]
if min(times) lt 0 then begin
	times=times+86400
	date=previousdate(date)
	geoparms.date=date
endif
starids=starids[si]
ENDIF ; ELSE starids=stars
;
numscan=n_elements(times)
;
; Get StationTable
get_stationtable
;
; Allocate the scans
scans=replicate(scan(),numscan)
;
; Put some data into scans
scans.time=times
scans.starid=starids
scans.photonrate=10
scans.photonrateerr=1
scans.vissq=1
scans.vissqerr=0.02
scans.vissqc=1
scans.vissqcerr=0.05
scans.tripleampc=1
scans.tripleampcerr=0.02
scans.triplephasec=0
scans.triplephasecerr=1.0/RAD
;
; Create the ScanTable
Iscan=lindgen(numscan)+1
ScanId=lindgen(numscan)+1
StarId=scans.starid
StartTime=scans.time
StopTime=scans.time
NumPoint=lonarr(numscan)
NumCoh=lonarr(numscan)
NumIncoh=lonarr(numscan)
Code=lonarr(numscan)+1
Station=lonarr(6,numscan)+1 & Station[0:NumSid-1,*]=1
;
ScanTable=build_scantable(Iscan,ScanId,StarId,Code,Station, $
		StartTime,StopTime,NumPoint, $
                NumCoh,NumIncoh)
;
; Do the astrometry
calcastrom
scans.fdlpos=scans.geodelay & scans.fdlposerr=1e-6
;
; Check whether delay line limits are exceeded
create_scanflagtable & n=0
parms=init_class('scan')
fdl_limit=system_config(system_id(systemid),'FDLLIMIT')
if ldl eq 0 then begin
for i=0,numscan-1 do begin
	if max(scans[i].fdlpos-min(scans[i].fdlpos)) gt fdl_limit then begin
		scanflagtable[n].time=scans[i].time
		scanflagtable[n].reason='FDL_LIMIT'
		scanflagtable[n].item=parms[0]
		n=n+1
	endif
	scans[i].fdlpos(genconfig.refstation-1)=-min(scans[i].fdlpos)
endfor
if n gt 0 then flagscandata,scanflagtable[0:n-1],flag=1
endif
;
freememory
GeoInfo=replicate(GeoParms,1)
GenInfo=replicate(GenConfig,1)
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1	; True: interferometry loaded
ds_options.v2=1
ds_options.tp=1
ds_options.ta=1
;
end
;-------------------------------------------------------------------------------
pro mockdata,pearl=pearl,poisson=poisson,model=model,triple=triple,init=init
;
; Replace observed visibilities and triple data with model values.
; Also includes photometry. Calls set_complexweight to set complex weights.
; Model values must have been computed before (except for option pearl).
;
; Options for model data, visibility data...
; pearl:   from pearl visibility data, given matching SysConfig and scans
; poisson: from model values, errors replaced with Poisson noise estimates
; else:    from model values, errors unchanged
;
; If model=1, use model to compute model values (only for poisson=1)
; If triple=1, set_complexweight will drop loose baselines
; If init=1, set all model visibilities to 1
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBaselines,ob_index,bl_index
common LocalPearl,star,pearl_channel
;
if n_elements(pearl) eq 0 then pearl=0
if n_elements(poisson) eq 0 then poisson=0
if n_elements(model) eq 0 then model=0
;
; Option "triple" used in call to set_complexweight
if n_elements(triple) eq 0 then triple=1
;
; Option "init" sets all model visibilities to unity
if n_elements(init) eq 0 then init=0
;
if pearl and n_elements(geninfo.date) gt 1 then begin
	print,'Error: pearl option cannot be used for multiple nights!'
	return
endif
;
for i=0,n_elements(geninfo.date)-1 do begin
;
if n_elements(bufferinfo) gt 1 then $
loadnight,geninfo[i].date,geoinfo[i].systemid,geninfo[i].configid
;
if checkdata([8,9]) ne 0 then return
;
IF pearl gt 0 THEN BEGIN
;
; This option produces absolute visibility phases, VisPhase
phase_rms=pearl	; VisPhaseErr initialized with phase RMS
;
index0=where(scans.starid eq star and scantable.code eq 1,NS0)
if NS0 lt max(unique(si)) then begin
	print,'Error(MOCKDATA): inconsistency between data structures!'
	return
endif
; Check if "triple" option was used in set_complexweight
index=where(scans[index0].vissqcerr gt 0,count_v)
index=where(scans[index0].complexweight ne 0,count_w)
if count_w lt count_v then scans[index0].vissqcerr=-abs(scans[index0].vissqcerr)
;
w=total(genconfig.wavelength,2)/genconfig.numoutbeam & w=w*1e9
indexj=whereequal(w,wl)
stationids=genconfig.stationid
for ob=0,genconfig.numoutbeam-1 do begin
for bl=0,genconfig.numbaseline[ob]-1 do begin
;	e=reform(scans(*).vissqcerr(ob,0:GenConfig.NumSpecChan(ob)-1,bl) > 0)
;	if GenConfig.NumSpecChan(ob) gt 1 then e=total(e,1)
;	index=where(scans.starid eq star and scantable.code eq 1 and e gt 0,NS)
 	index=index0
 	ns=ns0
	if NS gt 0 then begin
        m=where(stationids eq strmid(genconfig.baselineid[bl,ob],0,3))+1
        n=where(stationids eq strmid(genconfig.baselineid[bl,ob],4,3))+1
        id=100*m+n & id=id[0]
	for ch=0,genconfig.numspecchan[ob]-1 do begin
	for sc=0,NS-1 do begin
		jndex=where(bi eq id and indexj[ci-1] eq ch $
				     and si-1 eq sc,count)
		for k=0,count-1 do begin
			obi=ob_index[jndex[k]]
			bli=bl_index[jndex[k]]
			scans[index[sc]].complexvis(obi,ch,bli)=cv[jndex[k]]
			scans[index[sc]].visphase(obi,ch,bli)=cphase(cv[jndex[k]])
			scans[index[sc]].visphaseerr(obi,ch,bli)=phase_rms
			scans[index[sc]].vissqc(obi,ch,bli)=abs(cv[jndex[k]])^2
 			scans[index[sc]].vissqcerr(obi,ch,bli)= $
 			abs(scans[index[sc]].vissqcerr(obi,ch,bli))
		endfor
		if 0 then begin	; obsolete
		if count eq genconfig.numoutbeam then begin
			scans[index[sc]].complexvis(ob,ch,bl)=cv[jndex[ob]]
			scans[index[sc]].vissqc(ob,ch,bl)=abs(cv[jndex[ob]])^2
 			scans[index[sc]].vissqcerr(ob,ch,bl)= $
 			abs(scans[index[sc]].vissqcerr(ob,ch,bl))
		endif else if count eq 1 then begin
			scans[index[sc]].complexvis(ob,ch,bl)=cv[jndex]
			scans[index[sc]].vissqc(ob,ch,bl)=abs(cv[jndex])^2
 			scans[index[sc]].vissqcerr(ob,ch,bl)= $
 			abs(scans[index[sc]].vissqcerr(ob,ch,bl))
		endif
		endif
	endfor
	endfor
	endif 	; NS > 0
endfor
endfor
;
scans.tripleampcerr=-abs(scans.tripleampcerr)
scans.triplephasecerr=-abs(scans.triplephasecerr)
;
for tr=0,GenConfig.NumTriple-1 do begin
	nc=GenConfig.TripleNumChan[tr]
;
;	e=reform(scans(*).tripleampcerr(tr,0:GenConfig.TripleNumChan(tr)-1) > 0)
;	if GenConfig.TripleNumChan(tr) gt 1 then e=total(e,1)
;	index=where(scans.starid eq star and scantable.code eq 1 and e gt 0,NS)
 	index=index0
 	ns=ns0
	if NS gt 0 then begin
 	vis=reform(complexarr(3,nc,NS,/nozero),3,nc,NS)
 	err=reform(fltarr(3,nc,NS,/nozero),3,nc,NS)
	fBaseMatrix=intarr(GenConfig.NumSid,3)
	fBaseFactor=fltarr(3)+1
	for l=0,2 do begin
		ob=GenConfig.TripleBeam[l,tr]
		bl=GenConfig.TripleBase[l,tr]
		j1=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId[bl,ob],0,3))
		j2=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId[bl,ob],4,3))
		fBaseMatrix[j1,l]=+1
		fBaseMatrix[j2,l]=-1
		if l gt 0 then $
			fBaseFactor[l]=-total(fBaseMatrix[*,0]*fBaseMatrix[*,l])
	endfor
	for l=0,2 do begin
		ob=GenConfig.TripleBeam[l,tr]
		bl=GenConfig.TripleBase[l,tr]
		ch=GenConfig.TripleChan[0:GenConfig.TripleNumChan[tr]-1,l,tr]
		for j=0,nc-1 do begin
			vis[l,j,*]=scans[index].complexvis(ob,ch[j],bl)
			err[l,j,*]=scans[index].vissqcerr(ob,ch[j],bl)
		endfor
		if fBaseFactor[l] lt 0 then vis[l,*,*]=conj(vis[l,*,*])
	endfor
;
;	Form complex triple products
	result=reform(float(vis[0,*,*]*vis[1,*,*]*vis[2,*,*]),1,1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriple(tr,0,0:nc-1)=result
	result=reform(imaginary(vis[0,*,*]*vis[1,*,*]*vis[2,*,*]),1,1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriple(tr,1,0:nc-1)=result
;
;	Form complex triple error mask
	compltripleerr=reform(float( $
	(err[0,*,*] gt 0)*(err[1,*,*] gt 0)*(err[2,*,*] gt 0)),1,1,nc,NS)
	if nc eq 1 then result=reform(compltripleerr,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(compltripleerr)
	
;
;	Compute amp and phase
	result=reform(sqrt(scans[index].compltriple(tr,0,0:nc-1)^2 $
			  +scans[index].compltriple(tr,1,0:nc-1)^2),1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].tripleampc(tr,0:nc-1)=result
	compltripleerr=reform(compltripleerr,size(result,/dim))*2-1
 	scans[index].tripleampcerr(tr,0:nc-1)= $
 	abs(scans[index].tripleampcerr(tr,0:nc-1))*compltripleerr
	result=reform(atan(scans[index].compltriple(tr,1,0:nc-1), $
			   scans[index].compltriple(tr,0,0:nc-1)),1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].triplephasec(tr,0:nc-1)= result
 	scans[index].triplephasecerr(tr,0:nc-1)= $
 	abs(scans[index].triplephasecerr(tr,0:nc-1))*compltripleerr
;
	endif ; NS > 1
;
endfor
;
ENDIF ELSE IF poisson gt 0 THEN BEGIN
;
if init then begin
	scans.vissqm=1
	scans.tripleampm=1
	scans.triplephasem=0
endif
;
if model ne 0 then begin
	index=where(scans.starid eq gen_model.starid)
	scans[index].vissqcerr=0.1
	scans[index].tripleampcerr=0.1
	scans[index].triplephasecerr=0.1
	calcmodel,/noload
endif else index=indgen(n_elements(scans))
scans[index].photonrate=poisson
scans[index].vissqcerr=-1
scans[index].tripleampcerr=-1
scans[index].triplephasecerr=-1
genconfig.numbin=8
nframe=5000
a=fltarr(nframe)
g=a
;
for n=0,n_elements(index)-1 do begin
for t=0,genconfig.numtriple-1 do begin
	ob1=genconfig.triplebeam[0,t]
	ob2=genconfig.triplebeam[1,t]
	ob3=genconfig.triplebeam[2,t]
	bl1=genconfig.triplebase[0,t]
	bl2=genconfig.triplebase[1,t]
	bl3=genconfig.triplebase[2,t]
;
       	fBaseMatrix=intarr(GenConfig.NumSid,3)
       	fBaseFactor=fltarr(3)+1
       	for l=0,2 do begin
               	ob=GenConfig.TripleBeam[l,t]
               	bl=GenConfig.TripleBase[l,t]
               	j1=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId[bl,ob],0,3))
               	j2=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId[bl,ob],4,3))
               	fBaseMatrix[j1,l]=+1
               	fBaseMatrix[j2,l]=-1
               	if l gt 0 then $
		fBaseFactor[l]=-total(fBaseMatrix[*,0]*fBaseMatrix[*,l])
       	endfor
for j=0,genconfig.triplenumchan[t]-1 do begin
	b1=fringeframe(a,g,genconfig.wavelength[j,ob1]*1e6, $
		sqrt(scans[index[n]].vissqm(ob1,j,bl1)), $
		scans[index[n]].photonrate(ob1,j),/poisson, $
		phase=scans[index[n]].visphasem(ob1,j,bl1))
	b2=fringeframe(a,g,genconfig.wavelength[j,ob2]*1e6, $
		sqrt(scans[index[n]].vissqm(ob2,j,bl2)), $
		scans[index[n]].photonrate(ob2,j),/poisson, $
		phase=scans[index[n]].visphasem(ob2,j,bl2))
	b3=fringeframe(a,g,genconfig.wavelength[j,ob3]*1e6, $
		sqrt(scans[index[n]].vissqm(ob3,j,bl3)), $
		scans[index[n]].photonrate(ob3,j),/poisson, $
		phase=scans[index[n]].visphasem(ob3,j,bl3))
	v2=fringevissq(b1,1,200,error=v2e,/normal,/bin8)
	v2m=wmean(v2,v2e,v2me)
	scans[index[n]].vissqc(ob1,j,bl1)=v2m
	scans[index[n]].vissqcerr(ob1,j,bl1)=v2me
	v2=fringevissq(b2,1,200,error=v2e,/normal,/bin8)
	v2m=wmean(v2,v2e,v2me)
	scans[index[n]].vissqc(ob2,j,bl2)=v2m
	scans[index[n]].vissqcerr(ob2,j,bl2)=v2me
	v2=fringevissq(b3,1,200,error=v2e,/normal,/bin8)
	v2m=wmean(v2,v2e,v2me)
	scans[index[n]].vissqc(ob3,j,bl3)=v2m
	scans[index[n]].vissqcerr(ob3,j,bl3)=v2me
;
	v1=fringevis(b1)*(sqrt(4.0/sinc(1./8)^2)/avg(fringenphot(b1)))
	if fBaseFactor[0] lt 0 then v1=conj(v1)
	v2=fringevis(b2)*(sqrt(4.0/sinc(1./8)^2)/avg(fringenphot(b1)))
	if fBaseFactor[1] lt 0 then v2=conj(v2)
	v3=fringevis(b3)*(sqrt(4.0/sinc(1./8)^2)/avg(fringenphot(b1)))
	if fBaseFactor[2] lt 0 then v3=conj(v3)
	vt=v1*v2*v3
	vtm=total(vt)/nframe
;
	vtre=stddev(float(vt))/sqrt(nframe)
	vtie=stddev(imaginary(vt))/sqrt(nframe)
;		
	scans[index[n]].tripleampc(t,j)=abs(vtm)
	scans[index[n]].tripleampcerr(t,j)=sqrt(vtre^2+vtie^2)
	scans[index[n]].triplephasec(t,j)=cphase(vtm)
	scans[index[n]].triplephasecerr(t,j)=sqrt(vtre^2+vtie^2)/abs(vtm)
endfor
endfor
endfor
;
scans[index].photometry=scans[index].photometrym
set_complexweight,triple=triple
;
ENDIF ELSE BEGIN
;
if init then begin
	scans.vissqm=1
	scans.tripleampm=1
	scans.triplephasem=0
endif
;
scans.vissq=scans.vissqm
scans.vissqc=scans.vissqm		; This is the only important quantity
scans.vissqe=scans.vissqm
scans.vissqec=scans.vissqm
scans.tripleamp=scans.tripleampm
scans.tripleampc=scans.tripleampm	; This is the only important quantity
scans.tripleampe=scans.tripleampm
scans.tripleampec=scans.tripleampm
scans.triplephasec=scans.triplephasem	; This is the only important quantity
;
scans.photometry=scans.photometrym
set_complexweight,triple=triple
;
ENDELSE
;
if n_elements(bufferinfo) gt 1 then $
storenight,11
;
endfor
;
end
;************************************************************************Block 2
pro uvimage,uv=uv,lambda=lambda
;
; Compute and display (if uv keyword is set) an image of the visibility modulus 
; given a uv-coverage and a model. Monochromatic, specify lambda [m].
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(uv) eq 0 then uv=0
;
if n_elements(Date) eq 0 then begin
        print,'Warning(UVIMAGE): Date undefined!'
        parseidldate,systime(),y,m,d
        Date=nextdate(constrictordate(y,m,d))
        print,'Set date to: ',date
endif
if n_elements(SystemId) eq 0 then begin
        print,'Warning(UVIMAGE): SystemId undefined!'
        SystemId='NPOI'
        print,'Set SystemId to: ',SystemId
endif
;
if uv then begin
	i0=(!x.crange[0]*!x.s[1]+!x.s[0])*!d.x_size
	i1=(!x.crange[1]*!x.s[1]+!x.s[0])*!d.x_size
	j0=(!y.crange[0]*!y.s[1]+!y.s[0])*!d.y_size
	j1=(!y.crange[1]*!y.s[1]+!y.s[0])*!d.y_size
	imsze=[i1-i0+1,j1-j0+1]
	cellxsize=(abs(!x.crange[1]-!x.crange[0])/float(i1-i0))*1e6
	cellysize=(abs(!y.crange[1]-!y.crange[0])/float(j1-j0))*1e6
	pos=[i0,j0]
endif else begin
	imsze=[512,512]
	cellxsize=0.5e6
	cellysize=0.5e6
	cellxsize=8.0e6
	cellysize=8.0e6
	pos=[0,0]
endelse
;
if !d.name eq 'PS' then begin
	imsze=[456,456]
	cellxsize=40e6/imsze[0]
	cellysize=40e6/imsze[1]
endif
imsize=long(imsze)
cvf=complexarr(imsize[0],imsize[1])
u=(lindgen(imsize[0]*imsize[1]) mod imsize[0]) - (imsize[0]/2)
v=(lindgen(imsize[0]*imsize[1]) / imsize[0]) - (imsize[1]/2)
;
; Since we want the visibility image to be directly comparable
; to the source on the sky, we project it onto the sky just as
; plotuv has plotted it as projected onto the sky.
u=-u*cellxsize
v=+v*cellysize
;
; If no wavelength was specified, use a default
if n_elements(lambda) eq 0 then lambda=0.55e-6
; if n_elements(lambda) eq 0 then lambda=1.6e-6
;
; Display squared visibility amplitude in logarithmic scale
cvf[*]=modelvis(system_config(systemid,'MIDNIGHT')*3600,lambda,f,u,v)
map=abs(cvf)^2
; map=alog10(map/max(map)*1000+1)	; Rescale and add 1 to avoid log(0)
; map=byte(map/max(map)*127)
map=byte(map/max(map)*255)
; map=cphase(cvf)
; print,max(map),min(map)
; map=bytscl(map,max=2*!pi,min=-2*!pi)
;
if uv then begin
doco=0	; skip to Bob's code
if doco then begin
	tek_color
	u=-(findgen(imsize[0])-imsize[0]/2)*cellxsize/1e6
	v=+(findgen(imsize[1])-imsize[1]/2)*cellysize/1e6
	contour,map,u,v,levels=[0.25,0.5,0.75,0.9]^2*max(map),/over, $
			c_linestyle=[1,2,3,4]
endif else begin
;	Bob's code
	loadct,7
	device,set_graphics_function=7
	tv,map,pos[0],pos[1]
	device,set_graphics_function=3
;	Save the cvf,u and v data to a file
	outfile='uvimage.xdr'
	save,cvf,u,v,filename=outfile
	print,'uvimage data saved to ',outfile
; 	Use TVRD to save TV image to a png file
	filename = 'uvimage.png'
	WRITE_PNG, filename, TVRD(/TRUE)
	PRINT, 'File written to ', filename
endelse
endif else save,cvf,u,v
;
end
;-------------------------------------------------------------------------------
pro xyimage,lambda=lambda
;
; Display image of model by Fourier transforming uv-plane visibility.
; Image is monochromatic, given lambda.
;
if n_elements(lambda) eq 0 then lambda=0.55e-6
uvimage,lambda=lambda
restore
;
; image=dft(pearlimage(129,0.2),cmap,u,v)
;
r=size(cvf)
imsize=[r[1],r[2]]
; for i=0,r(2)-1 do cvf(*,i)=reverse(cvf(*,i))
; for i=0,r(1)-1 do cvf(i,*)=reverse(cvf(i,*))
image=shift(float(fft(shift(cvf,imsize[0]/2,imsize[1]/2),1)), $
	imsize[0]/2,imsize[1]/2)
for i=0,r[2]-1 do image[*,i]=reverse(image[*,i])
for i=0,r[1]-1 do image[i,*]=reverse(image[i,*],2)
;
loadct,0
tvscl,(image > 0)^(1/4.)
writefits,'xyimage.fits',image
;
save,cvf,u,v,image
end
;************************************************************************Block 3
pro addstar,stars
;
; Add one or more stars to the startable. Create the latter, if not existent.
; Leave old table intact but remove duplicate entries.
;
common StarBase,startable,notes
;
if n_elements(stars) eq 0 then begin
	print,'***Error(ADDSTAR): stars not defined!'
	return
endif else stars=strupcase(stars)
;
if n_elements(startable) ne 0 then table=startable
get_startable,stars
if n_elements(table) ne 0 then startable=merge_startable(table,startable)
;
si=sort(startable.starid)
startable=startable[si]
startable=startable[uniq(startable.starid)]
;
return
;
if n_elements(startable) eq 0 then get_startable,stars else begin
	stars=[startable.starid,stars]
	get_startable,unique(stars)
endelse
;
end
;-------------------------------------------------------------------------------
pro addcal,stars0,vlimit,klimit=klimit
;
; For a given list of stars, find the closest calibrators brighter than vlimit,
; and add them to the startable.
;
common StarBase,startable,notes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(stars0) eq 0 then begin
	print,'***Error(ADDCAL): stars not defined!'
	return
endif else stars0=strupcase(stars0)
;
if n_elements(klimit) eq 0 then klimit=0 else klimit=1
;
if n_elements(vlimit) eq 0 then vlimit=5.0
RAD=180/pi_circle
;
if n_elements(startable) ne 0 then table=startable
get_startable,stars0
ra0=startable.ra*15/RAD
dec0=startable.dec/RAD
;
; Read the diameter list with the calibrator information
n=0
d=0.0
z=0.0
a_0=0.0
a_1=0.0
a_2=0.0
c=''
status=dc_read_free(!oyster_dir+'starbase/diameter.bsc', $
	n,d,z,a_0,a_1,a_2,c,/col,resize=[1,2,3,4,5,6,7],ignore=['!'])
if status ne 0 then return
;
; Select the calibrators
index=where(c eq 'C') & n=n[index]
get_startable,'BSC'+string(n,format='(i4.4)')
get_jhk
rename_bsc
rename_starids,'bsc-fkv'
if klimit then index=where(startable.mk lt klimit,count) $
	  else index=where(startable.mv lt vlimit,count)
if count eq 0 then begin
	print,'***Error(ADDCAL): no stars bright enough found!'
	return
endif
;
ra=startable[index].ra*15/RAD
dec=startable[index].dec/RAD
stars=startable[index].starid
names=strcompress(startable[index].name)
cals=strarr(n_elements(stars0))
;
for i=0,n_elements(stars0)-1 do begin
	index=where(stars ne stars0[i])
	stars1=stars[index] & names1=names[index]
	ra1=ra[index] & dec1=dec[index]
	dist=sqrt(((ra1-ra0[i])*cos(dec1))^2+(dec1-dec0[i])^2)
	j=where(dist eq min(dist))
	cals[i]=stars1[j]
	print,'Adding calibrator ',cals[i],' (',names1[j],')', $
		' for star ',stars0[i],format='(a,a,a,a,a,a,a)'
endfor
;
if n_elements(table) ne 0 then begin
	startable=table
	addstar,cals
endif else get_startable,cals
;
end
;-------------------------------------------------------------------------------
pro removestar,stars
;
; Remove one or more star from the startable.
;
common StarBase,startable,notes
;
if n_elements(stars) eq 0 then begin
	print,'***Error(REMOVESTAR): stars not defined!'
	return
endif else stars=strupcase(stars)
;
n1=n_elements(startable)
if n1 eq 0 then return
;
for i=0,n_elements(stars)-1 do begin
	index=where(startable.starid ne stars[i],count)
	if count gt 0 then startable=startable[index] else begin
		print,'***Error(REMOVESTAR): no stars would remain in table!'
		return
	endelse
endfor
;
n2=n_elements(startable)
n=n1-n2
if n eq 1 then print,'Removed 1 star.' $
	  else print,'Removed ',n,' stars.',format='(a,i2,a)'
;
end
;-------------------------------------------------------------------------------
pro compileobslist
;
; Finalize information on a list of stars for observations. Also reorder
; by neighboring stars.
;
common StarBase,startable,notes
;
if n_elements(startable) eq 0 then begin
	print,'***Error(COMPILEOBSLIST): startable undefined!'
	return
endif
;
; Default information is from the BSC
rename_starids,'bat-bsc'
rename_starids,'hdn-bsc'
rename_starids,'fkv-bsc'
read_catalogs
;
; Get the diameters
get_diameter
;
; Get the names
rename_bsc
;
; Get the star positions from the FK5
table=startable
rename_starids,'bsc-fkv'
read_catalogs
table.starid=startable.starid
table.fkv=startable.fkv
table.ra=startable.ra
table.dec=startable.dec
startable=table
;
; Sort by RA to insure first stars are in the west
if n_elements(startable) eq 1 then si=0 else si=sort(startable.ra)
startable=startable[si]
;
; Go through list to determine the grouping
RAD=180/!pi
ra=startable.ra*15/RAD
dec=startable.dec/RAD
n=n_elements(startable)
si=intarr(n)
si[0]=1
j=0
for i=1,n-1 do begin
	index=where(si eq 0)
	dist=sqrt(((ra[index]-ra[j])*cos(dec[index]))^2+(dec[index]-dec[j])^2)
	j=index[where(dist eq min(dist))] & j=j[0]
	si[j]=i+1
endfor	
startable=startable[sort(si)]
;
end
;-------------------------------------------------------------------------------
function bglocation,ra0,dec0,fsize=fsize
;
; Return suitable location for measurement of background. Must not have
; any stars down to the magnitude limit of the used coordinate list within
; circle of diameter fsize. 
;
; Diameter, in degrees, of field required to be empty of stars
; The necessary list of star positions is contained in file skycoord,
; which is just a list od Ra and Dec values from a catalog saved into
; an XDR file.
; 
if n_elements(fsize) eq 0 then fsize=30./60
;
; Read (into ra,dec) stellar coordinates from a mag. lim. list for b.g. fields
restore,!oyster_dir+'npoi/skycoord.xdr'
;
RAD=180/!pi
;
; Center stellar field N (az=0, ha=0) at elevation zero, 
; and convert to (az,el) coord.
lat=-(90-dec0)
azel=hadec2azel(transpose([[ra-ra0],[dec]]),lat=lat)
index=where(azel[0,*] gt 180)
azel[0,index]=azel[0,index]-360
az=reform(azel[0,*])
el=reform(azel[1,*])
;
; Do a spiral search beginning south with field centers separated by fsize/2
dr=fsize/2
pa=0.
repeat begin
        az0=-sin(pa/RAD)*dr
        el0=+cos(pa/RAD)*dr
        d=sqrt((az-az0)^2+(el-el0)^2)
        pa=pa+2*asin((fsize/4)/dr)*RAD
        if pa gt 360 then begin
                pa=0.
                dr=dr+fsize/2
        endif
endrep until min(d) gt fsize/2
;
hadec=azel2hadec([az0,el0],lat=lat)
hadec[0]=hadec[0]+ra0
;
return,hadec
;
end
;-------------------------------------------------------------------------------
pro writeobslist,from_data,time=time
;
; Update rise and set times for stars listed in the startable, and
; write startable in obslist format. Time can be either GST or UTC.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
num_stars=n_elements(startable)
if num_stars eq 0 then begin
	print,'***Error(WRITEOBSLIST): no stars defined!'
	return
endif
;
if n_elements(from_data) eq 0 then from_data=0
;
riseset,uptime
if n_elements(uptime) eq 0 then return
if from_data then roseset,uptime
;
read_notes
;
openw,unit,Date+'.obs',/get_lun
;
if n_elements(time) eq 0 then time='UTC'
;
printf,unit, $
'! Observing list for: ',Date
case strupcase(time) of
	'UTC': printf,unit, $
'! Star        Name  Type m_V   D  Int Sp. PI   RA (BG)  Dec   | Rise (UT) Set'
	'GST': printf,unit, $
'! Star        Name  Type m_I   D  Int Sp. PI   RA (BG)  Dec   | Rise (GST) Set'
endcase
;
for i=0,n_elements(startable)-1 do begin
	case time of
		'UTC':begin
		      vi=0.0
		      rise=uptime[0,i]/3600
		      if rise lt 0 then rise=rise+24
		      set= uptime[1,i]/3600
		      end
		'GST':begin
		      r=[2.01,0.82,2.53]
		      r=[1.87,0.89,1.28]
		      vi=2.5*alog10(poly(startable[i].bv,r)/1.87)
		      rise=ut12gst(uptime[0,i],uptime[0,i])
		      set= ut12gst(uptime[1,i],uptime[1,i])
		      end
	endcase
	it=90
	star='CCCNNNN  '
	strput,star,startable[i].starid
;
;	Locate suitable background field for this star
	radec=bglocation(startable[i].ra,startable[i].dec)
;
	printf,unit,star,strcompress(startable[i].name), $
		startable[i].bflag,startable[i].mv-vi,startable[i].diameter, $
		it,strcompress(startable[i].spectrum,/remove_all),'____ ', $
		radec[0],radec[1],' | ', $
		hms(rise),hms(set), $
	format='(a9,1x,a11,1x,a1,1x,f4.1,1x,f4.1,1x,i3,1x,a2,1x,a5,f7.4,1x,f7.3,a3,a7,2x,a7)'
endfor
printf,unit,'!-------------------------------------------------------------------------------'
for i=0,n_elements(startable)-1 do begin
	list_note,startable[i].starid,note
	if strlen(note[0]) ne 0 then begin
		for j=0,n_elements(note)-1 do $
			printf,unit,'! '+strmid(note[j],0,78)
		printf,unit,'!-------------------------------------------------------------------------------'
	endif
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro reduceobslist,r
;
; Given a startable, try to replace close pairings (within r [deg]) with
; brighter of the two stars, or the middle star for compact groups.
;
common StarBase,startable,notes
;
RAD=180/!pi
;
for radius=1,nint(r) do begin
;
repeat begin
	n=n_elements(startable)
	nstars=intarr(n)
	ra=startable.ra*15/RAD
	dec=startable.dec/RAD
	x=cos(dec)*sin(ra)
	y=cos(dec)*cos(ra)
	z=sin(dec)
	for i=0,n-1 do begin
		dist=sqrt((x-x[i])^2+(y-y[i])^2+(z-z[i])^2)
		index=where(dist lt radius/RAD,count)
		nstars[i]=count
	endfor
	i=where(nstars eq max(nstars))
	dist=sqrt((x-x[i[0]])^2+(y-y[i[0]])^2+(z-z[i[0]])^2)
	stars=startable[where(dist lt radius/RAD)]
	if max(nstars) eq 2 then index=where(stars.mv ne min(stars.mv),count) $
			    else index=where(stars.starid ne startable[i[0]].starid,count)
	if count gt 0 then removestar,stars[index].starid
endrep until count eq 0
;
endfor
;
end
;-------------------------------------------------------------------------------
pro vignet,stations
;
; Display telescope shadows
;
n_az=37L
n_el=19L
az=dindgen(n_az)*360/(n_az-1)
el=dindgen(n_el)*90/(n_el-1)
r=fltarr(n_az*n_el)
t=fltarr(n_az*n_el)
;
for i=0,n_elements(stations)-1 do begin
shadow=dblarr(n_az*n_el)
;
loc=vlti_station(stations[i],ts)
;
; The AZ for shadow is zero for S and increasing Eastwards
status=linknload(!external_lib,'shadow',n_az,n_el,az,el,shadow,loc,r,t,ts)
;
if i eq 0 then z=shadow else z=z+shadow
;
endfor
;
z=z*127
;
loadct,0
i0=(!x.crange[0]*!x.s[1]+!x.s[0])*!d.x_size
i1=(!x.crange[1]*!x.s[1]+!x.s[0])*!d.x_size
j0=(!y.crange[0]*!y.s[1]+!y.s[0])*!d.y_size
j1=(!y.crange[1]*!y.s[1]+!y.s[0])*!d.y_size
imsze=[i1-i0+1,j1-j0+1]
device,set_graphics_function=6
tvscl,congrid(polar_surface(z,r,t),imsze[0],imsze[1]),i0,j0
; polar_contour,z,t,r,nlevels=8,fill=1,/over
device,set_graphics_function=3
;
end
;-------------------------------------------------------------------------------
pro plotuptime,from_data,stars=stars,stations=stations,polar=polar
;
; For the stars in startable and stations in GenConfig, compute rise 
; and set times and plot. Use stars and stations to initialize StarTable
; and GenConfig.
;
; Get actual times from scan data if requested (from_data=1).
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common LocalRiseSet,date_p,system_p,stars_p,stations_p, $
                times,gst,za_sun,za_moon,ha_moon,az_moon, $
                za_star_p,ha_star_p,az_star_p, $
                sun_ok,moon_ok,zas_ok,sid_ok_p,fdl_ok_p
;
if n_elements(stars) gt 0 then get_startable,stars
if n_elements(stations) gt 0 $
or n_elements(genconfig) eq 0 then get_sysconfig,stations=stations
if n_elements(polar) eq 0 then polar=0
;
num_stars=n_elements(startable)
if num_stars eq 0 then begin
	print,'***Error(PLOTUPTIME): no stars defined!'
	return
endif
if num_stars gt 40 then begin
	print,'***Error(PLOTUPTIME): too many stars to plot legibly!'
	return
endif
;
if n_elements(from_data) eq 0 then from_data=0
;
riseset,uptime,nighttime
if n_elements(uptime) eq 0 then return
if from_data then roseset,uptime
;
uptime=uptime/3600 & nighttime=nighttime/3600
index=where(uptime[1,*] lt uptime[0,*],count)
if count gt 0 then uptime[0,index]=uptime[0,index]-24
;
!p.multi=0
!p.color=tci(1)
IF NOT polar THEN BEGIN
if !d.window lt 0 then window,xsize=1000,ysize=300
x=findgen(num_stars)
y=fltarr(num_stars) & y[0]=13
xlabels1=startable.starid
xlabels2=xlabels1
if num_stars gt 20 then begin
	xlabels1[where(indgen(num_stars) mod 2 eq 0)]=' '
	xlabels2[where(indgen(num_stars) mod 2 eq 1)]=' '
endif else xlabels2[*]=' '
plot,x,y,/nodata,yrange=[nighttime[0],nighttime[1]], $
   xtickname=xlabels1,xticks=num_stars-1,xtickv=x,xstyle=2, $
   xtitle='Visibility for '+Date $
	 +' ('+strjoin(genconfig.stationid[0:genconfig.numsid-1],' ')+')', $
   ytitle='UT',charsize=1.5,ytickv=findgen(24), $
   yticks=fix(nighttime[1]-nighttime[0]+3)
t_x_c_n=!x.s[0]+(num_stars-1)*!x.s[1]+!d.x_ch_size/!d.x_vsize
t_x_c_d=(t_x_c_n-!x.s[0])/!x.s[1]
irange=fix(!y.crange)
for i=irange[0],irange[1] do begin
	if i gt nighttime[0] and i lt nighttime[1] then color=4 else color=6
	oplot,!x.crange,float([i,i]),psym=0,color=tci(color)
	all=0
	for j=0,num_stars-1 do $
		if i gt uptime[0,j] and i lt uptime[1,j] then all=all+1
	xyouts,t_x_c_d,float(i)-0.2,string(all,format='(i2)'),charsize=1.0
endfor
for i=0,num_stars-1 do begin
	oplot,fltarr(2)+i,uptime[0:1,i],thick=4,psym=0
endfor
ENDIF ELSE BEGIN
if !d.window lt 0 then window,xsize=600,ysize=600
!x.ticks=2
!x.tickname=[' ','S',' ']
!x.tickv=[-90,0,90]
!y.ticks=2
!y.tickname=[' ','W',' ']
!y.tickv=[-90,0,90]
if strupcase(!version.os_name) eq 'HP-UX' then !p.charsize=1.0 $
					  else !p.charsize=1.5
!x.margin=[3,3]
!y.margin=[3,3]
n_az=360
theta=findgen(n_az)/180*!pi
r=fltarr(n_az)+90
plot,r,theta,/polar,xrange=[-90,90],yrange=[-90,90],xstyle=1,ystyle=1, $
   title='Visibility '+Date $
	 +' ('+strjoin(genconfig.stationid[0:genconfig.numsid-1],' ')+')'
; Plot coordinate circles
for i=10,80,10 do begin
	r[*]=i
	oplot,r,theta,psym=0,linestyle=1,/polar
;	xyouts, -20.,r[0]+2,string(fix(90-r[0])),color=tci(4)
	xyouts,r[0]-17,0,string(fix(90-r[0])),color=4
endfor
; Plot coordinate cross
oplot,fltarr(180),findgen(180)-90
oplot,findgen(180)-90,fltarr(180)
theta=findgen(36)/18*!pi
usersym,sin(theta),cos(theta),/fill
ut=systime(/utc)
ut=float(strmid(ut,11,2))+float(strmid(ut,14,2))/60
oplot,za_moon,(az_moon+90)/180*!pi,/polar,psym=0,color=tci(7)
mndex=where(abs(times/3600-ut) eq min(abs(times/3600-ut)))
oplot,za_moon[mndex],(az_moon[mndex]+90)/180*!pi,/polar,psym=8,symsize=2,color=tci(7)
for i=0,num_stars-1 do begin
	index=where(sun_ok and zas_ok[*,i] and sid_ok_p[*,i] and fdl_ok_p[*,i],count)
	if count ge 1 then begin
;	Plot moon trail
;	oplot,za_moon_p(index,i),(az_moon_p(index,i)+90)/180*!pi,/polar,psym=0,color=tci(7)
;	mndex=where(abs(times/3600-ut) eq min(abs(times/3600-ut)))
;	oplot,za_moon_p(mndex,i),(az_moon_p(mndex,i)+90)/180*!pi,/polar, $
;		psym=8,symsize=2,color=tci(7)
;	Plot star trails
	oplot,za_star_p[index,i],(az_star_p[index,i]+90)/180*!pi,/polar,psym=0
; 	Plot 1hr intervals
	j=where(abs(times[index]/3600-nint(times[index]/3600)) lt 0.06)
	oplot,za_star_p[index[j],i],(az_star_p[index[j],i]+90)/180*!pi,/polar,psym=8
;	Plot current position of stars
	index=where(abs(times/3600-ut) eq min(abs(times/3600-ut)))
	oplot,za_star_p[index,i],(az_star_p[index,i]+90)/180*!pi,/polar, $
		psym=getsymbol(i+1),symsize=1.5,color=tci(3)
;	Print symbol and star label
	oplot,[-85.,-85.],86-[i*6,i*6],psym=getsymbol(i+1),color=tci(3)
	star_label=strtrim(startable[i].name,2)
	if strlen(star_label) eq 0 then $
	star_label=strtrim(startable[i].var,2)
	if strlen(star_label) eq 0 then $
	star_label=strtrim(startable[i].starid,2)
	xyouts,-80.,84-i*6,star_label,color=tci(3)
	endif
endfor
;
; For VINCI observations, overlay shadow map
if system_id(systemid) eq 'VLTI' then $
	vignet,genconfig.stationid
ENDELSE
;
set_screen
;
end
;-------------------------------------------------------------------------------
pro plotfov,stations=stations,dl=dl,bc=bc,polar=polar,iss=iss
;
; Plot a diagram indicating the hour angle range as a function of declination
; of a star to be observable with stations. If stations are not specified,
; ignore the delay limits; otherwise, fake the data and flag all data outside
; these limits.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common LocalRiseSet,date_p,system_p,stars_p,stations_p, $
                times,gst,za_sun,za_moon,ha_moon,az_moon, $
                za_star_p,ha_star_p,az_star_p, $
                sun_ok,moon_ok,zas_ok,sid_ok_p,fdl_ok_p
;
if n_elements(systemid) eq 0 then begin
	systemid=''
	read,prompt='Please enter systemid/instrument: ',systemid
	systemid=strupcase(systemid)
;	systemid='NPOI'
endif
;
if n_elements(polar) eq 0 then polar=0
;
; Call external ISS shadow code if iss=1
if n_elements(iss) eq 0 then iss=0
;
; Save startable
if n_elements(startable) ne 0 then startable_bck=startable
;
; Create list of artificial stars
n=19
n=37
get_startable,'BSC'+string(indgen(n)+1,format='(i4.4)')
startable.dec=-90+findgen(n)*(180/(n-1))
startable.ra=0
parseidldate,systime(),y,m,d
Date=nextdate(constrictordate(y,m,d))
get_sysconfig,stations=stations
;
; Overwrite configuration
if n_elements(dl) ne 0 then genconfig.delaylineid=dl
if n_elements(bc) ne 0 then genconfig.bcinputid=bc
get_stationtable
;
; get_geoparms
startable.ra=ut12gst(0,0)
;
if n_elements(stations) ne 0 then begin
;	fakescans,stations,dutycycle=0.05
;	roseset,uptime
;	riseset now can handle constraints such as FDL, etc.
	riseset,uptime,/force,iss=iss
endif else riseset,uptime,/force,iss=iss
si=where(total(uptime,1) ne 0,count)
ha=uptime/3600+geoparms.longitude/360*24
h=[reform(ha[0,si]),reform(ha[1,reverse(si)])]
d=[startable[si].dec,startable[reverse(si)].dec]
;
; Plot a filled polygon
set_screen
!p.charsize=1.5
!p.psym=0
!p.color=tci(1)
statns=''
for i=0,n_elements(stations)-1 do statns=statns+' '+stations[i]
;
IF polar THEN BEGIN
	hadec=transpose([[h],[d]])
        azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec)))
        az=reform(azel[0,*])/180*!pi
        za=90-reform(azel[1,*])
	if !d.window lt 0 then window,xsize=500,ysize=500
	n_az=360
	theta=findgen(n_az)/180*!pi
	r=fltarr(n_az)+90
	!x.ticks=2
	!x.tickname=[' ','S',' ']
	!x.tickv=[-90,0,90]
	!y.ticks=2
	!y.tickname=[' ','W',' ']
	!y.tickv=[-90,0,90]
	!p.charsize=1.5
	!x.margin=[3,3]
	!y.margin=[3,3]
	plot,r,theta,/polar,xrange=[-90,90],yrange=[-90,90],xstyle=1,ystyle=1, $
		title=statns
	for i=10,80,10 do begin
		r[*]=i
		oplot,r,theta,psym=0,/polar
		xyouts,r[0]-22,0,string(fix(90-r[0])),color=tci(4)
	endfor
	oplot,fltarr(180),findgen(180)-90
	oplot,findgen(180)-90,fltarr(180)
;
	x=-za*sin(az)
	y=+za*cos(az)
	polyfill,x,y,/line_fill,linestyle=1,thick=3
	!x.range=0
	!y.range=0
;
; 	Plot equatorial grid
	theta=findgen(36)/18*!pi
	usersym,sin(theta),cos(theta),/fill
	times1=times+(startable[0].ra-fix(startable[0].ra))*3600
	for j=0,n-1 do begin
	index=where(sun_ok and zas_ok[*,j],count)
	if count gt 0 and fix(startable[j].dec) mod 10 eq 0 then begin
	!p.color=tci(3)
	if startable[j].dec eq 0 then !p.color=tci(2)
	oplot,za_star_p[index,j],(az_star_p[index,j]+90)/180*!pi,/polar,psym=0
	k=where(abs(times1[index]/3600-nint(times1[index]/3600)) lt 0.06)
	k=where(abs(ha_star_p[index,j]-nint(ha_star_p[index,j])) lt 0.06)
	oplot,za_star_p[index[k],j],(az_star_p[index[k],j]+90)/180*!pi,/polar,psym=8
	endif
	endfor
	!p.color=tci(0)
ENDIF ELSE BEGIN
	window,xsize=500,ysize=640,/free
	plot,[-6,6],[-90,90],/nodata,xtitle='Hour angle',ytitle='Declination', $
		title=statns, $
		xticklen=1,yticklen=1,xthick=0.5,ythick=0.5, $
		xgridstyle=2,ygridstyle=2, $
		xticks=12,xtickv=findgen(13)-6, $
		ytickinterval=10, $
		xrange=[-6,6],yrange=[-90,90],xstyle=1,ystyle=1
	polyfill,h,d,/line_fill,linestyle=1,thick=2
ENDELSE
;
; Restore startable
; if n_elements(startable_bck) ne 0 then startable=startable_bck
;
end
;-------------------------------------------------------------------------------
pro selectup,from_data,mintime=min
;
; Extract list of stars from startable which are visible in a given night.
; Reduce startable accordingly. Get times from scan data if from_data=1.
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
num_stars=n_elements(startable)
if num_stars eq 0 then begin
	print,'***Error(SELECTUP): no stars defined!'
	return
endif
;
if n_elements(from_data) eq 0 then from_data=0
if not keyword_set(min) then min=3
;
riseset,uptime,nighttime 
if n_elements(uptime) eq 0 then return
if from_data then roseset,uptime
;
uptime=uptime/3600 & nighttime=nighttime/3600
index=where(uptime[1,*] lt uptime[0,*],count)
if count gt 0 then uptime[0,index]=uptime[0,index]-24
;
index=where(uptime[0,*] lt nighttime[1]-min or uptime[1,*] gt nighttime[0]+min)
startable=startable[index]
;
end
;-------------------------------------------------------------------------------
pro selectnew,dates=nd,scans=ns
;
; Remove from startable all stars observed before nd times, ns scans each.
;
common StarBase,startable,notes
;
if n_elements(nd) eq 0 then nd=2
if n_elements(ns) eq 0 then ns=4
;
for i=0,n_elements(startable)-1 do begin
	if n_elements(discard) eq 0 then discard=intarr(n_elements(startable))
	obsdates,startable[i].starid,dates,scans
	index=where(scans ge ns,count)
	if count ge nd then discard[i]=1
endfor
;
index=where(discard eq 0,count)
if count gt 0 then startable=startable[index]
;
end
;-------------------------------------------------------------------------------
pro selectdone,dates=nd,scans=ns
;
; Remove from startable all stars without any observations at lease nd nights,
; ns scans each.
;
common StarBase,startable,notes
;
if n_elements(nd) eq 0 then nd=2
if n_elements(ns) eq 0 then ns=4
;
for i=0,n_elements(startable)-1 do begin
	if n_elements(keep) eq 0 then keep=intarr(n_elements(startable))
	obsdates,startable[i].starid,dates,scans
	index=where(scans ge ns,count)
	if count ge nd then keep[i]=1
endfor
;
index=where(keep eq 1,count)
if count gt 0 then startable=startable[index]
;
end
;************************************************************************Block 4
pro biasbase,files
;
; This procedure updates the file biasbase.xdr in oyster/npoi with bias data
; from the specified files, if existent, i.e. non-zero. Therefore, there
; is no way knowing whether the non-zero coefficients have been derived
; from this night's data, or whether they have been used from another night
; using the get_bias procedure. The rule should be to write non-zero bias
; information only if it has been established with these data.
;
; Ideally, and ultimately, every night should include incoherent scans for
; the derivation of bias coefficients, but historically this has not been
; the case. Therefore this somewhat elaborate system to establish a bias
; database.
;
; Current list of dates entering biasbase:
; 2000-09-20 2000-09-26 2000-09-28 2000-09-29 2000-10-06 2000-10-18
; 2000-10-20 2000-11-29 2000-12-01 2001-06-21
;
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_params() eq 0 then begin
	print,'Usage: biasbase,files'
	print,'Required input: files.'
	return
endif
;
if n_elements(files) eq 1 then files=file_search(files[0])
nf=n_elements(files)
if strlen(files[0]) eq 0 then begin
	print,'***Error(BIASBASE): no files!'
	return
endif
maxConfig=NumSid 
for i=2,GenConfig.NumSid do $
	maxConfig=maxConfig+combinations(GenConfig.NumSid,i)
;
name=!oyster_dir+'npoi/biasbase.xdr'
result=file_search(name,count=fcount)
if fcount eq 1 then begin
	restore,name	; restore variable database
	bb=database & nb=n_elements(bb)
	database=alloc_biasbase(strarr(nb+nf*maxConfig),bb[0].bias,'      ')
	database[0:nb-1].date=bb.date
	database[0:nb-1].bias=bb.bias
	database[0:nb-1].config=bb.config
endif else nb=0
;
j=0
for i=0,nf-1 do begin
	hds_open,files[i],'READ',status
	if status ne 0 then return
	get_genconfig
	if i eq 0 and nb eq 0 then $
		database=alloc_biasbase(strarr(nf),genconfig.bias,'      ')
	if total(abs(genconfig.bias)) gt 0 then begin
		get_data
		configs=scanconfig()
		for k=0,n_elements(configs)-1 do begin
		database[nb+j].bias=genconfig.bias[*,*,*,*,k]
		database[nb+j].date=checkdate()
		database[nb+j].config=configs[k]
		j=j+1
		endfor
	endif
	hds_close
endfor
if j eq 0 then begin
	print,'No bias data found!'
	return
endif
database=database[0:nb+j-1]
;
database=database[uniq(database.date+database.config, $
			sort(database.date+database.config))]
save,database,filename=name
;
if fcount eq 1 then print,'Biasbase updated' else print,'Biasbase created'
;
end
;-------------------------------------------------------------------------------
pro obsbase,files
;
; Read ScanTable in HDS-format files, concatenate, and add to database,
; if database refers to an existing file. Otherwise, create database.
; The datebase file is kept in the npoi folder.
;
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_params() eq 0 then begin
	print,'Usage: obsbase,files'
	print,'Required input: files.'
	return
endif
;
if n_elements(files) eq 1 then files=file_search(files[0])
if strlen(files[0]) eq 0 then begin
	print,'***Error(OBSBASE): no files!'
	return
endif
hds_close
;
for i=0,n_elements(files)-1 do begin
	print,'Processing ',files[i]
	hds_open,files[i],'READ',status
	if status eq 0 then begin
		get_scantable
		get_sysconfig
		dat_annul
;
		if scantable[0].starttime/3600 gt 24 then $
			Date=nextdate(Date)
;
		scantable=scantable[where(scantable.code eq 1)]
		ns=n_elements(scantable)
		if i eq 0 then begin
			dates=strarr(ns)+Date
			stars=scantable.starid
			times=scantable.starttime
		endif else begin
			dates=[dates,strarr(ns)+Date]
			stars=[stars,scantable.starid]
			times=[times,scantable.starttime]
		endelse
	endif
;
endfor
parsedate,dates,y,m,d
umjd=unique(long(julian(y,m,d+times/86400,/mjd)*1000))
;
name=!oyster_dir+'npoi/obsbase.xdr'
result=file_search(name,count=fcount)
if fcount eq 1 then begin
	restore,name
	mask=intarr(n_elements(database))+1
	parsedate,database.date,y,m,d
	mjd=long(julian(y,m,d+database.time/86400,/mjd)*1000)
	for i=0,n_elements(umjd)-1 do begin
		index=where(mjd eq umjd[i],count)
		if count gt 0 then mask[index]=0
	endfor
	database=database[where(mask eq 1)]
	dates=[dates,database.date]
	stars=[stars,database.star]
	times=[times,database.time]
endif
database=alloc_obsbase(dates,stars,times)
entries=database.date+database.star+string(database.time)
database=database[uniq(entries,sort(entries))]
;
save,database,filename=name
;
if fcount eq 1 then print,'Database updated' else print,'Database created'
;
end
;-------------------------------------------------------------------------------
pro obsstars,stars
;
; List all stars ever observed with NPOI and listed in the database file.
;
;
name=!oyster_dir+'npoi/obsbase.xdr'
result=file_search(name,count=fcount)
if fcount eq 0 then begin
	print,'***Error(OBSSTARS): file not found!'
	return
endif
restore,name
stars=database.star
stars=stars[uniq(stars,sort(stars))]
;
if n_params() eq 0 then print,stars
;
end
;-------------------------------------------------------------------------------
pro obsnights,nights
;
; List all nights listed in the database file.
;
;
name=!oyster_dir+'npoi/obsbase.xdr'
result=file_search(name,count=fcount)
if fcount eq 0 then begin
        print,'***Error(OBSSTARS): file not found!'
        return
endif
restore,name
nights=database.date
nights=nights[uniq(nights,sort(nights))]
;
if n_params() eq 0 then $
	while(printindex(n_elements(nights),7,i,j)) do print,nights[i:j]
;
end
;-------------------------------------------------------------------------------
pro obsdates,star,dates,scans
;
; Find dates of observation for a given star.
;
;
if n_params() eq 0 then begin
	print,'Usage: obsdates,star,dates,scans'
	print,'Required input: star.'
	return
endif else star=strupcase(star)
;
name=!oyster_dir+'npoi/obsbase.xdr'
result=file_search(name,count=fcount)
if fcount eq 0 then begin
        print,'***Error(OBSSTARS): file not found!'
        return
endif
restore,name
;
if n_elements(star) eq 0 then begin
	print,'***Error(OBSDATES): star not specified!'
	return
endif
;
index=where(database.star eq star,count)
if count eq 0 then begin
	print,'Warning(OBSDATES): star not found!'
	dates=''
	scans=0
	return
endif
database=database[index]
dates=database.date
if count gt 1 then dates=dates[uniq(dates,sort(dates))]
;
num_dates=n_elements(dates)
scans=intarr(num_dates)
for i=0,num_dates-1 do begin
	index=where(database.date eq dates[i] $
		and database.star eq star,count)
	scans[i]=count
endfor
;
if n_params() ne 3 then begin
	n_d_p_l=7
	while(printindex(num_dates,n_d_p_l,i,j)) do begin
		print,dates[i:j]
		print,string(scans[i:j],format='(i7)')+'   '
	endwhile
endif
;

end
;-------------------------------------------------------------------------------
pro obssummary,stars
;
common StarBase,StarTable,Notes
;
if n_elements(stars) eq 0 then obsstars,stars
if n_elements(stars) eq 0 then return
;
if n_elements(StarTable) gt 0 then startable_save=StarTable
get_startable,stars
rename_starids,'fkv-bsc'
read_catalogs
rename_bsc
;
openw,unit,'obssummary.dat',/get_lun
;
for i=0,n_elements(stars)-1 do begin
	obsdates,stars[i],dates,scans
	if n_elements(dates) gt 0 then begin
		index=where(scans gt 2,count)
		if count gt 0 then begin
			printf,unit,stars[i],'(', $
				strcompress(StarTable[i].name),'):', $
				format='(a9,a1,a,a3)'
			printf,unit,dates[index],format='(100a11)'
			printf,unit,scans[index],format='(100(6x,i2,3x))'
		endif
	endif
endfor
;
free_lun,unit
if n_elements(save_startable) gt 0 then StarTable=save_startable
;
end
;-------------------------------------------------------------------------------
