;*******************************************************************************
; File: structure.pro
;
; Description:
; ------------
; This file contains all functions allocating structures. They are collated
; here because of the slight syntax difference between PV-WAVE and IDL
; regarding the definition of anonymous structures. All structures here
; are anonymous since they must allow repeated allocation with different
; dimensioning parameters.
;
; Block directory:
; ----------------
; Block 1: build_scantable,build_bgtable,build_stationtable,
;	   build_startable,build_npoitable,build_critable
;	   build_pointflagtable,build_inchflagtable,
;	   build_scanflagtable,build_bgflagtable
;	   build_cal_entries
; Block 2: allocmetroconfig,allocgeoparms,allocgenconfig,
;	   allocscan,allocbgscan,scan,bgscan
;	   alloc_obsbase,alloc_biasbase
; Block 3: nightinfo,location,velocity,magnitude,
;	   alloc_parallax,alloc_k1,alloc_k2,alloc_vsini
; Block 4: alloc_ds_options,alloc_ps_options,
;	   alloc_plotorbit_options,alloc_plotvel_options,alloc_plotmag_options,
;	   alloc_cal_entries,alloc_cal_options,alloc_fit_options,
;	   alloc_ellipse_options,alloc_orbit_options,alloc_component,
;	   alloc_limb_data,alloc_limb_table,alloc_map,alloc_strip
; Block 5: alloc_gen_model,alloc_star_struct,alloc_binary_struct
;
;************************************************************************Block 1
function build_scantable,Iscan,ScanId,StarId,Code,Station, $
			StartTime,StopTime,NumPoint, $
			NumCoh,NumIncoh
table=replicate({ $
	Iscan:Iscan[0], $
	ScanId:ScanId[0], $
	StarId:StarId[0], $
	Code:Code[0], $
	Station:Station[*,0], $
	StartTime:StartTime[0], $
	StopTime:StopTime[0], $
	NumPoint:NumPoint[0], $
	NumCoh:NumCoh[0], $
	NumIncoh:NumIncoh[0]},n_elements(Iscan))
table.Iscan=Iscan
table.ScanId=ScanId
table.StarId=StarId
table.Code=Code
table.Station=Station
table.StartTime=StartTime
table.StopTime=StopTime
table.NumPoint=NumPoint
table.NumCoh=NumCoh
table.NumIncoh=NumIncoh
return,table
end
;-------------------------------------------------------------------------------
function build_bgtable,Iscan,ScanId,Time,Ra,Dec
table=replicate({ $
	Iscan:Iscan[0], $
	ScanId:ScanId[0], $
	Time:Time[0], $
	Ra:Ra[0], $
	Dec:Dec[0]},n_elements(Iscan))
table.Iscan=Iscan
table.ScanId=ScanId
table.Time=Time
table.Ra=Ra
table.Dec=Dec
return,table
end
;-------------------------------------------------------------------------------
function build_stationtable,StationId,X,Y,Z,D
;
; Fields DL_ID (delayline ID) and BC_IN (beam combiner input) added
; for configuration control purposes.
;
table=replicate({ $
	StationId:StationId[0], $
	X:X[0], $
	Y:Y[0], $
	Z:Z[0], $
	D:D[0],DL_ID:0,BC_IN:0},n_elements(StationId))
table.StationId=StationId
table.X=X
table.Y=Y
table.Z=Z
table.D=D
return,table
end
;-------------------------------------------------------------------------------
function build_startable,starid,name,var,ra,rae,dec,dece, $
	mv,av,mj,mh,mk,flp,f12,bv,ub,ri,dmv,amv,by,m1,c1,beeta, $
	diameter,diameter2,omega,tilt,rapa,zexp, $
	zerospacing,a0,a1,a2,mass,mass2,feh, $
	pmra,pmrae,pmdec,pmdece,rv,vsini,px,pxe,d, $
	spectrum,type,type2,class,class2,sed,wvl, $
	teff,teff2,logg,logg2, $
	p,t,o,e,i,n,a,k1,k2,mf,m1sin3i,m2sin3i,a1sini,a2sini, $
	bflag,hflag,vflag,sflag,model,reference
table=replicate({ $
	starid:starid[0],name:name[0],var:var[0], $
	ra:ra[0],rae:rae[0],dec:dec[0],dece:dece[0], $
        bat:0,hdn:0L,bsc:0,fkv:0,hic:0L, $
	sao:0L,ads:0,wds:0L,fln:0, $
        mv:mv[0],av:av[0],mj:mj[0],mh:mh[0],mk:mk[0],flp:flp[0],f12:f12[0], $
	bv:bv[0],ub:ub[0],ri:ri[0],dmv:dmv[0],amv:amv[0], $
	by:by[0],m1:m1[0],c1:c1[0],beta:beeta[0], $
        diameter:diameter[0],diameter2:diameter2[0], $
	omega:omega[0],tilt:tilt[0],rapa:rapa[0],zexp:zexp[0], $
	zerospacing:zerospacing[0], $
	a0:a0[0],a1:a1[0],a2:a2[0], $
	mass:mass[0],mass2:mass2[0],feh:feh[0], $  
        pmra:pmra[0],pmrae:pmrae[0],pmdec:pmdec[0],pmdece:pmdece[0], $
	rv:rv[0],vsini:vsini[0],px:px[0],pxe:pxe[0],d:d[0], $
	spectrum:spectrum[0],type:type[0],type2:type2[0], $
	class:class[0],class2:class2[0], $
	sed:sed[*,0],wvl:wvl[*,0], $
        teff:teff[0],teff2:teff2[0],logg:logg[0],logg2:logg2[0], $
        p:p[0],t:t[0],o:o[0],e:e[0],i:i[0],n:n[0],a:a[0], $
	k1:k1[0],k2:k2[0], $
	mf:mf[0],m1sin3i:m1sin3i[0],m2sin3i:m2sin3i[0], $
	a1sini:a1sini[0],a2sini:a2sini[0], $
	bflag:bflag[0],hflag:hflag[0],vflag:vflag[0],sflag:sflag[0], $
	model:model[0],reference:reference[0]}, $
	n_elements(starid))
if n_elements(starid) eq 1 then begin
	table.starid=starid[0]
	table.name=name[0]
	table.var=var[0]
	table.ra=ra[0]
	table.rae=rae[0]
	table.dec=dec[0]
	table.dece=dece[0]
	table.bat=0
	table.hdn=0L
	table.bsc=0
	table.fkv=0
	table.hic=0L
	table.sao=0L
	table.ads=0
	table.wds=0L
	table.fln=0
	table.mv=mv[0]
	table.av=av[0]
	table.mj=mj[0]
	table.mh=mh[0]
	table.mk=mk[0]
	table.flp=flp[0]
	table.f12=f12[0]
	table.bv=bv[0]
	table.ub=ub[0]
	table.ri=ri[0]
	table.dmv=dmv[0]
	table.amv=amv[0]
	table.by=by[0]
	table.m1=m1[0]
	table.c1=c1[0]
	table.beta=beeta[0]
	table.diameter=diameter[0]
	table.diameter2=diameter2[0]
	table.omega=omega[0]
	table.tilt=tilt[0]
	table.rapa=rapa[0]
	table.zexp=zexp[0]
	table.zerospacing=zerospacing[0]
	table.a0=a0[0]
	table.a1=a1[0]
	table.a2=a2[0]
	table.mass=mass[0]
	table.mass2=mass2[0]
	table.feh=feh[0]
	table.pmra=pmra[0]
	table.pmrae=pmrae[0]
	table.pmdec=pmdec[0]
	table.pmdece=pmdece[0]
	table.rv=rv[0]
	table.vsini=vsini[0]
	table.px=px[0]
	table.pxe=pxe[0]
	table.d=d[0]
	table.spectrum=spectrum[0]
	table.type=type[0]
	table.type2=type2[0]
	table.class=class[0]
	table.class2=class2[0]
	table.sed=sed[*,0]
	table.wvl=wvl[*,0]
	table.teff=teff[0]
	table.teff2=teff2[0]
	table.logg=logg[0]
	table.logg2=logg2[0]
	table.p=p[0]
	table.t=t[0]
	table.o=o[0]
	table.e=e[0]
	table.i=i[0]
	table.n=n[0]
	table.a=a[0]
	table.k1=k1[0]
	table.k2=k2[0]
	table.mf=mf[0]
	table.m1sin3i=m1sin3i[0]
	table.m2sin3i=m2sin3i[0]
	table.a1sini=a1sini[0]
	table.a2sini=a2sini[0]
	table.bflag=bflag[0]
	table.hflag=hflag[0]
	table.vflag=vflag[0]
	table.sflag=sflag[0]
	table.model=model[0]
	table.reference=reference[0]
endif else begin
	table.starid=starid
	table.name=name
	table.var=var
	table.ra=ra
	table.rae=rae
	table.dec=dec
	table.dece=dece
	table.bat=intarr(n_elements(starid))
	table.hdn=lonarr(n_elements(starid))
	table.bsc=intarr(n_elements(starid))
	table.fkv=intarr(n_elements(starid))
	table.hic=lonarr(n_elements(starid))
	table.sao=lonarr(n_elements(starid))
	table.ads=intarr(n_elements(starid))
	table.wds=lonarr(n_elements(starid))
	table.fln=intarr(n_elements(starid))
	table.mv=mv
	table.av=av
	table.mj=mj
	table.mh=mh
	table.mk=mk
	table.flp=flp
	table.f12=f12
	table.bv=bv
	table.ub=ub
	table.ri=ri
	table.dmv=dmv
	table.amv=amv
	table.by=by
	table.m1=m1
	table.c1=c1
	table.beta=beeta
	table.diameter=diameter
	table.diameter2=diameter2
	table.omega=omega
	table.tilt=tilt
	table.rapa=rapa
	table.zexp=zexp
	table.zerospacing=zerospacing
	table.a0=a0
	table.a1=a1
	table.a2=a2
	table.mass=mass
	table.mass2=mass2
	table.feh=feh
	table.pmra=pmra
	table.pmrae=pmrae
	table.pmdec=pmdec
	table.pmdece=pmdece
	table.rv=rv
	table.vsini=vsini
	table.px=px
	table.pxe=pxe
	table.d=d
	table.spectrum=spectrum
	table.type=type
	table.type2=type2
	table.class=class
	table.class2=class2
	table.sed=sed
	table.wvl=wvl
	table.teff=teff
	table.teff2=teff2
	table.logg=logg
	table.logg2=logg2
	table.p=p
	table.t=t
	table.o=o
	table.e=e
	table.i=i
	table.n=n
	table.a=a
	table.k1=k1
	table.k2=k2
	table.mf=mf
	table.m1sin3i=m1sin3i
	table.m2sin3i=m2sin3i
	table.a1sini=a1sini
	table.a2sini=a2sini
	table.bflag=bflag
	table.hflag=hflag
	table.vflag=vflag
	table.sflag=sflag
	table.model=model
	table.reference=reference
endelse
return,table
end
;-------------------------------------------------------------------------------
function build_npoitable,starid,name,ra,dec,mv,spectrum, $
			 pmra,pmdec,rv,px,diameter
table=replicate({ $
	starid:starid[0],name:name[0],ra:ra[0],dec:dec[0], $
	mv:mv[0],spectrum:spectrum[0], $
	pmra:pmra[0],pmdec:pmdec[0], $
	rv:rv[0],px:px[0],diameter:diameter[0]},n_elements(starid))
table.starid=starid
table.name=name
table.ra=ra
table.dec=dec
table.mv=mv
table.spectrum=spectrum
table.pmra=pmra
table.pmdec=pmdec
table.rv=rv
table.px=px
table.diameter=diameter
return,table
end
;-------------------------------------------------------------------------------
function build_critable,id1,id2
table=replicate({ $
	id1:long(id1[0]),id2:long(id2[0])},n_elements(id1))
table.id1=id1
table.id2=id2
return,table
end
;-------------------------------------------------------------------------------
function build_pointflagtable,reason,item,baseline,triple,channel, $
			      inbeam,outbeam,time
table=replicate({ $
	reason:reason[0],item:item[0],baseline:baseline[0],triple:triple[0], $
	channel:channel[0],inbeam:inbeam[0], $
	outbeam:outbeam[0],time:time[0]},n_elements(reason))
table.reason=reason
table.item=item
table.baseline=baseline
table.triple=triple
table.channel=channel
table.inbeam=inbeam
table.outbeam=outbeam
table.time=time
return,table
end
;-------------------------------------------------------------------------------
function build_inchflagtable,reason,item,inbeam,time
table=replicate({ $
	reason:reason[0],item:item[0], $
	inbeam:inbeam[0],time:time[0]},n_elements(reason))
table.reason=reason
table.item=item
table.inbeam=inbeam
table.time=time
return,table
end
;-------------------------------------------------------------------------------
function build_scanflagtable,reason,item,baseline,triple,channel, $
			     inbeam,outbeam,time
table=replicate({ $
	reason:reason[0],item:item[0],baseline:baseline[0],triple:triple[0], $
	channel:channel[0],inbeam:inbeam[0], $
	outbeam:outbeam[0],time:time[0]},n_elements(reason))
table.reason=reason
table.item=item
table.baseline=baseline
table.triple=triple
table.channel=channel
table.inbeam=inbeam
table.outbeam=outbeam
table.time=time
return,table
end
;-------------------------------------------------------------------------------
function build_bgflagtable,reason,item,channel,outbeam,time
table=replicate({ $
	reason:reason[0],item:item[0],channel:channel[0], $
	outbeam:outbeam[0],time:time[0]},n_elements(reason))
table.reason=reason
table.item=item
table.channel=channel
table.outbeam=outbeam
table.time=time
return,table
end
;************************************************************************Block 2
function allocmetroconfig,NumPlate,NumSid,maxNumSensor
return,{ $
PlateIdx:lonarr(NumPlate), $
SidModel:{FeedBeamAng:dblarr(2,NumSid),FeedBeamAngErr:dblarr(2,NumSid), $
         SidAng:dblarr(2,NumSid),SidAngErr:dblarr(2,NumSid), $
	 ZeroAng:dblarr(2,NumSid),ZeroAngErr:dblarr(2,NumSid), $
	 AxisOffAng:dblarr(NumSid),AxisOffAngErr:dblarr(NumSid), $
	 MirrorOffAng:dblarr(NumSid),MirrorOffAngErr:dblarr(NumSid), $
	 PivotOffset:dblarr(4,NumSid),PivotOffsetErr:dblarr(4,NumSid)-1, $
	 CatsEyeOff:dblarr(NumSid),CatsEyeOffErr:dblarr(NumSid)}, $
FBAirTemp:{NumSensor:intarr(NumSid),label:strarr(maxNumSensor,NumSid)}, $
FBSolidTmp:{NumSensor:intarr(NumSid),label:strarr(maxNumSensor,NumSid)}}
;
end
;-------------------------------------------------------------------------------
function allocgeoparms
return,{Date:'',SystemId:'', $
	Latitude:0.d0,Longitude:0.d0,Altitude:0.d0, $
	EarthRadius:0.d0,J2:0.d0, $
	pole_x_coeffs:dblarr(3),pole_y_coeffs:dblarr(3), $
	ut1utc_coeffs:dblarr(3),TAI_UTC:0.d0,TDT_TAI:0.d0}
end
;-------------------------------------------------------------------------------
function allocgenconfig,NumSid,NumOutBeam,NumTriple, $
	maxBaseline,maxSpecChan,maxConfig,geninfo=geninfo
;
; Changes "Why 0?" applied 18.5.2021, has caused a problem
;
if n_elements(geninfo) eq 0 then geninfo=0
if n_elements(numoutbeam) eq 0 then numoutbeam=15
;
if geninfo then $
return,{Date:'',InstrCohint:0.d0,BeamCombinerId:0L,ConfigId:'DEFAULT', $
	RefStation:0L,NumPlate:0L,NumBin:0L, $
	NumSid:0L, $
	NumOutBeam:0L, $
		NumBaseline:lonarr(NumOutBeam), $
		NumSpecChan:lonarr(NumOutBeam), $
	NumTriple:0L} $
else $
return,{Date:'',InstrCohint:0.d0,BeamCombinerId:0L,ConfigId:'DEFAULT', $
	RefStation:0L,NumPlate:0L,NumBin:0L, $
	Config:strarr(maxConfig)+'DEFAULT', $
	APDFlux:fltarr(NumSid,maxSpecChan,NumOutBeam), $
	NATFlux:fltarr(NumSid,maxSpecChan,NumOutBeam), $
;Why 0?	NumSid:0L, $
	NumSid:long(NumSid), $
		StationId:strarr(NumSid)+'   ', $
		Diameter:fltarr(NumSid), $
		SidConId:lonarr(NumSid),SiderostatId:lonarr(NumSid), $
		DelayLineId:lonarr(NumSid),Stroke:fltarr(NumSid), $
		StarTrackerId:lonarr(NumSid),BCInputId:lonarr(NumSid), $
 		StationCoord:dblarr(4,NumSid), $
;Why 0?	NumOutBeam:0L, $
	NumOutBeam:long(NumOutBeam), $
		NumBaseline:lonarr(NumOutBeam), $
		NumSpecChan:lonarr(NumOutBeam), $
		NumBiasFreq:lonarr(NumOutBeam), $
 		SpectrometerId:sindgen(NumOutBeam), $
		FringeMod:lonarr(maxBaseline,NumOutBeam), $
		BiasMod:lonarr(maxBaseline,NumOutBeam), $
		BaselineId:strarr(maxBaseline,NumOutBeam)+'       ', $
 		Wavelength:dblarr(maxSpecChan,NumOutBeam), $
		WavelengthErr:dblarr(maxSpecChan,NumOutBeam), $
 		ChanWidth:dblarr(maxSpecChan,NumOutBeam), $
		ChanWidthErr:dblarr(maxSpecChan,NumOutBeam), $
		V2Bias:fltarr(2,maxBaseline,maxSpecChan,NumOutBeam,maxConfig), $
		Response:fltarr(2,maxSpecChan,NumOutBeam,maxConfig), $
		TrackCal:fltarr(3,maxBaseline,maxSpecChan,NumOutBeam,maxConfig), $
		OffsetCal:fltarr(2,maxBaseline,maxSpecChan,NumOutBeam,maxConfig), $
;Why 0?	NumTriple:0L, $
	NumTriple:long(NumTriple), $
		TABias:fltarr(2,maxSpecChan,NumTriple,maxConfig), $
	      	TripleBeam:lonarr(3,NumTriple), $
		TripleBase:lonarr(3,NumTriple), $
	      	TripleChan:lonarr(maxSpecChan,3,NumTriple), $
		TripleNumChan:lonarr(NumTriple)}
end
;-------------------------------------------------------------------------------
function allocscan,NIB,NOB,NTR,MC,MB
return,{Iscan:1,StarId:' ',Star:' ',Time:0.d0,Int_time:0.d0, $
	HA:0.d0,RA:0.d0,DEC:0.d0,ZA:0.d0,AZ:0.d0,MA:0.d0, $
	FDLPos:dblarr(NIB),FDLPosErr:fltarr(NIB)-1, $
	GeoDelay:dblarr(NIB),UVW:dblarr(NOB,MC,MB,3), $
	VisSq:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	VisSqErr:reform(fltarr(NOB,MC,MB),NOB,MC,MB)-1, $
	VisSqC:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	VisSqCErr:reform(fltarr(NOB,MC,MB),NOB,MC,MB)-1, $
	VisSqE:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	VisSqEErr:reform(fltarr(NOB,MC,MB),NOB,MC,MB)-1, $
	VisSqEC:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	VisSqECErr:reform(fltarr(NOB,MC,MB),NOB,MC,MB)-1, $
	ComplexVis:reform(complexarr(NOB,MC,MB),NOB,MC,MB), $
	ComplexWeight:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	VisPhase:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	VisPhaseErr:reform(fltarr(NOB,MC,MB),NOB,MC,MB)-1, $
	ComplTriple:reform(fltarr(NTR,2,MC),NTR,2,MC), $
	ComplTripleErr:reform(fltarr(NTR,2,MC),NTR,2,MC)-1, $
	TripleAmp:reform(fltarr(NTR,MC),NTR,MC), $
	TripleAmpErr:reform(fltarr(NTR,MC),NTR,MC)-1, $
	TripleAmpC:reform(fltarr(NTR,MC),NTR,MC), $
	TripleAmpCErr:reform(fltarr(NTR,MC),NTR,MC)-1, $
	TripleAmpE:reform(fltarr(NTR,MC),NTR,MC), $
	TripleAmpEErr:reform(fltarr(NTR,MC),NTR,MC)-1, $
	TripleAmpEC:reform(fltarr(NTR,MC),NTR,MC), $
	TripleAmpECErr:reform(fltarr(NTR,MC),NTR,MC)-1, $
	TriplePhase:reform(fltarr(NTR,MC),NTR,MC), $
	TriplePhaseErr:reform(fltarr(NTR,MC),NTR,MC)-1, $
	TriplePhaseC:reform(fltarr(NTR,MC),NTR,MC), $
	TriplePhaseCErr:reform(fltarr(NTR,MC),NTR,MC)-1, $
	VisSqM:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	VisPhaseM:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	ComplTripleM:reform(fltarr(NTR,2,MC),NTR,2,MC), $
	TripleAmpM:reform(fltarr(NTR,MC),NTR,MC), $
	TriplePhaseM:reform(fltarr(NTR,MC),NTR,MC), $
; Seeing data: R0=Fried parameter, SI=scintillation index
	R0:1.0, $
	Pres:760.0, $
	RHum:10.0, $
	Temp:10.0, $
	PWV:10.0, $
	SI:reform(fltarr(NOB,MC),NOB,MC), $
	T0:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	DelayRMS:reform(fltarr(NOB,MB),NOB,MB), $
	PhaseRMS:reform(fltarr(NOB,MB),NOB,MB), $
	DelayJitter2:reform(fltarr(NOB,MB),NOB,MB), $
	NATJitter2:reform(fltarr(NIB,2),NIB,2), $
; NPOI specific
	ParX:fltarr(NIB),ParXErr:fltarr(NIB), $
	ParY:fltarr(NIB),ParYErr:fltarr(NIB), $
	ParZ:fltarr(NIB),ParZErr:fltarr(NIB), $
	MetroDelay:fltarr(NIB),MetroDelayErr:fltarr(NIB), $
	WhiteDelay:fltarr(NIB),WhiteDelayErr:fltarr(NIB), $
	ModelDelay:fltarr(NIB),ModelDelayErr:fltarr(NIB), $
	GrpDelay:dblarr(NIB),GrpDelayErr:fltarr(NIB)-1, $
	DryDelay:dblarr(NIB),DryDelayErr:fltarr(NIB)-1, $
	WetDelay:dblarr(NIB),WetDelayErr:fltarr(NIB)-1, $
	NATJitter:fltarr(NIB),NATJitterErr:fltarr(NIB)-1, $
	NATCounts:fltarr(NIB),NATCountsErr:fltarr(NIB)-1, $
	DelayJitter:reform(fltarr(NOB,MB),NOB,MB), $
	DelayJitterErr:reform(fltarr(NOB,MB),NOB,MB)-1, $
	PhotonRate:reform(fltarr(NOB,MC),NOB,MC), $
	PhotonRateErr:reform(fltarr(NOB,MC),NOB,MC)-1, $
	BackgndRate:reform(fltarr(NOB,MC),NOB,MC), $
	BackgndErr:reform(fltarr(NOB,MC),NOB,MC)-1, $
; VLTI specific
	Photometry:reform(fltarr(NIB,NOB,MC),NIB,NOB,MC), $
	PhotometryErr:reform(fltarr(NIB,NOB,MC),NIB,NOB,MC)-1, $
	PhotometryC:reform(fltarr(NIB,NOB,MC),NIB,NOB,MC), $
	PhotometryCerr:reform(fltarr(NIB,NOB,MC),NIB,NOB,MC)-1, $
	PhotometryM:reform(fltarr(NIB,NOB,MC),NIB,NOB,MC), $
; VLTI/MIDI specific
	ChopThrow:fltarr(NIB),ChopAngle:fltarr(NIB), $
	AcqFilter:' ', AcqFOV:fltarr(NIB), $
	AcqPSF:fltarr(NIB,5),AcqFlux:fltarr(NIB), $
; VLTI/AMBER specific
	TrackOffset:reform(fltarr(NOB,MB),NOB,MB), $
	DiffPhase:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	DiffPhaseErr:reform(fltarr(NOB,MC,MB),NOB,MC,MB)-1, $
	DiffPhaseC:reform(fltarr(NOB,MC,MB),NOB,MC,MB), $
	DiffPhaseCErr:reform(fltarr(NOB,MC,MB),NOB,MC,MB)-1, $
	DiffPhaseM:reform(fltarr(NOB,MC,MB),NOB,MC,MB)}
; VLTI/PRIMA specific
;	Swapped:0, $
;	PrimetB:dblarr(NIB),PrimetBErr:dblarr(NIB)-1, $
;	DDLDelay:dblarr(NIB),DDLDelayErr:dblarr(NIB)-1, $
end
;-------------------------------------------------------------------------------
function allocbgscan,NOB,MC
return,{Time:0.d0,Rate:fltarr(NOB,MC),RateErr:fltarr(NOB,MC)}
end
;-------------------------------------------------------------------------------
function scan
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
NIB=GenConfig.NumSid
NOB=GenConfig.NumOutBeam
NTR=GenConfig.NumTriple
;
; We need to allocate space for at least one triple
if NTR eq 0 then NTR=1
;
; Find out about the maximum number of baselines/outputbeam: MB
MB=max(GenConfig.NumBaseline)
MB=n_elements(GenConfig.BaselineId[*,0])	; equalize dimensions
;
; Find out about the maximum number of spectral channels/outputbeam: MC
MC=max([GenConfig.NumSpecChan,GenConfig.TripleNumChan])
MC=n_elements(GenConfig.Wavelength[*,0])	; equalize dimensions
;
if NIB*NOB*NTR*MC*MB eq 0 then return,-1
return,allocscan(NIB,NOB,NTR,MC,MB)
end
;-------------------------------------------------------------------------------
function bgscan
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
NOB=GenConfig.NumOutBeam
;
; Find out about the maximum number of spectral channels/outputbeam: MC
MC=max(GenConfig.NumSpecChan)
;
return,allocbgscan(NOB,MC)
end
;-------------------------------------------------------------------------------
function alloc_obsbase,dates,stars,times
table=replicate({ $
	date:dates[0], $
	star:stars[0], $
	time:times[0]},n_elements(dates))
table.date=dates
table.star=stars
table.time=times
return,table
end
;-------------------------------------------------------------------------------
function alloc_biasbase,dates,bias,config
table=replicate({ $
	date:dates[0], $
	config:'      ', $
	bias:bias[*,*,*,*,0]},n_elements(dates))
table.date=dates
table.bias=bias
table.config=config
return,table
end
;************************************************************************Block 3
function nightinfo
;
; Allocate structure for bufferinfo to hold information for tags of the
; scans and genconfig structures. Those two have to be present, but their
; contents are irrelevant.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
num_tags=n_tags(scans)
num_levels=8	; Max expected number of data dimensions + 3
num_strings=0
for i=0,num_tags-1 do begin
	dims=size(scans.(i))
	if dims[dims[0]+1] eq 7 then num_strings=num_strings+1
endfor
;
nisc={size:lonarr(num_tags,num_levels), $
	maxlen:lonarr(num_strings+1), $
	NumScan:0}
;
num_tags=n_tags(genconfig)
num_levels=8	; Max expected number of data dimensions + 3
num_strings=0
for i=0,num_tags-1 do begin
	dims=size(genconfig.(i))
	if dims[dims[0]+1] eq 7 then num_strings=num_strings+1
endfor
;
nigc={size:lonarr(num_tags,num_levels),maxlen:lonarr(num_strings+1), $
	NumSid:0,NumOutBeam:0, $
	maxTriple:0,maxBaseline:0,maxChannel:0,maxConfig:0}
;
return,{file:'',xdr:0,nisc:nisc,nigc:nigc}
;
end
;-------------------------------------------------------------------------------
function location
;
return,{component:'',date:'',jy:0.d0,jd:0.d0, $
	rho:0.d0,theta:0.d0, $
	rhoM:0.d0,thetaM:0.d0, $
	emajor:0.d0,eminor:0.d0,pa:0.d0}
end
;-------------------------------------------------------------------------------
function velocity 
;
return,{component:'',date:'',jd:0.d0, $
	value:0.d0,error:0.d0, $
	valuem:0.d0,symbol:1,color:1}
end
;-------------------------------------------------------------------------------
function magnitude 
;
return,{component:'',filter:'',wavelength:0.d0,date:'',jd:0.d0, $
	value:0.d0,error:0.d0, $
	valuem:0.d0}
end
;-------------------------------------------------------------------------------
function alloc_parallax
;
return,{value:dblarr(10),error:dblarr(10),valuem:dblarr(10)}
;
end
;-------------------------------------------------------------------------------
function alloc_k1
;
return,{value:0.d0,error:0.d0,valuem:0.d0}
;
end
;-------------------------------------------------------------------------------
function alloc_k2
;
return,{value:0.d0,error:0.d0,valuem:0.d0}
;
end
;-------------------------------------------------------------------------------
function alloc_vsini
;
return,{value:0.d0,error:0.d0,valuem:0.d0}
;
end
;************************************************************************Block 4
function alloc_ds_options
;
; i=interferometry:
; (v2=V^2, ta=triple amp, vp=visibility phase, tp=triple phase, fl=int. flux)
; a=astrometry (c=circular), s=spectroscopy, p=photometry, pm=mod. comp. phot.,
; px=parallax,k1=k1,k2=K2,vsini=vsini
;
; Obsolete:
; ds_options.t: -1: no triple data, 0: V2 and triple data, 1: only triple data
;
return,{i:0.0,v2:0,ta:0,vp:0,tp:0,fl:0, $
	a:0.0,c:0.0,s:0.0,p:0.0,pm:0.0,px:0.0,k1:0.0,k2:0.0,vsini:0.0}
;
end
;-------------------------------------------------------------------------------
function alloc_ps_options
;
; e=errors, f=flagged, l=lines, a=all, d=3D, g=image, r=NoFIR, n=NoSort 
; t=NoTrace, o=all OBs, i=all IBs, m=Model, c=Color, s=Submit, p=Paper
; v=normalize amplitude ratio to 1 (see alloc_ps_options in plotwidget.pro)
;
return,{e:0,f:0,l:0,a:0,d:0,g:0,r:0,n:0,t:0,o:0,i:0,m:0,c:0,s:0,p:0,v:0}
;
end
;-------------------------------------------------------------------------------
function alloc_plotorbit_options
;
return,{e:0,f:0,l:0,o:0,j:0,c:0,s:0,t:0,component:''}
;
end
;-------------------------------------------------------------------------------
function alloc_plotvel_options
;
; p=phase, o=orbit, r=residuals, b=camera
;
return,{e:0,f:0,p:0,o:0,c:0,r:0,b:0,a:0,s:0,component:''}
;
end
;-------------------------------------------------------------------------------
function alloc_plotmag_options
;
return,{e:0,f:0,p:0,o:0,c:0,r:0,b:0,a:0,s:0,component:'',filter:''}
;
end
;-------------------------------------------------------------------------------
function alloc_cal_entries,num_f,num_i,num_t,num_st,num_sc,num_e
;
cal_entry={variable:-1,reason:'', $
        outbeam:-1,triple:-1,baseline:-1,channel:-1, $
        stars:strarr(num_st),calstars:strarr(num_st),scans:intarr(num_sc), $
        indicators:strarr(num_i),functions:strarr(num_i,num_f), $
        coeffs:fltarr(num_t),x_mid:fltarr(num_i),x_scl:fltarr(num_i)+1, $
        num_indct:-1,num_funcn:-1,num_coeffs:-1,num_scans:-1}
return,replicate(cal_entry,num_e)
;
end
;-------------------------------------------------------------------------------
function alloc_cal_options
;
; v=0: Vissq; v=1: TripleAmp; v=2: TriplePhase; 
; s=0: no smoothing; s=1: smoothing
; l=0: no loop; l=1: loop
return,{l:0,s:0,v:0,n:['VisSq','TripleAmp','TriplePhase']}
;
end
;-------------------------------------------------------------------------------
function alloc_fit_options
;
; o=one step; h=optimized; f=floating calibration
return,{o:0,h:0,f:0,alamda:1.d0,chifr:0.001d0,tolerance:1d-4}
;
end
;-------------------------------------------------------------------------------
function alloc_ellipse_options
;
return,{all:0,c:0,a:0,b:0,p:0,component:''}
;
end
;-------------------------------------------------------------------------------
function alloc_orbit_options
;
return,{all:0,a:0,e:0,i:0,w:0,n:0,p:0,t:0,component:''}
;
end
;-------------------------------------------------------------------------------
function alloc_component
;
return,{component:'',names:strarr(20)}
;
end
;-------------------------------------------------------------------------------
function alloc_limb_data,m,t,g,n,w,f,u,x,y,a,b
;
return,{m:m,t:t,g:g,n:n,w:w,f:f,u:u,x:x,y:y,a:a,b:b}
;
end
;-------------------------------------------------------------------------------
function alloc_limb_table,band,ulin,teff,logg
;
return,{band:band,ulin:ulin,teff:teff,logg:logg}
;
end
;-------------------------------------------------------------------------------
function alloc_map,icc,ccc,xcc,ycc,fcc,nx,ny
;
; In OYSTER, a map is really a CLEAN component list. All input arrays here
; are 1-dimensional vectors. It is therefore not really an image, though
; they can hold one.
;
if n_elements(nx) eq 0 then nx=0
if n_elements(ny) eq 0 then ny=0
;
return,{icc:icc,ccc:ccc,xcc:xcc,ycc:ycc,fcc:fcc,nx:nx,ny:ny}
;
end
;-------------------------------------------------------------------------------
function alloc_maps,r
;
; Obsolete! See next function.
;
; In OYSTER, a map is really a CLEAN component list. All input arrays here
; are 1-dimensional vectors. It is therefore not really an image, though
; they can hold one.
;
if n_elements(nx) eq 0 then nx=0
if n_elements(ny) eq 0 then ny=0
;
index=where(r.icc ne 0,n)
m=max(r.icc)
map=alloc_map(m,r.ccc,r.xcc,r.ycc,r.fcc)
maps=replicate(map,n)
;
j=0
for i=0,n-1 do begin
	maps[i].icc=r.icc(i)
	maps[i].ccc(0:r.icc(i)-1)=r.ccc(j:j+r.icc(i)-1)
	j=j+r.icc(i)
endfor
;
return,maps
;
end
;-------------------------------------------------------------------------------
function alloc_maps,icc,ccc,xcc,ycc,fcc,nx,ny
;
; In OYSTER, a map is really a CLEAN component list. All input arrays here
; are 1-dimensional vectors. It is therefore not really an image, though
; they can hold one.
;
if n_elements(nx) eq 0 then nx=0
if n_elements(ny) eq 0 then ny=0
;
index=where(icc ne 0,n)
m=max(icc)
map=alloc_map(m,ccc[0:m-1],xcc[0:m-1],ycc[0:m-1],fcc[0:m-1])
maps=replicate(map,n)
;
j=0
for i=0,n-1 do begin
	maps[i].icc=icc[i]
	maps[i].ccc(0:icc[i]-1)=ccc[j:j+icc[i]-1]
	maps[i].xcc(0:icc[i]-1)=xcc[j:j+icc[i]-1]
	maps[i].ycc(0:icc[i]-1)=ycc[j:j+icc[i]-1]
	maps[i].fcc(0:icc[i]-1)=fcc[j:j+icc[i]-1]
	j=j+icc[i]
endfor
;
return,maps
;
end
;-------------------------------------------------------------------------------
function alloc_strip,n,x,b,theta
;
return,{n:n,x:x,b:b,theta:theta}
;
end
;************************************************************************Block 5
function alloc_gen_model,num_wave
;
return,{starid:'', $
	 ra:0.d0, $
	 dec:0.d0, $
	 rv:0.d0, $
	 px:0.d0, $
	 sm:0.d0, $
	 wavelengths:dblarr(num_wave)}
;
end
;-------------------------------------------------------------------------------
function alloc_star_struct,num_wave,num_spot
;
magnitudes=dblarr(num_wave)
if num_wave eq 1 then magnitudes=magnitudes[0]
;
spot=fltarr(4)
spotparms=fltarr(4,num_spot)
;
return,{component:'', $
        wmc:'', $
        type:0, $
	sed:'', $
	model:'', $
	cellsize:0.d0, $
        mass:0.d0, $
        diameter:0.d0, $
        width:0.d0, $
	ratio:0.d0, $
	pa:0.d0, $
	omega:0.d0, $
	tilt:0.d0, $
        gr:0.d0, $
        albedo:0.d0, $
        teff:0.d0, $
        alpha:0.d0, $
        logg:0.d0, $
	xoff:0.d0, $
	yoff:0.d0, $
	spot:spot, $
	spotparms:spotparms, $
        magnitudes:magnitudes}
;
end
;-------------------------------------------------------------------------------
function alloc_binary_struct
;
return,{component:'', $
         method:0, $
         wdtype:0, $
         wdband:0, $
         massratio:0.d0, $
         semimajoraxis:0.d0, $
         inclination:0.d0, $
         ascendingnode:0.d0, $
         eccentricity:0.d0, $
         periastron:0.d0, $
         apsidalmotion:0.d0, $
         epoch:0.d0, $
         period:0.d0, $
         rho:0.d0, $
         theta:0.d0}
;
end
;-------------------------------------------------------------------------------
