pro analyze_vizier_phot,template,star=star,spec_class=spec_class, $
	t1=t1,t2=t2,dm=dm,l_dm=l_dm,binary=binary
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Example call: analyze_vizier_phot,template,star='HD27377',spec_class='B9V'
;
; For a given star, compile the SED with data from Vizier and save to disk.
; If the star is not defined, use the SED from the previous call.
; Additional parameters for a binary star: eff. temperatures t1 and t2,
; magnitude difference dm at wavelength l_dm [microns]
;
; Optional Pickles template on output: wave=template(*,0), flux=template(*,1)
; spec_class: e.g., 'B9V' to find closest template, or 'b9v' for specific file.
;
; Binary option: equal type and class only!
;
; Example: analyze_vizier_phot,template,star='HD217014',spec_class='g2iv'
; The keyword value will be assigned to spec_class in function pickles.
;
; plot,template(*,0),template(*,1)*1e6,psym=0, $
; 	xtitle='Wavelength ('+greek('mu')+'m]', $
; 	ytitle='F!D'+greek('lambda')+'!N * 10!E6!N ', $
; 	'(erg cm!E-2!N s!E-1!N'+greek('mu')+'m!E-1!N)'
; oplot,r(*,0),r(*,1)*1e6,psym=5
;
; The function to download data from vizier
forward_function vizier_phot
;
common LocalVizier,starid,phot,ferr,filt
common StarBase,startable,notes
;
if not keyword_set(star) then goto,ok_photometry else starid=oyster_starid(star)
;
if n_elements(startable) ne 0 then startable_bck=startable
get_startable,starid
;
phot_struct=vizier_phot(star) ; flux in Jy, wavelength in microns
if not isastruct(phot_struct) then return
if keyword_set(binary) then phot_struct.flux=phot_struct.flux/2
;
phot=[[phot_struct.wave],[phot_struct.flux],[phot_struct.ferr]]
filt=phot_struct.filt
ferr=phot_struct.ferr
nfilt=n_elements(filt)
;
; Remove data from some filters
if 1 then groom_vizier_phot,named_list='gerard'
nfilt=n_elements(filt)
;
; Remove outliers (not easy!)
ufilt=unique(filt)
for k=0,n_elements(ufilt)-1 do begin
	index=where(filt eq ufilt[k],n)
	if n ge 3 then begin
		mflux=median(phot[index,1])
		jndex=where(abs(phot[index,1]-mflux) gt 1e-7,m)
		if m gt 0 then ferr[index[jndex]]=-1
	endif
endfor
;
print,'Valid photometry found in '+strtrim(string(nfilt),2)+' filters.'
;
; Jump here if photometry has already been retrieved
ok_photometry:
;
if n_elements(starid) eq 0 then begin
	print,'Please specify star in call to this procedure!'
	return
endif
;
nfilt=n_elements(filt)
;
; Sort ascending to get good plots
index=sort(phot[*,0])
phot[*,0]=phot[index,0]
phot[*,1]=phot[index,1]
ferr=ferr[index]
filt=filt[index]
;
; Plot the photometry
!x.range=[0,2.5]
!y.range=[0,6e-6]
if startable.starid eq 'HDN217014' then !y.range=[0,3e-7]
if startable.starid eq 'HDN191110' then !y.range=[0,1e-6]
!x.margin=[12,3]
!p.charsize=2
plot,phot[*,0],phot[*,1],psym=3,title=starid, $
 	xtitle='Wavelength ('+greek('mu')+'m)', $
 	ytitle='F!D'+greek('lambda')+'!N '+ $
 	'(erg cm!E-2!N s!E-1!N'+greek('mu')+'m!E-1!N)';,/xlog,/ylog
errplot,phot[*,0],phot[*,1]-ferr/2,phot[*,1]+ferr/2,width=0.01
!x.margin=10
;
; Determine wavelength of SED maximum and estimate BB temperature
i=where(phot[*,1] eq max(phot[*,1])) & i=i[0]
print,'Wavelength [mu] of SED maximum:',float(phot[i,0])
teff_wien=wien(phot[i,0]*1e-6) & teff_wien=teff_wien[0]
print,'Temperature [K] of BB peaking at SED maximum:   '+ $
	strcompress(string(float(teff_wien)),/remove_all)
;
; Overplot blackbody (BB) curves
; Solid: 0, Dotted: 1, Dashed: 2, Dash-Dot: 3, Dashed-Long: 5
if startable.class eq 0 then begin
	print,'Warning: unknown class, assuming class V for ',star,'!'
	startable.class=5
endif
startable.teff=5689		; Teff from Boden et al. 2007 for HD 217014
startable.teff=11700		; Teff from Hummel et al. 2007 for HD 27376
startable.teff=teff_wien	; Teff from SED maximum
if strlen(startable.spectrum) gt 0 then begin
	startable.teff=teff_star(startable.spectrum); Teff from spectral type
	print,'Use BB temperature derived from stellar type: ', $
							startable.teff
endif else begin
	print,'Use temperature of BB peaking at SED maximum:    ', $
							float(teff_wien)
endelse
print,'Plot spectrum for BB with this temperature (dotted line)'
if startable.teff ne 0 then begin
	teff=startable.teff
	l_bb=phot[*,0]				; microns
	f_bb=blackbody(teff,l_bb*1e-6)
	idx=where(l_bb ge 0.3 and l_bb le 0.7)	; range to normalize
	f_rt=mean(f_bb[idx]/phot[idx,1])	; normalization ratio
	f_rt=median(f_bb[idx]/phot[idx,1])	; fits better at high flux
	l_bb=0.1+findgen(30)*0.1
	oplot,l_bb,blackbody(teff,l_bb*1e-6)/f_rt,linestyle=1
;	Re-compute spectrum
	f_bb=blackbody(teff,phot[*,0]*1e-6)/f_rt
endif
;
; Inform about outliers
h_index=where(alog(phot[*,1])/alog(f_bb) gt 1.1)
l_index=where(alog(phot[*,1])/alog(f_bb) lt 0.9)
print,'-------------------------------'
print,'Filters with outliers too high:'
for f=0,n_elements(h_index)-1 do print,filt[h_index[f]]
print,'Filters with outliers too low:'
for f=0,n_elements(l_index)-1 do print,filt[l_index[f]]
print,'-------------------------------'
;
; Fit Teff of blackbody spectrum
l=phot[*,0]*1e-6 	; [m]
y=phot[*,1]
w=fltarr(nfilt)+1	; weights
a=[startable.teff,1]
blackbody_pro,l,a,f	; get scaling factor f
a[1]=median(y/f)	; correct?
a[1]=mean(y/f)	; correct?
f=curvefit(l,y,w,a,fita=[1,0],function_name='blackbody_pro',/noderiv)
teff_bb=a[0]
print,'Temperature [K] of best-fit BB (dashed line): '+ $
	strcompress(string(float(teff_bb)),/remove_all)
l_bb=0.1+findgen(30)*0.1
oplot,l_bb,blackbody(teff_bb,l_bb*1e-6)*a[1],linestyle=2
; Re-compute spectrum at measured photometry
f_bb=blackbody(teff,phot[*,0]*1e-6)/f_rt
f_bb=blackbody(teff,phot[*,0]*1e-6)*a[1]
;
; Fit Teff of blackbody spectrum in log/log space
IF 0 THEN BEGIN
l=alog10(phot[*,0]*1e-6) 	; [m]
y=alog10(phot[*,1])
w=fltarr(nfilt)+1	; weights
a=[startable.teff,1]
blackbody_log_pro,l,a,f	; get scaling factor f
a[1]=median(10^y/10^f)
r=curvefit(l,y,w,a,fita=[1,0],function_name='blackbody_log_pro',/noderiv)
; oplot,1e6*10^l,10^r 	; check that the model fits the data
teff_bb=a[0]
f_bb=blackbody(teff_bb,phot[*,0]*1e-6)
idx=where(phot[*,0] ge 0.3 and phot[*,0] le 0.8)	; range to normalize
f_rt=mean(f_bb[idx]/phot[idx,1])		; normalize to SED
f_bb=blackbody(teff_bb,l_bb*1e-6)/f_rt
;oplot,l_bb,f_bb,linestyle=2
print,'Temperature [K] of best-fit BB: '+ $
	strcompress(string(float(teff_bb)),/remove_all)
ENDIF
;
; Fit Pickles template spectrum equal/nearest to the requested spectral class
if keyword_set(spec_class) then begin
;	spec_class=spec_class
;	Lower-case spec_class to select a specific template, e.g., g2iv
	if spec_class ne strupcase(spec_class) then begin
		template=pickles(star,template=spec_class)
		if template[0] eq 0 then begin
;			print,'Template not available'
			return
		endif
;	Upper-case spec_class to select nearest available template
	endif else begin
		template=pickles(startable.starid)
	endelse
	wave_p=template[*,0]*1e6	; [microns]
	flux_p=template[*,1]
	if total(flux_p) eq 0 then return ; No matching template was found
;	flux_p=flux_p*(max(f_bb)/max(flux_p)); Normalize template to BB
;	oplot,wave_p,flux_p,linestyle=0
;
;	Integrate (scaled) Pickles template fluxes over each filter.
;	Note that the template does not extend over all filter bandpasses!
;	Make copies of the variables in LOCALVIZIER
	phot_tpl=phot
	ferr_tpl=ferr
	filt_tpl=filt
	for i=0,n_elements(filt_tpl)-1 do begin
;		filter_c requires wl. in nm
		transmission=total(filter_c(wave_p*1e3,filt_tpl[i]))
		if transmission gt 0 then begin
		phot_tpl[i,1]=total(filter_c(wave_p*1e3,filt_tpl[i])*flux_p) $
			    /transmission
	    	endif else begin
;			print,'No transmission: '+filt_tpl(i)
			phot_tpl[i,1]=0
		endelse
	endfor
;	Use Pickles filter values to compute median ratio f
	index=where(phot_tpl[*,1] gt 0)
	f=median(phot[index,1]/phot_tpl[index,1])
	f=mean(phot[index,1]/phot_tpl[index,1]) ; more weight at higher fluxes
;	Scale template photometry by computed ratio
	phot_tpl[*,1]=phot_tpl[*,1]*f
	flux_tpl=phot_tpl[index,1]
	wave_tpl=phot_tpl[index,0]
	ferr_tpl=ferr_tpl[index]
	filt_tpl=filt_tpl[index]
	index=uniq(wave_tpl,sort(wave_tpl))
;	Plot photometry template
;	oplot,wave_tpl(index),flux_tpl(index),psym=-1	; filter values
	print,'Pickles template spectrum (solid line)'
	oplot,wave_p,f*flux_p,psym=0		; scaled Pickles temp.
;
;	Scale Pickles template as well (already done above)
	flux_p=flux_p*f
;
;	Integrate BB flux (the wavelength bin is 0.1 micron => normalize by 10)
	sigma=5.67e-5	; Boltzmann constant erg 1/cm^3 1/K^4
	wl_interval= $
		median(l_bb[1:n_elements(l_bb)-1]-l_bb[0:n_elements(l_bb)-2])
	f_bol_bb=total(f_bb)*wl_interval
;	Derive apparent diameter (/home/pti/nexsci/dev/sed/pieces/angularSize.m)
	theta=sqrt(4*f_bol_bb/(sigma*startable.teff^4))*2.062648e8
	print,'Derived BB angular diameter [mas]: ',theta
;
;	Integrate template flux (the bin is 0.0005 micron => normalize by 2000)
	wl_interval=median(wave_p[1:n_elements(wave_p)-1] $
			  -wave_p[0:n_elements(wave_p)-2])
	f_bol_flux_p=total(flux_p)*wl_interval
;	Derive apparent diameter
	theta=sqrt(4*f_bol_flux_p/(sigma*startable.teff^4))*2.062648e8
	print,'Derived TP angular diameter [mas]: ',theta
;
; (/home/pti/nexsci/dev/sed/pieces/angularSize.m)
; ## Angular size uncertainty calculation
; t1 = 2 * theta * sigmaTeff / Teff;  ## Teff error contribution
; t2 = 1/2 * theta * sigmaFbol / fBol; ## fBol error contribution
; sigmaTheta = sqrt(t1.^2.0 + t2.^2.0);  ## final error is both in quadrature
endif
;
; Restore pre-existing startable
if n_elements(startable_bck) ne 0 then startable=startable_bck
;
end
