pro average
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Averages PointData. Applies background correction. Stores the data in scans.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ModelFit,parameters,ds_options
;
; Check presence of data
if checkdata([1,2,6,7,8,10,11]) ne 0 then return
;
; Minimum number of valid data points needed for a scan
MINPOINTS=10
;
NS=n_elements(ScanTable)
;
; Allocate and initialize arrays for ScanData if not done before or up to date
flag=0
r=size(scans)
if r[n_elements(r)-2] ne 8 then flag=1
if not flag then if n_elements(scans) ne NS then flag=1
if not flag then if total(ScanTable.StarId eq scans.starid) ne NS then flag=1
if flag then begin
	scans=replicate(scan(),NS)
	s=size(scans[0])
	if s[n_elements(s)-2] ne 8 then begin
		print,'***Error(AVERAGE): GenConfig incomplete!'
		scans=-1
		return
	endif
	scans.starid=ScanTable.StarId
	if NS gt 1 then scans.iscan=lindgen(NS)+1 else scans.iscan=1
	bufferinfo=replicate(nightinfo(),1)
	ds_options.i=1
endif
; Set calibration factors
case system_id(SystemId) of
	'Mark3'   :normfactor=pi_circle^2/2
	'NPOI'    :normfactor=pi_circle^2/8/(1-1/sqrt(2)); Changed below!
	      else:begin
		   print,'***Error(AVERAGE): unknown system: ',SystemId,'!'
		   return
		   end
endcase
if GenConfig.NumTriple gt 0 then begin
      triple_factor=fltarr(NS,GenConfig.NumTriple,max(GenConfig.NumSpecChan))+1
      triple_norm_factor= $
	            fltarr(NS,GenConfig.NumTriple,max(GenConfig.NumSpecChan))+1
endif
;
; Choose the method for computing error bars, G, N, or P (see below)
UncertaintyEstimate='N'
;
; Set bias flag
if total(abs(genconfig.v2bias)) eq 0 then !v2bias=0 else !v2bias=1
if total(abs(genconfig.tabias)) eq 0 then !tabias=0 else !tabias=1
;
; Background rate, photon rate, visibilities
;
print,'Averaging rates and squared visibilities...'
;
for i=0,GenConfig.NumOutBeam-1 do begin
;
; Check for flagged channels so we can reduce the number of warnings
  dead_channel_index=total(reform(bgscans.rateerr[i,*] > 0),2)
  index=where(dead_channel_index eq 0,count)
  if count gt 1 then s='s' else s=''
  if count ge 1 then $
  print,'Warning(AVERAGE): ' $
	+string(count,format='(i3)')+' bad channel'+s+' in beam ' $
	+string(i+1,format='(i1)')+': '+string(index+1,format='(i2)')
  missing_bg_scans=intarr(n_elements(scantable))
for j=0,GenConfig.NumSpecChan[i]-1 do begin
for k=0,n_elements(Iscan)-1 do begin
;
  kk=Iscan[k]-1	; subtract 1 since arrays start with 0!
  StartTime=ScanTable[kk].StartTime
  StopTime=ScanTable[kk].StopTime
  sconf=scanconfig(Iscan[k],/starlist)
  kc=where(scanconfig(/starlist) eq sconf) & kc=kc[0]
  r=size(genconfig.v2bias) & if r[0] eq 4 then kc=0
;
; Check background measurement
  index=where(BGTable.ScanId eq ScanTable[kk].ScanId,bg_count)
  if bg_count gt 0 then begin
    bgr=bgscans[index].Rate(i,j)
    bgre=bgscans[index].RateErr(i,j)
    bgt=bgscans[index].Time
    bg_index=where(bgre gt 0,bg_count)
  endif
;
; Check photonrate
  PR=PhotonRate[i,j,Rec0[k]:RecN[k]] > 0.01
  PRE=PhotonRateErr[i,j,Rec0[k]:RecN[k]]
  pr_index=where(PRE gt 0,NumPoint)
;
; Catch frozen channels with the back ground rate = photonrate
  if NumPoint gt 0 and bg_count gt 0 then $
  if avg(PR[pr_index])-avg(bgr[bg_index]) eq 0 then NumPoint=0
;
; Only average scans with background and sufficient number of points
;
  IF NumPoint ge MINPOINTS AND bg_count gt 0 THEN BEGIN
;
;     Average the background data
;
      bgr=bgr[bg_index]
      bgre=bgre[bg_index]
      bgwt=1/bgre^2
      bgt=bgt[bg_index]
      sum_bgwt=total(bgwt)
      scans[kk].BackgndRate(i,j)=total(bgr*bgwt)/sum_bgwt
      scans[kk].BackgndErr(i,j)=1/sqrt(sum_bgwt)
      index_l=where(bgt lt StartTime,count_l)
      index_r=where(bgt gt StopTime,count_r)
;
;     Average photonrate and squared visibilities
;
      wt=1/PRE[pr_index]^2
;     wt(*)=1
      sum_wt=total(wt)
      scans[kk].PhotonRate(i,j)=total(PR[pr_index]*wt)/sum_wt $
			       -scans[kk].BackgndRate(i,j)
      case UncertaintyEstimate of
      'G':begin
;   	  Gaussian error propagation (error of weighted mean)
          scans[kk].PhotonRateErr(i,j)=1/sqrt(sum_wt)
	  end
      'N':begin
;         Normal distribution
	  scans[kk].PhotonRateErr(i,j)=stddev(PR[pr_index])/sqrt(NumPoint)
	  end
      'P':begin
;         Poisson noise theoretical estimate
          scans[kk].PhotonRateErr(i,j)=sqrt(abs(scans[kk].PhotonRate(i,j)) $
		  /(ScanTable[kk].NumCoh*ScanTable[kk].NumIncoh*NumPoint))
	  end
      endcase
;
      bgpoints=fltarr(NumPoint)
      if (count_l gt 0) and (count_r gt 0) then begin
      	x=(findgen(NumPoint)/NumPoint*(StopTime-StartTime)+StartTime)/3600
	coeffs=poly_fit(bgt/3600,bgr,1,measure_errors=bgre)
	bgpoints=poly(x,coeffs)
      endif else begin
      	bgpoints[*]=scans[kk].BackgndRate(i,j)
      endelse
      calfactor=avg(PR[pr_index]-bgpoints)^2    ; Used until 1999-07-09
      calfactors=PR[pr_index]-bgpoints
      index=where(calfactors ne 0,NumPoint)
      if NumPoint ge MINPOINTS then begin
	calfactors=calfactors[index]
	pr_index=pr_index[index]
      endif
;
      for l=0,GenConfig.NumBaseline[i]-1 do begin
;
;	Adjustment for incoherent fluxes
	calfactor=calfactors
	for is=0,genconfig.numsid-1 do $
	  if strpos(genconfig.baselineid[l,i],genconfig.stationid[is]) eq -1 $
	    and strpos(strjoin(genconfig.baselineid[0:genconfig.numbaseline[i]-1,i]), $
		       genconfig.stationid[is]) ge 0 $
	    and scantable[kk].station(is) then $
	    calfactor=calfactor-genconfig.apdflux[is,j,i]*avg(calfactors)
	is1=where(genconfig.stationid eq strmid(genconfig.baselineid[l,i],0,3))
	is2=where(genconfig.stationid eq strmid(genconfig.baselineid[l,i],4,3))
	if genconfig.apdflux[is1,j,i] eq 0 $
	or genconfig.apdflux[is2,j,i] eq 0 then alpha=1.0 $
					   else alpha=genconfig.apdflux[is1,j,i] $
						     /genconfig.apdflux[is2,j,i]
	calfactor=calfactor^2*(4*alpha)/(alpha+1)^2
;
;	NPOI has baselines at different fringe frequencies
	if system_id(SystemId) eq 'NPOI' then $
	normfactor=4.0/sinc(float(GenConfig.FringeMod[l,i])/GenConfig.NumBin)^2
;
;	Apply all corrections, i.e. bias and normalization
;	The bias correction was modified again Sep 26, 2007 because it was not
;	handling large BG rates properly in the scheme where these are zeroed
;	before computation of the bias. The two options are shown below, and
; 	for now we proceed with the one not requiring the zeroing of the BG.
	if NumPoint ge MINPOINTS then begin
          bias=(GenConfig.V2Bias[0,l,j,i,kc] $
              *(scans[kk].PhotonRate(i,j) $
		+scans[kk].BackgndRate(i,j))^(GenConfig.V2Bias[1,l,j,i,kc]))
	  bias=bias[0]*calfactor/normfactor	; use when not zeroing BG before
;	  bias=bias(0)*PR(pr_index)^2/normfactor; zero BG when doing bias fits
	  VSQ=VisSq[i,j,l,Rec0[k]:RecN[k]]-bias & VSQ=VSQ[pr_index]/calfactor
	  VSQE=VisSqErr[i,j,l,Rec0[k]:RecN[k]] & VSQE=VSQE[pr_index]/calfactor
	  index=where(VSQE gt 0,count)
	endif else count=0
;
;	Average the visibilities if we have enough of them
	if count gt MINPOINTS then begin
	  wt=1/VSQE[index]^2
;         wt(*)=1	; I don't know which is better
	  sum_wt=total(wt)
	  scans[kk].VisSq(i,j,l)= $
		(total(VSQ[index]*wt)/sum_wt)*normfactor ;$
;		This code (Nov 29, 2006) removed again Sep 26, 2007
;		-GenConfig.V2Bias(0,l,j,i,kc)*(scans(kk).PhotonRate(i,j) $
;					      +scans(kk).BackgndRate(i,j)*0) $
;						^(GenConfig.V2Bias(1,l,j,i,kc))
	  case system_id(SystemId) of
	  'Mark3' :scans[kk].VisSqErr(i,j,l)= $
			(stddev(VSQ[index])/sqrt(count))*normfactor
	  'NPOI'  :begin
		   case UncertaintyEstimate of
	       'G':begin
;   	           Gaussian error propagation (error of weighted mean)
		   scans[kk].VisSqErr(i,j,l)= $
		   	(1/sqrt(sum_wt))*normfactor
		   end
	       'N':begin
;                  Normal distribution
		   scans[kk].VisSqErr(i,j,l)= $
			(stddev(VSQ[index])/sqrt(NumPoint))*normfactor
		   end
	       'P':begin
;		   Estimate for Poisson noise, photonrate fluctuations neglected
		   scans[kk].VisSqErr(i,j,l)=sqrt( $
			pi_circle^4 $
			/(4*ScanTable[kk].NumCoh*ScanTable[kk].NumIncoh*count) $
			*(1/scans[kk].PhotonRate(i,j)^2 $
			  +(4/pi_circle^2)*abs(scans[kk].VisSq(i,j,l)) $
					      /scans[kk].PhotonRate(i,j)))
	           end
		   endcase
	     	   end
	  endcase
	endif else scans[kk].VisSqErr(i,j,l)=-1
;
;	Initialize E/EC variables
	scans[kk].VisSqE(i,j,l)=scans[kk].VisSq(i,j,l)
	scans[kk].VisSqEErr(i,j,l)=scans[kk].VisSqErr(i,j,l)
	scans[kk].VisSqC(i,j,l)=scans[kk].VisSq(i,j,l)
	scans[kk].VisSqCErr(i,j,l)=scans[kk].VisSqErr(i,j,l)
	scans[kk].VisSqEC(i,j,l)=scans[kk].VisSq(i,j,l)
	scans[kk].VisSqECErr(i,j,l)=scans[kk].VisSqErr(i,j,l)
;
;	Accumulate calibration factors for triples
        for tr=0,GenConfig.NumTriple-1 do begin
 	 for bl=0,2 do begin
   	  if (GenConfig.TripleBeam[bl,tr] eq i) and $
	     (GenConfig.TripleBase[bl,tr] eq l) then begin
           index=where( $
            GenConfig.TripleChan[0:GenConfig.TripleNumChan[tr]-1,bl,tr] eq j,n)
           if n gt 0 then begin
		triple_factor[kk,tr,index]= $
	        triple_factor[kk,tr,index]/(sqrt(avg(calfactor)/normfactor))
		triple_norm_factor[kk,tr,index]= $
		triple_norm_factor[kk,tr,index]*sqrt(normfactor)
	   	if scans[kk].VisSqErr(i,j,l) le 0 then $
		triple_factor[kk,tr,index] = 0
	   endif
	  endif
       	 endfor
	endfor
;
      endfor
;
  ENDIF ELSE BEGIN
;
;     We come here if there weren't enough data
      if bg_count eq 0 then begin
	if dead_channel_index[j] ne 0 then $
        print,'Warning(AVERAGE): no BackgndRate for OB=',i+1, $
  	    ', CH=',j+1,', scan=',kk+1, $
  	    ', Star: ',ScanTable[kk].starid, $
  	    format='(a,i1,a,i2,a,i3,a,a)'
        scans[kk].BackgndErr(i,j)=-1
      endif
      scans[kk].PhotonRateErr(i,j)=-1
      for l=0,GenConfig.NumBaseline[i]-1 do begin
  	scans[kk].VisSqErr(i,j,l)=-1
  	scans[kk].VisSqEErr(i,j,l)=-1
  	scans[kk].VisSqCErr(i,j,l)=-1
  	scans[kk].VisSqECErr(i,j,l)=-1
      endfor
      for t=0,GenConfig.NumTriple-1 do begin
       for l=0,2 do begin
        if (GenConfig.TripleBeam[l,t] eq i) then begin
          index=where( $
	      GenConfig.TripleChan[0:GenConfig.TripleNumChan[t]-1,l,t] eq j,n)
          if n gt 0 then begin
            triple_factor[kk,t,index]=0
	    scans[kk].TripleAmpErr(t,index)=-1
	    scans[kk].TripleAmpEErr(t,index)=-1
	    scans[kk].TripleAmpCErr(t,index)=-1
	    scans[kk].TripleAmpECErr(t,index)=-1
	    scans[kk].TriplePhaseErr(t,index)=-1
	    scans[kk].TriplePhaseCErr(t,index)=-1
          endif
         endif
        endfor
      endfor
;
  ENDELSE
;
endfor
endfor
endfor
;
; Triple products
;
print,'Averaging triple products...'
;
for t=0,GenConfig.NumTriple-1 do begin
for j=0,GenConfig.TripleNumChan[t]-1 do begin
for k=0,n_elements(Iscan)-1 do begin
;
kk=Iscan[k]-1	; subtract 1 since arrays start with 0!
sconf=scanconfig(Iscan[k],/starlist)
kc=where(scanconfig(/starlist) eq sconf) & kc=kc[0]
i_r=where(  TripleAmpErr[t,j,Rec0[k]:RecN[k]] gt 0,count_r)
i_i=where(TriplePhaseErr[t,j,Rec0[k]:RecN[k]] gt 0,count_i)
if count_r ne count_i then begin
  print,'***Error(AVERAGE): inconsistent triple editing!'
  return
endif
if count_r ge MINPOINTS and triple_factor[kk,t,j] ne 0 then begin
  CTR=ComplTriple[t,0,j,Rec0[k]:RecN[k]]
  CTI=ComplTriple[t,1,j,Rec0[k]:RecN[k]]
  CTER=ComplTripleErr[t,0,j,Rec0[k]:RecN[k]]
  CTEI=ComplTripleErr[t,1,j,Rec0[k]:RecN[k]]
  wtr=1/CTER[i_r]^2
  wtr[*]=1	; We are not sure complex triple errors are reliable
  wti=1/CTEI[i_i]^2
  wti[*]=1	; We are not sure complex triple errors are reliable
  sum_wtr=total(wtr)
  sum_wti=total(wti)
  scans[kk].ComplTriple(t,0,j)=total(CTR[i_r]*wtr)/sum_wtr*triple_factor[kk,t,j]
  scans[kk].ComplTriple(t,1,j)=total(CTI[i_i]*wti)/sum_wti*triple_factor[kk,t,j]
  case UncertaintyEstimate of
  'G':begin
;     Gaussian error propagation (error of weighted mean)
      scans[kk].ComplTripleErr(t,0,j)=1/sqrt(sum_wtr)*triple_factor[kk,t,j]
      scans[kk].ComplTripleErr(t,1,j)=1/sqrt(sum_wti)*triple_factor[kk,t,j]
      end
  'N':begin
;     Normal distribution
      scans[kk].ComplTripleErr(t,0,j)= $
	(stddev(CTR[i_r])/sqrt(count_r))*triple_factor[kk,t,j]
      scans[kk].ComplTripleErr(t,1,j)= $
	(stddev(CTI[i_i])/sqrt(count_i))*triple_factor[kk,t,j]
;     Based on D. Buscher's Theorem of the alignment uncertainty ellipse
      xr=complex(CTR[i_r],CTI[i_i]) $
	*exp(-i_complex*cphase(total(complex(CTR[i_r],CTI[i_i]))))
      scans[kk].ComplTripleErr(t,0,j)= $
	(stddev(float(xr))/sqrt(count_r))*triple_factor[kk,t,j]
      scans[kk].ComplTripleErr(t,1,j)= $
	(stddev(imaginary(xr))/sqrt(count_i))*triple_factor[kk,t,j]
      end
  'P':begin
;     Not available
      end
  endcase
;
; Compute amplitude and phase...
  bias=0
  tb=commonbeam(genconfig.triplebeam[*,t])
; if GenConfig.TripleBeam(0,t) eq GenConfig.TripleBeam(1,t) and $
;    GenConfig.TripleBeam(0,t) eq GenConfig.TripleBeam(2,t) then $
  if tb ne -1 then $
     bias=(GenConfig.TABias(0,j,t,kc) $
         *(scans(kk).PhotonRate(tb, $
			        GenConfig.TripleChan(j,0,t)) $
	  +scans(kk).BackgndRate(tb, $
			 	 GenConfig.TripleChan(j,0,t))) $
;	 ^(GenConfig.TABias(1,j,t,kc)+3))/triple_norm_factor(kk,t)
	 ^GenConfig.TABias[1,j,t,kc])
  scans[kk].TripleAmp(t,j)=sqrt(scans[kk].ComplTriple(t,0,j)^2+ $
			        scans[kk].ComplTriple(t,1,j)^2) $
;			  -bias*triple_factor(kk,t,j)
			  -bias
  if (scans[kk].ComplTriple(t,0,j) eq 0) and  $
     (scans[kk].ComplTriple(t,1,j) eq 0) then $
       scans[kk].TriplePhase(t,j)=0 $
  else scans[kk].TriplePhase(t,j)= $
		atan(scans[kk].ComplTriple(t,1,j), $
	     	     scans[kk].ComplTriple(t,0,j))
; ...and their errors
  TAUncertaintyEstimate='G'
  case TAUncertaintyEstimate of
  'G':begin
;     Gaussian error propagation, but correlations ignored
      if scans[kk].TripleAmp(t,j) eq 0 then begin
	scans[kk].TripleAmpErr(t,j)=-1
	scans[kk].TriplePhaseErr(t,j)=-1
      endif else begin
      	scans[kk].TripleAmpErr(t,j)= $
        sqrt((scans[kk].ComplTriple(t,0,j)*scans[kk].ComplTripleErr(t,0,j))^2 $
            +(scans[kk].ComplTriple(t,1,j)*scans[kk].ComplTripleErr(t,1,j))^2) $
       	        /abs(scans[kk].TripleAmp(t,j))
	scans[kk].TriplePhaseErr(t,j)=scans[kk].TripleAmpErr(t,j)/ $
				     scans[kk].TripleAmp(t,j)
      endelse
;     This uses the new ComplTripleErr's based on alignment of the uncertainty
;     ellipse with the triple product
      scans[kk].TripleAmpErr(t,j)=scans[kk].ComplTripleErr(t,0,j)
      scans[kk].TriplePhaseErr(t,j)=atan(scans[kk].ComplTripleErr(t,1,j), $
					 scans[kk].TripleAmp(t,j))
      end
  'N':begin
;     Normal distribution (for high-SNR case)
      scans[kk].TripleAmpErr(t,j)=(stddev(sqrt(CTR[i_r]^2+CTI[i_i]^2))/sqrt(coun
				 *triple_factor[kk,t,j]/sqrt(count_r)
      scans[kk].TriplePhaseErr(t,j)=stddev(atan(CTI[i_i],CTR[i_r]))/sqrt(count_r
      end
  'P':begin
;     Not available
      end
  endcase
;
; Initialize derived variables
  scans[kk].TripleAmpC(t,j)=scans[kk].TripleAmp(t,j)
  scans[kk].TripleAmpCErr(t,j)=scans[kk].TripleAmpErr(t,j)
  scans[kk].TripleAmpE(t,j)=scans[kk].TripleAmp(t,j)
  scans[kk].TripleAmpEErr(t,j)=scans[kk].TripleAmpErr(t,j)
  scans[kk].TripleAmpEC(t,j)=scans[kk].TripleAmp(t,j)
  scans[kk].TripleAmpECErr(t,j)=scans[kk].TripleAmpErr(t,j)
  scans[kk].TriplePhaseC(t,j)=scans[kk].TriplePhase(t,j)
  scans[kk].TriplePhaseCErr(t,j)=scans[kk].TriplePhaseErr(t,j)
endif else begin
  scans[kk].ComplTripleErr(t,0,j)=-1
  scans[kk].ComplTripleErr(t,1,j)=-1
  scans[kk].TripleAmpErr(t,j)=-1
  scans[kk].TripleAmpCErr(t,j)=-1
  scans[kk].TripleAmpEErr(t,j)=-1
  scans[kk].TripleAmpECErr(t,j)=-1
  scans[kk].TriplePhaseErr(t,j)=-1
  scans[kk].TriplePhaseCErr(t,j)=-1
endelse
;
endfor
endfor
endfor
;
; Time, delays, NAT data
;
print,'Averaging delays...'
;
for k=0,n_elements(Iscan)-1 do begin
;
; Time (change to median in Version 8)
	kk=Iscan[k]-1
	t=PointTime[Rec0[k]:RecN[k]]
	index=where(t ge 0,count)
;	if count gt 0 then scans(kk).Time=avg(t(index)) $
;		      else scans(kk).Time=-avg(abs(t))
	if count gt 0 then scans[kk].Time=median(t[index]) $
		      else scans[kk].Time=-avg(abs(t))
;
; DelayJitter
for i=0,GenConfig.NumOutBeam-1 do begin
for l=0,GenConfig.NumBaseline[i]-1 do begin
	v=DelayJitter[i,l,Rec0[k]:RecN[k]]
	ve=DelayJitterErr[i,l,Rec0[k]:RecN[k]]
    	index=where(ve gt 0,count)
    	if count ge MINPOINTS then begin
      		scans[kk].DelayJitter(i,l)=median(v[index])
      		scans[kk].DelayJitterErr(i,l)=1e-6
    	endif else scans[kk].DelayJitterErr(i,l)=-1e-6
endfor
endfor
;
; NATJitter, delay positions
for i=0,GenConfig.NumSid-1 do begin
;
	v=NATJitter[i,Rec0[k]:RecN[k]]
	ve=NATJitterErr[i,Rec0[k]:RecN[k]]
	index=where(ve gt 0,count)
	if count ge MINPOINTS then begin
		scans[kk].NATJitter(i)=median(v[index])
		scans[kk].NATJitterErr(i)=0.01
	endif else scans[kk].NATJitterErr(i)=-1
;
	v=NATCounts[i,Rec0[k]:RecN[k]]
	ve=NATCountsErr[i,Rec0[k]:RecN[k]]
	index=where(ve gt 0,count)
	if count ge MINPOINTS then begin
		scans[kk].NATCounts(i)=avg(v[index])
		scans[kk].NATCountsErr(i)=stddev(v[index])/sqrt(count)
	endif else scans[kk].NATCountsErr(i)=-1
;
	v=FDLPos[i,Rec0[k]:RecN[k]]
	ve=FDLPosErr[i,Rec0[k]:RecN[k]]
	index=where(ve gt 0,count)
	if count ge MINPOINTS then begin
		coeffs=polyfit(t[index]/3600,v[index],2,vft)
		scans[kk].FDLPos(i)=polynom(scans[kk].Time/3600,coeffs)
		scans[kk].FDLPosErr(i)=stddev(v[index]-vft)/sqrt(count)
	endif else begin
		scans[kk].FDLPos(i)=0
		scans[kk].FDLPosErr(i)=-1e-6
	endelse
;
	IF n_elements(GrpDelay) gt 1 and n_elements(MetroPos) gt 1 then BEGIN
	IF n_elements(GrpDelay[0,*]) ne n_elements(PointTime) $
	or n_elements(MetroPos[0,*]) ne n_elements(PointTime) then BEGIN
		if i eq 0 then $
		print,'Warning(AVERAGE): incompatible delay corrections!'
	ENDIF ELSE BEGIN
	d=FDLPos[i,Rec0[k]:RecN[k]]*(i ne GenConfig.RefStation-1) $
	 -GrpDelay[i,Rec0[k]:RecN[k]]*1d-6 $
	 -MetroPos[i,Rec0[k]:RecN[k]]*1d-6*(i ne GenConfig.RefStation-1)
	ferr=FDLPosErr[i,Rec0[k]:RecN[k]]
	gerr=GrpDelayErr[i,Rec0[k]:RecN[k]]
	merr=MetroPosErr[i,Rec0[k]:RecN[k]]
	index=where(ferr gt 0 and gerr gt 0 and merr gt 0,count)
	if count ge MINPOINTS then begin
		coeffs=polyfit(t[index]/3600,d[index],2,dft)
		scans[kk].GrpDelay(i)=polynom(scans[kk].Time/3600,coeffs)
		scans[kk].GrpDelayErr(i)=stddev(d[index]-dft)/sqrt(count)
		if i eq GenConfig.RefStation-1 then $
			scans[kk].GrpDelayErr(i)=1e-6
	endif else begin
		scans[kk].GrpDelay(i)=avg(d)
		scans[kk].GrpDelayErr(i)=-1e-6
	endelse
;
	d=FDLPos[i,Rec0[k]:RecN[k]]*(i ne GenConfig.RefStation-1) $
	 -DryDelay[i,Rec0[k]:RecN[k]]*1d-6 $
	 -MetroPos[i,Rec0[k]:RecN[k]]*1d-6*(i ne GenConfig.RefStation-1)
	derr=DryDelayErr[i,Rec0[k]:RecN[k]]
	index=where(ferr gt 0 and derr gt 0 and merr gt 0,count)
	if count ge MINPOINTS then begin
		coeffs=polyfit(t[index]/3600,d[index],2,dft)
		scans[kk].DryDelay(i)=polynom(scans[kk].Time/3600,coeffs)
		scans[kk].DryDelayErr(i)=stddev(d[index]-dft)/sqrt(count)
		if i eq GenConfig.RefStation-1 then $
			scans[kk].DryDelayErr(i)=1e-6
	endif else begin
		scans[kk].DryDelay(i)=avg(d)
		scans[kk].DryDelayErr(i)=-1e-6
	endelse
;
	d=FDLPos[i,Rec0[k]:RecN[k]]*(i ne GenConfig.RefStation-1) $
	 -WetDelay[i,Rec0[k]:RecN[k]]*1d-6 $
	 -MetroPos[i,Rec0[k]:RecN[k]]*1d-6*(i ne GenConfig.RefStation-1)
	werr=WetDelayErr[i,Rec0[k]:RecN[k]]
	index=where(ferr gt 0 and werr gt 0 and merr gt 0,count)
	if count ge MINPOINTS then begin
		coeffs=polyfit(t[index]/3600,d[index],2,dft)
		scans[kk].WetDelay(i)=poly(scans[kk].Time/3600,coeffs)
		scans[kk].WetDelayErr(i)=stddev(d[index]-dft)/sqrt(count)
		if i eq GenConfig.RefStation-1 then $
			scans[kk].WetDelayErr(i)=1e-6
	endif else begin
		scans[kk].WetDelay(i)=avg(d)
		scans[kk].WetDelayErr(i)=-1e-6
	endelse
	ENDELSE
;
	ENDIF
;
endfor
;
endfor
;
; Average inch data if present
if n_elements(Sol0) ne 0 and n_elements(SolN) ne 0 then inchav
;
calcastrom
calcviscal
;
print,'Finished averaging.'
;
end
