function blackbody,teff,lambda,fl_exp,jansky=jansky
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Return (specific) intensity B of a blackbody in units of mW/m2/sr/nm.
; (These units are the ones used by Van Hamme in his limb darkening tables.)
; Optionally, return in units of Jansky. fl_exp (output) is an exponent
; of the tilt of the spectrum at each wavelength: flux ~ (l/l_0)^-fl_exp.
; (see Corporaal et al. 2021, A&A 650, L12
;
; B*d_lambda*dA*cos(theta)*d_Omega is the amount of energy per unit time
; of radiation having a wavelength between lambda and lambda+d_lambda emitted
; by a projected surface area dA*cos(theta) into the solid angle d_Omega.
;
; Required input:
; teff [K] 	(scalar)
; lambda [m]	(vector)
;
if n_elements(jansky) eq 0 then jansky=0
;
teff=double(teff)
lambda=double(lambda)
;
unity_v=dblarr(n_elements(teff))+1
lambda_a=lambda#unity_v
flux=(1.191d-22/lambda_a^5)/(exp(1.439d-2/(lambda#teff))-1)
if jansky then flux=(flux/10e3)*(lambda_a*1e6)^2/3e-16
;
; Compute flux exponent k in flux_i/flux_0=(lambda_i/lambda_0)^k
lambda_l=lambda_a-1d-9	; 1 nm step, should be OK for all purposes
lambda_h=lambda_a+1d-9
flux_l=(1.191d-22/lambda_l^5)/(exp(1.439d-2/(lambda_l#teff))-1)
flux_h=(1.191d-22/lambda_h^5)/(exp(1.439d-2/(lambda_h#teff))-1)
fl_exp=alog(flux_h/flux_l)/alog(lambda_h/lambda_l)
fl_exp=alog10(flux_h/flux_l)/alog10(lambda_h/lambda_l)
;
if n_elements(flux) eq 1 then return,flux[0] else return,reform(flux)
;
; Old
return,(3.742d-30/lambda^5)/(exp(1.439d-2/(lambda*teff))-1)
;
end
