pro calcmodel,pearlmv=pearlmv,pearlcv=pearlcv,quiet=quiet,noload=noload
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Main model data computation function for interferometric, spectroscopic,
; astrometric, and photometric data. The common block variable ds_options
; specifies by weight which data sets are requested.
;
; Integration over time to account for fringe smearing has been included
; as of version 8.
;
; Option pearlmv: pass model visibility (mv) to Pearl for self-calibration
; Option pearlcv: pass mv and model flux to Pearl to subtract from cv
; These two options cannot be used together!
; The option pearlcv is useful for the analysis of triple stars.
;
; Keyword noload to just compute model for currently loaded data.
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,p_magnitudes
common Tables,scantable,bgtable,stationtable
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AuxData,parallaxes,k1,k2,vsini
common RandomModel,RandomSeed
common LocalRocheVis,LDinit
;
RAD=180/pi_circle
RandomSeed=0
LDinit=1
;
; http://www.idlcoyote.com/code_tips/underflow.html
; Disable reporting of underflow math errors
currentExcept = !Except
!Except = 0
void = Check_Math()
;
if n_elements(ds_options) eq 0 then begin
	print,'***Error(CALCMODEL): no data weights specified!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(CALCMODEL): no model!'
	return
endif
if n_elements(fit_options) eq 0 then fit_options=alloc_fit_options()
if n_elements(quiet) eq 0 then quiet=0
if n_elements(noload) eq 0 then noload=0
;
; Section for visibility data...................................................
;
if ds_options.i ne 0 then begin
;
if n_elements(GenInfo) eq 0 then begin
	print,'***Error(CALCMODEL): no data (GENINFO undefined)!'
	return
endif
;
if n_elements(GeoInfo) eq 0 then begin
	print,'***Error(CALCMODEL): no data (GEOINFO undefined)!'
	return
endif
;
; The model is always computed for data of all nights
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
; Save statistics of floating calibration factors
if fit_options.f then begin
	if not quiet then begin
	openw,unit,'calfactors.txt',/get_lun
	print,'Warning: floating calibration option selected!'
	print,'Use calfactorplot to see histograms of factors.'
	endif
;
	floating_notified=0
endif
;
; Do one night at a time
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
; Load a night
night=where(GenInfo.date eq nights[n] $
        and GeoInfo.systemid eq arrays[n] $
        and GenInfo.configid eq confgs[n],count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' $
	and not noload $
	and n_elements(bufferinfo) gt 1 then $
	loadnight,GenInfo[night[0]].date, $
		GeoInfo[night[0]].systemid, $
		GenInfo[night[0]].configid
; Save copy of uv-coordinates and compute offsets
uvw=scans.uvw
calcastrom,/quiet
uvw_off=uvw-scans.uvw
;
; Select data for model star
index0=where(scans.starid eq gen_model.starid and scantable.code eq 1,NS0)
if NS0 eq 0 then begin
; print,'***Error(CALCMODEL): no data selected for ',gen_model.starid,' on ', $
; 	nights(n)+' '+confgs(n),'!'
goto,NEXT
endif
;
time=abs(scans[index0].time)
r0=median(reform(scans[index0].r0,NS0))
am=median(1./cos(reform(scans[index0].za,NS0)/RAD))
int_time=median(reform(scans[index0].int_time,NS0))
; int_time=median(scantable(index0).stoptime-scantable(index0).starttime)
if not quiet and !int_time ne 0 then $
	print,'Median integration time [s] for '+Date+': ',int_time
;
; Baseline visibilities
;
for ob=0,GenConfig.NumOutBeam-1 do begin
;
; For multiple stellar systems, compute positions and apodize fluxes
if num_binary()*0 ge 1 then begin	; Obsolete
	magnitudes=star_model.magnitudes
	parsedate,date,y,m,d
	xy=reform(modelpos(julian(y,m,d)-2440000.d0))
	radius=sqrt(total(xy^2,2))
	rad2mas=180l*3600l*1000l/!pi
; 	Compute Airy's disk
	lamda=mean(genconfig.wavelength[0:genconfig.numspecchan[ob]-1,ob])
	airy=(lamda/genconfig.diameter[0])*rad2mas
	dm=-2.5*alog10(exp(-(2*radius/airy)^2))
	for i=0,n_elements(star_model)-1 do $
	star_model[i].magnitudes(*)=star_model[i].magnitudes(*)+dm[i]
endif
if num_binary() ge 1 then begin
;	Check for information on variable component magnitudes
;	It is assumed that the filter sequence matches wavelengths!
	if ds_options.pm then begin
		parsedate,date,y,m,d
		jd=julian(y,m,d)-2440000.d0
		wl=gen_model.wavelengths*1000	; [nm]
		index=where(abs(jd-p_magnitudes.jd) eq 0)
		if index[0] ne -1 then begin
		p_mags=p_magnitudes[index].value
		p_comp=p_magnitudes[index].component
		p_filt=p_magnitudes[index].filter
		for k=0,n_elements(star_model)-1 do begin
			index=where(p_comp eq star_model[k].component)
			if index[0] ge 0 then star_model[k].magnitudes=p_mags
		endfor
		endif
	endif
	magnitudes=star_model.magnitudes
	dm=apodize_m(r0,am)
;	Do not change the absolute magnitude of a cached image component
	index=where(star_model.type eq 12,count) & index=index[0]
	if count eq 1 then dm=dm-dm[index]
	index=where(star_model.type ne 0)
	star_model[index].magnitudes=star_model[index].magnitudes+dm[*,index]
endif
;
; Set up grid of wavelengths for band pass integrations
; Note that all functions in filter.pro use [nm] as units for lambda
lambda_grid=system_config(SystemId,'GRID')
if string(lambda_grid[0]) eq 'SPECTROMETER' then begin
	grid_type='SPECTROMETER'
	lambda_grid=genconfig.wavelength[0:genconfig.numspecchan[ob]-1,ob]*1e9
endif else begin
	grid_type='GRID'
	ch=indgen(genconfig.numspecchan[ob])
	jndex=where(lambda_grid*1e-9 le max(genconfig.wavelength[ch,ob] $
					   +genconfig.chanwidth[ch,ob]/2) $
		and lambda_grid*1e-9 ge min(genconfig.wavelength[ch,ob] $
					   -genconfig.chanwidth[ch,ob]/2),count)
;	The next statement was commented between 2012 and 2013, why?
	if count gt 0 then lambda_grid=lambda_grid[jndex]
endelse
lamda=lambda_grid/1d9	; convert to SI units
n_lam=n_elements(lamda)
nu=1/lamda
;
nc=GenConfig.NumSpecChan[ob]
vis=reform(complexarr(nc,NS0),nc,NS0)
pwr=reform(complexarr(nc,NS0),nc,NS0)
;
for bl=0,GenConfig.NumBaseline[ob]-1 do begin
;	Only compute model for scans which have data in at least one channel
;	Example: ups-Sgr-MIRC-2016Jul16.oifits
;	scans = Array[1, 8, 15, 48]: 48 scans, 15 baselines, 8 channels, 1 OB
	e0=reform(scans[*].vissqcerr(ob,0:GenConfig.NumSpecChan[ob]-1,bl) > 0)
	if GenConfig.NumSpecChan[ob] gt 1 then e=total(e0,1) else e=e0
	index=where(scans.starid eq gen_model.starid $
		and scantable.code eq 1 $
		and e gt 0,NS)
	if NS gt 0 then begin
	time=abs(scans[index].time)
	int_time=scans[index].int_time
	vis=reform(complexarr(nc,NS),nc,NS)
	pwr=reform(complexarr(nc,NS),nc,NS)
;
;	Set up integral over int_time
	time=scans[index].time	; Make backup of time stamps
	if !int_time ne 0 then num_int=max(nint(int_time/!int_time)) $
			  else num_int=0
	if num_int mod 2 eq 0 then num_int=num_int+1
	time_step=0
	if num_int gt 1 then $
	time_step=int_time/(num_int-1)	; array, [s]
	vis_grid=dcomplexarr(n_lam,NS)
;	Only compute model for wavelengths with data for SPECTROMETER
	if grid_type eq 'SPECTROMETER' then begin
		if n_lam gt 1 then begin
		if n_elements(scans) eq 1 then ee=e0 $ ; option added 2020
					  else ee=total(reform(e0),2)
		endif else ee=e0
		ei=where(ee gt 0,n_lam_ei)
	endif
	for i=-num_int/2,num_int/2 do begin
		scans[index].time=time+i*time_step[index]
		calcastrom,/quiet
		scans.uvw=scans.uvw+uvw_off
		um=[scans[index].uvw(ob,0,bl,0)*GenConfig.Wavelength[0,ob]]
		vm=[scans[index].uvw(ob,0,bl,1)*GenConfig.Wavelength[0,ob]]
		t=abs(scans[index].time)
		if grid_type eq 'SPECTROMETER' then begin
			u=nu[ei]#um
			v=nu[ei]#vm
			vis_grid[ei,*]=vis_grid[ei,*] $
			+reform(modelvis(t,lamda[ei],fluxes,u,v),n_lam_ei,NS)
		endif else begin
			u=nu#um
			v=nu#vm
			vis_grid=vis_grid $
				+reform(modelvis(t,lamda,fluxes,u,v),n_lam,NS)
		endelse
	endfor
	vis_grid=vis_grid/num_int
	scans[index].time=time
;	calcastrom,/quiet; commented Jan 4, 2020, added the next line
	scans.uvw=uvw
;
;	Bandpass integration
	if grid_type eq 'SPECTROMETER' then begin
;		High R spectrometer
		for sc=0,NS-1 do begin
;			vis(*,sc)=vis_grid(*,sc)/fluxes
;			scans(index(sc)).photometrym(0,ob,0:nc-1)=fluxes
			vis[ei,sc]=vis_grid[ei,sc]/fluxes
			scans[index[sc]].photometrym(0,ob,ei)=fluxes
		endfor
	endif else begin
		for ch=0,nc-1 do begin
			f=system_config(SystemId,'FILTERS')
			if n_elements(f) gt 1 then filter=f[ch,ob] $
					      else filter=f[0]
			if filter[0] eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch,ob) $
						 else $
				tm=call_function(filter,lambda_grid)
			if NS eq 1 then begin
				tm=reform(tm,1,n_lam)
				lam=reform(lamda,1,n_lam)
			endif else lam=lamda
			if total(tm) eq 0 then begin
				print,'***Error(CALCMODEL): wl grid too coarse!'
				stop
				return
			endif
;			ABCD fringe detection
			vis[ch,*]=(tm#vis_grid)/total(tm*fluxes)
;			Scanning interferometers
			pwr_grid=reform(abs(vis_grid)^2,n_lam,NS)
;			pwr(ch,*)=(tm^2#pwr_grid)/total(tm^2*fluxes^2)
			pwr[ch,*]=((lam^2*tm^2)#pwr_grid) $
				 /total(tm^2*fluxes^2*lam^2)
;			Store model SED
			scans[index].photometrym(0,ob,ch)=total(tm*fluxes) $
							 /total(tm)
		endfor
	endelse
	dimensions= $
	size(scans[index].complexvis(ob,0:GenConfig.NumSpecChan[ob]-1,bl))
	scans[index].complexvis(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(vis,dimensions[1:dimensions[0]])
;	Added 2016-06-06 for testing
	scans[index].complexweight(ob,0:GenConfig.NumSpecChan[ob]-1,bl)=1
;
	dimensions= $
	size(scans[index].vissqm(ob,0:GenConfig.NumSpecChan[ob]-1,bl))
;	Use amplitude (default) for ABCD method (e.g. NPOI)
	scans[index].vissqm(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(abs(vis)^2,dimensions[1:dimensions[0]])
	scans[index].visphasem(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(cphase(vis),dimensions[1:dimensions[0]])
	scans[index].diffphasem(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(cphase(vis),dimensions[1:dimensions[0]]) $
			-median(cphase(vis))
;	Use power for scanning interferometers (e.g. VINCI, PIONIER)
	if strpos(instrument_id(systemid),'VINCI') ge 0 or $
	   strpos(instrument_id(systemid),'PIONIER') ge 0 then $
	scans[index].vissqm(ob,0:GenConfig.NumSpecChan[ob]-1,bl)= $
		reform(pwr,dimensions[1:dimensions[0]])
;	Custom option for floating calibration
	if fit_options.f then begin
	for sc=0,n_elements(index)-1 do begin
		v=scans[index[sc]].vissqc(ob,0:nc-1,bl)
		e=scans[index[sc]].vissqcerr(ob,0:nc-1,bl)
		m=scans[index[sc]].vissqm(ob,0:nc-1,bl)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 then begin
;			nf=160	; Should be same as in triple section
;			f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
; 			Define range 1/2 to 2 in steps of 0.01 for cal. factor
			nf=150	; Should be same as in triple section
			f=findgen(nf+1)/100+0.5 & r=f
			nf=125	; Should be same as in triple section
			f=findgen(nf+1)*0.03+0.25 & r=f
			if not floating_notified and not quiet then begin
				print,'Floating cal. limits: ',min(f),max(f)
				floating_notified=1
			endif
			for i=0,nf do $
			r[i]=total(((v[eindex]*f[i]-m[eindex])/(e[eindex]))^2)
			i=where(r eq min(r)) & i=i[0]
			b=genconfig.baselineid[bl,ob]
			if i gt 0 and i lt nf-1 then begin ; Don't hit limits
				f[i]=sqrt(f[i])	; don't apply full scaling
				scans[index[sc]].vissqc(ob,0:nc-1,bl)= $
				scans[index[sc]].vissqc(ob,0:nc-1,bl)*f[i]
;				scans(index(sc)).vissqcerr(ob,0:nc-1,bl)= $
;				scans(index(sc)).vissqcerr(ob,0:nc-1,bl)*f(i)
;				print,date+' '+b,f(i)
				if not quiet then printf,unit,'2 ',f[i]
			endif else begin
				print,date+' '+b+' scan '+string(index[sc]+1, $
					format='(i3)') $
					+': limit hit for floating V2!'
				scans[index[sc]].vissqc(ob,0:nc-1,bl)= $
				scans[index[sc]].vissqc(ob,0:nc-1,bl)*f[i]
			endelse
		endif
	endfor
	endif
	endif	; NS > 0
endfor
; Reset magnitudes outside BL loop
if num_binary() ge 1 then star_model.magnitudes=magnitudes
endfor
;
if keyword_set(pearlmv) or keyword_set(pearlcv) then begin
; 	The following (modified) code is taken from pearldata (pearl.pro)
; 	Pass complex visibilities and model SED to Pearl
	mv=scans[index0].complexvis
	ow=scans[index0].complexweight
; 	Not all scans have photometry computed
	ti=3
	if n_elements(size(scans[index0].photometrym(0,0,*),/dim)) eq 2 $
	then ti=2
	fi=where(total(scans[index0].photometrym(0,0,*),ti) gt 0)
; 	Derive weight array from ow
	vw=total(ow,1)
	r=size(vw,/dim) & r=[1,r]
	if n_elements(index0) eq 1 then r=[r,1]
	vw=reform(vw,r)
;
	spectrometers=strarr(genconfig.numoutbeam)
	w=genconfig.wavelength	; save wavelength info
	if system_id(systemid) eq 'NPOI' then begin
; 		Assume same channel layout for NPOI!
		mean_w=total(w,2)/genconfig.numoutbeam
		for i=0,genconfig.numoutbeam-1 do $
			genconfig.wavelength[*,i]=mean_w
	endif
	for i=0,genconfig.numoutbeam-1 do $
		spectrometers[i]=strjoin(string( $
		genconfig.wavelength[0:genconfig.numspecchan[i]-1,i]*1e6))
	uspms=unique(spectrometers)
	for i=0L,n_elements(uspms)-1 do begin
	  k=where(spectrometers eq uspms[i])
	  fl0=reform( $
	  scans[index0[fi[0]]].photometrym(0,k[0],0:genconfig.numspecchan[i]-1))
;	  Code for removal of bad channels as in pearldata
	  e=reform(vw[k[0],0:GenConfig.NumSpecChan[k[0]]-1, $
			   0:max(GenConfig.NumBaseline)-1,*])
	  r=size(e,/dim)
	  if GenConfig.NumSpecChan[k[0]] eq 1 then begin
		r=[1,r]
		e=reform(e,r)
	  endif
;	  if n_elements(r) eq 1 then r=[r,1]
	  if NS0 eq 1 then r=[r,1]
	  jndex=where(total(reform(e, $
		GenConfig.NumSpecChan[k[0]], $
		produkt(r[1:n_elements(r)-1])),2) ne 0)
	  if jndex[0] ne -1 then begin
		if i eq 0 then fl=fl0[jndex] else fl=[fl,fl0[jndex]]
	  endif
	endfor
	genconfig.wavelength=w
;
	index=where(ow gt 0,count)
	if count gt 0 then begin
	   mv=mv[index]
	   if n eq 0 then mv_all=mv else mv_all=[mv_all,mv]
	   if n eq 0 then fl_all=fl else fl_all=[fl_all,fl]
	   if n eq n_elements(nights)-1 then begin
		if keyword_set(pearlmv) then pearlmodel,mv_all,/mv
		if keyword_set(pearlcv) then pearlmodel,mv_all,fl_all,/cv
	   endif
	endif
endif
;
; Triple visibilities
;
time=abs(scans[index0].time)
;
for tr=0,GenConfig.NumTriple-1 do begin
	nc=GenConfig.TripleNumChan[tr]
	triple=strjoin(unique(nameparse(strjoin(genconfig.baselineid[ $
		genconfig.triplebase[*,tr], $
		genconfig.triplebeam[*,tr]],'-'),'-')),'-')
;
; 	For multiple stellar systems, compute positions and apodize fluxes
	if num_binary()*0 ge 1 then begin	; obsolete
		parsedate,date,y,m,d
		xy=reform(modelpos(julian(y,m,d)-2440000.d0))
		radius=sqrt(total(xy^2,2))
		rad2mas=180l*3600l*1000l/!pi
; 		Compute Airy's disk (assume same wavelength for all baselines)
		lamda=mean(genconfig.wavelength[genconfig.triplechan[*,0,tr], $
						 genconfig.triplebeam[0,tr]])
		airy=(lamda/genconfig.diameter[0])*rad2mas
		dm=-2.5*alog10(exp(-(2*radius/airy)^2))
		magnitudes=star_model.magnitudes
		for i=0,n_elements(star_model)-1 do $
		star_model[i].magnitudes(*)=star_model[i].magnitudes(*)+dm[i]
	endif
	if num_binary() ge 1 then begin
		magnitudes=star_model.magnitudes
		dm=apodize_m(r0,am)
		index=where(star_model.type ne 0)
		star_model[index].magnitudes=star_model[index].magnitudes $
					    +dm[*,index]
	endif
;
;	Do not compute model for channels without data
;	Example: ups-Sgr-MIRC-2016Jul16.oifits
;	triplephase = Array[20, 8, 48]: 48 scans, 20 triples, 8 channels
	e0=reform(scans[*].triplephasecerr(tr,0:GenConfig.TripleNumChan[tr]-1) > 0)
	if GenConfig.TripleNumChan[tr] gt 1 then e=total(e0,1) else e=e0
	index=where(scans.starid eq gen_model.starid $
		and scantable.code eq 1 $
		and e gt 0,NS)
	if NS gt 0 then begin
	time=abs(scans[index].time)
	int_time=scans[index].int_time
 	vis=reform(complexarr(3,nc,NS),3,nc,NS)
;	Compute the BaseFactor, which is -1 if a baseline is flipped
;	The sum of the atmospheric phase over all baselines must be zero
	fBaseMatrix=intarr(GenConfig.NumSid,3)
	fBaseFactor=fltarr(3)+1
	for l=0,2 do begin
		ob=GenConfig.TripleBeam[l,tr]
		bl=GenConfig.TripleBase[l,tr]
		j1=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId[bl,ob],0,3))
		j2=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId[bl,ob],4,3))
		fBaseMatrix[j1,l]=+1
		fBaseMatrix[j2,l]=-1
		if l gt 0 then $
			fBaseFactor[l]=-total(fBaseMatrix[*,0]*fBaseMatrix[*,l])
	endfor
	for l=0,2 do begin
		ob=GenConfig.TripleBeam[l,tr]
		bl=GenConfig.TripleBase[l,tr]
		ch=GenConfig.TripleChan[0:GenConfig.TripleNumChan[tr]-1,l,tr]
		lambda_grid=system_config(SystemId,'GRID')
		if string(lambda_grid[0]) eq 'SPECTROMETER' then begin
			grid_type='SPECTROMETER'
			lambda_grid=genconfig.wavelength[ch,ob]*1e9
		endif else begin
			grid_type='GRID'
			jndex=where(lambda_grid*1e-9 le $
				max(genconfig.wavelength[ch,ob] $
				   +genconfig.chanwidth[ch,ob]/2) $
				and lambda_grid*1e-9 ge $
				min(genconfig.wavelength[ch,ob] $
				   -genconfig.chanwidth[ch,ob]/2),count)
			if count gt 0 then lambda_grid=lambda_grid[jndex]
		endelse
		lamda=lambda_grid/1d9
		n_lam=n_elements(lamda)
		nu=1/lamda
;
;		Set up integral over int_time
		time=scans[index].time  ; Make backup of time stamps
		if !int_time ne 0 then num_int=max(nint(int_time/!int_time)) $
				  else num_int=0
		if num_int mod 2 eq 0 then num_int=num_int+1
		time_step=0
		if num_int gt 1 then $
		time_step=int_time/(num_int-1)	; array, [s]
		vis_grid=dcomplexarr(n_lam,NS)
;		Only compute model for wavelengths with data for SPECTROMETER
		if grid_type eq 'SPECTROMETER' then begin
			if n_lam gt 1 then begin
			if n_elements(scans) eq 1 then ee=e0 $	; added 2020
						  else ee=total(e0,2)
			endif else ee=e0
			ei=where(ee gt 0,n_lam_ei)
		endif
		for i=-num_int/2,num_int/2 do begin
		  scans[index].time=time+i*time_step[index]
		  calcastrom,/quiet
		  scans.uvw=scans.uvw+uvw_off
		  um=[scans[index].uvw(ob,0,bl,0)*GenConfig.Wavelength[0,ob]]
		  vm=[scans[index].uvw(ob,0,bl,1)*GenConfig.Wavelength[0,ob]]
		  t=abs(scans[index].time)
		  if grid_type eq 'SPECTROMETER' then begin
			u=nu[ei]#um
			v=nu[ei]#vm
			vis_grid[ei,*]=vis_grid[ei,*] $
		 	  +reform(modelvis(t,lamda[ei],fluxes,u,v),n_lam_ei,NS)
		  endif else begin
			u=nu#um
			v=nu#vm
			vis_grid=vis_grid $
		 	  +reform(modelvis(t,lamda,fluxes,u,v),n_lam,NS)
		  endelse
		endfor
		vis_grid=vis_grid/num_int
		scans[index].time=time
;		calcastrom,/quiet; commented Jan 4, 2020, added the next line
		scans.uvw=uvw
;		Bandpass integration
		if grid_type eq 'SPECTROMETER' then begin
;			High R spectrometer
			for sc=0,NS-1 do begin
;				vis(l,*,sc)=vis_grid(*,sc)/fluxes
				vis[l,ei,sc]=vis_grid[ei,sc]/fluxes
			endfor
		endif else begin
		for j=0,nc-1 do begin
			f=system_config(SystemId,'FILTERS')
			if n_elements(f) gt 1 then filter=f[ch[j],ob] $
					      else filter=f[0]
			if filter eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch[j],ob) $
						 else $
				tm=call_function(filter,lambda_grid)
			if NS eq 1 then tm=reform(tm,1,n_lam)
			if total(tm) eq 0 then begin
				print,'***Error(CALCMODEL): wl grid too coarse!'
				return
			endif
			vis[l,j,*]=(tm#vis_grid)/total(tm*fluxes)
		endfor
		endelse
		if fBaseFactor[l] lt 0 then $
			vis[l,*,*]=conj(vis[l,*,*])
	endfor
;
;	Form complex triple products
	if grid_type eq 'SPECTROMETER' then begin
;
	result=reform(float(vis[0,ei,*]*vis[1,ei,*]*vis[2,ei,*]),1,1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriplem(tr,0,ei)=result
	result=reform(imaginary(vis[0,ei,*]*vis[1,ei,*]*vis[2,ei,*]),1,1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriplem(tr,1,ei)=result
;
;	Compute amp and phase
	result=reform(sqrt(scans[index].compltriplem(tr,0,ei)^2 $
		     +scans[index].compltriplem(tr,1,ei)^2),1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans[index].tripleampm(tr,ei)=result
	result=reform(atan(scans[index].compltriplem(tr,1,ei), $
		      scans[index].compltriplem(tr,0,ei)),1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans[index].triplephasem(tr,ei)= result
;
	endif else begin
;
	result=reform(float(vis[0,*,*]*vis[1,*,*]*vis[2,*,*]),1,1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriplem(tr,0,0:nc-1)=result
	result=reform(imaginary(vis[0,*,*]*vis[1,*,*]*vis[2,*,*]),1,1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].compltriplem(tr,1,0:nc-1)=result
;
;	Compute amp and phase
	result=reform(sqrt(scans[index].compltriplem(tr,0,0:nc-1)^2 $
		     +scans[index].compltriplem(tr,1,0:nc-1)^2),1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].tripleampm(tr,0:nc-1)=result
	result=reform(atan(scans[index].compltriplem(tr,1,0:nc-1), $
		      scans[index].compltriplem(tr,0,0:nc-1)),1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans[index].triplephasem(tr,0:nc-1)= result
;
	endelse
;
; 	If data set does not have valid triple amplitudes, use model values
	kndex=where(scans.tripleampcerr gt 0,count)
;	Was...:
;	if count eq 0 or not ds_options.ta then $
;	Now...:
	if count eq 0 then $
		scans[kndex].tripleampc=scans[kndex].tripleampm
;
;	Custom option for floating calibration
	if fit_options.f and ds_options.ta then begin
	for sc=0,n_elements(index)-1 do begin
		v=scans[index[sc]].tripleampc(tr,0:nc-1)
		k=where(v lt 0,n_neg)
		e=scans[index[sc]].tripleampcerr(tr,0:nc-1)
		m=scans[index[sc]].tripleampm(tr,0:nc-1)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 and n_neg eq 0 then begin
;			nf=160
;			f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
			nf=150	; Should be same as in baseline section
			f=findgen(nf+1)/100+0.5 & r=f
			nf=125	; Should be same as in triple section
			f=findgen(nf+1)*0.03+0.25 & r=f
			for i=0,nf do $
			r[i]=total(((v[eindex]*f[i]-m[eindex])/(e[eindex]))^2)
			i=where(r eq min(r)) & i=i[0]
			if i gt 0 and i lt nf-1 then begin ; Don't hit limits
				scans[index[sc]].tripleampc(tr,0:nc-1)= $
				scans[index[sc]].tripleampc(tr,0:nc-1)*f[i]
;				scans(index(sc)).tripleampcerr(tr,0:nc-1)= $
;				scans(index(sc)).tripleampcerr(tr,0:nc-1)*f(i)
				if not quiet then printf,unit,'3 ',f[i]
			endif else begin
				print,date+' '+triple+' scan ' $
					+string(index[sc]+1,format='(i3)') $
					+' : limit hit for floating TA!'
				scans[index[sc]].tripleampc(tr,0:nc-1)= $
				scans[index[sc]].tripleampc(tr,0:nc-1)*f[i]
			endelse
		endif
	endfor
	endif
;
;	Custom option for triple phases (disabled)
	if fit_options.f lt 0 then begin
	for sc=0,n_elements(index)-1 do begin
		p=scans[index[sc]].triplephasec(tr,0:nc-1)
		e=scans[index[sc]].triplephasecerr(tr,0:nc-1)
		m=scans[index[sc]].triplephasem(tr,0:nc-1)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 then begin
			nf=100
			f=findgen(nf+1)/2-nf/4 & f=f/RAD & r=f
			for i=0,nf do $
			r[i]=total(((p[eindex]+f[i]-m[eindex])/(e[eindex]))^2)
			i=where(r eq min(r)) & i=i[0]
			if i gt 0 and i lt nf-1 then begin ; Don't hit limits
				scans[index[sc]].triplephasec(tr,0:nc-1)= $
				scans[index[sc]].triplephasec(tr,0:nc-1)+f[i]
;				scans(index(sc)).triplephasecerr(tr,0:nc-1)= $
;				scans(index(sc)).triplephasecerr(tr,0:nc-1)*f(i)
			endif else begin
				print,date+' '+triple+' scan ' $
					+string(index[sc]+1,format='(i3)') $
					+' : limit hit for floating CP!'
				scans[index[sc]].triplephasec(tr,0:nc-1)= $
				scans[index[sc]].triplephasec(tr,0:nc-1)+f[i]
			endelse
		endif
	endfor
	endif
;
;	Unwrap the data phase by comparing it to the model
	for iscan=0,n_elements(index)-1 do begin
		repeat begin
		wrap=where((scans[index[iscan]].triplephasec(tr,*)- $
			    scans[index[iscan]].triplephasem(tr,*) $
					gt pi_circle) $
		       and (scans[index[iscan]].triplephasecerr(tr,*) gt 0), $
		       count)
		tcount=count
		if count gt 0 then scans[index[iscan]].triplephasec(tr,wrap)=$
				   scans[index[iscan]].triplephasec(tr,wrap) $
				  -2*pi_circle
		wrap=where((scans[index[iscan]].triplephasec(tr,*)- $
			    scans[index[iscan]].triplephasem(tr,*) $
					lt -pi_circle) $
		       and (scans[index[iscan]].triplephasecerr(tr,*) gt 0), $
			      count)
		tcount=tcount+count
		if count gt 0 then scans[index[iscan]].triplephasec(tr,wrap)=$
				   scans[index[iscan]].triplephasec(tr,wrap) $
				  +2*pi_circle
		endrep until tcount eq 0
	endfor
;
	endif ; NS > 1
; Reset magnitudes inside TR loop
if num_binary() ge 1 then star_model.magnitudes=magnitudes
endfor
;
; Update night stored with new model data
if strupcase(!version.os) ne 'WIN32' and n_elements(bufferinfo) gt 1 $
				     and not noload then storenight,11
;
NEXT:
ENDFOR
;
; Save statistics of floating calibration factors
if fit_options.f and not quiet then free_lun,unit
;
endif
;
; Section for positions.........................................................
;
if ds_options.a ne 0 then begin
;
if n_elements(positions) eq 0 then begin
	print,'***Error(CALCMODEL): no position data!'
	return
endif
;
; Use shortest wavelength from model
lambda=fltarr(1)+gen_model.wavelengths[0]*1e-6
;
p=modelpos(positions.jd)
list_poscomps,components
for i=0,n_elements(components)-1 do begin
	index=where(positions.component eq components[i],count)
	pos1=dblarr(count,2)
	if componentparse(components[i],comp_1,comp_2) ne 0 then return
	tflux=0.d0
	for k=0,strlen(comp_1)-1 do begin
		j=where(strmid(comp_1,k,1) eq star_model.component) & j=j[0]
		flux=stellarfluxes(star_model[j],lambda) $
		    *modelfluxes(star_model[j],lambda) & flux=total(flux)
		pos1=pos1+reform(p[index,j,*]*flux,count,2)
		tflux=tflux+flux
	endfor
	pos1=pos1/tflux
	pos2=dblarr(count,2)
	tflux=0.d0
	for k=0,strlen(comp_2)-1 do begin
		j=where(strmid(comp_2,k,1) eq star_model.component) & j=j[0]
		flux=stellarfluxes(star_model[j],lambda) $
		    *modelfluxes(star_model[j],lambda) & flux=total(flux)
		pos2=pos2+reform(p[index,j,*]*flux,count,2)
		tflux=tflux+flux
	endfor
	pos2=pos2/tflux
	dx=pos2[*,0]-pos1[*,0]
	dy=pos2[*,1]-pos1[*,1]
	rhom=sqrt(dx^2+dy^2)
	thetam=atan(dx,dy)
	if count eq 1 then begin
		positions[index].rhom=rhom[0]
		positions[index].thetam=thetam[0]
	endif else begin
		positions[index].rhom=rhom
		positions[index].thetam=thetam
	endelse
endfor
;
endif
;
; Section for velocities........................................................
if ds_options.s ne 0 then begin
;
if n_elements(velocities) eq 0 then begin
        print,'***Error(CALCMODEL): no velocity data!'
        return
endif
;
v=modelvel(velocities.jd)
list_velcomps,components
; Allow composite component of 2 or more stars, e.g. 'AB'
nv=n_elements(velocities)
nc=n_elements(components)
r=dblarr(nv,nc)
for i=0,n_elements(components)-1 do begin
	index=whereequal(star_model.component,components[i])
	if n_elements(index) ge 2 then begin
	r[*,i]=total(v[*,index]*((dblarr(nv)+1)#star_model[index].mass),2) $
	      /total(star_model[index].mass)
	endif else $
	r[*,i]=v[*,index]
endfor
for i=0,n_elements(components)-1 do begin
	index=where(velocities.component eq components[i])
	velocities[index].valuem=r[index,i]
endfor
;
endif
;
; Section for magnitudes........................................................
;
if ds_options.p gt 0 and not ds_options.pm then begin
;
if n_elements(p_magnitudes) eq 0 then begin
        print,'***Error(CALCMODEL): no magnitude data!'
        return
endif
;
filters=unique(p_magnitudes.filter)
wdbands=binary_model.wdband
;
for i=0,n_elements(filters)-1 do begin
;
case filters[i] of
	'U': 	begin
		lambda=findgen(23)*5+305
		tm=johnson_u(lambda)
		wdband=5
		end
	'B': 	begin
		lambda=findgen(40)*5+360
		tm=johnson_b(lambda)
		wdband=6
		end
	'V': 	begin
		lambda=findgen(54)*5+475
		tm=johnson_v(lambda)
		wdband=7
		end
	'R': 	begin
		lambda=findgen(23)*20+520
		tm=johnson_r(lambda)
		wdband=8
		end
	'I': 	begin
		lambda=findgen(26)*20+680
		tm=johnson_i(lambda)
		wdband=9
		end
	'J': 	begin
		lambda=findgen(31)*20+960
		tm=johnson_i(lambda)
		wdband=10
		end
	'H': 	begin
		lambda=findgen(17)*23+1460
		tm=johnson_i(lambda)
		wdband=45
		end
	'K': 	begin
		lambda=findgen(17)*50+1800
		tm=johnson_i(lambda)
		wdband=11
		end
	'Hp': 	begin
		lambda=findgen(113)*5+335
		tm=hipparcos_hp(lambda)
		wdband=25
		end
endcase
lambda=lambda*1d-9
;
index=where(p_magnitudes.filter eq filters[i])
jd=p_magnitudes[index].jd
;
binary_model.wdband=wdband
valuem=-2.5*alog10(reform(tm#modelflux(jd,lambda))/total(tm))
;
; Minimize median difference between model and data values
dm=median(p_magnitudes[index].value-valuem)
p_magnitudes[index].valuem=valuem+dm
;
endfor
;
binary_model.wdband=wdbands
;
endif
;
; Section for auxilliary data...................................................
;
; Parallax
if ds_options.px ne 0 then begin
if n_elements(parallaxes) eq 0 then begin
        print,'***Error(CALCMODEL): no parallax data!'
        return
endif
j=0
for i=0,n_elements(binary_model.component)-1 do begin
	if binary_model[i].method eq 1 then begin
		parallaxes.valuem[j]=modelpx(binary_model[i].component)
		j=j+1
	endif
endfor
endif
;
; K1
if ds_options.k1 ne 0 then begin
if n_elements(k1) eq 0 then begin
        print,'***Error(CALCMODEL): no K1 data!'
        return
endif
k1.valuem=modelk('A')
endif
;
; K2
if ds_options.k2 ne 0 then begin
if n_elements(k2) eq 0 then begin
        print,'***Error(CALCMODEL): no K2 data!'
        return
endif
k2.valuem=modelk('B')
endif
;
; V sin(i)
;
; Restore handling of floating point exceptions and math errors
; http://www.idlcoyote.com/code_tips/underflow.html
if 0 then begin
floating_point_underflow = 32
status = Check_Math() ; Get status and reset accumulated math error register.
IF(status AND NOT floating_point_underflow) NE 0 THEN $
	Message, 'IDL Check_Math() error: ' + StrTrim(status, 2)
!Except = currentExcept
endif
;
end
