pro calcoeffs,reason,unit
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Compute calibration coefficients from given calibrator data and fit selection.
; Store information in one element of array cal_entries for each baseline and
; channel.
;
; cal_options.v=0: Vissq
; cal_options.v=1: TripleAmp
; cal_options.v=2: TriplePhase
;
; cal_options.l=1: loop over baselines and channels
;
; The new self-calibration option (cal_options.s=1) was created for visibility
; spectra of lines, allowing the continuum to be self-calibrated to the level
; of the estimated visibility (using the continuum diameter).
;
common BaseFunctions,numlist,functions
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common CalInfo,cal_entries
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(reason) eq 0 then reason=''
if strlen(reason) ne 0 then cal_reason=reason $
		       else cal_reason=strmid(systime(),0,24)
;
index=where(cal_entries.num_coeffs gt 0,count_cal_in)
;
; Check indicator and function selection
if strlen(sel_indct[0]) eq 0 then begin
	print,'***Error(CALCOEFFS): no indicators selected!'
	return
endif
index=where(sel_funcn[*,0] ne '',count)
if count eq 0 then begin
	print,'***Error(CALCOEFFS): no base functions selected!'
	return
endif
if count lt n_elements(sel_indct) then begin
	print,'***Error(CALCOEFFS): no base functions selected for one or '+ $
		'more indicators!'
	return
endif
;
; Check calibrator star selection
if n_elements(cal_stars) eq 0 then begin
	print,'***Error(CALCOEFFS): cal_stars undefined!'
	return
endif else if cal_stars[0] eq '' then begin
	print,'***Error(CALCOEFFS): no calibration stars selected!'
	return
endif
;
; Prepare data selection via what is user-specified in the plot widgets
; Note that we use the 'y'-stream data selection!
flag=set_starsel()
if set_indexsel('y') ne 0 then begin
	print,'***Error(CALCOEFFS): invalid data selection!'
	return
endif
set_streamsel,'y',item,ds_ib,ds_ob,ds_tr,ds_ch,ds_bl,scan_index
if cal_options.v eq 0 then begin
	index=where(ds_ch ge genconfig.numspecchan[ds_ob],count)
	if count gt 0 then begin
		print,'***Error(CALCOEFFS): invalid channel selection!'
		return
	endif
	index=where(ds_bl ge genconfig.numbaseline[ds_ob],count)
	if count gt 0 then begin
		print,'***Error(CALCOEFFS): invalid baseline selection!'
		return
	endif
endif else begin
	index=where(ds_ch ge genconfig.triplenumchan[ds_tr],count)
	if count gt 0 then begin
		print,'***Error(CALCOEFFS): invalid triple channel selection!'
		return
	endif
endelse
scan_count=n_elements(scan_index)
;
; Obtain index into selected scans corresponding to calibrator selection
calscan_index=scan_index
for k=0,n_elements(calscan_index)-1 do begin
	index=where(cal_stars eq scantable[calscan_index[k]].starid,count)
	if count eq 0 then calscan_index[k]=-1
endfor
index=where(calscan_index ge 0,calscan_count)
if calscan_count eq 0 then begin
	print,'Warning(CALCOEFFS): No scans seelcted for this calibrator selection!'
	return
endif
calscan_index=calscan_index[where(calscan_index ge 0,calscan_count)]
;
; Set/determine the scaling parameters
x_mid=fltarr(n_elements(sel_indct))
x_scl=fltarr(n_elements(sel_indct))
for i=0,n_elements(sel_indct)-1 do begin
	index=where(indicators eq sel_indct[i]) & index=index[0]+1
	x=set_caldata(index,0,0,0,calscan_index,mid,scl)
	x_mid[i]=mid
	x_scl[i]=scl
endfor
;
; Create array of cal_entry structures, if not done before
if n_elements(cal_entries) eq 0 then create_calentries
;
; The loop option is equivalent to repeated calls to this function
if cal_options.l then begin
	num_b_loop=n_elements(ds_bl)
	num_c_loop=n_elements(ds_ch)
	ds_bl_bck=ds_bl
	ds_ch_bck=ds_ch
	num_p_loop=calscan_count
	if cal_options.s then num_c_loop=1 $
			 else num_p_loop=1
endif else begin
;	No loop, fit to all selected data
	num_b_loop=1
	num_c_loop=1
	num_p_loop=1
endelse
if cal_options.v eq 1 or cal_options.v eq 2 then num_b_loop=1
;
if num_c_loop gt 1 then ch_wout_data=intarr(num_c_loop)
;
close_unit=0
if n_elements(unit) eq 0 then begin
	openw,unit,'/dev/tty',/get_lun
	close_unit=1
endif
;
print,'Beginning calibration...'
;
FOR bloop=0,num_b_loop-1 DO BEGIN	; baselines
FOR ploop=0,num_p_loop-1 DO BEGIN	; scans
FOR cloop=0,num_c_loop-1 DO BEGIN	; channels
;
; Loop over selected baselines and channels of a selected output beam.
;
; Update the data selection
if cal_options.l then begin
	ds_bl=ds_bl_bck[bloop]
	ds_ch=ds_ch_bck[cloop]
	ds_pt=calscan_index[ploop]
	if cal_options.s then ds_ch=ds_ch_bck $
			 else ds_pt=calscan_index
endif else begin
	ds_pt=calscan_index
endelse
;
num_point=n_elements(ds_pt)
num_channel=n_elements(ds_ch)
num_baseline=n_elements(ds_bl) & if cal_options.v gt 0 then num_baseline=1
first=1
block=0
;
; Set up design matrix or average variable values, using selected cal data
for ib=0,num_baseline-1 do begin
for ic=0,num_channel-1 do begin
	channel=ds_ch[ic]
	baseline=ds_bl[ib]
	if cal_options.v eq 1 then begin
;	Item 31 is TripleAmp c/e
	R=set_plotdata(31,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	E=set_ploterr(31,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	endif else if cal_options.v eq 2 then begin
;	Item 33 is TriplePhaseC
	R=set_plotdata(33,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)/RAD
	E=set_ploterr(33,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)/RAD
	endif else begin
;	Item 27 is VisSq c/e
	R=set_plotdata(27,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	E=set_ploterr(27,ds_ib,ds_ob,ds_tr,channel,baseline,ds_pt)
	endelse
;
	index=where(strpos(sel_funcn[*,0],'S_') ge 0,count)
	if count ne 0 then begin
		do_smooth=1
		if first then begin
			smooth_time=float(strmid(sel_funcn[index[0],0],2,2))/60
			i=where(indicators eq sel_indct[index[0]]) & i=i[0]
			DM=set_caldata(i+1,0,0,0,indgen(n_elements(ScanTable)))
			RM=fltarr(calscan_count)
			RE=fltarr(calscan_count)
			first=0
		endif
		index=where(E le 0,count)
		if count gt 0 then E[index]=1
		wt=1/E^2
		if count gt 0 then wt[index]=0
		RM=RM+R*wt
		RE=RE+wt	; we store the weight here in RE for simplicity
	endif else begin
		do_smooth=0
		M=calmatrix(ds_pt,ds_ob,channel,baseline,x_mid,x_scl)
		if first then begin
		   nscans=n_elements(M[*,0])
		   nterms=n_elements(M[0,*])
		   if nterms gt n_elements(cal_entries[0].coeffs) then begin
			   print,'***Error(CALCOEFFS): too many terms!'
			   return
		   endif
		   DM=fltarr(nscans*num_baseline*num_channel,nterms,/nozero)
		   RM=fltarr(nscans*num_baseline*num_channel,/nozero)
		   RE=fltarr(nscans*num_baseline*num_channel,/nozero)
		   first=0
		endif
;		if cal_options.s then index=ploop else index=indgen(nscans)
		DM[block*nscans:(block+1)*nscans-1,*]=M
		RM[block*nscans:(block+1)*nscans-1]=R
		RE[block*nscans:(block+1)*nscans-1]=E
		block=block+1
	endelse
endfor
endfor
;
; Edit and weight the data
index=where(RE gt 0,count)
;
IF count EQ 0 THEN BEGIN
;
	if cal_options.s then begin
	print,'Warning(CALCOEFFS): no valid data points found! OB=',ds_ob+1, $
	', Bl=',ds_bl+1,', Pt=',ds_pt+1,format='(a,i2,a,i2,a,i3)'
	endif else begin
	if num_c_loop eq 1 then begin
	print,'Warning(CALCOEFFS): no valid data points found! OB=',ds_ob+1, $
	', Bl=',ds_bl+1,', Ch=',retroparse(ds_ch+1),format='(a,i2,a,i2,a,a)'
	endif else ch_wout_data[cloop]=cloop
	endelse
;
ENDIF ELSE BEGIN
;
if do_smooth then begin
;
	h=DM[scan_index]
	first_hour=min(h)-1
	last_hour=max(h)+1
	DM=DM[calscan_index]
;
	WT=RE[index]
	RM=RM[index]/WT
	DM=DM[index]
;
;
	coeffs=cal_entries[0].coeffs*0
	num_coeffs=n_elements(coeffs)
	hour_grid=first_hour $
		 +findgen(num_coeffs)/(num_coeffs-1)*(last_hour-first_hour)
	for i=0,num_coeffs-1 do begin
		exponents=((hour_grid[i]-DM)/smooth_time)^2
;		index=where(exponents gt 20,count)
;		if count gt 0 then exponents(index)=20
		index=where(exponents lt 20,count)
		if count eq 0 then index=where(exponents eq min(exponents))
		exponents=exponents[index] < 20
		WTE=WT[index]
		RME=RM[index]
		weighting=exp(-exponents)
		hwh=weighting*WTE
		coeffs[i]=total(RME*hwh)/total(hwh)
	endfor
	si=sort(DM)
	if cal_options.v eq 2 then $
	rms=sqrt(total((RM[si]-spline(hour_grid,coeffs,DM[si]))^2) $
                /n_elements(RM)) else $
	rms=sqrt(total((1-RM[si]/spline(hour_grid,coeffs,DM[si]))^2) $
                /n_elements(RM))
;
endif else begin
;
	RM=RM[index]
	RE=RE[index]
	if cal_options.s then RE[*]=1	; not yet figured out why
	DM=DM[index,*]
	if cal_options.s then begin
		v=medianve(RM,e)
		index=where(abs(RM-v) le 2*e)
		RM=RM[index]
		RE=RE[index]
		DM=DM[index,*]
	endif
	WT=1/RE
	RM=RM*WT
	for i=0,nterms-1 do DM[*,i]=DM[*,i]*WT
;
	RM=double(RM)
	DM=double(DM)
;
; 	Solve for calibration coefficients
	T=transpose(DM)
	N=T#DM
	R=T#RM
;
; 	Solution with SVD if small eigenvalues occur, direct otherwise
	if nterms gt 1 then begin
		svd,N,w,u,v
		small=where(w lt max(w)*1.0e-10,count)
		if count gt 0 then begin
			print,'Warning(CALCOEFFS):',count, $
				' eigenvalue(s) edited!'
			w[small]=0.0
			svbksb,u,w,v,R,coeffs
		endif else begin
			coeffs=invert(N)#R
		endelse
		if cal_options.v eq 2 then $
		rms=sqrt(total(((RM-(DM#coeffs))/WT)^2) $
		         /n_elements(RM)) else $
		rms=sqrt(total((1-RM/(DM#coeffs))^2) $
		         /n_elements(RM))
	endif else begin
		coeffs=1/N*R
		if cal_options.v eq 2 then $
		rms=sqrt(total(((RM-(DM*coeffs[0]))/WT)^2) $
		         /n_elements(RM)) else $
		rms=sqrt(total((1-RM/(DM*coeffs[0]))^2) $
		         /n_elements(RM))
	endelse
	num_coeffs=n_elements(coeffs)
;
endelse
;
; Store calibration information in array of structures cal_entries.
;
index=where(cal_entries.num_coeffs gt 0,count)
if count eq n_elements(cal_entries) then begin
	print,'***Error(CALCOEFFS): structure cal_entries full!'
	return
endif
id=count
;
num_indct=n_elements(sel_indct)
num_funcn=n_elements(sel_funcn[0,*])
num_stars=n_elements(ds_stars)
num_calstars=n_elements(cal_stars)
for ib=0,num_baseline-1 do begin
for ic=0,num_channel-1 do begin
	cal_entries[id].reason=cal_reason
	cal_entries[id].variable=cal_options.v
	cal_entries[id].outbeam=ds_ob
	cal_entries[id].triple=ds_tr
	cal_entries[id].channel=ds_ch[ic]
	cal_entries[id].baseline=ds_bl[ib]
	cal_entries[id].stars(0:num_stars-1)=ds_stars
	if cal_options.s then cal_entries[id].scans(0)=ds_pt $
			 else cal_entries[id].scans(0:scan_count-1)=scan_index
	cal_entries[id].calstars(0:num_calstars-1)=cal_stars
	cal_entries[id].indicators(0:num_indct-1)=sel_indct
	cal_entries[id].x_mid(0:num_indct-1)=x_mid
	cal_entries[id].x_scl(0:num_indct-1)=x_scl
	cal_entries[id].functions(0:num_indct-1,0:num_funcn-1)=sel_funcn
	cal_entries[id].coeffs(0:num_coeffs-1)=coeffs
	cal_entries[id].num_indct=num_indct
	cal_entries[id].num_funcn=num_funcn
	cal_entries[id].num_coeffs=num_coeffs
	if cal_options.s then cal_entries[id].num_scans=1 $
			 else cal_entries[id].num_scans=scan_count
	id=id+1
endfor
endfor
;
; Diagnostic messages
if cal_options.v eq 2 then trms='. (RMS [d] =' else trms='. (RMS [%] ='
if cal_options.v eq 2 then frms=RAD            else frms=100.0
;
if cal_options.l then begin	; Loop
;
; Baselines (VisSq)
if cal_options.v eq 0 then begin
;
if cal_options.s then begin
	if not close_unit then printf,unit, $
	'Solution calculated for OB=',ds_ob+1,', Bl=',ds_bl+1,', Pt=',ds_pt+1, $
 	trms,frms*rms,')',format='(a,i2,a,i2,a,i3,a,f6.2,a)' $
	else $
	printf,unit,ds_ob+1,ds_bl+1,ds_pt+1,trms,frms*rms, $
	format='(%"Solution calculated for OB=%d, Bl=%d, Pt=%d %s %6.2f)\r")'
endif else begin
	if not close_unit then printf,unit, $
	'Solution calculated for OB=',ds_ob+1,', Bl=',ds_bl+1,', Ch=',ds_ch+1, $
 		trms,frms*rms,')',format='(a,i2,a,i2,a,i3,a,f6.2,a)' $
	else $
	printf,unit,ds_ob+1,ds_bl+1,ds_ch+1,trms,frms*rms, $
	format='(%"Solution calculated for OB=%d, Bl=%d, Ch=%d %s %6.2f)\r")'
endelse
;
endif
;
; Triples (TripleAmp/Phase)
if cal_options.v ge 1 then begin
	if not close_unit then $
	printf,unit,'Solution calculated for TR=',ds_tr+1,', Ch=',ds_ch+1, $
 		trms,frms*rms,')',format='(a,i2,a,i3,a,f6.2,a)' $
	else $
	printf,unit,format='(%"Solution calculated for TR=%d, Ch=%d %s %6.2f)\r")', $
	ds_tr+1,ds_ch+1,trms,frms*rms
endif
;
endif else begin		; no Loop
	;
if cal_options.v eq 0 then begin
	if num_b_loop eq 1 then $
	printf,unit,format='(%"Combined solution calculated for all BL\r")'
endif
if cal_options.v ge 1 then $
	printf,unit,format='(%"Combined solution calculated for all TR\r")'
;
endelse				; end Loop
;
ENDELSE	; valid data points
;
ENDFOR	; channels
;
if num_c_loop gt 1 then begin
	index=where(ch_wout_data ne 0,count)
	if count gt 0 then begin
	if cal_options.v eq 0 then $
	print,'Warning(CALCOEFFS): no valid data points found! OB=',ds_ob+1, $
		', Bl=',ds_bl+1,', Ch=',retroparse(ch_wout_data[index]+1), $
		format='(a,i2,a,i2,a,a)'
	if cal_options.v ge 1 then $
	print,'Warning(CALCOEFFS): no valid data points found! TR=',ds_tr+1, $
		', Ch=',retroparse(ch_wout_data[index]+1), $
		format='(a,i2,a,a)'
	endif
endif
;
ENDFOR	; scans
ENDFOR	; baselines
;
if close_unit then free_lun,unit
;
; index=where(cal_entries.num_coeffs gt 0,count_cal_out)
; if count_cal_out gt count_cal_in then reason=cal_reason else reason=''
;
end
