function calcuv,hourangle,declination,utc,uvw
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Compute and return uvw coordinates, a right-handed coordinate system
; with v increasing to the N and u increasing to the E. This corresponds
; to astronomical images with E to the left and N up.
;
; To compute the baseline position angle of (u,v), use atan(u,v)=atan(u/v).
; The baseline PA then is the angle on the ground of the projected baseline
; starting from N over E.
;
; X,Y,Z coordinate system (see also Interferometry and Synthesis in Radio
; Astronomy, Thompson, Moran, & Swenson, page 86):
; X in the meridian plane, Y towards the east, Z towards the north pole.
; utc (in seconds) will be used for polar motion correction of the baselines.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
rad=180/pi_circle
ha=hourangle*(15/rad)
dec=declination/rad
;
n=n_elements(ha)
su=dblarr(n,3) & sv=su & sw=su
;
su[*,0]=+sin(ha)
su[*,1]=+cos(ha)
su[*,2]=+0
sv[*,0]=-sin(dec)*cos(ha)
sv[*,1]=+sin(dec)*sin(ha)
sv[*,2]=+cos(dec)
sw[*,0]=+cos(dec)*cos(ha)
sw[*,1]=-cos(dec)*sin(ha)
sw[*,2]=+sin(dec)
;
if n_elements(utc) eq 0 then utc=fltarr(n)
;
if n_elements(uvw) eq 0 then $
	uvw=dblarr(GenConfig.NumOutBeam, $
		max(GenConfig.NumSpecChan),max(GenConfig.NumBaseline),3,n)
;
for ob=0,GenConfig.NumOutBeam-1 do begin
	for bl=0,GenConfig.NumBaseline[ob]-1 do begin
		b_xyz=baselinecoord(ob,bl,utc)
		um=su[*,0]*b_xyz[0,*]+su[*,1]*b_xyz[1,*]+su[*,2]*b_xyz[2,*]
		vm=sv[*,0]*b_xyz[0,*]+sv[*,1]*b_xyz[1,*]+sv[*,2]*b_xyz[2,*]
		wm=sw[*,0]*b_xyz[0,*]+sw[*,1]*b_xyz[1,*]+sw[*,2]*b_xyz[2,*]
		for ic=0,GenConfig.NumSpecChan[ob]-1 do begin
			uvw[ob,ic,bl,0,*]=um/GenConfig.Wavelength[ic,ob]
			uvw[ob,ic,bl,1,*]=vm/GenConfig.Wavelength[ic,ob]
			uvw[ob,ic,bl,2,*]=wm/GenConfig.Wavelength[ic,ob]
		endfor
	endfor
endfor
;
return,uvw
;
end
