pro compile_calmaster
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Compile file diameter.hdn, which contains all HDN stars for which diameter
; information could be obtained. Also obtain information on duplicity, making
; this file useful for selecting calibrators for optical interferometers.
;
; This procedure will make use of the Borde, Merand, Verhoelst, and Van Belle
; calibrator lists.
;
common StarBase,StarTable,Notes
;
; Get StarTable based on extended HDN catalogue
get_hdn
num_hdn=n_elements(startable)
;
; Estimate diameters based on surface brightness
get_jhk_hdn
diameter_vk
startable.model='LD'
startable.bflag='.'
startable.sflag='5'
predicted=startable.diameter
;
; Compute T_eff and log(g) for limb darkening
teff_stars	; Teff will be zero for stars w/out full classification
logg_stars
;
; Compute limb darkening coefficients
limb_stars,[450,2500],/logarithmic
;
table=startable
;
; Read various calibrator catalogs
get_borde,/init
;
; Initialize comments array
comments=strarr(num_hdn)
;
; Read various lists; each one supercedes the previous one!
; Read Infrared Flux Method diameters
print,'Reading IRFM diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.irfm',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='irfm'
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt((1-7*u/15)/(1-u/3))
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 1st publication
print,'Reading DM1 diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm1',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm1'
		diam=d[i]
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 2nd publication
print,'Reading DM2 diameters...'
bsc=0
mv=0.0
ri=0.0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm2',bsc,mv,ri,d,/col, $
	resize=[1,2,3,4],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm2'
		diam=d[i]
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Tyler's NPOI diameters
print,'Reading TN diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.tn',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='tn'
		diam=d[i]
		if abs(diam-predicted[j])/diam gt 0.1 and predicted[j] ne 0 then begin
			comments[j]=comments[j]+', pred.=' $
				+string(predicted[j],format='(f5.2)')
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read bad calibrator CSV list from
; http://www.astro.lsa.umich.edu/~pmuirhea/calib.html
print,'Reading bad calibrators...'
bsc=0
d=0.0
s=dc_read_fixed(!oyster_dir+'starbase/diameter.bad',l,/col, $
	ignore=['!'],format='(a300)')
for i=0,n_elements(l)-1 do begin
	v=nameparse(l[i],',')
	hdn=cri(v[0],'hdn')
	j=where(startable.hdn eq hdn,count) & j=j[0]
	if count eq 1 then begin
		comments[j]='bad'
	endif
endfor
;
; Potential calibrators have d < 1 mas
index=where(startable.diameter lt 1)
startable[index].bflag='C'
;
; Check with Batten's catalogue for known spectroscopic binaries
bat=cri(startable.bsc,'bsc-bat')
index=where(bat ne -1)
startable[index].bflag='B'
startable.zerospacing=1
;
; Check with Hipparcos catalog for multiplicity information
table=startable
rename_starids,'bsc-hip'
read_catalogs
table[where(startable.hflag eq 'C' and startable.a gt 0 $
				   and startable.a lt 5)].bflag='B'
table[where(startable.hflag eq 'O')].bflag='B'
table.a=startable.a
table.dmv=startable.dmv
table.hflag=startable.hflag
startable=table
;
; These are known binaries missed (so far) by the algorithm
;
; Upsilon Tauri (occultation double)
j=where(startable.starid eq 'BSC1392',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Occultation'
endif
;
; Theta 1 Tauri (WDS)
j=where(startable.starid eq 'BSC1411',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Hyades'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC1427',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=200 mas'
endif
;
; Zeta Orionis (NPOI)
j=where(startable.starid eq 'BSC1948',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=42 mas'
endif
;
; Nu Geminorum (WDS)
j=where(startable.starid eq 'BSC2343',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=80 mas'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC5733',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=50-100 mas'
endif
;
; Sigma Herculis (WDS)
j=where(startable.starid eq 'BSC6168',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=67 mas'
endif
;
; Zeta Draconis (WDS)
j=where(startable.starid eq 'BSC6396',count) & j=j[0]
if count eq 1 then begin
        startable[j].bflag='B'
        comments[j]=comments[j]+'WDS'
endif
;
; Zeta Sagittae (WDS)
j=where(startable.starid eq 'BSC7546',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=146 mas'
endif
;
; FKV0782
j=where(startable.starid eq 'BSC7955',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Griffin SB'
endif
;
; FKV0131
j=where(startable.starid eq 'BSC1122',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'NPOI'
endif
;
; Update some comments
j=where(startable.starid eq 'BSC2421',count) & j=j[0]
if count eq 1 then begin
	comments[j]=comments[j]+'hb 1.32 mas @ 443 nm'
endif
j=where(startable.starid eq 'BSC5340',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='.'
	comments[j]=comments[j]+', aq=19.86 mas'
endif
;
; Write diameters to file
; put_diameter,comments
;
end
