pro compile_diameter
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Compile file diameter.bsc, which contains all BSC stars for which diameter
; information could be obtained. Also obtain information on duplicity, making
; this file useful for selecting calibrators for optical interferometers.
;
common StarBase,StarTable,Notes
;
; Get StarTable based on Bright Star Catalogue
get_bsc
index=where(startable.mv lt 5.5,num_bsc)
startable=startable[index]
;
; Initialize comments array
comments=strarr(num_bsc)
;
; Obtain Hipparcos parallaxes, and also update spectral types
t=startable
rename_starids,'bsc-hip'
read_catalogs
t.spectrum=startable.spectrum
t.class=startable.class
t.type=startable.type
t.px=startable.px
startable=t
;
; Assume that still unclassified stars are giants
index=where(startable.class eq 0)
startable[index].class=3
;
; Compute T_eff and log(g) for limb darkening
teff_stars
logg_stars
;
; Compute limb darkening coefficients
limb_stars
;
; For unclassified stars, set coefficients so that d_550/d_800=0.967
; Also assume they are giants, which is OK for estimated diameters here
startable[index].a0=1.28
startable[index].a1=-1.6/1e3
startable[index].a2=0
;
; Compute (B-V)0 and from that visual extinction Av
bv=startable.bv
bv_stars
av=3.1*((bv-startable.bv)>0)
;
; Obtain Av from extinction maps and correct V for extinction
print,'Estimating extinction...'
av_stars,/drimmel
index_av=where(bv eq 100 or abs(startable.av-av) gt 0.5,count_av)
if count_av gt 0 then startable[index_av].av=0
startable.mv=startable.mv-startable.av
index=where(startable.av ne 0,count)
if count gt 0 then comments[index]='Av '
;
; Compute estimated diameter from photometry for all stars (binary/single)
print,'Estimating diameters...'
startable.bflag='.'
; diameter_ri	; This is the UD diameter
; We now use V-K calibration because 2MASS gives us all the IR photometry
get_jhk_bsc
diameter_vk	; This is the LD diameter
predicted=startable.diameter
;
; Read various lists; each one supercedes the previous one!
; Read Infrared Flux Method diameters
print,'Reading IRFM diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.irfm',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='irfm '
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i];*sqrt((1-7*u/15)/(1-u/3))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 1st publication
print,'Reading DM1 diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm1',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm1 '
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt(((1-u/3)/(1-7*u/15)))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Dave's Mark III diameters, 2nd publication
print,'Reading DM2 diameters...'
bsc=0
mv=0.0
ri=0.0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.dm2',bsc,mv,ri,d,/col, $
	resize=[1,2,3,4],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='dm2 '
		u=poly(800.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt(((1-u/3)/(1-7*u/15)))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Tyler Nordgren's NPOI diameters
print,'Reading TN diameters...'
bsc=0
d=0.0
s=dc_read_free(!oyster_dir+'starbase/diameter.tn',bsc,d,/col, $
	resize=[1,2],ignore=['!'])
for i=0,n_elements(d)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]='tn '
		u=poly(750.0,[startable[j].a0,startable[j].a1,startable[j].a2])
		diam=d[i]*sqrt(((1-u/3)/(1-7*u/15)))
		if abs(diam-predicted[j])/diam gt 0.1 and $
			    predicted[j] ne 0 then begin
			comments[j]=comments[j]+'pred.=' $
				+string(predicted[j],format='(f5.2)')+' '
		endif
		startable[j].diameter=diam
	endif
endfor
;
; Read Brian's (Kloppenborg?) single star list
print,'Reading single star list...'
bsc=0
s=dc_read_free(!oyster_dir+'starbase/calibrators.brian',bsc,/col, $
	resize=[1],ignore=['!'])
for i=0,n_elements(bsc)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		comments[j]=comments[j]+'S '
	endif
endfor
;
; Read JPL calibrator list
print,'Reading JPL calibrator list...'
n='                                                                            '
n=n+n
s=dc_read_fixed(!oyster_dir+'starbase/calibrators.jpl',n,/col, $
	resize=[1],ignore=['#'],format='(a132)')
hdn=''
s=dc_read_free(!oyster_dir+'starbase/calibrators.jpl',hdn,/col, $
	resize=[1],ignore=['#'])
bsc=cri(long(strmid(hdn,3,6)),'hdn-bsc')
for i=0,n_elements(bsc)-1 do begin
	j=where(startable.bsc eq bsc[i],count) & j=j[0]
	if count eq 1 then begin
		if strpos(n[i],'Dbl') eq -1 and strpos(n[i],'SB') eq -1 then $
		comments[j]=comments[j]+'JPL '
	endif
endfor
;
; Potential calibrators have d < 1.1 mas
index=where(startable.diameter lt 1.1)
startable[index].bflag='C'
; Stars with unreliable colors should not be used as calibrators
; unless they have really small apparent diameters (< 0.5 mas)
if count_av gt 0 then begin
	comments[index_av]=comments[index_av]+'(B-V)! '
	startable[index_av].bflag='.'
	index=where(startable[index_av].diameter le 0.5,count)
	if count gt 0 then startable[index_av[index]].bflag='C'
endif
;
; Check with Batten's catalogue for known spectroscopic binaries
bat=cri(startable.bsc,'bsc-bat')
index=where(bat ne -1)
startable[index].bflag='B'
startable.zerospacing=1
;
; Check with Hipparcos catalog for multiplicity information
table=startable
rename_starids,'bsc-hip'
read_catalogs
table[where(startable.hflag eq 'C' and startable.a gt 0 $
				   and startable.a lt 5)].bflag='B'
table[where(startable.hflag eq 'O')].bflag='B'
table.a=startable.a
table.dmv=startable.dmv
table.hflag=startable.hflag
startable=table
;
; Check against BCR
table=startable	; startable will be modified
; get_bcr	; outdated
bad_cals	; new, bad cal. will have bflag='.' instead of 'C'
; index=whereequal(table.hdn,startable.hdn)
; startable=table
; if index(0) ge 0 then begin
	jndex=where(table.bflag eq 'C' and startable.bflag eq '.',count)
	if count gt 0 then comments[jndex]=comments[jndex]+'BCR '
; endif
;
; These are known binaries missed (so far) by the algorithm
; Tau Ori: listed as double in Simbad, poor results in SigOri reductions
j=where(startable.starid eq 'BSC1735',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Suspicious'
endif
; HR 6554 has a close companion which makes acquisition difficult
; (Information from R. Zavala May 6, 2013)
j=where(startable.starid eq 'BSC6554',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Physical'
endif
; HR 5774 has a close companion which makes acquisition difficult
; (Information from R. Zavala April 18, 2013)
j=where(startable.starid eq 'BSC5774',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Physical'
endif
; Upsilon Tauri (occultation double)
j=where(startable.starid eq 'BSC1392',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Occultation'
endif
;
; Theta 1 Tauri (WDS)
j=where(startable.starid eq 'BSC1411',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Hyades'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC1427',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=200 mas'
endif
;
; Zeta Orionis (NPOI)
j=where(startable.starid eq 'BSC1948',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=42 mas'
endif
;
; Nu Geminorum (WDS)
j=where(startable.starid eq 'BSC2343',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=80 mas'
endif
;
; (WDS)
j=where(startable.starid eq 'BSC5733',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=50-100 mas'
endif
;
; Sigma Herculis (WDS)
j=where(startable.starid eq 'BSC6168',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=67 mas'
endif
;
; Zeta Draconis (WDS)
j=where(startable.starid eq 'BSC6396',count) & j=j[0]
if count eq 1 then begin
        startable[j].bflag='B'
        comments[j]=comments[j]+'WDS'
endif
;
; Zeta Sagittae (WDS)
j=where(startable.starid eq 'BSC7546',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'a=146 mas'
endif
;
; FKV0782
j=where(startable.starid eq 'BSC7955',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'Griffin SB'
endif
;
; FKV0131
j=where(startable.starid eq 'BSC1122',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='B'
	comments[j]=comments[j]+'NPOI'
endif
;
; FKV0622 (BSC6175)
j=where(startable.starid eq 'BSC6175',count) & j=j[0]
if count eq 1 then begin
	startable[j].diameter=0.510
	comments[j]=comments[j]+'Gordon ea. 2018'
endif
;
; Update some comments
j=where(startable.starid eq 'BSC2421',count) & j=j[0]
if count eq 1 then begin
	comments[j]=comments[j]+'hb 1.32 mas @ 443 nm'
endif
j=where(startable.starid eq 'BSC5340',count) & j=j[0]
if count eq 1 then begin
	startable[j].bflag='.'
	comments[j]=comments[j]+'aq=19.86 mas'
endif
;
; Write diameters to file
num_stars=n_elements(startable)
if n_elements(comments) ne num_stars then comments=strarr(num_stars)
openw,unit,'diameter.bsc',/get_lun
printf,unit, $
'! NOTE: THIS FILE IS GENERATED BY OYSTER. ANY MODIFICATIONS'
printf,unit, $
'! SHOULD BY MADE TO COMPILE_DIAMETER.'
printf,unit, $
'!'
printf,unit, $
'!This file is read by OYSTER to obtain limb-darkened diameters of'
printf,unit, $
'!calibrator stars. These were either taken from observations listed below,'
printf,unit, $
'!or were estimated based on (V-K) and V  according to Mozurkewich et'
printf,unit, $
'!al. 2003 and van Belle et al. 2009. Estimates are +/- 5%, except possibly'
printf,unit, $
'!for supergiants.  Also given are coefficients of a quadratic fit to the'
printf,unit, $
'!linear limb darkening coefficients for the interval [450 nm,850 nm]:'
printf,unit, $
'!u_l=a0 + a1 * lambda + a2 * lambda^2, lambda = wavelength in nm.  Stars'
printf,unit, $
'!without luminosity class were assumed to be giants, and limb darkening'
printf,unit, $
'!fit coefficients will give u_l=0.0 at 800 nm, u_l=0.4 at 550 nm.  ZERO is'
printf,unit, $
'!the zero spacing visibility (unused). There is a flag to indicate type of'
printf,unit, $
'!star: B/C/. = binary or multiple/calibrator/everything else.  A calibrator'
printf,unit, $
'!is meant here to be a single star less than 1 mas in diameter.'
printf,unit, $
'!'
printf,unit, $
'!Stars found in the Batten catalog of spectroscopic binary orbits, as'
printf,unit, $
'!well as stars having C and O solutions in the Hipparcos catalog were'
printf,unit, $
'!labeled as multiple. There is a number of stars labeled as multiple'
printf,unit, $
'!based on previous Mark III and NPOI observations.'
printf,unit,'!'
printf,unit, $
'! Measurements:'
printf,unit, $
'! irfm  Infrared Flux Method (Blackwell & Shallis 1977)'
printf,unit, $
'! dm1:  Mozurkewich et al. 1991, first MarkIII diameter paper'
printf,unit, $
'! dm2:  Mozurkewich et al. 2003, second MarkIII diameter paper'
printf,unit, $
'! tn :  Nordgren et al. 1999, NPOI diameter paper'
printf,unit, $
'!'
printf,unit, $
'! Qualifier:'
printf,unit, $
'! Av: Non-zero Av correction applied to V based on E(B-V)'
printf,unit, $
'! (B-V)!: Unexpected (B-V) color, should not be used as calibrator'
printf,unit, $
'! S:    Brian Mason single star list'
printf,unit, $
'! JPL:  JPL calibrator candidate'
printf,unit, $
'! WDS:  Washington Double Star catalog'
printf,unit, $
'! BCR:  Bad Calibrator Registry'
printf,unit, $
'!       (http://www.eso.org/sci/observing/tools/bcr/bcr.html)'
printf,unit,'!'
printf,unit,'! BSC LD[mas]  ZERO    a0     a1     a2  cflag Comment'
printf,unit,'!'
for i=0,num_stars-1 do begin
	if startable[i].diameter ne 0 and startable[i].diameter lt 99 then begin
	printf,unit,startable[i].bsc,startable[i].diameter, $
		startable[i].zerospacing,startable[i].a0, $
		startable[i].a1*1000,startable[i].a2*1000000, $
		startable[i].bflag,startable[i].name,' '+comments[i], $
		format='(i4.4,2x,f6.3,2x,f4.2,2x,3(f6.3,1x),2x,a1,2x,a,a)'
	endif
endfor
free_lun,unit
;
end
