function componentflux,component,epochs,lamda
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Input parameters:
; component: 		A, B, C,..., A-B, AB-C, AB-CD,...
; epochs: 		dblarr(num_scan), Julian day (-2440000)
; lamda [m]: 		dblarr(num_wave)
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
forward_function componentflux
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
; Initialize constants
factor=(pi_circle^2)/(180L*3600L*1000L)
mas2rad=pi_circle/(180L*3600L*1000L)
;
num_scan=n_elements(epochs)
num_wave=n_elements(lamda)
;
; Replicate fluxes to same dimensions as [epochs,lamda]
flux=dblarr(num_wave,num_scan)
;
; Get length of component to find out whether component is single or double
len=strlen(component)
;
; Single star...................................................................
if len eq 1 then begin
;
; Find applicable model
index=where(star_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTFLUX): non-unique component definition!'
	return,-1
endif
model=star_model[index]
;
; This initializes the limb darkening database
fluxes=stellarfluxes(model,lamda,ld_coeffs)
fluxes=fluxes*modelfluxes(model,lamda)
for i=0,num_scan-1 do flux[*,i]=fluxes
if n_elements(flux) eq 1 then flux=flux[0]	; default model flux
;
case abs(model.type) of
;	Image cube
     12:begin
	nl=n_elements(channel_wavelengths)
	fluxes=fltarr(nl)
	for j=0,nl-1 do fluxes[j]=total(channel_images[*,*,j])
	flux=interpol(fluxes,channel_wavelengths,lamda)
	if n_elements(flux) eq 1 then flux=flux[0]
	return,flux
	end
; 	Rotating star, using Roche code V-1.4
     14:begin
	vis=rochevis(model,lamda,0,0,fluxes)
	for i=0,num_scan-1 do flux[*,i]=fluxes
	if n_elements(flux) eq 1 then flux=flux[0]
;	3.24 mas is the diameter of Vega, whose colors we use for normalization
	return,flux*(star_model[0].diameter/3.24)^2
	end
;
;	Ring or disk with hole, and temperature profile (Hillenbrand 1992),
;	also used by Malbet et al. (2005, "temperature gradient model").
;	T  ~ (radius/r_min)^-alpha, alpha should be -3/4
;	For the effect of tilt on flux, see Berger & Segransan (2007).
;	T  ~ (radius/r_min)^alpha, alpha should be -3/4.
     15:begin
	num_ring=100
	w_ring=(model.diameter/2)*model.width/num_ring
	r_ring=model.diameter/2+w_ring/2+findgen(num_ring)*w_ring
	alpha=model.alpha
	teff0=model.teff
	fluxes=fltarr(num_wave)
	for k=0,num_ring-1 do begin
		model.teff=-teff0*(r_ring[k]/r_ring[0])^alpha
		f=stellarfluxes(model,lamda,ld_coeffs)*2*!pi*r_ring[k]*w_ring $
		 *modelfluxes(model,lamda)*cos(model.tilt/RAD)
		fluxes=fluxes+f
	endfor
	model.teff=teff0
	for i=0,num_scan-1 do flux[*,i]=fluxes
	if n_elements(flux) eq 1 then flux=flux[0]
;	save,lamda,fluxes,filename='fluxes_comp_15.xdr'
	return,flux
	end
;
   else:begin
;	Return default flux computed by stellarfluxes
	return,flux
	end
;
endcase
;
endif
;
; Double star...................................................................
;
; Set applicable model
index=where(binary_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): non-unique component definition!'
	return,-1
endif
model=binary_model[index]
;
case model.method of
;
; 	Interacting binary, with LC/WD code
      4:begin
	f=lcmap(model,epochs,1)
	for i=0,num_scan-1 do begin
;	for j=0,num_wave-1 do begin
	flux[*,i]=f[i]
;	endfor
	endfor
	return,flux
;
; 	Old WD code
	for i=0,num_scan-1 do begin
	for j=0,num_wave-1 do begin
	map=wdmap(model,epochs[i],lamda[j])
	flux[j,i]=total(map.fcc)
	endfor
	endfor
	return,flux
        end
;
   else:
;
endcase
;
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
return,componentflux(comp_1,epochs,lamda)+componentflux(comp_2,epochs,lamda)
;
end
