function componentvis,component,time,lamda,fluxes,u,v
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Compute the complex visibility of a component. If double, add sub-components.
; Sub-components may themselves be binary, single stars, or both. The mode
; parameter determines which astrophysical scenario/method is to be used for
; the visibility computation of the components/sub-components.
;
; Input parameters:
; component: 		A, B, C,..., A-B, AB-C, AB-CD,...
; time [s]: 		dblarr(num_scan)
; lamda grid [m]: 	dblarr(num_wave)
; u,v [lamda]: 	dblarr(num_wave,num_scan)
;
; Output parameter:
; fluxes:		dblarr(numwave)
;
; The model can also provide a file name from which to restore an SED.
; This is then used to interpolate the fluxes.
;
; If only one value for time and wavelength is passed, assume u and v
; are arrays corresponding to an arbitrary coverage.
;
; Note: this function actually returns a correlated flux, as complex number so
; that components can be added with the proper phase term. After addition of all
; correlated fluxes, the visibility is obtained by normalizing with the total
; flux. That is also why this function returns the total flux contribution.
;
; Types renamed with version 6.12 (2 Aug 2007)
; 10->8; 11->9; 12->10; 14->11; 8->12; 16->13; 13->obsolete; 9->14; 17->16
;
; Sign for type introduced with version 8: if negative and the component is
; part of a multiple system, do not include it in the total correlated flux.
;
forward_function componentvis
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common RandomModel,RandomSeed
common AuxData,parallaxes,k1,k2,vsini
common LocalModelVis,comp_fluxes,md5seed
;
; Common block needed to include md5sum of image
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
RAD=180/pi_circle
MAS=1/3600000.d0
;
; Initialize constants
mas2rad=pi_circle/(180L*3600L*1000L)
;
num_scan=n_elements(time)
num_wave=n_elements(lamda)
;
; Get length of component to find out whether component is single or double
len=strlen(component)
;
; Single star...................................................................
if len eq 1 then begin
;
; Find applicable model
index=where(star_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): non-unique component definition!'
	return,-1
endif
model=star_model[index]
;
; Create directory for temporary products
if model.type eq 12 then begin
	tmp_products_dir='oyster.componentvis'
	if not file_test(tmp_products_dir,/directory) then $
		file_mkdir,tmp_products_dir
endif
;
; Allow rotation and stretch of all components: PA > 0 rotates counter-clockwise
; Ratio < 1 compresses E-W size of object
vr=v*cos(model.pa/RAD)+u*sin(model.pa/RAD)
ur=model.ratio*(u*cos(model.pa/RAD)-v*sin(model.pa/RAD))
;
; Get SED; all astrophysical types 12 and up compute their own fluxes
if abs(model.type) le 11 then fluxes=stellarfluxes(model,lamda,ld_coeffs) $
			 else fluxes=fltarr(num_wave)+1
;
; If an SED file (XDR format) has been specified, it overrides
if strlen(model.sed) gt 0 then begin
;	Read spectrum from XDR file: l[m], f
	restore,model.sed
	fb=f
;	Reduce resolution to half (Nyquist!) of that of the lamda array
	if n_elements(lamda) ge 2 then begin
		r=4*fix(abs(lamda[1]-lamda[0])/(l[1]-l[0]))
		if r gt 1 then fb=box(f,r)
	endif
	si=sort(lamda)
	fluxes=spline(l,fb,lamda[si])
	fluxes[si]=fluxes
endif
;
; Apply correction to fluxes specified by magnitudes
fluxes=fluxes*modelfluxes(model,lamda)
;
; Replicate fluxes to same dimensions as (u,v)
flux=dblarr(num_wave,num_scan)
for j=0,num_scan-1 do flux[*,j]=fluxes
if n_elements(flux) eq 1 then flux=flux[0]
;
; Compute contribution due to a star spot
if model.spot[3] ne 0 then begin
	print,'Warning: spot computation possible only with LCDC2015!'
	spot_amp=0
	spot_flux=0
	spot_phase=0
	spot_model=model
	spot_model.teff=model.spot[0]*model.teff
	spot_model.diameter=model.spot[3]
	spot_fluxes=stellarfluxes(spot_model,lamda) $
		   -fluxes*(spot_model.diameter/model.diameter)^2
	xy=dblarr(2)
	xy[0]=model.spot[1]*sin(model.spot[2]/RAD)*MAS/RAD
	xy[1]=model.spot[1]*cos(model.spot[2]/RAD)*MAS/RAD
	spot_phase=complexarr(num_wave,num_scan)
	if n_elements(spot_phase) eq 1 then begin
		spot_phase=exp(2*pi_circle*i_complex*(u*xy[0]+v*xy[1]))
	endif else begin
		for j=0,num_wave-1 do $
   	    	spot_phase[j,*]=exp(2*pi_circle*i_complex $
				*(u[j,*]*xy[0]+v[j,*]*xy[1]))
	endelse
	UD=dblarr(num_wave,num_scan)
	UD[*,*]=model.spot[3]
	if n_elements(UD) eq 1 then UD=UD[0]
	arg=pi_circle*mas2rad*sqrt(u^2+v^2)*UD
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
	spot_amp=2*beselj(arg,1)/arg
	if count gt 0 then spot_amp[index]=1
	spot_flux=flux
	for j=0,num_scan-1 do spot_flux[*,j]=spot_fluxes
endif else begin
	spot_amp=0
	spot_flux=0
	spot_phase=0
endelse
;
; Derive component phase from RA Dec offsets
phase=exp(2*pi_circle*i_complex*(u*model.xoff+v*model.yoff)*mas2rad)
;
case abs(model.type) of
;
;	Uncorrelated flux component
      0:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
	endif
;	For fiber-fed BCs, the FOV depends on lambda, so also the flux
;	GRAVITY: the FOV decreases towards the blue end of the spectrum!
	if strpos(genconfig.configid,'GRAVITY') eq 0 then begin
		airy=(lamda/genconfig.diameter[0])/mas2rad
		radius=airy[0]/2; distance from center in field of view [mas]
		fluxes=fluxes*exp(-(2*radius/airy)^2)
	endif
	return,modvis
	end
;
;	Uniform (elliptical) disk independent of wavelength, input diameter UD
      1:begin
	UD=dblarr(num_wave,num_scan)
	UD[*,*]=model.diameter
	if n_elements(UD) eq 1 then UD=UD[0]
	arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
	visamp=2*beselj(arg,1)/arg
	if count gt 0 then visamp[index]=1
;	save,lamda,fluxes,filename='fluxes_comp_1.xdr'
;	save,lamda,spot_fluxes,filename='fluxes_spot_1.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Uniform (elliptical) disk scales with wavelengths, input diameter LD
      2:begin
	UD=dblarr(num_wave,num_scan)
	UD1=sqrt((1-7*ld_coeffs/15)/(1-ld_coeffs/3))*model.diameter
	for j=0,num_wave-1 do UD[j,*]=UD1[j]
	if n_elements(UD) eq 1 then UD=UD[0]
	arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
	visamp=2*beselj(arg,1)/arg
	if count gt 0 then visamp[index]=1
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, linear law, input diameter LD
;	From: R. Hanbury Brown et al. 1974 MNRAS 167, 475
      3:begin
	alpha=dblarr(num_wave,num_scan)
	beeta=dblarr(num_wave,num_scan)
	for i=0,num_scan-1 do begin
		alpha[*,i]=1-ld_coeffs
		beeta[*,i]=ld_coeffs
	endfor
	if n_elements(alpha) eq 1 then alpha=alpha[0]
	if n_elements(beeta) eq 1 then beeta=beeta[0]
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	index=where(arg eq 0,count)
	if count gt 0 then arg[index]=1
;       visamp=(alpha*beselj(arg,1)/arg+beeta*sqrt(pi_circle/2)* $
;               sqrt(2/(pi_circle*arg))*(sin(arg)/arg-cos(arg))/ $
;               sqrt(arg*arg*arg))/(alpha/2+beeta/3)
;	Use simplified form
	visamp=(alpha*beselj(arg,1)/arg+beeta*(sin(arg)/arg-cos(arg))/arg^2) $
	       /(alpha/2+beeta/3)
	if count gt 0 then visamp[index]=1
;	save,lamda,fluxes,filename='fluxes_comp_3.xdr'
;	save,lamda,spot_fluxes,filename='fluxes_spot_3.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Limb-darkened disk, logarithmic law, input diameter LD
      4:begin
	modvis=complexarr(num_wave,num_scan)
	maps=limbmaps(model,lamda*1d9)
	for j=0,num_wave-1 do begin
		fcc_total=total(maps[j].fcc)
		for i=0,num_scan-1 do begin
			modvis[j,i]=mapvis(maps[j],ur[j,i],vr[j,i])/fcc_total
		endfor
	endfor
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
;	Limb-darkened disk, linear law, input diameter LD, from Kurucz
;	Note: type number hard coded in kurugrid (limb.pro)
      5:begin
	alpha=dblarr(num_wave,num_scan)
	beeta=dblarr(num_wave,num_scan)
	for i=0,num_scan-1 do begin
		alpha[*,i]=1-ld_coeffs
		beeta[*,i]=ld_coeffs
	endfor
	if n_elements(alpha) eq 1 then alpha=alpha[0]
	if n_elements(beeta) eq 1 then beeta=beeta[0]
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	visamp=(alpha*bessjy8(arg,1)/arg $
	        +beeta*sqrt(pi_circle/2)*bessjy8(arg,1.5)/arg^1.5) $
	       /(alpha/2+beeta/3)
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, Hestroffer law, input diameter LD, from Kurucz
;	Note: type number hard coded in kurugrid (limb.pro)
      6:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	arg_nu=1+(ld_coeffs/2)
	gaama=dblarr(num_wave,num_scan)
	bessl=dblarr(num_wave,num_scan)
	delta=dblarr(num_wave,num_scan)
	for j=0,num_wave-1 do begin
		gaama[j,*]=gamma(arg_nu[j]+1)
		bessl[j,*]=bessjy8(arg[j,*],arg_nu[j])
		delta[j,*]=(arg[j,*]/2)^arg_nu[j]
	endfor
	if n_elements(gaama) eq 1 then gaama=gaama[0]
	if n_elements(bessl) eq 1 then bessl=bessl[0]
	if n_elements(delta) eq 1 then delta=delta[0]
	visamp=gaama*(bessl/delta)
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, visibility from stellar atmosphere profile
;	Kurucz models
      7:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	profiles=kurufluxes(model,lamda,mu)
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		visamp=fltarr(num_uv)
		for i=0L,num_uv-1 do begin
			visamp[i]=provis(profiles,mu,arg[i])
		endfor
	endif else begin
		for j=0,num_wave-1 do begin
		for i=0,num_scan-1 do begin
			visamp[j,i]=provis(profiles[*,j],mu,arg[j,i])
		endfor
		endfor
	endelse
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
;	Limb-darkened disk, visibility from stellar atmosphere profile
;	Models from Jason Aufdenberg
      8:begin
	arg=pi_circle*model.diameter*mas2rad*sqrt(vr^2+ur^2)
	profiles=jasonfluxes(model,lamda,mu)
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		visamp=fltarr(num_uv)
		for i=0L,num_uv-1 do begin
			visamp[i]=provis(profiles,mu,arg[i])
		endfor
	endif else begin
		for j=0,num_wave-1 do begin
		for i=0,num_scan-1 do begin
			visamp[j,i]=provis(profiles[*,j],mu,arg[j,i])
		endfor
		endfor
	endelse
	return,complex(visamp)*flux+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Elliptical uniform disk
;	This is a numerical version of the analytical code for mode=1
      9:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		num_uv=n_elements(u)
		modvis=complexarr(num_uv)
		for i=0L,num_uv-1 do begin
			strip=edstrip(model,lamda,ur[i],vr[i])
			fcc_total=total(strip.b)
			modvis[i]=stripvis(strip,ur[i],vr[i])/fcc_total
		endfor
	endif else begin
		for i=0,num_scan-1 do begin
		for j=0,num_wave-1 do begin
			strip=edstrip(model,lamda[j],ur[j,i],vr[j,i])
			fcc_total=total(strip.b)
			modvis[j,i]=stripvis(strip,ur[j,i],vr[j,i])/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase
        end
;
;	Gaussian elliptical disk
;	Reference: "difmap" software by M. Shepherd.
     10:begin
	GD=dblarr(num_wave,num_scan)
	GD[*,*]=model.diameter
	if n_elements(GD) eq 1 then GD=GD[0]
	uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
	index=where(uvr gt 5,count)
	if count gt 0 then uvr[index]=5
;	visamp=exp(-0.3606737602*uvr*uvr)
	visamp=exp(-(1./(4.*alog(2)))*uvr*uvr)
	if count gt 0 then visamp[index]=0
; 	save,lamda,fluxes,filename='fluxes_comp_10.xdr'
	return,complex(visamp)*flux*phase+complex(spot_amp)*spot_flux*spot_phase
	end
;
; 	Pearson disk (FFT of which is exponential)
     11:begin
	modvis=complexarr(num_wave,num_scan)
;
;	Create image with Pearson radial profile
	num=41
	map=indgen(num*num)
	xcc=float((map mod num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
	ycc=float((map  /  num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
;	Make clumpy
	nc=20
	cr=fltarr(nc)+0.1	; clump radii
	xc=randomu(RandomSeed,nc)*2-1
	yc=randomu(RandomSeed,nc)*2-1
	xc=[xc,0]
	yc=[yc,0]
	cr=[cr,0.1]	; add central clump
	nc=nc+1
	index=intarr(n_elements(map))
	for k=0,nc-1 do begin
		jndex=where(sqrt((xcc-xc[k])^2+(ycc-yc[k])^2) lt cr[k])
		index[jndex]=1
	endfor
	xcc=xcc[where(index eq 1)]
	ycc=ycc[where(index eq 1)]
; 	Compute radius squared
	r=sqrt(xcc^2+ycc^2)
; 	Use only circular region with r<1
	index=where(r lt 1)
	xcc=xcc[index]
	ycc=ycc[index]
	r=r[index]
	n=n_elements(r)
;	Create array of to hold maps
	map=alloc_map(n,lonarr(n),fltarr(n),fltarr(n),fltarr(n))
	maps=replicate(map,num_wave)
;	Pearson width = 1.5 (1 = Lorentzian, inf. = Gaussian)
	a=[1.0,1.5,20.0]
	for j=0,num_wave-1 do begin
		fcc=fluxes[j]*pearson_funct(r*a[2]*10,a)
		maps[j].xcc=xcc*model.diameter/2
		maps[j].ycc=ycc*model.diameter/2
		maps[j].fcc=fcc
	endfor
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
		fcc_total=total(maps[0].fcc)
                for i=0L,num_uv-1 do begin
		modvis[i]=mapvis(maps[0],ur[i],vr[i])/fcc_total
		endfor
;	For model
	endif else begin
		for j=0,num_wave-1 do begin
		fcc_total=total(maps[j].fcc)
		for i=0,num_scan-1 do begin
		modvis[j,i]=mapvis(maps[j],ur[j,i],vr[j,i])/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
; 	Pearson disk (FFT of which is exponential)
;	(Place holder for new model type, still to be defined)
    111:begin
	modvis=complexarr(num_wave,num_scan)
;
;	Create image with Pearson radial profile
	num=41
	map=indgen(num*num)
	xcc=float((map mod num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
	ycc=float((map  /  num)-(num-1)/2)/((num-1)/2)  ; [-1,1]
;	Make clumpy
	nc=20
	cr=fltarr(nc)+0.1	; clump radii
	if RandomSeed[0] eq 0 then do_plot=1 else do_plot=0
	xc=randomu(RandomSeed,nc)*2-1
	yc=randomu(RandomSeed,nc)*2-1
	xc=[xc,0]
	yc=[yc,0]
	cr=[cr,0.1]	; add central clump
	nc=nc+1
	index=intarr(n_elements(map))
	for k=0,nc-1 do begin
		jndex=where(sqrt((xcc-xc[k])^2+(ycc-yc[k])^2) lt cr[k])
		index[jndex]=1
	endfor
	xcc=xcc[where(index eq 1)]
	ycc=ycc[where(index eq 1)]
; 	Compute radius squared
	r=sqrt(xcc^2+ycc^2)
; 	Use only circular region with r<1
	index=where(r lt 1)
	xcc=xcc[index]
	ycc=ycc[index]
	r=r[index]
	n=n_elements(r)
	if do_plot then plot,xcc,ycc,psym=3
;	Create array of to hold maps
	map=alloc_map(n,lonarr(n),fltarr(n),fltarr(n),fltarr(n))
	maps=replicate(map,num_wave)
;	Pearson width = 1.5 (1 = Lorentzian, inf. = Gaussian)
	a=[1.0,1.5,20.0]
	for j=0,num_wave-1 do begin
		fcc=fluxes[j]*pearson_funct(r*a[2]*10,a)
		maps[j].xcc=xcc*model.diameter/2
		maps[j].ycc=ycc*model.diameter/2
		maps[j].fcc=fcc
	endfor
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
                num_uv=n_elements(u)
                modvis=complexarr(num_uv)
		fcc_total=total(maps[0].fcc)
                for i=0L,num_uv-1 do begin
		modvis[i]=mapvis(maps[0],ur[i],vr[i])/fcc_total
		endfor
;	For model
	endif else begin
		for j=0,num_wave-1 do begin
		fcc_total=total(maps[j].fcc)
		for i=0,num_scan-1 do begin
		modvis[j,i]=mapvis(maps[j],ur[j,i],vr[j,i])/fcc_total
		endfor
		endfor
	endelse
	return,modvis*flux*phase+complex(spot_amp)*spot_flux*spot_phase
        end
;
; 	Image cube (third axis: wavelength)
     12:begin
;	If diameter is specified, it is the clean beam
	if model.diameter gt 0 then begin
		ur=u
		vr=v
	endif
	cellsize=model.cellsize
	if strlen(model.model) gt 0 then $
		if model.cellsize ne 0 then $
			readimages,model.model,/quiet,cellsize=cellsize else $
			readimages,model.model,/quiet
;	Obtain md5sum of input for visibility computation
	md5file=tmp_products_dir+'/'+specname(model.model)+'_' $
		+strtrim(string(randomu(md5seed)*6),2)+'_md5sum.dat'
	openw,unit,md5file,/get_lun
	writeu,unit,ur,vr,lamda,model, $
		channel_images,channel_wavelengths,images_cellsize
	free_lun,unit
	spawn,'md5sum '+md5file,md5sum
	spawn,'rm -f '+md5file
	words=nameparse(md5sum)
	xdrfile=file_search(tmp_products_dir+'/vis_comp_12.'+words[0]+'.xdr')
;
	IF strlen(xdrfile) ne 0 THEN BEGIN
;		print,'Restoring previously saved vis_comp_12 data...'
		restore,xdrfile
	ENDIF ELSE BEGIN
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
;		For uvimage, which calls with num_scan=num_wave=1
		map=chmap(model,lamda)
		radius=sqrt(map.xcc^2+map.ycc^2)
		map.fcc=map.fcc*apodize_f(radius,lamda[0]) $
			       *modelfluxes(model,lamda[0])
		num_uv=n_elements(u)
		modvis=complexarr(num_uv)
		fcc_total=total(map.fcc)
		for i=0L,num_uv-1 do begin
			modvis[i]=mapvis(map,ur[i],vr[i])/fcc_total
		endfor
		if strlen(model.sed) eq 0 then begin
			fluxes=fcc_total*modelfluxes(model,lamda)
			flux=fluxes
		endif
	endif else begin
;		For model
		for i=0,num_scan-1 do begin
		for j=0,num_wave-1 do begin
			map=chmap(model,lamda[j])
			radius=sqrt(map.xcc^2+map.ycc^2)
			map.fcc=map.fcc*apodize_f(radius,lamda[j]) $
				       *modelfluxes(model,lamda[j])
			fcc_total=total(map.fcc)
			if fcc_total gt 0 then $
			modvis[j,i]=mapvis(map,ur[j,i],vr[j,i])/fcc_total
			if strlen(model.sed) eq 0 then begin
				fluxes[j]=fcc_total*modelfluxes(model,lamda[j])
				flux[j,i]=fluxes[j]
			endif
		endfor
		endfor
	endelse
;	Obtain md5sum for input to this computation
	openw,unit,md5file,/get_lun
	writeu,unit,ur,vr,lamda,model, $
		channel_images,channel_wavelengths,images_cellsize
	free_lun,unit
	spawn,'md5sum '+md5file,md5sum
	words=nameparse(md5sum)
	xdrfile=tmp_products_dir+'/vis_comp_12.'+words[0]+'.xdr'
	save,fluxes,flux,modvis,phase,filename=xdrfile
	spawn,'rm -f '+md5file
	ENDELSE
;
;	Correct for restoring beam effect if diameter is specified
	if model.diameter gt 0 then begin
		GD=dblarr(num_wave,num_scan)
		GD[*,*]=model.diameter
		vr=v*cos(model.pa/RAD)+u*sin(model.pa/RAD)
		ur=model.ratio*(u*cos(model.pa/RAD)-v*sin(model.pa/RAD))
		if n_elements(GD) eq 1 then GD=GD[0]
		uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
		index=where(uvr gt 5,count)
		if count gt 0 then uvr[index]=5
		visamp=exp(-0.3606737602*uvr*uvr)
		if count gt 0 then visamp[index]=0.01
		modvis=modvis/visamp
	endif
;
	return,modvis*flux*phase
        end
;
; 	"Effective temperature" PEARL image
     13:begin
;	If diameter is specified, it is the clean beam
	if model.diameter gt 0 then begin
		ur=u
		vr=v
	endif
	if strlen(model.model) gt 0 then $
		if model.cellsize ne 0 then $
			readimage,model.model,cellsize=cellsize else $
			readimage,model.model
	fluxes=fltarr(num_wave)+1
	flux=dblarr(num_wave,num_scan)+1
	modvis=complexarr(num_wave,num_scan)
	modvis[*]=imgvis(ur,vr,lamda)
;
;	Correct for restoring beam effect
	if model.diameter gt 0 then begin
		GD=dblarr(num_wave,num_scan)
		GD[*,*]=model.diameter
		if n_elements(GD) eq 1 then GD=GD[0]
		uvr=pi_circle*GD*mas2rad*sqrt(vr^2+ur^2)
		index=where(uvr gt 5,count)
		if count gt 0 then uvr[index]=5
		visamp=exp(-0.3606737602*uvr*uvr)
		if count gt 0 then visamp[index]=0.01
		modvis=modvis/visamp
	endif
;
	return,modvis*phase
        end
;
; 	Rotating star, using Roche code V-1.4
     14:begin
	modvis=complexarr(num_wave,num_scan)
	if n_elements(modvis) eq 1 then begin
		num_uv=n_elements(ur)
		modvis=complexarr(num_uv)
		modvis[*]=rochevis(model,fltarr(num_uv)+lamda,ur,vr,fluxes)
		flux[0]=fluxes[0]
	endif else begin
		for i=0,num_scan-1 do begin
		modvis[*,i]=rochevis(model,lamda,ur[*,i],vr[*,i],fluxes)
		flux[*,i]=fluxes
		endfor
	endelse
	fluxes=fluxes*modelfluxes(model,lamda)
	flux=flux*modelfluxes(model,lamda)
	return,modvis*flux*phase
        end
;
;	Ring or disk with hole, and temperature profile (Hillenbrand 1992),
;	also used by Malbet et al. (2005, "temperature gradient model").
;	Diameter refers to the inner diameter (2*r_min), width is r_max-r_min.
;	For the effect of tilt on flux, see Berger & Segransan (2007).
;	T  ~ (radius/r_min)^alpha, alpha should be -3/4.
     15:begin
	num_ring=100
	w_ring=(model.diameter/2)*model.width/num_ring
	r_ring=model.diameter/2+w_ring/2+findgen(num_ring)*w_ring
	alpha=model.alpha
	teff0=model.teff
	visamp=fltarr(num_wave,num_scan)
	if n_elements(visamp) eq 1 then begin
		num_uv=n_elements(u)
		visamp=complexarr(num_uv)
	endif
	fluxes=fltarr(num_wave)
	for k=0,num_ring-1 do begin
		arg=pi_circle*r_ring[k]*2*mas2rad*sqrt(vr^2+ur^2)
		model.teff=-teff0*(r_ring[k]/r_ring[0])^alpha
;		Compute black body fluxes of each ring and sum up
		f=stellarfluxes(model,lamda,ld_coeffs)*2*!pi*r_ring[k]*w_ring $
		 *cos(model.tilt/RAD)*modelfluxes(model,lamda); <-- scaling
		fluxes=fluxes+f
		flux=dblarr(num_wave,num_scan)
		for j=0,num_scan-1 do flux[*,j]=f
		if n_elements(flux) eq 1 then flux=flux[0]
;		Compute visiblity of ringlet
		visamp=visamp+flux*beselj(arg,0)
	endfor
;	save,lamda,fluxes,filename='fluxes_comp_15.xdr'
	model.teff=teff0
	return,complex(visamp)*phase
	end
;
;	Result file (*.i###) from DUSTY or other shell simulation
     16:begin
	arg=pi_circle*2*mas2rad*sqrt(vr^2+ur^2)
;	Read and decode file, last # line before data must contain wavelengths
	readshell,model.model,l_shell,r_shell,f_shell
	nl=n_elements(l_shell)
	nr=n_elements(r_shell)
	r_shell=r_shell*abs(model.diameter/2)	; Scale with diameter/2
;	Extract rows corresponding to central star
;	We assume that these have the highest intensities!
	tf=total(f_shell,1)
	index_s=where(tf/max(tf) gt 0.99,count_s)
	num_ring=nr-count_s
	visamp_shell=fltarr(num_wave,num_scan)
	if n_elements(visamp_shell) eq 1 then begin
		num_uv=n_elements(u)
		visamp_shell=complexarr(num_uv)
	endif
	fluxes=fltarr(num_wave)
	flux=dblarr(num_wave,num_scan)
;
;	V(q)/F(q)/F(0); F(q)=2pi Int(I*J0(2pi*q*x)*x*dx)
	x0=r_shell[count_s:nr-1]
	f0=dblarr(num_wave,nr)
;	Interpolate fluxes onto lamda grid
	for k=0,nr-1 do begin
		index=where(lamda ge l_shell[0] and lamda le l_shell[nl-1])
		f0[index,k]=interpol(f_shell[*,k],l_shell,lamda[index])
;		Extrapolate to smaller lamda as a constant
		index=where(lamda lt l_shell[0],count)
		if count gt 0 then f0[index,k]=f_shell[0,k]
;		Extrapolate to longer lamda as a constant
		index=where(lamda gt l_shell[nl-1],count)
		if count gt 0 then f0[index,k]=f_shell[nl-1,k]
	endfor
;	Integrate rings other than those of the central star
	f=f0[*,count_s:nr-1]
	for j=0,num_wave-1 do begin
		y0=f[j,*]*2*pi_circle*x0 & y0=reform(y0)
;		y0=gsmooth(x0,y0,15)
		y0=apodize_f(x0,lamda[j])*y0*modelfluxes(model,lamda[j])
		maxrad=max(x0-x0[0]) < 1000	; Maximum radius 1"
		x1=findgen(1000)/1000*maxrad+x0[0]
		y1=interpol(y0,x0,x1)
;		y1=spline(x0,y0,x1)
;		fluxes(j)=int_tabulated(x0,y0,/double)
		fluxes[j]=int_tabulated(x1,y1,/double)
		for i=0,num_scan-1 do begin
;			y=beselj(x0*arg(j,i),0)*y0
			y=beselj(x1*arg[j,i],0)*y1
;			visamp_shell(j,i)=int_tabulated(x0,y,/double)
			visamp_shell[j,i]=int_tabulated(x1,y,/double)
			flux[j,i]=fluxes[j]
		endfor
	endfor
;
;	Add back transform of central star
	if model.diameter gt 0 then begin
		UD=dblarr(num_wave,num_scan)
		UD[*,*]=r_shell[count_s]*2
		if n_elements(UD) eq 1 then UD=UD[0]
		arg=pi_circle*UD*mas2rad*sqrt(vr^2+ur^2)
		index=where(arg eq 0,count)
		if count gt 0 then arg[index]=1
		visamp_star=2*beselj(arg,1)/arg
		if count gt 0 then visamp_star[index]=1
;		f=total(f_shell(*,0:count_s-1),2)/count_s
;		f=interpol(f,l_shell,lamda)*!pi*(UD(0)/2)^2
		f=f0[*,0]*!pi*(UD[0]/2)^2
		f=f*modelfluxes(model,lamda)
		fluxes=fluxes+f
		flux=dblarr(num_wave,num_scan)
		for j=0,num_scan-1 do flux[*,j]=f
		if n_elements(flux) eq 1 then flux=flux[0]
		return,(complex(visamp_shell)+complex(visamp_star)*flux)*phase
	endif else begin
		return,complex(visamp_shell)*phase
	endelse
	end
;
endcase
endif
;
; Double star...................................................................
;
; Find applicable model
index=where(binary_model.component eq component,count)
if count ne 1 then begin
	print,'***Error(COMPONENTVIS): no or non-unique component definition!'
	return,-1
endif
model=binary_model[index]
;
parsedate,date,y,m,d
midnight=system_config(systemid,'MIDNIGHT')
jd=(julian(y,m,d)-2440000)+time/86400
jd0=(julian(y,m,d)-2440000)+(time*0+midnight/24)
;
; Do the astrometry (i.e. compute xy) for binary components
case model.method of
;
;	Normal detached binary, positions derived from orbit
      1:begin
	o_parms=dblarr(8)
	o_parms[0]=model.semimajoraxis*MAS/RAD
	o_parms[1]=model.eccentricity
	o_parms[2]=model.inclination/RAD
	o_parms[3]=model.periastron/RAD
	o_parms[4]=model.apsidalmotion/RAD
	o_parms[5]=model.ascendingnode/RAD
	o_parms[6]=model.period
	o_parms[7]=model.epoch
	xy=true2app(jd,o_parms,rho,theta,gen_model.ra,gen_model.dec,z)
	x=xy[*,0] & if n_elements(jd) eq 1 then x=x[0]
	y=xy[*,1] & if n_elements(jd) eq 1 then y=y[0]
	end
;
;	Normal detached binary, position from model
      2:begin
	xy=dblarr(n_elements(time),2)
	xy[*,0]=model.rho*sin(model.theta/RAD)*MAS/RAD
	xy[*,1]=model.rho*cos(model.theta/RAD)*MAS/RAD
	end
;
; 	Normal detached binary, positions from model w/orbital motion
;	Positions calculated for MIDNIGHT need to be derived for actual epochs
      3:begin
	o_parms=dblarr(8)
	o_parms[0]=model.semimajoraxis*MAS/RAD
	o_parms[1]=model.eccentricity
	o_parms[2]=model.inclination/RAD
	o_parms[3]=model.periastron/RAD
	o_parms[4]=model.apsidalmotion/RAD
	o_parms[5]=model.ascendingnode/RAD
	o_parms[6]=model.period
	o_parms[7]=model.epoch
	xyO=true2app(jd, o_parms,rhoO,thetaO,gen_model.ra,gen_model.dec)
	xy0=true2app(jd0,o_parms,rho0,theta0,gen_model.ra,gen_model.dec)
	dxy=xyO-xy0
	xy=dblarr(n_elements(time),2)
	xy[*,0]=model.rho*sin(model.theta/RAD)*MAS/RAD+dxy[*,0]
	xy[*,1]=model.rho*cos(model.theta/RAD)*MAS/RAD+dxy[*,1]
	end
;
; 	Interacting binary, with LC/WD code
      4:begin
	modvis=complexarr(num_wave,num_scan)
	fluxes=fltarr(num_wave)+1
	maps=lcmap(model,jd,5)
	for i=0,num_scan-1 do begin
	fcc_total=total(maps[i].fcc)
	for j=0,num_wave-1 do begin
	modvis[j,i]=mapvis(maps[i],u[j,i],v[j,i]);/fcc_total
	fluxes[j]=fcc_total
	endfor
	endfor
	return,modvis	; Special case: this code returns the visibility
;
;	Old WD code
	modvis=complexarr(num_wave,num_scan)
	fluxes=fltarr(num_wave)+1
	for i=0,num_scan-1 do begin
	for j=0,num_wave-1 do begin
	map=wdmap(model,jd[i],lamda[j])
	fcc_total=total(map.fcc)
	modvis[j,i]=mapvis(map,u[j,i],v[j,i])/fcc_total
	endfor
	endfor
	return,modvis	; Special case: this code returns the visibility
        end
;
endcase
;
; Compute component visibilities (must follow astrometry because of mode 4)
if modelparse(component,comp_1,comp_2) eq -1 then return,-1
modvis_1=componentvis(comp_1,time,lamda,fluxes1,u,v)
modvis_2=componentvis(comp_2,time,lamda,fluxes2,u,v)
;
; Process negative model type: contribution to be ignored
if strlen(comp_1) eq 1 then begin
	index=where(star_model.component eq comp_1)
	comp_fluxes[*,index]=fluxes1
	if star_model[index].type lt 0 then modvis_1=0
endif
if strlen(comp_2) eq 1 then begin
	index=where(star_model.component eq comp_2)
	comp_fluxes[*,index]=fluxes2
	if star_model[index].type lt 0 then modvis_2=0
endif
;
; Compute total flux to be returned
fluxes=fluxes1+fluxes2
;
; Compute masses for center-of-mass referencing
if componentparse(component,comp_1,comp_2) eq -1 then return,-1
masses=componentmass(comp_1,comp_2,num_body,index_comp)
;
; Establish phase center for bandwidth smearing computation
if model.component eq topbincomp() then begin
	!cop=strupcase(!cop)
	if !cop eq 'COM' then begin
;		Common center of mass
		factor1=-(1-masses[0]/total(masses))
		factor2=+(1-masses[1]/total(masses))
	endif else if !cop eq 'COP' then begin
;		Move phase center to most massive component
		factor1=0.
		factor2=1.
	endif else begin
;		Photometric center (COL)
		factor1=-(1-total(fluxes1)/total(fluxes))
		factor2=+(1-total(fluxes2)/total(fluxes))
	endelse
	index=where(star_model.type lt 0,count)
	for i=0,count-1 do fluxes=fluxes-comp_fluxes[*,index[i]]
endif else begin
;	For lower level binary components, reference to center of mass
	factor1=-(1-masses[0]/total(masses))
	factor2=+(1-masses[1]/total(masses))
endelse
;
; Allocate phase offsets and compute
phase1=complexarr(num_wave,num_scan)
if n_elements(phase1) eq 1 then phase1=complexarr(n_elements(u))
phase2=phase1
;
if num_wave gt 1 then begin
	for j=0,num_wave-1 do begin
	   phase1[j,*]=exp(2*pi_circle*i_complex*factor1 $
		*(u[j,*]*xy[*,0]+v[j,*]*xy[*,1]))
	   phase2[j,*]=exp(2*pi_circle*i_complex*factor2 $
		*(u[j,*]*xy[*,0]+v[j,*]*xy[*,1]))
	endfor
endif else begin
	   phase1[*]=exp(2*pi_circle*i_complex*factor1 $
		*(u[*]*xy[0,0]+v[*]*xy[0,1]))
	   phase2[*]=exp(2*pi_circle*i_complex*factor2 $
		*(u[*]*xy[0,0]+v[*]*xy[0,1]))
endelse
;
return,modvis_1*phase1+modvis_2*phase2
;
end
