pro fakedata,stations,stars,dutycycle=dutycycle,ldl=ldl,npoi=npoi, $
	starids=starids,times=times
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Create fake scan data for list of stations and stars. Use defaults for date
; and SystemId if not provided. One needs to specify the stations; if stars
; are not defined, they will be taken from the startable (or keyword starids).
; Units of dutycycle are hours, and corresponds to interval between successive
; scans on the same star. The schedule will cycle through all available
; stars within one hour, whether realistic or not.
;
; Using keywords starids *and* times [s], a specific schedule can be enforced.
;
; NPOI: ldl=1 (i.e. LDL in use) if not specified differently.
;
; Example:
; date='2023-12-16'
; systemid='VLTI/GRAVITY_LR'
; get_startable,gen_model.starid
; get_sysconfig
; fakedata,['AA0','AB5','AJ2','AJ6']
; readmodel,'delori.model'
; calcmodel
; mockdata
;
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common ModelFit,parameters,ds_options
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
;
if n_elements(ldl) eq 0 then ldl=1 else ldl=ldl gt 0
if n_elements(npoi) eq 0 then npoi=0 else npoi=1
;
; Shortcut to simulate NPOI data
if npoi then begin
	systemid='NPOI/6way'
	stations=['E07','AC0','AE0','AW0','W07','AN0']
	if n_elements(date) eq 0 then begin
		idldate=systime()
		parseidldate,idldate,y,m,d
		date=nextdate(constrictordate(y,m,d))
	endif
endif
;
if n_elements(stations) eq 0 then stations=genconfig.stationid
if n_elements(stars) eq 0 then begin
	if n_elements(startable) ne 0 then stars=startable.starid $
	else begin
		print,'Error: no stars defined and no StarTable present!'
		return
	endelse
endif
;
RAD=180/!pi
;
; Check inputs
flag=0
NumSid=n_elements(stations)
if NumSid gt 0 then begin
	sid_index=where(strlen(strcompress(stations,/remove)) gt 0,NumSid)
	if NumSid eq 0 then flag=1
endif else flag=1
if flag then begin
	print,'***Error(FAKEDATA): no stations specified!'
	return
endif
if n_elements(starids) ne 0 then stars=unique(starids)
if n_elements(stars) eq 0 then begin
	if n_elements(startable) eq 0 then begin
		print,'***Error(FAKEDATA): no stars specified!'
		return
	endif
	stars=startable.starid
endif else get_startable,stars
;
get_sysconfig,stations=stations
genconfig.configid=instrument_id(systemid)
if checkdata([7,8]) ne 0 then return
;
IF n_elements(times) eq 0 or n_elements(times) ne n_elements(starids) $
	THEN BEGIN
; Compute the visibility range
riseset,uptime,nighttime,ldl=ldl,/force
if uptime[0] gt uptime[1] then uptime[0]=uptime[0]-86400
;
; Create time stamps: believe it or not, but CALIB crashes if time stamps
; fall on full hours!
if n_elements(dutycycle) eq 0 then dutycycle=1.0 ; repeat after this many hours
seed=0
ti=randomu(seed,n_elements(stars))*3600*dutycycle
if instrument_id(systemid) eq 'PRIMA' then ti[*]=0
t0=findgen(nint(24/dutycycle))*3600*dutycycle $
	 -(geoparms.longitude/15+12)*3600
for i=0,n_elements(stars)-1 do begin
;	if min(t) lt 0 then t=t+86400
	t=t0+ti[i]
	ids=strarr(n_elements(t)) & ids[*]=stars[i]
	index=where(t gt uptime[0,i] and t lt uptime[1,i] $
		and t gt nighttime[0] and t lt nighttime[1],n)
;	index=where(t gt uptime(0,i) and t lt uptime(1,i),n)
	if n eq 0 then begin
		print,'Warning(FAKEDATA): ',stars[i],' not visible!'
	endif else begin
		if n_elements(times) eq 0 then begin
			times=t[index]
			starids=ids[index]
		endif else begin
			times=[times,t[index]]
			starids=[starids,ids[index]]
		endelse
	endelse
endfor
if n_elements(times) eq 0 then return
si=sort(times)
times=times[si]
if min(times) lt 0 then begin
	times=times+86400
	date=previousdate(date)
	geoparms.date=date
endif
starids=starids[si]
ENDIF ; ELSE starids=stars
;
numscan=n_elements(times)
;
; Get StationTable
get_stationtable
;
; Allocate the scans
scans=replicate(scan(),numscan)
;
; Put some data into scans
scans.time=times
scans.starid=starids
scans.photonrate=10
scans.photonrateerr=1
scans.vissq=1
scans.vissqerr=0.02
scans.vissqc=1
scans.vissqcerr=0.05
scans.tripleampc=1
scans.tripleampcerr=0.02
scans.triplephasec=0
scans.triplephasecerr=1.0/RAD
;
; Create the ScanTable
Iscan=lindgen(numscan)+1
ScanId=lindgen(numscan)+1
StarId=scans.starid
StartTime=scans.time
StopTime=scans.time
NumPoint=lonarr(numscan)
NumCoh=lonarr(numscan)
NumIncoh=lonarr(numscan)
Code=lonarr(numscan)+1
Station=lonarr(6,numscan)+1 & Station[0:NumSid-1,*]=1
;
ScanTable=build_scantable(Iscan,ScanId,StarId,Code,Station, $
		StartTime,StopTime,NumPoint, $
                NumCoh,NumIncoh)
;
; Do the astrometry
calcastrom
scans.fdlpos=scans.geodelay & scans.fdlposerr=1e-6
;
; Check whether delay line limits are exceeded
create_scanflagtable & n=0
parms=init_class('scan')
fdl_limit=system_config(system_id(systemid),'FDLLIMIT')
if ldl eq 0 then begin
for i=0,numscan-1 do begin
	if max(scans[i].fdlpos-min(scans[i].fdlpos)) gt fdl_limit then begin
		scanflagtable[n].time=scans[i].time
		scanflagtable[n].reason='FDL_LIMIT'
		scanflagtable[n].item=parms[0]
		n=n+1
	endif
	scans[i].fdlpos(genconfig.refstation-1)=-min(scans[i].fdlpos)
endfor
if n gt 0 then flagscandata,scanflagtable[0:n-1],flag=1
endif
;
freememory
GeoInfo=replicate(GeoParms,1)
GenInfo=replicate(GenConfig,1)
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1	; True: interferometry loaded
ds_options.v2=1
ds_options.tp=1
ds_options.ta=1
;
end
