pro fitchisq,chisqfiles,component=component,file=localfile,sigma=sigma,cf=cf
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Given a list of YYYY-MM-DD.chisq files (from gridchisq), read these and
; fit the (global) minimum position including an error ellipse and write
; results to file fitchisq.psn. Sigma is the desired increase in total chisq,
; and the ellipse is fit to the corresponding contour. The factor cf allows
; to account for correlated data and reduces the number of degrees of freedom.
; Correlations may involve adjacent channels depending on the calibration.
;
; The ellipse orientation follows the astronomical rule: +RA to the left!
;
; Formal errors correspond to sigma=1 and cf=1 (default).
;
; The gridchisq file must have fine sampling, use option chi2map with gridchisq.
; More reliable is the error ellipse computed by cleanbeam(dirtybeam(star))!
;
; An astrometry file (*.psn) can be specified via file, which evaluates
; the Chi^2 value nearest the position given for the epoch in the localfile.
; This allows to pick a (local) minimum closest to the local astrometry.
; Specify the component if there is more than one when using localfile.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common MarquardtFit,fit_options
;
debug=1
;
if n_elements(chisqfiles) eq 0 then begin
	chisqfiles='????-??-??.chisq'
	chisqfiles='*.chisq'
	print,'Processing all "*.chisq" files...'
endif
;
chisq_files=file_search(chisqfiles)
if strlen(chisq_files[0]) eq 0 then begin
	print,'GRIDPSN: Files not found!'
	retall
endif
for i=0,n_elements(chisq_files)-1 do begin
	r=strsplit(chisq_files[i],'.',/extract)
	if r[n_elements(r)-1] ne 'chisq' then begin
		print,'Error: input files need to have extension "chisq"!'
		retall
	endif
endfor
;
if n_elements(component) eq 0 then component='A-B'
;
if n_elements(localfile) eq 0 then local=0 else local=1
;
if local then begin
	local_file=file_search(localfile)
;	Check extension
	if strpos(local_file[0],'psn') eq -1 then begin
		print,'Error: file does not have .psn extension!'
		return
	endif
	if strlen(local_file[0]) eq 0 then begin
		print,'Local astrometry file not found!'
		return
	endif
	load_astrometry,local_file[0]
	index=where(positions.component eq component)
	local_positions=positions[index]
	r_local=sin(local_positions.theta)*local_positions.rho
	d_local=cos(local_positions.theta)*local_positions.rho
endif
if n_elements(sigma) eq 0 then sigma=1
if n_elements(cf) eq 0 then cf=1 else cf=float(cf)
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
fit_options.tolerance=1e-7
;
openw,unit,'fitchisq.psn',/get_lun
printf,unit, $
'!Comp Julian Year  Rho   Theta    Major  Minor   PA     Rho_err  Theta_err Chi2
;
rad=180/!pi
ellipse_options=alloc_ellipse_options()
ellipse_options.component=component
ellipse_options.a=1
ellipse_options.b=1
ellipse_options.p=1
positions=replicate({component:'',rho:0.0,theta:0.0},8)
positions.component=component
;
n=n_elements(chisq_files)
ny=long(sqrt(n))
nx=nint(n/float(ny))
if nx*ny lt n then nx=nx+1
!p.multi=0
!p.multi=[0,nx,ny]
!p.multi=[0,4,4,0,0]
if debug then !p.multi=[0,2,2,0,1]
;
for i_file=0,n_elements(chisq_files)-1 do begin
	print,''
	print,'=========== Analyzing file '+chisq_files[i_file]+' ==========='
	date=''
	restore,chisq_files[i_file]; restores n_deg_free,chisq,rho,theta,date
;
	if strlen(date) eq 0 then date=strmid(chisq_files[i_file],0,10) ; old method
	parsedate,date,y,m,d
	midnight=system_config('NPOI','MIDNIGHT')
	midnight=system_config('VLTI','MIDNIGHT')
	d=d+midnight/24
	jy=jd2jy(julian(y,m,d))
;	Defaults
	emajor=1.0
	eminor=emajor
	pa=0
	rho_error=0
	theta_error=0
;
	n_deg_free=nint(n_deg_free/cf)	; cf reduces the number of data points
	min_red_chisq=min(chisq)
	index=where(chisq eq min_red_chisq) & index=index[0]
	ij=whereindex(index,chisq)
	print,'Minimum chi^2 found at rho= ', $
		strtrim(string(rho[index]),1), $
		', theta= ',strtrim(string(theta[index]),1)
	x=sin(theta[index]/rad)*rho[index]
	y=cos(theta[index]/rad)*rho[index]
	print,'Minimum chi^2 found at X [mas]= ', $
		strtrim(string(-float(x)),1), $
		', Y [mas]= ',strtrim(string(float(y)),1)
	print,'Minimum reduced chi^2: ',min_red_chisq
	print,'Number of degrees of freedom:',n_deg_free
	print,'Normalize minimum chi^2 to 1'
	chisq=chisq/min_red_chisq
;	print,'Minimum total chi^2: ',min_red_chisq*n_deg_free
;
	r=sin(theta/rad)*rho
	d=cos(theta/rad)*rho
	cellsize=float(r[0]-r[1])
	rr=r[*,0]
	dd=d[0,*] & dd=reform(dd)
	csr=median(rr-shift(rr,-1))
	csd=median(dd-shift(dd,1))
	if nint(csr*1e6) ne nint(csd*1e6) then begin
		print,'Error: different cellsize in x and y directions!'
		return
	endif
;
	if local then begin
;		Find the grid cell closest to the local astrometry
		i=where(abs(local_positions.jy-jy) lt 1.0/(365.25*24),count)
		i=i[0]
		if count eq 0 then begin
			print,'Julian Year not found!'
			continue
		endif
		dr=r_local[i]-r
		dd=d_local[i]-d
		dist_local=sqrt(dr^2+dd^2)
		j=where(dist_local eq min(dist_local)) & j=j[0]
		min_dist=dist_local[index]
		print,'Local chi^2 found at rho= ',rho[j],', theta= ',theta[j]
		print,'Local chi^2: ',chisq[j],', min(dist)=',min_dist
		print,'============================================'
;		ij is the index of the minimum of chisq: chis(ij(0),ij(1))
		ij=whereindex(j,chisq)
	endif
;
; Attempt to fit ellipse to contour at sigma, map size should be less than 2 mas
	dims=size(chisq,/dim)
;
	chisq=chisq*n_deg_free	; Convert to total chisq,
;				  formal error now corresponds to increase by 1
	print,'Converted to total chi^2:'
       	print,'     Maximum='+strtrim(string(max(chisq)),1)
	print,'     Minimum='+strtrim(string(min(chisq)),1)
	min_chisq=min(chisq)
	c0=1			; Default sigma, used w/polynomial fit
	max_chisq=min_chisq+c0	; Correct, though test shows chi^2(red.)>3
	max_chisq=max(chisq)	; Start with highest value, checking edges
;
;	Closed contour check
	if min(chisq[0,*]) le max_chisq then max_chisq=min(chisq[0,*])
	if min(chisq[*,0]) le max_chisq then max_chisq=min(chisq[*,0])
	print,"Maximum chi^2 of closed cont's.: ",max_chisq
	if (max_chisq-min_chisq) ge sigma then begin
		max_chisq=min_chisq+sigma
		print,'Maximum chi^2 contour to fit (min + sigma): ',max_chisq
	endif else begin
		print,'***Requested contour too high:',min(chisq)+sigma
		max_chisq=min_chisq+(max_chisq-min_chisq)*0.25	; Use 25%
		print,'Maximum chi^2 contour allowed (min/max=25%): ',max_chisq
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files[i_file]+' (contour high)' else $
		bad_files=[bad_files,chisq_files[i_file]+' (contour high)']
		print,''
		continue
	endelse
;
;	"r": right ascension ("left, right")
;	"d":     declination ("down, up")
	r_ell=fltarr(8)	; left, right, down, up, low.l., up.r., up.l., low.r.
	d_ell=fltarr(8)
; 	Walk from minimum along the axes and diagonals
	mflag=0	; set to 1 if walk hits edge of map
;	Cardinal axes: left, right, down, up
	last_chisq=0
	for i=1,dims[0]/2 do begin	; "left"
		curr_chisq=chisq[(ij[0]-i)>0,ij[1]]
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if i gt ij[0] then mflag=1
	r_ell[0]=r[(ij[0]-i)>0,ij[1]]-r[ij[0],ij[1]]
	d_ell[0]=d[(ij[0]-i)>0,ij[1]]-d[ij[0],ij[1]]
	last_chisq=0
	for i=1,dims[0]/2 do begin	; "right"
		curr_chisq=chisq[(ij[0]+i)<dims[0]-1,ij[1]]
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if ij[0]+i ge dims[0] then mflag=1
	r_ell[1]=r[(ij[0]+i)<dims[0]-1,ij[1]]-r[ij[0],ij[1]]
	d_ell[1]=d[(ij[0]+i)<dims[0]-1,ij[1]]-d[ij[0],ij[1]]
	last_chisq=0
	for j=1,dims[1]/2 do begin	; "down"
		curr_chisq=chisq[ij[0],(ij[1]-j)>0]
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if j gt ij[1] then mflag=1
	r_ell[2]=r[ij[0],(ij[1]-j)>0]-r[ij[0],ij[1]]
	d_ell[2]=d[ij[0],(ij[1]-j)>0]-d[ij[0],ij[1]]
	last_chisq=0
	for j=1,dims[1]/2 do begin	; "up"
		curr_chisq=chisq[ij[0],(ij[1]+j)<dims[1]-1]
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if ij[1]+j ge dims[1] then mflag=1
	r_ell[3]=r[ij[0],(ij[1]+j)<dims[1]-1]-r[ij[0],ij[1]]
	d_ell[3]=d[ij[0],(ij[1]+j)<dims[1]-1]-d[ij[0],ij[1]]
;	Diagonals
	for i=1,dims[0]/2 do begin	; "lower-left"
		j=i
		curr_chisq=chisq[(ij[0]-i)>0,(ij[1]-j)>0]
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij[0] then mflag=1
	if j gt ij[1] then mflag=1
	r_ell[4]=r[(ij[0]-i)>0,(ij[1]-j)>0]-r[ij[0],ij[1]]
	d_ell[4]=d[(ij[0]-i)>0,(ij[1]-j)>0]-d[ij[0],ij[1]]
	for i=1,dims[0]/2 do begin	; "upper-right"
		j=i
		curr_chisq=chisq[(ij[0]+i)<dims[0]-1,(ij[1]+j)<dims[1]-1]
		if curr_chisq ge max_chisq then break
	endfor
	if ij[0]+i ge dims[0] then mflag=1
	if ij[1]+j ge dims[1] then mflag=1
	r_ell[5]=r[(ij[0]+i)<dims[1]-1,(ij[1]+j)<dims[1]-1]-r[ij[0],ij[1]]
	d_ell[5]=d[(ij[0]+i)<dims[1]-1,(ij[1]+j)<dims[1]-1]-d[ij[0],ij[1]]
	for i=1,dims[1]/2 do begin	; "upper-left"
		j=i
		curr_chisq=chisq[(ij[0]-i)>0,(ij[1]+j)<dims[1]-1]
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij[0] then mflag=1
	if ij[1]+j ge dims[1] then mflag=1
	r_ell[6]=r[(ij[0]-i)>0,(ij[1]+j)<dims[1]-1]-r[ij[0],ij[1]]
	d_ell[6]=d[(ij[0]-i)>0,(ij[1]+j)<dims[1]-1]-d[ij[0],ij[1]]
	for i=1,dims[1]/2 do begin	; "lower-right"
		j=i
		curr_chisq=chisq[(ij[0]+i)<dims[0]-1,(ij[1]-j)>0]
		if curr_chisq ge max_chisq then break
	endfor
	if ij[0]+i ge dims[0] then mflag=1
	if j gt ij[1] then mflag=1
	r_ell[7]=r[(ij[0]+i)<dims[0]-1,(ij[1]-j)>0]-r[ij[0],ij[1]]
	d_ell[7]=d[(ij[0]+i)<dims[0]-1,(ij[1]-j)>0]-d[ij[0],ij[1]]
;
	if local then begin
		if min_dist gt 1 then mflag=1
	endif
;
	if mflag then begin
		print,''
		print,'***Warning: Contour outside map! Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files[i_file]+' (contour outside)' else $
		bad_files=[bad_files,chisq_files[i_file]+' (contour outside)']
		wait,1
	endif else begin
;		Fit error ellipse to contour at sigma
		positions.rho=sqrt(r_ell^2+d_ell^2)
		positions.theta=atan(r_ell,d_ell)
		i=where(positions.rho eq max(positions.rho)) & i=i[0]
		pa=positions[i].theta
		emajor=positions[i].rho	; semi-major axis
		i=where(positions.rho eq min(positions.rho)) & i=i[0]
		eminor=positions[i].rho	; semi-minor axis
		e_parms=[0,0,emajor,eminor,pa]
		ellipse_options.c=1	; fit center 1/0
;
		print,'Fitting ellipse to contour at min.+sigma...'
		fitellipse ; & e_parms0=e_parms
;
		if debug then begin
			!x.range=[max(r_ell),min(r_ell)]
			xl=!x.range[0]-!x.range[1]
			!y.range=[min(d_ell),max(d_ell)]
			yl=!y.range[1]-!y.range[0]
			if yl gt xl then !x.range=!x.range*yl/xl
			if yl lt xl then !x.range=!x.range*xl/yl
			if i_file eq 0 then window,xsize=800,ysize=700
			plotellipse,0
			j=sort(positions.theta)
			oplot,[r_ell[j],r_ell[j[0]]],[d_ell[j],d_ell[j[0]]], $
				psym=6,thick=3
		endif
		emajor=e_parms[2]
		eminor=e_parms[3]
		pa=(e_parms[4]*rad) mod 180
		if eminor gt emajor then begin
			v=emajor
			emajor=eminor
			eminor=v
			pa=(pa+90) mod 180
		endif
		if pa lt 0 then pa=pa+180
		rho_error=0
		theta_error=0
;
		printf,unit,'! '+date+', ellipse at sigma'
		printf,unit,' '+component,jy, $
      			rho[index],theta[index], $
			emajor,eminor,pa, $
			rho_error,theta_error,min(chisq), $
		format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       		'2x,f7.3,1x,f8.3,1x,f7.1)'
;
	endelse	; mflag
endfor
;
free_lun,unit
spawn,'cat fitchisq.psn'
;
print,'------------------------------'
if n_elements(bad_files) gt 0 then begin
	print,'Failed to analyze these files:'
	for i=0,n_elements(bad_files)-1 do print,bad_files[i]
endif
if local then positions=local_positions
;
end
