pro fitmodel,chisq,ndata,error=error
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Fit model of hierarchical stellar system to combined data sets.
; Note: if a file "Z_h_steps.xdr" exists, use it (fit_options.h=1) to
; read parameter increments for the computation of the numerical derivatives.
; The parameter increments will be stored in gen/star/binary_error structures.
; Please see procedure setincrements for how to create this file.
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
;
; Check data
if ds_options.i and n_elements(scans) eq 0 then begin
	print,'***Error[FITMODEL]: no interferometric data!'
	return
endif
if ds_options.a and n_elements(positions) eq 0 then begin
	print,'***Error[FITMODEL]: no astrometric data!'
	return
endif
if ds_options.s and n_elements(velocities) eq 0 then begin
	print,'***Error[FITMODEL]: no radial velocities!'
	return
endif
;
; Check parameters
icheck=0
if n_elements(parameters) eq 0 then icheck=1 else $
if fix(total(strlen(parameters.names))) eq 0 then icheck=1
if icheck then begin
	print,'***Error[FITMODEL]: no modelfit parameters defined!'
	return
endif
;
; Check model
icheck=0
for i=0,n_elements(binary_model)-1 do begin
	if binary_model[i].method eq 1 then begin
		if binary_model[i].semimajoraxis eq 0 then icheck=1
	endif
	if binary_model[i].method eq 2 then begin
		if binary_model[i].rho eq 0 then icheck=1
	endif
endfor
if icheck then begin
	print,'***Error (FITMODEL): errors in parameter values!'
	return
endif
;
; Initialize parameter array a with hierarchical model
modelparmcopy,-1,a
ma=n_elements(a)
ia=lonarr(ma)+1
mfit=ma
if mfit eq 0 then begin
	print,'***Error[FITMODEL]: no fit parameters selected!'
	return
endif
;
; Set data.
;!quiet=1
calcmodel,/quiet
marquardtdata,y,ysig,ymod
ndata=n_elements(y)
if ndata eq 1 and ysig[0] eq 0 then ndata=0
if ndata lt ma then begin
	print,'***Error[FITMODEL]: not enough data!'
	return
endif
x=dindgen(ndata)
;
; If this keyword is set, use the parameter steps computed by modelparmcopy
if fit_options.h then begin
	file=file_search('Z_h_steps.xdr')
	if strlen(file) eq 0 then begin
		print,'Error: Z_h_steps.xdr not found!'
		return
	endif else begin
		restore,'Z_h_steps.xdr'
		print,'Using optimized steps in Z_h_steps.xdr.'
	endelse
endif
;
IF keyword_set(error) THEN BEGIN
;
marquardt,'modfuncs',x,y,ysig,ndata,a,ia,ma,chisq,covar,/error
if ndata-mfit ne 0 then chisq=chisq/(ndata-mfit)
;
ENDIF ELSE BEGIN
;
; Run marquardt alogrithm
marquardt,'modfuncs',x,y,ysig,ndata,a,ia,ma,chisq,covar
if ndata-mfit ne 0 then chisq=chisq/(ndata-mfit)
;
if total(abs(covar)) gt 0 then begin
	print,'Normalized covariance matrix:'
	print,'Selected parameters in column-wise order!'
;
;	Establish list order (index)...
	parameters_bck=parameters
	np=n_elements(parameters)
	index=intarr(np)-1
;
	n=0
	p_len=strlen(parameters.component)
	i=where(p_len ge strlen('System A'),ni)
	if ni gt 0 then begin
	if n_elements(nameparse(parameters[i[0]].component,'-')) eq 1 then begin
		index[n]=i[0]
		n=n+1
	endif
	endif
	i=where(p_len eq 1,ni)
	j=sort(parameters[i].component)
	for k=0,ni-1 do begin
		index[n]=i[j[k]]
		n=n+1
	endfor
	i=where(strpos(parameters.component,'-') ge 0,ni)
	if ni gt 0 then begin
		j=sort(strlen(parameters[i].component))
		for k=0,ni-1 do begin
			index[n]=i[j[k]]
			n=n+1
		endfor
	endif
;
;	Sort the parameters
	parameters=parameters[index[where(index ge 0)]]
;
	parms=strarr(ma)
	b=blanks(5)
	print,b,format='(1x,a5,$)'
	k=0
	for i=0,n_elements(parameters)-1 do begin
		j=where(strlen(parameters[i].names) ne 0,n)
		for j=0,n-1 do begin
			print,parameters[i].names(j),format='(1x,a5,$)'
			parms[k]=strmid(parameters[i].names(j),0,5)
			k=k+1
		endfor
	endfor
	print,''	; to start newline
	;
	b='-----'
	bs=strarr(ma)
	for i=1,ma-1 do bs[i]=bs[i-1]+b+'-'
;	Uncomment to swap lines for blanks
;	bs=''
	for i=0,ma-1 do begin
		j=indgen(ma-i)
		values=string(fltarr(ma-i),format='(100(1x,f5.2))')
		covar_ij=sqrt(covar[j+i,j+i]*covar[i,i])
		if total(covar_ij) ne 0 then $
		values=string(covar[j+i,i]/covar_ij,format='(100(1x,f5.2))')
		chars=strarr(6-strlen(parms[i])) & chars[*]=' '
;
;		Swap comments to swap blanks for lines
		print,parms[i]+strjoin(chars),bs[i],values
;		print,parms(i)+blanks(5-strlen(parms(i))),bs,values
;		bs=bs+b+' '
	endfor
	parameters=parameters_bck
endif else begin
	print,'Ill defined parameters for this model!'
	return
endelse
;
; Store new parameters into hierarchical model
modelparmcopy,1,a
calcmodel,/quiet
;!quiet=0
;
ENDELSE
;
; Store diagonal elements of covariance matrix
; ...save the model first, and replace with model errors
gen_model_bck=gen_model
gen_model=gen_error
star_model_bck=star_model
star_model=star_error
binary_model_bck=binary_model
binary_model=binary_error
; ...clear all floats
gen_model.rv=0
for i=0,n_tags(star_model)-1 do $
	if size(star_model.(i),/type) eq 5 $
	or size(star_model.(i),/type) eq 4 then star_model.(i)=0
for i=0,n_tags(binary_model)-1 do $
	if size(binary_model.(i),/type) eq 5 $
	or size(binary_model.(i),/type) eq 4 then binary_model.(i)=0
; ...extract diagonal elements of covariance matrix
index=indgen(n_elements(a))
e=sqrt(covar[index,index]*chisq)	; Scale with sqrt(chisq)!
modelparmcopy,1,e
; ...copy model errors, replace model
gen_error=gen_model
gen_model=gen_model_bck
star_error=star_model
star_model=star_model_bck
binary_error=binary_model
binary_model=binary_model_bck
;
; As of version 8.10 (July 2014)
print,'Errors scaled with sqrt(chisq)!'
;
print,'FitModel complete.'
;
end
