pro fitnights,plotselection=plotselection,chi2map=chi2map,sigma=sigma,cf=cf
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Fit interferometry (usually including separation and position angle,
; possibly with diameters and magnitude differences) to single/multiple nights,
; obtaining initial estimates for rho and theta from the orbit model.
;
; The fit parameters must be selected through the widgets (FIT|Interferometry)!
;
; Combines different configurations of the same night into a single fit.
; The reference epoch for the astrometry corresponds to the MIDNIGHT value for
; each interferometer.
;
; plotselection: if true, process only the nights selected in the plot widget
;
; chi2map: if true, use previously created gridchisq maps (YYYY-MM-DD.chisq)
; to fit error ellipses to the local minima. The maps should have a pixel
; size of 0.01 mas and 100 x 100 or more pixels.
;
; Parameters sigma (default: 1) and cf (default: 7) are passed to fitchisq.
; cf is the correlation factor. For NPOI, we assume a full correlation of the
; first 7 channels (all others are usually not included in the fitting process)
; Selections in papers: 73 Leo sigma=1, cf=10; Alpha Dra sigma=4, cf=1
;
; Note: error ellipse axes are semi-major/minor axes!
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common LocalFitNights,cbcf
;
RAD=180/pi_circle
;
index=where(parameters.names[0] eq 'Rho' or parameters.names[0] eq 'Theta',n)
if n eq 0 then begin
	print,'Neither Rho nor Theta selected to fit!'
	return
endif
for i=0,n-1 do begin
	j=where(binary_model.component eq parameters[index[i]].component,nj)
	if binary_model[j].method eq 1 and binary_model[j].period eq 0 then $
	begin
		print,'Error: selected component has no orbit defined!'
		return
	endif
endfor
;
if n_elements(cbcf) eq 0 then cbcf=7.0
;
; Passed to fitchisq:
if n_elements(cf) eq 0 then cf=7.0 else cf=float(cf)
if n_elements(sigma) eq 0 then sigma=1.0 else sigma=float(sigma)
;
; Save geoinfo, geninfo, and bufferinfo
geoinfo_bck=geoinfo
geninfo_bck=geninfo
if n_elements(bufferinfo) gt 1 then bufferinfo_bck=bufferinfo
list_nights,ds_nights_bck
;
if keyword_set(plotselection) then begin
; 	Here the nights are selected in the plot widget
	if n_elements(ds_nights) eq 0 then plotinit=init_plot('amoeba','pt')
	if set_nightsel() ne 0 then return
	nfit=n_elements(ds_nights)
endif else begin
	files=''
	if n_elements(bufferinfo) gt 1 then $
	files=bufferinfo.file
	dates=geninfo.date
	udates=unique(dates)
	nfit=n_elements(udates)
endelse
;
; Save other current settings
methods=binary_model.method
binary_model_bck=binary_model
weight_i=ds_options.i
weight_s=ds_options.s
weight_p=ds_options.p
weight_a=ds_options.a
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
chifr=fit_options.chifr
;
; Assume all components w/defined orb. elements are used to compute (rho,theta)
index=where(binary_model.period ne 0,count)
if count gt 0 then binary_model[index].method=1
;
; Make sure we find the minimum
fit_options.chifr=0.001
;
; Make sure modelchisq() uses only interferometry
ds_options.i=1
ds_options.s=0
ds_options.p=0
ds_options.a=0
;
; Make backup of parameters
parameters_bck=parameters
;
; Open results file
openw,unit,'fitnights.psn',/get_lun
printf,unit, $
'!                                 Semi-  Semi-                                '
printf,unit, $
'!Comp Julian Year  Rho   Theta    Major  Minor   PA     Rho_err  Theta_err Chi2
;
for i=0,nfit-1 do begin
	if keyword_set(plotselection) then begin
		print,ds_nights[i]
		k=where(geninfo_bck.date+' ' $
		       +geoinfo_bck.systemid+' ' $
		       +geninfo_bck.configid eq ds_nights[i],count)
;		This ensures only this data set is used
		geoinfo=geoinfo_bck[k]
		geninfo=geninfo_bck[k]
		bufferinfo=bufferinfo_bck[k]
;		Calcmodel does not load data if only one night is in buffer
;		Only read from buffer if mode is AMOEBA
		if count eq 1 and n_elements(bufferinfo_bck) gt 1 $
		then loadnight,ds_nights[i]
	endif else begin
		k=where(dates eq udates[i],count)
		print,'Date = ',udates[i],', files = ',count
		geoinfo=geoinfo_bck[k]
		geninfo=geninfo_bck[k]
		if n_elements(bufferinfo_bck) gt 1 then $
		bufferinfo=bufferinfo_bck[k]
		ds_nights=ds_nights_bck[k]
;		Calcmodel does not load data if only one night is in buffer
;		Only read from buffer if mode is AMOEBA
		if count eq 1 and n_elements(bufferinfo_bck) gt 1 $
		then loadnight,ds_nights[0]
	endelse
	printf,unit,'!     '+date
	midnight=system_config(systemid,'MIDNIGHT')
	parsedate,geninfo[0].date,y,m,d
	d=d+midnight/24
	epoch_jd=julian(y,m,d)
	epoch_jy=jd2jy(epoch_jd)
;	Obtain predictions for (rho,theta) for this night, requires method=1
	for j=0,num_binary()-1 do begin
		rt=binarypos(epoch_jd,binary_model[j].component)
		binary_model[j].rho=rt[0]
		binary_model[j].theta=rt[1]
	endfor
	if binary_model[0].method eq 4 then begin
;		TBC (used only in UX Ari paper
;		parameters.names=['Semimajoraxis','Ascendingnode']
	endif else begin
		binary_model.method=2
	endelse
;	For those components which have orbital elements include orb. motion
	index=where(methods eq 1,count)
	if count gt 0 then binary_model[index].method=3
;	Fit model
	fitmodel,chisq,ndata
;	writemodel,geninfo(0).date+'.model.xdr'
	parameters=parameters_bck
	binary_model.method=methods
;
;	If successful, compute error ellipse and write results
	IF ndata ne 0 THEN BEGIN
;
	locations=replicate(location(),num_binary())
;
	FOR j=0,num_binary()-1 DO BEGIN
;	Don't print components which were not fit
	IF (binary_error[j].rho NE 0 AND $
	    binary_error[j].theta NE 0) OR $
	   (binary_error[j].semimajoraxis NE 0 AND $
	    binary_error[j].ascendingnode) THEN BEGIN
;
;	Compute CLEAN beam size
	print,'------------Computing CLEAN beam size------------'
	if keyword_set(plotselection) then begin
		db=dirtybeam(gen_model.starid)
	endif else begin
		if n_elements(bufferinfo_bck) gt 1 then $
		loadnight,ds_nights[0]
		db=dirtybeam(gen_model.starid)
		for k=1,n_elements(ds_nights)-1 do begin
			loadnight,ds_nights[k]
			db=db+dirtybeam(gen_model.starid)
		endfor
		db=db/n_elements(ds_nights)
	endelse
	rms=abs(min(db)/max(db))	; This does not consider vis. RMS
	e_parms=cleanbeam(db)
	if e_parms[0] lt 0 then begin
;		Clean-up before returning
		if nfit gt 1 then binary_model=binary_model_bck
		geoinfo=geoinfo_bck
		geninfo=geninfo_bck
		if n_elements(bufferinfo_bck) gt 1 then $
		bufferinfo=bufferinfo_bck
		ds_options.i=weight_i
		ds_options.s=weight_s
		ds_options.p=weight_p
		ds_options.a=weight_a
		fit_options.chifr=chifr
	endif
	e_parms[0]=e_parms[0]/2		; Convert to semi-major axis
	e_parms[1]=e_parms[1]/2		; Convert to semi-minor axis
	e_parms[2]=e_parms[2]/RAD
;
	if binary_model[0].method ne 4 then binary_model.method=2
	locations[j].jy=epoch_jy
	locations[j].component=binary_model[j].component
;	These should be the photocenter coordinates
	rt=binarypos(epoch_jd,binary_model[j].component)
	locations[j].rho=rt[0]
	locations[j].theta=rt[1]/RAD
	if locations[j].theta lt 0 then locations[j].theta= $
					locations[j].theta+2*pi_circle
;
;	In the following, we try different methods to estimate
;	the uncertainty ellipse
	locations[j].pa=e_parms[2]
;
;	Derive from CLEAN beam, reduction factor from map RMS
;	Fomalont, Image Analysis, in Perley et al., Synth. Im. Rad. Ast., p.219
	dsp=ds_options.tp
	dsa=ds_options.ta
	ds_options.ta=-1
	ds_options.tp=-1
	marquardtdata,y,ysig,ymod
	ds_options.tp=dsp
	ds_options.ta=dsa
	rms=stddev(y-ymod)/sqrt(2)
	f=2.0/rms
	locations[j].emajor=e_parms[0]/f
	locations[j].eminor=e_parms[1]/f
;
;	Adjust uncertainty ellipse to equal an increase
;	in total chisq by 1*reduced chisq
	x=locations[j].rho*sin(locations[j].theta) $
	 -locations[j].emajor*sin(locations[j].pa)
	y=locations[j].rho*cos(locations[j].theta) $
	 -locations[j].emajor*cos(locations[j].pa)
	binary_model[j].rho=sqrt(x^2+y^2)
	binary_model[j].theta=atan(x,y)*RAD
	calcmodel,/quiet
	a=abs(modelchisq()-chisq)/locations[j].emajor^2
	f1=locations[j].emajor/sqrt((max([chisq,1])/ndata)/a)
	x=locations[j].rho*sin(locations[j].theta) $
	 -locations[j].eminor*cos(locations[j].pa)
	y=locations[j].rho*cos(locations[j].theta) $
	 +locations[j].eminor*sin(locations[j].pa)
	binary_model[j].rho=sqrt(x^2+y^2)
	binary_model[j].theta=atan(x,y)*RAD
	calcmodel,/quiet
	a=abs(modelchisq()-chisq)/locations[j].eminor^2
	f2=locations[j].eminor/sqrt((max([chisq,1])/ndata)/a)
	f=sqrt(f1*f2)	; maintain beam ratio
	locations[j].emajor=e_parms[0]/f
	locations[j].eminor=e_parms[1]/f
;
;	All of the above may return too optimistic errors
	f=5.0	; Standard radio CLEAN beam/5 criterion
	f=7.0	; Used for Eta Virginis NPOI data
	f=22.0	; Used for UX Ari CHARA data
	f=40.0	; Used in Mizar A NPOI data
	f=40.0	; Used for 41 Eri PIONIER data
	f=cbcf  ; User user-defined conversion factor
	locations[j].emajor=e_parms[0]/f
	locations[j].eminor=e_parms[1]/f
;
;	Grid fit, if requested (not triples or higher)
;	Use fitchisq procedure also to extract error ellipse
	if n_elements(chi2map) ne 0 then begin
	chisqfile=Date+'.chisq'
	fitchisq,chisqfile,sigma=sigma,cf=cf,component=binary_model[j].component
	status=dc_read_free('fitchisq.psn',c,jy,r,t,ea,eb,pa,/col,ignore=['!'])
	locations[j].emajor=ea;/sqrt(ndata)
	locations[j].eminor=eb;/sqrt(ndata)
	locations[j].pa=pa/RAD
	endif
;
	r=ellipse(locations[j].emajor, $
	          locations[j].eminor, $
	          locations[j].theta-e_parms[2])
	factor=(locations[j].rho/100)/r
	if factor gt 1 and geoinfo[0].SystemId eq 'not_obsolete' then begin
		d_pa=e_parms[2]-locations[j].theta
		if d_pa lt 0 then d_pa=d_pa+!pi
		new_parms=apollonius(locations[j].emajor, $
				     locations[j].eminor, $
				     d_pa,factor)
		locations[j].emajor=new_parms[0]
		locations[j].eminor=new_parms[1]
		locations[j].pa=(new_parms[2]+locations[j].theta) mod !pi
	endif
;
; 	Write results to file
	printf,unit,locations[j].component,locations[j].jy, $
	      	locations[j].rho,locations[j].theta*RAD, $
		locations[j].emajor,locations[j].eminor,locations[j].pa*RAD, $
		binary_error[j].rho,binary_error[j].theta,chisq, $
		format='(a5,1x,f10.5,1x,f7.3,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
		       	'3x,f7.3,1x,f8.3,1x,f6.1)'
;
	binary_model.method=methods
	ENDIF
	ENDFOR
	ENDIF
print,'-------------------------------------------------'
print,''
;
endfor
;
; Close file
free_lun,unit
;
; Print information about CBCF used
print,'CLEAN beam conversion factor used: ',cbcf
;
; Set some of the parms back to previous values
if nfit gt 1 then binary_model=binary_model_bck
geoinfo=geoinfo_bck
geninfo=geninfo_bck
if n_elements(bufferinfo_bck) gt 1 then $
bufferinfo=bufferinfo_bck
; if n_elements(files) eq 0 then begin
; 	geoinfo=geoinfo_bck
; 	geninfo=geninfo_bck
; 	bufferinfo=bufferinfo_bck
; endif
;
ds_options.i=weight_i
ds_options.s=weight_s
ds_options.p=weight_p
ds_options.a=weight_a
fit_options.chifr=chifr
;
end
