pro get_genconfig,stations
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Read GenConfig object in HDS data file. Store a copy in GenInfo.
; If file not open, or stations are defined, close file and create
; object from inputs.
;
; GenConfig fields and their meaning for NPOI:
;
; Date			Not the "data date", but UT date of the night.
;			This date field is set when reading data.
; BeamCombinerID	1: 3-way, 2: 6-way, 3: 4-way astrometry.
; InstrCohInt		Instrumental coherent integration time in ms, i.e. 2 ms.
; RefStation		Reference station, first = 1.
; NumBin		Number of phase bins., e.g. 8 (3-way) or 64 (6-way).
;
; INPUT BEAM
;
; NumSid	Number of siderostats used.
; StationID	AC0,AE0,AN0,AW0,N00,..,W01,..,E02,...,: Name of pier station.
; SiderostatID	1,2,...: unique IDs of the siderostats/NATs; NAT data is stored
;		by this ID, not in the order of StationID!
; SidConId	This is the ID of the sidcon rack;
; DelaylineID	1-6: 1 is closest to south wall, furthest from E-room.
; Stroke	Stroke amplitude in m for each FDL.
; LDL ID        1-6: from E to W.
; StarTrackerID	1-6: Quadcell ID used for angle tracking.
; BCInputID	1-6: 1 is western most (FDL side), 6 is east (Manifold side).
;		With 6-way, there is a 1-to-1 correspondence to DelaylineID.
; StationCoord	x,y,z,d: d is delay from station to array center.
;
; OUTPUT BEAM
;
; NumOutBeam		Number of spectrometers.
; SpectrometerID	Hardware ID of spectrometer.
; NumBaseline		Number of baselines in a spectrometer.
; FringeMod		Baseline Modulation, e.g. 1,2,3...
; NumBiasFreq		Number of frequencies used for bias computation
; BiasMod		Bias frequencies
; NumSpecChan		Number of spectral channels in a spectrometer.
; BaselineID		N09-W03,...: baseline names. Orientation sensitive.
; Wavelength		in m, for each channel and spectrometer.
; WavelengthErr
; ChanWidth		Width of spectral channel, in m.
; ChanWidthErr
;
; TRIPLE
;
; NumTriple	Number of computed triple combinations.
; TripleBeam	For each baseline, which spectrometer.
; TripleBase    For each baseline, which baseline.
; TripleChan	For each baseline, channel numbers involved.
; TripleNumChan	For each triple, how many channels.
;
; NPOI array diagram
;
;                                                          E-arm AE
;                                                           /\
;                                                          /  \AE
;                                              AN         /
;                                     N-arm----------- NF/\
;                                             	      3 /6 \AC
;                                         6-pipe      2/ 5
;                                          stack      1 /4
;                                             	     / /
;               S6  S5  S4  S3  S2  S1              / WF
;        LDL 1                                   P1/ /  \AW
;                                                 /|/    \
;        LDL 2                                 P2/ |     W-arm
;                                               /|/|
;        LDL 3  <------------E-------------- P3/ | |
;                ------------W---------------> |/| |
;                                              | | |
;                                             /| | |
;        LDL 4                             P4/ | | |   <- Manifold room area
;                                           /| | | |
;        LDL 5                           P5/ | | | |
;                                         /| | | | |
;        LDL 6                         P6/ | | | | |
;                                        | | | | | |
;                                        6 5 4 3 2 1   -> south
;                                        | | | | | |
;
;
;                Beam combiner            --------6-way--------------   3-way
;                                                                     (before 20
;
;                   /--------------  6     N
;              M3b /-\-------------  5     W07
;                 /-\-\------------  4     W
;                  \ \ \
;                   \ \ \
;                    \ \ \ S1
;              /------\-\-/--------  3     E                              W(4)
;          M1 /-\------\-/-\-------  2     C                              E(3)
;            /-\-\------/-\-\------  1     E02                            C(2)
;             \ \ \      \ \ \
;              \ \ \ S2   \ \ \
;         /-----\-\-/------\-/ \
;    /-----\-----\-/--------/M2 \
;     \  /--\-----/--------------/M3a
;      \  \  \     \
;       \  \  \    (unused)
; Sp. ID 3  2  1                       |
; 3-way CE CW EW (before 2000)         |
; 3-way CW EW CE (2000 and after)      V
; 6-way 12 13 23 (2002 and after)     west
;       45 46 56
;
; All spectrometers have 50 micron pinholes (but see note below)
;
; Note: Sp3 has had 25 micron pinhole in place between August 1, 2000,
;		and September 23, 2002.
;       Sp1 channel 12 is dead (broken fiber; I think beginning in 2000)
;
; Some notes on the array configuration:
; Light is intercepted by siderostats, and dropped vertically by the
; Elevator cans (inside astrometric huts, and on piers close to imaging
; stations). Feedpipes transfer light to array center, where the array
; center cans transfer light into the manifold room. Here they are dropped
; again by the periscopes, and fed into the long delay lines. There are 3
; feed pipes per arm, 9 total. There are six delay pipes, always connected
; to the same delay lines in the laboratory.
;
; In order to compute the path lengths, one file (station file) contains
; the paths from the siderostats to the array center. Another file gives
; for each combination of delayline ID and feedpipe ID the path through
; the array center to the beamcombiner. Another file will give for each
; combination of delayline ID and BeamCombinerInput ID the final pieces
; of path lengths for a specific BeamCombiner.
;
; Since the advent of the 6-way beam combiner and the Huggins fringe engine,
; observers have the capability to switch in and out stations during the night.
; Strictly speaking, these actions correspond to configuration changes which
; are not allowed in a single data container file. However, we have introduced
; a configuration ID index into some GenConfig fields, for now just Bias and
; Response, since the results for these calibration data depend on the
; particular configuration.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; Create object from input parameters if defined, or no file open
if n_elements(stations) gt 0 or hds_state() eq 0 then begin
	hds_close
;	GenConfig valid
	if n_elements(stations) eq 0 and n_elements(size(GenConfig)) eq 4 $
		then stations=GenConfig.StationId[0:GenConfig.NumSid-1]
;	GenConfig invalid
	if n_elements(stations) eq 0 and n_elements(size(GenConfig)) eq 3 $
		then stations=system_config(systemid,'STATIONS')
;
	GenConfig=system_config(systemid,'CONFIG',stations=stations)
	result=size(GenConfig)
	if result[n_elements(result)-2] eq 8 then get_stationtable
	genconfig.diameter=system_config(system_id(systemid),'DIAMETER', $
		stations=genconfig.stationid[0])
	genconfig.configid=instrument_id(systemid)
	return
endif
;
; Change to toplevel in HDS file
toplevel
;
; Get format
get_format,format
;
; Get Date and SystemId
get_datum
get_systemid
;
; Extract configid
words=nameparse(systemid,'/')
if n_elements(words) eq 2 then configid=words[1]
;
; If refstation is defined, do not change it (see exceptions below)
if n_elements(size(GenConfig)) eq 4 then refstation=GenConfig.refstation $
			            else refstation=99
;
; If reading a CONSTRICTOR (pointdata) file, do read the reference station info
if format eq 'CONSTRICTOR' then refstation=99
;
; If reading a COBRA (raw data) file, do read the reference station info
if format eq 'COBRA' then refstation=99
;
; As of Version 8, we always read the refstation
refstation=99
;
; If date field is defined, keep it until changed when data is read
if n_elements(size(GenConfig)) eq 4 then datum=GenConfig.Date $
			            else datum=date
;
; Find dimensioning information and allocate GenConfig
dat_find,'GENCONFIG'
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
MaxSid=NumSid
dat_annul
dat_find,'OUTPUTBEAM'
cmp_get0i,'NUMOUTBEAM',NumOutBeam
cmp_get1i,'NUMBASELINE',NumOutBeam,NumBaseline
MaxBaseline=max(NumBaseline)
cmp_get1i,'NUMSPECCHAN',NumOutBeam,NumSpecChan
MaxSpecChan=max(NumSpecChan)
dat_annul
name='TRIPLE'
dat_there,name,reply
if reply then begin
	dat_find,name
	cmp_get0i,'NUMTRIPLE',NumTriple
	dat_annul
endif else NumTriple=0
MaxNumTriple=max([1,NumTriple])
dat_annul
;
MaxConfig=system_config(SystemId,'MAXCONFIG')
case string(maxConfig) of
	'scans':MaxConfig=scanconfig(/maxconfig)
	'stars':MaxConfig=n_elements(scanconfig(/starlist))
	   else:MaxConfig=fix(MaxConfig)
endcase
if MaxConfig le 0 then begin
	print,'***Error(GET_GENCONFIG): unable to get maxConfig!'
	return
endif
GenConfig=allocgenconfig(MaxSid,NumOutBeam,MaxNumTriple, $
			 MaxBaseline,MaxSpecChan,MaxConfig)
;
; Now load data into temporary variables
;
dat_find,'GENCONFIG'
;
IF format EQ 'INCHWORM' THEN BEGIN
;
cmp_get0i,'NUMPLATE',NumPlate
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
cmp_get1i,'SIDEROSTATID',NumSid,SiderostatId
SidConId=SiderostatId
dat_there,'STATIONID',reply
; This object was only implemented just before Nick left in 2002
if reply eq 1 then begin
	cmp_get1c,'STATIONID',NumSid,StationId
	GenConfig.StationId=StationId
endif
if refstation eq 99 then refstation=1
GenConfig.RefStation=refstation
GenConfig.NumSid=NumSid
GenConfig.SiderostatId=SiderostatId
GenConfig.SidConId=SidConId
GenConfig.NumPlate=NumPlate
print,'GenConfig (INCHWORM) loaded.'
;
ENDIF ELSE BEGIN
;
dat_there,'NUMPLATE',reply
if reply then cmp_get0i,'NUMPLATE',NumPlate
cmp_get0d,'INSTRCOHINT',InstrCohint
cmp_get0i,'BEAMCOMBINERID',BeamCombinerId
dat_there,'NUMBIN',reply
if reply eq 1 then cmp_get0i,'NUMBIN',NumBin else NumBin=8
dat_there,'REFSTATION',reply
if reply eq 1 and refstation eq 99 then cmp_get0i,'REFSTATION',refstation
if refstation eq 99 then refstation=1	; in case reply=0
dat_there,'CONFIGID',reply
if reply eq 1 and n_elements(configid) eq 0 then $
	cmp_get0c,'CONFIGID',configid ; else configid='UNKNOWN'
;
; Get components in OutputBeam table............................................
dat_find,'OUTPUTBEAM'
cmp_get0i,'NUMOUTBEAM',NumOutBeam
cmp_get1i,'NUMBASELINE',NumOutBeam,NumBaseline
cmp_get1i,'NUMSPECCHAN',NumOutBeam,NumSpecChan
cmp_get1c,'SPECTROMETERID',NumOutBeam,SpectrometerId
;
MaxBaseline=max(NumBaseline)
cmp_shape,'BASELINEID',ndim,dims
icheck=0
if dims[0] ne MaxBaseline then icheck=1
if dims[1] ne NumOutBeam then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of BaselineId '+ $
		'inconsistent with declarations!'
	return
endif
cmp_getnc,'BASELINEID',ndim,dims,BaselineId
index=where(strlen(BaselineId) eq 0,count)
if count gt 0 then BaselineId[index]='ST1-ST2'
;
MaxSpecChan=Max(NumSpecChan)
cmp_shape,'WAVELENGTH',ndim,dims
icheck=0
if dims[0] ne MaxSpecChan then icheck=1
if dims[1] ne NumOutBeam then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of Wavelength '+ $
		'inconsistent with declarations!'
	return
endif
cmp_getnd,'WAVELENGTH',ndim,dims,Wavelength
cmp_getnd,'WAVELENGTHERR',ndim,dims,WavelengthErr
cmp_getnd,'CHANWIDTH',ndim,dims,ChanWidth
cmp_getnd,'CHANWIDTHERR',ndim,dims,ChanWidthErr
;
V2Bias=0
name='BIAS'
dat_there,name,reply
if reply eq 1 then begin
	cmp_shape,name,ndim,dims
	cmp_getnd,name,ndim,dims,V2Bias
	if ndim eq 5 then numConfig=dims[ndim-1] $
		     else numConfig=1
	if numConfig gt MaxConfig then begin
		V2Bias=V2Bias[*,*,*,*,0:MaxConfig-1]
		numConfig=MaxConfig
	endif
endif else numConfig=MaxConfig
;
MaxNumBaseline=max(NumBaseline)
cmp_shape,'FRINGEMOD',ndim,dims
icheck=0
if dims[0] ne MaxNumBaseline then icheck=1
if dims[1] ne NumOutBeam then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of FringeMod '+ $
		'inconsistent with declarations!'
	return
endif
cmp_getni,'FRINGEMOD',ndim,dims,FringeMod
;
name='NumBiasFreq'
dat_there,name,reply
if reply then begin
	cmp_get1i,'NUMBIASFREQ',NumOutBeam,NumBiasFreq
	cmp_getni,'BIASMOD',ndim,dims,BiasMod
endif else begin
	NumBiasFreq=0
	BiasMod=0
endelse
dat_annul
;
; Get components in InputBeam table.............................................
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
if n_elements(NumPlate) eq 0 then NumPlate=NumSid
cmp_get1c,'STATIONID',NumSid,StationId
cmp_get1i,'SIDEROSTATID',NumSid,SiderostatId
dat_there,'SIDCONID',reply
if reply eq 1 then cmp_get1i,'SIDCONID',NumSid,SidConId $
	      else SidConId=SiderostatId
if total(abs(SidConId)) eq 0 then SidConId=SiderostatId
cmp_get1i,'DELAYLINEID',NumSid,DelayLineId
cmp_get1r,'STROKE',NumSid,Stroke
if n_elements(Stroke) eq 0 then begin
	Stroke=fltarr(NumSid)
	if SystemId eq 'NPOI' and NumSid eq 3 then Stroke=[0,900d-9,1800d-9]
endif
cmp_get1i,'STARTRACKERID',NumSid,StarTrackerId
cmp_get1i,'BCINPUTID',NumSid,BCInputId
;
; Get station coordinates
StationCoord=dblarr(4,NumSid)
dat_find,'STATIONCOORD'
dat_shape,ndim,dims
icheck=0
if dims[0] ne 4 then icheck=1
if dims[1] ne NumSid then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of STATIONCOORD'+ $
		'inconsistent with declarations!'
	return
endif
dat_getnd,ndim,dims,StationCoord
dat_annul
dat_annul
;
; Get components in Triple table................................................
NumTriple=0
TripleBeam=0
TripleBase=0
TripleChan=0
TripleNumChan=0
TABias=0
name='TRIPLE'
dat_there,name,reply
if reply eq 1 then begin
	dat_find,name
	cmp_get0i,'NUMTRIPLE',NumTriple
	if NumTriple gt 0 then begin
;
		name='BIAS'
		dat_there,name,reply
		if reply eq 1 then begin
			cmp_shape,name,ndim,dims
			cmp_getnd,name,ndim,dims,TABias
			if ndim eq 4 then numConfig=dims[ndim-1] $
		     		else numConfig=1
			if numConfig gt MaxConfig then begin
				TABias=TABias[*,*,*,0:MaxConfig-1]
				numConfig=MaxConfig
			endif
		endif ; else numConfig=MaxConfig
;
		cmp_shape,'OUTPUTBEAM',ndim,dims
		icheck=0
		if ndim ne 2 then icheck=1
		if dims[0] ne 3 then icheck=1
		if dims[1] ne NumTriple then icheck=1
		if icheck eq 1 then begin
		 print,'***Error(GET_GENCONFIG): found dimensions of '+name+ $
				'inconsistent with declarations!'
		 return
		endif
		cmp_getni,'OUTPUTBEAM',ndim,dims,TripleBeam
		cmp_getni,'BASELINE',ndim,dims,TripleBase
;
		cmp_get1i,'NUMSPECCHAN',NumTriple,TripleNumChan
;
		cmp_shape,'SPECCHAN',ndim,dims
		icheck=0
		if ndim ne 3 then icheck=1
		if dims[0] lt Max(TripleNumChan) then icheck=1
	 	if dims[1] ne 3 then icheck=1
		if dims[2] ne NumTriple then icheck=1
		if icheck eq 1 then begin
		 print,'***Error(GET_GENCONFIG): found dimensions of '+name+ $
				'inconsistent with declarations!'
		 return
		endif
		cmp_getni,'SPECCHAN',ndim,dims,TripleChan
		TripleChan=TripleChan[0:max(TripleNumChan)-1,*,*]
	endif
endif
;
; Now store temporary variables into previously allocated structure
;
GenConfig.Date=datum
GenConfig.RefStation=refstation
GenConfig.NumSid=NumSid
GenConfig.NumPlate=NumPlate
GenConfig.InstrCohint=InstrCohint
GenConfig.BeamCombinerId=BeamCombinerId > 1
GenConfig.ConfigId=strcompress(string(GenConfig.BeamCombinerId),/remove_all)
GenConfig.ConfigId=configid
GenConfig.NumBin=NumBin
GenConfig.StationId[0:NumSid-1]=StationId
GenConfig.SiderostatId[0:NumSid-1]=SiderostatId
GenConfig.SidConId[0:NumSid-1]=SidConId
GenConfig.DelayLineId[0:NumSid-1]=DelayLineId
GenConfig.Stroke[0:NumSid-1]=Stroke
GenConfig.BCInputId[0:NumSid-1]=BCInputId
GenConfig.StarTrackerId[0:NumSid-1]=StarTrackerId
GenConfig.StationCoord[*,0:NumSid-1]=StationCoord
GenConfig.NumOutBeam=NumOutBeam
GenConfig.NumBaseline[0:NumOutBeam-1]=NumBaseline
GenConfig.NumBiasFreq[0:NumOutBeam-1]=NumBiasFreq
GenConfig.NumSpecChan[0:NumOutBeam-1]=NumSpecChan
GenConfig.SpectrometerId[0:NumOutBeam-1]=SpectrometerId
GenConfig.BaselineId[0:MaxBaseline-1,0:NumOutBeam-1]=BaselineId
GenConfig.Wavelength[0:MaxSpecChan-1,0:NumOutBeam-1]=Wavelength
GenConfig.WavelengthErr[0:MaxSpecChan-1,0:NumOutBeam-1]=WavelengthErr
GenConfig.ChanWidth[0:MaxSpecChan-1,0:NumOutBeam-1]=ChanWidth
GenConfig.ChanWidthErr[0:MaxSpecChan-1,0:NumOutBeam-1]=ChanWidthErr
GenConfig.V2Bias[*,0:MaxBaseline-1,0:MaxSpecChan-1, $
		   0:NumOutBeam-1,0:numConfig-1]=V2Bias
GenConfig.TABias[*,0:MaxSpecChan-1,0:max([1,NumTriple])-1,0:numConfig-1]=TABias
GenConfig.FringeMod[0:MaxNumBaseline-1,0:NumOutBeam-1]=FringeMod
GenConfig.BiasMod[0:MaxNumBaseline-1,0:NumOutBeam-1]=BiasMod
GenConfig.NumTriple=NumTriple
if NumTriple gt 0 then begin
	GenConfig.TripleBeam[*,0:NumTriple-1]=TripleBeam
	GenConfig.TripleBase[*,0:NumTriple-1]=TripleBase
	GenConfig.TripleNumChan=TripleNumChan
	GenConfig.TripleChan[0:max(TripleNumChan)-1,*,0:NumTriple-1]=TripleChan
endif
genconfig.diameter=system_config(system_id(systemid),'DIAMETER', $
	stations=genconfig.stationid[0])
print,'GenConfig loaded.'
;
; Make sure configid is set to something useful
if strlen(configid) eq 0 then configid=strjoin(StationId,'-')
configid=strict_configid(configid)
;
ENDELSE
;
; Check fringe modulation against strokes
if system_id(systemid) eq 'NPOI' then begin
for i=0,genconfig.numoutbeam-1 do begin
for j=0,genconfig.numbaseline[i]-1 do begin
        i1=where(genconfig.stationid eq strmid(genconfig.baselineid[j,i],0,3))
        i2=where(genconfig.stationid eq strmid(genconfig.baselineid[j,i],4,3))
        if nint((genconfig.stroke[i2]-genconfig.stroke[i1])/1e-6) ne $
           genconfig.fringemod[j,i] and genconfig.beamcombinerid ne 1 then $
        print,'Warning(GET_GENCONFIG): inconsistent modulation on beam ' $
                +string(i+1,format='(i1)') $
                +', baseline '+genconfig.baselineid[j,i]+'!'
endfor
endfor
endif
;
; Return to first level
toplevel
;
; Store a copy in GenInfo after clearing it, for use in AMOEBA
; if n_elements(GenInfo) le 1 then GenInfo=replicate(GenConfig,1)
;
end
