pro get_geoparms,sysid,datum
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Read GeoParms object in HDS file. If no file is open, set defaults
; based on SystemID. If sysid or datum are defined, close files and
; create object from inputs.
;
; Read earth orientation data in file mark3.dat.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
format=''
if hds_state() eq 1 then get_format,format
;
; Default Earth parameters
EarthRadius=6378140.d0
J2=0.00335281d0
;
; If sysid and/or datum are supplied, close file and initialize
if n_elements(sysid) ne 0 or n_elements(datum) ne 0 then hds_close
if n_elements(sysid) ne 0 then systemid=sysid
if n_elements(datum) ne 0 then date=datum
;
get_systemid,systemid
get_datum,date
;
; Create object from input parameters if no file open, or wrong format
IF hds_state() eq 0 or format eq 'INCHWORM' THEN BEGIN
	Longitude=system_config(systemid,'longitude')
	Latitude=system_config(systemid,'latitude')
	Altitude=system_config(systemid,'altitude')
ENDIF
;
; Read from HDS file if one is open
IF hds_state() eq 1 THEN BEGIN
;
toplevel
;
; Get site coordinates. Compare NPOI coordinates to coordinates of
; Perkins 72" telescope: Lon=111:32:09.3, Lat=35:05:48.6, Alt=2198 m
;
if format ne 'INCHWORM' then begin
	dat_find,'GEOPARMS'
	cmp_get0d,'LATITUDE',Latitude
	cmp_get0d,'LONGITUDE',Longitude	; East Longitude!
	cmp_get0d,'ALTITUDE',Altitude
	dat_there,'EARTHRADIUS',reply
	if reply eq 1 then begin
		cmp_get0d,'EARTHRADIUS',EarthRadius
		cmp_get0d,'J2',value
		if value ne 0 then J2=value
	endif
	dat_annul
endif
;
ENDIF
;
; Set TAI-UTC. See file ser7 in directory ser7 on maia.usno.navy.mil.
; If a new leap second is introduced, copy the first line with the proper
; date and TAI-UTC entered to the end of the list below. Note that if, for
; example, a leap second is to be introduced June 30th, the seconds in the
; evening of that day will go 23:59:58, 23:59:59, 23:59:60, 0:00:00, i.e.
; the leap second will be the last second of that day.
;
TAI_UTC=28
parsedate,Date,y,m,d
if julian(y,m,d) ge julian(1994L,7L,1L) then TAI_UTC=29
if julian(y,m,d) ge julian(1996L,1L,1L) then TAI_UTC=30
if julian(y,m,d) ge julian(1997L,7L,1L) then TAI_UTC=31
if julian(y,m,d) ge julian(1999L,1L,1L) then TAI_UTC=32
if julian(y,m,d) ge julian(2006L,1L,1L) then TAI_UTC=33
if julian(y,m,d) ge julian(2009L,1L,1L) then TAI_UTC=34
if julian(y,m,d) ge julian(2012L,7L,1L) then TAI_UTC=35
if julian(y,m,d) ge julian(2015L,7L,1L) then TAI_UTC=36
if julian(y,m,d) ge julian(2017L,1L,1L) then TAI_UTC=37
; Last updated/checked 26 Oct 2023
;
; As of May, 1994, TDT=TAI+32.184s
TDT_TAI=32.184
;
; Get UT1-UTC data from VLBI polar motion measurements provided through
; maia.usno.navy.mil:ser7/mark3.out.
;
; Read one line, see what date it is...
status=dc_read_free(!oyster_dir+'usno/mark3.dat',year,month,day,mjd, $
	pole_x,pole_x_e,pole_y,pole_y_e,ut1_utc,ut1_utc_e,/col, $
	filters=['P'],ignore=['$TEXT_IN_NUMERIC'],nskip=3,nrecs=1)
if status ne 0 then begin
	print,'Warning(GET_GEOPARMS): failed to read UT1-UTC data!'
	mjd=0
endif
; ...and skip ahead to make reading faster
nskip=long(julian(y,m,d)-(mjd+2400000.5d0)) & nskip=nskip[0]
neighborhood=10
nskip=nskip-neighborhood
if nskip lt 0 then nskip=0
spawn,'wc -l '+!oyster_dir+'usno/mark3.dat',r
words=nameparse(r[0])
nlines=long(words[0])
if nskip gt nlines-neighborhood then nskip=0
status=dc_read_free(!oyster_dir+'usno/mark3.dat',year,month,day,mjd, $
        pole_x,pole_x_e,pole_y,pole_y_e,ut1_utc,ut1_utc_e,/col, $
        filters=['P'],ignore=['$TEXT_IN_NUMERIC'],nskip=nskip, $
	nrecs=2*neighborhood,resize=[1,2,3,4,5,6,7,8,9,10])
if status ne 0 then begin
	print,'Warning(GET_GEOPARMS): failed to read UT1-UTC data!'
	mjd=0
endif
;
; Believe it or not, but on 2002-03-12 the predicted ut1_utc_e was zero
index=where(ut1_utc_e eq 0,count)
if count gt 0 then ut1_utc_e[index]=median(ut1_utc_e)
;
; Fit 2nd order polynomials through data.
index=where((mjd+2400000.5d0) eq julian(y,m,d),count)
index=index[0]
if count eq 0 then begin
	print,'Warning(GET_GEOPARMS): date of observation not found '+ $
		'in VLBI data base...'
	ut1utc_coeffs=dblarr(3)
	pole_x_coeffs=dblarr(3)
	pole_y_coeffs=dblarr(3)
endif else if (index eq 0) or (index ge n_elements(mjd)-2) then begin
	print,'Warning(GET_GEOPARMS): date too near to limits '+ $
		'in VLBI data base...'
	ut1utc_coeffs=dblarr(3)
	pole_x_coeffs=dblarr(3)
	pole_y_coeffs=dblarr(3)
endif else begin
;
; 	Do a quadratic fit through the UT1-UTC data for 4 days starting
;	with the night before the date of observation.
	t=fltarr(4)
	t[0]=mjd[index-1]-mjd[index]
	t[1]=0
	t[2]=mjd[index+1]-mjd[index]
	t[3]=mjd[index+2]-mjd[index]
	v=ut1_utc[index-1:index+2]
	e=ut1_utc_e[index-1:index+2]*10000
	ut1utc_coeffs=poly_fit(t,v,2,measure_errors=e,yfit=vfit)
        sig=stddev(v-vfit)
	if sig gt 0.001 then begin
		print,'Warning(GET_GEOPARMS): STDV (',sig,'[s]) of fit '+ $
			'to UT1-UTC large!'
	endif
;
;	Do a quadratic fit through pole_x data
	v=pole_x[index-1:index+2]
	e=pole_x_e[index-1:index+2]*1000
	pole_x_coeffs=poly_fit(t,v,2,measure_errors=e,yfit=vfit)
        sig=stddev(v-vfit)
	if sig gt 0.001 then begin
		print,'Warning(GET_GEOPARMS): STDV (',sig,'[m]) of fit '+ $
			'to pole_x large!'
	endif
;
;	Do a quadratic fit through pole_y data
	v=pole_y[index-1:index+2]
	e=pole_y_e[index-1:index+2]*1000
	pole_y_coeffs=poly_fit(t,v,2,measure_errors=e,yfit=vfit)
        sig=stddev(v-vfit)
	if sig gt 0.001 then begin
		print,'Warning(GET_GEOPARMS): STDV (',sig,'[m]) of fit ', $
			'to pole_y large!'
	endif
endelse
;
; Now allocate GeoParms structure
if n_elements(GeoParms) eq 0 then GeoParms=allocGeoParms()
;
; Fill in values
GeoParms.Date=Date
GeoParms.SystemId=system_id(SystemId)
GeoParms.Latitude=Latitude
GeoParms.Longitude=Longitude
GeoParms.Altitude=Altitude
GeoParms.EarthRadius=EarthRadius
GeoParms.J2=J2
GeoParms.pole_x_coeffs=pole_x_coeffs
GeoParms.pole_y_coeffs=pole_y_coeffs
GeoParms.ut1utc_coeffs=ut1utc_coeffs
GeoParms.TAI_UTC=TAI_UTC
GeoParms.TDT_TAI=TDT_TAI
;
; Store a copy in GeoInfo after clearing it, for use in AMOEBA
; if n_elements(GeoInfo) le 1 then GeoInfo=replicate(GeoParms,1)
;
print,'GeoParms loaded for '+date+'.'
;
end
