pro get_metroconfig,no_alloc=no_alloc
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Get metrology configuration.
;
; Note the conversion factor for motor angles in the sid model:
; angle [deg] = motor counts * 360e-8.
;
; Note that mudroom hydra channel 6 measures the air temperature, whereas
; the remaining channels 1-5,7-10 correspond to the probes measuring the
; temperatures of various solid parts, and have the corresponding labels
; as listed below.
;
; Siderostat model parameters are sent in the SidModel packet for each
; sidData.N file. The N extension specifies the SidCon ID. The models
; are stored in this order so that OYSTER has to use the SidConId
; to re-order the data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; If no_alloc=1, this procedure will not re-allocate MetroConfig. For multiple
; night loading, re-allocation would cause a structure conflict with MetroInfo.
if n_elements(no_alloc) eq 0 then no_alloc=0 else no_alloc=no_alloc gt 0
;
if hds_state() eq 0 then begin
	print,'***Error(GET_METROCONFIG): no HDS file open!'
	return
endif
toplevel
;
; Check whether this object is available
dat_there,'METROCONFIG',reply
if not reply then begin
;	print,'***Error(GET_METROCONFIG): MetroConfig object not found!'
	return
endif
;
; Get number of siderostats and their SidCon IDs
dat_find,'GENCONFIG'
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
cmp_get1i,'SIDEROSTATID',NumSid,SiderostatId
dat_there,'SIDCONID',reply
if reply eq 1 then cmp_get1i,'SIDCONID',NumSid,SidConId $
	      else SidConId=SiderostatId
dat_annul
dat_annul
;
; Read format
get_format,format
if format eq 'CHAMELEON' or format eq 'INCHWORM' then max_alloc=1 $
						 else max_alloc=0
;
; Determine number of siderostats for which metrology was configured and rec.
dat_find,'GENCONFIG'
dat_there,'NUMPLATE',reply
if reply then cmp_get0i,'NUMPLATE',NumPlate else NumPlate=NumSid
dat_annul
;
; Allocate MetroConfig structure
if max_alloc then begin
	MaxSid=6
	MaxPlate=6
endif else begin
	MaxSid=NumSid
	MaxPlate=NumPlate
endelse
if n_elements(MetroConfig) eq 0 or no_alloc eq 0 or max_alloc eq 0 then $
MetroConfig=allocmetroconfig(MaxPlate,MaxSid,9)
;
; If number of plates is not equal to number of siderostats, assume
; standard 4-way metroconfig and determine index into allocated objects
if NumPlate ne NumSid then begin
	AStationIDs=['AC0','AE0','AW0','AN0']
	if NumPlate ne n_elements(AStationIDs) then begin
		print,'***Error(GET_METROCONFIG): unknown metro configuration!'
		return
	endif
	for i=0,NumPlate-1 do $
		MetroConfig.PlateIdx[i]=where(GenConfig.StationId eq AStationIDs[i])
endif else MetroConfig.PlateIdx=indgen(NumPlate)
pidx=MetroConfig.PlateIdx
;
; Locate main container
dat_find,'METROCONFIG',status & if status ne 0 then return
;
; Read feed beam air sensor configuration
dat_find,'FBAIRTEMPCONF',status
IF status eq 0 THEN BEGIN
NumSensor=intarr(NumSid)
for i=0,NumPlate-1 do begin
	dat_cell,1,i+1,status
	if status eq 0 then begin
		cmp_get0i,'NUMSENSOR',v
		if pidx[i] ne -1 then NumSensor[pidx[i]]=v
		dat_annul
	endif
endfor
dat_annul
MetroConfig.FBAirTemp.NumSensor[0:NumSid-1]=NumSensor
Metroconfig.FBAirTemp.Label[*]='Sid room air, near NAT (below stellar beam)'
ENDIF
;
; Read feed beam air sensor configuration
dat_find,'FBSOLIDTMPCONF',status
IF status eq 0 THEN BEGIN
NumSensor=intarr(NumSid)
for i=0,NumPlate-1 do begin
	dat_cell,1,i+1,status
	if status eq 0 then begin
		cmp_get0i,'NUMSENSOR',v
		if pidx[i] ne -1 then NumSensor[pidx[i]]=v
		dat_annul
	endif
endfor
dat_annul
MetroConfig.FBSolidTmp.NumSensor[0:NumSid-1]=NumSensor
labels=['Feed can, level B (top mirror level plate, INSIDE can)', $
        'Feed can, level C (middle mirror level plate, INSIDE can)', $
        'Feed can, level D (bottom mirror level plate, INSIDE can)', $
        'Feed can exterior, south side', $
        'NAT mount base', $
        'Elevator can top plate, INSIDE can', $
        'Elevator can middle (movable) plate, INSIDE can', $
        'Elevator can  bottom plate, INSIDE can', $
        'Elevator can exterior']
for i=0,NumSid -1 do MetroConfig.FBSolidTmp.Label[*,i]=labels
;
; For a few months, the center hut probes were at different locations
labels=['Feed can, level B (top mirror level plate, INSIDE can)', $
        'Feed can, level C (middle mirror level plate, INSIDE can)', $
        'Feed can, level D (bottom mirror level plate, INSIDE can)', $
        'Feed can exterior, south side', $
        'Center feed can concrete pier', $
        'Elevator can top plate, INSIDE can', $
        'Elevator can middle (movable) plate, INSIDE can', $
        'Elevator can  bottom plate, INSIDE can', $
        'Exterior, bottom seal plate center feed can']
parsedate,date,y,m,d
if julian(y,m,d) ge julian(2000,10,1) and julian(y,m,d) le julian(2001,4,2) then
	MetroConfig.FBSolidTmp.Label[*,0]=labels
; Here is the N hut for astrometry with the 6-way combiner
labels=['North array center, level B (top    mirror support plate, inside can)',
	'North array center, level C (middle mirror support plate, inside can)', $
        'North array center, level D (bottom mirror support plate, inside can)',
        'North array center can EXTERIOR, north side', $
	'North NAT mirror mount base, north side', $
	'North elevator skeleton top plate (inside can)', $
        'North elevator skeleton elevator (movable) plate (inside can)', $
        'North elevator skeleton bottom plate (inside can)', $
        'North elevator can EXTERIOR']
if julian(y,m,d) ge julian(2002,3,10) and NumSid gt 2 then $
MetroConfig.FBSolidTmp.Label(*,3)=labels
ENDIF
;
; Read siderostat model parameters
dat_find,'SIDMODEL',status
IF status eq 0 THEN BEGIN
FeedBeamAng=dblarr(2,NumSid)
FeedBeamAngErr=dblarr(2,NumSid)
SidAng=dblarr(2,NumSid)
SidAngErr=dblarr(2,NumSid)
ZeroAng=dblarr(2,NumSid)
ZeroAngErr=dblarr(2,NumSid)
AxisOffAng=dblarr(NumSid)
AxisOffAngErr=dblarr(NumSid)
MirrorOffAng=dblarr(NumSid)
MirrorOffAngErr=dblarr(NumSid)
CatsEyeOff=dblarr(NumSid)
CatsEyeOffErr=dblarr(NumSid)
for i=0,NumSid-1 do begin
	j=where(SidConId eq (i+1)) & j=j[0]
	if j eq -1 then j=i
	dat_cell,1,i+1
	dat_there,'FEEDBEAMANG',reply
	if reply then begin
		cmp_get1d,'FEEDBEAMANG',2,v
		FeedBeamAng[*,j]=v
		cmp_get1d,'FEEDBEAMANGERR',2,v
		FeedBeamAngErr[*,j]=v
		cmp_get1d,'SIDANG',2,v
		SidAng[*,j]=v
		cmp_get1d,'SIDANGERR',2,v
		SidAngErr[*,j]=v
		cmp_get1d,'ZEROANG',2,v
		ZeroAng[*,j]=v
		cmp_get1d,'ZEROANGERR',2,v
		ZeroAngErr[*,j]=v
		cmp_get0d,'AXISOFFANG',v
		AxisOffAng[j]=v
		cmp_get0d,'AXISOFFANGERR',v
		AxisOffAngErr[j]=v
		cmp_get0d,'MIRROROFFANG',v
		MirrorOffAng[j]=v
		cmp_get0d,'MIRROROFFANGERR',v
		MirrorOffAngErr[j]=v
		if SidAng[0,j] gt 180 then begin
			SidAng[0,j]=SidAng[0,j]-180
			SidAng[1,j]=180-SidAng[1,j]
			ZeroAng[0,j]=ZeroAng[0,j]+180
		endif
	endif
	dat_annul
endfor
dat_annul
MetroConfig.SidModel.FeedBeamAng[*,0:NumSid-1]=feedbeamang
MetroConfig.SidModel.FeedBeamAngErr[*,0:NumSid-1]=feedbeamangErr
MetroConfig.SidModel.SidAng[*,0:NumSid-1]=sidang
MetroConfig.SidModel.SidAngErr[*,0:NumSid-1]=sidangErr
MetroConfig.SidModel.ZeroAng[*,0:NumSid-1]=zeroang
MetroConfig.SidModel.ZeroAngErr[*,0:NumSid-1]=zeroangErr
MetroConfig.SidModel.AxisOffAng[0:NumSid-1]=axisoffang
MetroConfig.SidModel.AxisOffAngErr[0:NumSid-1]=AxisOffangErr
MetroConfig.SidModel.MirrorOffAng[0:NumSid-1]=Mirroroffang
MetroConfig.SidModel.MirrorOffAngErr[0:NumSid-1]=MirrorOffangErr
MetroConfig.SidModel.CatsEyeOff[0:NumSid-1]=CatsEyeOff
MetroConfig.SidModel.CatsEyeOffErr[0:NumSid-1]=CatsEyeOffErr
ENDIF
;
; Return to first level
toplevel
;
print,'MetroConfig loaded.'
;
end
