pro get_oifits,fitsfile,freememory=freememory,pionierfree=pionierfree, $
	stationfile=station_file,photfile=sed_file, $
	uvcheck=uvcheck,date_obs=date_obs_in
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
forward_function tag_exist
forward_function matisse_zerocounts
;
; Procedure to read interferometric data from a single OI-FITS file.
; Stores data in AMOEBA buffer. Note that strings read from OIFITS have
; trailing blanks for an as yet unknown reason.
;
; Checks presence of file "????-??-??_oiDiam.fits" and reads it to find out
; which stars are calibrators (this file is produced by pndrsCalibrate).
;
; Checks the presence of file seeing.txt and reads it to fill scans.r0.
;
; A spectrum [lambda in microns, flux] stored in sed_file can be supplied.
;
; If freememory is true (1), then free memory before storing new data (default)
; If pionierfree, assign each BL to a new OB for calibration (default=no)
;
; If specified, date_obs sets the date of observation for all data sets.
;
common Hds,path,hds_file_stub
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,StarTable,Notes
common ModelFit,parameters,ds_options
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
common LocalChaSel,chafiles
common LocalFitSel,oiffiles
;
RAD=180/pi_circle
;
; Initialize spectral_cal for PIONEER
common Pionier,pionier_spectral_cal
pionier_spectral_cal='generic_c'
;
; The IDs UU? are inserted to handle the 2-letter U1, U2,... designations.
; They are translated to the designations (UT1,...) by left-shifting indices.
stationids=[ $
	'UT1','UT2','UT3','UT4','UU1','UU2','UU3','UU4', $ $
	'AA0','AA1','AB0','AB1','AB2','AB3','AB4','AB5','AC0','AC1','AC2', $
	'AC3','AD0','AD1','AD2','AE0','AG0','AG1','AG2','AH0','AI1','AI2', $
	'AJ1','AJ2','AJ3','AJ4','AJ5','AJ6','AK0','AL0','AM0']
;
if n_elements(fitsfile) eq 0 then fitsfile='oidata.fits'
if strlen(fitsfile) eq 0 then return
files=file_search(fitsfile)
if strlen(files[0]) eq 0 then begin
	print,'***Error(GET_OIFITS): file not found!'
	return
endif
print,'Reading file: '+fitsfile
;
; Close any open data file
hds_close
;
; Read OI data
read_oidata,fitsfile,oiarray,oitarget,oiwavelength,oivis,oivis2,oit3,oiflux, $
	/inventory
if fitsfile eq 'temp_do_not_keep.fits' then begin
	merged=1
	if n_elements(oivis) gt 0 then oivis=oivis[sort(oivis.mjd)]
	if n_elements(oivis2) gt 0 then oivis2=oivis2[sort(oivis2.mjd)]
	if n_elements(oit3) gt 0 then oit3=oit3[sort(oit3.mjd)]
	if n_elements(oiflux) gt 0 then oiflux=oiflux[sort(oiflux.mjd)]
endif else merged=0
check_oiextns,fitsfile,n_oivis,n_oivis2,n_oit3,n_oiflux
if n_oivis eq 0 and n_oivis2 eq 0 and n_oit3 eq 0 and n_oiflux eq 0 then begin
	print,'No data to load.'
	return
endif
;
; Check OIFITS arrays
if 1 then $
check_oifits,oiarray,oitarget,oiwavelength, $
	oivis1=oivis,oivis2=oivis2,oit3=oit3,oiflux=oiflux
;
; Merge split polarisations (WIP)
if 0 then begin
n_vis=0
if n_oivis2 gt 0 then n_vis=n_oivis2 else if n_oivis gt 0 then n_vis=n_oivis
if n_vis gt 0 then begin
	if n_oivis2 gt 0 then begin
		if n_elements(oivis2.time)/unique(oivis2.time) eq 2 then begin
		endif
	endif
	if n_oivis gt 0 then begin
	endif
endif
endif
;
; Get some information from header (ESO headers only)
d=mrdfits(fitsfile,0,h)
temp=fitshparse(h,'ISS.AMBI.TEMP',/quiet)
pres=fitshparse(h,'ISS.AMBI.PRES',/quiet)
rhum=fitshparse(h,'ISS.AMBI.RHUM',/quiet)
seeing=(fitshparse(h,'ISS.AMBI.FWHM.START',/quiet) $
       +fitshparse(h,'ISS.AMBI.FWHM.END',/quiet))/2.
taunot=(fitshparse(h,'ISS.AMBI.TAU0.START',/quiet) $
       +fitshparse(h,'ISS.AMBI.TAU0.END',/quiet))/2.
pwv=(fitshparse(h,'ISS.AMBI.IWV.START',/quiet) $
    +fitshparse(h,'ISS.AMBI.IWV.END',/quiet))/2.
;
; OYSTER's UT Date has only a single value, hence times are wrt to UT=0
; Check night with observations before UT=0 => date+1d and times-1d (neg.)
systemid=strtrim(oiarray[0].arrname,2)
night_start=24-system_config(systemid[0],'MIDNIGHT'); UT of when night begins
!date_change=0
n2=0
if n_elements(oivis2) ne 0 then begin
	oivis2.date_obs=strmid(oivis2.date_obs,0,10)
	oivis2.time=(oivis2.mjd mod 1)*86400
	idx2=where((oivis2.mjd-long(oivis2.mjd))*24 gt night_start,n2)
	if n2 gt 0 then begin
		!date_change=1
		date_obs=strmid(oivis2[idx2].date_obs,0,10)
		oivis2[idx2].date_obs=nextdate(date_obs)
		oivis2[idx2].time=oivis2[idx2].time-86400
	endif
	if keyword_set(date_obs_in) then oivis2.date_obs=date_obs_in
endif
if n_elements(oivis) ne 0 then begin
	oivis.date_obs=strmid(oivis.date_obs,0,10)
	oivis.time=(oivis.mjd mod 1)*86400
	idxv=where((oivis.mjd-long(oivis.mjd))*24 gt night_start,nv)
	if nv gt 0 then begin
		!date_change=1
		date_obs=strmid(oivis[idxv].date_obs,0,10)
		oivis[idxv].date_obs=nextdate(date_obs)
		oivis[idxv].time=oivis[idxv].time-86400
	endif
	if keyword_set(date_obs_in) then oivis.date_obs=date_obs_in
endif
if n_elements(oit3) ne 0 then begin
	oit3.date_obs=strmid(oit3.date_obs,0,10)
	oit3.time=(oit3.mjd mod 1)*86400
	idx3=where((oit3.mjd-long(oit3.mjd))*24 gt night_start,n3)
	if n3 gt 0 then begin
		!date_change=1
		date_obs=strmid(oit3[idx3].date_obs,0,10)
		oit3[idx3].date_obs=nextdate(date_obs)
		oit3[idx3].time=oit3[idx3].time-86400
	endif
	if keyword_set(date_obs_in) then oit3.date_obs=date_obs_in
endif
if n_elements(oiflux) ne 0 then begin
	oiflux.date_obs=strmid(oiflux.date_obs,0,10)
	oiflux.time=(oiflux.mjd mod 1)*86400
	idxf=where((oiflux.mjd-long(oiflux.mjd))*24 gt night_start,nf)
	if nf gt 0 then begin
		!date_change=1
		date_obs=strmid(oiflux[idxf].date_obs,0,10)
		oiflux[idxf].date_obs=nextdate(date_obs)
		oiflux[idxf].time=oiflux[idxf].time-86400
	endif
	if keyword_set(date_obs_in) then oiflux.date_obs=date_obs_in
endif
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1  ; True: interferometry loaded
;
; Assemble startable
if n_elements(oitarget) eq 0 then begin
	starid=''
	read,starid, $
	prompt='Target table not present, please enter star ID [HDN123456]: ', $
	get_startable,starid
endif else begin
	targets=unique(strcompress(oitarget.target,/remove_all),si)
;	Make sure order of targets and oitarget is the same
	oitargets=oitarget[si]	; Note that oitarget is preserved!
;	Create and fill the StarTable
	create_startable,targets
	startable.name=targets
	startable.ra=oitargets.raep0/15
	startable.dec=oitargets.decep0
	startable.rae=oitargets.ra_err
	startable.dece=oitargets.dec_err
	startable.rv=oitargets.sysvel
	startable.pmra=oitargets.pmra
	startable.pmdec=oitargets.pmdec
	startable.pmrae=oitargets.pmra_err
	startable.pmdece=oitargets.pmdec_err
	startable.px=oitargets.parallax
	startable.pxe=oitargets.para_err
	startable.spectrum=oitargets.spectyp
;	startable_id will be an index into startable given a OI target_id
	if n_elements(oivis2) gt 0 then begin
	startable_id=intarr(n_elements(unique(oivis2.target_id))+1)
	startable_st=strarr(n_elements(unique(oivis2.target_id))+1) ; temporary
	endif else begin
	startable_id=intarr(n_elements(unique(oivis.target_id))+1)
	startable_st=strarr(n_elements(unique(oivis.target_id))+1) ; temporary
	endelse
;	Establish and assign the OYSTER StarId, to be used by this command:
;	scans(k).starid=startable(startable_id(v2(index(0)).target_id)).starid
	for i=0,n_elements(oitarget)-1 do begin
		j=where(startable.name eq $
			strcompress(oitarget[i].target,/remove_all))
;		startable_id is an index into startable given a target_id
		startable_id[oitarget[i].target_id]=j
		if valid_cat(targets[j]) eq 1 then starid=targets[j] $
					      else starid=cri_vlti(targets[j])
		if (strlen(starid) eq 0 or valid_cat(starid) eq 0) $
			and !owner eq 'chummel' then begin
			print,'Warning: '+targets[j]+' is not a recognized ID!'
			print,'Please add object to '+ $
				!oyster_dir+'starbase/vlti.usr'
		endif
		if strlen(starid) eq 0 then begin
			iras_id=irasid(startable[j].ra,startable[j].dec)
			if strlen(iras_id) gt 0 then begin
				startable[j].starid='IRAS'+iras_id
			endif else $
			startable[j].starid='OBJ'+esoid(startable[j].ra, $
							startable[j].dec)
		endif else begin
			startable[j].starid=starid
		endelse
;		Temporarily store the target name for this oi_target
		startable_st[oitarget[i].target_id]=starid
	endfor
;	Remove identical StarIDs with different names
	starids=unique(startable.starid,si)
	startable=startable[si]
	targets=targets[si]
	for i=0,n_elements(oitarget)-1 do begin
		j=where(startable.starid eq startable_st[oitarget[i].target_id])
		startable_id[oitarget[i].target_id]=j
	endfor
;	Read catalog positions if target has an HD number
	index=where(strpos(startable.starid,'HDN') eq 0,count)
	if count gt 0 then begin
		startable_bck=startable
		startable=startable[index]
		startable.hdn=long(strmid(startable.starid,3,6))
		read_catalogs
		startable_bck[index].ra=startable.ra
		startable_bck[index].dec=startable.dec
		startable=startable_bck
	endif
;
endelse
;
; Determine number of nights
if n_elements(oivis) ne 0 then begin
	nights_vc=unique(oivis.date_obs)
	num_nights_vc=n_elements(nights_vc)
endif else begin
	nights_vc=''
	num_nights_vc=0
endelse
print,'Number of nights with complex vis data:   ',num_nights_vc
if n_elements(oivis2) ne 0 then begin
	ds_options.v2=1
	nights_v2=unique(oivis2.date_obs)
	num_nights_v2=n_elements(nights_v2)
endif else begin
	nights_v2=''
	num_nights_v2=0
endelse
print,'Number of nights with squared vis data:   ',num_nights_v2
if n_elements(oit3) ne 0 then begin
	ds_options.tp=1
	ds_options.ta=0
	nights_t3=unique(oit3.date_obs)
	num_nights_t3=n_elements(nights_t3)
endif else begin
	nights_t3=''
	num_nights_t3=0
endelse
print,'Number of nights with complex triple data:',num_nights_t3
if n_elements(oiflux) ne 0 then begin
	ds_options.fl=1
	nights_fl=unique(oiflux.date_obs)
	num_nights_fl=n_elements(nights_fl)
endif else begin
	nights_fl=''
	num_nights_fl=0
endelse
print,'Number of nights with flux data:          ',num_nights_fl
nights=unique([nights_vc,nights_v2,nights_t3,nights_fl])
index=where(strlen(nights) gt 0,count)
if count eq 0 then begin
	print,'***Error(GET_OIFITS): no data!'
	return
endif
nights=nights[index]
num_nights=n_elements(nights)
print,'Total number of nights: ',num_nights
;
; Detect missing time stamps
if n_elements(oivis) ne 0 then begin
	if total(oivis.time) eq 0 then oivis.time=(oivis.mjd mod 1)*86400
	uinsname=unique(oivis.insname)
	for i=0,n_elements(oivis.insname)-1 do $
	if oivis[i].time eq 0 then $
	   oivis[i].time=where(uinsname eq oivis[i].insname)*3600
endif
if n_elements(oivis2) ne 0 then begin
	if total(oivis2.time) eq 0 then oivis2.time=(oivis2.mjd mod 1)*86400
	uinsname=unique(oivis2.insname)
	for i=0,n_elements(oivis2.insname)-1 do $
	if oivis2[i].time eq 0 then $
	   oivis2[i].time=where(uinsname eq oivis2[i].insname)*3600
endif
if n_elements(oit3) ne 0 then begin
	if total(oit3.time) eq 0 then oit3.time=(oit3.mjd mod 1)*86400
	uinsname=unique(oit3.insname)
	for i=0,n_elements(oit3.insname)-1 do $
	if oit3[i].time eq 0 then $
	   oit3[i].time=where(uinsname eq oit3[i].insname)*3600
endif
if n_elements(oiflux) ne 0 then begin
	if total(oiflux.time) eq 0 then oiflux.time=(oiflux.mjd mod 1)*86400
	uinsname=unique(oiflux.insname)
	for i=0,n_elements(oiflux.insname)-1 do $
	if oiflux[i].time eq 0 then $
	   oiflux[i].time=where(uinsname eq oiflux[i].insname)*3600
endif
;
; Detect identical time stamps for different observations
if n_elements(oivis2) ne 0 then begin
	epochs=strcompress(oivis2.date_obs $
		+startable[startable_id[oivis2.target_id]].starid $
		+string(oivis2.time),/remove_all)
	times=strcompress(oivis2.date_obs $
		+string(oivis2.time),/remove_all)
	if n_elements(unique(times)) lt n_elements(unique(epochs)) then begin
;		Initialize time stamps
		epoch=epochs[0]
		time=hms2h(strmid(times[0],11,12))*3600
		for i=1,n_elements(epochs)-1 do begin
			if epochs[i] ne epoch then begin
				time=time+600
				epoch=epochs[i]
			endif
			oivis2[i].time=time
		endfor
	endif
endif
;
; Check whether the different instrument names really are different
nins=n_elements(oiwavelength)
insn=strarr(nins)
for i=0,nins-1 do begin
	p=oiwavelength[i].eff_wave
	for j=0,oiwavelength[i].nwave-1 do insn[i]=insn[i]+string((*p)(j))
endfor
if n_elements(unique(insn)) eq 1 then begin
	oiwavelength=oiwavelength[0]
	if n_elements(oivis) ne 0 then oivis[*].insname=oiwavelength.insname
	if n_elements(oivis2) ne 0 then oivis2[*].insname=oiwavelength.insname
	if n_elements(oit3) ne 0 then oit3[*].insname=oiwavelength.insname
	if n_elements(oiflux) ne 0 then oiflux[*].insname=oiwavelength.insname
endif
;
; Reduce array names to standard
if n_elements(oivis) ne 0 then begin
	for i=0,n_elements(oivis)-1 do $
		oivis[i].arrname=system_id(oivis[i].arrname)
endif
if n_elements(oivis2) ne 0 then begin
	for i=0,n_elements(oivis2)-1 do $
		oivis2[i].arrname=system_id(oivis2[i].arrname)
endif
if n_elements(oit3) ne 0 then begin
	for i=0,n_elements(oit3)-1 do $
		oit3[i].arrname=system_id(oit3[i].arrname)
endif
if n_elements(oiflux) ne 0 then begin
	for i=0,n_elements(oiflux)-1 do $
		oiflux[i].arrname=system_id(oiflux[i].arrname)
endif
;
; Reduce oiarray if duplicate entries exist
if n_elements(oiarray) ne 0 then begin
	oiarrays=oiarray
	oiarray.arrname=''
	oiarray.extver=1
	s=unique(stringof(oiarray),index)
	oiarray=oiarrays[index]
	for i=0,n_elements(oiarray)-1 do $
		oiarray[i].arrname=system_id(oiarray[i].arrname)
endif
;
; Determine number of data sets (differing in date and/or array)
; 2nd lines allow GRAVITY SC and FT in 2 OBs, ConfigID='GRAVITY_SCFT'
if n_elements(oivis) ne 0 $
;	then oivis_sets=oivis.date_obs+' '+oivis.arrname+' '+oivis.insname $
	then oivis_sets=oivis.date_obs+' '+oivis.arrname $
	else oivis_sets=''
if n_elements(oivis2) ne 0 $
;	then oivis2_sets=oivis2.date_obs+' '+oivis2.arrname+' '+oivis2.insname $
	then oivis2_sets=oivis2.date_obs+' '+oivis2.arrname $
	else oivis2_sets=''
if n_elements(oit3) ne 0 $
;	then oit3_sets=oit3.date_obs+' '+oit3.arrname+' '+oit3.insname $
	then oit3_sets=oit3.date_obs+' '+oit3.arrname $
	else oit3_sets=''
if n_elements(oiflux) ne 0 $
;	then oiflux_sets=oiflux.date_obs+' '+oiflux.arrname+' '+oiflux.insname $
	then oiflux_sets=oiflux.date_obs+' '+oiflux.arrname $
	else oiflux_sets=''
all_sets=[oivis_sets,oivis2_sets,oit3_sets,oiflux_sets]
index=where(strlen(all_sets) gt 0,count)
if count gt 0 then all_sets=all_sets[index]
sets=unique(all_sets)
num_sets=n_elements(sets)
index=where(sets[0] eq oiflux_sets,count)
if count ge 1 then sets=reverse(sets)	; The first set must have config info
print,'Number of datasets:',num_sets
for i=0,num_sets-1 do print,sets[i]
;
; If table oiarray missing (John, Sylvestre) or incomplete (Peter),
; create default table. Also initialize the station indices in the data.
arrnames=strarr(num_sets)
for i=0,num_sets-1 do begin
	words=nameparse(sets[i],' ')
	arrnames[i]=words[0]	; Does not work if array name has blanks
;	3/2 words in a data set name (arrname, insname, date_obs), see above
;	arrnames(i)=strjoin(words(0:n_elements(words)-3),' ')
	arrnames[i]=strjoin(words[0:n_elements(words)-2],' ')
endfor
arrnames=unique(arrnames)
if n_elements(oiarray) eq 0 then begin
	if n_elements(station_file) ne 0 then begin
		l=''
		status=dc_read_free(station_file,l,/col)
		stations=strarr(3,n_elements(l))
		for i=0,n_elements(l)-1 do begin
			words=nameparse(l[i])
			stations[0:n_elements(words)-1,i]=words
		endfor
	endif
	def_index=unique(oivis2.sta_index)
	def_stations='A'+string(def_index,format='(i2.2)')
	num_tel=n_elements(def_stations)
	oiarrays=replicate({ $
		arrname:'', $
		sta_name:'', $
		sta_index:0, $
		arrayx:0.d0, $
		arrayy:0.d0, $
		arrayz:0.d0, $
		frame:'UNKNOWN', $
		staxyz:fltarr(3) $
		},num_tel*n_elements(arrnames))
	for i=0,n_elements(arrnames)-1 do begin
;		Group all telescopes for same array together
		oiarrays[i*num_tel:(i+1)*num_tel-1].arrname=arrnames[i]
;		Use info from station file, if available
		if n_elements(stations) gt 0 $
		then oiarrays[i*num_tel:(i+1)*num_tel-1].sta_name=stations[*,i] $
		else oiarrays[i*num_tel:(i+1)*num_tel-1].sta_name=def_stations
		if n_elements(stations) gt 0 $
		then oiarrays[i*num_tel:(i+1)*num_tel-1].sta_index=def_index $
		else oiarrays[i*num_tel:(i+1)*num_tel-1].sta_index=def_index
	endfor
	oiarray=oiarrays
endif
;
; Obtain info for allocation of geninfo and geoinfo
;
MaxSid=n_elements(unique(oiarray.sta_name))
MS=0
MB=0
MC=max(oiwavelength.nwave)
maxConfig=10
for i=0,num_nights_vc-1 do begin
	j=where(oivis.date_obs eq nights_vc[i],jc)
	if jc gt 0 then begin
		specs=unique(oivis[j].insname)
                num_spec=n_elements(specs)
                MS=max([MS,num_spec])
                vc=oivis[j]
                for j=0,num_spec-1 do begin
                k=where(vc.insname eq specs[j])
                MB=max([MB,n_elements(unique(strjoin(vc[k].sta_index)))])
                endfor
        endif
endfor
for i=0,num_nights_v2-1 do begin
	j=where(oivis2.date_obs eq nights_v2[i],jc)
	if jc gt 0 then begin
		specs=unique(oivis2[j].insname)
		num_spec=n_elements(specs)
		MS=max([MS,num_spec])
		v2=oivis2[j]
		for j=0,num_spec-1 do begin
		k=where(v2.insname eq specs[j])
		MB=max([MB,n_elements(unique(strjoin(v2[k].sta_index)))])
		endfor
	endif
endfor
MT=1
for i=0,num_nights_t3-1 do begin
	j=where(oit3.date_obs eq nights_t3[i],jc)
	if jc gt 0 then begin
		MT=max([MT,n_elements(unique(oit3[j].insname))]) $
		  *max([MT,n_elements(unique(strjoin(oit3.sta_index)))])
	endif
endfor
; Check for problems due to missing vc, v2, or t3
if MB eq 0 then begin
	MS=1
	MB=3
endif
; Starting in Version 6, we now allocate a reduced size version of geninfo
if num_sets gt 1 then begin
   geninfo=replicate(allocgenconfig(MaxSid,MS,MT,MB,MC,maxConfig,/geninfo), $
		  num_sets)
   geoinfo=replicate(allocgeoparms(),num_sets)
endif
baselineij=strarr(MB,MS)
;
; Now loop over all nights
;
for i=0,num_sets-1 do begin
;
; 	Starting in Version 6, we no longer allocate geninfo with the full
;	capacity. Therefore, we repeat the allocation for this particular night.
	if total(strlen(strcompress(oiarray.sta_name,/remove_all))) eq 0 then $
		oiarray.sta_name=oiarray.tel_name
	MaxSid=n_elements(unique(oiarray.sta_name))
	MS=0				; spectrometers
	MT=1				; triples
	MB=0				; baselines
	MC=max(oiwavelength.nwave)	; channels
	if n_elements(unique(oiarray.arrname)) eq 1 then begin
		systemid=strtrim(oiarray[0].arrname,2)
		maxConfig=system_config(system_id(systemid),'MAXCONFIG')
		case string(maxConfig) of
			'scans':maxConfig=scanconfig(/maxconfig)
			'stars':maxConfig=n_elements(scanconfig(/starlist))
			   else:maxConfig=fix(maxConfig)
		endcase
	endif else begin
		maxConfig=10
	endelse
	count_vc=0
	if n_elements(oivis) ne 0 then $
	index_vc=where(oivis_sets eq sets[i],count_vc)
	if count_vc gt 0 then begin
		specs=unique(oivis[index_vc].insname)
                num_spec=n_elements(specs)
                MS=max([MS,num_spec])
                vc=oivis[index_vc]
                for j=0,num_spec-1 do begin
                k=where(vc.insname eq specs[j])
                MB=max([MB,n_elements(unique(strjoin(vc[k].sta_index)))])
                endfor
		date=strmid(oivis[index_vc[0]].date_obs,0,10)
        endif
	count_v2=0
	if n_elements(oivis2) ne 0 then $
	index_v2=where(oivis2_sets eq sets[i],count_v2)
	if count_v2 gt 0 then begin
		specs=unique(oivis2[index_v2].insname)
		num_spec=n_elements(specs)
		MS=max([MS,num_spec])
		v2=oivis2[index_v2]
		for j=0,num_spec-1 do begin
		k=where(v2.insname eq specs[j])
		MB=max([MB,n_elements(unique(strjoin(v2[k].sta_index)))])
		endfor
		date=strmid(oivis2[index_v2[0]].date_obs,0,10)
	endif
	count_t3=0
	if n_elements(oit3) ne 0 then $
	index_t3=where(oit3_sets eq sets[i],count_t3)
	if count_t3 gt 0 then begin
;		MT=max([MT,n_elements(unique(strjoin(oit3.sta_index)))])
		MT=max([MT,n_elements(unique(oit3.insname))]) $
		  *max([MT,n_elements(unique(strjoin(oit3.sta_index)))])
		date=strmid(oit3[index_t3[0]].date_obs,0,10)
	endif
	count_fl=0
	if n_elements(oiflux) ne 0 then $
	index_fl=where(oiflux_sets eq sets[i],count_fl)
	if count_fl gt 0 then begin
		specs=unique(oiflux[index_fl].insname)
		num_spec=n_elements(specs)
		MS=max([MS,num_spec])
		fl=oiflux[index_fl]
		date=strmid(oiflux[index_fl[0]].date_obs,0,10)
		oiflux_max_time_offset=0; time offsets [s] between VIS2 and FLUX
	endif
; 	Check for problems due to missing vc, v2, or t3
	if MB eq 0 then begin
		MS=1
		MB=3
	endif
;	Set MB to maximum possible (only pndrs: may have CLP while missing BL!)
	MaxBase=(MaxSid*(MaxSid-1))/2
	if n_elements(oiwavelength) eq 1 then k=0 else k=i
	if strpos(oiwavelength[k].insname,'PIONIER') ge 0 and $
			count_v2 gt 0 and count_t3 gt 0 then $
		if MB lt MaxBase then MB=(MaxSid*(MaxSid-1))/2
;
;	Allocate genconfig, in the following, we fill GenConfig and geoinfo
	genconfig=allocgenconfig(MaxSid,MS,MT,MB,MC,maxConfig)
	geoparms=allocgeoparms()
;
;	Determine what kind of data were recorded
;
	num_spec=0
	num_triple=0
	int_time_vc=0
	if count_vc gt 0 then begin
;	Here we have a night with complex visibility data
		vc=oivis[index_vc]
		time_vc=vc.time
		int_time_vc=vc.int_time
;		int_time must be in seconds!
		if medianve(int_time_vc/60) gt 60 $
			then int_time_vc=int_time_vc/1000
		vb=vc
	endif else time_vc=0
	int_time_v2=0
	if count_v2 gt 0 then begin
;	Here we have a night with squared visibility data
		v2=oivis2[index_v2]
		time_v2=v2.time
		int_time_v2=v2.int_time
;		int_time must be in seconds!
		if medianve(int_time_v2/60) gt 60 $
			then int_time_v2=int_time_v2/1000
		vb=v2
	endif else time_v2=0
	int_time_t3=0
	if count_t3 gt 0 then begin
;	Here we have a night with complex triple data
		t3=oit3[index_t3]
		time_t3=t3.time
		int_time_t3=t3.int_time
;		int_time must be in seconds!
		if medianve(int_time_t3/60) gt 60 $
			then int_time_t3=int_time_t3/1000
	endif else time_t3=0
	int_time_fl=0
	if count_fl gt 0 then begin
;	Here we have a night with flux data
		fl=oiflux[index_fl]
		time_fl=fl.time
		int_time_fl=fl.int_time
;		int_time must be in seconds!
		if medianve(int_time_fl/60) gt 60 $
			then int_time_fl=int_time_fl/1000
	endif else time_fl=0
;
;	If we we have either v2 or vc, we need to extract geoparms and genconfig
	if count_vc gt 0 or count_v2 gt 0 then begin
;
;		Assemble info for GeoInfo
		r_earth=6378.136d3
		k=where(strtrim(oiarray.arrname,2) $
			eq strtrim(vb[0].arrname,2)) & k=k[0]
		systemid=strtrim(oiarray[k].arrname,2)
		geoparms.systemid=systemid
		geoparms.earthradius=r_earth
		if oiarray[k].arrayx+oiarray[k].arrayy+oiarray[k].arrayz eq 0 $
		then oiarray[k].frame='Unknown'
		case oiarray[k].frame of
		'GEOCENTRIC': begin
			      RAD=180/!pi
			      altitude=sqrt(oiarray[k].arrayx^2 $
					   +oiarray[k].arrayy^2 $
					   +oiarray[k].arrayz^2)-r_earth
			      latitude=asin(oiarray[k].arrayz $
					       /(altitude+r_earth))*RAD
			      longitude=atan(oiarray[k].arrayy, $
					     oiarray[k].arrayx)*RAD
			      end
		else:	      begin
			      latitude= $
			      system_config(system_id(systemid),'LATITUDE')
			      longitude= $
			      system_config(system_id(systemid),'LONGITUDE')
			      altitude= $
			      system_config(system_id(systemid),'ALTITUDE')
			      oiarray[k].frame='GEOCENTRIC'
			      end
		endcase
		geoparms.altitude=altitude
		geoparms.latitude=latitude
		geoparms.longitude=longitude
;
;		Assemble info for GenConfig
;
;		Stations
		sta_index=oiarray.sta_index		; 2020-06-26
		sta_index=unique(vb.sta_index)		; original (no diff.)
		num_sid=n_elements(sta_index)
		stations=strarr(num_sid)+'___'
		siderostats=intarr(num_sid)
		siderostat_ids=intarr(num_sid)
		for j=0,num_sid-1 do begin
			siderostats[j]=sta_index[j]
			index=where(oiarray.sta_index eq sta_index[j])
			if strmid(oiarray[index[0]].tel_name,0,3) eq 'SID' $
				then k=3 else k=2
;			tel_name=strmid(oiarray(index(0)).tel_name,k,1)
;			if strlen(strcompress(tel_name,/remove_all)) eq 0 then $
;				siderostat_ids(j)=j else $
;			siderostat_ids(j)=fix(strmid( $
;				oiarray(index(0)).tel_name,k,1))
			siderostat_ids[j]=fix(strmid( $
			   strcompress(oiarray[index[0]].tel_name,/remove_all),k,1))
			station=stations[j]
			sta_name=strtrim(oiarray[index].sta_name,2)
			strput,station,sta_name[0]
			stations[j]=station
		endfor
		genconfig.numsid=num_sid
		genconfig.stationid[0:num_sid-1]=stations
;		genconfig.siderostatid(0:num_sid-1)=siderostats
		if systemid eq 'VLTI' then $
			genconfig.siderostatid[0:num_sid-1]=siderostat_ids
		genconfig.diameter=system_config(systemid,'DIAMETER', $
						stations=genconfig.stationid)
		genconfig.refstation=1
		cl=cos(latitude/RAD)
		sl=sin(latitude/RAD)
		for j=0,num_sid-1 do begin
		  k=where(oiarray.sta_index eq sta_index[j]) & k=k[0]
		  staxyz=oiarray[k].staxyz
		  if total(staxyz) ne 0 then begin
;		  Geocentric system (the OIFITS standard)
		  genconfig.stationcoord[0:2,j]=equatorial2horizon( $
					      geocentric2equatorial(staxyz[*]))
;		  Issue discovered by Regis Lachaume for GRAVITY pipeline,
;		  but true for all VLTI pipelines (PNDRS does not write coord.):
;		  OIFITS files have horizon coordinates instead of geocentric
		  if systemid eq 'VLTI' and stddev(oiarray.staxyz[2]) lt 0.1 $
			  then genconfig.stationcoord[0:2,j]=staxyz*[-1,-1,1]
		  endif
		endfor
;
;		Spectrometers (a.k.a. instruments)
		specs=unique(vb.insname)
		if strpos(specs[0],'GRAVITY') ge 0 then begin
			reverse_specs=1
			si=sort(specs)
			specs=specs[reverse(si)]	; SC 1st
		endif else reverse_specs=0
		num_spec=n_elements(specs)
		genconfig.numoutbeam=num_spec
;
;		Baselines
		for j=0,num_spec-1 do begin
			k=where(vb.insname eq specs[j])
;			b=unique(strjoin(vb(k).sta_index))
			b=remove_dup(strjoin(vb[k].sta_index))
			genconfig.numbaseline[j]=n_elements(b)
			for l=0,n_elements(b)-1 do begin
				baselineij[l,j]=b[l]
				ij=long(nameparse(b[l]))
				st1=stations[where(sta_index eq ij[0])]
				st2=stations[where(sta_index eq ij[1])]
				genconfig.baselineid[l,j]=st1+'-'+st2
			endfor
			k=where(oiwavelength.insname eq specs[j])
			nw=oiwavelength[k].nwave
			genconfig.numspecchan[j]=nw
			genconfig.wavelength[0:nw-1,j]= $
				*oiwavelength[k].eff_wave
			genconfig.wavelengtherr[0:nw-1,j]= $
				*oiwavelength[k].eff_wave*0.01
			genconfig.chanwidth[0:nw-1,j]= $
				*oiwavelength[k].eff_band
		endfor
;
;		Prepare ConfigID (must follow OYSTER standards)
		configid=shortest(strtrim(vb.insname,2))
;
;		Instrument-specific adjustments of ConfigID
		if configid eq 'GRAVITY_' then configid='GRAVITY_SCFT'
;
		if strpos(configid,'MATISSE') ge 0 then begin
			if genconfig.numoutbeam eq 1 then begin
			if median(genconfig.wavelength) le 6e-6 then b='L'
			if median(genconfig.wavelength) gt 6e-6 then b='N'
			configid=configid+'-'+b
			case b of ; Time [s] offsets FLUX relative to VIS2
				'L':oiflux_max_time_offset=200
				'N':oiflux_max_time_offset=200
			endcase
			endif else begin	; numoutbeam=2
				configid=configid+'-'+'LN'
				oiflux_max_time_offset=200
			endelse
		endif
;
		if strpos(configid,'PIONIER_Pnat') ge 0 then begin
			words=nameparse(configid,['(',')'])
			words=nameparse(words[1],'/')
			nchan=string(genconfig.numspecchan,format='(i1)')
			if n_elements(unique(words)) eq 1 $
				then configid='PIONIER_FREE' $
				else configid='PIONIER_GRS'+nchan
		endif else begin
			if strpos(configid,'PIONIER') eq 0 then begin
				nchan=string(genconfig.numspecchan,format='(i1)')
				if nchan eq 1 then configid='PIONIER_FREE' $
					      else configid='PIONIER_GRISM'
			endif
		endelse
;
;		Replace unwanted characters in ConfigID
		genconfig.configid=stredit(configid,['(','/',')'],['_','-',' '])
;
		words=strtrim(specs[0],2)
		genconfig.beamcombinerid=beamcombiner_id(system_id(systemid) $
					+'/'+words[0])
	endif
;	If we we have t3, we need to extract geoparms and genconfig
	if count_t3 gt 0 then begin
;
;		Assemble info for GenConfig
		r_earth=6378.136d3
		k=where(strtrim(oiarray.arrname,2) eq strtrim(t3[0].arrname,2))
		k=k[0]
		systemid=strtrim(oiarray[k].arrname,2)
		geoparms.systemid=systemid
		geoparms.earthradius=r_earth
		geoparms.latitude=system_config(system_id(systemid),'LATITUDE')
		geoparms.longitude=system_config(system_id(systemid),'LONGITUDE')
		geoparms.altitude=system_config(system_id(systemid),'ALTITUDE')
;
;		Only retrieve specs if there are no v2 or vc data
		if n_elements(vb) eq 0 then begin
			specs=unique(t3.insname)
			if strpos(specs[0],'GRAVITY') ge 0 then begin
				reverse_specs=1
				si=sort(specs)
				specs=specs[reverse(si)]
			endif else reverse_specs=0
			num_spec=n_elements(specs)
			genconfig.numoutbeam=num_spec
		endif
;
;		triples=unique(strjoin(t3.sta_index)+' '+t3.insname)	; alternative
;		triples=unique(t3.insname+' '+strjoin(t3.sta_index))
		triples=remove_dup(t3.insname+' '+strjoin(t3.sta_index))
		if reverse_specs then begin
			si=sort(triples)
			triples=triples[reverse(si)]
		endif
		num_triple=n_elements(triples)
		genconfig.numtriple=num_triple
;
		sta_index=oiarray.sta_index		; 2020-06-26
		sta_index=unique(t3.sta_index)		; original (no diff.)
		num_sid=n_elements(sta_index)
		stations=strarr(num_sid)+'___'
		siderostats=intarr(num_sid)
		siderostat_ids=intarr(num_sid)
		for j=0,num_sid-1 do begin
			siderostats[j]=sta_index[j]
			index=where(oiarray.sta_index eq sta_index[j])
			if strmid(oiarray[index[0]].tel_name,0,3) eq 'SID' $
				then k=3 else k=2
;			tel_name=strmid(oiarray(index(0)).tel_name,k,1)
;			if strlen(strcompress(tel_name,/remove_all)) eq 0 then $
;				siderostat_ids(j)=j else $
;			siderostat_ids(j)=fix(strmid( $
;				oiarray(index(0)).tel_name,k,1))
			siderostat_ids[j]=fix(strmid( $
			   strcompress(oiarray[index[0]].tel_name,/remove_all),k,1))
			station=stations[j]
			sta_name=strtrim(oiarray[index[0]].sta_name,2)
			strput,station,sta_name[0]
			stations[j]=station
		endfor
		genconfig.numsid=num_sid
		genconfig.stationid[0:num_sid-1]=stations
;		genconfig.siderostatid(0:num_sid-1)=siderostats
		if systemid eq 'VLTI' then $
			genconfig.siderostatid[0:num_sid-1]=siderostat_ids
		genconfig.diameter=system_config(systemid,'DIAMETER', $
						stations=genconfig.stationid)
		genconfig.refstation=1
		configid=shortest(strtrim(t3.insname,2))
;
;		Instrument-specific adjustments of ConfigID
		if configid eq 'GRAVITY_' then configid='GRAVITY_SCFT'
		if strpos(configid,'GRAVITY') ge 0 then begin
			oiflux_max_time_offset=0.001
		endif
;
		if strpos(configid,'MATISSE') ge 0 then begin
			if genconfig.numoutbeam eq 1 then begin
			if median(genconfig.wavelength) le 6e-6 then b='L'
			if median(genconfig.wavelength) gt 6e-6 then b='N'
			configid=configid+'-'+b
			case b of ; Time [s] offsets FLUX relative to VIS2
				'L':oiflux_max_time_offset=200
				'N':oiflux_max_time_offset=200
			endcase
			endif else begin	; numoutbeam=2
				configid=configid+'-'+'LN'
				oiflux_max_time_offset=200
			endelse
		endif
		if strpos(configid,'PIONIER_Pnat') ge 0 then begin
			words=nameparse(configid,['(',')'])
			words=nameparse(words[1],'/')
			nchan=string(genconfig.numspecchan,format='(i1)')
			if n_elements(unique(words)) eq 1 $
				then configid='PIONIER_FREE' $
				else configid='PIONIER_GRS'+nchan
		endif else begin
			if strpos(configid,'PIONIER') eq 0 then begin
				nchan=string(genconfig.numspecchan,format='(i1)')
				if nchan eq 1 then configid='PIONIER_FREE' $
					      else configid='PIONIER_GRISM'
			endif
		endelse
;
;		Replace unwanted characters in ConfigID
		genconfig.configid=stredit(configid,['(','/',')'],['_','-',' '])
;
		words=strtrim(specs[0],2)
		genconfig.beamcombinerid=beamcombiner_id(system_id(systemid) $
					+'/'+words[0])
;
;		For data sets with only triple products initialize baselines
		if total( $
		   strlen( $
		   strcompress( $
		   strjoin( $
		   genconfig.baselineid),/remove_all))) eq 0 $
		then begin
			genconfig.numoutbeam=1
			genconfig.numbaseline=3
			genconfig.baselineid[0]='ST1-ST2'
			genconfig.baselineid[1]='ST2-ST3'
			genconfig.baselineid[2]='ST3-ST1'
			genconfig.baselineid[0]=stations[0]+'-'+stations[1]
			genconfig.baselineid[1]=stations[1]+'-'+stations[2]
			genconfig.baselineid[2]=stations[2]+'-'+stations[0]
			genconfig.triplebase[0]=0
			genconfig.triplebase[1]=0
			genconfig.triplebase[2]=0
		endif
;		Identify baselines
		for j=0,num_triple-1 do begin
			t=triples[j]
;			index=where(strjoin(t3.sta_index)+' '+t3.insname eq t)	; alternative
			index=where(t3.insname+' '+strjoin(t3.sta_index) eq t)
			words=nameparse(t)
;			ijk=long(words(0:2))	; alternative
			ijk=long(words[1:3])
			st0=strtrim(oiarray[where(strtrim(oiarray.arrname,2) $
						      eq systemid $
				and oiarray.sta_index eq ijk[0])].sta_name,2)
			st1=strtrim(oiarray[where(strtrim(oiarray.arrname,2) $
						      eq systemid $
				and oiarray.sta_index eq ijk[1])].sta_name,2)
			st2=strtrim(oiarray[where(strtrim(oiarray.arrname,2) $
						      eq systemid $
				and oiarray.sta_index eq ijk[2])].sta_name,2)
;			If oiarray has duplicate entries
			st0=st0[0] & st1=st1[0] & st2=st2[0]
			st0=stations[where(strpos(stations,st0) ge 0)]
			st0=st0[0]
			st1=stations[where(strpos(stations,st1) ge 0)]
			st1=st1[0]
			st2=stations[where(strpos(stations,st2) ge 0)]
			st2=st2[0]
			b1=st0+'-'+st1
			b2=st1+'-'+st2
			b3=st2+'-'+st0
			for k=0,genconfig.numoutbeam-1 do begin
			i1=where(genconfig.baselineid $
				(0:genconfig.numbaseline[k]-1,k) eq b1 $
			      or genconfig.baselineid $
				(0:genconfig.numbaseline[k]-1,k) eq breve(b1))
			i2=where(genconfig.baselineid $
				(0:genconfig.numbaseline[k]-1,k) eq b2 $
			      or genconfig.baselineid $
				(0:genconfig.numbaseline[k]-1,k) eq breve(b2))
			i3=where(genconfig.baselineid $
				(0:genconfig.numbaseline[k]-1,k) eq b3 $
			      or genconfig.baselineid $
				(0:genconfig.numbaseline[k]-1,k) eq breve(b3))
			if i1[0] ge 0 then begin
				genconfig.triplebase[0,j]=i1[0]
;				if specs(k) eq words(3) then $	; alternative
				if specs[k] eq words[0] then $
				genconfig.triplebeam[0,j]=k
			endif
			if i2[0] ge 0 then begin
				genconfig.triplebase[1,j]=i2[0]
;				if specs(k) eq words(3) then $	; alternative
				if specs[k] eq words[0] then $
				genconfig.triplebeam[1,j]=k
			endif
			if i3[0] ge 0 then begin
				genconfig.triplebase[2,j]=i3[0]
;				if specs(k) eq words(3) then $	; alternative
				if specs[k] eq words[0] then $
				genconfig.triplebeam[2,j]=k
			endif else begin
				if 0 then begin
;				pndrs may have CLP while missing BL!
				i3=where(strlen(strcompress( $
					genconfig.baselineid,/remove_all)) eq 0)
				i3=i3[0]
				genconfig.baselineid[i3[0]]=b3
				genconfig.triplebase[2,j]=i3[0]
				st1=stations[where(stations eq strmid(b3,0,3))]
				st2=stations[where(stations eq strmid(b3,3,3))]
				genconfig.baselineid[i3[0]]=st1+'-'+st2
				endif
			endelse
			endfor
;			k=where(strtrim(oiwavelength.insname) eq specs(j)) ; alternative
			k=where(strtrim(oiwavelength.insname) eq words[0])
			nw=oiwavelength[k].nwave
			genconfig.triplenumchan[j]=nw
			genconfig.wavelength[0:nw-1,k]= $
				*oiwavelength[k].eff_wave
			genconfig.wavelengtherr[0:nw-1,k]= $
				*oiwavelength[k].eff_wave*0.01
			genconfig.chanwidth[0:nw-1,k]= $
				*oiwavelength[k].eff_band
			lambda=*oiwavelength[k].eff_wave
			for l=0,nw-1 do begin
			for k=0,2 do begin
			index=where(genconfig.wavelength $
				(0:nw-1,genconfig.triplebeam[k,j]) eq lambda[l])
			genconfig.triplechan[l,k,j]=index
			genconfig.numspecchan[genconfig.triplebeam[k,j]] $
				=genconfig.triplenumchan[j]
			endfor
			endfor
		endfor
	endif
;	Establish a common vector of time stamps
	time=[time_vc,time_v2,time_t3]
	int_time=[int_time_vc,int_time_v2,int_time_t3]
	index=where(time ne 0,count)
	if count gt 0 then begin
		time=unique(time[index],ui)
		if n_elements(ui) gt 1 then int_time=int_time[ui]	;
		num_scan=n_elements(time)
		scans=replicate(scan(),num_scan)
		scans.iscan=indgen(num_scan)+1
		scans.time=time
		scans.temp=temp
		scans.pres=pres
		scans.rhum=rhum
		scans.r0=seeing
		scans.t0=taunot
		scans.pwv=pwv
;
		true=byte('T') & true=true[0]
;		Work on every scan
		FOR k=0,num_scan-1 do BEGIN
;
		FOR j=0,num_spec-1 do BEGIN
			nw=genconfig.numspecchan[j]
		FOR l=0,genconfig.numbaseline[j]-1 do BEGIN
;
option=1	; Testing purpose, option 2 has bugs
if option eq 1 then begin
;		Complex visibility data (both amp/phase and complex vis/err)
		if count_vc gt 0 then begin
		oivis_ok=0
		index=where(vc.time eq time[k] $
			and vc.insname eq specs[j] $
			and strjoin(vc.sta_index) eq baselineij[l,j],count)
		if count gt 0 then begin
		index=index[0]
;		Look for OIFITS2-compliant VISAMP and VISPHI
		if tag_exists('VISAMP',vc) and tag_exists('VISPHI',vc) then begin
		if vc[0].visamp ne !NULL and vc[0].visphi ne !NULL then begin
			scans[k].diffphasec(j,0:nw-1,l) $
				=(*vc[index].visphi)(0:nw-1)
			scans[k].diffphasecerr(j,0:nw-1,l) $
				=((*vc[index].visphierr)(0:nw-1)) < 3.4e38
			vc_real=((*vc[index].visamp)(0:nw-1)) $
				*cos(((*vc[index].visphi)(0:nw-1))/RAD)
			vc_imag=((*vc[index].visamp)(0:nw-1)) $
				*sin(((*vc[index].visphi)(0:nw-1))/RAD)
			scans[k].complexvis(j,0:nw-1,l)=complex(vc_real,vc_imag)
			scans[k].complexweight(j,0:nw-1,l)= $
				byte(*vc[index].flag) ne true
			oivis_ok=1
		endif
		endif
		endif
		if not oivis_ok then begin
		if tag_exists('RVIS',vc) and tag_exists('IVIS',vc) then begin
		if vc[0].rvis ne !NULL and vc[0].ivis ne !NULL then begin
;			RVIS and IVIS are the real and imaginary part of the
;			complex coherent flux
			scans[k].complexvis(j,0:nw-1,l)= $
				complex(*vc[index].rvis,*vc[index].ivis)
			scans[k].complexweight(j,0:nw-1,l)= $
				byte(*vc[index].flag) ne true
			oivis_ok=1
		endif
		endif
		endif
		if not oivis_ok then begin
;		Look for non-compliant GRAVITY VISDATA and VISERR columns
		if tag_exists('VISDATA',vc) and tag_exists('VISERR',vc) then begin
;			print,'Warning: this code not yet validated...'
		if vc[0].visdata ne !NULL and vc[0].viserr ne !NULL then begin
			if k+j+l eq 0 then $
			print,'Warning: non-compliant tags VISDATA and VISERR!'
			scans[k].diffphasec(j,0:nw-1,l) $
				=cphase((*vc[index].visdata)(0:nw-1))
			idx=where(abs((*vc[index].visdata)(0:nw-1)) gt 0,nidx)
			if nidx gt 0 then $
				scans[k].diffphasecerr(j,idx,l) $
					=abs((*vc[index].viserr)(idx)) $
					/abs((*vc[index].visdata)(idx))
			idx=where(byte(*vc[index].flag) eq true,nidx)
			if nidx ge 1 then scans[k].diffphasecerr(j,idx,l)=-1
;			VISDATA is the complex flux, not mormalized
			scans[k].complexvis(j,0:nw-1,l)= $
				(*vc[index].visdata)(0:nw-1)
			scans[k].complexweight(j,0:nw-1,l)= $
				byte((*vc[index].flag)(0:nw-1)) ne true
			oivis_ok=1
		endif
		endif
		endif
		endif	; count_vc
;
endif else begin
;	This part is an update of the code above, but has issues
;		Complex visibility data (both amp/phase and complex vis/err)
		if count_vc gt 0 then begin
		index=where(vc.time eq time[k] $
			and vc.insname eq specs[j] $
			and strjoin(vc.sta_index) eq baselineij[l,j],count)
		if count gt 0 then begin
;		Look for OIFITS2-compliant VISAMP and VISPHI
		oivis_ok=0
		if tag_exists('VISAMP',vc) and tag_exists('VISPHI',vc) then begin
		if vc[0].visamp ne !NULL and vc[0].visphi ne !NULL then begin
		for i=0,count-1 do begin
			scans[k].diffphasec(j,index,l)=*(vc[index[i]].visphi)
			scans[k].diffphasecerr(j,index,l)=*(vc[index[i]].visphierr)
			vc_real=(*vc[index[i]].visamp)*cos(*vc[index[i]].visphi/RAD)
			vc_imag=(*vc[index[i]].visamp)*sin(*vc[index[i]].visphi/RAD)
			scans[k].complexvis(j,index,l)=complex(vc_real,vc_imag)
			scans[k].complexweight(j,index,l)=*(vc[index[i]].flag) ne true
		endfor
		oivis_ok=1
		endif
		endif
		if not oivis_ok then begin
;		Look for RVIS and IVIS, the real and imaginary parts of the
;		complex coherent flux
		if tag_exists('RVIS',vc) and tag_exists('IVIS',vc) then begin
		if vc[0].rvis ne !NULL and vc[0].ivis ne !NULL then begin
		for i=0,count-1 do begin
			scans[k].complexvis(j,index,l)= $
				complex(*vc[index[i]].rvis,*vc[index[i]].ivis)
			scans[k].complexweight(j,index,l)= $
				*(vc[index[i]].flag) ne true
		endfor
		oivis_ok=1
		endif
		endif
		endif
		if not oivis_ok then begin
;		Look for non-compliant GRAVITY VISDATA and VISERR columns
		if tag_exists('VISDATA',vc) and tag_exists('VISERR',vc) then begin
;			print,'Warning: this code not yet validated...'
		if vc[0].visdata ne !NULL and vc[0].viserr ne !NULL then begin
			if k+j+l eq 0 then $
			print,'Warning: non-compliant tags VISDATA and VISERR!'
		for i=0,count-1 do begin
			scans[k].diffphasec(j,index,l)=cphase(*vc[index[i]].visdata)
			scans[k].diffphase(j,index,l)=scans[k].diffphasec(j,*,l)
			idx=where(abs((*vc[index[i]].visdata)) gt 0,nidx)
			if nidx gt 0 then $
				scans[k].diffphasecerr(j,i[idx],l) $
					=abs(((*vc[index[i]].viserr)(idx))) $
					/abs(((*vc[index[i]].visdata)(idx)))
			idx=where(byte(*vc[index[i]].flag) eq true,nidx)
			if nidx ge 1 then scans[k].diffphasecerr(j,i[idx],l)=-1
			scans[k].diffphaseerr(j,index,l)=scans[k].diffphasecerr(j,*,l)
;			VISDATA is the complex flux, not mormalized
			scans[k].complexvis(j,index,l)=(*vc[index[i]].visdata)
			scans[k].complexweight(j,index,l)=(*vc[index[i]].flag) ne true
		endfor
		oivis_ok=1
		endif
		endif
		endif
;
		scans[k].starid=startable[ $
			startable_id[vc[index[0]].target_id]].starid
		scans[k].star=targets[startable_id[vc[index[0]].target_id]]
		scans[k].uvw(j,0:nw-1,l,0)=vc[index].ucoord $
					  /genconfig.wavelength[0:nw-1,j]
		scans[k].uvw(j,0:nw-1,l,1)=vc[index].vcoord $
					  /genconfig.wavelength[0:nw-1,j]
;		Read additional columns if file is output of GRAVITY astrometry
		if strpos(fitsfile,'ASTROREDUCED') ge 0 then begin
			scans.vissqcerr=1 ; used as error for UVW values
		endif
;
		endif	; count
		endif	; count_vc
endelse
;
;		Squared visibility data, min(err) set to 0.001 unless flag=true
		if count_v2 gt 0 then begin
		index=where(v2.time eq time[k] $
			and v2.insname eq specs[j] $
			and strjoin(v2.sta_index) eq baselineij[l,j],count)
		if count gt 0 then begin
		index=index[0]
		scans[k].vissqc(j,0:nw-1,l)=(*v2[index].vis2data)(0:nw-1)
;		There have been NaNs in these data
		nan=where(finite((*v2[index].vis2err)(0:nw-1)) eq 0,n_nan)
		if n_nan gt 0 then (*v2[index].vis2err)(nan)=-1
		scans[k].vissqcerr(j,0:nw-1,l)=(*v2[index].vis2err)(0:nw-1) > 0.001
		eindex=where(byte(*(v2[index[0]].flag)) eq true,ecount)
		if ecount ge 1 then scans[k].vissqcerr(j,eindex,l)=-1
;		The MATISSE pipeline flags more FL data than VIS data!
;		Optionally use FL flags here (experimental!)
		if strpos(configid,'MATISSE') ge 0 $
			and num_nights_fl gt 0 then begin
			index=where(fl.time eq time[k] $
				and fl.insname eq specs[j],count)
; 			Comment next line if VIS2 flags are to be used for FL
;			count=0
			if count gt 0 then begin
			   index=index[0]
			   eindex=where(byte(*fl[index].flag) eq true,ecount)
			   if ecount ge 1 then scans[k].vissqcerr(j,eindex,l)=-1
			endif
		endif
		jndex=where(finite(scans[k].vissqc(j,0:nw-1,l)) ne 1,count)
		if count gt 0 then begin
			scans[k].vissqc(j,jndex,l)=1.0
			scans[k].vissqcerr(j,jndex,l)=-1.0
		endif
		scans[k].starid=startable[ $
			startable_id[v2[index[0]].target_id]].starid
		scans[k].star=targets[startable_id[v2[index].target_id]]
		scans[k].uvw(j,0:nw-1,l,0)=v2[index].ucoord $
					  /genconfig.wavelength[0:nw-1,j]
		scans[k].uvw(j,0:nw-1,l,1)=v2[index].vcoord $
					  /genconfig.wavelength[0:nw-1,j]
		endif
		endif
;
		ENDFOR	; baselines
;
;		Flux data, min(err) set to 0.001 unless flag=true
		for l=0,num_sid-1 do begin
		if count_fl gt 0 then begin
		oiflux_time_offset=median(fl.time-time[k])
		if oiflux_max_time_offset ne 0 and not merged and l eq 0 and j eq 0 then $
		print,'Median FLUX-VIS2 time [s]: ',oiflux_time_offset,', scan:',fix(k)
		if oiflux_max_time_offset ne 0 then $
		index=where(abs(fl.time-time[k]) le oiflux_max_time_offset $
			and fl.insname eq specs[j] $
			and fl.sta_index eq siderostats[l],count) $
		else $
		index=where(fl.time eq time[k] $
			and fl.insname eq specs[j] $
			and fl.sta_index eq siderostats[l],count)
		if count gt 0 then begin
		 index=index[0]
		 scans[k].photometry(l,j,0:nw-1) $
			 =(*fl[index].fluxdata)(0:nw-1)
		 scans[k].photometryerr(l,j,0:nw-1) $
			 =(*fl[index].fluxerr)(0:nw-1) > 0.001
		 eindex=where(byte(*fl[index].flag) eq true,ecount)
;		 The MATISSE pipeline flags more FL data than VIS data!
;		 Optionally use VIS2 flags here (experimental!)
		 if strpos(configid,'MATISSE') ge 0 then $
		 eindex=where(byte(*v2[index].flag) eq true,ecount)
;		 Comment next line if instead the FL flags are used for VIS2
;		 if ecount ge 1 then scans(k).photometryerr(l,j,eindex)=-1
		 jndex=where(finite(scans[k].photometry(l,j,0:nw-1)) ne 1,count)
		 if count gt 0 then begin
			scans[k].photometry(l,j,jndex)=1.0
			scans[k].photometryerr(l,j,jndex)=-1.0
		 endif
		 scans[k].starid=startable[ $
			 startable_id[fl[index[0]].target_id]].starid
		 scans[k].star=targets[startable_id[fl[index[0]].target_id]]
		endif
		endif
		endfor	; stations
;
		ENDFOR	; spectrometers
;
		FOR j=0,num_triple-1 DO BEGIN
			nw=genconfig.triplenumchan[j]
;
;		Triple product data, min(err) set to 0.001 unless flag=true
		index=where(t3.time eq time[k] $
		  and t3.insname+' '+strjoin(t3.sta_index) eq triples[j],count)
		if count gt 0 then begin
		index=index[0]
		scans[k].tripleampc(j,0:nw-1)=(*t3[index].t3amp)(0:nw-1)
		if total(finite((*t3[index].t3amperr)(0:nw-1))) ne 0 then $
			scans[k].tripleampcerr(j,0:nw-1) $
			=(*t3[index].t3amperr)(0:nw-1) > 0.001
; 		Flip sign of phase (based on Iota Peg observations)
		scans[k].triplephasec(j,0:nw-1) $
			=((-(*t3[index].t3phi)(0:nw-1)) mod 360)/RAD
		if min(scans[k].triplephasec(j,0:nw-1)) lt -!pi $
			then scans[k].triplephasec(j,0:nw-1) $
				=scans[k].triplephasec(j,0:nw-1)+2*!pi
		scans[k].triplephasecerr(j,0:nw-1) $
			=(*t3[index].t3phierr)(0:nw-1)/RAD > 0.001
		eindex=where(byte(*t3[index].flag) eq true,ecount)
		if ecount ge 1 then begin
			scans[k].tripleampcerr(j,eindex)=-1
			scans[k].triplephasecerr(j,eindex)=-1
		endif
		jndex=where(finite(scans[k].tripleampc(j,0:nw-1)) ne 1,count)
		if count gt 0 then begin
			scans[k].tripleampc(j,jndex)=1.0
			scans[k].tripleampcerr(j,jndex)=-1.0
			scans[k].triplephasec(j,jndex)=1.0
			scans[k].triplephasecerr(j,jndex)=-1.0
		endif
		jndex=where(finite(scans[k].triplephasec(j,0:nw-1)) ne 1,count)
		if count gt 0 then begin
			scans[k].triplephasec(j,jndex)=1.0
			scans[k].triplephasecerr(j,jndex)=-1.0
			scans[k].tripleampc(j,jndex)=1.0
			scans[k].tripleampcerr(j,jndex)=-1.0
		endif
		scans[k].starid=startable[ $
			startable_id[t3[index[0]].target_id]].starid
		scans[k].star=targets[startable_id[t3[index[0]].target_id]]
;		Make sure we follow the rules...
		fBaseMatrix=intarr(GenConfig.NumSid,3)
		fBaseFactor=fltarr(3)+1
	        for l=0,2 do begin
                	ob=GenConfig.TripleBeam[l,j]
                	bl=GenConfig.TripleBase[l,j]
                	j1=where(GenConfig.StationId eq $
				strmid(GenConfig.BaselineId[bl,ob],0,3))
                	j2=where(GenConfig.StationId eq $
				strmid(GenConfig.BaselineId[bl,ob],4,3))
                	fBaseMatrix[j1,l]=+1
                	fBaseMatrix[j2,l]=-1
                	if l gt 0 then fBaseFactor[l]= $
				-total(fBaseMatrix[*,0]*fBaseMatrix[*,l])
        	endfor
		l=0
		scans[k].uvw(genconfig.triplebeam[l,j], $
			     0:genconfig.triplenumchan[j]-1, $
			     genconfig.triplebase[l,j],0) $
			=t3[index].u1coord*fBaseFactor[l] $
			/genconfig.wavelength[0:genconfig.triplenumchan[j]-1, $
					      genconfig.triplebeam[l,j]]
		scans[k].uvw(genconfig.triplebeam[l,j], $
			     0:genconfig.triplenumchan[j]-1, $
			     genconfig.triplebase[l,j],1) $
			=t3[index].v1coord*fBaseFactor[l] $
			/genconfig.wavelength[0:genconfig.triplenumchan[j]-1, $
					      genconfig.triplebeam[l,j]]
		l=1
		scans[k].uvw(genconfig.triplebeam[l,j], $
			     0:genconfig.triplenumchan[j]-1, $
			     genconfig.triplebase[l,j],0) $
			=t3[index].u2coord*fBaseFactor[l] $
			/genconfig.wavelength[0:genconfig.triplenumchan[j]-1, $
					      genconfig.triplebeam[l,j]]
		scans[k].uvw(genconfig.triplebeam[l,j], $
			     0:genconfig.triplenumchan[j]-1, $
			     genconfig.triplebase[l,j],1) $
			=t3[index].v2coord*fBaseFactor[l] $
			/genconfig.wavelength[0:genconfig.triplenumchan[j]-1, $
					      genconfig.triplebeam[l,j]]
		l=2
		scans[k].uvw(genconfig.triplebeam[l,j], $
			     0:genconfig.triplenumchan[j]-1, $
			     genconfig.triplebase[l,j],0) $
			=(-t3[index].u1coord-t3[index].u2coord)*fBaseFactor[l] $
			/genconfig.wavelength[0:genconfig.triplenumchan[j]-1, $
					      genconfig.triplebeam[l,j]]
		scans[k].uvw(genconfig.triplebeam[l,j], $
			     0:genconfig.triplenumchan[j]-1, $
			     genconfig.triplebase[l,j],1) $
			=(-t3[index].v1coord-t3[index].v2coord)*fBaseFactor[l] $
			/genconfig.wavelength[0:genconfig.triplenumchan[j]-1, $
					      genconfig.triplebeam[l,j]]
		endif
;
		ENDFOR	; triples
;
		ENDFOR	; scans
;
;		Here we have a good night
		genconfig.date=checkdate()
		geoparms.date=date
; 		Default is to clear AMOEBA's buffer and initialize Info arrays
		if n_elements(freememory) eq 0 then freememory=1
		if freememory then begin
			freememory
			GenInfo=replicate(GenConfig,1)
			GeoInfo=replicate(GeoParms,1)
			if num_sets gt 1 then $
			freememory=0	; OIFITS file can have more than one set
		endif
;		Next 4 lines commented for OIFITS in load_interferometry
;		if i eq 0 then freememory
;		if i eq 0 and num_sets gt 1 then begin
;			bufferinfo=replicate(nightinfo(),num_sets)
;		endif
		get_scantable
;		Store INT_TIME information indirectly in scantable
		scantable.stoptime=scantable.starttime+median(int_time)
; 		Store photometric INT_TIME needed for flux calibration
		if int_time_fl[0] gt 0 then $
		scans.int_time=median(int_time_fl) else $
		scans.int_time=scantable.stoptime-scantable.starttime
;
;		Use the uv-coordinates from the data file
		calcastrom,/skipuv
		scans.vissq=scans.vissqc
		scans.vissqerr=scans.vissqcerr
		scans.vissqe=scans.vissq
		scans.vissqeerr=scans.vissqerr
		scans.vissqec=scans.vissqc
		scans.vissqecerr=scans.vissqcerr
		scans.tripleamp=scans.tripleampc
		scans.tripleamperr=scans.tripleampcerr
		scans.tripleampe=scans.tripleamp
		scans.tripleampeerr=scans.tripleamperr
		scans.tripleampec=scans.tripleampc
		scans.tripleampecerr=scans.tripleampcerr
		scans.triplephase=scans.triplephasec
		scans.triplephaseerr=scans.triplephasecerr
;
; 		When reading VLTI data produced by, e.g. pndrs,
;		convert station names to standard OYSTER format.
		if systemid eq 'VLTI' then begin
		vlt_stn=strmid(stationids,1,2)+'_'
		for n=0,genconfig.numsid-1 do begin
		  j=where(strpos(vlt_stn,genconfig.stationid[n]) eq 0,count)
		  if j[0] gt 7 then begin
			  genconfig.stationid[n]=stationids[j]
		  endif else if j[0] gt 3 then begin
;			  Shift index to replace proper UT designation
			  genconfig.stationid[n]=stationids[j-4]
		  endif
		endfor
		if count gt 0 then begin
		  print,'Station names converted.'
		  get_stationtable,update=0
		endif
		for ob=0,genconfig.numoutbeam-1 do begin
;		for n=0,genconfig.numbaseline(ob)-1 do begin
		for n=0,MB-1 do begin	; pndrs: n(BL) < n(n-1) with all tr!
		j=where(strpos(vlt_stn,strmid(genconfig.baselineid[n,ob],0,3)) $
		    eq 0)
		k=where(strpos(vlt_stn,strmid(genconfig.baselineid[n,ob],4,3)) $
		    eq 0)
	    	j=j[0] & k=k[0]
		if j ne -1 and k ne -1 then begin
		  if j le 7 then j=j-4; shift index to replace label for UTs
		  if k le 7 then k=k-4; shift index to replace label for UTs
		  genconfig.baselineid[n,ob]=stationids[j]+'-'+stationids[k]
		endif
		endfor
		endfor
		bl_ids=genconfig.baselineid
		genconfig.configid=genconfig.configid+'_' $
;		Use vlt_stn string (could be ambiguous)
;		+strjoin(genconfig.stationid(0:num_sid-1),'-')
;		Use baseline string (unique), but could be very loooong...
;		+strjoin(genconfig.baselineid,'_')
;		Use unique station string (shorter than baseline string)
;		+unique_station_config(genconfig.baselineid)
;		Use unique station string (with blanks removed)
		+unique_station_config( $
			bl_ids[where(strlen(strtrim(bl_ids)) gt 0)])
;		Remove possible occurrences of double-underscores
		words=nameparse(genconfig.configid,'__')
		genconfig.configid=strjoin(words,'_')
		endif	; systemid='VLTI'
;
; 		If reading PIONIER files with mode FREE,
;		allocate OB for each baseline (enable for special cases)
		if n_elements(pionierfree) eq 0 then pionierfree=0
		if pionierfree and strpos(genconfig.configid,'PIONIER') eq 0 $
		 and genconfig.numspecchan[0] eq 1 then set_pionierfree
;
;		Buffer scandata if more than one data set is in the OIFITS file
		if num_sets gt 1 then begin
;			Next 6 lines commented 2016-02-24
;			g=geninfo(i)
;			struct_assign,genconfig,g
;			geninfo(i)=g
;			g=geoinfo(i)
;			struct_assign,geoparms,g
;			geoinfo(i)=g
			storenight
		endif else begin
;			Next 2 lines commented 2016-02-24
;			geninfo=genconfig
;			geoinfo=geoparms
		endelse
	endif
endfor
;
; Complete some information in StarTable...
; PIONIER data reduction: which stars were used as calibrators?
oidiam_file=file_search('.','????-??-??_oiDiam.fits')
if strlen(oidiam_file[0]) gt 0 then begin
	i=where(strpos(oidiam_file,date) ge 0,count)
	if count gt 0 then begin
		t=mrdfits(oidiam_file[i[0]],1,header)
		index=where(strpos(t.target,'INTERNAL') lt 0)
		t=t[index]
		if n_elements(t) eq n_elements(startable) then begin
			index=where(t.iscal eq 1,count)
			if count gt 0 then startable[index].bflag='C'
		endif
	endif
endif
; If any star name is an OYSTER Star ID, fill HDN or HIC numbers
for i=0,n_elements(startable)-1 do begin
	if valid_cat(startable[i].starid) then begin
;		id=long(strmid(startable(i).starid,3,6))
;		There are cases such as HIP_5300
		id=long(strmid(strjoin(nameparse(startable[i].starid, $
			['_','-'])),3,6))
		case strmid(startable[i].starid,0,3) of
		'HDN':startable[i].hdn=id
		'HIP':startable[i].hic=id
		'BSC':startable[i].bsc=id
		'FKV':startable[i].fkv=id
		else:
		endcase
	endif
endfor
; Is this still needed?
startable=startable[uniq(startable.starid)]
;
pipefile=fitshparse(h,'PIPEFILE',/quiet)+''
if strpos(pipefile,'RAW_INT') ge 0 or $
   strpos(pipefile,'CAL_INT') ge 0 then begin
;
; 	MATISSE flux calibration: counts -> Jansky
	if strpos(genconfig.configid,'MATISSE_disabled') ge 0 and $
		  genconfig.numoutbeam eq 1 then begin
;	Remove instrumental response first: photometry -> photometry!
	matisse_response; Comment this for matisse_cal_response
;
;	...then calibrate in Jy: photometry -> photometryc!
;	...unless we process data to compute the zerocounts
	if strpos(pwd(),'InputZeroCounts') lt 0 then begin
	restore,!oyster_dir+'source/matisse/zerocounts.xdr'
	if strpos(genconfig.configid,'-L_') ge 0 then begin
		band='L'
		ins_mode=fitshparse(h,'INS.DIL.ID')
	endif
	if strpos(genconfig.configid,'-N_') ge 0 then begin
		band='N'
		ins_mode=fitshparse(h,'INS.DIN.ID')
	endif
	if strpos(genconfig.stationid[0],'A') ge 0 then array='AT'
	if strpos(genconfig.stationid[0],'U') ge 0 then array='UT'
	det_mode=fitshparse(h,'DET.READ.CURNAME')
	det_dit=strmid( $
		strtrim(string(double(fitshparse(h,'DET.SEQ1.DIT'))),1),0,10)
	tpl_nexp=strtrim(string(fix(fitshparse(h,'TPL.NEXP'))),1)
	index=where(zerocounts.band eq band and $
		    zerocounts.array eq array and $
		    zerocounts.ins_mode eq ins_mode and $
		    zerocounts.det_mode eq det_mode and $
		    zerocounts.det_dit eq det_dit and $
		    zerocounts.tpl_nexp eq tpl_nexp,count)
	coeff_factor=1.
	if count eq 0 then begin
		if tpl_nexp eq '40' then begin
		index=where(zerocounts.band eq band and $
		    zerocounts.array eq array and $
		    zerocounts.ins_mode eq ins_mode and $
		    zerocounts.det_mode eq det_mode and $
		    zerocounts.det_dit eq det_dit and $
		    zerocounts.tpl_nexp eq '52',count)
	    	if count gt 0 and band eq 'L' then coeff_factor=1./2
	    	if count gt 0 and band eq 'N' then coeff_factor=2
		endif
		if tpl_nexp eq '52' then begin
		index=where(zerocounts.band eq band and $
		    zerocounts.array eq array and $
;		    zerocounts.ins_mode eq ins_mode and $
;		    zerocounts.det_mode eq det_mode and $
;		    zerocounts.det_dit eq det_dit and $
		    zerocounts.tpl_nexp eq '40',count)
	    	if count gt 0 and band eq 'L' then coeff_factor=2
	    	if count gt 0 and band eq 'N' then coeff_factor=1./2
		endif
	endif
; 	Compute median of coefficients for each telescope (siderostat)
	if count ne 0 then begin
;	Found zerocount coefficients
	zcs=fltarr(genconfig.numsid)
	if count eq 1 then zcs=zerocounts[index].zcs*coeff_factor $
	else for i=0,genconfig.numsid-1 do $
		zcs[i]=median(zerocounts[index].zcs(i)*coeff_factor)
; 	Compute estimated L or N band fluxes in Jy for each telescope
	fln=matisse_zerocounts(zcs=zcs,tm=tm)
; 	Calibrate scans.photometry
	i=0	; Single-star observation only
	for k=0,n_elements(scans)-1 do begin
	for j=0,genconfig.numsid-1 do begin
;	 Compute total flux and normalize to target flux in L or N band
	 f_band=total(median(reform(scans[k].photometry(j,0,*)),5)*tm)/total(tm)
	 if f_band gt 0 then begin
	 if fitshistory(h,'mat_mergeAllOiFits') then f_band=f_band/4
	 scans[k].photometryc(j,0,*)=scans[k].photometry(j,0,*)*fln[i,j]/f_band
	 scans[k].photometrycerr(j,0,*)=scans[k].photometryerr(j,0,*)/f_band
	 endif else scans[k].photometrycerr(j,0,*)=-1
	endfor
	endfor
	endif	; Data calibrated in Jy
;
	endif	; Not called to compile zero counts
	endif	; MATISSE
endif	; Reflex pipeline product
;
; Read spectrum if available, single night and target only
if n_elements(sed_file) eq 0 then sed_file='sed.dat'
sed_file=file_search(sed_file)
if strlen(sed_file) gt 0 and num_sets eq 1 $
			  and n_elements(startable) eq 1 then begin
	status=dc_read_free(sed_file,lambda,flux,/col)
	if status eq 0 then begin
		print,'Read spectrum (SED) from file '+sed_file
        	for k=0,genconfig.numoutbeam-1 do begin
        	nc=genconfig.numspecchan[k]
        	for i=0,n_elements(scans)-1 do scans[i].photometry(0,k,0:nc-1) $
	                =spline(lambda,flux,genconfig.wavelength[0:nc-1,k]*1e6)
       		endfor
	endif
	scans.photometryc=scans.photometry
endif else begin
;	scans.photometry=0
endelse
;
; Read seeing data from text files, if available (single night only)
; dfits PIONI*.fits | fitsort ISS.AMBI.FWHM.START > r0.txt
; dfits PIONI*.fits | fitsort ISS.AMBI.FWHM.END >> r0.txt
; sort r0.txt > r0.dat
; sed -i '1,2d' r0.dat
; Remove all lines without r0 values or other extra text
if n_elements(r0_file) eq 0 then r0_file='r0.dat'
r0_file=file_search(r0_file)
if strlen(r0_file) gt 0 and num_sets eq 1 then begin
	l=''
	status=dc_read_fixed(r0_file,l,/col,format='(a80)')
	if status eq 0 then begin
		print,'Read r0 data from file '+r0_file
		nl=n_elements(l)
		t=fltarr(nl)
		r0=fltarr(nl)
		ipos=strpos(l[0],'.')
		for k=0,nl-1 do begin
			words=nameparse(l[k])
			t[k]=hms2h(strmid(words[0],ipos+12,12))
			r0[k]=float(words[1])
		endfor
		index=where(r0 gt 0,count)
		if count gt 0 then begin
			t=t[index]
			r0=r0[index]
			index=where(t gt 24-system_config(systemid[0], $
				'MIDNIGHT'),count)
			if count gt 0 then t[index]=t[index]-24
			scans.r0=gsmooth(t,r0,0.02,scans.time/3600)
		endif
	endif
endif else scans.r0=1	; Default 1" r0
; dfits PIONI*.fits | fitsort ISS.AMBI.TAU0.START >> t0.txt
; dfits PIONI*.fits | fitsort ISS.AMBI.TAU0.END >> t0.txt
; sort t0.txt > t0.dat
; sed -i '1,2d' t0.dat
; Remove all lines without t0 values or other extra text
if n_elements(t0_file) eq 0 then t0_file='t0.dat'
t0_file=file_search(t0_file)
if strlen(t0_file) gt 0 and num_sets eq 1 then begin
	l=''
	status=dc_read_fixed(t0_file,l,/col,format='(a80)')
	if status eq 0 then begin
		print,'Read t0 data from file '+t0_file
		nl=n_elements(l)
		t=fltarr(nl)
		t0=fltarr(nl)
		ipos=strpos(l[0],'.')
		for k=0,nl-1 do begin
			words=nameparse(l[k])
			t[k]=hms2h(strmid(words[0],ipos+12,12))
			t0[k]=float(words[1])
		endfor
		index=where(t0 gt 0,count)
		if count gt 0 then begin
			t=t[index]
			t0=t0[index]
			index=where(t gt 24-system_config(systemid[0], $
				'MIDNIGHT'),count)
			if count gt 0 then t[index]=t[index]-24
;			t0=fltarr(NOB,MC,MB)
			r=gsmooth(t,t0,0.02,scans.time/3600)
			for i=0,n_elements(scans)-1 do scans[i].t0=r[i]
		endif
	endif
endif else scans.t0=0.001	; Default 0.001s t0
if strlen(r0_file) eq 0 and strlen(t0_file) eq 0 then begin
;	Alternatively, check if file has header with r0/t0/PWV values
	d=mrdfits(fitsfile,0,header)
	s_start=float(fitshparse(header,'ISS.AMBI.FWHM.START',/quiet))
	s_end=float(fitshparse(header,'ISS.AMBI.FWHM.END',/quiet))
	if s_start gt 0 and s_end gt 0 then scans.r0=(s_start+s_end)/2.
	t_start=float(fitshparse(header,'ISS.AMBI.TAU0.START',/quiet))
	t_end=float(fitshparse(header,'ISS.AMBI.TAU0.END',/quiet))
	if t_start gt 0 and t_end gt 0 then scans.t0[*]=(t_start+t_end)/2.
	p_start=float(fitshparse(header,'ISS.AMBI.IWV.START',/quiet))
	p_end=float(fitshparse(header,'ISS.AMBI.IWV.END',/quiet))
	if p_start gt 0 and p_end gt 0 then scans.pwv=(p_start+p_end)/2.
endif
;
; Test validity of station coordinates
uvw_bck=scans.uvw
uv=reform(scans[*].uvw(*,*,*,0:1))
calcastrom
uv_c=reform(scans[*].uvw(*,*,*,0:1))
;
n=size(uv,/dim)
print,'Checking uv-coordinates for case '+string(n_elements(n),format='(i1)')
case n_elements(n) of
	1:	begin
		u=uv[*]
		v=uv[*]
		u_c=uv_c[*]
		v_c=uv_c[*]
		end
	2:	begin
		u=uv[*,0]
		v=uv[*,1]
		u_c=uv_c[*,0]
		v_c=uv_c[*,1]
		end
	3:	begin
		u=uv[*,*,0]
		v=uv[*,*,1]
		u_c=uv_c[*,*,0]
		v_c=uv_c[*,*,1]
		end
	4:	begin
		u=uv[*,*,*,0]
		v=uv[*,*,*,1]
		u_c=uv_c[*,*,*,0]
		v_c=uv_c[*,*,*,1]
		end
	5:	begin
		u=uv[*,*,*,0,*]
		v=uv[*,*,*,1,*]
		u_c=uv_c[*,*,*,0,*]
		v_c=uv_c[*,*,*,1,*]
		end
endcase
index=where(u lt 0,count)
if count gt 0 then begin
	u[index]=-u[index]
	v[index]=-v[index]
endif
index=where(u_c lt 0,count)
if count gt 0 then begin
	u_c[index]=-u_c[index]
	v_c[index]=-v_c[index]
endif
index=where(v lt 0,count)
if count gt 0 then begin
	u[index]=-u[index]
	v[index]=-v[index]
endif
index=where(v_c lt 0,count)
if count gt 0 then begin
	u_c[index]=-u_c[index]
	v_c[index]=-v_c[index]
endif
uv=[[u],[v]]
uv_c=[[u_c],[v_c]]
index=where(uv ne 0)
;
if keyword_set(uvcheck) then begin
	uvcheck_file=basename(fitsfile)+'_uvcheck.txt'
	openw,unit,uvcheck_file,/get_lun
	printf,unit,fitsfile
endif
print,'Median absolute deviation in uv [m]:', $
	medianve(abs(uv[index]-uv_c[index]))*median(genconfig.wavelength)
if keyword_set(uvcheck) then $
printf,unit,'Median absolute deviation in uv [m]:', $
	medianve(abs(uv[index]-uv_c[index]))*median(genconfig.wavelength)
print,'Maximum absolute deviation in uv [m]:', $
	max(abs(uv[index]-uv_c[index]))*mean(genconfig.wavelength)
if keyword_set(uvcheck) then $
printf,unit,'Maximum absolute deviation in uv [m]:', $
	max(abs(uv[index]-uv_c[index]))*mean(genconfig.wavelength)
;
p=100*medianve((uv[index]-uv_c[index])/uv[index],e)
e=e*100
print,'Median percentage deviation in uv: '+ $
		string(p,format='(f6.3)')+'% +/- '+string(e,format='(f7.2)')+'%'
if keyword_set(uvcheck) then $
printf,unit,'Median percentage deviation in uv: '+ $
		string(p,format='(f6.3)')+'% +/- '+string(e,format='(f7.2)')+'%'
;
if p gt 1 or e gt 1 then begin
	print,'Warning: station coordinates inconsistent with uvw coordinates!'
	print,'The uv coordinates of the data file will be used.'
endif
scans.uvw=uvw_bck
if keyword_set(uvcheck) then free_lun,unit
;
; Update the menu options in the DataInterferometry GUI
if n_elements(opmenua_wid) eq 0 then opmenua_wid=0
if n_elements(opmenup_wid) eq 0 then opmenup_wid=0
if opmenua_wid ne 0 and opmenup_wid ne 0 then begin
	types=['Vsqr','TAmp','Both','None']
	sds=3
	if ds_options.v2 then sds=0
	if ds_options.ta then sds=1
	if ds_options.v2 and ds_options.ta then sds=2
	widget_control,opmenua_wid,set_droplist_select=sds
	types=['VPhs','TPhs','Both','None']
	sds=3
	if ds_options.vp then sds=0
	if ds_options.tp then sds=1
	if ds_options.vp and ds_options.tp then sds=2
	widget_control,opmenup_wid,set_droplist_select=sds
	print,'Interferometric data type selection updated.'
endif
;
; Make sure amoeba knows about this file, unless this procedure was called
; by load_interferometry with a temporary file.
chafiles=''
if fitsfile ne 'temp_do_not_keep.fits' then $
oiffiles=fitsfile
;
; Name of file is used by OYSTER for auxilliary files
if fitsfile ne 'temp_do_not_keep.fits' then $
	hds_file_stub=basename(fitsfile) else $
	hds_file_stub=genconfig.date+'_' $
		+system_id(systemid)+'_' $
		+genconfig.configid
;
; Enforce strict formatting of ConfigId (currently disabled)
genconfig.configid=strict_configid(genconfig.configid)
;
end
