pro get_wds2007,starids
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Note: this procedure accesses the WDS orbit catalog with the old format.
;
; Allocate startable and read orbital element data from WDS orbit solution file.
; Read the orbital element data from the "Sixth Catalog of Orbits of
; Visual Binary Stars" by W.I. Hartkopf and B.D. Mason, USNO.
; The file is !catalogs_dir/wds/orb6orbits.txt, and can by downloaded from
; http://ad.usno.navy.mil/ad/wds/orb6.html. Unfortunately, this file does
; not contain the equinox for the ascending node. We get this information
; from the ephemeris file orb6ephem.txt.
;
; This procedure uses or prepares an XDR file for faster access.
; Set startable with data from requested stars.
;
common StarBase,StarTable,Notes
;
RAD=180/!pi
;
; Read wdsorb.xdr, if it exists
orbfile=!catalogs_dir+'proprietary/wds2007/orb6orbits.xdr'
result=file_search(orbfile)
IF strlen(result[0]) ne 0 THEN restore,orbfile $
			  ELSE BEGIN
;
; Read orb6ephem.txt
;
;000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000011111111111111111111111111111111
;000000000011111111112222222222333333333344444444445555555555666666666677777777778888888888999999999900000000001111111111222222222233
;012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901
;00014+3937  HLD  60       4  Hei1963a  2000 1961 1960-85    173.9  1.191   173.5  1.197   173.1  1.203   172.8  1.208   172.4  1.214
;
rec=''
status=dc_read_fixed(!catalogs_dir+'proprietary/wds2007/orb6ephem.txt',rec,/col, $
        resize=[1],format='(a132)',ignore=['!'])
num_ephem=n_elements(rec)
wds1=strmid(rec,0,10)
;
equinx_string=strmid(rec,39,4)
index=where(equinx_string eq '    ',count)
if count gt 0 then equinx_string[index]='   0'
equinx=double(equinx_string)
index=where(equinx eq 0,count)
if count gt 0 then equinx[index]=2000
;
; Read orb6orbits.txt
;
; Format template of txtorbits.txt (text format)
;00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001111111111111111111111111111111111111111111111111111111111111111
;00000000001111111111222222222233333333334444444444555555555566666666667777777777888888888899999999990000000000111111111122222222223333333333444444444455555555556666
;01234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123
;23 37 23.2 +07 37 26  23374+0737FOX 102AB   16873 222068 116571   8.32   8.42     221.06     0.413     46.4    282.0    1987.71    0.354    40.6    Sey2000b 4   P E
;RA,Dec (2000)....... WDS....... Name....... ADS.. HD.... HIP...  V1...* V2...*  ....P..... ....a..... ...i.... Omega..* ....T..... ...e... omega... Referenc G N P E
;
rec=''
status=dc_read_fixed(!catalogs_dir+'proprietary/wds2007/orb6orbits.txt',rec,/col, $
	resize=[1],format='(a164)',ignore=['!'])
num_orbit=n_elements(rec)
wds2=strmid(rec,22,10)
if num_orbit ne num_ephem then begin
	print,'***Error(GET_WDS): files corrupt?'
	return
endif
;
; Remove records with missing data in various fields
index=where(strlen(strcompress(strmid(rec,139,07),/remove_all)) ne 0,num_orbit) & rec=rec[index] & equinx=equinx[index]
index=where(strlen(strcompress(strmid(rec,120,10),/remove_all)) ne 0,num_orbit) & rec=rec[index] & equinx=equinx[index]
index=where(strlen(strcompress(strmid(rec,111,07),/remove_all)) ne 0,num_orbit) & rec=rec[index] & equinx=equinx[index]
;
; Decode the data
;
wds=long(strmid(rec,21,5)+strmid(rec,28,4))
index=where(strmid(rec,27,1) eq '-')
sign=strarr(n_elements(rec))
sign[*]='+'
sign[index]='-'
wdsstarids='WDS'+sign+stringl(wds,format='(i9.9)')
wds[index]=-wds[index]
;
hic_string=strmid(rec,57,6)
index=where(strlen(strcompress(hic_string,/remove_all)) eq 0,count)
if count gt 0 then hic_string[index]='0'
hic=long(hic_string)
;
rah=double(strmid(rec,0,2))
ram=double(strmid(rec,3,2))
ras=double(strmid(rec,6,4))
ra=rah+ram/60+ras/3600
;
decd=double(strmid(rec,12,2))
decm=double(strmid(rec,15,2))
decs_string=strmid(rec,15,2)
index=where(strlen(strcompress(decs_string,/remove_all)) eq 0,count)
if count gt 0 then decs_string[index]='0'
decs=double(decs_string)
dec=decd+decm/60+decs/3600
index=where(strmid(rec,11,1) eq '-',count)
if count gt 0 then dec[index]=-dec[index]
;
ads_string=strmid(rec,44,5)
index=where(strlen(strcompress(ads_string,/remove_all)) eq 0,count)
if count gt 0 then ads_string[index]='0'
ads=long(ads_string)
;
mag_string=strmid(rec,65,5)
index=where(strlen(strcompress(mag_string,/remove_all)) eq 0,count)
if count gt 0 then mag_string[index]='+100'
index=where(strpos(mag_string,'var') ne -1,count)
if count gt 0 then mag_string[index]='+100'
ma=double(mag_string)
;
mag_string=strmid(rec,72,5)
index=where(strlen(strcompress(mag_string,/remove_all)) eq 0,count)
if count gt 0 then mag_string[index]='+100'
index=where(strpos(mag_string,'var') ne -1,count)
if count gt 0 then mag_string[index]='+100'
mb=double(mag_string)
;
dmv=mb-ma
index=where(ma gt 99 or mb gt 99,count)
if count gt 0 then dmv[index]=100
;
period=double(strmid(rec,81,12))*365.25		; days
amajor=double(strmid(rec,92,9))			; arc seconds
inclin=double(strmid(rec,102,6))		; degrees
ascnod=double(strmid(rec,111,7)) & ascnod=precess_pa(ascnod/RAD,equinx,ra,dec,1)*RAD
epochp=double(strmid(rec,120,10))
epoch_jd=lonarr(num_orbit)
for i=0,num_orbit-1 do epoch_jd[i]=julian(long(epochp[i]),1,1)
years=intarr(num_orbit)+365
index=where(((long(epochp) mod 4) eq 0) and ((long(epochp) mod 400) ne 0),count)
if count gt 0 then years[index]=366
epoch_jd=epoch_jd+(epochp mod 1)*years
eccent=double(strmid(rec,131,6))
argper=double(strmid(rec,139,7))
name=strmid(rec,32,11)
;
create_startable,wdsstarids
StarTable.name=strcompress(name)
StarTable.bflag='B'
StarTable.ra=ra
StarTable.dec=dec
StarTable.ads=ads
StarTable.wds=wds
StarTable.hic=hic
;
StarTable.mv=cmag(ma,mb)
StarTable.dmv=dmv
StarTable.p=period
StarTable.a=amajor
StarTable.i=inclin
StarTable.n=ascnod
StarTable.t=epoch_jd
StarTable.e=eccent
StarTable.o=argper
;
; Save this catalog to an XDR file
save,StarTable,filename=!catalogs_dir+'proprietary/wds2007/orb6orbits.xdr'
;
ENDELSE
;
n=n_elements(starids)
if n ne 0 then begin
	index=lonarr(n)-1
	m=0
	for i=0,n-1 do begin
		k=where(startable.starid eq starids[i],l)
		if l gt 1 then index=[index,-1]
		if l gt 0 then index[i+m:i+m+l-1]=k
		if l gt 1 then m=m+1
	endfor
	jndex=where(index eq -1,count)
	for i=0,count-1 do print,'Star not found: ',starids[jndex[i]]
	jndex=where(index ge 0,count)
	if count gt 0 then begin
		index=index[jndex]
		startable=startable[index]
	endif
endif
;
end
