function glushneva,starid,lammda
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Return the count rate as a function of wavelength for a given star.
; Find the closest match in class and type from a data base of secondary
; photometric standards (Glushneva et al. 1992, A&AS 92, 1), and scale
; with the blackbody flux ratio arising from the temperature mismatch, if any.
; Also extrapolate the flux from 767.5 nm to 850 nm using the black body law.
;
; Return lammda [m]
;
forward_function teff_star
;
common StarBase,startable,notes
;
; Save startable if it exists
if n_elements(startable) eq 0 then get_startable,starid
table=startable
;
; Get the information for the star
index=where(startable.starid eq starid,count)
if count eq 0 then begin
	get_startable,starid
	index=0
endif
type=startable[index[0]].type
class=startable[index[0]].class
mv=startable[index[0]].mv
teff_stars
teff=startable[index[0]].teff
;
; Get the stars for which spectra are available
s=dc_read_free(!atmospheres_dir+'glushneva/table3',bsc,/col)
stars='BSC'+string(bsc,format='(i4.4)')
get_startable,stars
teff_stars
;
; Find the closest match and read flux data
index=where(startable.class eq class,count)
if count eq 0 then begin
	print,'***Error(GLUSHNEVA): class not found!'
	return,0
endif
startable=startable[index]
sep=abs(startable.type-type)
index=where(sep eq min(sep)) & index=index[0]
jndex=where(stars eq startable[index].starid) & jndex=jndex[0]
print,'Found matching star: ',stars[jndex]
var=fltarr(91)
s=dc_read_free(!atmospheres_dir+'glushneva/table3',var,nrec=1,nskip=jndex)
fluxes=median(var[1:85],3)
lammda=(findgen(85)*5+322.5)*1e-9
;
; Calculate the shape correction for a mismatch in temperature, normalize to V
f_ratio=(exp(1.439d-2/(lammda*abs(startable[index].teff)))-1) $
       /(exp(1.439d-2/(lammda*abs(teff)))-1)
f_ratio=f_ratio/f_ratio[46]
;
; Adjust fluxes for temperature and brightness mismatch
fluxes=fluxes*f_ratio*10^((startable[index].mv-mv)/2.5)
;
; Check the V flux against absolute calibration of 3.75e-9 erg/cm^2/s/A
fv0=total(fluxes*johnson_v(lammda*1e9))*10^(mv/2.5)/total(johnson_v(lammda*1e9))
print,'Predicted absolute flux V=0: ',fv0*10,', should be 3.75'
factor=0.375/fv0
fluxes=fluxes*factor
;
; Extend arrays for the long wavelength extrapolation
lammda=[lammda,(findgen(25)*5+747.5)*1e-9]
f_ratio=median(fluxes[80:84])/ $
	((3.742d-30/lammda[82]^5)/(exp(1.439d-2/(lammda[82]*abs(teff)))-1))
fluxes=median([fluxes,f_ratio*((3.742d-30/lammda[85:109]^5) $
		      /(exp(1.439d-2/(lammda[85:109]*abs(teff)))-1))],7)
;
; Restore original startable, if needed
if n_elements(table) ne 0 then startable=table
;
; This is the energy (ergs) of a photon of wavelength lammda
hnu=6.6262e-27*2.997925e8/lammda
;
; Return rate/cm^2/s/nm
return,(fluxes/hnu)*1e-7
;
end
