pro gridfit,num_grid,grid_cell,component=component,files=files,gridpsn=gridpsn
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Given a loaded binary model (method 1), predict rho/theta for the
; observation date, and look for the global minimum in ChiSq by varying r/t
; for the specified component, e.g. 'A-B'. Make sure no data other than
; interferometry is loaded because of use of modelchisq() to compute ChiSq.
;
; Write Chi^2 surface data to file YYYY-MM-DD.chisq.
;
; Loop over files if given, otherwise use the currently loaded data.
; e.g. gridfit,150,0.04,files='200?-??-??.cha'
; e.g. gridfit,150,0.04,files='201?-??-??.cha'
; e.g. gridfit,150,0.04,files='201[7-8]-??-??.cha'
;
; This procedure can be called also by gridfit_parallel
;
; If keyword gridpsn is set, num_grid=100 and grid_cell=cb/200 are used.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common MarquardtFit,fit_options
common OysterBridge,obd
;
rad=180/!pi
;
; Restore data if run within IDL bridge object
if n_elements(obd) ne 0 then restore,obd
;
if n_elements(binary_model) eq 0 then begin
	modelfile=''
	read,modelfile,prompt='Please enter name of model file: '
	readmodel,modelfile
endif
if n_elements(component) eq 0 and num_binary() gt 1 then begin
	component=''
	read,component,prompt='Please enter component (e.g. A-B): '
endif
if n_elements(component) ne 0 then k=where(binary_model.component eq component)$
			      else k=0
method_input=binary_model[k].method
;
if keyword_set(gridpsn) then begin
	print,'Parameters for use with gridpsn requested!'
	cb=cleanbeam(dirtybeam(gen_model.starid))
	grid_cell=cb[1]/100
	num_grid=150
	print,'Will use num_grid',num_grid,' and grid_cell ',grid_cell
endif
if n_elements(num_grid) eq 0 then num_grid=100L else num_grid=long(num_grid)
if n_elements(grid_cell) eq 0 then grid_cell=0.5
;
print,'Grid cell [mas]: ',grid_cell
print,'Size of search box [mas]: ',num_grid*grid_cell
;
if n_elements(files) ne 0 then begin
	files=file_search(files)
	if strlen(files[0]) eq 0 then begin
		print,'GRIDFIT: Files not found!'
		return
	endif
	n_files=n_elements(files)
endif else n_files=1
;
FOR i_file=0,n_files-1 DO BEGIN
;
status=0
if n_elements(files) ne 0 then get_data,files[i_file],status
if status ne 0 then continue
;
parsedate,date,y,m,d
d=float(d)+system_config(systemid,'MIDNIGHT')/24.
for i=0,n_elements(binary_model)-1 do begin
	rt=binarypos(julian(y,m,d),binary_model[i].component)
	if rt[1] lt 0 then rt[1]=rt[1]+360
	binary_model[i].rho=rt[0]
	binary_model[i].theta=rt[1]
endfor
binary_model.method=2
;
x0=-binary_model[k].rho*sin(binary_model[k].theta/rad)
y0=+binary_model[k].rho*cos(binary_model[k].theta/rad)
;
n=num_grid*num_grid
grid_x=(lindgen(n) mod num_grid)*grid_cell
grid_y=(lindgen(n)/num_grid)*grid_cell
grid_x=reform(grid_x,num_grid,num_grid)-num_grid/2*grid_cell+x0
grid_y=reform(grid_y,num_grid,num_grid)-num_grid/2*grid_cell+y0
;
rho=sqrt(grid_x^2+grid_y^2)
theta=atan(-grid_x,grid_y)*rad
;
chisq=fltarr(num_grid,num_grid)
calcmodel,/quiet
print,'Starting location rho= ',binary_model[k].rho, $
			', theta=',binary_model[k].theta
print,'Start at location RA = ', $
	binary_model[k].rho*sin(binary_model[k].theta/rad), $
	', Dec = ',binary_model[k].rho*cos(binary_model[k].theta/rad)
print,'Starting reduced Chi^2: ',modelchisq(n_deg_free)
;
completed=1
;
for i=0L,n-1 do begin
	binary_model[k].rho=rho[i]
	binary_model[k].theta=theta[i]
	calcmodel,/quiet
	chisq[i]=modelchisq()
	if ((i*100)/n)/10 eq completed then begin
		print,string(13b),completed*10,'% completed',format='(a,i,a,$)'
		completed=completed+1
	endif
endfor
print,''
;
index=where(chisq eq min(chisq)) & index=index[0]
if theta[index] lt 0 then theta[index]=theta[index]+360
print,'Minimum Chi^2 found at rho= ',rho[index],', theta=',theta[index]
print,'Location of minimim at RA = ',rho[index]*sin(theta[index]/rad), $
			   ', Dec = ',rho[index]*cos(theta[index]/rad)
print,'Minimum red. Chi^2 : ',min(chisq)
; print,'Index = ',index mod num_grid,index/num_grid
binary_model[k].rho=rho[index]
binary_model[k].theta=theta[index]
print,'Model values updated.'
print,'-------------------------------------------------------------------'
calcmodel,/quiet
;
magnitudes=star_model.magnitudes
diameters=star_model.diameter
posangles=star_model.pa
rho_best=rho[index]
theta_best=theta[index]
chisq_best=chisq[index]
if n_elements(files) eq 0 then files=Date
filebase=basename(files[i_file]) & filebase=filebase[0]
num_base=strlen(file_search(filebase+'.chisq'))
if num_base[0] eq 0 then begin
	outfile=filebase+'.chisq'
endif else begin
	words=nameparse(systime())
	outfile=filebase+'_'+strjoin(words,'_')+'.chisq'
endelse
; outfile=basename(files(i_file))+'.chisq' & outfile=outfile(0)
save,n_deg_free,chisq,theta,rho, $
	magnitudes,diameters,rho_best,theta_best,chisq_best, $
	filename=outfile
print,"To show the Chi^2 surface, run: gridplot,'"+outfile+"',/rd"
;
binary_model.method=method_input
;
ENDFOR
;
end
