pro gridpsn_1,chisqfiles,component=component,local=localfile,sigma=sigma,cf=cf
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Original version, no longer in use as of 2023
;
; Given a list of YYYY-MM-DD.chisq files (from gridfit), read these and
; fit the (global) minimum position including an error ellipse and write
; results to file fitchisq.psn. sigma is the desired increase in total chisq,
; and the ellipse is fit to the corresponding contour. The factor cf allows
; to account for correlated data and reduces the number of degrees of freedom.
; Correlations may involve adjacent channels depending on the calibration.
;
; Formal errors correspond to sigma=1 and cf=1 (default).
;
; The gridfit file must have fine sampling, use option gridpsn with gridfit.
;
; An astrometry file (*.psn) can be specified via localfile, which evaluates
; the Chi^2 value nearest the position given for the epoch in the localfile.
; This allows to pick a (local) minimum closest to the local astrometry.
; Specify the component if there is more than one when using localfile.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common MarquardtFit,fit_options
;
if n_elements(chisqfiles) eq 0 then begin
	chisqfiles='????-??-??.chisq'
	chisqfiles='*.chisq'
	print,'Processing all CHISQ files...'
endif
;
chisq_files=file_search(chisqfiles)
if strlen(chisq_files[0]) eq 0 then begin
	print,'GRIDPSN: Files not found!'
	return
endif
;
if n_elements(component) eq 0 then component='A-B'
;
if n_elements(localfile) eq 0 then local=0 else local=1
if local then begin
	local_file=file_search(localfile)
	if strlen(local_file[0]) eq 0 then begin
		print,'Local astrometry file not found!'
		return
	endif
	load_astrometry,local_file[0]
	index=where(positions.component eq component)
	local_positions=positions[index]
	r_local=sin(local_positions.theta)*local_positions.rho
	d_local=cos(local_positions.theta)*local_positions.rho
endif
if n_elements(sigma) eq 0 then sigma=1
if n_elements(cf) eq 0 then cf=1 else cf=float(cf)
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
fit_options.tolerance=1e-7
;
openw,unit,'fitchisq.psn',/get_lun
printf,unit, $
'!Comp Julian Year  Rho   Theta    Major  Minor   PA     Rho_err  Theta_err Chi2'
;
rad=180/!pi
ellipse_options=alloc_ellipse_options()
ellipse_options.component=component
ellipse_options.a=1
ellipse_options.b=1
ellipse_options.p=1
positions=replicate({component:'',rho:0.0,theta:0.0},8)
positions.component=component
;
for i_file=0,n_elements(chisq_files)-1 do begin
	print,''
	print,'Analyzing file ',chisq_files[i_file]
;
	date=strmid(chisq_files[i_file],0,10)
	parsedate,date,y,m,d
	midnight=system_config('NPOI','MIDNIGHT')
	d=d+midnight/24
	jy=jd2jy(julian(y,m,d))
;	Defaults
	emajor=1.0
	eminor=emajor
	pa=0
	rho_error=0
	theta_error=0
;
	restore,chisq_files[i_file]	; restores n_deg_free,chisq,rho,theta
	n_deg_free=nint(n_deg_free/cf)	; cf reduces the number of data points
	min_red_chisq=min(chisq)
	index=where(chisq eq min_red_chisq) & index=index[0]
	ij=whereindex(index,chisq)
	print,'Minimum chi^2 found at rho= ', $
		float(rho[index]),', theta= ',float(theta[index])
	print,'Minimum reduced chi^2: ',min_red_chisq
	print,'Number of degrees of freedom:',n_deg_free
;	print,'Minimum total chi^2: ',min_red_chisq*n_deg_free
	print,'Normalizing reduced chi^2 to 1'
	chisq=chisq/min_red_chisq
;
	r=sin(theta/rad)*rho
	d=cos(theta/rad)*rho
	cellsize=float(r[0]-r[1])
	rr=r[*,0]
	dd=d[0,*] & dd=reform(dd)
	csr=median(rr-shift(rr,-1))
	csd=median(dd-shift(dd,1))
	if nint(csr*1e6) ne nint(csd*1e6) then begin
		print,'Error: different cellsize in x and y directions!'
		return
	endif
	cell_size=csr	; mas
;
	if local then begin
;		Find the grid cell closest to the local astrometry
		i=where(abs(local_positions.jy-jy) lt 1.0/(365.25*24),count)
		i=i[0]
		if count eq 0 then begin
			print,'Julian Year not found!'
			continue
		endif
		dr=r_local[i]-r
		dd=d_local[i]-d
		dist_local=sqrt(dr^2+dd^2)
		j=where(dist_local eq min(dist_local)) & j=j[0]
		min_dist=dist_local[index]
		print,'Local chi^2 found at rho= ',rho[j],', theta= ',theta[j]
		print,'Local chi^2: ',chisq[j],', min(dist)=',min_dist
		print,'============================================'
		ij=whereindex(j,chisq)
	endif
;
; Attempt to fit ellipses to contours, map size should be less than 2 mas
	dims=size(chisq,/dim)
;
	chisq=chisq*n_deg_free	; Convert to total chisq,
;				  formal error now corresponds to increase by 1
	print,'Converted to total chi^2:'
       	print,'     Maximum='+strtrim(string(max(chisq)),1)
	print,'     Minimum='+strtrim(string(min(chisq)),1)
	min_chisq=min(chisq)
	c0=1			; Default sigma, used w/polynomial fit
	max_chisq=min_chisq+c0	; Correct, though test shows chi^2(red)>3
	max_chisq=max(chisq)	; Start with highest value, checking edges
;
;	Closed contour check
	if min(chisq[0,*]) le max_chisq then max_chisq=min(chisq[0,*])
	if min(chisq[*,0]) le max_chisq then max_chisq=min(chisq[*,0])
	print,"Maximum chi^2 of closed cont's.: ",max_chisq
	if (max_chisq-min_chisq) ge sigma then begin
		max_chisq=min_chisq+sigma
		print,'Maximum chi^2 contour to fit (min + sigma): ',max_chisq
	endif else begin
		print,'***Error: requested contour too high:',sigma
		max_chisq=min_chisq+(max_chisq-min_chisq)*0.25	; Use 25%
		print,'Maximum chi^2 contour to fit (min/max=25%): ',max_chisq
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files[i_file]+' (contour high)' else $
		bad_files=[bad_files,chisq_files[i_file]+' (contour high)']
		print,''
		continue
	endelse
;
	r_ell=fltarr(8)
	d_ell=fltarr(8)
	z_r_ell=fltarr(8)
	z_d_ell=fltarr(8)
; 	Walk from minimum along the axes and diagonals
	mflag=0	; set to 1 if walk hits edge of map
	qflag=0	; set to 1 if quasol fails
;	Cardinal axes
	for i=1,dims[0]/2 do begin	; "left"
		curr_chisq=chisq[(ij[0]-i)>0,ij[1]]
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij[0] then mflag=1
	r_ell[0]=r[(ij[0]-i)>0,ij[1]]-r[ij[0],ij[1]]
	d_ell[0]=d[(ij[0]-i)>0,ij[1]]-d[ij[0],ij[1]]
	for i=1,dims[0]/2 do begin	; "right"
		curr_chisq=chisq[(ij[0]+i)<dims[0]-1,ij[1]]
		if curr_chisq ge max_chisq then break
	endfor
	if ij[0]+i ge dims[0] then mflag=1
	r_ell[1]=r[(ij[0]+i)<dims[0]-1,ij[1]]-r[ij[0],ij[1]]
	d_ell[1]=d[(ij[0]+i)<dims[0]-1,ij[1]]-d[ij[0],ij[1]]
;	Fit quadratic polynomial to minimum...
	x=r[*,0]-r[index]
	y=chisq[*,ij[1]] & y0=min(y) & y=y-y0
	j=where(y le max_chisq,nj)
	if nj lt 8 then begin
		print,'Warning: sampling of minimum too coarse!'
		print,'Rerun gridfit with smaller grid cell size!'
		print,'Current grid cell size is ',cellsize
		return
	endif
	x=x[j] & y=y[j]
	c=poly_fit(x,y,2,yfit)
	c[0]=-c0 & c[1]=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z[si]
	if total(z) eq 0 then qflag=1
	z_r_ell[0]=z[1]
	z_r_ell[1]=z[0]
	z_d_ell[0:1]=0
	for j=1,dims[1]/2 do begin	; "down"
		curr_chisq=chisq[ij[0],(ij[1]-j)>0]
		if curr_chisq ge max_chisq then break
	endfor
	if j gt ij[1] then mflag=1
	r_ell[2]=r[ij[0],(ij[1]-j)>0]-r[ij[0],ij[1]]
	d_ell[2]=d[ij[0],(ij[1]-j)>0]-d[ij[0],ij[1]]
	for j=1,dims[1]/2 do begin	; "up"
		curr_chisq=chisq[ij[0],(ij[1]+j)<dims[1]-1]
		if curr_chisq ge max_chisq then break
	endfor
	if ij[1]+j ge dims[1] then mflag=1
	r_ell[3]=r[ij[0],(ij[1]+j)<dims[1]-1]-r[ij[0],ij[1]]
	d_ell[3]=d[ij[0],(ij[1]+j)<dims[1]-1]-d[ij[0],ij[1]]
;	Fit quadratic polynomial to minimum...
	x=reform(d[0,*])-d[index]
	y=reform(chisq[ij[0],*]) & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	if n_elements(j) lt 8 then begin
		print,'Error: sampling of minimum too coarse!'
		print,'Rerun gridfit with smaller grid cell size!'
		print,'Current grid cell size is ',cellsize
		return
	endif
	x=x[j] & y=y[j]
	c=poly_fit(x,y,2,yfit)
	c[0]=-c0 & c[1]=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z[si]
	if total(z) eq 0 then qflag=1
	z_d_ell[2]=z[0]
	z_d_ell[3]=z[1]
	z_r_ell[2:3]=0
;	Diagonals
	for i=1,dims[0]/2 do begin	; "lower-left"
		j=i
		curr_chisq=chisq[(ij[0]-i)>0,(ij[1]-j)>0]
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij[0] then mflag=1
	if j gt ij[1] then mflag=1
	r_ell[4]=r[(ij[0]-i)>0,(ij[1]-j)>0]-r[ij[0],ij[1]]
	d_ell[4]=d[(ij[0]-i)>0,(ij[1]-j)>0]-d[ij[0],ij[1]]
	for i=1,dims[0]/2 do begin	; "upper-right"
		j=i
		curr_chisq=chisq[(ij[0]+i)<dims[0]-1,(ij[1]+j)<dims[1]-1]
		if curr_chisq ge max_chisq then break
	endfor
	if ij[0]+i ge dims[0] then mflag=1
	if ij[1]+j ge dims[1] then mflag=1
	r_ell[5]=r[(ij[0]+i)<dims[1]-1,(ij[1]+j)<dims[1]-1]-r[ij[0],ij[1]]
	d_ell[5]=d[(ij[0]+i)<dims[1]-1,(ij[1]+j)<dims[1]-1]-d[ij[0],ij[1]]
;	Fit quadratic polynomial to minimum...
	idx=indgen(dims[0])-dims[0]/2
	idx0=ij[0]+idx
	idx1=ij[1]+idx
	i=where(idx0 ge 0 and idx1 ge 0 and $
		idx0 lt dims[0] and idx1 lt dims[1],j)
	idx0=idx0[i]
	idx1=idx1[i]
	x=sqrt((r[idx0,idx1]-r[index])^2 $
	      +(d[idx0,idx1]-d[index])^2)*signof(idx[i])
	y=chisq[idx0,idx1] & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	x=x[j] & y=y[j]
	c=poly_fit(x,y,2,yfit)
	c[0]=-c0 & c[1]=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z[si]
	if total(z) eq 0 then qflag=1
	z_d_ell[4]=z[0];/sqrt(2)
	z_d_ell[5]=z[1];/sqrt(2)
	z_r_ell[4]=z[1];/sqrt(2)
	z_r_ell[5]=z[0];/sqrt(2)
	for i=1,dims[1]/2 do begin	; "upper-left"
		j=i
		curr_chisq=chisq[(ij[0]-i)>0,(ij[1]+j)<dims[1]-1]
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij[0] then mflag=1
	if ij[1]+j ge dims[1] then mflag=1
	r_ell[6]=r[(ij[0]-i)>0,(ij[1]+j)<dims[1]-1]-r[ij[0],ij[1]]
	d_ell[6]=d[(ij[0]-i)>0,(ij[1]+j)<dims[1]-1]-d[ij[0],ij[1]]
	for i=1,dims[1]/2 do begin	; "lower-right"
		j=i
		curr_chisq=chisq[(ij[0]+i)<dims[0]-1,(ij[1]-j)>0]
		if curr_chisq ge max_chisq then break
	endfor
	if ij[0]+i ge dims[0] then mflag=1
	if j gt ij[1] then mflag=1
	r_ell[7]=r[(ij[0]+i)<dims[0]-1,(ij[1]-j)>0]-r[ij[0],ij[1]]
	d_ell[7]=d[(ij[0]+i)<dims[0]-1,(ij[1]-j)>0]-d[ij[0],ij[1]]
;	Fit quadratic polynomial to minimum...
	idx=indgen(dims[0])-dims[0]/2
	idx0=ij[0]+idx
	idx1=ij[1]-idx
	i=where(idx0 ge 0 and idx1 ge 0 and $
		idx0 lt dims[0] and idx1 lt dims[1],j)
	idx0=idx0[i]
	idx1=idx1[i]
	x=sqrt((r[idx0,idx1]-r[index])^2 $
	      +(d[idx0,idx1]-d[index])^2)*signof(idx[i])
	y=chisq[idx0,idx1] & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	if n_elements(j) lt 8 then begin
		print,''
		print,'***Number of contour points: ',n_elements(j)
		print,'***Warning: could not define contour! Skipping '+date
		wait,1
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files[i_file]+' (contour poor)' else $
		bad_files=[bad_files,chisq_files[i_file]+' (contour poor)']
		continue
	endif
	x=x[j] & y=y[j]
	c=poly_fit(x,y,2,yfit)
	c[0]=-c0 & c[1]=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z[si]
	if total(abs(z)) eq 0 then qflag=1
	z_d_ell[6]=z[1];/sqrt(2)
	z_d_ell[7]=z[0];/sqrt(2)
	z_r_ell[6]=z[1];/sqrt(2)
	z_r_ell[7]=z[0];/sqrt(2)
;
	if local then begin
		if min_dist gt 1 then mflag=1
	endif
;
	if mflag then begin
		print,''
		print,'***Warning: Contour outside map! Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files[i_file]+' (contour outside)' else $
		bad_files=[bad_files,chisq_files[i_file]+' (contour outside)']
		wait,1
	endif
	if not mflag then begin
;	Fit error ellipse to contour at sigma
	positions.rho=sqrt(r_ell^2+d_ell^2)
	positions.theta=atan(r_ell,d_ell)
	i=where(positions.rho eq max(positions.rho)) & i=i[0]
	pa=positions[i].theta
	emajor=positions[i].rho	; semi-major axis
	i=where(positions.rho eq min(positions.rho)) & i=i[0]
	eminor=positions[i].rho	; semi-minor axis
	e_parms=[0,0,emajor,eminor,pa]
	ellipse_options.c=1	; fit center 1/0
	print,'Running marquardt to fit ellipse to contour at min.+sigma...'
	fitellipse & e_parms0=e_parms
;
;	emajor=e_parms(2)*cf
;	eminor=e_parms(3)*cf
	emajor=e_parms[2]
	eminor=e_parms[3]
	pa=(e_parms[4]*rad) mod 180
	if eminor gt emajor then begin
		v=emajor
		emajor=eminor
		eminor=v
		pa=(pa+90) mod 180
	endif
	if pa lt 0 then pa=pa+180
	rho_error=0
	theta_error=0
;
	printf,unit,'! '+date+', ellipse at sigma followed by formal error ellipse'
	printf,unit,' '+component,jy, $
      		rho[index],theta[index], $
		emajor,eminor,pa, $
		rho_error,theta_error,min(chisq), $
	format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       	'3x,f7.3,1x,f8.3,1x,f6.1)'
;
;	Fit formal error ellipse w/c0=1 and quadratic fit
	if qflag then begin
		print,''
		print,'***Error: sol. of quadratic eq. failed. Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files[i_file]+' (solution failed)' else $
		bad_files=[bad_files,chisq_files[i_file]+' (solution failed)']
		wait,1
	endif else begin
		positions.rho=sqrt(z_r_ell^2+z_d_ell^2)
		positions.theta=atan(z_r_ell,z_d_ell)
		i=where(positions.rho eq max(positions.rho)) & i=i[0]
		pa=positions[i].theta
		pa=e_parms0[4]
		emajor=positions[i].rho	; semi-major axis
		i=where(positions.rho eq min(positions.rho)) & i=i[0]
		eminor=positions[i].rho	; semi-minor axis
		e_parms=[0,0,emajor,eminor,pa]
		ellipse_options.c=0	; fit center 1/0
;		ellipse_options.p=0
		print,''
		print,'Running marquardt to fit formal error ellipse...'
		fitellipse
;
		emajor=e_parms[2]
		eminor=e_parms[3]
		pa=(e_parms[4]*rad) mod 180
		if eminor gt emajor then begin
			v=emajor
			emajor=eminor
			eminor=v
			pa=(pa+90) mod 180
		endif
		if pa lt 0 then pa=pa+180
		rho_error=0
		theta_error=0
;
		printf,unit,'!'+component,jy, $
      		rho[index],theta[index], $
		emajor,eminor,pa, $
		rho_error,theta_error,min(chisq), $
		format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       	'3x,f7.3,1x,f8.3,1x,f6.1)'
	endelse	; qflag
	endif	; mflag
endfor
;
free_lun,unit
spawn,'cat fitchisq.psn'
;
print,'------------------------------'
if n_elements(bad_files) gt 0 then begin
	print,'Failed to analyze these files:'
	for i=0,n_elements(bad_files)-1 do print,bad_files[i]
endif
if local then positions=local_positions
;
end
