function innes,t_parms,prograde=prograde
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Given the orbital period, epoch of periastron, and the 5 fit parameters
; of an apparent ellipse fit, compute the 7 orbital elements.
; The analysis follows closely R.M. Green, Spherical Astronomy, pp.470
;
; Center x =        t_parms(0)
; Center y =        t_parms(1)
; Semi-major axis = t_parms(2)
; Semi-major axis = t_parms(3)
; Position angle =  t_parms(4) [rad]
; Period =          t_parms(5)
; Epoch =           t_parms(6)
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(t_parms) eq 0 then $
	t_parms=[e_parms,binary_model[0].period,binary_model[0].epoch+2440000]
o_parms=t_parms
RAD=180/pi_circle
;
; Check inputs
if n_elements(t_parms) eq 0 then begin
	print,'***Error(INNES): orbit parms undefined!'
	return,o_parms
endif
;
if n_elements(prograde) eq 0 then begin
	jd=positions.jd
	theta=positions.theta
	rho=positions.rho
	s=sort(jd)
	jd=jd[s]
	theta=theta[s]
	rho=rho[s]
;
; 	Determine orbital sense
	dtheta=theta-shift(theta,1)
	index=where(dtheta lt -pi_circle,count)
	if count gt 0 then dtheta[index]=dtheta[index]+2*pi_circle
	index=where(dtheta ge 0,count1)
	index=where(dtheta lt 0,count2)
	if count1 gt count2 then begin
		print,'Orbit is PROGRADE.'
		prograde=1
	endif else begin
		print,'Orbit is RETROGRADE.'
		prograde=0
	endelse
endif
;
; Determine eccentricity
ecc=sqrt(t_parms[2]^2-t_parms[3]^2)/t_parms[2]
alpha=atan(-t_parms[0],-t_parms[1])
beeta=t_parms[4]-alpha
cp=sqrt(t_parms[3]^2/(1-(ecc*cos(beeta))^2))
ca=sqrt(t_parms[0]^2+t_parms[1]^2)
o_parms[1]=ca/cp
;
; Determine coordinates (x1,y1) of periastron P
ap=cp-ca
x1=ap*sin(alpha)
y1=ap*cos(alpha)
;
; Compute eccentric anomaly for the point on the orbit where T_anom=90
E_anom=acos(o_parms[1])
;
; Compute time for this point
t=o_parms[6]+o_parms[5]/(2*pi_circle)*(E_anom-o_parms[1]*sqrt(1-o_parms[1]^2))
;
; Use equal area law to find coordinates (x2,y2) of point R
e_area=t_parms[2]*t_parms[3]*pi_circle
r_area=(t-o_parms[6])/o_parms[5]*e_area
x2p=x1
y2p=y1
dalpha=2*pi_circle/3600
if not prograde then dalpha=-dalpha
alpha2=alpha
a=0
repeat begin
	alpha2=alpha2+dalpha
	r=ellipse(t_parms[2],t_parms[3],alpha2-t_parms[4])
	x2=r*sin(alpha2)+t_parms[0]
	y2=r*cos(alpha2)+t_parms[1]
	da=sqrt(total(crossp([x2,y2,0.0],[x2p,y2p,0.0])^2))/2
	a=a+da
	x2p=x2
	y2p=y2
endrep until a gt r_area
;
; Compute Thiele-Innes constants
X1=x1/(1-o_parms[1])
Y1=y1/(1-o_parms[1])
X2=x2/(1-o_parms[1]^2)
Y2=y2/(1-o_parms[1]^2)
;
; Compute argument of periastron
wpw=atan((X1-Y2),(X2+Y1))
wmw=atan((X1+Y2),(X2-Y1))
if (X1-Y2)*sin(wpw) lt 0 then wpw=wpw+pi_circle
if (X1+Y2)*sin(wmw) gt 0 then wmw=wmw+pi_circle
o_parms[3]=((wpw+wmw)/2) mod (2*pi_circle)
;
; Compute argument of ascending node
o_parms[4]=(wpw-wmw)/2
if o_parms[4] lt 0 then o_parms[4]=o_parms[4]+2*pi_circle
;
; If e near 0, set w=0 and re-do epoch
if o_parms[1] lt 0.001 then begin
	o_parms[6]=o_parms[6]-o_parms[3]/2/pi_circle*o_parms[5]
	o_parms[3]=0
endif
;
; Compute inclination
o_parms[2]=atan(sqrt(-(X1+Y2)*sin(wpw)/(X1-Y2)/sin(wmw)))*2
;
; Compute semi-major axis
o_parms[0]=(X1-Y2)/sin(wpw)/(1+cos(o_parms[2]))
;
print,'__________________________________'
print,'Thiele-Innes estimates: '
print,'Semi-major axis = ',o_parms[0]
print,'Eccentricity =    ',o_parms[1]
print,'Inclination =     ',o_parms[2]*RAD
print,'Periastron =      ',o_parms[3]*RAD
print,'Ascending node =  ',o_parms[4]*RAD
print,'Period =          ',o_parms[5]
print,'Epoch =           ',o_parms[6]
print,'_______________***________________'
;
return,o_parms
end
