pro limbdata,restore=do_restore
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Read Van Hamme's monochromatic limb darkening coefficients and fluxes
; and interpolate onto grid from B to N bands (2 nm stepsize).
; Also add NextGen stellar fluxes for cooler stars. Save all data to
; XDR file. NextGen data from https://phoenix.ens-lyon.fr/Grids/
;
; If called with restore=1, just re-read the limbdata.xdr file
;
common LimbBase,limb_data
;
limbfile=!atmospheres_dir+'vanhamme/limbdata.xdr'
f=file_search(limbfile) & f=f[0]
if n_elements(do_restore) eq 0 then do_restore=0
if do_restore and strlen(f) ne 0 then begin
	restore,limbfile
	print,'Restored '+limbfile+'.'
	print,'Min. Teff = ',min(limb_data.t),', max. Teff = ',max(limb_data.t)
	m=n_elements(limb_data.m)
	min_m=min(limb_data.m)
	u=fltarr(m)
	t=limb_data.t
	for i=0,m-1 do begin
		index=where(limb_data.n eq (i+min_m))
		u[i]=mean(limb_data.u[index])
	endfor
	index=where(u ne 0,count)
	if count gt 0 then begin
		print,'Limb-darkening coefficients non-zero for ' $
		+string(nint(min(t[index])),format='(i5)')+' < T < ' $
		+string(nint(max(t[index]),/long),format='(i5)')
	endif else begin
		print,'Warning: all limb-darkening coefficients zero!'
	endelse
	print,'See '+!atmospheres_dir+'vanhamme/Notes.txt for more information.'
	return
endif else print,'File limbdata.xdr not found, recreate instead...'
;
print,'Reading VanHamme data...'
modelfile=!atmospheres_dir+'vanhamme/table3a'
status=dc_read_free(modelfile,m,t,g,/col)
m=fix(m)	; m = Model number for t = Teff, g = log(g)
;
datafile=!atmospheres_dir+'vanhamme/table3c.xdr'
f=file_search(datafile) & f=f[0]
if strlen(f) ne 0 then restore,datafile else begin
	datafile=!atmospheres_dir+'vanhamme/table3c'
	status=dc_read_free(datafile,n,w,u,qlin,x,y,qlog,a,b,qsqr,f,/col)
	save,n,w,u,qlin,x,y,qlog,a,b,qsqr,f,filename=datafile+'.xdr'
endelse
n=fix(n)
; n = model number, w = wavelength, f = flux, u = lin. coeff, x,y = log coeff.
;
; Remove all models with Teff < teff_limit
; (only 3500, 4000, 8000, 10000 K allowed)
teff_limit=10000
if getenv('USER') eq !owner then begin
	index=where(t lt teff_limit,count)
	for i=0,count-1 do begin
		nndex=where(n eq m[index[i]])
		n[nndex]=0	; The lowest valid model number is 1
	endfor
	if count gt 0 then begin
		m[index]=0
		index=where(m ne 0)
		m=m[index]
		t=t[index]
		g=g[index]
	endif
endif
;
; Use 300 nm < lambda < 20 mu for interpolation
i=where(w ge 300 and w le 20000 and n ne 0)
n0=n[i] & w0=w[i] & f0=f[i] & u0=u[i] & x0=x[i] & y0=y[i] & a0=a[i] & b0=b[i]
nw=n_elements(unique(w0))
nm=n_elements(unique(n0))
if n_elements(w0)/nw ne nm then begin
	print,'File corrupt: '+datafile
	return
endif
l0=unique(w0)
; Prepare grid from B to N bands
l=findgen(((14000-min(w0))/2)+1)*2+min(w0)
nl=n_elements(l)
n=intarr(n_elements(l)*nm)
w=fltarr(n_elements(l)*nm)
f=fltarr(n_elements(l)*nm)
u=fltarr(n_elements(l)*nm)
x=fltarr(n_elements(l)*nm)
y=fltarr(n_elements(l)*nm)
a=fltarr(n_elements(l)*nm)
b=fltarr(n_elements(l)*nm)
; Interpolate onto grid variables
for i=0,nm-1 do begin
	i0=i*nw
	i1=(i+1)*nw-1
	j0=i*nl
	j1=(i+1)*nl-1
	w[j0:j1]=l
;	n(j0:j1)=i+1 bug!
	n[j0:j1]=n0[i0]
	f[j0:j1]=interpol(f0[i0:i1],l0,l)
	u[j0:j1]=interpol(u0[i0:i1],l0,l)
	x[j0:j1]=interpol(x0[i0:i1],l0,l)
	y[j0:j1]=interpol(y0[i0:i1],l0,l)
	a[j0:j1]=interpol(a0[i0:i1],l0,l)
	b[j0:j1]=interpol(b0[i0:i1],l0,l)
endfor
;
; Replace all models for Teff < teff_limit with NextGen models
if getenv('USER') eq !owner then begin
	print,'Reading NextGen data...'
;	Use only NextGen models cooler than teff_limit
	if teff_limit eq 3500 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-3][0-4]*.xdr')
	if teff_limit eq 4000 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-3][0-9]*.xdr')
	if teff_limit eq 8000 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-7][0-9]*.xdr')
	if teff_limit eq 10000 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-9][0-9]*.xdr')
	models=specname(modelfiles)
	k=n_elements(modelfiles)
	m1=max(m)+1+indgen(k)		; assign model numbers
;	Get Teff and Log(g) from the filename!
	t1=float(strmid(models,3,2))*100
	g1=float(strmid(models,6,3))
	m=[m,m1]
	t=[t,t1]
	g=[g,g1]
	restore,modelfiles[0]		; restores l,s
	nw=n_elements(l)
	n1=intarr(k*nw)
	w1=fltarr(k*nw)
	f1=fltarr(k*nw)
	u1=fltarr(k*nw)
	x1=fltarr(k*nw)
	y1=fltarr(k*nw)
	a1=fltarr(k*nw)
	b1=fltarr(k*nw)
	for i=0,k-1 do begin
		restore,modelfiles[i]	; restores l,s
		n1[i*nw:(i+1)*nw-1]=m1[i]
		w1[i*nw:(i+1)*nw-1]=l
;		Apply scale factor (nextgen_scale)
		f1[i*nw:(i+1)*nw-1]=s*3.61267e-08
	endfor
	n=[n,n1]
	w=[w,w1]
	f=[f,f1]
	u=[u,u1]
	x=[x,x1]
	y=[y,y1]
	a=[a,a1]
	b=[b,b1]
endif
;
limb_data=alloc_limb_data(m,t,g,n,w,f,u,x,y,a,b)
;
; Save the XDR file
save,limb_data,filename=limbfile
;
end
