pro load_interferometry,files_in
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Procedure to load multiple .cha/.xdr/.*fits scan averaged data files and store
; each in memory using storenight. Expand GenConfig and GeoParms into arrays of
; structures, geninfo and geoinfo, respectively, to have some configuration data
; readily available for all stored data.
;
; Modified version for reading pndrs *calibrated* data files.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,StarTable,Notes
common LocalAmoebaBuffer,BUFFERSIZE,BUFFERLIMIT
common ModelFit,parameters,ds_options
;
common LocalChaSel,chafiles
common LocalFitSel,oiffiles
;
; Next 4 lines commented 2016-02-24
; Clear buffer
; freememory
; GenInfo=replicate(GenConfig,1)
; GeoInfo=replicate(GeoParms,1)
;
files=file_search(strtrim(files_in,2))
if strlen(files[0]) eq 0 then begin
	print,'***Error(LOAD_INTERFEROMETRY): files not found!'
	return
endif
num_files=n_elements(files)
max_scans=num_files*10	; total number of observations
ds_options.i=0
; Next two variables for associating stars and calibrators (VLTI only)
stars=strarr(num_files)
bflag=strarr(num_files)+'.'
;
; Combine if all files are FITS for same night, array, and configuration
temp_do_not_keep=0
f_ext=unique(extname(files))
if num_files gt 1 and n_elements(f_ext) eq 1 and f_ext[0] EQ 'fits' then begin
	print,'Please wait, reading OIFITS files...'
	date_obs=strarr(num_files)
	arr_config=strarr(num_files)
	spec_config=strarr(num_files)
;	Environmental data: T, pressure, %hum., seeing, coherence time, PWV
;	Read here (one per file!) and pass to combined file in next step.
;	Note that a single file may however contain more than one observation!
	emjd=dblarr(num_files)
	temp=fltarr(num_files)
	pres=fltarr(num_files)
	rhum=fltarr(num_files)
	s_start=fltarr(num_files)	; seeing
	s_end=fltarr(num_files)
	t_start=fltarr(num_files)	; T0
	t_end=fltarr(num_files)
	p_start=fltarr(num_files)	; PVW
	p_end=fltarr(num_files)
;
	for i=0,num_files-1 do begin
		d=mrdfits(files[i],0,h,/silent)
		if i eq 0 then first_header=h	; Header for the merged file
		obsname=fitshparse(h,'OBS.NAME',/quiet)	; OB name
		dateobs=fitshparse(h,'DATE-OBS',/quiet)
;		Ignore these files (e.g., PIONIER calibrations)
		if strpos(obsname,'SpectralCalib') ge 0 then files[i]=''
		if strpos(obsname,'Calibrations') ge 0 then files[i]=''
		if strlen(files[i]) eq 0 then continue
;		Extract target name
		stars[i]=strjoin( $
			nameparse(fitshparse(h,'OBJECT',/quiet),'_'), $
			'')
;		VLTI: if CAL, OBJECT is "STD"
		if strpos(stars[i],'STD') eq 0 then $ ;stars(i)=obsname
		stars[i]=strjoin( $
			nameparse(fitshparse(h,'OBS.TARG.NAME',/quiet),'_'), $
			'')
;		VLTI: is this a calibrator observation?
		if strpos(obsname,'CAL_') eq 0 then begin
			bflag[i]='C'
		endif
;		Extract valid OYSTER star ID
		stars[i]=cri_vlti(fitshparse(h,'OBS.TARG.NAME',/quiet))
;		Read header information
		temp[i]=fitshparse(h,'ISS.AMBI.TEMP',/quiet)
		pres[i]=fitshparse(h,'ISS.AMBI.PRES',/quiet)
		rhum[i]=fitshparse(h,'ISS.AMBI.RHUM',/quiet)
		s_start[i]=float(fitshparse(h,'ISS.AMBI.FWHM.START',/quiet))
		s_end[i]=float(fitshparse(h,'ISS.AMBI.FWHM.END',/quiet))
		t_start[i]=float(fitshparse(h,'ISS.AMBI.TAU0.START',/quiet))
		t_end[i]=float(fitshparse(h,'ISS.AMBI.TAU0.END',/quiet))
		p_start[i]=float(fitshparse(h,'ISS.AMBI.IWV.START',/quiet))
		p_end[i]=float(fitshparse(h,'ISS.AMBI.IWV.END',/quiet))
		fits_info,files[i],n_ext=n_ext,extname=ext,/silent
		index=where(strpos(ext,'OI_VIS2') ge 0,count)
		d=mrdfits(files[i],index[0],h,/silent)
		emjd[i]=mean(d.mjd)
;
;		Get info on array stations (e.g., A0)/telescopes (e.g., UT1)
		exten=find_oiextn(files[i],'OI_ARRAY')
		oidata=mrdfits(files[i],exten,h,/silent)
		sta_names=strcompress(oidata.sta_name,/remove_all)
		sta_index=oidata.sta_index
		tel_names=strarr(max(sta_index)+1)
		tel_names[sta_index]=strcompress(oidata.tel_name,/remove_all)
;		Get info on UT of local midnight
		systemid=strtrim(fitshparse(h,'ARRNAME',/quiet))
		night_start=24-system_config(systemid[0],'MIDNIGHT')
;		Read file header for time of observation
		d=mrdfits(files[i],0,h,/silent)
		dateobs=fitshparse(h,'DATE-OBS',/quiet)
		if strlen(dateobs) eq 0 then begin
			fits_info,files[i],n_ext=n_ext,extname=ext,/silent
			index=where(strpos(ext,'OI_VIS2') ge 0,count)
			d=mrdfits(files[i],index[0],h,/silent)
			date_obs[i]=jd2date(d[0].mjd,/mjd)
		endif else date_obs[i]=strmid(dateobs,0,10)
		time_start=0
		if strlen(dateobs) gt 10 then $
			time_start=hms2h(strmid(dateobs,11,8))
;		Check on date changes if observations started before UT=0
		if time_start gt night_start then begin
			!date_change=1
			date_obs[i]=nextdate(date_obs[i])
		endif else !date_change=0
;		Extract info on configurations (stations)
		exten=find_oiextn(files[i],'OI_VIS2')
		oidata=mrdfits(files[i],exten,h,/silent)
		arr_config[i]=strjoin( $
			unique(strjoin(tel_names[oidata.sta_index],'-')),'_')
;		Extract info on spectral setups (spectrometers)
		exten=find_oiextn(files[i],'OI_WAVELENGTH')
		oidata=mrdfits(files[i],exten,h,/silent)
		spec_config[i]=strjoin(string(oidata.eff_wave),' ')
	endfor
;
	index=where(strlen(files) gt 0,num_files)
	if num_files eq 0 then begin
		print,'ERROR: no valid observations selected!'
		files_in=''
		return
	endif
	date_obs=date_obs[index]
	arr_config=arr_config[index]
	spec_config=spec_config[index]
	emjd=emjd[index]
	temp=temp[index]
	rhum=rhum[index]
	pres=pres[index]
	s_start=s_start[index]
	s_end=s_end[index]
	t_start=t_start[index]
	t_end=t_end[index]
	p_start=p_start[index]
	p_end=p_end[index]
	files=files[index]
	stars=stars[index]
	bflag=bflag[index]
	if n_elements(unique(date_obs)) eq 1 and $
	   n_elements(unique(arr_config)) eq 1 and $
	   n_elements(unique(spec_config)) eq 1 then temp_do_not_keep=1
endif
;
IF temp_do_not_keep THEN BEGIN
;
;	Create and read combined OIFITS file
;
	chafiles=''
	oiffiles=files
;
;	Open temporary file to combine the matching FITS files
;	Number of columns increases (added by OYSTER) in OI_TARGET extensions
;	MNTSTA column is removed from OI_ARRAY table
;	No changes in OI_WAVELENGTH
;
	outfile='temp_do_not_keep.fits'
	print,'Please wait, merging OIFITS files...'
;	Sort files by time
	files=files[sort(emjd)]
	merge_oidata,outfile=outfile,infiles=files
;	Add header from first file
	modfits,outfile,0,first_header
	get_oifits,outfile,date_obs=date_obs[0]
	spawn,'rm -f '+outfile
	sttbl=stationtable
;
;	Sort environmental data
	si=sort(emjd)
	etim=(emjd[si]-date2jd(date_obs,/mjd))*86400
	temp=temp[si]
	rhum=rhum[si]
	pres=pres[si]
	fwhm=(s_start[si]+s_end[si])/2
	tau0=(t_start[si]+t_end[si])/2
	pwvc=(p_start[si]+p_end[si])/2
;
	n=n_elements(scans)
	if n_elements(temp) eq n then begin
;		Copy environmental data
		scans.r0=(s_start+s_end)/2
		for i=0,n-1 do scans[i].t0=(t_start[i]+t_end[i])/2
		scans.pwv=(p_start+p_end)/2
		scans.temp=temp
		scans.rhum=rhum
		scans.pres=pres
	endif else begin
;		Interpolate if some files had more then one observation
		if n_elements(unique([scans.time,etim])) eq n then begin
		index=where(scans.time lt etim[0],count)
		if count gt 0 then begin
			etim=[scans[index].time,etim]
			fwhm=[scans[index].r0,fwhm]
			tau0=[scans[index].t0(0,0),tau0] ; [NOB,MC,MB]
			pwvc=[scans[index].pwv,pwvc]
			temp=[scans[index].temp,temp]
			rhum=[scans[index].rhum,rhum]
			pres=[scans[index].pres,pres]
		endif
		index=where(scans.time gt etim[0],count)
		if count gt 0 then begin
			etim=[etim,scans[index].time]
			fwhm=[etim,scans[index].r0]
			tau0=[etim,scans[index].t0(0,0)] ; [NOB,MC,MB]
			pwvc=[etim,scans[index].pwv]
			temp=[etim,scans[index].temp]
			rhum=[etim,scans[index].rhum]
			pres=[etim,scans[index].pres]
		endif
		scans.r0=fwhm
		scans.t0=tau0
		scans.pwv=pwvc
		scans.temp=temp
		scans.rhum=rhum
		scans.pres=pres
		endif
	endelse
;
;	Calibrator info from SCI_ and CAL_ files if produced by pndrsReduce
;	(headers include VLTI header with OB names, "CAL" for VLTI calibrator)
	for i=0,n_elements(startable)-1 do begin
		j=where(stars eq startable[i].name,count)
		if count eq 0 then $
		j=where(stars eq startable[i].starid,count)
		if count ge 1 then startable[i].bflag=bflag[j[0]]
	endfor
;
;	Calibrator info from SCI_ and CAL_ files if produced by pndrsCalibrate
;	(headers do not include the VLTI file header, must parse name instead)
	f_pndrs_cal=file_search('????-??-??_CAL_*_oidataCalibrated.fits')
	n_cal=n_elements(f_pndrs_cal)
	for i=0,n_cal-1 do begin
		words=nameparse(f_pndrs_cal[i],'_')
		k=n_elements(words)-2
		j=where(startable.starid eq words[k],count)
		if count eq 1 then startable[j].bflag='C'
	endfor
;
;	Special feature: if reading files in PNDRS tree, read oiDiam file
	dir=pwd()
	words=nameparse(dir,'/')
	index=where(strpos(words,'_abcd') gt 0,count)
	civil_date=strmid(words[index[0]],0,10)
	oidiam_file=strjoin(words[0:index[0]],'/')+'/'+civil_date+'_oiDiam.fits'
	if count eq 1 and file_test(oidiam_file) then begin
		d=mrdfits(oidiam_file,1,h)
		for i=0,n_elements(startable)-1 do begin
			j=where(strtrim(d.target) eq startable[i].name,count)
			if count eq 1 then $
			startable[i].diameter=d[j].diam
		endfor
	endif
;
ENDIF ELSE BEGIN ; temp_do_not_keep=0
;
;	Read and store files separately
;
;	Save lists of files to restore after reading each one of them
	chafiles_to_load=''
	index_cha=where(extname(files) eq 'cha',num_cha)
	if num_cha gt 0 then chafiles_to_load=files[index_cha]
	oiffiles_to_load=''
	index_oif=where(extname(files) eq 'fits',num_oif)
	if num_oif gt 0 then oiffiles_to_load=files[index_oif]
;
; if num_files gt 30 then BUFFERLIMIT=0 else BUFFERLIMIT=200000000L
;
	files=strcompress(files,/remove_all)
	first=1
	j=0
;
	hds_close
;
	for i=0,num_files-1 do begin
	result=file_search(files[i],count=fcount)
	if fcount ne 0 then begin
		if strpos(files[i],'xdr') ge 0 then begin
			get_xdr,files[i],freememory=first
		endif else if strpos(files[i],'fits') ge 0 then begin
			get_oifits,files[i],freememory=first
			print,unique(genconfig.baselineid)
		endif else begin
			get_scandata,files[i],freememory=first
			hds_close
		endelse
		startable.bflag=bflag[i]
		get_stationtable,update=0 ; Do not update station coordinates!
		if first or num_files eq 1 then begin
			first=0
			table=startable
			sttbl=stationtable
		endif else begin
			table=merge_startable(table,startable)
			sttbl=[sttbl,stationtable]
		endelse
		calcastrom,/skipuv	; Do not re-compute uv-coordinates!
		storenight,file=files[i]
		j=j+1
	endif else print,'***Error(LOAD_INTERFEROMETRY): file not found: ', $
			files[i],'!'
	endfor
	if n_elements(table) eq 0 then return else StarTable=table
;
;	Restore filelists
	if num_cha gt 0 then chafiles=files[index_cha]
	if num_oif gt 0 then oiffiles=files[index_oif]
;
; Get names and diameters for FKV and BSC stars in the startable (no OIFITS!)
	starids=startable.starid
	stars=startable.name
	rename_starids,'fkv-bsc'
	index=where(startable.bsc ne 0,count)
	get_diameter
	t=startable
	read_catalogs
	rename_bsc
	if count gt 0 then stars[index]=startable[index].name
	startable=t
	startable.starid=starids
	startable.name=stars
;
ENDELSE
;
; Compute all estimated visibilities
calcviscal
;
index=uniq(sttbl.stationid,sort(sttbl.stationid))
stationtable=sttbl[index]
;
if num_files eq 1 or temp_do_not_keep then plotinit=init_plot('scan','pt') $
		  		      else plotinit=init_plot('amoeba','pt')
ds_options.i=1
;
; Special section to flag interferometry
if !owner eq 'chummel' and pwd() eq '/science/73leo' then begin
	flag_channels,[11,12,13,14,15,16],/all
	print,'For data in /science73leo, channels 11 to 16 were flagged!'
endif
;
end
