function magfilter,teff,logg,filter,radius,parallax
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Integrate model stellar atmosphere over a named filter.
; Atmosphere parameters teff and logg in K and cgs, respectively.
; Radius in solar units, parallax in mas.
; Subtract Vega zero points.
;
forward_function magfilter
;
; Vega zero point data
vega_filter=['U','B','V','R','I']
vega_colors=[0.03,0.03,0.03,0.07,0.10]		; from Johnson et al. 1966
vega_colors=[0.026,0.026,0.026,0.066,0.096]	; Deane Peterson (version 2)
; Get Vega zero points with model T=9500 K, d=3.24 mas, px=129 mas, (mass=2)
vega_zp=[-22.7879,-22.4989,-21.9033,-21.1903,-20.5634]-vega_colors
; Get Vega zero points with model T=9442 K, d=3.24 mas, px=128.9 mas, (mass=2.4)
vega_zp=[-22.7633,-22.4774,-21.8882,-21.1780,-20.5523]-vega_colors
; Calibration with measured flux in V (Allen's Astrophysical Quantities, p.150)
vega_flux_v=3.44e-8
vega_zp=vega_zp-vega_zp[2]-2.5*alog10(vega_flux_v)
; Adding data for infrared filters
vega_filter=[vega_filter,'J','H','K','L','M','N','Q']
vega_fluxes=[3.31e-9,1.15e-9,4.14e-10,6.59e-11,2.11e-11,9.63e-13,7.18e-14]
vega_zp=[vega_zp,-2.5*alog10(vega_fluxes)]
;
; Here, we choose to derive Vega zero points from the model atmosphere
if n_params() eq 1 then begin
;	This call is for Vega
	filter=teff
	teff=9500.
	logg=4.1
	radius=2.73
	parallax=128.9
endif else begin
	if n_elements(radius) eq 0 then radius=2.73
	if n_elements(parallax) eq 0 then parallax=128.9
endelse
if limbgrid(teff,logg,lammda,limbdu,fluxes) ne 0 then return,-1
;
fluxes=fluxes*(radius/2.73)^2*(parallax/128.9)^2
;
if n_elements(filter) eq 0 then begin
	print,'***Error(MAGFILTER): no filter specified!'
	return,-1
endif
;
case filter of
'U':	begin
	tm=johnson_u(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[0]
	end
'B':	begin
	tm=johnson_b(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[1]
	end
'V':	begin
	tm=johnson_v(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[2]
	end
'R':	begin
	tm=johnson_r(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[3]
	end
'I':	begin
	tm=johnson_i(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[4]
	end
'J':	begin
	tm=johnson_j(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[5]
	end
'H':	begin
	tm=johnson_h(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[6]
	end
'K':	begin
	tm=johnson_k(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[7]
	end
'L':	begin
	tm=johnson_lp(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[8]
	end
'M':	begin
	tm=johnson_m(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[9]
	end
'N':	begin
	tm=johnson_n(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[10]
	end
'Q':	begin
	tm=johnson_q[lammda]
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[11]
	end
'Hp':	begin
	tm=hipparcos_hp(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'u': 	begin
	tm=stroemgren_u(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'v': 	begin
	tm=stroemgren_v(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'b': 	begin
	tm=stroemgren_b(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'y': 	begin
	tm=stroemgren_y(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm800': begin
	tm=mark3_800(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm550': begin
	tm=mark3_550(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm500': begin
	tm=mark3_500(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm450': begin
	tm=mark3_450(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'mhan': begin
	tm=mark3_han(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'mhab': begin
	tm=mark3_hab(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm500o':begin
	tm=mark3_500o(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
else:	begin
	print,'Warning(MAGFILTER): unknown filter!'
	flux=-1
	end
endcase
;
if n_params() eq 1 then vega_zp=0 $
		   else vega_zp=magfilter(filter)
return,-2.5*alog10(flux)-vega_zp
;
; Using Vega zero points from literature
; return,-2.5*alog10(flux)+zp
;
end
