function pickles,starid,template=template
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Return the count rate as a function of wavelength for a given star.
; Spectra are taken from Pickles library for closest match in class and type.
; We use the UVK spectra in dat_uvk which are extended to 25 microns.
;
; A Stellar Spectral Flux Library: 1150–25000 Å
; A.J. Pickles, 1998
; https://iopscience.iop.org/article/10.1086/316197
;
; Keyword template: must not include "uk", e.g., "g2iv" for "ukg2iv"!
; If template is specified, use it. If not available, suggest alternatives.
; If template is not specified, use type and class from StarTable and
; find the nearest template/templates.
;
; Return [lammda [m], flux]
;
forward_function teff_star
;
common StarBase,startable,notes
;
; 	Pickles library (file names, e.g.: a47iv )
	files= $
	["o5v"    ,"o9v"    ,"b0v"    ,"b1v"    ,"b3v", $
	 "b57v"   ,"b8v"    ,"b9v"    ,"a0v"    ,"a2v", $
	 "a3v"    ,"a5v"    ,"a7v"    ,"f0v"    ,"f2v", $
	 "f5v"    ,"f6v"    ,"f8v", $
	 "g0v"    , $
	 "g2v"    ,"g5v"    ,"g8v", $
	 "k0v"    ,"k2v"    ,"k3v"    ,"k4v", $
	 "k5v"    ,"k7v"    ,"m0v"    ,"m1v"    ,"m2v", $
	 "m2.5v"  ,"m3v"    ,"m4v"    ,"m5v"    ,"m6v", $
	 "b2iv"   ,"b6iv"   ,"a0iv"   ,"a47iv"  ,"f02iv", $
	 "f5iv"   ,"f8iv"   ,"g0iv"   ,"g2iv"   ,"g5iv", $
	 "g8iv"   ,"k0iv"   ,"k1iv"   ,"k3iv"   ,"o8iii", $
	 "b12iii" ,"b3iii"  ,"b5iii"  ,"b9iii"  ,"a0iii", $
	 "a3iii"  ,"a5iii"  ,"a7iii"  ,"f0iii"  ,"f2iii", $
	 "f5iii"  ,"g0iii"  ,"g5iii"  , $
	 "g8iii"  ,"k0iii"  , $
	 "k1iii"  ,"k2iii"  , $
	 "k3iii"  ,"k4iii", $
	 "k5iii"  ,"m0iii", $
	 "m1iii"  ,"m2iii"  ,"m3iii"  ,"m4iii"  ,"m5iii", $
	 "m6iii"  ,"m7iii"  ,"m8iii"  ,"m9iii"  , $
	 "b2ii"   ,"b5ii"   ,"f0ii"   ,"f2ii"   ,"g5ii", $
	 "k01ii"  ,"k34ii"  ,"m3ii"   ,"b0i"    ,"b1i", $
	 "b3i"    ,"b5i"    ,"b8i"    ,"a0i"    ,"a2i", $
	 "f0i"    ,"f5i"    ,"f8i"    ,"g0i"    ,"g2i", $
	 "g5i"    ,"g8i"    ,"k2i"    ,"k3i"    ,"k4i",  $
	 "m2i"]
;
; 	Pickles library (spectral types, e.g., a4.7iv)
	spectra= $
	["o5v"    ,"o9v"    ,"b0v"    ,"b1v"    ,"b3v", $
	 "b5.7v"   ,"b8v"    ,"b9v"    ,"a0v"    ,"a2v", $
	 "a3v"    ,"a5v"    ,"a7v"    ,"f0v"    ,"f2v", $
	 "f5v"    ,"f6v"    ,"f8v", $
	 "g0v"    , $
	 "g2v"    ,"g5v"    ,"g8v", $
	 "k0v"    ,"k2v"    ,"k3v"    ,"k4v", $
	 "k5v"    ,"k7v"    ,"m0v"    ,"m1v"    ,"m2v", $
	 "m2.5v"  ,"m3v"    ,"m4v"    ,"m5v"    ,"m6v", $
	 "b2iv"   ,"b6iv"   ,"a0iv"   ,"a4.7iv"  ,"f0.2iv", $
	 "f5iv"   ,"f8iv"   ,"g0iv"   ,"g2iv"   ,"g5iv", $
	 "g8iv"   ,"k0iv"   ,"k1iv"   ,"k3iv"   ,"o8iii", $
	 "b1.2iii" ,"b3iii"  ,"b5iii"  ,"b9iii"  ,"a0iii", $
	 "a3iii"  ,"a5iii"  ,"a7iii"  ,"f0iii"  ,"f2iii", $
	 "f5iii"  ,"g0iii"  ,"g5iii"  , $
	 "g8iii"  ,"k0iii"  , $
	 "k1iii"  ,"k2iii"  , $
	 "k3iii"  ,"k4iii", $
	 "k5iii"  ,"m0iii", $
	 "m1iii"  ,"m2iii"  ,"m3iii"  ,"m4iii"  ,"m5iii", $
	 "m6iii"  ,"m7iii"  ,"m8iii"  ,"m9iii"  , $
	 "b2ii"   ,"b5ii"   ,"f0ii"   ,"f2ii"   ,"g5ii", $
	 "k0.1ii" ,"k3.4ii" ,"m3ii"   ,"b0i"    ,"b1i", $
	 "b3i"    ,"b5i"    ,"b8i"    ,"a0i"    ,"a2i", $
	 "f0i"    ,"f5i"    ,"f8i"    ,"g0i"    ,"g2i", $
	 "g5i"    ,"g8i"    ,"k2i"    ,"k3i"    ,"k4i",  $
	 "m2i"]
;
;	Translate spectral types into grid cells
;	o,b,a,f,g,k,m => 1,2,3,4,5,6,7
;	i,ii,iii,iv,v => 1,2,3,4,5
;
; 	Info: library file name syntax and spectra differ in a few cases:
; 	index=where(files ne spectra)
; 	print,files(index)
; 	b57v a47iv f02iv b12iii k01ii k34ii
; 	print,spectra(index)
; 	b5.7v a4.7iv f0.2iv b1.2iii k0.1ii k3.4ii
;
;	Extract stellar types and classes of the library templates
	spec_parse,spectra,types,classes,t2,c2
	types_bck=types
	classes_bck=classes
;
IF keyword_set(template) THEN  BEGIN
;	Use specified template, if available
	index=where(spectra eq template,count)
	if count eq 1 then begin
		sfile=!atmospheres_dir+'pickles/dat_uvk/uk'+spectra[index[0]]
	endif else begin
		spec_parse,template,type,class
		print,''
		print,'Template not available, consider these alternatives:'
		index=where(types eq type[0],count)
		print,'Same type (line profiles differ): ',spectra[index]
		index=where(classes eq class[0],count)
		print,'Same class (spectral shapes differ):'
		print,spectra[index]
		return,0
	endelse
;
ENDIF ELSE BEGIN
;
;	Find closest template matching type and class of star
	if n_elements(startable) eq 0 then get_startable,starid
; 	Make backup copy of startable
	startable_bck=startable
	st_index=where(startable.starid eq starid,count)
	if count eq 0 then begin
		get_startable,starid
		st_index=0
	endif
	mv=startable[st_index].mv
;
;	Extract stellar type (e.g., B9) and class (e.g., V) of target
	type=fix(startable[st_index].type)
	class=fix(startable[st_index].class)
;	Compute effective temperature of star
	teff_stars
	teff_star=startable[st_index].teff
;
; 	Find closest match for target class
	index=where(classes eq class,count)
	if count eq 0 then begin
		class=' class '+string(class,format='(i2)')+' ('+starid+')!'
		print,'Warning(PICKLES): substituting class V for unknown'+class
			class=5
		index=where(classes eq class)
	endif
; 	Find closest match for target type
	dt=abs(types[index]-type)
	jndex=where(dt eq min(dt))
;	Make index to index into files/spectra
	index=index[jndex]
	print,'Pickles: loading file for class '+strupcase(files[index])
	sfile=!atmospheres_dir+'pickles/dat_uvk/uk'+files[index]
	teff_temp=teff_star[spectra[index]]
	print,'Effective temperature of closest template:' $
		+strcompress(string(teff_temp))
	template='uk'+files[index]
ENDELSE
;
pfile=file_search(sfile+'.xdr') & pfile=pfile[0]
if strlen(pfile) eq 0 then begin
	pfile=sfile+'.dat'
;	print,'Reading file: '+pfile
	s=dc_read_free(pfile,lammda,fluxes,/col,ignore=['#'])
	if getenv('USER') eq !owner then $
		save,lammda,fluxes,filename=sfile+'.xdr'
endif else begin
	restore,pfile
	print,'Restored file: ',pfile
endelse
lammda=lammda*1e-10
nl=n_elements(lammda)
lammda=lammda[0:nl-2]
fluxes=fluxes[0:nl-2]
;
; If not requesting a specific template, make adjustments
IF NOT keyword_set(template) THEN BEGIN
;
; Calculate the shape correction for a mismatch in temperature and apply
teff_temp=teff_star[spectra[index]]
print,'Calculating spectral shape corr. for T_eff = ' $
	+strcompress(string(teff_star))
f_ratio=(exp(1.439d-2/(lammda*abs(teff_temp)))-1) $
       /(exp(1.439d-2/(lammda*abs(teff_star)))-1)
fluxes=fluxes*f_ratio
;
; Normalize V flux to 3.75e-9 erg/cm^2/s/A
fv0=total(fluxes*johnson_v(lammda*1e9))*10^(mv/2.5)/total(johnson_v(lammda*1e9))
factor=0.375/fv0
fluxes=fluxes*factor
;
; Extend to 25 microns to include N and Q bands, using black body
; lammda_nq=(findgen(230)/10+2.5)*1e-6
; fluxes_nq=(3.742d-30/lammda_nq^5)/(exp(1.439d-2/(lammda_nq*abs(teff)))-1)
; index=where(lammda gt 2.45e-6)
; factor_nq=median(fluxes(index))/fluxes_nq(0)
; lammda=[lammda,lammda_nq]
; fluxes=[fluxes,fluxes_nq*factor_nq]
; Obsolete: the UVK spectra already go to 25 microns!
;
ENDIF
;
; Restore original startable, if applicable
if n_elements(startable_bck) ne 0 then startable=startable_bck
;
; This is the energy (ergs) of a photon of wavelength lammda
hnu=6.6262e-27*2.997925e8/lammda
;
; Replace zero's in flux with smallest value in flux
index=where(fluxes ne 0,count)
min_flux=min(fluxes[index])
index=where(fluxes eq 0,count)
if count gt 0 then fluxes[index]=min_flux
;
; Return rate/cm^2/s/nm
; return,(fluxes/hnu)*1e-7
;
; Return spectrum
return,[[lammda],[fluxes]]
;
end
