pro readimage,fitsfile,cellsize=cellsize
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Read a FITS image and convert into effective temperature Pearl image.
; Effective temperature and log(g) maps should be found in the image extensions,
; if not, the procedure will ask them to be defined using XROI.
;
; Cellsize in arcseconds.
;
forward_function mrdfits,pearlimage
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common StarBase,StarTable,Notes
;
cmmap=mrdfits(fitsfile,0,header,/silent)
;
nx=n_elements(cmmap[*,0])
ny=n_elements(cmmap[0,*])
imsze=[nx,ny]
;
index=where(strpos(header,'RA---') ge 0,exists)
if exists then begin
	words=nameparse(header[index[0]+2])
	cdelt1=abs(float(words[2]))*3600000
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt1=cdelt1*180/!pi
endif else cdelt1=25.0
index=where(strpos(header,'DEC--') ge 0,exists)
if exists then begin
	words=nameparse(header[index[0]+2])
	cdelt2=abs(float(words[2]))*3600000
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt2=cdelt2*180/!pi
endif else cdelt2=25.0
if cdelt1 ne cdelt2 then begin
	print,'***Error(READIMAGE): cell size in RA and Dec different!'
	return
endif
if n_elements(cellsize) eq 0 then cellsze=cdelt1 else cellsze=cellsize
;
cm=pearlimage(imsze,cellsze,/relax)
cc=cm
cm.maps[0].map=cmmap
cc.maps[0].map=0
;
; Check wavelength axis
index=where(strpos(header,'WAVE') ge 0,exists)
if not exists then begin
;	print,'***Error(READIMAGE): wavelength axis undefined in header!'
;	read,wavelength,prompt='Please enter min and max wavelengths: '
	wavelength=1.0	; For this type of image wavelength is not used
endif else begin
	words=nameparse(header[index[0]+1])
	crval3=float(words[2])
	words=nameparse(header[index[0]+2])
	cdelt3=float(words[2])
	words=nameparse(header[index[0]+3])
	crpix3=float(words[2])
	wavelength=crval3
endelse
;
; Initialize default model for this type of image
flag=0
if n_elements(gen_model) eq 0 then begin
	flag=1
endif else begin
	if n_elements(gen_model.wavelengths) $
	ne n_elements(wavelength) then flag=1
endelse
if flag then begin
	if n_elements(gen_model) ne 0 then starid=gen_model.starid
	num_wave=1
	gen_model=alloc_gen_model(num_wave)
	gen_error=gen_model
	gen_model=alloc_gen_model(num_wave)
	gen_error=gen_model
	if n_elements(startable) eq 1 then gen_model.starid=startable.starid $
				      else gen_model.starid='CCCNNNN'
	if n_elements(starid) ne 0 then gen_model.starid=starid
	gen_model.wavelengths=wavelength	; model file wavelengths [mu]
	num_spot=2
	star_model=alloc_star_struct(num_wave,num_spot)
	star_error=star_model
	star_model[0].component='A'
	star_model[0].type=13
	binary_struct=alloc_binary_struct()
	binary_model=replicate(binary_struct,1)
endif
;
; Check if map is AIPS/MX ICL map
index=where(strpos(header,"IMCLASS='ICL") ge 0,exists)
if exists then begin
	rad=180/!pi
	mas2rad=rad*3600000.0
	cclist=mrdfits(fitsfile,1,header,/silent)
	cm.maps[0].map=0
	for i=0,n_elements(cclist)-1 do begin
		j=where(abs(cm.x-cclist[i].deltax/rad) lt cellsze/mas2rad $
		    and abs(cm.y-cclist[i].deltay/rad) lt cellsze/mas2rad,count)
		if count eq 0 then begin
			print,'***Error (READIMAGE): could not assign CC!'
			return
		endif
		cm.maps[0].map(j)=cm.maps[0].map(j)+cclist[i].flux
	endfor
endif else begin
; 	Read FITS extensions for eff. temp. and log(g) region maps
	etmap=mrdfits(fitsfile,1,header,/silent)
	lgmap=mrdfits(fitsfile,2,header,/silent)
endelse
;
if n_elements(etmap) le 1 then begin
	print,'------------------------------------------------------------------'
	print,'This type of image requires the definition of Teff/log(g) regions.'
	print,'Please use XROI to define regions of interest, close (X) when done.'
	print,'The largest regions will be processed first, smaller ones override.'
	print,'------------------------------------------------------------------'
	scale_factor=4
	xroi,bytscl(rebin(cm.maps[0].map,imsze*scale_factor),top=200), $
		roi_geometry=roigeo,regions_out=regout,/block
	if n_elements(roigeo) eq 0 then return
;
	print,'Now enter Teff/log(g) for each region in order of decreasing size.'
	si=reverse(sort(roigeo.area))
	roigeo=roigeo[si]
	regout=regout[si]
	etmap=fltarr(nx,ny)
	lgmap=fltarr(nx,ny)
	nr=n_elements(regout)
	for ireg=0,nr-1 do begin
		read,teff,logg,prompt='Please enter Teff,log(g) for region '+string(ireg,forma
		regout[ireg]->scale,fltarr(2)+1./float(scale_factor)
; The following command is not compiled by GDL
;		mask=regout(ireg)->computemask(dimension=[nx,ny])
		etmap[where(mask eq 255)]=teff
		lgmap[where(mask eq 255)]=logg
;		Make sure there are no pixels without initialization
		if ireg eq 0 then begin
			etmap[*]=teff
			lgmap[*]=logg
		endif
	endfor
	ext_header=["XTENSION= 'IMAGE' / IMAGE extension", $
		    "BITPIX  = -32 /  Number of bits per data pixel", $
		    "NAXIS   = 2 / Number of data axes", $
		    "NAXIS1  = "+string(nx)+" /", $
		    "NAXIS2  = "+string(ny)+" /", $
		    "PCOUNT  = 0 / No Group Parameters", $
		    "GCOUNT  = 1 / One Data Group", $
		    "EXTNAME = 'Teff' /", $
		    "END     "]
	writefits,fitsfile,etmap,ext_header,/append
	ext_header[where(strpos(ext_header,'EXTNAME') ge 0)]="EXTNAME = 'log(g)' /"
	writefits,fitsfile,lgmap,ext_header,/append
;
endif
;
; Here we have a valid eff. temp. map, now we use it to initialize the rest
etmap=fix(etmap)
index=where(etmap gt 0)
etlg=string(etmap[index],format='(i5)')+' '+string(lgmap[index],format='(f3.1)')
uetlg=unique(etlg)
nuetlg=n_elements(uetlg)
rt=fltarr(nuetlg)
rg=rt
for i=0,nuetlg-1 do begin
	words=nameparse(uetlg[i])
	rt[i]=float(words[0])
	rg[i]=float(words[1])
endfor
cm.teff=etmap
cm.logg=lgmap
cc.teff=etmap
cc.logg=lgmap
;
; Read convolving beam, if any
; a=0.
; read,a,b,c,prompt='Please enter major, minor axis [mas] and PA of restoring be
; if a ne 0 then cb=cleankernel(cm,a) else cb=0
;
end
