pro readimages,fitsfile,cellsize=cellsize,wavelengths=wavelengths,quiet=quiet
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Read a FITS file containing an image cube to be used for modeling.
; Input: cellsize [mas], wavelengths [microns]. Wavelengths should be
; specified if they are not in the header of the FITS file (WAVE keyword).
; Instead of a cube, the images can be specified using wildcards,
; and in that case they will be assembled into a cube. Here one also has the
; option to load a model file first, and the wavelengths specified there
; take the place of the wavelengths required by this routine.
;
; Recognizes BSMEM images by PIXELATION keyword, reverses RA.
;
; The images and related information are stored in common ChannelImage.
; Function chmap (model.pro) is used to compute image at given wavelength.
;
; Wavelength [m]
;
; A few words on passing keyword parameters using this example:
; The call has to use this syntax: readimages,file,cellsize=cellsize_return,...
; The value of cellsize is then returned in cellsize_return.
;
; General example with this code inserted below after the common blocks:
; help,cellsize
; cellsize=4
; return
;
; Execute:
; cellsize=3
; cellsize_inout=6
; readimages,cellsize=cellsize_inout
; Inside function:
; CELLSIZE        INT =  6 	; the value "inout" is copied to cellsize
; On return:
; print,cellsize,cellsize_inout
;        3       4		; cellsize not changed, cellsize_inout changed
; If called with cellsize=cellsize, cellsize = 4 on output!
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common ReadImagesLocal,message_counter
;
if n_elements(message_counter) eq 0 then message_counter=0
;
; Check if multiple files were given, one image per wavelength
files=file_search(fitsfile)
if strlen(files[0]) eq 0 then begin
	print,'***Error: file does not exist!'
	channel_images=0
	return
endif
nz=n_elements(files)
;
; Read first file and get dimensions and header
channel_image=mrdfits(files[0],0,header,/silent)
index=where(strpos(header,'NAXIS1') ge 0)
words=nameparse(header[index[0]],'=')
naxis1=fix(words[1])
index=where(strpos(header,'NAXIS2') ge 0)
words=nameparse(header[index[0]],'=')
naxis2=fix(words[1])
index=where(strpos(header,'NAXIS3') ge 0,count)
if count eq 1 then begin
	words=nameparse(header[index[0]],'=')
	naxis3=fix(words[1])
endif else naxis3=1
nx=n_elements(channel_image[*,0])
ny=n_elements(channel_image[0,*])
imsze=[nx,ny]
;
; Check if image was written by HOCHUNK3D and select the "All photons" cube
if n_elements(size(channel_image,/dim)) eq 4 then begin
	if message_counter eq 0 then $
	print,'Detected image computed by HOCHUNK3D!'
	message_counter=1
	channel_image=channel_image[*,*,*,0]
	wavelengths=mrdfits(files[0],1,header_ext,/silent)
	wavelengths=wavelengths.wavelength ; microns, converted to m below
	index=where(strpos(header,'RMAXI') ge 0)
	words=nameparse(header[index[0]],'=')
	dmaxi=double(words[1])*2	; full size of image [cm]
	pixel=dmaxi/nx
	if gen_model.px eq 0 then begin
		print,'Error: please specify parallax in model file!'
		return
	endif
	cellsize=gen_model.px*pixel/a_unit
endif
;
; Check if image was written by BSMEM
index=where(strpos(header,'PIXELATION') ge 0,count)
if count eq 1 then begin
	flipra=1	; BSMEM images have W to the left
	words=nameparse(header[index[0]],'=')
	cellsize=float(words[1])
	print,'BSMEM image: cellsize = ',cellsize
endif else flipra=0
;
; Check for wavelength or frequency axis
index=where(strpos(header,'WAVE') ge 0,wave_exists)
index=where(strpos(header,'FREQ') ge 0 and $
	    strpos(header,'RESTFREQ') lt 0,freq_exists)
if not wave_exists and not freq_exists then begin
	if n_elements(wavelengths) eq 0 then begin
		if not keyword_set(quiet) then $
		print,'Wavelength axis undefined in header of image!'
		if n_elements(gen_model) eq 0 then begin
			print,'***Error: no wavelengths defined in model!'
			return
		endif
		channel_wavelengths=gen_model.wavelengths*1e-6
		if not keyword_set(quiet) then $
		print,'Wavelengths for images taken from model.'
	endif else begin
		channel_wavelengths=wavelengths*1e-6
	endelse
endif else begin
	index=where(strpos(header,'WAVE') ge 0,exists)
	if exists and n_elements(files) eq 1 then begin
		index=where(strpos(header,'CRVAL3') ge 0)
		words=nameparse(header[index[0]],'=')
		crval3=float(words[1])
		index=where(strpos(header,'CDELT3') ge 0)
		words=nameparse(header[index[0]],'=')
		cdelt3=float(words[1])
		index=where(strpos(header,'CRPIX3') ge 0)
		words=nameparse(header[index[0]],'=')
		crpix3=float(words[1])
		channel_wavelengths=findgen(naxis3)*cdelt3+crval3 $
				   -(crpix3-1)*cdelt3
	endif
	index=where(strpos(header,'FREQ') ge 0 and $
		    strpos(header,'RESTFREQ') lt 0,exists)
	if exists and n_elements(files) eq 1 then begin
		index=where(strpos(header,'CRVAL3') ge 0)
		words=nameparse(header[index[0]],'=')
		crval3=float(words[1])
		index=where(strpos(header,'CDELT3') ge 0)
		words=nameparse(header[index[0]],'=')
		cdelt3=float(words[1])
		index=where(strpos(header,'CRPIX3') ge 0)
		words=nameparse(header[index[0]],'=')
		crpix3=float(words[1])
		channel_wavelengths=c_light/(findgen(naxis3)*cdelt3+crval3 $
				   -(crpix3-1)*cdelt3)
	endif
endelse
; If more than one image specified, assemble into cube
if nz gt 1 then begin
	if nz ne n_elements(channel_wavelengths) then begin
		print,'***Error: different number of files and wavelengths!'
		return
	endif
	channel_images=fltarr(nx,ny,nz)
	for i=0,nz-1 do begin
		channel_images[*,*,i]=mrdfits(files[i],0,header,/silent) > 0
		if keyword_set(flipra) then $
		for j=0,ny-1 do channel_images[*,j,i] $
		       =reverse(channel_images[*,j,i])
		index=where(strpos(header,'WAVE') ge 0,exists)
		if exists then begin
			words=nameparse(header[index[0]+1])
			crval3=float(words[2])
			channel_wavelengths[i]=crval3
		endif
	endfor
;	Sort by wavelength
	si=sort(channel_wavelengths)
	channel_wavelengths=channel_wavelengths[si]
	channel_images[*,*,*]=channel_images[*,*,si]
endif else begin
;	channel_images=mrdfits(fitsfile(0),0,header,/silent) > 0
	channel_images=channel_image
;	Replace image pixels with zero flux by minimum non-zero flux
	index=where(channel_images eq 0,count)
	if count gt 0 then channel_images[index]= $
				min(channel_images[where(channel_images gt 0)])
	if keyword_set(flipra) then begin
		for i=0,naxis3-1 do $
		for j=0,ny-1 do $
		channel_images[*,j,i]=reverse(channel_images[*,j,i])
	endif
endelse
; Get pixel scales
flag=0
index=where(strpos(header,'RA---') ge 0,exists)
if not exists then begin
	index=where(strpos(header,'PXSCAL1') ge 0 $
		 or strpos(header,'SCALE') ge 0,exists)
	if not exists then begin
		flag=1
	endif else begin
		words=nameparse(header[index[0]])
		if n_elements(words) eq 5 then words[4]=strupcase(words[4])
		index=where(strpos(words,'ARCSEC') ge 0,count)
		if count ne 0 then f=1000.0 else f=1.0
		cdelt1=abs(float(words[2]))*f	; [mas]
	endelse
	if flag eq 1 then begin
		index=where(strpos(header,'CDELT1') ge 0,exists)
		if not exists then begin
			flag=2
			if not keyword_set(cellsize) then begin
			print,'Warning(READIMAGES): pixel scale undefined!'
			print,'Please supply scale in keyword cellsize [mas]!'
			endif else cdelt1=cellsize
		endif else begin
			words=nameparse(header[index[0]])
			cdelt1=abs(float(words[2]))*3600000; convert deg to mas
			index=where(strpos(words,'radians') ge 0,count)
			if count gt 0 then cdelt1=cdelt1*180/!pi
		endelse
	endif
endif else begin
	words=nameparse(header[index[0]])
	index=where(strpos(header,'CDELT'+strmid(words[0],5,1)) ge 0)
	words=nameparse(header[index[0]])
	cdelt1=abs(float(words[2]))*3600000	; convert deg to mas
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt1=cdelt1*180/!pi
	cellsize=cdelt1
endelse
;
flag=0
index=where(strpos(header,'DEC--') ge 0,exists)
if not exists then begin
	index=where(strpos(header,'PXSCAL2') ge 0 $
		 or strpos(header,'SCALE') ge 0,exists)
	if not exists then begin
		flag=1
	endif else begin
		words=nameparse(header[index[0]])
		if n_elements(words) eq 5 then words[4]=strupcase(words[4])
		index=where(strpos(words,'ARCSEC') ge 0,count)
		if count ne 0 then f=1000.0 else f=1.0
		cdelt2=abs(float(words[2]))*f	; [mas]
	endelse
	if flag eq 1 then begin
		index=where(strpos(header,'CDELT2') ge 0,exists)
		if not exists then begin
			flag=2
			if not keyword_set(cellsize) then begin
			print,'Warning(READIMAGES): pixel scale undefined!'
			print,'Please supply scale in keyword cellsize [mas]!'
			endif else cdelt2=cellsize
		endif else begin
			words=nameparse(header[index[0]])
			cdelt2=abs(float(words[2]))*3600000; convert deg to mas
			index=where(strpos(words,'radians') ge 0,count)
			if count gt 0 then cdelt2=cdelt2*180/!pi
		endelse
	endif
endif else begin
	words=nameparse(header[index[0]])
	index=where(strpos(header,'CDELT'+strmid(words[0],5,1)) ge 0)
	words=nameparse(header[index[0]])
	cdelt2=abs(float(words[2]))*3600000	; convert deg to mas
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt2=cdelt2*180/!pi
	cellsize=cdelt2
endelse
if n_elements(cdelt1) ne 0 and n_elements(cdelt2) ne 0 then begin
	if cdelt1 ne cdelt2 then begin
		print,'***Error(READIMAGES): cell size in RA and Dec different!'
		return
	endif
endif
;
; If cellsize was specified on input, use it to set cdelt1 and cdelt2
if n_elements(cellsize) ne 0 then images_cellsize=cellsize
;
; Initialize default model for this type of image
flag=0
if n_elements(gen_model) eq 0 then begin
	flag=1
endif else begin
;	Causes problems, therefore commented out Feb 2017
;	if n_elements(gen_model.wavelengths) $
;	ne n_elements(channel_wavelengths) then flag=1
endelse
; Protect procedure argument
if n_elements(wavelengths) ne 0 then wavelengths_in=wavelengths
if flag then begin
	if n_elements(gen_model) ne 0 then starid=gen_model.starid
	wavelengths=[min(channel_wavelengths),max(channel_wavelengths)]*1e6
	if wavelengths[0] eq wavelengths[1] then wavelengths=wavelengths[0]
	num_wave=n_elements(wavelengths)
	gen_model=alloc_gen_model(num_wave)
	gen_error=gen_model
	if n_elements(startable) eq 1 then gen_model.starid=startable.starid $
				      else gen_model.starid='CCCNNNN'
	if n_elements(starid) ne 0 then gen_model.starid=starid
	gen_model.wavelengths=wavelengths	; model file wavelengths [mu]
	num_spot=2
	star_model=alloc_star_struct(num_wave,num_spot)
	star_error=star_model
	star_model[0].component='A'
	star_model[0].type=12
	binary_struct=alloc_binary_struct()
	binary_model=replicate(binary_struct,1)
endif
; restore procedure argument
if n_elements(wavelengths_in) ne 0 then wavelengths=wavelengths_in
;
end
