pro readmodel,modelfile
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; This procedure sets up a hierarchical model format and reads a file
; with the model information ("commands"). It also initializes the two
; structures identical to the model structures, which hold the
; parameter error information. A model may include images (or image cubes),
; these require the definition of the cellsize [mas], unless the image
; has been loaded before calling this procedure.
;
; The model file may also contain commands which analyze the model parameters
; and print the derived information (e.g., parallax).
;
common StarBase,StarTable,Notes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common AstroConstants,m_sun,r_sun,a_unit,s_year
common ReadImagesLocal,message_counter
;
message_counter=0
;
f=file_search(modelfile)
if strlen(f[0]) eq 0 then begin
	print,'Error(READMODEL): file not found!'
	return
endif else modelfile=f[0]
;
; Disabled 12.12.2023, function now in ww_modelfit
; Initialize modelfit
; if init_modelfit() ne 0 then return
;
; If in XDR format, just restore all components
if strpos(modelfile,'xdr') eq strlen(modelfile)-3 then begin
	restore,modelfile
	return
endif
;
; Read command file into string array
;
commands=''
status=dc_read_fixed(modelfile,commands,resize=[1],format='(a120)',/col)
if status ne 0 then begin
	print,'***Error(READMODEL): error reading file or file not found!'
	return
endif
;
; Remove comment lines (square brackets included in 2025
commands=commands[where(strmid(strcompress(commands,/remove_all),0,1) ne ';')]
;
; Allocate arrays
;
num_binaries=10
num_stars=2*num_binaries
index=where(strpos(strupcase(commands),'WAVELENGTHS') ne -1,count)
if count ge 1 then for i=0,count-1 do r=execute(commands[index[i]]) else begin
	print,'***Error(READMODEL): Check definition of WAVELENGTHS!'
	return
endelse
;
; General parameters
num_wave=n_elements(wavelengths)
if num_wave eq 0 then begin
	wavelengths=1
	num_wave=1
endif
num_spot=2
starid=''
px=0.d0					; Parallax [mas]
rv=0.d0					; radial velocity [km/s]
sm=1.d0					; System mass
;
; Star parameters
name    	=strarr(num_stars)	; A, B, C, D,...
wmc		=strarr(num_stars)	; WMC designation, e.g. 'Aa'
type		=intarr(num_stars)+1	; Component model type
sed		=strarr(num_stars)	; SED data file
model		=strarr(num_stars)	; Model atmosphere file
cellsize	=dblarr(num_stars)	; Cell size for images w/out header
mass		=dblarr(num_stars)+1	; solar masses
diameter	=dblarr(num_stars)	; Diameter of disk or inner rim [mas]
width    	=dblarr(num_stars)	; Disk width / inner radius [0-1]
ratio		=dblarr(num_stars)+1	; Minor axis/major axis
pa		=dblarr(num_stars)	; Major axis position angle, 0=N
omega      	=dblarr(num_stars)	; omega (rotation) / omega breakup
omega      	=dblarr(num_stars)	; ratio of axial to orbital rate
tilt       	=dblarr(num_stars)	; inclination of rot. axis, 90=eq.on
gr		=dblarr(num_stars)	; exponent in gravity darkening law
albedo		=dblarr(num_stars)	; bolometric albedo
teff		=dblarr(num_stars)	; effective temperature
alpha		=dblarr(num_stars)	; accretion disk: T ~ r^alpha
logg		=dblarr(num_stars)	; log(g)
xoff		=dblarr(num_stars)	; RA offset in mas (positive to East)
yoff		=dblarr(num_stars)	; Dec offset in mas (positive to North)
spot		=dblarr(4,num_stars)	; Teff-factor,radius,angle,diameter
spotparms	=dblarr(4,num_spot,num_stars)
magnitudes	=dblarr(num_wave,num_stars)
;
; Binary parameters
component	=strarr(num_binaries)	; A-B, AB-C, AB-CD,...
method		=intarr(num_binaries)
wdtype		=intarr(num_binaries)
wdband		=intarr(num_binaries)
massratio	=dblarr(num_binaries)	;
semimajoraxis	=dblarr(num_binaries)	; mas
inclination	=dblarr(num_binaries)	; degrees
ascendingnode	=dblarr(num_binaries)	; degrees, component RV positive
eccentricity	=dblarr(num_binaries)
periastron	=dblarr(num_binaries)	; degrees, of primary!
apsidalmotion	=dblarr(num_binaries)	; degrees/year
epoch		=dblarr(num_binaries)	; secondary at periastron
period		=dblarr(num_binaries)	; days
rho             =dblarr(num_binaries)	; mas
theta           =dblarr(num_binaries)	; degrees
;
; The model may contain analytic commands which can only be executed after
; the model was updated! Here we implemented handling of modelpx()
command_i=''
command_n=0
for i=0,n_elements(commands)-1 do begin
	if strpos(commands[i],'modelpx') ge 0 then begin
		command_i=[command_i,commands[i]]
		command_n=command_n+1
	endif else if strpos(commands[i],'some_other_command') ge 0 then begin
		command_i=[command_i,commands[i]]
		command_n=command_n+1
	endif else r=execute(commands[i])
endfor
if n_elements(command_i) gt 1 then command_i=command_i[1:n_elements(command_i)-1
;
; Allocate structures, arrays, store information, and remove zero entries
;
; General model parameters
gen_model=alloc_gen_model(num_wave)
wavelengths=double(wavelengths)
si=sort(wavelengths)
gen_model.wavelengths=wavelengths[si]
gen_error=gen_model
if total(strlen(starid)) gt 0 then begin
	gen_model.starid=starid
	gen_model.rv=rv
	gen_model.px=px
	gen_model.sm=sm
	if n_elements(StarTable) eq 0 then get_startable,starid
;
; 	If StarTable has only one entry, use that name for gen_model.starid
	if n_elements(startable) eq 1 then gen_model.starid=startable.starid
	index=where(StarTable.starid eq gen_model.starid,count)
	if count eq 0 then begin
		addstar,starid
		index=where(StarTable.starid eq starid)
	endif
	gen_model.ra=StarTable[index[0]].ra
	gen_model.dec=StarTable[index[0]].dec
;	set_parallax,StarTable(index(0)).px*1000,StarTable(index(0)).pxe,0
endif
;
; Star model parameters
star_struct=alloc_star_struct(num_wave,num_spot)
star_model=replicate(star_struct,num_stars)
star_error=star_model
if total(strlen(starid)) gt 0 then begin
	star_model.component		=name
	star_model.wmc			=wmc
	star_model.type			=type
	star_model.sed			=sed
	star_model.model		=model
	star_model.cellsize		=cellsize
	star_model.mass			=mass
	star_model.diameter		=diameter
	star_model.width     		=width
	star_model.ratio		=ratio
	star_model.pa			=pa
	star_model.omega  		=omega
	star_model.tilt   		=tilt
	star_model.gr			=gr
	star_model.albedo		=albedo
	star_model.teff			=teff
	star_model.alpha		=alpha
	star_model.logg			=logg
	star_model.xoff			=xoff
	star_model.yoff			=yoff
	star_model.spot			=spot
	star_model.spotparms		=spotparms
	star_model.magnitudes		=reform(magnitudes[si,*])
;
	star_error.component		=name
;
	star_model=star_model[where(strlen(star_model.component) ne 0)]
	star_error=star_error[where(strlen(star_model.component) ne 0)]
endif
;
; Binary model parameters
binary_struct=alloc_binary_struct()
binary_model=replicate(binary_struct,num_binaries)
binary_error=binary_model
index=where(epoch gt 0,count)
if count gt 0 then epoch[index]=epoch[index]-2440000.d0
if total(strlen(component)) gt 0 then begin
	binary_model.component		=component
	binary_model.method		=method
	binary_model.wdtype		=wdtype
	binary_model.wdband		=wdband
	binary_model.massratio		=massratio
	binary_model.semimajoraxis	=semimajoraxis
	binary_model.inclination	=inclination
	binary_model.ascendingnode	=ascendingnode
	binary_model.eccentricity	=eccentricity
	binary_model.periastron		=periastron
	binary_model.apsidalmotion	=apsidalmotion
	binary_model.epoch		=epoch
	binary_model.period		=period
	binary_model.rho		=rho
	binary_model.theta		=theta
;
	binary_error.component		=component
;
;	The following was found not to work anymore on May 6, 2024
;	c=fltarr(num_binaries)
;	for i=0,num_binaries-1 do $
;	 for j=1,n_tags(binary_model)-1 do c(i)=c(i)+total(binary_model(i).(j))
;	index=where(c ne 0,count)
;	if count eq num_binaries then $
;		print,'Warning: maximum of ' $
;		      +string(num_binaries,format='(i2)') $
;		      +' binary model parameters found, please check model!'
;	binary_model=binary_model(where(c ne 0))
;	binary_error=binary_error(where(c ne 0))
;
;	Replaced on May 6, 2024
	binary_model=binary_model[where(strlen(binary_model.component) ne 0)]
	binary_error=binary_error[where(strlen(binary_model.component) ne 0)]
endif
;
if checkmodel() eq -1 then return
;
; Execute embedded analytic model commands now that the model was created
for j=0,command_n-1 do r=execute(command_i[j])
command_n=0
;
if total(strlen(binary_model.component)) ne 0 then begin
; 	adjustfluxes,topbincomp(),dblarr(num_wave)+1
; 	adjustmasses,topbincomp()
endif
if n_elements(star_model) gt 1 and total(abs(star_model.teff)) eq 0 then begin
; 	For non-physical models, compute all as blackbody (req. by Pearl)
	star_model.teff=-5555
	print,'Note: setting all Teff to -5555 K!'
endif
; fitwaveparms
;
; Check availability of images for code=12
index=where(star_model.type eq 12,count)
;
for i=0,count-1 do begin
;
if strlen(star_model[index[i]].model) gt 0 then begin
	image=file_search(star_model[index[i]].model)
	if strlen(image[0]) eq 0 then begin
		print,'Error: image file not found: '+star_model[index[i]].model
		return
	endif else begin
		readimages,image,cellsize=cellsize_return
		cellsize[i]=cellsize_return
	endelse
endif
;
endfor
;
; Check availability of images for code=13
index=where(star_model.type eq 13,count)
for i=0,count-1 do begin
	if strlen(star_model[index[i]].model) gt 0 then begin
	image=file_search(star_model[index[i]].model)
	if strlen(image[0]) eq 0 then begin
		print,'Error: image file not found: '+star_model[index[i]].model
		return
	endif else readimage,image
	endif else begin
		print,'Warning: image not specified, assume it is loaded!'
	endelse
endfor
; Check availability of parallax and Teff for code=14
index=where(star_model.type eq 14,count)
if count gt 0 then print,'Note: you must set a value for the parallax!'
for i=0,count-1 do begin
	if star_model[i].teff eq 0 then print,'***Error: Teff must be non-zero!'
endfor
;
print,'Model file read and checked successfully.'
;
end
