pro riseset,uptime,nighttime,force=force,ldl=ldl,iss=iss
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Given a StarTable, compute the visibility for a given date and system.
; Return uptime(2,num_stars), the rise and set times for the interferometer
; horizon in seconds. Also return nighttime(2), beginning and end in seconds
; of a night.
;
; Note that only an average siderostat horizon (set by motor angle limits)
; is used. The delays, which depend on the specific configuration, are
; taken into account.
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common LocalRiseSet,date_p,system_p,stars_p,stations_p, $
                times,gst,za_sun,za_moon,ha_moon,az_moon, $
		za_star_p,ha_star_p,az_star_p, $
		sun_ok,moon_ok,zas_ok,sid_ok_p,fdl_ok_p
;
; Call external ISS shadow code if iss=1
if n_elements(iss) eq 0 then iss=0
;
; If force=1, always recompute circumstances
if n_elements(force) eq 0 then force=0
;
num_stars=n_elements(StarTable)
if num_stars eq 0 then begin
        print,'***Error(RISESET): no stars defined!'
        return
endif
;
if n_elements(Date) eq 0 then begin
	print,'Warning(RISESET): Date undefined!'
        parseidldate,systime(),y,m,d
        Date=nextdate(constrictordate(y,m,d))
	print,'Set date to: ',date
endif
if n_elements(SystemId) eq 0 then begin
	print,'Warning(RISESET): SystemId undefined!'
	SystemId='NPOI'
	print,'Set SystemId to: ',SystemId
endif
if system_config(system_id(systemid),'EXIST') ne 1 then begin
	print,'***Error(RISESET): SystemId unknown!'
	return
endif
;
if n_elements(date_p) eq 0 then date_p=''
if n_elements(system_p) eq 0 then system_p=''
if n_elements(stars_p) eq 0 then stars_p=''
if n_elements(stations_p) eq 0 then stations_p=''
;
; New computation of GST, Sun and Moon necessary if date or system changed
IF Date NE date_p OR SystemId NE system_p OR force THEN BEGIN
;
; 	Get GeoParms
	hds_close
	get_sysconfig
;
; 	Allocate array of UT hours
	num_ut=200
	times=dindgen(num_ut)/num_ut*86400-(geoparms.longitude/15+12)*3600
;
; 	Compute UT1
	ut1=utc2ut1(times)
;
; 	Compute Greenwich apparent sidereal time
	gst=ut12gst(times,ut1)
;
; 	Get Right Ascension and declination of the Sun
	topoplanet,times,intarr(num_ut)+10,ra_sun,dec_sun,distance
;
; 	Compute hour angle of the Sun
	ha_sun=hourangle(gst,ra_sun)
;
; 	Compute zenith angle of the Sun
	za_sun=zenithangle(ha_sun,dec_sun)
;
; 	Get Right Ascension and declination of the Moon (w/ephemeris access)
;	topoplanet,times,intarr(num_ut)+11,ra_moon,dec_moon,distance
;
; 	Get Right Ascension and declination of the Moon (w/out ephemeris access)
	parsedate,Date,y,m,d
	tjd=(times+GeoParms.TAI_UTC+GeoParms.TDT_TAI)/86400+julian(y,m,d)
	moonpos,tjd,ra_moon,dec_moon	; from idlastro library
	ra_moon=ra_moon/15
;
; 	Compute hour angle of the moon
	ha_moon=hourangle(gst,ra_moon)
;
; 	Compute zenith angle of the moon
	za_moon=zenithangle(ha_moon,dec_moon)
;
; 	Compute azimuth of the moon
	hadec=transpose([[ha_moon],[dec_moon]])
	azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec)))
	az_moon=reform(azel[0,*])
;
ENDIF
;
; Compute night time
za_sun_min=108.0
index=where(za_sun gt za_sun_min,count)
nighttime=[times[index[0]],times[index[count-1]]]
;
; Allocate ZA, HA, and AZ
num_star=n_elements(StarTable)
num_ut=n_elements(times)
za_star=dblarr(num_ut,num_star)
ha_star=dblarr(num_ut,num_star)
az_star=dblarr(num_ut,num_star)
;
; These are binary 1/0 (T/F) indices for various conditions
sun_ok=intarr(num_ut)
moon_ok=intarr(num_ut,num_star)
zas_ok=intarr(num_ut,num_star)
sid_ok=intarr(num_ut,num_star)
fdl_ok=intarr(num_ut,num_star)+1
;
genconfig_checksum=strjoin(genconfig.stationid)
genconfig_checksum=''
for i=0,n_tags(genconfig)-1 do genconfig_checksum=genconfig_checksum $
				 +strjoin(string(genconfig.(i)),/single)
;
; Compute angles for the stars
stars=strarr(num_ut)
for i=0,num_star-1 do begin
	j=where(stars_p eq StarTable[i].starid,count) & j=j[0]
	if count eq 1 and Date eq date_p and not force $
		      and genconfig_checksum eq stations_p then begin
		ha_star[*,i]=ha_star_p[*,j]
		za_star[*,i]=za_star_p[*,j]
		az_star[*,i]=az_star_p[*,j]
		sid_ok[*,i]=sid_ok_p[*,j]
		fdl_ok[*,i]=fdl_ok_p[*,j]
	endif else begin
		stars[*]=StarTable[i].starid
		topostar,times,stars,StarTable,ra_app,dec_app
		ha_star[*,i]=hourangle(gst,ra_app)
		za_star[*,i]=zenithangle(ha_star[*,i],dec_app)
		hadec=transpose([[ha_star[*,i]],[dec_app]])
		azel=horizon2azel(equatorial2horizon(hadec2equatorial(hadec)))
		az_star[*,i]=reform(azel[0,*])
		sid_ok[*,i]=sidlimits(ha_star[*,i],dec_app)
		fdl_ok[*,i]=fdllimits(ha_star[*,i],dec_app)
		if keyword_set(ldl) then fdl_ok[*,i]=1
		if system_id(systemid) eq 'VLTI' then begin
			sid_ok[*,i]=1
			for j=0,genconfig.numsid-1 do begin
			case genconfig.stationid[j] of
;			Here AZ is 0 towards S, increasing Westwards
;				      el azl azh
;			Data from station_parameters.dat, cols 7-12
			'AA0':limits=[48,164,210,0,0,0]
			'AA1':limits=[33,162,195,0,0,0]
			'AB0':limits=[52,150,198,0,0,0]
			'AB1':limits=[35,153,187,0,0,0]
			'AB2':limits=[35,153,187,0,0,0]
			'AC0':limits=[53,136,185,0,0,0]
			'AC1':limits=[35,143,178,0,0,0]
			'AD0':limits=[48,111,156,0,0,0]
			'AE0':limits=[37,97,134,0,0,0]
			'AG2':limits=[35,63,97,35,134,168]
			'AJ3':limits=[36,112,147,68,219,282]
			'AJ4':limits=[50,92,139,59,257,311]
			'AJ5':limits=[68,39,102,36,292,327]
			'AJ6':limits=[50,2,49,0,0,0]
			'AM0':limits=[30,145,175,0,0,0]
	 		else:limits=[0,0,360,0,0,0]
			endcase
			for k=0,n_elements(times)-1 do begin
			el=azel[1,k]
			shadow=0.d0
			if iss then begin
; 			Convert to 0 towards S, incr. Eastwards
			az=(azel[0,k]+180) mod 360
			n_az=1l
			n_el=1l
			r=0.0
			t=0.0
			loc=vlti_station(genconfig.stationid[j],ts)
			status=linknload(!external_lib,'shadow', $
					n_az,n_el,az,el,shadow,loc,r,t,ts)
			endif else begin
; 			Convert to 0 towards S, incr. Westwards
			az=(360+180-azel[0,k]) mod 360
			if az gt limits[1] and az lt limits[2] $
					   and el lt limits[0] then shadow=1
			if az gt limits[4] and az lt limits[5] $
					   and el lt limits[3] then shadow=1
			endelse
			if shadow ne 0 then sid_ok[k,i]=0
			endfor
			endfor
		endif
	endelse
endfor
;
; if SystemId ne 'NPOI' then sid_ok(*,*)=1
; if SystemId ne 'NPOI' then fdl_ok(*,*)=1
;
uptime=fltarr(2,num_star)
index=where(za_sun gt za_sun_min,count)
if count ge 1 then sun_ok[index]=1
;
za_star_max=system_config(system_id(systemid),'ZALIMITMAX')
za_star_min=system_config(system_id(systemid),'ZALIMITMIN')
for i=0,num_star-1 do begin
	index=where(za_star[*,i] lt za_star_max $
		and za_star[*,i] gt za_star_min,count)
	if count ge 1 then zas_ok[index,i]=1
	index=where(zas_ok[*,i] and sid_ok[*,i] and fdl_ok[*,i],count)
	if StarTable[i].starid eq 'FKV0000' then begin
		index=indgen(num_ut)
		count=num_ut
	endif
;
	if count gt 1 then begin
		index=contiguous(index)
		count=n_elements(index)
		i0=index[0]
		i1=index[count-1]
		uptime[0,i]=times[i0]
		uptime[1,i]=times[i1]
	endif
endfor
;
if n_params() eq 0 then $
for i=0,num_star-1 do print,StarTable[i].starid,'  ',hms(uptime[*,i]/3600)
;
; Save the results for next call
date_p=Date
system_p=SystemId
stars_p=StarTable.starid
genconfig_checksum=strjoin(genconfig.stationid)
genconfig_checksum=''
for i=0,n_tags(genconfig)-1 do genconfig_checksum=genconfig_checksum $
				 +strjoin(string(genconfig.(i)),/single)
stations_p=genconfig_checksum
ha_star_p=ha_star
za_star_p=za_star
az_star_p=az_star
sid_ok_p=sid_ok
fdl_ok_p=fdl_ok
;
end
