pro set_visampphase
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Compute VisAmp and VisPhase, as well as their respective errors, from the
; complex quantities ComplexVis and ComplexVisErr for plotting
; and editing. Phases are counter-clockwise from 1+0i in radians.
;
; The following is a citation from D F Buschers Thesis (p. 18):
; "A complex quantity [like ComplexVis] may have different variances along
; different directions in the complex plane. This can be determined for any
; arbitrary direction from just two parameters, var_1(C) and var_2(C), which
; are defined for a complex quantity C as:
; var_1(C) = E(CC*) - E(C)E(C)*, var_2(C) = E(CC) - E(C)E(C).
; With these definitions, the variance of C along a direction at an angle t
; to the real axis is: var(C,t)=1/2(var_1(C)+Re{var_2(C)e^(-2it)}).
; Thus var_2(C) represents the part of the noise which varies with direction
; in the complex plane and var_1(C) represents the circularly symmetric term."
;
; In this program, it is assumed that ComplexVis(*,0,*,*,*) is the real part
; of ComplexVis, and ComplexVis(*,1,*,*,*) is the imaginary part. However, we
; assume that there is no correlation between the real and imaginary parts, so
; that the error of the complex visibility is also complex, with the real part
; corresponding to the error of the real part of the visibility.
;
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
; 	Compute VisAmp
	CVR=ComplexVis[*,0,*,*,*]
	CVI=ComplexVis[*,1,*,*,*]
	VisAmp[*,*,*,*]=sqrt(CVR^2+CVI^2)
;
;	Compute VisPhase, replacing undefined values
	index=where(CVR eq 0 and CVI eq 0,count)
	if count gt 0 then begin
		CVR[index]=1
		CVI[index]=1
		VisAmp[index]=2
	endif
	VisPhase[*,*,*,*]=atan(CVI,CVR)
;
;	Compute VisAmpErr. If VisAmp is zero, above replacements
;	will give a reasonable estimate.
	VisAmpErr[*,*,*,*]= $
		sqrt((CVR*ComplexVisErr[*,0,*,*,*])^2 $
		    +(CVI*ComplexVisErr[*,1,*,*,*])^2) $
		/VisAmp
;
;	Reset VisAmp replacements and flag phase of zero amplitudes
	if count gt 0 then begin
		VisAmp[index]=0
		VisPhaseErr[index]=-1
	endif
;
;	Compute VisPhaseErr for valid data points
	index=where(VisAmp gt 0,count)
	if count gt 0 then VisPhaseErr[index]=VisAmpErr[index]/VisAmp[index]
;
;	Free memory
	CVR=0
	CVI=0
;
;	Make sure all derived data is flagged where the input data is
	CVRE=ComplexVisErr[*,0,*,*,*]
	CVIE=ComplexVisErr[*,1,*,*,*]
	index=where(CVRE le 0 or CVIE le 0,count)
	if count gt 0 then begin
		VisAmpErr[index]=-1
		VisPhaseErr[index]=-1
	endif
;
end
