pro set_viserrors,use_e
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Compute the calibration error based on temporal fluctuations of the
; calibrator visibilities squared and triple amplitudes, and add in
; quadrature to the error of the calibrated visibilities for selected stars.
; To compute the calibration error, use the EC visibilities to compute
; chisq, and compute the additive error as sqrt((chisq-1)*e^2/z^2), where
; e is the intrinsic error, and z is the calibrated visibility (which is
; not necessarily unity if the calibrator is resolved). The latter two
; are mean quantities, i.e. the average of the values. Other means might
; be considered too (geometric, mean of the squares, etc.). The aim is to
; produce unity chisq for the calibrator data.
;
; Derive selected program and calibrator stars from the selection made
; in the appropriate widgets.
;
; If use_e=1, use the normalized amplitudes (E) instead of the
; calibrated normalized amplitudes (EC). This is useful if the latter were
; not calibrated for any reason. In this case, we "calibrate" by fitting
; polynomials to the data of each channel, before computing the chisq.
;
;
common SolInfo,cal_stars,cal_options,indicators,sel_indct,sel_funcn
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; See whether uncalibrated but normalized amplitudes are to be used
if n_elements(use_e) ne 0 then use_e=use_e gt 0 else use_e=0
;
; Check if calibrator stars are defined
if n_elements(cal_stars) eq 0 then begin
	print,'***Error(SET_VISERRORS): calibrator list undefined!'
	return
endif
;
; See if calibrator stars were selected
if total(strlen(cal_stars)) eq 0 then begin
	print,'***Error(SET_VISERRORS): no calibrators selected!'
	return
endif
;
; Obtain scan index corresponding to calibrator star selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
        index=where(cal_stars eq stars[k],count)
        if count eq 0 then begin
                index=where(scans.StarId eq stars[k])
                mask[index]=-1
        endif
endfor
calscan_index=where(mask eq 0,calscan_count)
MINSCAN=2
if calscan_count lt MINSCAN then begin
	print,'***Error(SET_VISERRORS): not enough calibrator scans!'
	return
endif
;
; Check if program stars are defined
flag=set_starsel()
if set_indexsel('y') ne 0 then return
if n_elements(ds_stars) eq 0 then begin
	print,'***Error(SET_VISERRORS): program star list undefined!'
	return
endif
;
; See if program stars were selected
if total(strlen(ds_stars)) eq 0 then begin
	print,'***Error(SET_VISERRORS): no program stars selected!'
	return
endif
;
; Obtain scan index corresponding to program star selection
list_stars,stars
mask=intarr(n_elements(scans))
for k=0,n_elements(stars)-1 do begin
	index=where(ds_stars eq stars[k],count)
	if count eq 0 then begin
		index=where(scans.StarId eq stars[k])
		mask[index]=-1
	endif
endfor
scan_index=where(mask eq 0,scan_count)
;
; Determine r.m.s. of calibrator scans for each output beam and channel
for ob=0,GenConfig.NumOutBeam-1 do begin
for bl=0,GenConfig.NumBaseline[ob]-1 do begin
;
bad_ch=intarr(GenConfig.NumSpecChan[ob])
for ch=0,GenConfig.NumSpecChan[ob]-1 do begin
;
	if use_e then begin
		cv=scans[calscan_index].VisSqE(ob,ch,bl)
		ce=scans[calscan_index].VisSqEErr(ob,ch,bl)
		ct=scans[calscan_index].time/3600
		index=where(ce gt 0,count)
		if count gt MINSCAN then begin
			cv_fit=poly(ct,poly_fit(ct[index],cv[index],1))
			cv=cv/cv_fit
			ce=ce/cv_fit
		endif
	endif else begin
		cv=scans[calscan_index].VisSqEC(ob,ch,bl)
		ce=scans[calscan_index].VisSqECErr(ob,ch,bl)
	endelse
	index_ce=where(ce gt 0,count_ce)
	if count_ce gt MINSCAN then begin
		chisq=total(((cv[index_ce]-1)/ce[index_ce])^2)/(count_ce-1)
		if chisq lt 1 then chisq=1
;		Reset error bars first
		j=where(scans[scan_index].VisSq(ob,ch,bl) ne 0,count)
		scans[scan_index[j]].vissqcerr(ob,ch,bl)= $
		 scans[scan_index[j]].vissqerr(ob,ch,bl) $
		 *(scans[scan_index[j]].vissqc(ob,ch,bl) $
		  /scans[scan_index[j]].vissq(ob,ch,bl))
		w=sqrt(chisq-1) $
		 *avg(scans[calscan_index[index_ce]].VisSqECErr(ob,ch,bl) $
		     /scans[calscan_index[index_ce]].VisSqEC(ob,ch,bl))
		j=where(scans[scan_index].VisSqCErr(ob,ch,bl) lt 0,count)
;		Add in quadrature
		scans[scan_index].VisSqCErr(ob,ch,bl)= $
		 sqrt(scans[scan_index].VisSqCErr(ob,ch,bl)^2 $
		    +(scans[scan_index].VisSqC(ob,ch,bl)*w)^2)
		if count gt 0 then begin
			scans[scan_index[j]].VisSqCErr(ob,ch,bl)= $
		       -scans[scan_index[j]].VisSqCErr(ob,ch,bl)
		endif
	endif else begin
		bad_ch[ch]=ch+1
	endelse
endfor
index=where(bad_ch ne 0,count)
if count gt 0 then $
	print,'Warning(SET_VISERRORS): not enough cal scans for ob=', $
	ob+1,', bl=',bl+1,', ch=',retroparse(bad_ch[index]), $
	format='(a,i1,a,i1,a,a)'
endfor
endfor
;
; Determine r.m.s. of calibrator scans for each output triple and channel
for tr=0,GenConfig.NumTriple-1 do begin
bad_ch=intarr(GenConfig.TripleNumChan[tr])
for ch=0,GenConfig.TripleNumChan[tr]-1 do begin
;
;	Triple amplitudes
	if use_e then begin
		cv=scans[calscan_index].TripleAmpE(tr,ch)
		ce=scans[calscan_index].TripleAmpEErr(tr,ch)
		ct=scans[calscan_index].time/3600
		index=where(ce gt 0,count)
		if count gt MINSCAN then begin
			cv_fit=poly(ct,poly_fit(ct[index],cv[index],1))
			cv=cv/cv_fit
			ce=ce/cv_fit
		endif
	endif else begin
		cv=scans[calscan_index].TripleAmpEC(tr,ch)
		ce=scans[calscan_index].TripleAmpECErr(tr,ch)
	endelse
	index_ce=where(ce gt 0,count_ce)
	if count_ce gt MINSCAN then begin
;		Reset error bar first
		j=where(scans[scan_index].TripleAmp(tr,ch) ne 0,count)
		scans[scan_index[j]].TripleAmpCErr(tr,ch)= $
		 scans[scan_index[j]].TripleAmpErr(tr,ch) $
		 *scans[scan_index[j]].TripleAmpC(tr,ch) $
		 /scans[scan_index[j]].TripleAmp(tr,ch)
		chisq=total(((cv[index_ce]-1)/ce[index_ce])^2)/(count_ce-2)
		if chisq lt 1 then chisq=1
		w=sqrt(chisq-1) $
		 *avg(scans[calscan_index[index_ce]].TripleAmpCErr(tr,ch) $
		     /scans[calscan_index[index_ce]].TripleAmpC(tr,ch))
		j=where(scans[scan_index].TripleAmpCErr(tr,ch) lt 0,count)
		scans[scan_index].TripleAmpCErr(tr,ch)= $
		 sqrt(scans[scan_index].TripleAmpCErr(tr,ch)^2 $
		    +(scans[scan_index].TripleAmpC(tr,ch)*w)^2)
		if count gt 0 then begin
			scans[scan_index[j]].TripleAmpCErr(tr,ch)= $
		       -scans[scan_index[j]].TripleAmpCErr(tr,ch)
		endif
	endif else begin
		bad_ch[ch]=ch+1
	endelse
;
endfor
index=where(bad_ch ne 0,count)
if count gt 0 then $
	print,'Warning(SET_VISERRORS): not enough cal scans for tr=', $
	tr+1,', ch=',retroparse(bad_ch[index]),format='(a,i2,a,a)'
endfor
;
; Determine r.m.s. of calibrator scans for each output triple and channel
for tr=0,GenConfig.NumTriple-1 do begin
bad_ch=intarr(GenConfig.TripleNumChan[tr])
for ch=0,GenConfig.TripleNumChan[tr]-1 do begin
;
;	Closure phases
;	if use_e then begin
;		cv=scans(calscan_index).TriplePhase(tr,ch)
;		ce=scans(calscan_index).TriplePhaseErr(tr,ch)
;		ct=scans(calscan_index).time/3600
;		index=where(ce gt 0,count)
;		if count gt MINSCAN then begin
;			cv_fit=poly(ct,poly_fit(ct(index),cv(index),1))
;			cv=cv-cv_fit
;		endif
;	endif else begin
		cv=scans[calscan_index].TriplePhaseC(tr,ch)
		ce=scans[calscan_index].TriplePhaseCErr(tr,ch)
;	endelse
	index_ce=where(ce gt 0,count_ce)
	if count_ce gt MINSCAN then begin
;		Reset error bar first
		scans[scan_index].TriplePhaseCErr(tr,ch)= $
		 scans[scan_index].TriplePhaseErr(tr,ch)
		chisq=total(((cv[index_ce]-0)/ce[index_ce])^2)/(count_ce-2)
		if chisq lt 1 then chisq=1
		w=sqrt(abs(stddev(cv[index_ce])^2-median(ce[index_ce])^2/chisq))
		j=where(scans[scan_index].TriplePhaseCErr(tr,ch) lt 0,count)
		scans[scan_index].TriplePhaseCErr(tr,ch)= $
		 sqrt(scans[scan_index].TriplePhaseCErr(tr,ch)^2+w^2)
		if count gt 0 then begin
			scans[scan_index[j]].TriplePhaseCErr(tr,ch)= $
		       -scans[scan_index[j]].TriplePhaseCErr(tr,ch)
		endif
	endif else begin
		bad_ch[ch]=ch+1
	endelse
;
endfor
index=where(bad_ch ne 0,count)
if count gt 0 then $
	print,'Warning(SET_VISERRORS): not enough cal scans for tr=', $
	tr+1,', ch=',retroparse(bad_ch[index]),format='(a,i2,a,a)'
endfor
;
print,'VisErrors set for stars',' '+ds_stars,'.',format='(7a)'
;
end
