function setlcparms,model,jds,spotparms
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Initialize parameters for the Wilson-Devinney code (LCDC, 2015).
; jds are the Julian day epochs (-2440000).
; Return the parameters in form of an array, also return spotparms.
;
; Primary: increasing spot longitude rotates star against orbital sense
;		longitude zero spot faces secondary
; Secondary: increasing longitude rotates star against orbital sense
;		longitude zero spot faces primary
; Spot latitude increases from North to South.
;
; For light curves (MPAGE=1), star 1 is, by definition, the one at superior
; conjunction near phase zero when parameter PSHIFT is entered as zero. It
; will be the one eclipsed near phase zero if there are eclipses.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common AuxData,parallaxes,k1,k2,vsini
;
RAD=180/pi_circle
;
; Restore float array "effwvls" and string array "filters" for WD bands
restore,!oyster_dir+'source/lcdc2015/ibands.xdr'
;
if componentparse(model.component,comp_1,comp_2) ne 0 then return,-1
star1=where(star_model.component eq comp_1) & star1=star1[0]
star2=where(star_model.component eq comp_2) & star2=star2[0]
model1=star_model[star1]
model2=star_model[star2]
;
num_parms=85
lcparms=dblarr(num_parms)
max_spots=2
max_stars=2
spotparms=dblarr(4,max_spots,max_stars)
;
; spotparms(0,0,0)=180. 	; Latitude [deg], 0 - 180
; spotparms(1,0,0)=0.		; Longitude [deg], 0 - 360
; spotparms(2,0,0)=20.		; Radius [deg]
; spotparms(3,0,0)=0.8		; Temperature factor
;
for j=0,max_spots-1 do begin
	spotparms[*,j,0]=model1.spotparms[*,j]
	spotparms[*,j,1]=model2.spotparms[*,j]
endfor
spotparms[0:2,*,*]=spotparms[0:2,*,*]/RAD
nsp1=fix(total(total(abs(model1.spotparms),1) gt 0)); Number of spots on star 1
nsp2=fix(total(total(abs(model2.spotparms),1) gt 0)); Number of spots on star 2
;
; Parameters (in order read by kmlc): implicit real*8(a-h,o-z)
mpage=5			; output: 1=light curves, 2=RVs, 5=images
nref=1			; number of reflections if mref=2
mref=1			; simple treatment of reflection effect
ifsmv1=0 		; star spot control #1 (0/1, 0=stationary spot)
ifsmv2=0 		; star spot control #2 (0/1, 1=spot moves in long.)
icor1=1			; proximity and eclipse effects star 1 if 1
icor2=1			; proximity and eclipse effects star 2 if 1
if3b=0			; light time effect due to 3rd body
ld1=1			; linear cosine law for limb darkening od star 1
ld2=1			; linear cosine law for limb darkening od star 2
kspev=0			; whether spots can evolve
kspot=1			; whether old spot model applies (1) or new one (2)
nomax=1			; spot growth trapezoidal
ifcgs=0			; physical units for light curves
ktstep=0		; only for mpage=6
jdphs=2			; independent variable time (1) or phase (2)
hjd0=model.epoch	; ephemeris reference time (for jdphs=1)
pzero=model.period	; orbital period in days (at hjd0)
phases=(jds-model.epoch)$
      /pzero mod 1	; phases of binary, superior conjunction of primary at 0
dpdt=0.			; rate of period change
pshift=0. 		; constant phase shift
delph=0.		;
nga=1			; NGA=1 there is no phase smearing
stdev=0.		;
noise=0.		;
seed=0.			;
hjdst=min(jds)		; start time for output points (jdphs=1)
hjdsp=max(jds)		; stop  time for output points (jdphs=1)
hjdin=1			; increment of time for output points (jdphs=1)
phn=0			; Phase of normalization for normalized light/magnitude
phstrt=min(phases)	; start phase, for jdphs=2
phstop=max(phases)	; stop phase, for jdphs=2
phin=0.1  		; phase increment, for jdphs=2
phobs=0.		; phase of tobs datum
lsp=1			; whether TOBS and TAVSP are for star 1 or star 2
tobs=abs(model1.teff)/10000  ; Spectroscopic temperature at phase PHOBS
tavsp=abs(model1.teff)/10000 ; Computed flux-weighted mean surface temperature
mode=model.wdtype 	; -1: X-ray binaries
			; 0: no constraints
			; 2: detached binary
			; 3: overcontact binaries
			; 4: semi-detached binaries
			; 5: Algol-types
			; 6: double-contact
ipb=1  			; 0: compute L2 from T2; 1: decouple L from T
ifat1=0 		; 0 = Blackbody radiation, 1 = stellar atmosphere
ifat2=0 		; see ifat1
n1=20 			; number of latitude rows per hemisphere
n2=20 			; see n1, longitude elements scale with sine of latitude
perr0=(model.periastron+180)/RAD 	; in radians, +180 as of Feb 22, 2019!
dperdt=0.		; the first time derivative of ω
the=0. 			; semi-duration of primary eclipse
vunit=1 		; unit for radial velocity input and output (km/s)
e=model.eccentricity	; eccentricity
k1_m=modelk(model1.component)
k2_m=modelk(model2.component)
a1sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k1_m*model.period
a2sini=13751.0*sqrt(1.d0-model.eccentricity^2)*k2_m*model.period
rsun=6.96e5 		; radius of sun [km]
a=((a1sini+a2sini)/sin(model.inclination/RAD))/rsun
f1=model1.omega 	; ratio of axial rotation rate to orbital rate
f2=model2.omega
vga=gen_model.rv/vunit	; systemic velocity (in units of vunit)
vga=0			; set to zero, this is added by modelvel
xincl=model.inclination	; orbital inclination in degrees
xincl=(90-xincl)+90	; Bug in WD?: prograde and retrograde exchanged!
gr1=model1.gr 		; exponent in gravity darkening law
gr2=model2.gr 		; see gr1, convective envelopes: 0.3, radiative: 1.0
abunin=0.		; [M/H] as defined logarithmically, relative to the Sun
fspot1=0.		; spot angular drift rates in longitude
fspot2=0.		; rate 1.0000 means drift matches mean orbital rate
tavh=abs(model1.teff)/10000 	; effective temperature in 10000 K
tavc=abs(model2.teff)/10000	; see tavh, make sure tavh/tavc are positive
alb1=model1.albedo	; bolometric albedo
alb2=model2.albedo	; see alb1, convective envelopes: 0.5-1.0, radiative:1.0
rm=model2.mass/model1.mass
			; mass ratio
poth=model.semimajoraxis/(model1.diameter/2)
			; omega potential estimate = semi-major axis / radius
potc=model.semimajoraxis/(model2.diameter/2)
			; see poth, for second star
xbol1=0.4       	; coefficient in bolometric limb darkening law
xbol2=0.4		; see xbol1
ybol1=0.0		; coefficient for logarithmic term
ybol2=0.0		; see ybol1
dpclog=alog10(1000./modelpx(model.component))
			; logarithm (base 10) of distance (d) in parsecs
;
iband=model.wdband	; photometric band offered by lcdc, e.g. 7 for V-band
; hl,cl			; narrowband (5%) luminosities
l0=effwvls[iband-1]*1e-9; lamda [m]
lamda_min=l0/1.025
lamda_max=l0*1.025
nl=100
lamda=lamda_min+(lamda_max-lamda_min)*findgen(nl)/(nl-1)
hl=mean(stellarfluxes(model1,lamda,x1)) & x1=mean(x1)
cl=mean(stellarfluxes(model2,lamda,x2)) & x2=mean(x2)
if ipb eq 1 then begin
	tl=hl+cl
	cl=cl/hl
	hl=1.0
;	hl=hl/tl
;	cl=cl/tl
endif else begin
	hl=1.
	cl=1.
endelse
;
; Tests with parameters from Binary Stars, A Pictorial Atlas, Terrell et al.
; TU Cam, page 199, mode/wdtype=2
if 0 then begin
	poth=3.548
	potc=5.701
	rm=0.47
	pzero=2.9333
	xincl=77.6
; Beta Per, page 239, mode/wdtype=5 (Algols, secondary star fills roche lobe)
	poth=5.151
	potc=2.299
	rm=0.227
	pzero=2.867
	xincl=82.31
endif
;
y1=0.0			; see xh coefficient for logarithmic term
y2=0.0			; see yh, (unused if ld=1)
el3=0.0			; third light
opsf=0.			; opacity in a specific band, related to ISM
zero=0.0 		; zero point for output magnitudes
factor=1.0		; scaling factor for normalized light column
wl=l0*1e6		; wavelength in microns, only used for mpage=3
aextinc=0.		; ISM extinction in band
calib=1.		; flux calibration constant (cgs) for star of mag=0
;
lcparms[0] = mpage
lcparms[1] = nref
lcparms[2] = mref
lcparms[3] = ifsmv1
lcparms[4] = ifsmv2
lcparms[5] = icor1
lcparms[6] = icor2
lcparms[7] = ld1
lcparms[8] = mode
lcparms[9] = ipb
lcparms[10]= ifat1
lcparms[11]= ifat2
lcparms[12]= n1
lcparms[13]= n2
lcparms[14]= perr0
lcparms[15]= the
lcparms[16]= vunit
lcparms[17]= phn
lcparms[18]= phstrt
lcparms[19]= phstop
lcparms[20]= phin
lcparms[21]= e
lcparms[22]= a
lcparms[23]= f1
lcparms[24]= f2
lcparms[25]= vga
lcparms[26]= pshift
lcparms[27]= xincl
lcparms[28]= gr1
lcparms[29]= gr2
lcparms[30]= tavh
lcparms[31]= tavc
lcparms[32]= alb1
lcparms[33]= alb2
lcparms[34]= poth
lcparms[35]= potc
lcparms[36]= rm
lcparms[37]= xbol1
lcparms[38]= xbol2
lcparms[39]= ybol1
lcparms[40]= ybol2
lcparms[41]= wl
lcparms[42]= hl
lcparms[43]= cl
lcparms[44]= x1
lcparms[45]= x2
lcparms[46]= y1
lcparms[47]= y2
lcparms[48]= el3
lcparms[49]= zero
lcparms[50]= factor
lcparms[51]= nsp1
lcparms[52]= nsp2
lcparms[53]= if3b
lcparms[54]= ld2
lcparms[55]= kspev
lcparms[56]= kspot
lcparms[57]= nomax
lcparms[58]= ifcgs
lcparms[59]= ktstep
lcparms[60]= jdphs
lcparms[61]= hjd0
lcparms[62]= pzero
lcparms[63]= dpdt
lcparms[64]= delph
lcparms[65]= nga
lcparms[66]= stdev
lcparms[67]= noise
lcparms[68]= seed
lcparms[69]= hjdst
lcparms[70]= hjdsp
lcparms[71]= hjdin
lcparms[72]= phobs
lcparms[73]= lsp
lcparms[74]= tobs
lcparms[75]= tavsp
lcparms[76]= dperdt
lcparms[77]= abunin
lcparms[78]= fspot1
lcparms[79]= fspot2
lcparms[80]= dpclog
lcparms[81]= iband
lcparms[82]= opsf
lcparms[83]= aextinc
lcparms[84]= calib
;
; Use LCDC to compute Omega potentials which match the diameters.
; Diameters of components constrained by the model (e.g. Algols)
; are not checked as the input diameter is ignored in these cases.
pothc=lcpotentials(lcparms)
lcparms[34]= pothc[0]
lcparms[35]= pothc[1]
;lcparms(34)= 5.033
;lcparms(35)= 4.281
;
return,lcparms
;
end
