pro solveastrom
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Solve for new station positions and/or star positions. Input data are
; station based delays. Because of that, the solution for any station is
; independent from the others. The delays for the reference station are
; zero, so that the corrections for this station will also be zero.
; For coefficients, check with Hummel et al. 1994, AJ 108, 326. Coefficient
; for the delay offset is negative.
;
; The coordinates in GenConfig are updated, as well as those in the
; stationtable. However, the constant term in the stationtable, corresponding
; to a path to a common reference in the lab, is incrementally updated
; instead of copying the value from GenConfig.
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([8,9]) ne 0 then return
;
; Select all scans with valid time stamps
index=where(scans.time gt 0 and scans.starid ne 'FKV0000',count)
index=where(scans.time gt 0,count)
if count eq 0 then begin
	print,'***Error(SOLVEASTROM): no valid stellar scans found!'
	return
endif
;
rad=180/pi_circle
ha=scans[index].ha*(15/rad)
dec=scans[index].dec/rad
starid=scans[index].starid
geodelay=scans[index].GeoDelay
;
; Select the observed delays
case fit_data of
	'FDL':begin
	      obsdelay=scans[index].FDLPos
	      obsdelay[GenConfig.RefStation-1,*]=0
	      obsdelayerr=scans[index].FDLPosErr
	      end
	'GRP':begin
	      obsdelay=scans[index].GrpDelay
	      obsdelayerr=scans[index].GrpDelayErr
	      end
	'DRY':begin
	      obsdelay=scans[index].DryDelay
	      obsdelayerr=scans[index].DryDelayErr
	      end
	'WET':begin
	      obsdelay=scans[index].WetDelay
	      obsdelayerr=scans[index].WetDelayErr
	      end
endcase
;
; Make sure data with a bad reference are not used. Change reference if needed.
for i=0,GenConfig.NumSid-1 do $
obsdelayerr(i,*)=abs(obsdelayerr[i,*]) $
	     *signof(obsdelayerr[i,*],obsdelayerr[GenConfig.RefStation-1,*])
;
; Check station and star selection
nstn=n_elements(fit_stations)
nstr=n_elements(fit_stars)
if nstn gt 0 then if fit_stations[0] eq '' then nstn=0
if nstr gt 0 then if fit_stars[0] eq '' then nstr=0
;
; Remove the white light source from list
if nstr gt 0 then begin
	index= $
	where(fit_stars ne 'FKV0000',nstr)
	if nstr gt 0 then fit_stars0=fit_stars[index]
endif
;
; Remove reference station from list, if included
if nstn gt 0 then begin
	index= $
	where(fit_stations ne GenConfig.StationId[GenConfig.RefStation-1],nstn)
	if nstn gt 0 then fit_stations0=fit_stations[index]
endif
if nstn+nstr eq 0 then begin
	print,'***Error(SOLVEASTROM): no fits selected!'
	return
endif
;
scan_index=indgen(n_elements(starid))
if nstr gt 0 then begin
	id=0
	for i=0,nstr-1 do begin
		index=where(starid eq fit_stars0[i],count)
		if count gt 0 then scan_index[id:id+count-1]=index
		id=id+count
	endfor
	if id gt 0 then scan_index=scan_index[0:id-1] else begin
		print,'***Error(SOLVEASTROM): no data for selected stars!'
		return
	endelse
endif
ch=cos(ha[scan_index])
sh=sin(ha[scan_index])
cd=cos(dec[scan_index])
sd=sin(dec[scan_index])
starid=starid[scan_index]
index=where(starid eq 'FKV0000',count)
if count gt 0 then begin
	ch[index]=0
	cd[index]=0
	sh[index]=0
	sd[index]=0
endif
geodelay=geodelay[*,scan_index]
obsdelay=obsdelay[*,scan_index]
obsdelayerr=obsdelayerr[*,scan_index]
;
; Determine maximum number of rows
index=where(obsdelayerr gt 0,nrow)
;
; Add one row for removal of RA degeneracy
if nstr gt 0 then nrow=nrow+1
;
; Determine number of columns
ncol=nstn*4+nstr*2
;
; Allocate arrays
m=make_array(nrow,ncol,/double)
r=make_array(nrow,/double)
s=make_array(nrow,/string)
;
; Set up design matrix
id=0
for i=0,GenConfig.NumSid-1 do begin
    index_e=where(obsdelayerr[i,*] gt 0,count_e)
    if count_e gt 0 and i ne GenConfig.RefStation-1 then begin
    wt=1/obsdelayerr[i,index_e]
    r[id:id+count_e-1]=(obsdelay[i,index_e]-geodelay[i,index_e])*wt
    s[id:id+count_e-1]=starid[index_e]
    count=0
    if nstn gt 0 then index=where(fit_stations0 eq GenConfig.StationId[i],count)
    if count ne 0 then begin
		m[id:id+count_e-1,index*4+0]=+cd[index_e]*ch[index_e]*wt; X
		m[id:id+count_e-1,index*4+1]=-cd[index_e]*sh[index_e]*wt; Y
		m[id:id+count_e-1,index*4+2]=+sd[index_e]*wt		; Z
		m[id:id+count_e-1,index*4+3]=-wt			; D
    endif
;
;   Transform horizon station coordinates to equatorial
    StationCoord= $
	horizon2equatorial(GenConfig.StationCoord[*,i] $
			  -GenConfig.StationCoord[*,GenConfig.RefStation-1])
    for j=0,nstr-1 do begin
	index=where(obsdelayerr[i,index_e] gt 0 $
		and starid[index_e] eq fit_stars0[j],count)
	if count gt 0 then begin
;		Right ascension
		m[id+index,nstn*4+0+j*2]=wt[index]*( $
		   +StationCoord[1]*cd[index_e[index]]*ch[index_e[index]] $
		   +StationCoord[0]*cd[index_e[index]]*sh[index_e[index]])
;		Declination
		m[id+index,nstn*4+1+j*2]=wt[index]*( $
		   -StationCoord[0]*sd[index_e[index]]*ch[index_e[index]] $
		   +StationCoord[1]*sd[index_e[index]]*sh[index_e[index]] $
		   +StationCoord[2]*cd[index_e[index]])
	endif
    endfor
    id=id+count_e
    endif
endfor
;
; Fix the right ascension degeneracy if both stations and stars are fitted
; The large weighting factor forces the solution.
index=where(obsdelayerr gt 0)
wscale=1/median(obsdelayerr[index])
if nstr gt 0 and nstn gt 0 then $
	m[nrow-1,nstn*4]=wscale $
		*max(abs(genconfig.stationcoord[0:3,0:genconfig.numsid-1]))
;
; Remove zero rows in the design matrix
index=where(avg(abs(m),1) ne 0,count)
if count gt 0 then begin
	m=m[index,*]
	r=r[index]
	s=s[index]
endif
;
; Remove data from white light source, or, if only this one, solve for C only
stars=unique(s)
if n_elements(stars) eq 1 and stars[0] eq 'FKV0000' then begin
	do_c_only=1
	nstr=0
	m=m[*,indgen(nstn)*4+3]
endif else begin
	do_c_only=0
;	index=where(s ne 'FKV0000')
;	Do not remove the data, it was conditioned to constrain the const.term
;	m=m(index,*)
;	r=r(index)
;	s=s(index)
endelse
;
; Enough data?
nrow=n_elements(m[*,0])
ncol=n_elements(m[0,*])
if nrow lt ncol then begin
	print,'***Error(SOLVEASTROM): not enough data!'
	return
endif
;
t=transpose(m)
n=t#m
y=t#r
;
if n_elements(n) eq 1 then begin
	s=y/n
endif else begin
	svd,n/max(n),w,u,v		; Singular value decomposition
; 	print,'Eigenvalues (normalized): ',w/max(w)
	small=where(w lt max(w)*1.0e-8,count)
	if count gt 0 then begin
;	print,'SVD: will edit',count,' singular values!'
		print,'***Error(SOLVEASTROM): singular matrix!'
		return
		w[small]=0
	endif
;	svbksb,u,w,v,y,s	; SVD solution
;
	in=invert(n,status)
	s=in#y			; Direct solution, is more precise
endelse
;
print,'Solution computed.'
;
; Update the station coordinates, also copy the ref. station values
if n_elements(stationtable) eq 0 then get_stationtable,update=0
for i=0,nstn do begin
	if i eq nstn then j=GenConfig.RefStation-1 $
		     else j=long(where(GenConfig.StationId $
				eq fit_stations0[i]),0)
	ct0=GenConfig.StationCoord[3,j]
	if do_c_only then begin
	GenConfig.StationCoord[3,j]=GenConfig.StationCoord[3,j] $
				   +s[i]
	endif else if i lt nstn then begin
	GenConfig.StationCoord[*,j]=GenConfig.StationCoord[*,j] $
				   +equatorial2horizon(s[i*4:(i+1)*4-1])
	endif
	ct1=GenConfig.StationCoord[3,j]
;	Update stations in stationtable too
	if i eq nstn $
	then k=long(where(stationtable.stationid $
			eq genconfig.stationid[genconfig.refstation-1])) $
	else k=long(where(stationtable.stationid eq fit_stations0[i]),0)
	stationtable[k].x=genconfig.stationcoord[0,j]
	stationtable[k].y=genconfig.stationcoord[1,j]
	stationtable[k].z=genconfig.stationcoord[2,j]
	stationtable[k].d=stationtable[k].d+(ct1-ct0)
endfor
;
; For single configuration fit, store config info
for i=0,GenConfig.NumSid-1 do begin
	j=where(stationtable.stationid eq genconfig.stationid[i]) & j=j[0]
	stationtable[j].DL_ID=genconfig.delaylineid[i]
	stationtable[j].BC_IN=genconfig.bcinputid[i]
endfor
if nstn gt 0 then print,'GenConfig.StationCoord updated.'
;
; Update the star positions
for i=0,nstr-1 do begin
	j=long(where(StarTable.starid eq fit_stars0[i]),0)
	StarTable[j].ra  =StarTable[j].ra +s[nstn*4+0+i*2]*RAD/15
	StarTable[j].rae =sqrt(in[nstn*4+0+i*2,nstn*4+0+i*2])*RAD/15
	StarTable[j].dec =StarTable[j].dec+s[nstn*4+1+i*2]*RAD
	StarTable[j].dece=sqrt(in[nstn*4+1+i*2,nstn*4+1+i*2])*RAD
endfor
if nstr gt 0 then print,'StarTable updated.'
;
; Update the astrometry
calcastrom	; for scandata
calcgeo		; for pointdata
;
end
