pro solvevolvox
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Solve global astrometry using sparse matrix Cholesky decomposition.
; To read about sparse matrix technology, see book with same title by
; Sergio Pissanetsky, 1984, Academic Press Inc., London
;
common FitInfo,fit_stations,fit_stars,fit_data,fit_nights,fit_parms
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([12,13]) ne 0 then return
;
; Check station and star selection
nstn=n_elements(fit_stations)
nstr=n_elements(fit_stars)
if nstn gt 0 then if fit_stations[0] eq '' then nstn=0
if nstr gt 0 then if fit_stars[0] eq '' then nstr=0
;
; Sanity check
if nstr+nstn eq 0 then begin
	print,'***Error(SOLVEVOLVOX): no fits were selected!'
	return
endif
;
RAD=180/pi_circle
TOL=1e-8
;
; Set nominal dimensions and allocate sparse matrix arrays
nast=2L				; parms/star
nbnd=nstr*nast			; width of border band
nstp=4				; station coordinate parameters
narc=nstp*nstn			; arc parms/night
nnit=n_elements(fit_nights)	; number of nights
nrow=nnit*narc+nbnd		; Full dimension of square matrix
;
numr=nrow-nbnd
numv=nnit*((narc*(narc+1))/2+narc*nbnd)
numt=(nbnd*(nbnd+1))/2
vd=dblarr(numv+numt)		; row-wise sparse part of matrix
jv=uintarr(numv+numt)		; column numbers in vd, beginning with 1
iv=ulonarr(nrow+1)		; where row i begins in vd, beginning with 1
ni=uintarr(nnit)		; dimensions of sub-matrices
rd=dblarr(nrow)			; right hand side
;
; Number of observations per star
if nstr gt 0 then sc=intarr(nstr)
;
ns=0U				; sub-matrix counter
r0=0L				; rd counter
n0=0L				; vd counter
;
nsolved=intarr(nnit)		; was this night included? (1=Yes)
nzcount=intarr(nnit)		; number of non-zero border band columns
;
FOR night=0,nnit-1 DO BEGIN
;
print,'Now processing ',fit_nights[night],'...'
;
loadnight,fit_nights[night]
calcastrom
;
; Select all scans with valid time stamps
index=where(scans.time gt 0 and scans.starid ne 'FKV0000',count)
if count eq 0 then begin
	print,'Warning(SOLVEVOLVOX): no valid data on ',fit_nights[night],'!'
	goto,SKIP
endif
ha=scans[index].ha*(15/rad)
dec=scans[index].dec/rad
starid=scans[index].starid
geodelay=scans[index].GeoDelay
case fit_data of
	'FDL':begin
	      obsdelay=scans[index].FDLPos
	      obsdelay[GenConfig.RefStation-1,*]=0
	      obsdelayerr=scans[index].FDLPosErr
	      end
	'GRP':begin
	      obsdelay=scans[index].GrpDelay
	      obsdelayerr=scans[index].GrpDelayErr
	      end
	'DRY':begin
	      obsdelay=scans[index].DryDelay
	      obsdelayerr=scans[index].DryDelayErr
	      end
	'WET':begin
	      obsdelay=scans[index].WetDelay
	      obsdelayerr=scans[index].WetDelayErr
	      end
endcase
;
; Determine which fit stations are to be included
nstn0=0
if nstn gt 0 then begin
	stn_index=intarr(nstn)
	for i=0,GenConfig.NumSid-1 do begin
		if i ne GenConfig.RefStation-1 then begin
		   index=where(fit_stations eq GenConfig.StationId[i],count)
		   if count gt 0 then stn_index[index]=1
		endif
	endfor
	index=where(stn_index eq 1,nstn0)
	if nstn0 gt 0 then fit_stations0=fit_stations[index]
endif
;
; Extract data for stars if selected
scan_index=indgen(n_elements(starid))
if nstr gt 0 then begin
	id=0
	for i=0,nstr-1 do begin
		index=where(starid eq fit_stars[i],count)
		if count gt 0 then scan_index[id:id+count-1]=index
		id=id+count
	endfor
	if id gt 0 then scan_index=scan_index[0:id-1] else begin
		print,'Warning(SOLVEVOLVOX): no data for stars on ', $
				fit_nights[night],'!'
		goto,SKIP
	endelse
endif
ch=cos(ha[scan_index])
sh=sin(ha[scan_index])
cd=cos(dec[scan_index])
sd=sin(dec[scan_index])
starid=starid[scan_index]
geodelay=geodelay[*,scan_index]
obsdelay=obsdelay[*,scan_index]
obsdelayerr=obsdelayerr[*,scan_index]
;
; Determine maximum number of rows
index=where(obsdelayerr gt 0,nr)
;
; Determine number of columns
nc=nstn0*nstp+nstr*nast
;
; Allocate design matrix for this night
m=make_array(nr,nc,/double)
r=make_array(nr,/double)
;
; Set up design matrix
id=0
for i=0,GenConfig.NumSid-1 do begin
    index_e=where(obsdelayerr[i,*] gt 0,count_e)
    if count_e gt 0 and i ne GenConfig.RefStation-1 then begin
    wt=1/obsdelayerr[i,index_e]
    r[id:id+count_e-1]=(obsdelay[i,index_e]-geodelay[i,index_e])*wt
    count=0
    if nstn0 gt 0 then index=where(fit_stations0 eq GenConfig.StationId[i],count
    if count ne 0 then begin
		m[id:id+count_e-1,index*4+0]=+cd[index_e]*ch[index_e]*wt; X
		m[id:id+count_e-1,index*4+1]=-cd[index_e]*sh[index_e]*wt; Y
		m[id:id+count_e-1,index*4+2]=+sd[index_e]*wt		; Z
		m[id:id+count_e-1,index*4+3]=-wt			; D
    endif
;
;   Transform horizon station coordinates to equatorial
    StationCoord= $
	horizon2equatorial(GenConfig.StationCoord[*,i] $
			  -GenConfig.StationCoord[*,GenConfig.RefStation-1])
    for j=0,nstr-1 do begin
	if j eq 0 then sc1=sc*0
	index=where(obsdelayerr[i,index_e] gt 0 $
		and starid[index_e] eq fit_stars[j],count)
	if count gt 0 then begin
		sc1[j]=count
;		Right ascension
		m[id+index,nstn0*nstp+0+j*nast]=wt[index]*( $
		   +StationCoord[1]*cd[index_e[index]]*ch[index_e[index]] $
		   +StationCoord[0]*cd[index_e[index]]*sh[index_e[index]])
;		Declination
		m[id+index,nstn0*nstp+1+j*nast]=wt[index]*( $
		   -StationCoord[0]*sd[index_e[index]]*ch[index_e[index]] $
		   +StationCoord[1]*sd[index_e[index]]*sh[index_e[index]] $
		   +StationCoord[2]*cd[index_e[index]])
	endif
    endfor
    id=id+count_e
    endif
endfor
;
; Remove zero rows in the design matrix
index=where(total(abs(m),2) ne 0)
m=m[index,*]
r=r[index]
;
; Remove zero columns in border band
if nstr gt 0 then begin
	nzindex=where(total(abs(m[*,nstn0*nstp:nc-1]),1) ne 0,nzc)
	index=nstn0*nstp+nzindex
	if nstn0 gt 0 then index=[indgen(nstn0*nstp),index]
	m=m[*,index]
	nc=nstn0*nstp+nzc
endif
;
; Normalized matrix
t=transpose(m)
n=t#m
y=t#r
;
; Check singularity of station coordinate solution
if nstn0 gt 0 then begin
	svd,n[0:nstn0*nstp-1,0:nstn0*nstp-1],w
	if min(w)/max(w) lt 1e-4 then goto,SKIP
endif
;
; Check singularity of full solution
svd,n,w
w=w[sort(w)]
if w[1]/w[n_elements(w)-1] lt 1e-6 then goto,SKIP
;
; This night is now part of the solution
nsolved[night]=1
ns=ns+1
ni[ns-1]=nstn0*nstp
;
; Fill in right hand side
if nstn0 gt 0 then begin
	rd[r0:r0+ni[ns-1]-1]=y[0:ni[ns-1]-1]
	r0=r0+ni[ns-1]
endif
if nstr gt 0 then begin
	nzcount[ns-1]=nzc
	sc=sc+sc1
	rd[numr+nzindex]=rd[numr+nzindex]+y[ni[ns-1]:nc-1]
endif
;
; Shift border columns
if nstr gt 0 then begin
	n1=0
	for l=0,ns-2 do begin
		for k=0,ni[l]-1 do begin
			n2=iv[n1+k]+ni[l]-k-1
			jv[n2:n2+nzcount[l]-1]=jv[n2:n2+nzcount[l]-1]+ni[ns-1]
		endfor
		n1=n1+ni[l]
	endfor
endif
;
; Fill in sparse border
n1=ni[ns-1]+nzcount[ns-1]
if ns gt 1 then n3=fix(total(ni[0:ns-2])) else n3=0
nl=-1
for k=0,ni[ns-1]-1 do begin
	iv[n3+k]=n0+1
	vd[n0:n0+n1-1]=n[k,k:k+n1-1]
	jvc=indgen(ni[ns-1]-k)+k
	if nstr gt 0 then jvc=[jvc,ni[ns-1]+nzindex]
	jv[n0:n0+n1-1]=jvc+1+n3
	nl=n0+n1-1
	n0=n0+n1
	n1=n1-1
endfor
;
; Fill corner matrix
for k=0,nzcount[ns-1]-1 do begin
	index=nzindex[k:nzcount[ns-1]-1]-nzindex[k] $
	     +nzindex[k]*nbnd-(nzindex[k]*(nzindex[k]-1))/2
	vd[numv+index]=vd[numv+index]+n[ni[ns-1]+k:nc-1,ni[ns-1]+k]
endfor
;
SKIP:
;
ENDFOR
;
; Remove degeneracies due to RA zero point and missing stars
print,'Removing known degeneracies...'
if nstr gt 0 then begin
	minobs=nast
	index=where(sc lt minobs,icount)
	jndex=where(sc ge minobs,jcount)
	cd=dblarr(nbnd)
	if jcount gt 0 then cd[jndex[0]*nast]=1
	if icount gt 0 then begin
		cd[index*nast]=1
		cd[index*nast+1]=1
	endif
	wscale=max(abs(vd[where(vd ne 0)]))
	k=lindgen(nbnd)
	vd[numv+k*nbnd-(k*(k-1))/2]=vd[numv+k*nbnd-(k*(k-1))/2]+cd*wscale
endif
;
; Finish sparse matrix setup
print,'Finishing sparse matrix...'
ni=ni[0:ns-1]
index=byte(rd)*0+1 & if numr gt r0 then index[r0:numr-1]=0
rd=rd[where(index eq 1)]
nm=uint(n_elements(rd))
;
index=byte(vd)*0+1 & if numv gt nl+1 then index[nl+1:numv-1]=0
vd=vd[where(index eq 1)]
;
n1=nbnd
n3=fix(total(ni))
n0=nl+1
for k=0,nbnd-1 do begin
	iv[n3+k]=n0+1
	jv[n0:n0+n1-1]=indgen(n1)+1+k+n3
	nl=n0+n1-1
	n0=n0+n1
	n1=n1-1
endfor
iv=[iv[0:nm-1],iv[nm-1]+1]
jv=jv[0:nl]
;
; Display TV image with sparse matrix
print,'Displaying sparse matrix...'
numpix=min([600,nm])
window,/free,xsize=numpix,ysize=numpix
image=bytarr(numpix,numpix)
for k=0,nm-1 do begin
        i1=iv[k]-1
        i2=iv[k+1]-2
	j1=nint(float(jv[i1:i2]-1)/nm*numpix)
	j2=nint((nm-1-(fltarr(i2-i1+1)+k))/nm*numpix)
        image[j1,j2]=1
endfor
if 0 then begin
for k=0,nbnd-1 do begin
	j1=nint((nm-findgen(nbnd-k)-1)/nm*numpix)
	j2=nint((nbnd-fltarr(nbnd-k)-1-k)/nm*numpix)
	image[j1,j2]=1
endfor
endif
tvscl,image
;
; Solve matrix using Hyper-Cholesky
if nstr eq 0 then ns=ns-1
if nstn eq 0 then ns=0
print,'Computing solution..., nm= ',nm,', ns= ',ns
vt=vd
hichol,vt,iv,jv,ni,ns,nm
sl=dblarr(nm)
rn=-rd
hisolve,vt,iv,jv,ni,ns,nm,rn,sl
;
; Improve solution by iteration
for iter=0,1 do begin
	hires,vd,iv,jv,rd,rn,sl,nm
	hisolve,vt,iv,jv,ni,ns,nm,rn,sl
endfor
;
; Compute smallest eigenvalue by inverse iteration
print,'Computing eigenvalues...'
rd[*]=1
s=dblarr(nm)
sum=0.d0
tiny=1d-3
i=0
niter=30
repeat begin
	ev=sum
	s[*]=0
	hisolve,vt,iv,jv,ni,ns,nm,rd,s
	sum=sqrt(total(s^2))
	rd[*]=s/sum
	i=i+1
endrep until abs(1-ev/sum) lt tiny or i eq niter
sev=1/sum
;
; Compute largest eigenvalue using the power method
u=dblarr(nm)+1
sum=0.d0
j=0
repeat begin
	ev=sum
	hires,vd,iv,jv,rd*0,rd,u,nm
	sum=sqrt(total(double(rd)^2))
	u[*]=rd/sum
	j=j+1
endrep until abs(1-ev/sum) lt tiny or j eq niter
lev=sum
;
; Check matrix condition
if i eq niter or j eq niter then begin
	print,'***Error(SOLVEVOLVOX): did not converge; return w/out app. sol.'
	return
endif
print,'Ratio largest/smallest eigenvalue= ',lev/sev
if sev/lev lt TOL then begin
	print,'***Error(SOLVEVOLVOX): matrix singular; return w/out app. sol.!'
	return
endif
;
; Calculate diagonal elements of inverse stellar parameter matrix
print,'Computing errors...'
if nstr gt 0 then begin
	se=fltarr(nbnd)
	for i=nm-nbnd,nm-1 do begin
		rd[*]=0 & rd[i]=1
		s[*]=0
		hisolve,vt,iv,jv,ni,ns,nm,rd,s
		se[i-nm+nbnd]=sqrt(rd[i])
	endfor
endif
;
; Apply solutions
;
; Update the star positions
for i=0,nstr-1 do begin
	j=long(where(StarTable.starid eq fit_stars[i]),0)
	StarTable[j].ra  =StarTable[j].ra +sl (nm-nstr*2+0+i*nast)*RAD/15
	StarTable[j].rae =se[i*nast+0]*RAD/15
	StarTable[j].dec =StarTable[j].dec+sl (nm-nstr*2+1+i*nast)*RAD
	StarTable[j].dece=se[i*nast+1]*RAD
endfor
if nstr gt 0 then print,'StarTable updated.'
;
ns=0
;
FOR night=0,nnit-1 DO BEGIN
;
loadnight,fit_nights[night]
;
if nsolved[night] then begin
;
; Update the station coordinates
if nstn gt 0 then begin
	stn_index=intarr(nstn)
	for i=0,GenConfig.NumSid-1 do begin
		if i ne GenConfig.RefStation-1 then begin
		index=where(fit_stations eq GenConfig.StationId[i],count)
		if count gt 0 then stn_index[index]=1
		endif
	endfor
	index=where(stn_index eq 1,nstn0)
	if nstn0 gt 0 then fit_stations0=fit_stations[index]
endif
for i=0,nstn0-1 do begin
	j=long(where(GenConfig.StationId eq fit_stations0[i]),0)
	GenConfig.StationCoord[*,j]=GenConfig.StationCoord[*,j] $
			   +equatorial2horizon(sl[ns+i*nstp:ns+(i+1)*nstp-1])
endfor
j=where(geninfo.date eq fit_nights[night])
; geninfo(j).stationcoord=genconfig.stationcoord
if nstn0 gt 0 then print,'GenConfig.StationCoord updated for ',fit_nights[night]
ns=ns+nstn0*nstp
;
; Update the astrometry
calcastrom
storenight,11
;
endif
;
ENDFOR
;
; Inform about unusable nights
if nstn gt 0 then begin
	index=where(nsolved eq 0,count)
	if count gt 0 then begin
		print,'These nights have not been used:'
		print,fit_nights[index]
	endif
endif
;
; Inform about stars not solved for
if nstr gt 0 then begin
	index=where(sc lt minobs,count)
	if count gt 0 then begin
		print,'These stars have not been solved for:'
		print,fit_stars[index]
	endif
endif
;
end
