function system_config,systemid,key,stations=stations
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; This function returns general site and system information for the given key.
; System configuration wavelengths are in meters, but grid wavelengths in nm.
;
; key='MIDNIGHT' calls this function 2nd time
;
forward_function system_config
;
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
common Pionier,pionier_spectral_cal
;
if n_elements(key) eq 0 then begin
	print,'Error(system_config): must also specify key!'
	return,''
endif
key=strupcase(key)
;
; This key creates a new GenConfig (not all systems implemented!)
if key eq 'CONFIG' then begin
	get_systemid,systemid	; store systemid in common block
	stations=strcompress(stations,/remove)
	sid_index=where(strlen(stations) gt 0,NumSid)
endif
;
case system_id(systemid) of
;
; Observatories
'OHP':begin	; Observatoire Haute Provence
	case key of
	'LONGITUDE': 	return,+5.7122D0
	'LATITUDE':  	return,43.9317D0
	'ALTITUDE':  	return,650.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'Ondrejov':begin	; Ondrejov Observatory
	case key of
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'CalarAlto':begin
	case key of
	'LONGITUDE': 	return,-2.546822D0
	'LATITUDE':  	return,37.220888D0
	'ALTITUDE':  	return,2150.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'LaSilla':begin
	case key of
	'LONGITUDE': 	return,-70.051187D0
	'LATITUDE':  	return,-29.255122D0
	'ALTITUDE':  	return,2375.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'Paranal':begin
	case key of
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'OCA':begin	; Observatorio Cerro Armazones
	case key of
	'LONGITUDE': 	return,-70.196389D0
	'LATITUDE':  	return,24.598056D0
	'ALTITUDE':  	return,2817.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
;
; Aperture masks
'JWST_SIM':begin	; The JWST aperture mask
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['P01','P02','P03','P04','P05','P06','P07']
	'DIAMETER':	return,9.468544e-1 ; Beauty Contest 2022, JSB
	'LONGITUDE': 	return,-70.7367D0
	'LATITUDE':  	return,-30.241D0
	'ALTITUDE':  	return,2722.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'GRID':		return,findgen(132)*40+11700-5240.0/2
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	endcase
	end
'T-ReCS':begin	; The TrecS aperture mask
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['A01','A02','A03','A04','A05','A06','A07']
	'DIAMETER':	return,1.5 ; John Monnier, email from 20.3.2005
	'LONGITUDE': 	return,-70.7367D0
	'LATITUDE':  	return,-30.241D0
	'ALTITUDE':  	return,2722.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'GRID':		return,findgen(132)*40+11700-5240.0/2
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	endcase
	end
'NACO':	begin	; The 7Holes SAM aperture mask, Ks and Lp bands
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['A01','A02','A03','A04','A05','A06','A07']
	'DIAMETER':	return,1.20	; 1.5 mm holes, pupil diameter 10 mm
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'GRID':		return,findgen(50)*80+1000
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	endcase
	end
;
; Long-baseline interferometers
'Mark3':begin
	case key of
	'EXIST': 	return,1
	'STATIONS': 	return,['NAS','SAS']
	'DIAMETER':	return,0.05
	'LONGITUDE': 	return,-118.05917D0
	'LATITUDE':  	return,34.216694D0
	'ALTITUDE':  	return,1742.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,35
	'GRID': 	return,findgen(250)*2+401
	'MAXCONFIG':	return,1
	'FILTERS':begin
		index=where(mark3_configs eq instrument_id(systemid))
		case index[0]+1 of
		1:return,[['generic_c','mark3_486'], $
			  ['mark3_486','mark3_486']]
		2:return,[['generic_c','mark3_550'], $
			  ['mark3_656','mark3_550']]
		3:return,[['generic_c','mark3_656'], $
			  ['mark3_657','mark3_550']]
		4:return,[['generic_c','mark3_450'], $
			  ['mark3_700','mark3_550']]
		5:return,[['generic_c','mark3_623'], $
			  ['mark3_754','mark3_712']]
		6:return,[['generic_c','mark3_450'], $
			  ['mark3_799','mark3_550']]
		7:return,[['generic_c','mark3_450'], $
			  ['mark3_800','mark3_500']]
		8:return,[['generic_c','mark3_450'], $
			  ['mark3_800','mark3_550']]
		9:return,[['generic_c','mark3_452'], $
			  ['mark3_800','mark3_500']]
	       10:return,[['generic_c','mark3_500'], $
			  ['mark3_800','mark3_550']]
	       11:return,[['generic_c','mark3_500'], $
			  ['mark3_800','mark3_656']]
	       12:return,[['generic_c','mark3_500'], $
			  ['mark3_800','mark3_550']]
	       13:return,[['generic_c','mark3_656'], $
			  ['mark3_800','mark3_550']]
		endcase
		return,'generic_c'
		case instrument_id(systemid) of
		'450':return,[['generic_c','mark3_450'], $
			      ['mark3_800','mark3_550']]
		'500':return,[['generic_c','mark3_500'], $
			      ['mark3_800','mark3_550']]
		endcase
		end
	endcase
	end
'PTI': 	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['NAS','SAS']
	'DIAMETER':	return,0.40
	'LONGITUDE': 	return,-116.8633D0
	'LATITUDE':  	return,33.3567D0
	'ALTITUDE':  	return,1687.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,100
	'GRID':		return,findgen(250)*2+2000	; [nm]
	'MAXCONFIG':	return,1
	'FILTERS':	begin
			case instrument_id(systemid) of
			'K1':return,'pti_k'
			'K5':return,['pti_k1','pti_k2', $
				     'pti_k3','pti_k4', $
				     'pti_k5']
			endcase
			end
	'CONFIG':	begin
		maxNumSid=2
		maxNumOutBeam=3
		maxNumTriple=1
		maxBaseline=1
		maxSpecChan=5
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
; 		Assume pair-wise beam combination
		case instrument_id(systemid) of
			'K1' :GenConfig.BeamCombinerId=1
			'K5' :GenConfig.BeamCombinerId=2
		endcase
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
		GenConfig.DelayLineId=[1,2]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId[0,0]=genconfig.stationid[0]+'-' $
					 +genconfig.stationid[1]
		GenConfig.NumBaseline[0]=1
;
; 		Remove the instrument string for backward compatibility
		SystemId='PTI'
;
		case GenConfig.BeamCombinerId of
;	  		Single channel K filter
		      1:begin
			GenConfig.NumSpecChan=1
			GenConfig.Wavelength=2.2e-6
			GenConfig.WavelengthErr=1e-9
			GenConfig.ChanWidth=0.03*GenConfig.Wavelength
			GenConfig.ChanWidthErr=1e-9
			end
		      2:begin
			GenConfig.NumSpecChan=5
			GenConfig.Wavelength=[2.009,2.106,2.203,2.299,2.396]*1e-6
			GenConfig.WavelengthErr=1e-9
			GenConfig.ChanWidth=0.097e-6
			GenConfig.ChanWidthErr=1e-9
			end
		endcase
;
; 		Set number of independent triples
		GenConfig.NumTriple=0
;
		return,GenConfig
		end
	endcase
	end
'IOTA':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['A00','B00','C00']
	'DIAMETER':	return,0.45
	'LONGITUDE': 	return,-110.8849D0
	'LATITUDE':  	return,31.6916D0
	'ALTITUDE':  	return,2564.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,65
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,90
	'GRID':		return,findgen(150)*2+1500; [nm], H-band, 1.51-1.79 mu
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	'CONFIG':begin
		BeamCombinerId=1
;
		maxNumSid=3
		maxNumOutBeam=1
		maxNumTriple=1
		maxBaseline=3
		maxSpecChan=1
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
		case instrument_id(systemid) of
			'IONIC3' :GenConfig.BeamCombinerId=1
			'IONIC'  :GenConfig.BeamCombinerId=2
		endcase
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		if GenConfig.NumSid ne 3 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 3!'
			return,-1
		endif
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,2,3]
		GenConfig.BaselineId[0,0]=genconfig.stationid[0]+'-' $
				         +genconfig.stationid[1]
		GenConfig.BaselineId[1,0]=genconfig.stationid[1]+'-' $
				         +genconfig.stationid[2]
		GenConfig.BaselineId[2,0]=genconfig.stationid[2]+'-' $
				         +genconfig.stationid[0]
		GenConfig.NumBaseline=3
;
		NumChan=1; must be less than or equal to maxSpecChannel
		for ob=0,GenConfig.NumOutBeam-1 do begin
			GenConfig.NumSpecChan[ob]=NumChan
			GenConfig.Wavelength[*,ob]=1.65e-6
			GenConfig.WavelengthErr[*,ob]=1e-9
			GenConfig.ChanWidth[*,ob]=0.03*GenConfig.Wavelength[*,ob]
			GenConfig.ChanWidthErr[*,ob]=1e-9
		endfor
;		Triple
		GenConfig.NumTriple=1
		GenConfig.TripleBeam=[0,1,2]
		GenConfig.TripleBase=[0,0,0]
		GenConfig.TripleNumChan[0]=NumChan
		for bl=0,2 do begin
		GenConfig.TripleChan[0:NumChan-1,bl,0]=indgen(NumChan)
		endfor
;
; 		Remove the instrument string for backward compatibility
		SystemId='IOTA'
;
		return,GenConfig
		end
	endcase
	end
'COAST':begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['CC0','CW2','CE1','CN4']
	'DIAMETER':	return,0.40
	'LONGITUDE': 	return,-(00.D0+02.D0/60+31.96D0/3600)
	'LATITUDE':  	return,  52.D0+09.D0/60+49.96D0/3600
	'ALTITUDE':  	return,17.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,115
	'GRID':		return,findgen(400)+630	; [nm]
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	'CONFIG':begin
; 	Pair-wise combination of 4 stations for simulation purposes
;
		NumOutBeam=(NumSid*(NumSid-1))/2
		NumTriple=(NumSid-1)*(NumSid-2)/2
		NumBaseline=1
		NumSpecChan=71
		maxSpecChan=NumSpecChan
		maxConfig=1
		GenConfig=allocGenConfig(NumSid,NumOutBeam,NumTriple, $
					 NumBaseline,maxSpecChan,maxConfig)
;
		GenConfig.BeamCombinerId=1
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		GenConfig.NumSid=NumSid
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumSpecChan
		GenConfig.TripleNumChan=NumSpecChan
		for i=0,NumOutBeam-1 do $
		GenConfig.Wavelength[*,i]=(findgen(NumSpecChan)*5+650)*1e-9
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=5e-9
		GenConfig.ChanWidthErr=1e-9
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		ob=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		GenConfig.BaselineId[0,ob]=genconfig.stationid[i]+'-' $
					   +genconfig.stationid[j]
		ob=ob+1
		endfor
		endfor
;
		tr=0
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		ob1=where(GenConfig.BaselineId[0,*] eq genconfig.stationid[0]+'-' $
						      +genconfig.stationid[i])
		ob2=where(GenConfig.BaselineId[0,*] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[j])
		ob3=where(GenConfig.BaselineId[0,*] eq genconfig.stationid[0]+'-' $
						      +genconfig.stationid[j])
		GenConfig.TripleBeam[*,tr]=[ob1,ob2,ob3]
		for l=0,2 do begin
			GenConfig.TripleChan[*,l,tr]=indgen(NumSpecChan)
			GenConfig.TripleChan[*,l,tr]=indgen(NumSpecChan)
		endfor
		tr=tr+1
		endfor
		endfor
;
		return,GenConfig
		end
	endcase
	end
'NPOI':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['AC0','AE0','AW0','AN0']
	'DIAMETER':	return,0.12
	'LONGITUDE': 	return,-(111.D0+32.D0/60+ 6.D0/3600)
	'LATITUDE':  	return,   35.D0+ 5.D0/60+48.D0/3600
	'ALTITUDE':  	return,2200.66D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,35
	'GRID': begin
		case instrument_id(systemid) of
		'3way':return,findgen(250)*2+401
		'6way':return,findgen(250)*2+401
		'nway':return,findgen(250)*2+401
		'else':return,findgen(250)*2+401
		endcase
		end
	'FILTERS':begin
		return,'generic_c'
		end
	'MAXCONFIG':begin
		case instrument_id(systemid) of
		'3way':return,'stars'
		'nway':return,'stars'
		  else:return,'scans'
		endcase
		end
	'CONFIG':begin
;
; 		Set beam combiner ID, which is identical to instrument ID
		case instrument_id(systemid) of
			'3way' :BeamCombinerId=1
			'6way' :BeamCombinerId=2
			'nway' :BeamCombinerId=9
			else:   BeamCombinerId=1
		endcase
;
; 		Remove the instrument string for backward compatibility
		SystemId='NPOI'
;
		case BeamCombinerId of
;
;		3way
	      1:begin
		maxNumSid=3
		maxNumOutBeam=3
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)/2
		maxBaseline=1
		maxSpecChan=32
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
		GenConfig.BeamCombinerId=BeamCombinerId
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		if GenConfig.NumSid ne 3 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 3!'
			return,-1
		endif
;
		NumOutBeam=3
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,2,3]
		GenConfig.BaselineId[0,0]=genconfig.stationid[0]+'-' $
				         +genconfig.stationid[1]
		GenConfig.BaselineId[0,1]=genconfig.stationid[1]+'-' $
				         +genconfig.stationid[2]
		GenConfig.BaselineId[0,2]=genconfig.stationid[2]+'-' $
				         +genconfig.stationid[0]
		GenConfig.NumBaseline=1
;
		NumChan=32; must be less than or equal to maxSpecChannel
		for ob=0,GenConfig.NumOutBeam-1 do begin
			GenConfig.NumSpecChan[ob]=NumChan
			GenConfig.Wavelength[*,ob]= $
			1e-9/((1./850)+findgen(GenConfig.NumSpecChan[ob])*4e-5)
			GenConfig.WavelengthErr[*,ob]=1e-9
			GenConfig.ChanWidth[*,ob]=0.03*GenConfig.Wavelength[*,ob]
			GenConfig.ChanWidthErr[*,ob]=1e-9
		endfor
;		Triple
		GenConfig.NumTriple=1
		GenConfig.TripleBeam=[0,1,2]
		GenConfig.TripleBase=[0,0,0]
		GenConfig.TripleNumChan[0]=NumChan
		for bl=0,2 do begin
		GenConfig.TripleChan[0:NumChan-1,bl,0]=indgen(NumChan)
		endfor
;
		return,GenConfig
		end
;
;		6way
	      2:begin
;
		maxNumSid=6
		maxNumOutBeam=3
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)/2
		maxBaseline=6	; 4 stations per output beam
		maxSpecChan=16
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
		GenConfig.BeamCombinerId=BeamCombinerId
		GenConfig.NumBin=64
		strokes=1e-6*[-1,4,-3,3,-4,2]
; 		3-way combination sequence for spectrometers 1-3
		outputs=[[2,3],[1,3],[1,2]]
; 		The 6-way combiner is two 3-way combiners
		stations6=strarr(6)
		stations6[0:n_elements(stations)-1]=stations
		doublet=reform(stations6,3,2)
; 		Each spectrometer has these inputs
		quad=strarr(4,3)
		for i=0,2 do quad[*,i]=[doublet[outputs[*,i]-1,0], $
					doublet[outputs[*,i]-1,1]]
;
		for ob=0,3-1 do begin
		bl=0
		for s1=0,4-2 do begin
		for s2=s1+1,4-1 do begin
		if strlen(quad[s1,ob]) ne 0 and strlen(quad[s2,ob]) ne 0 $
			then begin
			m=where(stations eq quad[s1,ob]) & m=m[0]
			n=where(stations eq quad[s2,ob]) & n=n[0]
			k=nint((strokes[m]-strokes[n])/1e-6)
			if k gt 0 then $
			GenConfig.BaselineId[bl,ob]=quad[s1,ob]+'-'+quad[s2,ob]
			if k lt 0 then $
			GenConfig.BaselineId[bl,ob]=quad[s2,ob]+'-'+quad[s1,ob]
			GenConfig.FringeMod[bl,ob]=abs(k)
			bl=bl+1
		endif
		endfor
		endfor
		GenConfig.NumBaseline[ob]=bl
		endfor
;
		GenConfig.NumOutBeam=3
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.Stroke[0:NumSid-1]=strokes[sid_index]
		GenConfig.RefStation=1
		GenConfig.DelayLineId[0:NumSid-1]=sid_index+1
		GenConfig.BCInputId[0:NumSid-1]=sid_index+1
;
		for ob=0,GenConfig.NumOutBeam-1 do begin
			GenConfig.NumSpecChan[ob]=maxSpecChan
			GenConfig.Wavelength[*,ob]= $
			1e-9/((1./850)+findgen(GenConfig.NumSpecChan[ob])*4e-5)
			GenConfig.WavelengthErr[*,ob]=1e-9
			GenConfig.ChanWidth[*,ob]=0.03*GenConfig.Wavelength[*,ob]
			GenConfig.ChanWidthErr[*,ob]=1e-9
		endfor
;
; 		Get space for (n-1)(n-2)/2 independent triples
		GenConfig.NumTriple=(NumSid-1)*(NumSid-2)/2
		TripleBeam=lonarr(3,maxNumTriple)
		TripleBase=lonarr(3,maxNumTriple)
		TripleChan=lonarr(maxSpecChan,3,maxNumTriple)
		TripleNumChan=lonarr(maxNumTriple)
		tr=0
		s1=0
		FOR s2=1,NumSid-2 DO BEGIN
		FOR s3=s2+1,NumSid-1 DO BEGIN
;	 	This is the first baseline in the triple
		for ob=0,GenConfig.NumOutBeam-1 do begin
		for bl=0,GenConfig.NumBaseline[ob]-1 do begin
			if strpos(GenConfig.BaselineId[bl,ob], $
				GenConfig.StationId[s1]) ne -1 $
		        and strpos(GenConfig.BaselineId[bl,ob], $
		       		GenConfig.StationId[s2]) ne -1 $
			then begin
				TripleBeam[0,tr]=ob
				TripleBase[0,tr]=bl
				TripleChan[*,0,tr]=indgen(maxSpecChan)
			endif
		endfor
		endfor
;	 	This is the second baseline in the triple
		for ob=0,GenConfig.NumOutBeam-1 do begin
		for bl=0,GenConfig.NumBaseline[ob]-1 do begin
			if strpos(GenConfig.BaselineId[bl,ob], $
				GenConfig.StationId[s1]) ne -1 $
		        and strpos(GenConfig.BaselineId[bl,ob], $
				GenConfig.StationId[s3]) ne -1 $
		      	then begin
				TripleBeam[1,tr]=ob
				TripleBase[1,tr]=bl
				TripleChan[*,1,tr]=indgen(maxSpecChan)
			endif
		endfor
		endfor
;	 	This is the third baseline in the triple
		for ob=0,GenConfig.NumOutBeam-1 do begin
		for bl=0,GenConfig.NumBaseline[ob]-1 do begin
			if strpos(GenConfig.BaselineId[bl,ob], $
				GenConfig.StationId[s2]) ne -1 $
		       and strpos(GenConfig.BaselineId[bl,ob], $
		       		GenConfig.StationId[s3]) ne -1 $
		      then begin
				TripleBeam[2,tr]=ob
				TripleBase[2,tr]=bl
				TripleChan[*,2,tr]=indgen(maxSpecChan)
			endif
		endfor
		endfor
		TripleNumChan[tr]=maxSpecChan
		tr=tr+1
		ENDFOR
		ENDFOR
;
		GenConfig.TripleBeam=TripleBeam
		GenConfig.TripleBase=Triplebase
		GenConfig.TripleChan=TripleChan
		GenConfig.TripleNumChan=TripleNumChan
;
		return,GenConfig
		end
;
; 		Pair-wise combination of 6 stations for simulation purposes
	     9:	begin
;
		NumOutBeam=(NumSid*(NumSid-1))/2
		NumTriple=(NumSid-1)*(NumSid-2)/2
		NumBaseline=1
		NumSpecChan=1
		maxSpecChan=2
		maxConfig=1
		GenConfig=allocGenConfig(NumSid,NumOutBeam,NumTriple, $
					 NumBaseline,maxSpecChan,maxConfig)
;
		GenConfig.BeamCombinerId=9
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		GenConfig.NumSid=NumSid
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumSpecChan
		GenConfig.TripleNumChan=NumSpecChan
		GenConfig.Wavelength=550e-9
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=20e-9
		GenConfig.ChanWidthErr=1e-9
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		ob=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		GenConfig.BaselineId[0,ob]=genconfig.stationid[i]+'-' $
					   +genconfig.stationid[j]
		ob=ob+1
		if i eq 0 then begin
		endif
		endfor
		endfor
;
		tr=0
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		ob1=where(GenConfig.BaselineId[0,*] eq $
			genconfig.stationid[0]+'-'+genconfig.stationid[i])
		ob2=where(GenConfig.BaselineId[0,*] eq $
			genconfig.stationid[i]+'-'+genconfig.stationid[j])
		ob3=where(GenConfig.BaselineId[0,*] eq $
			genconfig.stationid[0]+'-'+genconfig.stationid[j])
		GenConfig.TripleBeam[*,tr]=[ob1,ob2,ob3]
		tr=tr+1
		endfor
		endfor
;
		return,GenConfig
		end
;
		endcase
		end
	endcase
	end
'CHARA':begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['S1_','S2_','E1_','E2_','W1_','W2_']
	'DIAMETER':	return,1.0
	'LONGITUDE': 	return,-118.0570D0
	'LATITUDE':  	return,34.2244D0
	'ALTITUDE':  	return,1725.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,80
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,400
;	'GRID':		return,findgen(400)*4+1000
	'GRID':		return,'SPECTROMETER'
	'MAXCONFIG':	return,1
	'FILTERS':	begin
			return,'generic_c'	; we don't have filter curves yet
			case instrument_id(systemid) of
			'CLASSIC':return,'kprime'
			'CLIMB'  :return,'kprime'
			     else:return,'generic_c'
			endcase
			end
	'CONFIG':	begin
;
		maxNumSid=6
		maxNumOutBeam=2
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)	; H+K combined
		maxBaseline=(maxNumSid*(maxNumSid-1))/2
		maxSpecChan=500
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
		GenConfig.BeamCombinerId=beamcombiner_id(systemid)
;
		case GenConfig.BeamCombinerId/10 of
;
;		CLASSIC
		1:begin
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.NumSpecChan[0]=1
		case instrument_id(systemid) of
			'CLASSIC':GenConfig.Wavelength=2.13e-6
			     else:GenConfig.Wavelength=2.13e-6
		endcase
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=0.35e-6
		GenConfig.ChanWidthErr=1e-9
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.NumBaseline=(NumSid*(NumSid-1))/2
		GenConfig.RefStation=1
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		GenConfig.BaselineId[bl,*]=genconfig.stationid[i]+'-' $
					  +genconfig.stationid[j]
		bl=bl+1
		endfor
		endfor
;
; 		Set number of independent triples
		GenConfig.NumTriple=0
;
		end
;
;		CLIMB
	     	2:begin
;
		NumOutBeam=2
		GenConfig.NumOutBeam=NumOutBeam
;
		NumChan=1
		GenConfig.NumSpecChan=NumChan
		GenConfig.Wavelength[0,*]=[1.65,2.13]*1e-6
		GenConfig.WavelengthErr[0,*]=1e-9
		GenConfig.ChanWidth[0,*]=0.03*GenConfig.Wavelength[0,*]
		GenConfig.ChanWidthErr[0,*]=1e-9
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.NumBaseline=3
		GenConfig.RefStation=1
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,2,3]
		GenConfig.BaselineId[0,*]=genconfig.stationid[0]+'-' $
				         +genconfig.stationid[1]
		GenConfig.BaselineId[1,*]=genconfig.stationid[1]+'-' $
				         +genconfig.stationid[2]
		GenConfig.BaselineId[2,*]=genconfig.stationid[2]+'-' $
				         +genconfig.stationid[0]
;		Triple
		GenConfig.NumTriple=2
		GenConfig.TripleBeam=[[0,0,0],[1,1,1]]
		GenConfig.TripleBase=[[0,1,2],[0,1,2]]
		GenConfig.TripleNumChan[*]=NumChan
		for bl=0,2 do begin
		GenConfig.TripleChan[0:NumChan-1,bl,0]=indgen(NumChan)
		GenConfig.TripleChan[0:NumChan-1,bl,1]=indgen(NumChan)
		endfor
;
		end
;
;		MIRC
		3:begin
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		lambda=[1.79+1.51,2.38+1.97]/2	; Center wavelengths
		widths=[1.79-1.51,2.38-1.97]	; H and K bandwidths
		lambda=lambda[0]		; MIRC is only H-band
		widths=widths[0]
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=widths/lambda*35	; R=35
			2: NumChan=widths/lambda*150	; R=150
			3: NumChan=widths/lambda*450	; R=450
		endcase
		NumChan=fix(NumChan)+1
		GenConfig.NumSpecChan[0]=NumChan
;		H-band, 1.51-1.79 microns
		l=findgen(NumChan[0])/(NumChan[0]-1)
		GenConfig.Wavelength[0:NumChan[0]-1,0]=(l*0.29+1.51)*1e-6
		GenConfig.ChanWidth[0:NumChan[0]-1,0]=0.29e-6/NumChan[0]
;		K-band, 1.97-2.38 microns
;		l=findgen(NumChan(1))/(NumChan(1)-1)
;		GenConfig.Wavelength(0:NumChan(1)-1,1)=(l*0.41+1.97)*1e-6
;		GenConfig.ChanWidth(0:NumChan(1)-1,1)=0.41e-6/NumChan(1)
;
		GenConfig.WavelengthErr=0.1e-6
		GenConfig.ChanWidthErr=1e-9
;
		NumTriple=(NumSid-1)*(NumSid-2)	; H+K combined
		NumBaseline=(NumSid*(NumSid-1))/2
;
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		GenConfig.NumSid=NumSid
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumChan
		GenConfig.TripleNumChan=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		GenConfig.BaselineId[bl,*]=genconfig.stationid[i]+'-' $
					  +genconfig.stationid[j]
		bl=bl+1
		endfor
		endfor
;
		tr=0
		for k=0,NumOutBeam-1 do begin
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId[*,k] eq genconfig.stationid[0]+'-' $
						      +genconfig.stationid[i])
		bl2=where(GenConfig.BaselineId[*,k] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[j])
		bl3=where(GenConfig.BaselineId[*,k] eq genconfig.stationid[0]+'-' $
						      +genconfig.stationid[j])
		GenConfig.TripleBeam[*,tr]=k
		GenConfig.TripleBase[*,tr]=[bl1,bl2,bl3]
		GenConfig.TripleNumChan[tr]=NumChan[k]
		for l=0,2 do begin
			GenConfig.TripleChan[0:NumChan[k]-1,l,tr]=indgen(NumChan[k])
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		end
	endcase
;
; 	Remove the instrument string for backward compatibility
;	SystemId='CHARA'
	return,GenConfig
	end
;
	endcase
	end
'Keck':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['K01','K02']
	'DIAMETER':	return,10.0
	'LONGITUDE': 	return,-155.4733D0
	'LATITUDE':  	return,19.8267D0
	'ALTITUDE':  	return,4123.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,45
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,90
	'GRID':		return,findgen(250)*2+2000	; [nm]
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	'CONFIG':begin
		maxNumSid=2
		maxNumOutBeam=3
		maxNumTriple=1
		maxBaseline=1
		maxSpecChan=1
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
; 		Assume pair-wise beam combination
		GenConfig.BeamCombinerId=1
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations
		GenConfig.RefStation=1
		GenConfig.DelayLineId=[1,2]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId[0,0]='K01-K02'
		GenConfig.NumBaseline[0]=1
;
		GenConfig.NumSpecChan=1
		GenConfig.Wavelength=2.2e-6
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=0.03*GenConfig.Wavelength
		GenConfig.ChanWidthErr=1e-9
;
; 		Set number of independent triples
		GenConfig.NumTriple=0
;
		return,GenConfig
		end
	endcase
	end
'VLTI':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':begin
		instrument=nameparse(instrument_id(systemid),'-')
		case instrument[0] of
		'VINCI':  return,['VE0','VG0']
		'IONIC':  return,['VE0','VG0']
		'MIDI' :  return,['UT2','UT3']
		'AMBER':  return,['UT1','UT2','UT3']
		'AMBER':  return,['AA0','AG1','AJ3']
		'PIONIER':return,['UT1','UT2','UT3','UT4']
		'GRAVITY':return,['AA0','AB2','AD0','AC1']
		'GRAVITY':return,['AD0','AG2','AJ3','AK0']
		'MATISSE':return,['UT1','UT2','UT3','UT4']
		   else:  return,['UT1','UT2']
		endcase
		end
	'DIAMETER':	begin
			case strmid(stations[0],0,1) of
			'A' :return,1.8
			'U' :return,8.0
			else:return,1.8
			endcase
			end
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	begin
			if n_elements(stations) ne 0 then begin
			case stations[0] of
;			'AA0':return,65
;			'AG1':return,64
			 else:return,99
			endcase
			endif else return,99	; was 105 m
			end
	'GRID':begin
		case instrument_id(systemid) of
		'VINCI-MONA':	return,findgen(100)*6+1900
		'VINCI-IONIC':	return,findgen(100)*6+1900
		'MIDI-N-PRISM':	return,findgen(264)*40+10340-5240.0
		'MIDI-N-GRISM':	return,'SPECTROMETER'
		'AMBER-LR':	return,findgen(400)*5+1000
		'AMBER-MR':	return,'SPECTROMETER'
		'AMBER-HR':	return,'SPECTROMETER'
		'PIONIER':	return,findgen(200)*2+1500; [nm], H-band
		'PIONIER-FREE':	return,findgen(200)*2+1500; [nm], H-band
		'PIONIER-GRISM':return,findgen(200)*2+1500; [nm], H-band
		'GRAVITY-LR':	return,findgen(100)*5+1950; K-band, R=40
		'GRAVITY-MR':	return,findgen(600)*1+1950; K-band, R=400
		'GRAVITY-HR':	return,'SPECTROMETER'	  ; K-band, R=4000
		'MATISSE-LN':   return,findgen(230)*11+2600 ; LM,R=34
		'MATISSE-L-LR': return,findgen(230)*11+2600 ; LM,R=34
		'MATISSE-L-MR': return,'SPECTROMETER'; R=506
		'MATISSE-L-HR': return,'SPECTROMETER'; R=959
		'MATISSE-L-HP': return,'SPECTROMETER'; R=?
		'MATISSE-N-LR': return,findgen(510)*10+7950 ; N, R=30
		'MATISSE-N-HR': return,'SPECTROMETER'; R=218
		endcase
		end
	'FILTERS':begin
		case instrument_id(systemid) of
		'VINCI-MONA':	return,'vinci_k'
		'VINCI-IONIC':	return,'vinci_k'
		'PIONIER-FREE':	return,pionier_spectral_cal
		else:		return,'generic_c'
		endcase
		end
	'MAXCONFIG':return,20
	'CONFIG':begin
		maxNumSid=4
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)/2+1
		maxBaseline=maxNumSid*(maxNumSid-1)/2
		maxNumOutBeam=3
		case instrument_id(systemid) of
;			These limits specified again later!
			'AMBER-LR': maxSpecChan=30
			'AMBER-MR': maxSpecChan=600
			'AMBER-HR': maxSpecChan=1000
			'MIDI-N-PRISM': maxSpecChan=60
			'MIDI-N-GRISM': maxSpecChan=160
			'PIONIER-GRISM': maxSpecChan=10
			'PIONIER-FREE': maxSpecChan=1
			'GRAVITY-LR': maxSpecChan=20
			'GRAVITY-MR': maxSpecChan=200
			'GRAVITY-HR': maxSpecChan=2000
			'MATISSE-LN':maxSpecChan=100
			'MATISSE-L-LR':maxSpecChan=100
			'MATISSE-L-MR':maxSpecChan=500
			'MATISSE-L-HR':maxSpecChan=1000
			'MATISSE-L-HP':maxSpecChan=1000
			'MATISSE-N-LR':maxSpecChan=100
			'MATISSE-N-HR':maxSpecChan=1000
			else:	maxSpecChan=100
		endcase
		maxConfig=20
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
; 		Set beam combiner ID, which is identical to instrument ID
		case instrument_id(systemid) of
			'VINCI-MONA' :GenConfig.BeamCombinerId=11
			'VINCI-IONIC':GenConfig.BeamCombinerId=12
			'MIDI-N-PRISM' :GenConfig.BeamCombinerId=21
			'MIDI-N-GRISM' :GenConfig.BeamCombinerId=22
			'AMBER-LR'   :GenConfig.BeamCombinerId=31
			'AMBER-MR'   :GenConfig.BeamCombinerId=32
			'AMBER-HR'   :GenConfig.BeamCombinerId=33
			'PIONIER-GRISM':GenConfig.BeamCombinerId=41
			'PIONIER-FREE' :GenConfig.BeamCombinerId=42
			'GRAVITY-LR' :GenConfig.BeamCombinerId=51
			'GRAVITY-MR' :GenConfig.BeamCombinerId=52
			'GRAVITY-HR' :GenConfig.BeamCombinerId=53
			'MATISSE-LN'  :GenConfig.BeamCombinerId=61
			'MATISSE-L-LR':GenConfig.BeamCombinerId=62
			'MATISSE-L-MR':GenConfig.BeamCombinerId=63
			'MATISSE-L-HR':GenConfig.BeamCombinerId=64
			'MATISSE-L-HP':GenConfig.BeamCombinerId=65
			'MATISSE-N-LR':GenConfig.BeamCombinerId=66
			'MATISSE-N-HR':GenConfig.BeamCombinerId=67
			         else:GenConfig.BeamCombinerId=1
		endcase
;
; 		Remove the instrument string for backward compatibility
;		SystemId='VLTI'
;
		case GenConfig.BeamCombinerId/10 of
;	  	VINCI
	      1:begin
		if GenConfig.NumSid ne 2 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 2!'
			return,-1
		endif
	  	NumOutBeam=1
		NumChan=1
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=0
;
		GenConfig.DelayLineId=[1,2]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId[0,0]=genconfig.stationid[0]+'-' $
				         +genconfig.stationid[1]
		GenConfig.NumBaseline[0]=1
;
		GenConfig.NumSpecChan=NumChan
		GenConfig.Wavelength[0:NumChan-1,0]=2.2e-6
		GenConfig.WavelengthErr[0:NumChan-1,0]=0.1e-6
		GenConfig.ChanWidth=0.03*GenConfig.Wavelength
		GenConfig.ChanWidthErr[0:NumChan-1,0]=1e-9
;
		return,GenConfig
		end
;		MIDI
	      2:begin
		if GenConfig.NumSid ne 2 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 2!'
			return,-1
		endif
		NumOutBeam=1
		NumChan=60
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=0
;
		GenConfig.DelayLineId=[2,3]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId[0,0]=genconfig.stationid[0]+'-' $
				         +genconfig.stationid[1]
		GenConfig.NumBaseline[0]=1
;
		GenConfig.NumSpecChan=NumChan
		GenConfig.Wavelength[0:NumChan-1,0]=(findgen(NumChan)*0.1+7.5)*1e-6
		GenConfig.WavelengthErr[0:NumChan-1,0]=0.05e-6
		GenConfig.ChanWidth=0.1e-6
		GenConfig.ChanWidthErr[0:NumChan-1,0]=1e-9
;
		return,GenConfig
		end
;		AMBER
	      3:begin
		if GenConfig.NumSid ne 3 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 3!'
			return,-1
		endif
		NumOutBeam=3
;		3-way combination
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,3,5]
		GenConfig.BaselineId[0,*]=genconfig.stationid[0]+'-' $
				         +genconfig.stationid[1]
		GenConfig.BaselineId[1,*]=genconfig.stationid[1]+'-' $
				         +genconfig.stationid[2]
		GenConfig.BaselineId[2,*]=genconfig.stationid[2]+'-' $
				         +genconfig.stationid[0]
		GenConfig.NumBaseline=3
;
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=[30,30,30]		; R=35
			2: NumChan=[600,600,600]	; R=1500
			3: NumChan=[1000,1000,1000]	; R=10000
		endcase
		GenConfig.NumSpecChan=NumChan
;		J-band, 1.08-1.34 microns
		l=findgen(NumChan[0])/(NumChan[0]-1)
		GenConfig.Wavelength[0:NumChan[0]-1,0]=(l*0.26+1.08)*1e-6
		GenConfig.ChanWidth[0:NumChan[0]-1,0]=0.26e-6/NumChan[0]
;		H-band, 1.51-1.79 microns
		l=findgen(NumChan[1])/(NumChan[1]-1)
		GenConfig.Wavelength[0:NumChan[1]-1,1]=(l*0.29+1.51)*1e-6
		GenConfig.ChanWidth[0:NumChan[1]-1,1]=0.29e-6/NumChan[1]
;		K-band, 1.97-2.38 microns
		l=findgen(NumChan[2])/(NumChan[2]-1)
		GenConfig.Wavelength[0:NumChan[2]-1,2]=(l*0.41+1.97)*1e-6
		GenConfig.ChanWidth[0:NumChan[2]-1,2]=0.41e-6/NumChan[2]
;
		GenConfig.WavelengthErr=0.1e-6
		GenConfig.ChanWidthErr=1e-9
;
;		Triple
		GenConfig.NumTriple=3
		GenConfig.TripleBeam=[[0,0,0],[1,1,1],[2,2,2]]
		GenConfig.TripleBase=[[0,1,2],[0,1,2],[0,1,2]]
		GenConfig.TripleChan=0
		for tr=0,2 do begin
		GenConfig.TripleNumChan[tr]=NumChan[tr]
		for bl=0,2 do begin
		GenConfig.TripleChan[0:NumChan[tr]-1,bl,tr]=indgen(NumChan[tr])
		endfor
		endfor
;
		return,GenConfig
		end
;		PIONIER
	      4:begin
		if GenConfig.NumSid ne 4 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 4!'
			return,-1
		endif
		NumChan=7
		GenConfig.NumSpecChan=NumChan
;		H-band, 1.51-1.79 microns
		l=findgen(NumChan[0])/(NumChan[0]-1)
		GenConfig.Wavelength[0:NumChan[0]-1,0]=(l*0.29+1.51)*1e-6
		GenConfig.ChanWidth[0:NumChan[0]-1,0]=0.29e-6/NumChan[0]
;
		GenConfig.WavelengthErr=0.1e-6
		GenConfig.ChanWidthErr=1e-9
;
		NumOutBeam=1
		NumTriple=((NumSid-1)*(NumSid-2))/2+1
		NumBaseline=(NumSid*(NumSid-1))/2
;
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumChan
		GenConfig.TripleNumChan=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
;
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		GenConfig.BaselineId[bl,*]=genconfig.stationid[i]+'-' $
					  +genconfig.stationid[j]
		bl=bl+1
		endfor
		endfor
;
		if 0 then begin
		tr=0
		k=0
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId[*,k] eq genconfig.stationid[0]+'-' $
						      +genconfig.stationid[i])
		bl2=where(GenConfig.BaselineId[*,k] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[j])
		bl3=where(GenConfig.BaselineId[*,k] eq genconfig.stationid[0]+'-' $
						      +genconfig.stationid[j])
		GenConfig.TripleBeam[*,tr]=k
		GenConfig.TripleBase[*,tr]=[bl1,bl2,bl3]
		GenConfig.TripleNumChan[tr]=NumChan[k]
		for l=0,2 do begin
			GenConfig.TripleChan[0:NumChan[k]-1,l,tr]=indgen(NumChan[k])
		endfor
		tr=tr+1
		endfor
		endfor
		endif
;
;		Borrowed from GRAVITY (below)
;
		tr=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		for k=j+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[j])
		bl2=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[j]+'-' $
						      +genconfig.stationid[k])
		bl3=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[k])
		GenConfig.TripleBeam[*,tr]=0
		GenConfig.TripleBase[*,tr]=[bl1,bl2,bl3]
		GenConfig.TripleNumChan[tr]=NumChan
		for l=0,2 do begin
			GenConfig.TripleChan[0:NumChan-1,l,tr] $
				=indgen(NumChan)
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		return,GenConfig
		end
;		GRAVITY
	      5:begin
		if GenConfig.NumSid ne 4 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 4!'
			return,-1
		endif
		NumTriple=1*(((NumSid-1)*(NumSid-2))/2+1)
		NumBaseline=(NumSid*(NumSid-1))/2
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=20	; R=40
			2: NumChan=200	; R=400
			3: NumChan=2000	; R=4000
		endcase
		GenConfig.NumSpecChan=0
		GenConfig.NumSpecChan[0]=NumChan
;		K-band, 2.05-2.45 microns
		l=findgen(NumChan)/(NumChan-1)
		GenConfig.Wavelength[0:NumChan-1,0]=(l*0.4+2.05)*1e-6
		GenConfig.ChanWidth[0:NumChan-1,0]=0.4e-6/NumChan
;
		GenConfig.WavelengthErr[0:NumChan-1,0]=0.1e-6
		GenConfig.ChanWidthErr[0:NumChan-1,0]=1e-9
;
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.TripleNumChan[0:3]=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		GenConfig.BaselineId[bl,*]=genconfig.stationid[i]+'-' $
					  +genconfig.stationid[j]
		bl=bl+1
		endfor
		endfor
;
		tr=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		for k=j+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[j])
		bl2=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[j]+'-' $
						      +genconfig.stationid[k])
		bl3=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[k])
		GenConfig.TripleBeam[*,tr]=0
		GenConfig.TripleBase[*,tr]=[bl1,bl2,bl3]
		GenConfig.TripleNumChan[tr]=NumChan
		for l=0,2 do begin
			GenConfig.TripleChan[0:NumChan-1,l,tr] $
				=indgen(NumChan)
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		return,GenConfig
		end
;		MATISSE
	      6:begin
		if GenConfig.NumSid ne 4 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 4!'
			return,-1
		endif
		NumTriple=1*(((NumSid-1)*(NumSid-2))/2+1)
		NumBaseline=(NumSid*(NumSid-1))/2
		GenConfig.StationId[0:NumSid-1]=stations[sid_index]
		GenConfig.RefStation=1
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=15	; R=30
			2: NumChan=250	; R=500
			3: NumChan=500	; R=1000
			4: NumChan=15	; R=30
			5: NumChan=110	; R=220
		endcase
		case GenConfig.BeamCombinerId mod 10 of
			1: l_min=2.5	; R=30
			2: l_min=2.5	; R=500
			3: l_min=2.5	; R=1000
			4: l_min=8.0	; R=30
			5: l_min=8.0	; R=220
		endcase
		case GenConfig.BeamCombinerId mod 10 of
			1: l_max=5.5	; R=30
			2: l_max=5.5	; R=500
			3: l_max=5.5	; R=1000
			4: l_max=13.0	; R=30
			5: l_max=13.0	; R=220
		endcase
		width=(l_max-l_min)/NumChan
		case GenConfig.BeamCombinerId mod 10 of
			1: width=15	; R=30
			2: width=250	; R=500
			3: width=500	; R=1000
			4: width=15	; R=30
			5: width=110	; R=220
		endcase
		GenConfig.NumSpecChan=0
		GenConfig.NumSpecChan[0]=NumChan
		l=findgen(NumChan)/(NumChan-1)
		GenConfig.Wavelength[0:NumChan-1,0]=(l*width+l_min)*1e-6
		GenConfig.ChanWidth[0:NumChan-1,0]=width
;
		GenConfig.WavelengthErr[0:NumChan-1,0]=0.1e-6
		GenConfig.ChanWidthErr[0:NumChan-1,0]=1e-9
;
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.TripleNumChan[0:3]=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		GenConfig.BaselineId[bl,*]=genconfig.stationid[i]+'-' $
					  +genconfig.stationid[j]
		bl=bl+1
		endfor
		endfor
;
		tr=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		for k=j+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[j])
		bl2=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[j]+'-' $
						      +genconfig.stationid[k])
		bl3=where(GenConfig.BaselineId[*,0] eq genconfig.stationid[i]+'-' $
						      +genconfig.stationid[k])
		GenConfig.TripleBeam[*,tr]=0
		GenConfig.TripleBase[*,tr]=[bl1,bl2,bl3]
		GenConfig.TripleNumChan[tr]=NumChan
		for l=0,2 do begin
			GenConfig.TripleChan[0:NumChan-1,l,tr] $
				=indgen(NumChan)
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		return,GenConfig
		end
;
		endcase
		end
	endcase
	end
else: 	begin
	if key eq 'MIDNIGHT' then begin
	print,'Warning: SystemId ('+systemid+') not known, assume midnight=8 UT'
	return,8.
	endif else $
        print,'***Error(SYSTEM_CONFIG): unknown SystemId: '+ $
                                          systemid+' !'
	end
endcase
;
return,-1
;
end
