function vizier_phot,star_in
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Prepare and execute Python script to download photometry from Vizier.
;
; Example: r=vizier_phot('HD217014')
; Vizier returns wavelength in GHz and flux in Jy
;
; Return: [wave (microns),flux [CGS: erg s^-1 cm^-2 mum^-1],err,filter]
; See also Boden 2007, New Astronomy Reviews 51 (2007) 617–622
;	"Calibrating optical/IR interferometry visibility data"
;
; Conversion Jy to CGS system: 3.27e3 Jy = 3.32e-12 erg/(s*cm^2*um)
; => CGS = Jy * 1.0153e-15
;
; Example (wavelength in microns):
; r=vizier_phot('HD27376')
;
; On machine PTI Ubuntu, go to:
; https://irsa.ipac.caltech.edu/cgi-bin/Radar/nph-discovery
; and add Star ID (e.g., "HD 27376" in box at bottom and "Search IRSA"
;
common StarBase,startable,notes
;
if n_elements(star_in) eq 0 then begin
	print,'Error: you must supply a star ID (e.g., HD 123456)!'
	return,0
endif else begin
	star=vizier_starid(strcompress(star_in,/remove_all))
endelse
;
IF 1 THEN BEGIN
;
; Get data from VIZIER
com_file=star+'.inp'
sed_file=star+'.out'
openw,unit,com_file,/get_lun
printf,unit,"import warnings"
printf,unit,"import astropy"
printf,unit,"from astropy.table import Table"
printf,unit,'warnings.filterwarnings("ignore")'
radius='5'
printf,unit, $
'sed=Table.read("https://vizier.cds.unistra.fr/viz-bin/sed?-c={'+star+'}&-c.rs={
;sed=Table.read("https://vizier.cds.unistra.fr/viz-bin/sed?-c={HD2737}&-c.rs={5}
printf,unit,"import tabulate"
printf,unit,"from tabulate import tabulate"
printf,unit,"open('"+sed_file+"','w').write(tabulate(sed))"
free_lun,unit
;
ENDIF ELSE BEGIN
;
; Read VOT table (not working yet, only as command to web browser)
command='/usr/bin/wget https://vizier.cds.unistra.fr/viz-bin/sed?-c="'+star+'"'
spawn,command
;
sed_file=star+'.vot'
com_file=star+'.inp'
openw,unit,com_file,/get_lun
printf,unit,"import astropy"
printf,unit,"from astropy.table import Table"
printf,unit,'radius="5"'
printf,unit,'target="'+star+'"'
printf,unit, $
  'sed=Table.read("https://vizier.cds.unistra.fr/viz-bin/sed?-c={"+target+"}&-c.
printf,unit,"import tabulate"
printf,unit,"from tabulate import tabulate"
printf,unit,"open('"+sed_file+"','w').write(tabulate(sed))"
free_lun,unit
;
ENDELSE
;
; Run Python script
print,'Excuting Python script '+com_file+' to access VIZIER...'
spawn,"/usr/bin/python < "+com_file
print,'Output SED [Jy vs frequency] saved to: '+sed_file
;
status=dc_read_fixed(sed_file,l,/col,format='(a192)')
num_line=n_elements(l)
;
; Decode first line dashes for formatting and perform some checks
words=nameparse(l[0],' ')
if n_elements(words) ne 10 then begin
	print,'Error: expected number (10) of columns not found!'
endif
column_l=strlen(words)
;
; Put | in each row where there is a blank in the first line between the dashes
for i=0,n_elements(column_l)-1 do begin
	for j=0,num_line-1 do strput,l,'|',total(column_l[0:i])+i*2
endfor
;
; Remove first and last line (these should look like "------ ------")
l=l[1:num_line-1]
;
; Vizier output is in flux[Jy] and freq.[GHz].
; Extract photometry as flux [erg 1/cm^2 1/s 1/micron] from input in Jy
; Convert freq [GHz] to wavelength [micron]: lambda [mu] = 3.0e5 / freq
; Example: 1.5 Jy @ 3.48x10^4 GHz l=8.62 mu: F=6.06e-11 erg/(s*cm^2*mu)
; flux(i)=1.0153e-15*float(words(7))	; erg/cm^2/s/micron
;
; Conversions:
; (Eureka Scientific,  Inc., 2452 Delmer Street, Suite 100, Oakland, CA 94602)
; Adapted for wavelength measured in microns (original: Angstroem)
; F_lambda = 3.0x10^(-1) S(Jy) / lambda^2
;	with [lambda] = micron (mu): 1 mu = 10000 A
;
; Verification using "Magnitude to Flux Density converter" at
; irsa.ipac.caltech.edu/data/SPITZER/docs/dataanalysistools/tools/pet/magtojy/
; HD 34029: flux = 3230 Jy at 136890 GHz corresponds to K = -1.71 (SimBad: K = -
;
; Columns (epoch 2000):
;RA2000|DE2000|Catalog|Cat.ID|tbl_86|tbl_87|SED_freq|SED_flux|SED_eflux|filter
;  deg |  deg |       |      |  d   |      |  GHz   |   Jy   |    Jy   |
;
; Write raw SED data
sed_file=star+'.sed'
print,'Writing SED Flux [erg/cm^2 mu] vs wavelength to '+sed_file
openw,unit,sed_file,/get_lun
for i=0,n_elements(l)-1 do printf,unit,l[i]
free_lun,unit
;
; Extract data line by line
cite=strarr(n_elements(l))	; citation, e.g., J/MNRAS/495/1943/table3
unid=strarr(n_elements(l))	; unique ID created for each record
flux=dblarr(n_elements(l))	; Jy
ferr=dblarr(n_elements(l))	; Jy
wave=flux
fnan=strarr(n_elements(l))	; char, NaN or blank
filt=strarr(n_elements(l))	; char, filter name
;
for i=0,n_elements(l)-1 do begin
	words=nameparse(l[i],'|')
	cite[i]=strcompress(words[2],/remove_all)
	unid[i]=strcompress(words[6]+words[7]+words[8]+words[9])
	wave[i]=3.0e5/float(words[6])		; microns
	flux[i]=3.0e-9*float(words[7])/wave[i]^2; erg/cm^2/s/micron
	fnan[i]=strtrim(words[8],2)
	if isnumeric(fnan[i]) then $	; can be NaN or a value
	ferr[i]=3.0e-9*((float(words[7])+float(fnan[i])) $
		       -(float(words[7])-float(fnan[i])))/(2*wave[i]^2)
;	Extract filters and and fix some names
	filt[i]=strcompress(words[9],/remove_all)
endfor
;
; Fix some filter names to enforce name consistency w/filter_c (filter.pro)
index=where(strmid(filt,0,4) eq 'SDSS',count)	; has ' at end of name
for j=0,count-1 do filt[index[j]]=strmid(filt[index[j]],0,6)
index=where(strmid(filt,0,9) eq 'Johnson:L',count); has ' at end of name
for j=0,count-1 do filt[index[j]]=strmid(filt[index[j]],0,9)
index=where(strmid(filt,0,3) eq 'PAN' and strlen(filt) eq 16,count)
for j=0,count-1 do begin
	words=strsplit(filt[index[j]],'/',/extract)
	strgs=strsplit(words[1],':',/extract)
	filt[index[j]]=words[0]+':'+strgs[1]
endfor
;
; Spitzer detectors: IRAC I1/2/3/4; IRS.Blue/Red, MIPS.24/70/160
;
; Spitzer/IRAC
index=where(strpos(filt,'Spitzer/IRAC:3.6') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:3.6'
index=where(strpos(filt,'Spitzer/IRAC:4.5') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:4.5'
index=where(strpos(filt,'Spitzer/IRAC:5.7') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:5.7'
index=where(strpos(filt,'Spitzer/IRAC:7.9') ge 0,count)
for k=0,count -1 do filt[index[k]]='IRAC:7.9'
; Spitzer/MIPS 24mu, 70um, 160mu; J/ApJ/785/33/table2
; https://cdsarc.u-strasbg.fr/ftp/pub/J/other/ApJ/785/33/
index=where(strpos(filt,'Spitzer/MIPS_24') ge 0,count)
for k=0,count -1 do filt[index[k]]='MIPS:24'
index=where(strpos(filt,'Spitzer/MIPS_70') ge 0,count)
for k=0,count -1 do filt[index[k]]='MIPS:70'
index=where(strpos(filt,'Spitzer/MIPS_160') ge 0,count)
for k=0,count -1 do filt[index[k]]='MIPS:160'
;
; http://vizier.cds.unistra.fr/viz-bin/VizieR?-source=J/PASP/119/994/table5
index=where(cite eq 'J/PASP/119/994/table5' and filt eq ':=12um',count)
if count gt 0 then filt[index]='IRAS:12'
index=where(cite eq 'J/PASP/119/994/table5' and filt eq ':=24um',count)
if count gt 0 then filt[index]='MIPS:24'
index=where(cite eq 'J/PASP/119/994/table5' and filt eq ':=25um',count)
if count gt 0 then filt[index]='MIPS:24' ; We don't have the transmission
; Photometry in the MIPS 70µm, PACS 70µm, and PACS 100µm bands
; http://vizier.cds.unistra.fr/viz-bin/VizieR?-source=J/MNRAS/495/1943/table3
index=where(cite eq 'J/MNRAS/495/1943/table3' and filt eq ':=70um',count)
if count gt 0 then filt[index]='PACS:blue'
index=where(cite eq 'J/MNRAS/495/1943/table3' and filt eq ':=100um',count)
if count gt 0 then filt[index]='PACS:green'
; https://vizier.cds.unistra.fr/viz-bin/VizieR-3?-source=J/MNRAS/488/3588
index=where(cite eq 'J/MNRAS/488/3588/table2' and filt eq ':=70um',count)
if count gt 0 then filt[index]='Spitzer/MIPS:70'
index=where(cite eq 'J/MNRAS/488/3588/table2' and filt eq ':=100um',count)
if count gt 0 then filt[index]='PACS:green'
;
; CTIO
index=where(strpos(filt,'CTIO/DECam:u') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:u'
index=where(strpos(filt,'CTIO/DECam:g') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:g'
index=where(strpos(filt,'CTIO/DECam:r') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:r'
index=where(strpos(filt,'CTIO/DECam:i') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:i'
index=where(strpos(filt,'CTIO/DECam:z') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:z'
index=where(strpos(filt,'CTIO/DECam:Y') ge 0,count)
for k=0,count -1 do filt[index[k]]='DECam:Y'
;
; GAIA
index=where(strpos(filt,'GAIA/') ge 0,count)
for k=0,count -1 do begin
	words=nameparse(filt[index[k]],'/')
	filt[index[k]]=words[1]
endfor
;
stop
; Remove NaN entries and those without error bar
index=where(fnan eq 'nan',nan)
print,'Number of NaN entries to be removed:',nan
index=where(fnan ne 'nan',nan)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
index=where(ferr eq 0,count)
print,'Number of entries without error bar:',count
index=where(ferr gt 0,count)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
; Use higher quality measurements
if 0 then begin
index=where(abs(ferr) lt 1e-7,count)
l=l[index]
unid=unid[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
endif
;
; Many records are duplicates
index=uniq(unid,sort(unid))
l=l[index]
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
filt=filt[index]
;
; Write SED data
sed_file=star+'.viz'
openw,unit,sed_file,/get_lun
for i=0,n_elements(l)-1 do printf,unit,l[i]
free_lun,unit
;
; Print data for filters under review...
if 0 then begin
for i=0,n_elements(l)-1 do begin
    if filt[i] eq 'Gaia:G' then print,wave[i],flux[i],ferr[i]
    if filt[i] eq 'GAIA3:G' then print,wave[i],flux[i],ferr[i],'  '+filt[i]
    if filt[i] eq 'GAIA3:Grp' then print,wave[i],flux[i],ferr[i],'  '+filt[i]
    if filt[i] eq 'GAIA3:Gbp' then print,wave[i],flux[i],ferr[i],'  '+filt[i]
endfor
endif
;
index=sort(wave)
return,{wave:wave,flux:flux,ferr:ferr,filt:filt}
;
end
