function vizier_sed,star_in,template,pickles=spec_class
COMPILE_OPT STRICTARR,STRICTARRSUBS
;
; Prepare and execute Python script to download photometry from Vizier.
;
; Example: r=vizier_sed('HD217014,template,pickles='g2iv')
; Vizier returns wavelength in GHz and flux in Jy
;
; Return: [wave (microns),flux (erg cm^-2 s^-1 mum^-1)]
; See also Boden 2007, New Astronomy Reviews 51 (2007) 617–622
;	"Calibrating optical/IR interferometry visibility data"
;
; Optional Pickles template on output: wave=template(*,0), flux=template(*,1)
; spec_class: e.g., 'B9V' to find closest template, or 'b9v' for specific file.
;
; If keyword pickles is used to pass a spectral class, also return closest
; spectrum (wave_p,flux_p) from the Pickles library.
;
; Example (wavelength in microns):
; r=vizier_sed('HD27376',template,pickles='B9V')
; plot,template(*,0),template(*,1)*1e6,psym=0, $
; 	xtitle='Wavelength ('+greek('mu')+'m]', $
; 	ytitle='F!D'+greek('lambda')+'!N * 10!E6!N ', $
; 	'(erg cm!E-2!N s!E-1!N'+greek('mu')+'m!E-1!N)'
; oplot,r(*,0),r(*,1)*1e6,psym=5
;
; On machine PTI Ubuntu, go to:
; https://irsa.ipac.caltech.edu/cgi-bin/Radar/nph-discovery
; and add Star ID (e.g., "HD 27376" in box at bottom and "Search IRSA"
;
;
common StarBase,startable,notes
;
; Star name must not have blanks!
; if n_elements(star_in) ne 0 then star=strtrim(star_in,2)
; star_in=oyster_starid(star_in)
;
if n_elements(star_in) eq 0 then begin
; 	if n_elements(startable) eq 1 then begin
; 		print,'Star not specified, using ',startable.starid
; 		star=vizier_starid(startable.starid)
; 	endif else begin
		print,'Error: you must supply a star ID (e.g., HD 123456)!'
		return,0
;	endelse
endif else begin
	star=vizier_starid(strtrim(star_in,2))
endelse
;
sed_file=star+'.txt'
command_file=star+'.inp'
openw,unit,command_file,/get_lun
printf,unit,"import astropy"
printf,unit,"from astropy.table import Table"
printf,unit,"import tabulate"
printf,unit,"from tabulate import tabulate"
printf,unit,"import warnings"
printf,unit,'warnings.filterwarnings("ignore")'
printf,unit,"radius=1"
printf,unit,'target="'+star+'"'
printf,unit, $
'sed=Table.read(f"https://vizier.cfa.harvard.edu/viz-bin/sed?-c={target}&-c.rs={radius}")'
printf,unit,"open('"+sed_file+"','w').write(tabulate(sed))"
free_lun,unit
print,'Excuting Python script '+command_file+' to access VIZIER...'
spawn,"/usr/bin/python < "+command_file
wait,5
;
status=dc_read_fixed(sed_file,l,/col,format='(a185)')
num_line=n_elements(l)
;
; Decode first line dashes for formatting and perform some checks
words=nameparse(l[0],' ')
if n_elements(words) ne 10 then begin
	print,'Error: expected number (10) of columns not found!'
endif
column_l=strlen(words)
;
; Put | in each row where there is a blank in the first line between the dashes
for i=0,n_elements(column_l)-1 do begin
	for j=0,num_line-1 do strput,l,'|',total(column_l[0:i])+i*2
endfor
;
; Remove first and last line (these should look like "------ ------")
l=l[1:num_line-1]
;
; Vizier output is in flux[Jy] and freq.[GHz].
; Extract photometry as flux [erg 1/cm^2 1/s 1/micron] from input in Jy
; Convert freq [GHz] to wavelength [micron]: lambda [mu] = 3.0e5 / freq
;
; Conversion:
; (Eureka Scientific,  Inc., 2452 Delmer Street, Suite 100, Oakland, CA 94602)
; F_lambda = 3.0x10^(-5) S(Jy) / lambda^2
;	with [lambda] = Angstrom (A): 1 A =0.1 nm
;
; F_lambda = 3.0x10^(-1) S(Jy) / lambda^2
;	with [lambda] = micron (mu): 1 mu = 10000 A
;
; Verification using "Magnitude to Flux Density converter" at
; irsa.ipac.caltech.edu/data/SPITZER/docs/dataanalysistools/tools/pet/magtojy/
; HD 34029: flux = 3230 Jy at 136890 GHz corresponds to K = -1.71 (SimBad: K = -1.78)
;
; Columns (epoch 2000):
;RA2000|DE2000|Catalog|Cat.ID|tbl_86|tbl_87|SED_freq|SED_flux|SED_eflux|filter
;  deg |  deg |       |      |  d   |      |  GHz   |   Jy   |    Jy   |
;
flux=dblarr(n_elements(l))
wave=flux
fnan=strarr(n_elements(l))
;
for i=0,n_elements(l)-1 do begin
	words=nameparse(l[i],'|')
	wave[i]=3.0e5/float(words[6])			; microns
	flux[i]=3.e-1*float(words[7])/wave[i]^2
	fnan[i]=strtrim(words[8],2)
endfor
;
index=where(fnan ne 'nan')
wave=wave[index]
flux=flux[index]
ferr=float(fnan[index])
;
index=where(ferr gt 0,index)
wave=wave[index]
flux=flux[index]
ferr=ferr[index]
;
flux=flux/2e8	; "Andy" fudge factor
ferr=ferr/2e8
;
; Optionally return nearest Pickles template spectrum
if keyword_set(spec_class) then begin
;	Lower-case spec_class to select a specific template
	if spec_class ne strupcase(spec_class) then begin
		flux_p=pickles(star_in,wave_p,template=spec_class)
;	Upper-case spec_class to select nearest available template
	endif else begin
		flux_p=pickles(star_in,wave_p)
	endelse
	wave_p=wave_p*1e6	; convert to microns
	flux_p=flux_p/2e4 ; Andy fudge factor
	template=[[wave_p],[flux_p]]
endif
;
index=sort(wave)
return,[[wave[index]],[flux[index]],[ferr[index]]]
;
end
