;*******************************************************************************
; File: mygravitygui.pro
; Front-end for Reflex GRAVITY data reduction
; Part of OYSTER (Christian Hummel)
;
; Block directory:
; ----------------
;
; Block 1: alloc_gravity_options,singledual,dfits_gravity,rename2arc_gravity,
;	   rename_wkf_products,rename_viscal_products,find_files
; Block 2: analyzegravityendproducts,analyzegravityconfigids,gravityconfigids,
;	   selectgravityfiles,
;	   gravitygenconfig,gravitystarids
; Block 3: gravitygui_event,gravitygui_productdirs,gravitygui_configid,
;	   files_event,gravitygui_options
;	   mygravitygui,mygravitypipe,gravity_cleanup,gravity_compact,
;	   gravity_reduce,gravity_fluxcal,gravity_zerocounts,gravity_brg
; Block 4: plotspectrumshift,specphot
; Block 5: molecfit2l,l2molecfit,molecfit_key,splice_index,merge_spectra,
;	   set_contint,gravity_normcont,
;	   gravity_kernel,gravity_kernels
; Block 6: gravity_tellcorr,
;	   gravity_molecfit,gravity_datfile,gravity_tellfit,gravity_tellpower
; Block 7: gravity_prepare_tellrem,gravity_tellrem,
; Block 8: gravity_display_acq,gravity_north_angle,gravity_acq_scale,
;	   gravity_split,gravity_oivis_check
; Block 9: bpcru
;
;************************************************************************Block 1
function alloc_gravity_options
;
return,{$
	viscorrection:'VFACTOR' $
	}
;
end
;-------------------------------------------------------------------------------
function singledual,insmode
;
if strpos(insmode,'SINGLE') eq 0 then return,'SINGLE'
if strpos(insmode,'DUAL') eq 0 then return,'DUAL'
if strpos(insmode,'ASTROREDUCED') eq 0 then return,'DUAL'
;
end
;-------------------------------------------------------------------------------
function dfits_gravity,r
;
ipos=strpos(r(1),'GRAVI')
jpos=strpos(r(0),'FT.ROBJ')
jpos=strpos(r(0),'OBS.NAME')
return,strmid(r,ipos,29)+' '+strmid(r,jpos,strlen(r(1))-jpos)
return,strmid(r,ipos,strlen(r(1))-ipos)
;
end
;-------------------------------------------------------------------------------
pro rename2arc_gravity,fitsfiles
;
if n_elements(fitsfiles) eq 0 then fitsfiles='GRAVITY_GEN_???*.fits'
;
files=file_search(fitsfiles)
;
for i=0,n_elements(files)-1 do begin
;
	words=nameparse(files(i),['.','_'])
	instrument=words(0)
	shortinstr=strmid(instrument,0,5)
;
	fitsfile=obj_new('fitsfile',files(i))
	prihead=fitsfile->prihead()
	arcfile=prihead->getpar('ARCFILE')
	pipefile=string(prihead->getpar('PIPEFILE'))
	obj_destroy,fitsfile
	if strpos(pipefile,instrument) eq 0 and $
	   strpos(arcfile,shortinstr) eq 0 then begin
		ipos=25
		jpos=strpos(files(i),'.fits')
		suffix=strmid(files(i),ipos,jpos-ipos)
		destfile=instrument+'_'+strmid(arcfile,6,19)+'_'+suffix+'.fits'
	command='mv '+files(i)+' '+destfile
	spawn,command
	endif
;
endfor
;
end
;-------------------------------------------------------------------------------
pro rename_wkf_products
;
; The GRAVITY pipeline does not include the date/time string in the
; file name of the reduced data, this procedure transfers this information
; from the folder (created by the pipleline) to the file names as well.
;
; Must be in a folder reflex_end_products/YYYY-MM-DDTHH:MM:SS.SSS
;
dirs=file_search('GRAVI*')
;
for i=0,n_elements(dirs)-1 do begin
	cd,dirs(i),current=rep
	words=nameparse(pwd(),'/')
	date=strmid(words(n_elements(words)-1),6,19)
	f=file_search('*.fits')
	for j=0,n_elements(f)-1 do begin
		spawn,'mv '+f(j)+' '+basename(f(j))+'_'+date+'.fits'
	endfor
	cd,rep
endfor
;
end
;-------------------------------------------------------------------------------
pro rename_viscal_products
;
; The GRAVITY pipeline does not include the date/time string in the
; file name of the calibrated data, this procedure transfers this information
; from the folder (created by the pipleline) to the file names as well.
;
; Must be in a folder reflex_end_products/YYYY-MM-DDTHH:MM:SS.SSS
;
dirs=file_search('CAL*')
if strlen(dirs(0)) eq 0 then dirs=file_search('SCI*')
;
for i=0,n_elements(dirs)-1 do begin
	cd,dirs(i),current=rep
	words=nameparse(pwd(),'/')
	words=nameparse(words(n_elements(words)-1),'_')
	date=words(n_elements(words)-1)
	f=file_search('*.fits')
	for j=0,n_elements(f)-1 do begin
		spawn,'mv '+f(j)+' '+basename(f(j))+'_'+date+'.fits'
	endfor
	cd,rep
endfor
;
end
;-------------------------------------------------------------------------------
function find_files,directory,searchstrng
;
; Recursive file search.
;
words=nameparse(searchstrng)
for i=0,n_elements(words)-1 do begin
	spawn,'find -L '+directory+' -iname '+words(i),f
	if i eq 0 then files=f else files=[files,f]
endfor
return,files
;
end
;************************************************************************Block 2
function analyzegravityendproducts,udate,write=write,all=all
;
; Called by analyzegravityconfigids. Return dates for which pipeline
; products were found (using searchstrng). If given a date, return
; all files available for that date.
;
common GravityPipeline,rootdatadir,productsdir,searchstrng,configid,recipes
;
; Directory/file tree of reflex_end_products:
; First level  down: YYYY-MM-DDTHH:MM:SS.FFF (date of running the recipe)
; Second level down: GRAVI.YYYY-MM-DDTHH:MM:SS.FFF (date of observation)
; Third level  down: OBS.NAME_YYYY-MM-DDTHH:MM:SS_SINGLE_SCI_VIS.fits
; Third level  down: OBS.NAME_YYYY-MM-DDTHH:MM:SS_SINGLE_CAL_VIS.fits
; Third level  down: OBS.NAME_YYYY-MM-DDTHH:MM:SS_DUAL_SCI_VIS.fits
; Third level  down: OBS.NAME_YYYY-MM-DDTHH:MM:SS_DUAL_CAL_VIS.fits
; Third level  down: OBS.NAME_YYYY-MM-DDTHH:MM:SS_ASTROREDUCED.fits
;
; file_search does not follow symbolic links!
; f=file_search(productsdir,'*single*',/fold_case)
; if searchstrng eq "'*.fits'" then spawn,'ls *.fits',f $
if searchstrng(0) eq '*.fits' then begin
	spawn,'ls *.fits',f
; else spawn,'find -L '+productsdir+' -iname '+searchstrng,f
endif else begin
	f=find_files(productsdir,searchstrng)
;	words=nameparse(searchstrng)
;	for i=0,n_elements(words)-1 do begin
;		spawn,'find -L '+productsdir+' -iname '+words(i),files
;		if i eq 0 then f=files else f=[f,files]
;	endfor
endelse
f=f(where(strlen(f) gt 0))
;
n=n_elements(f)
dates=strarr(n)
times=strarr(n)
;
strings=stringregex(f,'????-??-??T??:??:??')
dates=strmid(strings,0,10)
times=strmid(strings,11,8)
index=where(hms2h(times) lt 12,count)
for i=0,count-1 do dates(i)=previousdate(dates(i))
;
udates=unique(dates)
;
if n_elements(write) eq 0 then write=0
;
if n_elements(all) eq 0 then all=0
if all then udate=udates
files=''
;
if n_elements(udate) ne 0 then begin
	for i=0,n_elements(udate)-1 do begin
		index=where(dates eq udate(i),count)
		if count gt 0 then files=[files,f(index)]
		if write then begin
			openw,unit,'product_files.txt',/get_lun
			for i=0,count-1 do printf,unit,f(index(i))
			free_lun,unit
		endif
	endfor
	return,files(1:n_elements(files)-1)
endif else return,udates
;
end
;-------------------------------------------------------------------------------
function analyzegravityconfigids
;
; Determine how many different configurations are present in the data
; of the select products directory. Configurations are different for different
; P2VMIDs, baselines, read-out windows, and spectral setups.
;
dates=analyzegravityendproducts()
files=analyzegravityendproducts(dates(0))
for i=1,n_elements(dates)-1 do files=[files,analyzegravityendproducts(dates(i))]
;
if strlen(files(0)) eq 0 then return,'No GRAVITY raw files'
;
n=n_elements(files)
configurations=strarr(n)
;
for i=0,n-1 do begin
;
	fitsfile=obj_new('fitsfile',files(i))
	prihead=fitsfile->prihead()
	obsstart=strtrim(prihead->getpar('OBS START'))
	ftobjnam=strtrim(prihead->getpar('FT ROBJ NAME'))
	scobjnam=strtrim(prihead->getpar('INS SOBJ NAME'))
	if ftobjnam ne scobjnam then insmode1='DUAL' else insmode1='SINGLE'
	if insmode1 eq 'DUAL' and strpos(files(i),'ASTROREDUCED') ge 0 $
		then insmode1='ASTRO'
	insmode2=strtrim(prihead->getpar('INS SPEC RES'))
	insmode3=strtrim(prihead->getpar('INS POLA MODE'))
	dprtype=strtrim(prihead->getpar('PRO SCIENCE'))
	j=where(dprtype eq 'T',count) 
	if count gt 0 then dprtype(j)='SCI'
	j=where(dprtype eq 'F',count) 
	if count gt 0 then dprtype(j)='CAL'
;
	obstimes=strmid(obsstart,11,8)
	obsstart=strmid(obsstart,0,10)
;	Reduced file date stamps should refer to UT date, not civil date
	if hms2h(obstimes) gt 12 then obsstart=nextdate(obsstart)
	
	words=nameparse(insmode1,'_')
	insmode1=words(0)
;
	insmode=insmode1+'-'+insmode2+'-'+insmode3
;
	station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
	station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
	station_3=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION3')))
	station_4=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION4')))
;
	configurations(i)=obsstart+'_'+insmode $
				 +'_'+station_1 $
				 +'-'+station_2 $
				 +'-'+station_3 $
				 +'-'+station_4
;
	obj_destroy,fitsfile
;
endfor
;
return,unique(configurations)
;
end
;-------------------------------------------------------------------------------
function gravityconfigids,obht
;
; Return a unique set of configids currently loaded.
;
n=n_elements(obht)
configurations=strarr(n)
;
for i=0,n-1 do begin

	obsstart=obht(i).OBSSTART
	insmode1=obht(i).ISSFEEDMODE
	insmode2=obht(i).INSSPEC
	insmode3=obht(i).INSPOLAMODE
;
	obstimes=strmid(obsstart,11,8)
	obsstart=strmid(obsstart,0,10)
;	Reduced file date stamps should refer to UT date, not civil date
	if hms2h(obstimes) gt 12 then obsstart=nextdate(obsstart)
	words=nameparse(insmode1,'_')
	insmode1=words(0)
;
	insmode=insmode1+'-'+insmode2+'-'+insmode3
;
	station_1=vlti_stationid(strtrim(obht(i).ISSCONFSTATION1,2))
	station_2=vlti_stationid(strtrim(obht(i).ISSCONFSTATION2,2))
	station_3=vlti_stationid(strtrim(obht(i).ISSCONFSTATION3,2))
	station_4=vlti_stationid(strtrim(obht(i).ISSCONFSTATION4,2))
;
	configurations(i)=strcompress(obsstart+'_'+insmode $
				 +'_'+station_1 $
				 +'-'+station_2 $
				 +'-'+station_3 $
				 +'-'+station_4,/remove_all)
endfor
;
return,configurations
;
end
;-------------------------------------------------------------------------------
pro selectgravityfiles,all=all
;
; Displays the Gorgonzola GUI to select files. Stores the results in
; common block. If all are requested, return just those corresponding
; to the currently selected configuration, without displaying the GUI.
;
common GravityGuiWids,gravity_wid,files_wid,configids_wid,pipeline_wid
common GravityPipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common GravityObservation,obht,objindex
common GravityOptions,gravity_options
;
keys=['OBS START','FT ROBJ NAME','INS SOBJ NAME','ISS FEED MODE', $
      'INS SPEC','INS POLA MODE', $
      'PRO REC1 PARAM17 VALUE', $
      'PRO REC1 PARAM18 VALUE', $
      'ISS CONF STATION1','ISS CONF STATION2','ISS CONF STATION3','ISS CONF STATION4']
print,'Please wait, reading files...'
searchstrings=nameparse(searchstrng)
if keyword_set(all) then begin
	first=1
	for i=0,n_elements(searchstrings)-1 do begin
		ob_i=obj_new('fitsfilelist',dir=productsdir, $
			searchString=searchstrings(i))
		if heap_refcount(ob_i) eq 0 then continue
		obht_i=ob_i->getHeaderKeys(keys)
		nobht_i=n_elements(obht_i)
;		ob_i->setHeader,obht_i,keys
		obsfiles=ob_i->files()
		obht_i.filename=obsfiles
		obht_i.issfeedmode='SINGLE'
		for j=0,nobht_i-1 do begin
			if obht_i(j).ftrobjname ne obht_i(j).inssobjname $
			then obht_i(j).issfeedmode='DUAL'
			if obht_i(j).issfeedmode eq 'DUAL' $
				and strpos(obht_i(j).filename,'ASTRO') ge 0 $
				then obht_i(j).issfeedmode='ASTRO'
		endfor
		if first then begin
			ob=ob_i
			obht=obht_i
			nobht=nobht_i
			first=0
		endif else begin
			ob=[ob,ob_i]
			obht=[obht,obht_i]
			nobht=nobht+nobht_i
		endelse
		obj_destroy,ob_i
	endfor
endif else begin
	ob=obj_new('fitsfilelist',dir=productsdir, $
		guikey=keys,guisearch="'*.fits'")
; 	The header table initially contains all files
	obht=ob->headertable()
	nobht=n_elements(obht)
	obsfiles=ob->files()
	files=obht.filename
	n=n_elements(obsfiles)
	if strlen(obsfiles(0)) eq 0 then n=0
	if n gt 0 then begin
		index=intarr(n)
		for i=0,n-1 do begin
			words=nameparse(obsfiles(i))
			index(i)=where(files eq words(0))
		endfor
		obht=obht(index)
		obht.filename=obsfiles
	endif
	obj_destroy,ob
endelse
;
; Reduce field ISSFEEDMODE to DUAL or SINGLE
; for i=0,n_elements(obht)-1 do obht(i).issfeedmode=singledual(obht(i).issfeedmode)
; for i=0,n_elements(obht)-1 do obht(i).insmode=singledual(obht(i).prorec1raw1catg)
;
; For option all, we do however select according to the current configid
if keyword_set(all) then begin
	configurations=gravityconfigids(obht)
	index=where(configurations eq configid,count)
	if count gt 0 then obht=obht(index)
endif
;
; Select files according to viscorrection preference (VFACTOR/NONE)
obht.prorec1param17value=strcompress(obht.prorec1param17value,/remove_all)
obht.prorec1param18value=strcompress(obht.prorec1param18value,/remove_all)
if gravity_options.viscorrection eq 'VFACTOR' then begin
	index=where(obht.prorec1param17value eq 'VFACTOR' or $
		    obht.prorec1param18value eq 'VFACTOR',count)
	if count gt 0 then obht=obht(index)
endif
if gravity_options.viscorrection eq 'NONE' then begin
	index=where(obht.prorec1param17value eq 'NONE' or $
		    obht.prorec1param18value eq 'NONE',count)
	if count gt 0 then obht=obht(index)
endif
;
; The obsindex shall refer exactly to those files needed for a visibility
objindex=indgen(n_elements(obht))	; all files for GRAVITY
;
; Sort with increasing time
si=sort(obht.obsstart)
obht=obht(si)
;
; Display selected files in list widget
if n_elements(files_wid) eq 0 then files_wid=0l
if widget_info(files_wid,/valid) then $
widget_control,files_wid,set_value=obht.filename,set_uvalue=obht.filename
;
end
;-------------------------------------------------------------------------------
pro gravitygenconfig,filename,genconfig
;
; Not used currently!
;
; Open specified file, and extract information into given genconfig.
; This does not include channel wavelengths and widths.
;
; Note on FINITO: there is a fixed mapping of IPs and FINITO channels:
; IP1 -> F_CH2
; IP3 -> F_CH0
; IP5 -> F_CH1
; The reference channel of FINITO is 0, i.e. the tracking baselines are 
; IP1-IP3 and IP5-IP3.
;
; Open file
fitsfile=obj_new('fitsfile',filename)
prihead=fitsfile->prihead()
;
; Read configuration
detdit=strtrim(prihead->getpar('DET2 SEQ1 DIT'),2)
insmode1=singledual(strtrim(prihead->getpar('ISSFEEDMODE')))
insmode2=strtrim(prihead->getpar('INS POLA MODE'))
insmode3=strtrim(prihead->getpar('INS SPEC RES'))
dprtype=strtrim(prihead->getpar('PRO SCIENCE'))
j=where(dprtype eq 'T',count) 
if count gt 0 then dprtype(j)='SCI'
j=where(dprtype eq 'F',count) 
if count gt 0 then dprtype(j)='CAL'
insmode=insmode1+'-'+insmode2+'-'+insmode3
configid=''
;
station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
station_3=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION3')))
station_4=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION4')))
stations=[station_1,station_2,station_3,station_4]
telescope_1=strtrim(prihead->getpar('ISS CONF T1NAME'))
telescope_2=strtrim(prihead->getpar('ISS CONF T2NAME'))
telescope_3=strtrim(prihead->getpar('ISS CONF T3NAME'))
telescope_4=strtrim(prihead->getpar('ISS CONF T4NAME'))
;
delayline_1=strtrim(prihead->getpar('ISS CONF DL1'))
delayline_2=strtrim(prihead->getpar('ISS CONF DL2'))
delayline_3=strtrim(prihead->getpar('ISS CONF DL3'))
delayline_4=strtrim(prihead->getpar('ISS CONF DL4'))
delaylines=[delayline_1,delayline_2,delayline_3,delayline_4]
delaylineids=strmid(delaylines,2,1)
refname=prihead->getpar('DEL REF NAME')
bcinput_1=long(strtrim(prihead->getpar('ISS CONF INPUT1'))) > 1
bcinput_2=long(strtrim(prihead->getpar('ISS CONF INPUT2'))) > 3
bcinput_3=long(strtrim(prihead->getpar('ISS CONF INPUT3'))) > 5
bcinput_4=long(strtrim(prihead->getpar('ISS CONF INPUT4'))) > 7
bcinputs=[bcinput_1,bcinput_2,bcinput_3,bcinput_4]
;
; Close file
obj_destroy,prihead
obj_destroy,fitsfile
;
; Store configuration in genconfig
index=where(strlen(stations) gt 0,numsid)
genconfig.numsid=numsid
genconfig.stationid(0)=stations(0)
genconfig.stationid(1)=stations(1)
genconfig.stationid(2)=stations(2)
genconfig.stationid(3)=stations(3)
genconfig.baselineid(0,*)=genconfig.stationid(0)+'-'+genconfig.stationid(1)
genconfig.baselineid(1,*)=genconfig.stationid(0)+'-'+genconfig.stationid(2)
genconfig.baselineid(2,*)=genconfig.stationid(0)+'-'+genconfig.stationid(3)
genconfig.baselineid(3,*)=genconfig.stationid(1)+'-'+genconfig.stationid(2)
genconfig.baselineid(4,*)=genconfig.stationid(1)+'-'+genconfig.stationid(3)
genconfig.baselineid(5,*)=genconfig.stationid(2)+'-'+genconfig.stationid(3)
genconfig.delaylineid=delaylineids
ref_count=0
if not isnumeric(refname) then $
ref_index=where(delaylineids eq long(strmid(refname,2,1)),ref_count)
if ref_count eq 0 then ref_index=0
genconfig.refstation=ref_index+1
genconfig.bcinputid=bcinputs
genconfig.numoutbeam=2
genconfig.numbaseline=((numsid-1)*numsid)/2
genconfig.instrcohint=detdit*1000	; ms
;
genconfig.configid=insmode+'+'+configid+'+' $
;	+genconfig.stationid(0)+'-'+genconfig.stationid(1)
	+strjoin(genconfig.stationid,'-')
;
if strpos(telescope_1,'UT') ge 0 then genconfig.diameter=8 $
				 else genconfig.diameter=1.8
;
end
;-------------------------------------------------------------------------------
function gravitystarids,scans
;
; Not used currently!
;
; Adopt a single starid (the first) for stars of the same name.
; This is important for the OBJ designation in case it was derived
; from apparent coordinates.
;
stars=unique(scans.star)
for i=0,n_elements(stars)-1 do begin
	index=where(scans.star eq stars(i))
	scans(index).starid=scans(index(0)).starid
endfor
;
return,scans
;
end
;************************************************************************Block 3
function gravitygui_event,event
;
; This is the call back for buttons on the main MyGravityGui.
;
common GravityGuiWids,gravity_wid,files_wid,configids_wid,pipeline_wid
common GravityPipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common GravityObservation,obht,objindex
common GravityOptions,gravity_options
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
case event.value of
;
'Show/select product files':begin
;		Clean desktop a little
		wdall
		set_screen
		!x.range=0
		!y.range=0
		!p.psym=0
;		Store current OYSTER observation
		if n_elements(geninfo) ne 0 then begin
;			storenight,11
;			save,recipes,filename='Recipes_'+genconfig.configid+'.xdr'
		endif
;
		selectgravityfiles
;
;		If a complete observation was selected, get additional info
		if objindex(0) ge 1 then begin 
		fitsfiles=obht(objindex(0)).filename
		fitsfile=obj_new('fitsfile',fitsfiles(0))
		prihead=fitsfile->prihead()
		airmass=strtrim(prihead->getpar('ISS AIRM END'))
		seeing=strtrim(prihead->getpar('ISS AMBI FWHM'))
		dispersion=strtrim(prihead->getpar('INS MODE'))
		obj_destroy,prihead
		obj_destroy,fitsfile
		print,'Target: ',obht(objindex(0)).ftrobjname
		print,'Airmass:',airmass,', seeing:',seeing
;		Load corresponding OYSTER scans structure
		if n_elements(geninfo) ne 0 then begin
			currentconfig=unique(gravityconfigids(obht(objindex)))
			index=where(geninfo.configid eq currentconfig,count)
;			This is currently not strictly needed
			if count eq 1 then begin
				datum=geninfo(index).date
				loadnight,datum,currentconfig
				restore,'Recipes_'+currentconfig+'.xdr'
			endif
		endif
		endif
		end
'GRAVITY':	begin
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'FT.ROBJ.NAME '+ $
			'DET2.SEQ1.DIT '+ $
			'DET2.NDIT ',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_gravity(r),width=142
		end
'FT':	begin
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'FT.ROBJ.NAME '+ $
			'QC.PHASE_FT12.RMS  '+ $
			'QC.PHASE_FT13.RMS  '+ $
			'QC.PHASE_FT14.RMS  '+ $
			'QC.PHASE_FT23.RMS  '+ $
			'QC.PHASE_FT24.RMS  '+ $
			'QC.PHASE_FT34.RMS  ',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_gravity(r),width=142
		end
'COUDE':	begin
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'FT.ROBJ.NAME '+ $
			'COU.AO2.T0_MEAN COU.AO2.FWHM_MEAN ',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_gravity(r),width=142
		end
'AMBIENT':	begin
		selectgravityfiles,/all
		spawn,'dfits '+strjoin(obht.filename,' ')+' | fitsort '+ $
			'OBS.NAME '+ $
			'FT.ROBJ.NAME '+ $
			'ISS.AMBI.FWHM.START ISS.AMBI.FWHM.END '+ $
			'ISS.AMBI.TAU0.START ISS.AMBI.TAU0.END '+ $
			'ISS.AMBI.WINDSP',r
;			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=dfits_gravity(r),width=142
		end
'OYSTER': 	begin
		mygravitypipe
		oyster
		end
'Pipeline':	begin
		!p.multi=[0,2,2]
		mygravitypipe
		end
'Delete?':	begin
		for i=0,n_elements(obht.filename)-1 do $
			print,obht(i).filename
		answer=' '
		read,prompt='Delete these files? (yes/no): ',answer
		if answer eq 'yes' then spawn,'rm -f '+strjoin(obht.filename,' ')
		if answer ne 'yes' then print,'Files not removed.'
		end
'Cleanup':	begin
;		Remove end products if more recent ones exist
		files=unique(specname(obht.filename))
		for i=0,n_elements(files)-1 do begin
			j=where(strmatch(obht.filename,'*'+files(i)))
			f=obht(j).filename
			n=n_elements(f)
			u_len_f=unique(strlen(f))
			if n gt 1 and n_elements(u_len_f) eq 1 then begin
				si=sort(f)
				f=f(si)
				spawn,'rm -f '+f(0:n-2)
			endif
		endfor
;		Remove directories which now only contain sky files
		f=file_search('.','*SINGLE_SKY*')
		dirs=strarr(n_elements(f))
		for i=0,n_elements(f)-1 do begin
			words=nameparse(f(i),'/')
			dirs(i)=strjoin(words(0:n_elements(words)-2),'/')
		endfor
		dirs=unique(dirs(where(strlen(dirs) ne 0)))
		for i=0,n_elements(dirs)-1 do begin
			cd,dirs(i),current=old_dir
			f_scical=file_search('*VIS.fits')
			f_sky=file_search('*_SKY*.fits')
			f_all=file_search('*')
			cd,old_dir
;			if strlen(f_scical) eq 0 then spawn,'rm -rf '+dir
;			Better to look for dirs with all files of type SKY
			if n_elements(f_sky) eq n_elements(f_all) then $
				spawn,'rm -rf '+dirs(i)
		endfor
		f=file_search('.','README')
		if strlen(f(0)) ne 0 then spawn,'rm -f '+strjoin(f,' ')
;		Remove empty directory trees
		spawn,'du reflex_end_products',r
		for i=0,n_elements(r)-1 do begin
			words=nameparse(r(i))
			if long(words(0)) lt 10 then print,'rm -rf '+words(1)
		endfor
		selectgravityfiles,/all
		end
'Rename':	begin
		for i=0,n_elements(obht.filename)-1 do begin
			fitsfile=obj_new('fitsfile',obht(i).filename)
			prihead=fitsfile->prihead()
			dateobs=strmid(strtrim(prihead->getpar('DATE-OBS')),0,23)
			obj_destroy,fitsfile
			outfile='GRAVITY.'+dateobs+'.fits'
			if obht(i).filename ne outfile then begin
				command='mv '+obht(i).filename+' '+outfile
				print,command
				spawn,command
			endif
		endfor
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
function gravitygui_productsdirs,event
;
; Function called when starting gravity gui to return lists of directories
; with GRAVITY data. It is first called with the rootdatadir as "event".
;
; File names to recognize in reflex_end_products and reflex_tmp_products:
; OBS.NAME_YYYY-MM-DDTHH:MM:SS_SINGLE_SCI_VIS.fits
; OBS.NAME_YYYY-MM-DDTHH:MM:SS_SINGLE_CAL_VIS.fits
; OBS.NAME_YYYY-MM-DDTHH:MM:SS_DUAL_SCI_VIS.fits
; OBS.NAME_YYYY-MM-DDTHH:MM:SS_DUAL_CAL_VIS.fits
; OBS.NAME_YYYY-MM-DDTHH:MM:SS_ASTROREDUCED.fits
; OBS.NAME_YYYY-MM-DDTHH:MM:SS_SINGLE_SCI_VIS_singlesciviscalibrated.fits
; OBS.NAME_YYYY-MM-DDTHH:MM:SS_DUAL_SCI_VIS_singlesciviscalibrated.fits
;
common GravityPipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common GravityGuiWids,gravity_wid,files_wid,configids_wid,pipeline_wid
;
; Set products directory
r=size(event)
; Check for structure event (indicates call via menu selection)
if r(n_elements(r)-2) eq 8 then begin
	widget_control,event.id,get_uvalue=productsdirs
	productsdir=productsdirs(event.index)
endif else productsdir=event
;
; Reflex search strings
r_string="'*_*_???_VIS.fits'"
r_string="'*_SINGLE_???_VIS*.fits'"
r_string="'*_*_???_VIS.fits'"
r_string="'*_ASTROREDUCED*.fits'"
r_string="'*_*_???_VIS.fits' '*_SCI_VIS_*sciviscalibrated.fits' '*_ASTROREDUCED*.fits'"
; Gasgano search strings
g_string="'*single???vis_????.fits'"
;
case productsdir of
'reflex_end_products':	begin
		productsdir=rootdatadir+'/reflex_end_products'
		searchstrng=r_string
		end
'reflex_tmp_products':	begin
		productsdir=rootdatadir+'/reflex_tmp_products'
		searchstrng=r_string
		end
'gasgano':	begin
		productsdir=rootdatadir+'/gasgano'
		searchstrng=g_string
		end
else:		begin
;		Come here the first time to see which data are available
		IF r(n_elements(r)-2) ne 8 THEN BEGIN 
		productsdirs=''
		productsdir=rootdatadir
;		Check working directory for FITS files
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		searchstrng='*.fits'
		searchstrng='\*.fits'
;		spawn,'find -L . -iname '+searchstrng,f
;		spawn,'ls '+searchstrng0,f
		spawn,'ls *.fits',f
		if strlen(f(0)) ne 0 then productsdirs=[productsdirs,productsdir]
		endif
;		Check for Reflex end products
		productsdir='reflex_end_products'
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		searchstrng=r_string
;		spawn,'find -L '+productsdir+' -iname '+searchstrng,f
		f=find_files(productsdir,searchstrng)
		index=where(strlen(f) ne 0,count)
		if count ge 1 then productsdirs=[productsdirs,productsdir]
;		if strlen(f(0)) ne 0 then productsdirs=[productsdirs,productsdir]
		endif
;		Check for Reflex tmp products
		productsdir='reflex_tmp_products'
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		searchstrng=r_string
;		spawn,'find -L '+productsdir+' -iname '+searchstrng,f
		f=find_files(productsdir,searchstrng)
		index=where(strlen(f) ne 0,count)
		if count ge 1 then productsdirs=[productsdirs,productsdir]
;		if strlen(f(0)) ne 0 then productsdirs=[productsdirs,productsdir]
		endif
;		Check for gasgano products
		productsdir='gasgano'
		if strlen(file_search(productsdir,/test_dir)) ne 0 then begin
		searchstrng=g_string
;		spawn,'find -L '+productsdir+' -iname '+searchstrng,f
		f=find_files(productsdir,searchstrng)
		index=where(strlen(f) ne 0,count)
		if count ge 1 then productsdirs=[productsdirs,productsdir]
;		if strlen(f(0)) ne 0 then productsdirs=[productsdirs,productsdir]
		endif
		n=n_elements(productsdirs)
		if n gt 1 then productsdirs=productsdirs(1:n-1)
		return,productsdirs
		ENDIF ELSE BEGIN
		searchstrng='*.fits'
		searchstrng='\*.fits'
		ENDELSE
		end
endcase
;
if widget_info(configids_wid,/valid) then begin
	configids=analyzegravityconfigids()
	widget_control,configids_wid,set_value=configids,set_uvalue=configids
	configid=configids(0)
	widget_control,configids_wid,/sensitive
endif
;
return,productsdir
;
end
;-------------------------------------------------------------------------------
function gravitygui_configid,event
;
; Callback for a selection of a configuration, returns all matching file names.
;
common GravityPipeline,rootdatadir,productsdir,searchstrng,configid,recipes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common GravityGuiWids,gravity_wid,files_wid,configids_wid,pipeline_wid
;
; Set configid
r=size(event)
if r(n_elements(r)-2) eq 8 then begin
	widget_control,event.id,get_uvalue=configids
	configid=configids(event.index)
endif else configid=event
selectgravityfiles,/all
;
end
;-------------------------------------------------------------------------------
function files_event,event
;
; When clicking on a file, start fv on it.
;
widget_control,event.id,get_uvalue=files
if n_elements(files) ne 0 then begin
print,'Starting fv on ',files(event.index)
spawn,'fv -cmap 2 '+files(event.index)+' &'
endif
;
end
;-------------------------------------------------------------------------------
function gravitygui_options,event
;
common GravityOptions,gravity_options
;
widget_control,event.id,get_uvalue=command,get_value=values
gravity_options.viscorrection=values(event.index)
;if command eq 'VFACTOR' then gravity_options.viscorrection=values(event.index)
;if command eq 'NONE' then gravity_options.errors=values(event.index)
;
end
;-------------------------------------------------------------------------------
pro mygravitygui
;
; Creates the MyGravityGui main widget.
;
common GravityGuiWids,gravity_wid,files_wid,configids_wid,pipeline_wid
common GravityOptions,gravity_options
common GravityObservation,obht,objindex
common GravityPipeline,rootdatadir,productsdir,searchstrng,configid,recipes
;
; Initialization and cleanup
if n_elements(gravity_wid) eq 0 then gravity_wid=0L
if n_elements(configids_wid) eq 0 then configids_wid=0L
if widget_info(gravity_wid,/valid) then widget_control,gravity_wid,/destroy
;
; Rootdatadir shall be a directory which may contain the following folders:
; reflex_end_products, reflex_tmp_products, gasgano. 
; It may also contain reduced files.
spawn,'pwd',rootdatadir & rootdatadir=rootdatadir(0) & productsdir=''
;
; Allocate options
gravity_options=alloc_gravity_options()
;
gravity_wid=widget_base(/column,title=rootdatadir)
;
row_wid=widget_base(gravity_wid,/row,/frame)
;
templates=['Show/select product files']
button_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='gravitygui_event',/return_name)
;
; Get list of directories with data products available
productsdirs=gravitygui_productsdirs(rootdatadir)
if strlen(productsdirs(0)) eq 0 then return
productsdir=productsdirs(0)
opmenup_wid=widget_droplist(row_wid,event_func='gravitygui_productsdirs', $
	uvalue=productsdirs,value=productsdirs)
; if productsdir ne rootdatadir then $
productsdir=gravitygui_productsdirs(productsdir)
;
templates=['GRAVITY','FT','COUDE','AMBIENT']
templates=['GRAVITY','FT','AMBIENT']
button_wid=cw_bgroup(gravity_wid,/row,templates, $
	event_func='gravitygui_event',/return_name)
;
files_wid=widget_list(gravity_wid,value=strarr(10)+' ', $
	event_func='files_event', $
	xsize=15,ysize=10)
;
templates=['Delete?','Cleanup']
button_wid=cw_bgroup(gravity_wid,/row,templates, $
	event_func='gravitygui_event',/return_name)
;
col_wid=widget_base(gravity_wid,/col,/frame)
;
configids=analyzegravityconfigids()
configids_wid=widget_droplist(col_wid,event_func='gravitygui_configid', $
	uvalue=configids,value=configids)
configid=configids(0)
widget_control,configids_wid,sensitive=0
;
row_wid=widget_base(col_wid,/row)
;
templates=['OYSTER']
pipeline_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='gravitygui_event',/return_name)
;
row2_wid=widget_base(col_wid,/row,/frame)
;
viscorrection=['VFACTOR','NONE']
opmenue_wid=widget_droplist(row2_wid,event_func='gravitygui_options', $
	uvalue='viscorrection',value=viscorrection)
;
widget_control,gravity_wid,/realize
xmanager,'mygravitygui',gravity_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro mygravitypipe
;
; Processes all GRAVITY files returned by selectgravityfiles and stores results 
; in OYSTER scans structure. Data should be of same configuration.
;
; Mygravitypipe uses the instructions contained in the recipes structure for 
; the selection of the reduction method. The default recipes are created
; and stored when initializing OYSTER.
;
common GravityObservation,obht,objindex
common GravityGuiWids,gravity_wid,files_wid,configids_wid,pipeline_wid
common GravityOptions,gravity_options
common GravityPipeline,rootdatadir,productsdir,searchstrng,configid,recipes
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ModelFit,parameters,ds_options
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
common AmbientData,AmbientTable
;
common AstroData,StarId,AstroTime,AstroMJD,AstroFlag,StaIndex,GDELAY_DISP, $
       OPD_MET_TELFC_MCORR,OPD_MET_FC_CORR,GDELAY,GDELAY_FT,Z_DISPI,Z_DISPJ
;
RAD=180/!pi
;
selectgravityfiles,/all
;
; In DUAL mode, the targets in the SCI and REF channels are different
if obht(0).issfeedmode eq 'DUAL' then dual=2 else dual=1
if obht(0).issfeedmode eq 'ASTRO' then astro=1 else astro=0
;
; Determine how many observations there are
count_obs=n_elements(objindex)
;
; Load first file
print,'Reading first file: ',obht(0).filename+'...'
if astro then begin
	d=mrdfits(obht(0).filename,4,header)	; OIVIS table
endif else begin
	get_oifits,obht(0).filename
	gravity_compact
	ns=n_elements(scans)
endelse
scantable.starid=scans.starid
;
; Read header for more information
d=mrdfits(obht(0).filename,0,header)
scans.r0=(float(fitshparse(header,'FWHMSTART')) $
	 +float(fitshparse(header,'FWHMEND')))/2
scans.t0=(float(fitshparse(header,'TAU0START')) $
	 +float(fitshparse(header,'TAU0END')))/2
scans.pres=float(fitshparse(header,'AMBIPRES'))
scans.rhum=float(fitshparse(header,'AMBIRHUM'))
scans.temp=float(fitshparse(header,'AMBITEMP'))
scans.int_time=float(fitshparse(header,'DET2SEQ1DIT')) $
	      *float(fitshparse(header,'DET2NDIT'))
;
scans_all=scans
scantable_all=scantable
;
scans_cal_all=intarr(ns)
if dual eq 2 then scans_cal_all(indgen(ns/2)*2)=1
if strpos(strupcase(obht(0).filename),'CAL') ge 0 then $
if dual eq 1 then scans_cal_all(*)=1 else scans_cal_all=fix(scans_cal_all eq 0)
;
FOR iobs=1,count_obs-1 DO BEGIN
	print,'Reading file: ',obht(iobs).filename+'...'
	if astro then begin
		d=mrdfits(obht(0).filename,4,header)
	endif else begin
		get_oifits,obht(iobs).filename
		gravity_compact
		ns=n_elements(scans)
;
		d=mrdfits(obht(iobs).filename,0,header)
		scans.r0=(float(fitshparse(header,'FWHMSTART')) $
			 +float(fitshparse(header,'FWHMEND')))/2
		scans.t0=(float(fitshparse(header,'TAU0START')) $
			 +float(fitshparse(header,'TAU0END')))/2
		scans.pres=fitshparse(header,'AMBIPRES')
		scans.rhum=fitshparse(header,'AMBIRHUM')
		scans.temp=fitshparse(header,'AMBITEMP')
		scans.int_time=float(fitshparse(header,'DET2SEQ1DIT')) $
			      *float(fitshparse(header,'DET2NDIT'))
;	
		scans_all=[scans_all,scans]
		scantable_all=[scantable_all,scantable]
;
		scans_cal=intarr(ns)
		if dual eq 2 then scans_cal(indgen(ns/2)*2)=1
		if strpos(strupcase(obht(iobs).filename),'CAL') ge 0 then $
		if dual eq 1 then scans_cal(*)=1 $
			     else scans_cal=fix(scans_cal eq 0)
		scans_cal_all=[scans_cal_all,scans_cal]
	endelse
ENDFOR
stations=strarr(4)
for i=0,3 do begin
	stations(i) $
		=fitshparse(header,'ISSCONFSTATION'+string(i+1,format='(i1)'))
	j=where(genconfig.stationid eq 'A'+stations(i))
	if j(0) eq -1 then $
	j=where(genconfig.stationid eq 'U'+stations(i))
	genconfig.siderostatid(j)=i+1
endfor
genconfig.beamcombinerid=51
if max(genconfig.numspecchan) gt 30 then genconfig.beamcombinerid=52
if max(genconfig.numspecchan) gt 300 then genconfig.beamcombinerid=53
;
scans=scans_all
scans_cal=scans_cal_all
scantable=scantable_all
si=sort(scans.time)
scans=scans(si)
scans_cal=scans_cal(si)
scantable=scantable(si)
;
; Fix a problem which occurs once in a while
index=where(scans.star eq 'Name',count)
if count gt 0 then scans(index).star=scans(index).starid
;
; Store names for stars in StarTable
list_stars,starids,names
get_startable,starids,names=names
;
; Identify calibrators in StarTable
for i=0,n_elements(startable)-1 do begin
	j=where(scans.star eq startable(i).name,count)
	if count gt 0 then begin
	scans(j).starid=startable(i).starid
	if scans_cal_all(j(0)) eq 1 then startable(i).bflag='C' $
				    else startable(i).bflag='.'
	endif
endfor
scantable.starid=scans.starid
;
words1=nameparse(genconfig.configid,'_')
words2=nameparse(instrument_id(systemid),'-')
genconfig.configid=words2(1)+'_'+words1(1)
geninfo.configid=genconfig.configid
filestub=genconfig.date+'_'+genconfig.configid
filestub=configid
put_xdr,filestub
;
; Also save in OIFITS
; put_oifits,filestub+'.fits'
;
; Cleanup
gravity_cleanup
;
end
;-------------------------------------------------------------------------------
pro gravity_cleanup
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
gravity_compact
;
; Flag FT data (OB=2)
; scans(*).vissqerr(1,*,*)=-1
; scans(*).vissqeerr(1,*,*)=-1
; scans(*).vissqcerr(1,*,*)=-1
; scans(*).vissqecerr(1,*,*)=-1
; scans(*).tripleamperr(4:7,*,*)=-1
; scans(*).tripleampeerr(4:7,*,*)=-1
; scans(*).tripleampcerr(4:7,*,*)=-1
; scans(*).tripleampecerr(4:7,*,*)=-1
; scans(*).triplephasecerr(4:7,*,*)=-1
; scans(*).triplephaseerr(4:7,*,*)=-1
; scans(*).triplephasecerr(4:7,*,*)=-1
;
; Find and flag bad data
medianob
set_compltriplem
outliers,10
;
; Flag data with huge error bars
e=scans.vissqcerr
index=where(e gt 0.1,count)
if count gt 0 then e(index)=-e(index)
scans.vissqcerr=e
ea=scans.tripleampcerr
ep=scans.triplephasecerr
index=where(e gt 0.1,count)
if count gt 0 then begin
	ea(index)=-ea(index)
	ep(index)=-ep(index)
endif
scans.tripleampcerr=ea
scans.triplephasecerr=ep
;
end
;-------------------------------------------------------------------------------
pro gravity_compact
;
; Move OIVIS2 and OI_T3	time stamps which are different due to frame selection
; to common grid. Thus, data for one OB should end up with one time stamp.
;
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
i0=0
i=1
ns=n_elements(scans)
;
repeat begin
if scans(i<(ns-1)).time-scans(i0).time gt 5*60 or i eq ns then begin
;
index0=i0+indgen(i-i0)
targets=unique(scans(index0).starid)
;
FOR j=0,n_elements(targets)-1 DO BEGIN
;
jndex=where(scans(index0).starid eq targets(j))
jndex0=index0(jndex)
s=scans(jndex0(0))
t=scantable(jndex0(0))
if n_elements(jndex0) gt 1 then begin
for ob=0,genconfig.numoutbeam-1 do begin
	nc=genconfig.numspecchan(ob)
	for bl=0,genconfig.numbaseline(ob)-1 do begin
		nf=scans(jndex0).vissqcerr(ob,0:nc-1,bl)*0+1
		vissqc=scans(jndex0).vissqc(ob,0:nc-1,bl)
		vissqcerr=scans(jndex0).vissqcerr(ob,0:nc-1,bl)
		diffphasec=scans(jndex0).diffphasec(ob,0:nc-1,bl)
		diffphasecerr=scans(jndex0).diffphasecerr(ob,0:nc-1,bl)
		index=where(vissqcerr lt 0,count) ; note "lt"
		if count gt 0 then begin
			vissqc(index)=0
			vissqcerr(index)=0
			diffphasec(index)=0
			diffphasecerr(index)=0
			nf(index)=0
		endif
;		index=where(diffphasecerr lt 0,count)
;		if count gt 0 then begin
;			diffphasec(index)=0
;			diffphasecerr(index)=0
;			nf(index)=0
;		endif
		vissqc=total(vissqc,3)/(total(nf,3) > 1)
		vissqcerr=total(vissqcerr,3)/(total(nf,3) > 1)
		diffphasec=total(diffphasec,3)/(total(nf,3) > 1)
		diffphasecerr=total(diffphasecerr,3)/(total(nf,3) > 1)
		index=where(total(nf,3) eq 0,count)
		if count gt 0 then begin
			vissqcerr(index)=-1
			diffphasecerr(index)=-1
		endif
		s.vissqc(ob,0:nc-1,bl)=vissqc
		s.vissqcerr(ob,0:nc-1,bl)=vissqcerr
		s.vissqec(ob,0:nc-1,bl)=vissqc
		s.vissqecerr(ob,0:nc-1,bl)=vissqcerr
		s.vissq(ob,0:nc-1,bl)=vissqc
		s.vissqerr(ob,0:nc-1,bl)=vissqcerr
		s.vissqe(ob,0:nc-1,bl)=vissqc
		s.vissqeerr(ob,0:nc-1,bl)=vissqcerr
		s.diffphasec(ob,0:nc-1,bl)=diffphasec
		s.diffphasecerr(ob,0:nc-1,bl)=diffphasecerr
	for k=0,2 do begin
		nf=scans(jndex0).uvw(ob,0:nc-1,bl,k)*0+1
		uvw=scans(jndex0).uvw(ob,0:nc-1,bl,k)
		index=where(uvw eq 0,count)
		if count gt 0 then begin
			nf(index)=0
		endif	
		uvw=total(uvw,3)/(total(nf,3) > 1)
		index=where(total(nf,3) eq 0,count)
		if count gt 0 then uvw(index)=0
		s.uvw(ob,0:nc-1,bl,k)=uvw
	endfor
	endfor
	for ib=0,genconfig.numsid-1 do begin
		nf=scans(jndex0).photometryerr(ib,ob,0:nc-1)*0+1
		photometry=scans(jndex0).photometry(ib,ob,0:nc-1)
		photometryerr=scans(jndex0).photometryerr(ib,ob,0:nc-1)
		index=where(photometryerr le 0,count) ; note "le"
		if count gt 0 then begin
			photometry(index)=0
			photometryerr(index)=0
			nf(index)=0
		endif	
		photometry=total(photometry,4)/(total(nf,4) > 1)
		photometryerr=total(photometryerr,4)/(total(nf,4) > 1)
		index=where(total(nf,4) eq 0,count)
		if count gt 0 then photometryerr(index)=-1
		s.photometry(ib,ob,0:nc-1)=photometry
		s.photometryerr(ib,ob,0:nc-1)=photometryerr
	endfor
endfor
for tr=0,genconfig.numtriple-1 do begin
	nc=genconfig.triplenumchan(tr)
	nf=scans(jndex0).triplephasecerr(tr,0:nc-1)*0+1
	tripleampc=scans(jndex0).tripleampc(tr,0:nc-1)
	tripleampcerr=scans(jndex0).tripleampcerr(tr,0:nc-1)
	triplephasec=scans(jndex0).triplephasec(tr,0:nc-1)
	triplephasecerr=scans(jndex0).triplephasecerr(tr,0:nc-1)
	index=where(triplephasecerr lt 0,count)
	if count gt 0 then begin
		tripleampc(index)=0
		tripleampcerr(index)=0
		triplephasec(index)=0
		triplephasecerr(index)=0
		nf(index)=0
	endif	
	tripleampc=total(tripleampc,3)/(total(nf,3) > 1)
	tripleampcerr=total(tripleampcerr,3)/(total(nf,3) > 1)
	triplephasec=total(triplephasec,3)/(total(nf,3) > 1)
	triplephasecerr=total(triplephasecerr,3)/(total(nf,3) > 1)
	index=where(total(nf,3) eq 0,count)
	if count gt 0 then tripleampcerr(index)=-1
	if count gt 0 then triplephasecerr(index)=-1
	s.tripleampc(tr,0:nc-1)=tripleampc
	s.tripleampcerr(tr,0:nc-1)=tripleampcerr
	s.triplephasec(tr,0:nc-1)=triplephasec
	s.triplephasecerr(tr,0:nc-1)=triplephasecerr
	s.tripleamp(tr,0:nc-1)=tripleampc
	s.tripleamperr(tr,0:nc-1)=tripleampcerr
	s.tripleampe(tr,0:nc-1)=tripleampc
	s.tripleampeerr(tr,0:nc-1)=tripleampcerr
	s.triplephase(tr,0:nc-1)=triplephasec
	s.triplephaseerr(tr,0:nc-1)=triplephasecerr
endfor
endif
if n_elements(s2) eq 0 then begin
	s2=s
	t2=t
endif else begin
	s2=[s2,s]
	t2=[t2,t]
endelse
;
ENDFOR
;
i0=i
endif	; i-i0 gt 1
i=i+1
endrep until i gt n_elements(scans)
;
scans=s2
scantable=t2
;
end
;-------------------------------------------------------------------------------
pro gravity_reduce,files=files,fn=fn,brg=brg
;
; brg=1:flag data in Brackett-gamma line
; Flag all data of the FT channel.
; Flag data for all channels in selected telluric regions.
; Reduce (by a factor of 10), the valid data for fitting.
;
; If provided with a list of files, process these and save with extension
; _red.fits'. This option does not currently allow to pass keyword brg.
; Create list of files like this: files=file_search('GRAVI.*.fits') 
; Then reduce lengths of file names for GRAVITY SV files:
; rename _singlesciviscalibrated_red _red *_red.fits
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if not keyword_set(fn) then fn=10
;
; Process all files if specified
if keyword_set(files) then begin
	for i=0,n_elements(files)-1 do begin
		get_oifits,files(i)
		gravity_reduce,fn=fn,brg=brg
		put_oifits,basename(files(i))+'_red.fits'
	endfor
	return
endif
;
; Flag FT channel
if genconfig.numoutbeam eq 2 then begin
	scans.vissqcerr(1,*,*)=-abs(scans.vissqcerr(1,*,*))
	scans.triplephasecerr(4:7,*,*)=-abs(scans.triplephasecerr(4:7,*,*))
endif
;
; Remove Brackett-gamma line if fitting just the continuum
if keyword_set(brg) then begin
index=where(genconfig.wavelength gt 2.05e-6 $
	and genconfig.wavelength lt 2.25e-6)
scans(*).vissqcerr(0,index,*)=-abs(scans(*).vissqcerr(0,index,*))
scans(*).triplephasecerr(0:3,index)=-abs(scans(*).triplephasecerr(0:3,index))
scans(*).tripleampcerr(0:3,index)=-abs(scans(*).tripleampcerr(0:3,index))
endif
;
; Tellurics
index=where(genconfig.wavelength gt 2.000e-6 $
	and genconfig.wavelength lt 2.030e-6)
scans(*).vissqcerr(0,index,*)=-abs(scans(*).vissqcerr(0,index,*))
scans(*).triplephasecerr(0:3,index)=-abs(scans(*).triplephasecerr(0:3,index))
scans(*).tripleampcerr(0:3,index)=-abs(scans(*).tripleampcerr(0:3,index))
;
; Tellurics
index=where(genconfig.wavelength gt 2.050e-6 $
	and genconfig.wavelength lt 2.080e-6)
scans(*).vissqcerr(0,index,*)=-abs(scans(*).vissqcerr(0,index,*))
scans(*).triplephasecerr(0:3,index)=-abs(scans(*).triplephasecerr(0:3,index))
scans(*).tripleampcerr(0:3,index)=-abs(scans(*).tripleampcerr(0:3,index))
;
; Reduce by a factor of fn
index=where(genconfig.wavelength gt 1.990e-6 $
	and genconfig.wavelength lt 2.450e-6,count)
j=where(index mod fn ne 0)	
scans(*).vissqcerr(0,index(j),*)= $
	-abs(scans(*).vissqcerr(0,index(j),*))
scans(*).triplephasecerr(0:3,index(j))= $
	-abs(scans(*).triplephasecerr(0:3,index(j)))
scans(*).tripleampcerr(0:3,index(j))= $
	-abs(scans(*).tripleampcerr(0:3,index(j)))
;
end
;-------------------------------------------------------------------------------
pro gravity_fluxcal
;
; Calibrate Photometry of science targets using photometry of calibrators
; normalized with NextGen spectrum closest in Teff and log(g).
; It is recommended to have run gravity_compact before.
;
; Set up grid of wavelengths for band pass integrations
; Note that all functions in filter.pro use [nm] as units for lambda
;
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
for is=0,n_elements(startable)-1 do begin
;
teff=teff_star(startable(is).spectrum)
logg=logg_star(startable(is).spectrum)
r=limbgrid(teff,logg,lambda_atm,limbdu,flux_atm)
;
index_is=where(scans.starid eq startable.starid,count_is)
;
for ob=0,GenConfig.NumOutBeam-1 do begin
;
lambda_grid=system_config(SystemId,'GRID')
if string(lambda_grid(0)) eq 'SPECTROMETER' then begin
	grid_type='SPECTROMETER'
	lambda_grid=genconfig.wavelength(0:genconfig.numspecchan(ob)-1,ob)*1e9
	fluxes=interpol(flux_atm,lambda_atm,lambda_grid)
endif else begin
	grid_type='GRID'
	lambda_grid=lambda_atm
	ch=indgen(genconfig.numspecchan(ob))
	jndex=where(lambda_grid*1e-9 le max(genconfig.wavelength(ch,ob) $
					   +genconfig.chanwidth(ch,ob)/2) $
		and lambda_grid*1e-9 ge min(genconfig.wavelength(ch,ob) $
					   -genconfig.chanwidth(ch,ob)/2),count)
	if count gt 0 then begin
		lambda_grid=lambda_grid(jndex)
		fluxes=flux_atm(jndex)
	endif
endelse
;
lambda=lambda_grid/1d9	; convert to SI units
num_lambda=n_elements(lambda)
nu=1/lambda
;
nc=GenConfig.NumSpecChan(ob)
;
for ib=0,GenConfig.NumSid-1 do begin
;
;	Bandpass integration
	if grid_type eq 'SPECTROMETER' then begin
;		High R spectrometer
		for sc=0,NS-1 do begin
			teff(index)= $
			 interpol(flux_grid,lambda_grid,genconfig.wavelength)
;			Store stellar template fluxes
			for i=0,count_is-1 do $
			scans(index_is(i)).photometrym(ib,ob,0:nc-1)=fluxes
		endfor
	endif else begin
		for ch=0,nc-1 do begin
			filters=system_config(SystemId,'FILTERS')
			if n_elements(filters) gt 1 then filter=filters(ch,ob) $
						    else filter=filters(0)
			if filter(0) eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch,ob) $
						 else $
				tm=call_function(filter,lambda_grid)
;			if NS eq 1 then begin
;				tm=reform(tm,1,num_lambda)
;				lam=reform(lambda,1,num_lambda)
;			endif else lam=lambda
			lam=lambda
			if total(tm) eq 0 then begin
				print,'***Error(GRAVITY_FLUX): grid too coarse!'
				return
			endif
;			Store stellar template fluxes
			for i=0,count_is-1 do begin
			scans(index_is(i)).photometrym(ib,ob,ch)= $
				total(tm*fluxes)/total(tm)
;			scans(index_is(i)).photometry(ib,ob,ch)= $
;			scans(index_is(i)).photometry(ib,ob,ch)/cal
			endfor
		endfor
	endelse
;
endfor
;
endfor
endfor
;
end
;-------------------------------------------------------------------------------
function gravity_zerocounts,zcs
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
;
get_jhk
;
t=startable
;
zerocounts=fltarr(n_elements(startable),genconfig.numsid)
medflux=zerocounts
;
valid_scans=intarr(n_elements(scans),genconfig.numsid)+1
;
photometry=reform(scans.photometry(*,0,*))
photometryerr=reform(scans.photometryerr(*,0,*))
for i=0,n_elements(scans)-1 do begin
	photometry(*,*,i)=photometry(*,*,i)/scans(i).int_time
	for j=0,genconfig.numsid-1 do begin
	index=where(photometryerr(j,*,i) lt 0,count)
	if count eq n_elements(photometry(j,*,i)) then valid_scans(i,j)=0
	endfor
endfor
;
for i=0,n_elements(startable)-1 do begin
	for j=0,genconfig.numsid-1 do begin
		index=where(scans.starid eq startable(i).starid $
			and valid_scans(*,j) eq 1,count)
		if count gt 0 then begin
			medflux(i,j)=median(photometry(j,*,index))
			zerocounts(i,j)=10^(startable(i).mk/2.5)*medflux(i,j)
		endif
	endfor
endfor
;
zct=total(zerocounts,2)
index=where(zct gt 0,count)
if count gt 0 then zc=zerocounts(index,*)
zcs=fltarr(genconfig.numsid)
for i=0,genconfig.numsid-1 do zcs(i)=mean(zc(*,i))
;
return,zerocounts
;
end
;-------------------------------------------------------------------------------
pro gravity_brg
;
; Flag data for all channels outside the Br-gamma region (2.157-2.175 microns)
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
index=where(genconfig.wavelength lt 2.157e-6 $
	or genconfig.wavelength gt 2.175e-6)
;
scans(*).vissqcerr(*,index,*)=-abs(scans(*).vissqcerr(*,index,*))
scans(*).triplephasecerr(*,index)=-abs(scans(*).triplephasecerr(*,index))
scans(*).tripleampcerr(*,index)=-abs(scans(*).tripleampcerr(*,index))
;
end
;************************************************************************Block 4
pro plotspectrumshift
;
; Analyze fringe-less GRAVITY test data
; Run in /vlti/gravity/2018-04-10
;
dir='data_with_raw_calibs/'
p=mrdfits(dir+'GRAVI.2018-04-11T03:54:48.919_preproc_0000.fits',4,h)
l0=mrdfits(dir+'GRAVI.2018-04-11T03:54:48.919_preproc_0000.fits',6,h)
l1=mrdfits(dir+'GRAVI.2018-04-11T03:54:48.919_preproc_0000.fits',7,h)
;
window,xsize=800
;
ipol=1
case ipol of
	0:l=l0
	1:l=l1
endcase
;
case ipol of
0:	begin
	plot, l.eff_wave*1e6,p(0).data1,xrange=[1.99,2.04],yrange=[0,1500]
	oplot,l.eff_wave*1e6,p(0).data12,color=tci(2)
	oplot,l.eff_wave*1e6,p(0).data24,color=tci(3)
	end
1:	begin
	plot, l.eff_wave*1e6,p(0).data25,xrange=[1.99,2.04],yrange=[0,1500]
	oplot,l.eff_wave*1e6,p(0).data37,color=tci(2)
	oplot,l.eff_wave*1e6,p(0).data48,color=tci(3)
	end
endcase	
;
end
;-------------------------------------------------------------------------------
pro specphot
;
; p=fluxes
; w=wavelengths
;
; Analyze fringe-less test data for ETC testing
;
dir='data_with_raw_calibs/'
!p.charsize=1.0
;
; Alpha Leo (fringe-less data)
; Read preproc file which containes the raw extracted spectra
p=mrdfits(dir+'GRAVI.2018-04-11T03:13:39.278_preproc_0000.fits',4,h)
w=mrdfits(dir+'GRAVI.2018-04-11T03:13:39.278_singlecalvis_0000.fits',3,h)
;
flux_sc=        p(0).data1*0
;
; Integrate spectra over all outputs and time stamps
for i=0,n_elements(p)-1 do begin
flux_sc=flux_sc+p(i).data1
flux_sc=flux_sc+p(i).data2
flux_sc=flux_sc+p(i).data3
flux_sc=flux_sc+p(i).data4
flux_sc=flux_sc+p(i).data5
flux_sc=flux_sc+p(i).data6
flux_sc=flux_sc+p(i).data7
flux_sc=flux_sc+p(i).data8
flux_sc=flux_sc+p(i).data9
flux_sc=flux_sc+p(i).data10
flux_sc=flux_sc+p(i).data11
flux_sc=flux_sc+p(i).data12
flux_sc=flux_sc+p(i).data13
flux_sc=flux_sc+p(i).data14
flux_sc=flux_sc+p(i).data15
flux_sc=flux_sc+p(i).data16
flux_sc=flux_sc+p(i).data17
flux_sc=flux_sc+p(i).data18
flux_sc=flux_sc+p(i).data19
flux_sc=flux_sc+p(i).data20
flux_sc=flux_sc+p(i).data21
flux_sc=flux_sc+p(i).data22
flux_sc=flux_sc+p(i).data23
flux_sc=flux_sc+p(i).data24
endfor
;
; Read OI_FLUX table computed by Reflex (flat-flux=false)
d=mrdfits('reflex_end_products/2018-06-01T16:37:12/GRAVI.2018-04-11T03:13:39.278/CAL_HD087901_2018-04-11T03:12:44_SINGLE_CAL_VIS.fits',12,h)
;
flux_ts=d(0).flux*0
;
; Integrate over all stations and time stamps
for i=0,n_elements(d)-1 do flux_ts=flux_ts+d(i).flux
;
plot,w.eff_wave*1e6,flux_ts, $
	title='Total flux telescopes (white) and extracted spectra (green)'
oplot,w.eff_wave*1e6,flux_sc,color=tci(3)
;
return
;
; Spica
p=mrdfits(dir+'GRAVI.2018-04-11T03:54:48.919_preproc_0000.fits',4,h)
w=mrdfits(dir+'GRAVI.2018-04-11T03:54:48.919_singlecalvis_0000.fits',4,h)
;
flux_sc=        p(0).data1
flux_sc=flux_sc+p(0).data2
flux_sc=flux_sc+p(0).data3
flux_sc=flux_sc+p(0).data4
flux_sc=flux_sc+p(0).data5
flux_sc=flux_sc+p(0).data6
flux_sc=flux_sc+p(0).data7
flux_sc=flux_sc+p(0).data8
flux_sc=flux_sc+p(0).data9
flux_sc=flux_sc+p(0).data10
flux_sc=flux_sc+p(0).data11
flux_sc=flux_sc+p(0).data12
flux_sc=flux_sc+p(0).data13
flux_sc=flux_sc+p(0).data14
flux_sc=flux_sc+p(0).data15
flux_sc=flux_sc+p(0).data16
flux_sc=flux_sc+p(0).data17
flux_sc=flux_sc+p(0).data18
flux_sc=flux_sc+p(0).data19
flux_sc=flux_sc+p(0).data20
flux_sc=flux_sc+p(0).data21
flux_sc=flux_sc+p(0).data22
flux_sc=flux_sc+p(0).data23
flux_sc=flux_sc+p(0).data24
flux_sc=flux_sc+p(0).data25
flux_sc=flux_sc+p(0).data26
flux_sc=flux_sc+p(0).data27
flux_sc=flux_sc+p(0).data28
flux_sc=flux_sc+p(0).data29
flux_sc=flux_sc+p(0).data30
flux_sc=flux_sc+p(0).data31
flux_sc=flux_sc+p(0).data32
flux_sc=flux_sc+p(0).data33
flux_sc=flux_sc+p(0).data34
flux_sc=flux_sc+p(0).data35
flux_sc=flux_sc+p(0).data36
flux_sc=flux_sc+p(0).data37
flux_sc=flux_sc+p(0).data38
flux_sc=flux_sc+p(0).data39
flux_sc=flux_sc+p(0).data40
flux_sc=flux_sc+p(0).data41
flux_sc=flux_sc+p(0).data42
flux_sc=flux_sc+p(0).data43
flux_sc=flux_sc+p(0).data44
flux_sc=flux_sc+p(0).data45
flux_sc=flux_sc+p(0).data46
flux_sc=flux_sc+p(0).data47
flux_sc=flux_sc+p(0).data48
;
end
;************************************************************************Block 5
function molecfit2l,l,l_min,l_max
;
; Convert molecfit lambda [-1,1] to wavelength shift, given l_min and l_max.
; e.g.: print,molecfit2l(1.945e-03,1.990,2.450): 0.000447
;
common LocalMolecfit,min_l,max_l
;
if n_elements(l_min) eq 0 then l_min=min_l
if n_elements(l_max) eq 0 then l_max=max_l
;
min_l=l_min
max_l=l_max
;
return,(l+1)*(l_max-l_min)/2+l_min-((l_min+l_max)/2)
;
end
;-------------------------------------------------------------------------------
function l2molecfit,l,l_min,l_max
;
; Convert absolute lambda, given l_min and l_max, to molecfit lambda [-1,1].
;
common LocalMolecfit,min_l,max_l
;
if n_elements(l_min) eq 0 then l_min=min_l
if n_elements(l_max) eq 0 then l_max=max_l
;
min_l=l_min
max_l=l_max
;
return,(l-l_min)*2/(l_max-l_min)-1
;
end
;-------------------------------------------------------------------------------
pro molecfit_key,molecfit_keys,key,value
;
; Replave value for given key in list of key/value pairs.
;
index=where(strpos(molecfit_keys,key) eq 0,count) & index=index(0)
if count ge 1 then begin
	words=nameparse(molecfit_keys(index))
	words(1)=value
	molecfit_keys(index)=strjoin(words,' ')
endif else begin
	print,'Key not found: '+key
endelse
;
end
;-------------------------------------------------------------------------------
function merge_spectra,l1,f1,l2,f2,dl,l
;
; Merge two GRAVITY spectra. dl given in microns
;
; d1=mrdfits('AA0/gravity_fit.fits',1,h)
; d2=mrdfits('AG1/gravity_fit.fits',1,h)
; dl=-median(d1(1:1741).lambda-d1(0:1740).lambda)*0.5
; f=merge_spectra(d1.lambda,d1.flux,d2.lambda,d2.flux,dl,l)
; window_slide,xsize=10000
; plot,l,f
; oplot,l(indgen(1742)*2),f(indgen(1742)*2),psym=1,color=tci(2)
; oplot,l(indgen(1742)*2+1),f(indgen(1742)*2+1),psym=1,color=tci(3)
;
; -1 < dl < +1
; dl=0.5
; d=mrdfits('molecfit_gravity_fit_HDN070555_1_AA0.fits',1,h)
; l1=d.lambda
; l1=findgen(1742)
; f1=d.flux
; d=mrdfits('molecfit_gravity_fit_HDN070555_1_AG1.fits',1,h)
; l2=findgen(1742)-dl
; f2=d.flux
; window_slide,xsize=10000
; plot,l1,f1,psym=-1,xrange=[0,1000]
; oplot,l2,f2+2000,color=tci(3),psym=1
; window_slide,xsize=10000
; f2=d.flux+2000
; plot,smooth(f1-f2,30),psym=-1,xrange=[0,1000]
; oplot,smooth(f1-f2,30,/edge_truncate),psym=-1,color=tci(3)
; df=smooth(f1-f2,30,/edge_truncate)
; window_slide,xsize=10000
; plot,findgen(1742),flux,psym=-1,xrange=[0,1000]
; oplot,findgen(1742)-0.5,d.flux+2000+df,color=tci(3),psym=1
;
; The following are just notes...
if 0 then begin
dl=-0.5
d=mrdfits('molecfit_gravity_fit_HDN070555_1_AA0.fits',1,h)
l1=findgen(1742)
f1=d.flux
d=mrdfits('molecfit_gravity_fit_HDN070555_1_AG1.fits',1,h)
l2=findgen(1742)+dl
f2=d.flux
df=smooth(f1-f2,30,/edge_truncate)
n=n_elements(d)
l=fltarr(n*2)
l(indgen(n)*2)=l2
l(indgen(n)*2+1)=l1
f=fltarr(n*2)
f(indgen(n)*2)=f2+df
f(indgen(n)*2+1)=f1
window_slide,xsize=10000
plot,l,f,psym=-1
endif
;
n=n_elements(l1)
;
dx=dl/median(l1(1:n-1)-l1(0:n-2))
x1=findgen(n)
x2=findgen(n)+dx
df=smooth(f1-f2,30,/edge_truncate)
;
x=fltarr(2*n)
f=fltarr(2*n)
;
x(indgen(n)*2)=x2
x(indgen(n)*2+1)=x1
f(indgen(n)*2)=f2+df
f(indgen(n)*2+1)=f1
;
; window_slide,xsize=10000
; plot,x,f,psym=-1
;
l=x*median(l1(1:n-1)-l1(0:n-2))+min(l1)
;
return,f
;
end
;-------------------------------------------------------------------------------
function splice_index,l1,l2,s
;
; min(l2) < min(l1)
;
s=fix(median(l2-l1)/median(shift(l1,-1)-l1))
n_l2=n_elements(l2)
index=shift(lindgen(n_l2),s)
return,index(0:n_l2-abs(s)-1)
;
end
;-------------------------------------------------------------------------------
function merge_spectra,l1,f1,l2,f2,dw,l
;
; Merge two GRAVITY spectra. dw given in microns
;
; d1=mrdfits('AA0/gravity_fit.fits',1,h)
; d2=mrdfits('AG1/gravity_fit.fits',1,h)
; l1=d1.lambda
; l2=d2.lambda
; f1=d1.flux
; f2=d2.flux
; window_slide,xsize=10000
; plot,l1,f1
; oplot,l2,f2,psym=1,color=tci(2)
; dw=-median(d1(1:1741).lambda-d1(0:1740).lambda)*0.5
; f=merge_spectra(d1.lambda,d1.flux,d2.lambda,d2.flux,dw,l)
; plot,l,f
; oplot,l(indgen(1742)*2),f(indgen(1742)*2),psym=1,color=tci(2)
; oplot,l(indgen(1742)*2+1),f(indgen(1742)*2+1),psym=1,color=tci(3)
;
index=splice_index(l1,l2,s)
n=n_elements(index)
l=dblarr(n*2)
f=dblarr(n*2)
if s lt 0 then begin
	df=smooth(f1(index+s)-f2(index),30,/edge_truncate)
	l(lindgen(n)*2)=l2(index)
	l(lindgen(n)*2+1)=l1(index+s)
	f(lindgen(n)*2)=f2(index)+df
	f(lindgen(n)*2+1)=f1(index+s)
endif else if s eq 0 then begin
	if min(l2) lt min(l1) then begin
	df=smooth(f2(index)-f1(index),30,/edge_truncate)
	l(lindgen(n)*2)=l2(index)
	l(lindgen(n)*2+1)=l1(index)
	f(lindgen(n)*2)=f2(index)+df
	f(lindgen(n)*2+1)=f1(index)
	endif else begin
	df=smooth(f2(index+s)-f1(index),30,/edge_truncate)
	l(lindgen(n)*2)=l1(index)
	l(lindgen(n)*2+1)=l2(index)
	f(lindgen(n)*2)=f1(index)+df
	f(lindgen(n)*2+1)=f2(index)
	endelse
endif
;
; window_slide,xsize=10000
; plot,x,f,psym=-1
;
return,f
;
end
;-------------------------------------------------------------------------------
function merge_spectra,l1,f1,l2_in,f2,dw,l
;
; Merge two GRAVITY spectra. dw given in microns
;
; Example:
; d1=mrdfits('AA0/gravity_fit.fits',1,h)
; d2=mrdfits('AG1/gravity_fit.fits',1,h)
; n=n_elements(d1)
; l1=d1.lambda
; l2=d2.lambda
; f1=d1.flux
; f2=d2.flux
; window_slide,xsize=20000
; plot,l1,f1,psym=-1
; oplot,l2,f2,psym=-1,color=tci(2)
; dw=-median(d1(1:n-1).lambda-d1(0:n-2).lambda)*0.5
; f12=merge_spectra(d1.lambda,d1.flux,d2.lambda,d2.flux,dw,l12)
; window_slide,xsize=20000
; plot,l12,f12
; oplot,l12(indgen(n)*2),f12(indgen(n)*2),psym=1,color=tci(2)
; oplot,l12(indgen(n)*2+1),f12(indgen(n)*2+1),psym=1,color=tci(3)
;
; d2=mrdfits('AG1/gravity_fit.fits',1,h)
; d3=mrdfits('AJ2/gravity_fit.fits',1,h)
; n=n_elements(d2)
; l2=d2.lambda
; l3=d3.lambda
; f2=d2.flux
; f3=d3.flux
; window_slide,xsize=20000
; plot,l2,f2,psym=-1
; oplot,l3,f3,psym=-1,color=tci(2)
; dw=-median(d2(1:n-1).lambda-d2(0:n-2).lambda)*0.5
; f23=merge_spectra(d2.lambda,d2.flux,d3.lambda,d3.flux,dw,l23)
; window_slide,xsize=20000
; plot,l23,f23
; oplot,l23(indgen(n)*2),f23(indgen(n)*2),psym=1,color=tci(2)
; oplot,l23(indgen(n)*2+1),f23(indgen(n)*2+1),psym=1,color=tci(3)
;
; d3=mrdfits('AJ2/gravity_fit.fits',1,h)
; d4=mrdfits('AK0/gravity_fit.fits',1,h)
; l3=d3.lambda
; l4=d4.lambda
; f3=d3.flux
; f4=d4.flux
; window_slide,xsize=20000
; plot,l3,f3,psym=-1
; oplot,l4,f4,psym=-1,color=tci(2)
; dw=-median(d3(1:n-1).lambda-d3(0:n-2).lambda)*0.5
; f34=merge_spectra(d3.lambda,d3.flux,d4.lambda,d4.flux,dw,l34)
; window_slide,xsize=20000
; plot,l34,f34
; oplot,l34(indgen(n)*2),f34(indgen(n)*2),psym=1,color=tci(2)
; oplot,l34(indgen(n)*2+1),f34(indgen(n)*2+1),psym=1,color=tci(3)
;
; n=n_elements(l12)
; window_slide,xsize=10000
; plot,l12,f12
; oplot,l34,f34,psym=1,color=tci(2)
; dw=-median(l12(1:n-1)-l12(0:n-2))*0.5
; f=merge_spectra(l12,f12,l34,f34,dw,l)
; plot,l,f
; oplot,l(indgen(n)*2),f(indgen(n)*2),psym=1,color=tci(2)
; oplot,l(indgen(n)*2+1),f(indgen(n)*2+1),psym=1,color=tci(3)
;
n=n_elements(l1)
dl=median(l1(1:n-1)-l1(0:n-2))
dx=dw/dl
;
l2=l2_in+dw
;
df=smooth(f1-f2,30,/edge_truncate)
;
x1=findgen(n)
x2=findgen(n)+dx
;
x=fltarr(2*n)
x(indgen(n)*2)=x2
x(indgen(n)*2+1)=x1
l=x*dl+min(l1)
;
f=fltarr(2*n)
f(indgen(n)*2)=f2+df
f(indgen(n)*2+1)=f1
;
; window_slide,xsize=10000
; plot,x,f,psym=-1
;
return,f
;
end
;-------------------------------------------------------------------------------
function set_contint,wavelength,flux
;
window,xsize=2000,/free
plot,wavelength,flux,/ynozero
;
l=fltarr(100)
il=0
;
if !d.window eq -1 then begin
	print,'***Error: no plot window open!'
	return,-1
endif
;
print,'___________________________________________'
print,'Click left mouse button on plot window.'
print,'Click middle or right mouse button to exit!'
print,'__________________***______________________'
;
cursor,x,y,/down & icom=!err
device,set_graphics_function=6
y1=!y.crange(0) & y2=!y.crange(1)
yrange=y2-y1
y1=y1-0.1*yrange & y2=y2+0.1*yrange
oplot,[x,x],[y1,y2],psym=0
;
repeat begin
	cursor,xn,y,/change & icom=!err
	oplot,[x,x],[y1,y2],psym=0
	oplot,[xn,xn],[y1,y2],psym=0
	print,format='(%"Wavelength: %s\r",$)', $
	string(xn,format='(f7.4)')
	x=xn
	if icom eq 1 then begin
		l(il)=float(string(xn,format='(f7.4)'))
		il=il+1
;		if il eq 0 then begin
;			l(il)=ls
;			il=il+1
;		endif else begin
;			if ls gt l(il) then begin
;				l(il)=ls
;				il=il+1
;			endif
;		endelse
	endif
endrep until (icom eq 4) or (icom eq 2)
;
if icom eq 4 then oplot,[x,x],[y1,y2],psym=0
device,set_graphics_function=3
;
if il eq 0 then return,-1
l=unique(l(where(l gt 0)))
if n_elements(l) mod 2 ne 0 then begin
	print,'***Error: incomplete interval!'
	return,-1
endif
n_int=n_elements(l)/2
l_int=reform(l,2,n_int)
;
return,l_int
;
end
;-------------------------------------------------------------------------------
function gravity_normcont,wavelength,flux,l_int_in,degree,r
;
; l_int=gravity_contint(d.lambda,d.flux)
; gravity_fitcont,d.lambda,d.flux,l_int
;
w=wavelength
f=flux
;
n=n_elements(wavelength)
l_int=l_int_in
index=where(l_int(1,*) gt w(0))
l_int=l_int(*,index)
index=where(l_int(0,*) lt w(n-1))
l_int=l_int(*,index)
;
; Decrease degree of polynomial to fit
case fix((max(l_int_in)-min(l_int_in))/(max(l_int)-min(l_int))) of
	5: degree=3
	4: degree=3
	3: degree=4
	2: degree=4
	1: degree=5
endcase
;
;window,xsize=2000,/free
;plot,w,f,/ynozero
;
dims=size(l_int,/dim)
n_int=dims(1)
;
w_int=total(l_int,1)/2
f_int=fltarr(n_int)
for i=0,n_int-1 do $
	f_int(i)=median(f(where(w gt l_int(0,i) and w lt l_int(1,i))))
r=polyfit(w_int,f_int,degree)
; oplot,wavelength,polynom(wavelength,r),color=tci(3)
;
return,flux/polynom(wavelength,r)
;
end
;-------------------------------------------------------------------------------
function gravity_kernel,wl,fl
;
; Restore continuum interval selection l_int
molecfit_dir='/usr/local/src/molecfit/'
restore,molecfit_dir+'oyster/gravity_l_int.xdr'
;
fl_norm=gravity_normcont(wl,fl,l_int)
;
n=15
lag=indgen(n)-n/2
kernel=a_correlate(fl,lag)
;
return,kernel
;
end
;-------------------------------------------------------------------------------
pro gravity_kernels,wl,fl
;
; Kernel for CO2 region (does not work well)
; index=where(l ge 2.063 and l le 2.073)
; l=l(index)
; f=f(index)
; r=poly_fit(l,f,1,yfit)
; f=f/yfit
; kernel=a_correlate(f,lag)
;
; Restore continuum interval selection l_int
restore,molecfit_dir+'oyster/gravity_l_int.xdr'
;
fl_norm=gravity_normcont(wl,fl,l_int)
fl_n=fl_norm-median(fl_norm,10)
n_fl=n_elements(fl_norm)
;
n_lg=31
lag=indgen(n_lg)-n_lg/2
;
kernels=fltarr(n_lg,n_fl)
for i=n_lg/2,n_fl-n_lg/2-1 do $
	kernels(*,i)=a_correlate(fl_n(i-n_lg/2:i+n_lg/2),lag)
;
m=11
kernels=kernels(n_lg/2-m/2:n_lg/2+m/2,*)
for i=0,n_lg/2-1 do kernels(*,i)=kernels(*,n_lg/2)
for i=n_fl-n_lg/2,n_fl-1 do kernels(*,i)=kernels(*,n_fl-n_lg/2-1)
;
openw,unit,'gravity_kernels.dat',/get_lun
for i=0,n_fl-1 do printf,unit,kernels(*,i)>0,format='(11(f5.3,1x))'
free_lun,unit
;
end
;************************************************************************Block 6
pro gravity_tellcorr
;
; Obsolete! See following procedure for multi-processor computing.
;
; Correct photometry for telluric absorption, using a call to molecfit.
; Read XDR file and run gravity_compact before calling this procedure.
;
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Molecfit installation directory
molecfit_dir=!oyster_dir+'source/molecfit/'
molecfit_dir='/usr/local/src/molecfit/'
;
; Read default parameter file in oyster/source/molecfit
par_file='gravity.par'
par_file_def=molecfit_dir+'oyster/'+par_file
par_lines=''
status=dc_read_fixed(par_file_def,par_lines,format='(a80)',/col)
;
for is=0,n_elements(scans)-1 do begin
; 	for ob=0,genconfig.numoutbeam-1 do begin
	ob=0	; Just process the science combiner
	nc=genconfig.numspecchan(ob)
	wl=genconfig.wavelength(*,ob)*1e6	; convert to micron
	p=fltarr(nc)
	g=intarr(nc)+1
	tw=fltarr(nc)
;	Computed weighted sum of fluxes of all telescopes
	for ib=0,genconfig.numsid-1 do begin
		e=scans(is).photometryerr(ib,ob,0:nc-1)
		g=g*(e>0)
		w=1./e^2
		p=p+scans(is).photometry(ib,ob,0:nc-1)*w
		tw=tw+w
	endfor
	p=p/tw
	e=1./sqrt(tw)
	index=where(g eq 0,count)
	if count gt 0 then e(index)=-1
;	photometry=scans(is).photometry(ib,ob,0:nc-1)
;	photometryerr=scans(is).photometryerr(ib,ob,0:nc-1)
;	index=where(photometryerr gt 0,count)
	index=where(g gt 0,count)
	if count gt 0 then begin
;		fl=photometry(index)
		ib=0
		fl=p(index)
		wl=wl(index)
;			
;		Write data to local file
		filename=pwd()+'/molecfit_gravity.dat'
		openw,unit,filename,/get_lun
		for i=0,count-1 do printf,unit,wl(i),fl(i)
		free_lun,unit
;		Update parameter file
		molecfit_key,par_lines,'filename', $
			filename
		molecfit_key,par_lines,'cont_const', $
			string(median(fl))
		case instrument_id(systemid) of
;		grep "FWHM of Gaussian in pixels" molecfit_gravity*.res
		'GRAVITY-LR': res_gauss=1.6 
		'GRAVITY-MR': res_gauss=1.6 
		'GRAVITY-HR': res_gauss=1.6 
		endcase
		molecfit_key,par_lines,'res_gauss', $
			res_gauss
		molecfit_key,par_lines,'obsdate', $
			string(date2jd(date,/mjd))
		molecfit_key,par_lines,'utc', $
			string(scans(is).time)
		molecfit_key,par_lines,'telalt', $
			string(90.-scans(is).za)
		molecfit_key,par_lines,'rhum', $
			string(scans(is).rhum)
		molecfit_key,par_lines,'pres', $
			string(scans(is).pres)
		molecfit_key,par_lines,'temp', $
			string(scans(is).temp)
		molecfit_key,par_lines,'m1temp', $
			string(scans(is).temp)
; 		Write parameter file
		openw,unit,par_file,/get_lun
		for i=0,n_elements(par_lines)-1 do printf,unit,par_lines(i)
		free_lun,unit
;
;		Call molecfit (fit molecules to narrow wavelength intervals)
		spawn,molecfit_dir+'bin/molecfit  '+par_file,r_m
;		Call calctrans (compute transmission over entire range)
		spawn,molecfit_dir+'bin/calctrans '+par_file,r_c
;		Move files with results from the molecfit dir. to local
		suffix='_'+scans(is).starid+'_'+strtrim(string(is+1),1) $
		      +'_'+genconfig.stationid(ib)
		suffix='_'+scans(is).starid+'_'+strtrim(string(is+1),1)
		spawn,'mv -f '+molecfit_dir+'/oyster/' $
			+'molecfit_gravity_fit.ps molecfit_gravity_fit' $
			+suffix+'.ps'
		spawn,'mv -f '+molecfit_dir+'/oyster/' $
			+'molecfit_gravity_fit.fits molecfit_gravity_fit' $
			+suffix+'.fits'
		spawn,'mv -f '+molecfit_dir+'/oyster/' $
			+'molecfit_gravity_fit.res molecfit_gravity_fit' $
			+suffix+'.res'
		spawn,'mv -f '+molecfit_dir+'/oyster/' $
			+'molecfit_gravity_tac.ps molecfit_gravity_tac' $
			+suffix+'.ps'
		spawn,'mv -f '+molecfit_dir+'/oyster/' $
			+'molecfit_gravity_tac.fits molecfit_gravity_tac' $
			+suffix+'.fits'
;
;		Read corrected spectrum and store in PhotometryC (station 0)
		case instrument_id(systemid) of
		'GRAVITY-LR': n=10 
		'GRAVITY-MR': n=50 
		'GRAVITY-HR': n=100 
		endcase
		d=mrdfits('molecfit_gravity_tac'+suffix+'.fits',1,header)
		f=median(d.cflux)
		fm=median(d.cflux,n)
		nndex=where(fm eq 0,count)
		if count ge 1 then stop
		scans(is).photometryc(ib,ob,index)=f*d.cflux/fm
		scans(is).photometrycerr(ib,ob,index)= $
		scans(is).photometryerr(ib,ob,index)
		scans(is).photometrycerr(ib,ob,index)=e(index)
;		d=mrdfits('molecfit_gravity_fit'+suffix+'.fits',1,header)
;		scans(is).photometryc(ib,ob,index)=d.flux/d.mflux
;		scans(is).photometrycerr(ib,ob,index)= $
;		scans(is).photometryerr(ib,ob,index)/d.mflux
;		scans(is).photometryerr(ib,ob,index)/d.mflux
;		scans(is).photometrycerr(ib,ob,index)=e(index)
;
	endif
;	endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro gravity_tellcorr,highres=highres, $
prepare=prepare,runmfit=runmfit,runcalc=runcalc,collect=collect
;
; Multi-processor version!!
; Read XDR file and run gravity_compact before calling this procedure.
;
; Correct photometry for telluric absorption, using a call to molecfit.
; It is recommended to have run gravity_compact before, or start with
; a regular .cha file, e.g., get_data,'2017-04-05_VLTI_HR_AA0-AG1-AJ2-AK0.cha'.
;
; prepare=1: add spectra of all telescopes
; prepare not set: fit spectra separately
;
common LocalGravityTellCorr,basedir,suffix,par_files,tel
;
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Molecfit installation directory
molecfit_dir='/usr/local/src/molecfit/'
;
; Read default parameter file in oyster/source/molecfit
par_file=molecfit_dir+'oyster/gravity.par'
par_lines=''
status=dc_read_fixed(par_file,par_lines,format='(a80)',/col)
par_files=strarr(genconfig.numsid)+'gravity.par'
;
ob=where(genconfig.numspecchan gt 5) ; Just process the science combiner
ob=ob(0)
nc=genconfig.numspecchan(ob)
;
if n_elements(highres) eq 0 then highres=0
;
; All following options, if true, cause their exclusive execution
; If option 'prepare' is not set, run molecfit for each telescope
if n_elements(prepare) eq 0 then prepare=genconfig.numsid else begin
	if prepare ge 1 then begin
		runcalc=0
		runmfit=0
		collect=0
	endif
	endelse
; If one of the following options is false, the option will not be executed.
; The default is to execute them, independent of the other option seetings
if n_elements(runmfit) eq 0 then runmfit=1 else begin
	if runmfit then begin
		prepare=0
		runcalc=0
		collect=0
	endif
	endelse
if n_elements(runcalc) eq 0 then runcalc=1 else begin
	if runcalc then begin
		prepare=0
		runmfit=0
		collect=0
	endif
	endelse
if n_elements(collect) eq 0 then collect=1 else begin
	if collect then begin
		prepare=0
		runcalc=0
		runmfit=0
	endif
	endelse
;
IF prepare GE 1 THEN BEGIN
; Prepare directories with data and parameter files
basedir=strarr(n_elements(scans))
suffix=strarr(genconfig.numsid)
ib=0
for is=0,n_elements(scans)-1 do begin
	basedir(is)=pwd()+'/'+scans(is).starid+'_'+strtrim(string(is+1),1)
; 	Check if data are available (in DUAL mode, they may not be)
	index=where(scans(is).photometryerr(ib,ob,0:nc-1) gt 0,count)
	if count eq 0 then basedir(is)=''
endfor
for ib=0,genconfig.numsid-1 do suffix(ib)=genconfig.stationid(ib)
tel=strmid(suffix(0),0,1)
for is=0,n_elements(scans)-1 do begin
	if strlen(basedir(is)) ne 0 then begin
	p=fltarr(nc)
	g=intarr(nc)+1	; Logical vector for good channels in all IBs
	tw=fltarr(nc)
	wl=genconfig.wavelength(0:nc-1,ob)*1e6	; convert to micron
;	Preparation for cross-correlation
	dwl=median(wl(1:nc-1)-wl(0:nc-2))
;
;	The reference spectrum to align all others to
	r=scans(is).photometry(0,ob,0:nc-1)
;
;	Create data arrays for tellrem FITS file (must have equal
;	flux=fltarr(nc)		; Initialized below
	qual=fltarr(nc)+1	; QUAL column for FITS file
	errs=fltarr(nc)		; Flux errors
	wave=wl			; Wavelengths
;
;	Compute weighted sum of spectra from all telescopes
	for ib=0,genconfig.numsid-1 do begin
		e=reform(scans(is).photometryerr(ib,ob,0:nc-1))
		index=where(e gt 0,count)
		if count gt 0 then begin
			w=1./e(index)^2
			p(index)=p(index)+scans(is).photometry(ib,ob,index)*w
			tw(index)=tw(index)+w
			e_median=median(e(index))
		endif
		g=g*(e>0)
	endfor
	index=where(g eq 0,count)
	if count gt 0 then e(index)=e_median
	index=where(g gt 0,count)
	p(index)=p(index)/tw(index)
	e(index)=1./sqrt(tw(index))
	qual(index)=0	; flag: 0 for good values, 1 for bad
	errs=e
;
	if count gt 0 then begin
	for ib=0,prepare-1 do begin
		if prepare eq 1 then suffix(ib)='All' $
		else p=scans(is).photometry(ib,ob,0:nc-1)
;		Only use channels with data from all input beams
		fl=p(index)
		wl=wl(index)
		flux=reform(p)
;		Compute and save kernel
;		jndex=where(wl ge 2.36 and wl le 2.45)
;		kernel=gravity_kernel(wl(jndex),fl(jndex))
;		Normalize continuum
		restore,molecfit_dir+'oyster/gravity_l_int.xdr'
		fl=gravity_normcont(wl,fl,l_int,degree,coeffs)
		flux=flux/polynom(wave,coeffs)
		errs=errs/polynom(wave,coeffs)
;		Create sub-directories for scans and stations
		spawn,'mkdir -p '+basedir(is)
		spawn,'mkdir -p '+basedir(is)+'/'+suffix(ib)
		cd,basedir(is)+'/'+suffix(ib),current=old_dir
		save,index,filename='index.xdr'
		if highres then begin
;			Read wavelength sol. from previous run if requested
			spawn,"grep 'Chip 1, coef 0:' gravity_fit.res",r
			words=nameparse(r,':')
			wlc_const=float(words(1))
			status=dc_read_free('gravity.dat',l,f,/col)
			w0=molecfit2l(wlc_const,min(l),max(l))
		endif else begin
;			Estimate wavelength offset
			restore,molecfit_dir+'oyster/gravity_tac_hr.xdr'
			mt=mt(index)
			ml=ml(index)
			n=10
			l=findgen(2*n+1)-n
			r=c_correlate(fl,mt,l); cross-correlate with ref. spec.
			r_index=where(r eq max(r)) & r_index=r_index-n
			w0=r_index*dwl
			m=4 ; fit cubic spline for better determination of max.
			n=n*m
			x=(findgen(2*n+1)-n)/n
			y=spline(l,r,x)
			y_index=where(y eq max(y)) & y_index=y_index-n
			w0=-y_index(0)*dwl/m
			print,'Estimated wavelength offset = [microns]',w0
			w0=0
		endelse
;		Write data file, including wavelength offset if requested
		filestub=pwd()+'/gravity';+'_'+suffix(ib)
		filename=filestub+'.dat'
		openw,unit,filename,/get_lun
		for i=0,count-1 do printf,unit,wl(i)-w0,fl(i)
		free_lun,unit
;		Set the coef. 0 of wavelength sol. to zero as we subtr. w0 above
		wlc_const=0
;		Update parameter file
		molecfit_key,par_lines,'filename', $
			filename
		molecfit_key,par_lines,'output_dir', $
			basedir(is)+'/'+suffix(ib)
		molecfit_key,par_lines,'output_name', $
			'gravity';+'_'+suffix(ib)
		molecfit_key,par_lines,'cont_const', $
			string(median(fl))
		molecfit_key,par_lines,'wlc_const', $
			wlc_const
		case instrument_id(systemid) of
;		grep "FWHM of Gaussian in pixels" molecfit_gravity*.res
		'GRAVITY-LR': res_gauss=1.6 
		'GRAVITY-MR': res_gauss=1.6 
		'GRAVITY-HR': res_gauss=1.6 
		endcase
		relcol=' 1.0 1.06 1.0'
;		Read result file from previous run to speed up fitting
		res_file=file_search('*.res') & res_file=res_file(0)
		if strlen(res_file) gt 0 then begin
			spawn,"grep 'FWHM of Gaussian in pixels:' "+res_file,r
			words=nameparse(r,':')
			res_gauss=float(words(1))
			spawn,"grep 'list_molec' "+par_files(ib),r
			r=r(n_elements(r)-1)
			words=nameparse(r,':')
			molecules=nameparse(words(1))
			nm=n_elements(molecules)
			relcol=''
			lines=''
			status=dc_read_fixed(res_file,lines,/col,format='(a80)')
			i1=where(strpos(lines,'RELATIVE MOLECULAR GAS') ge 0)
			i2=where(strpos(lines,'MOLECULAR GAS COLUMNS IN') ge 0)
			lines=lines(i1+1,i2-1)
			for j=0,nm-1 do begin
				k=where(strpos(lines,molecules(0)+':') ge 0)
				words=nameparse(lines(k(0)),':')
				relcol=relcol+' '+string(float(words(1)))
			endfor
		endif
;		Update values in parameter file
		molecfit_key,par_lines,'res_gauss', $
			res_gauss
		molecfit_key,par_lines,'relcol', $
			relcol
		molecfit_key,par_lines,'obsdate', $
			string(date2jd(date,/mjd))
		molecfit_key,par_lines,'utc', $
			string(scans(is).time)
		molecfit_key,par_lines,'telalt', $
			string(90.-scans(is).za)
		molecfit_key,par_lines,'rhum', $
			string(scans(is).rhum)
		molecfit_key,par_lines,'pres', $
			string(scans(is).pres)
		molecfit_key,par_lines,'temp', $
			string(scans(is).temp)
		molecfit_key,par_lines,'m1temp', $
			string(scans(is).temp)
		molecfit_key,par_lines,'geoelev', $
			string(geoparms.altitude)
		molecfit_key,par_lines,'longitude', $
			string(geoparms.longitude)
		molecfit_key,par_lines,'latitude', $
			string(geoparms.latitude)
; 		Write parameter file in basedir
;		par_files(ib)='gravity_'+suffix(ib)+'.par'
;		par_files(ib)='gravity.par'
		openw,unit,par_files(ib),/get_lun
		for i=0,n_elements(par_lines)-1 do printf,unit,par_lines(i)
		free_lun,unit
;		Also write FITS file for use with tellrem
		gravity_prepare_tellrem,filestub,ob,is,flux,errs,qual
;
		cd,old_dir
	endfor
	endif
	endif	; strlen(basedir(is)) ne 0
endfor
print,'Finished preparation of parameter and data files.'
ENDIF	; prepare
;
; Spawn molecfit for all files, but no more than 2 running simultaneously
IF runmfit THEN BEGIN
index=where(strlen(basedir) gt 0)
for is=0,n_elements(scans)-1 do begin
	if strlen(basedir(is)) ne 0 then begin
	cd,basedir(is),current=top_dir
	if tel eq 'U' then prepare=n_elements(file_search('U??'))
	if tel eq 'A' then prepare=n_elements(file_search('A??'))
	if is gt index(0) then begin
	n_max=!cpu.hw_ncpu/2
	repeat begin
		wait,2
		spawn,'pgrep lnfl',r_lnfl
		spawn,'pgrep lblrtm',r_lblrtm
		r=[r_lnfl,r_lblrtm]
		n=n_elements(r)
		index=where(strlen(r) ne 0,n)
	endrep until n le n_max
	endif
;	Call molecfit (fit molecules to narrow wavelength intervals)
	for ib=0,prepare-1 do begin
;		if prepare eq 1 then suffix(ib)='All'
		cd,suffix(ib),current=old_dir
		spawn,molecfit_dir+'bin/molecfit  '+par_files(ib)+' &',r
		cd,old_dir
	endfor
	if is eq 0 then wait,10	; wait 10s for GDAS file to be retrieved!
	cd,top_dir
	endif	; strlen(basedir(is)) ne 0
endfor
; Wait until all lblrtm processes have finished
repeat begin
	spawn,'pgrep lblrtm',r
	wait,2
endrep until strlen(r(0)) eq 0
wait,5	; Make sure all processes finished
print,'Finished running molecfit.'
ENDIF	; runmfit
;
; Spawn calctrans for all files, but no more than 2 running simultaneously
IF runcalc THEN BEGIN
for is=0,n_elements(scans)-1 do begin
	if strlen(basedir(is)) ne 0 then begin
	cd,basedir(is),current=top_dir
	if tel eq 'U' then prepare=n_elements(file_search('U??'))
	if tel eq 'A' then prepare=n_elements(file_search('A??'))
	n_max=!cpu.hw_ncpu
	if prepare gt 1 then n_max=0
	flag=0
	repeat begin
		wait,2
		spawn,'pgrep lnfl',r_lnfl
		spawn,'pgrep lblrtm',r_lblrtm
		r=[r_lnfl,r_lblrtm]
		index=where(strlen(r) ne 0,count)
		if count le n_max then flag=flag+1
	endrep until flag eq 2
;	Call calctrans (compute transmission over entire range)
	for ib=0,prepare-1 do begin
;		if prepare eq 1 then suffix(ib)='All'
		cd,suffix(ib),current=old_dir
		spawn,molecfit_dir+'bin/calctrans '+par_files(ib)+' &',r
		cd,old_dir
	endfor
	cd,top_dir
	endif	; strlen(basedir(is)) ne 0
endfor
; Wait until all lblrtm processes have finished
flag=0
repeat begin
	wait,2
	spawn,'pgrep lnfl',r_lnfl
	spawn,'pgrep lblrtm',r_lblrtm
	if strlen(r_lnfl(0)) eq 0 and strlen(r_lblrtm(0)) eq 0 then flag=flag+1
endrep until flag eq 2
; wait,5	; We still had mrdfits read errors in the following code
print,'Finished running calctrans.'
ENDIF	; runcalc
;
; Collect the results and store in scans structure
IF collect THEN BEGIN
case instrument_id(systemid) of
	'GRAVITY-LR': n=10 
	'GRAVITY-MR': n=50 
	'GRAVITY-HR': n=100 
endcase
;
for is=0,n_elements(scans)-1 do begin
	if strlen(basedir(is)) ne 0 then begin
	cd,basedir(is),current=top_dir
	if tel eq 'U' then prepare=n_elements(file_search('U??'))
	if tel eq 'A' then prepare=n_elements(file_search('A??'))
	for ib=0,prepare-1 do begin
;		if prepare eq 1 then suffix(ib)='All'
		cd,suffix(ib),current=old_dir
		restore,'index.xdr'
;
;		Read corrected spectrum and store in PhotometryC
		if strlen(file_search('gravity_tac.fits')) eq 0 then begin
		  print,'Error: could not find result file gravity_tac.fits!'
		  print,'Current directory: ',basedir(is)
		endif else begin
		  d=mrdfits('gravity_tac.fits',1,header)
		  f=median(d.cflux)
		  fm=median(d.cflux,n)
		  nndex=where(fm eq 0,count)
		  if count ge 1 then stop
;		  scans(is).photometryc(ib,ob,index)=f*d.cflux/fm
		  scans(is).photometryc(ib,ob,index)=d.cflux
		  if prepare eq 1 then f=sqrt(genconfig.numsid)
		  scans(is).photometrycerr(ib,ob,index)=( $
		  scans(is).photometryerr(ib,ob,index)/d.mtrans)/ $
		  (median(scans(is).photometry(ib,ob,index))*f)
		endelse
;
		cd,old_dir
	endfor
	cd,top_dir
	endif	; strlen(basedir(is) ne 0
endfor
ENDIF	; collect
;
; restore,!oyster_dir+'source/gravity/telltrans.xdr'
; s=dc_read_free('/home/chummel/oyster/source/gravity/telldata.txt',l,t,/col)
; Antoine's quicklook code:
; tell = tellTrans(r['wl'][w]/1.00025, width=2.1 if r['SPEC RES']=='HIGH' 
;						 else 1.4)
; def tellTrans(wl, width=2.3):
; res = []
; gwl = np.gradient(wl)
; for i in range(len(wl)):
;     res.append(np.mean(__tran[ np.abs(__lbda-wl[i])<width*gwl[i]/2 ]))
; return np.array(res)
;
; https://docs.scipy.org/doc/numpy-1.13.0/reference/generated/numpy.gradient.html
; >>> f = np.array([1, 2, 4, 7, 11, 16], dtype=np.float)
; >>> np.gradient(f)
; array([ 1. ,  1.5,  2.5,  3.5,  4.5,  5. ])
; >>> np.gradient(f, 2)
; array([ 0.5 ,  0.75,  1.25,  1.75,  2.25,  2.5 ])
;
end
;-------------------------------------------------------------------------------
pro gravity_molecfit
;
; Run in parent folder of station subdirectories.
; Combines wavelength-calibrated gravity.dat files and runs molecfit.
;
molecfit_dir='/usr/local/src/molecfit/'
;
prefix=file_search('A??')
;
for i=0,n_elements(prefix)-1 do begin
	status=dc_read_free(prefix(i)+'/'+'gravity.dat',li,fi,/col)
	if i eq 0 then begin
		l=li
		f=fi
	endif else begin
		l=[l,li]
		f=[f,fi]
	endelse
endfor
;
si=sort(l)
l=l(si)
f=f(si)
;
dat_file='gravity.dat'
openw,unit,dat_file,/get_lun
for i=0,n_elements(l)-1 do printf,unit,l(i),f(i)
free_lun,unit
;
par_file='gravity.par'
par_lines=''
status=dc_read_fixed(prefix(0)+'/'+par_file,par_lines,format='(a80)',/col)
molecfit_key,par_lines,'filename',pwd()+'/'+dat_file
molecfit_key,par_lines,'output_dir',pwd()
openw,unit,par_file,/get_lun
for i=0,n_elements(par_lines)-1 do printf,unit,par_lines(i)
free_lun,unit
;
spawn,molecfit_dir+'bin/molecfit  '+par_files(ib)+' &',r
;
end
;-------------------------------------------------------------------------------
pro gravity_datfile,wlc_const,dat_file
;
; Correct a molecfit input data file (spectrum) for a wavlength shift wlc_const
;
if n_elements(wlc_const) eq 0 then begin
	res_file=file_search('*.res') & res_file=res_file(0)
	if strlen(res_file) eq 0 then return
	spawn,"grep 'Chip 1, coef 0:' "+res_file,r
	words=nameparse(r,':')
	wlc_const=float(words(1))
endif
if n_elements(dat_file) eq 0 then dat_file='gravity.dat'
;
status=dc_read_free(dat_file,l,f,/col)
w0=molecfit2l(wlc_const,min(l),max(l))
l=l-w0
;
openw,unit,dat_file,/get_lun
for i=0,n_elements(l)-1 do printf,unit,l(i),f(i)
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro gravity_tellfit,file
;
; Read molecfit output file and plot raw and calibrated fluxes.
; These plots correspond to how Alain displayes the results.
;
common LocalMolecfit,min_l,max_l
;
if n_elements(file) eq 0 then file='gravity_tac.fits'
;
d=mrdfits(file,1,h)
min_l=min(d.lambda)
max_l=max(d.lambda)
;
set_screen
!p.charsize=1.5
;
window,/free
plot,d.mlambda,d.mtrans,yrange=[0,2],xtitle='Wavelength [microns]', $
;	ytitle='Atm. transmission (lower), corr. spectrum (upper)', $
	ytitle='Raw spectrum (lower), corr. spectrum (upper)', $
	/ynozero,/nodata
; oplot,d.mlambda,d.mtrans,color=tci(2)	; Atm. transmission
oplot,d.lambda,d.flux/median(d.cflux),color=tci(2)
; oplot,d.mlambda,1+d.cflux/max(d.cflux)
; y=d.cflux/max(d.cflux)
; oplot,d.mlambda,y-median(y,100)+1.5
oplot,d.mlambda,(d.cflux/median(d.cflux))+0.5,color=tci(3)
;
; Add plot of NextGen model spectrum in K band
restore,'~/oyster/atmospheres/nextgen/lte42-3.5-0.0.NextGen.spec.xdr'
l=l/1000
index=where(l gt 1.9 and l lt 2.5)
s=s(index) & s=s/median(s)
l=l(index)
r=poly_fit(l,s,2,sfit)
oplot,l,s-sfit+1.2
;
; Plot high-resolution results
; window_slide,xsize=10000
window,/free
plot,d.lambda,d.flux,yrange=[0,1.5]
oplot,d.mlambda,d.mtrans,color=tci(2)	; shows offset if wlc_const ne 0
oplot,d.lambda,d.cflux,color=tci(3)
oplot,d.mlambda,d.flux/d.mtrans,color=tci(3)
;
end
;-------------------------------------------------------------------------------
pro gravity_tellpower,file
;
; Read molecfit output file and plot flux power spectrum.
;
if n_elements(file) eq 0 then file='molecfit_gravity_tac.fits'
;
d=mrdfits(file,1,h)
n=n_elements(d)
;
sampling_int=median(d(1:n-1).lambda-d(0:n-2).lambda)
series=poweroftwo(d.flux)
tp=powerspectrum(series,sampling_int,f,p,f_avg,p_avg)
;
!x.ticks=0
!y.ticks=0
!p.charsize=1.5
;
plot,f_avg,alog10(p_avg),xrange=[f(1),1/(2*sampling_int)],xstyle=1,/xlog, $
     xtitle='Frequency [Hz]',psym=0, $
     title=!p.title, $
     xgridstyle=1,xticklen=1,ygridstyle=1,yticklen=1
;
end
;************************************************************************Block 7
pro gravity_prepare_tellrem,filestub,ob,is,flux,errs,qual
;
; Prepare FITS file, links to TAPE1 and TAPE3, and tellrem.info file
; to prepare a run of tellrem. (Read FITS file with rddat,file,l,f,e)
;
common LocalGravityTellCorr,basedir,suffix,par_files,tel
;
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
fitsfile=filestub+'.fits'
nc=genconfig.numspecchan(ob)
llambda=genconfig.wavelength(0:nc-1,ob)*1e9	; nm
dlambda=median(llambda(1:nc-1)-llambda(0:nc-2))
parseidldate,systime(),file_date
j=where(startable.starid eq scans(is).starid)
header=["SIMPLE  =                    T / Written by IDL", $
        "BITPIX  =                  -32 / Number of bits per data pixel", $
        "NAXIS   =                    1 / Number of data axes", $
        "NAXIS1  =         "+string(nc)+" /", $
	"EXTEND  =                    T / FITS data may contain extensions", $
	"DATE    = '"+file_date+"'/ File creation date", $
	"CRPIX1  =                  1.0 /", $
	"CRVAL1  =     "+string(llambda(0))+" / nm", $
	"CDELT1  =     "+string(dlambda)+" / nm", $
	"CTYPE1  = 'LINEAR  '           /", $
	"CUNIT1  = 'nm      '           / Wavelength unit ", $
	"ORIGIN  = 'ESO     '           / European Southern Observatory", $
	"INSTRUME= 'GRAVITY '           / Instrument used", $
	"OBJECT  = '"+scans(is).star+"'            / Original target", $
	"RA      =     "+string(startable(j).ra*15) $
		     +" / "+hms(startable(j).ra,/aspro)+" RA (J2000) (deg)", $
	"DEC     =     "+string(startable(j).dec) $
		     +" / "+dms(startable(j).dec,/aspro)+" DEC (J2000) (deg)", $
	"EQUINOX =               2000.0 / Standard FK5 (years)", $
	"RADECSYS= 'FK5     '           / Coordinate reference frame", $
	"EXPTIME = 308.0                / Exposure time (sec)", $
	"MJD-OBS = "+string(date2jd(date)+(scans(is).time)/86400-2400000.5) $
		+"     / ",$
      	"DATE-OBS= '"+Date+"T"+strmid(hms(scans(is).time/3600,/aspro),0,8) $
		     +"'/ Observing date", $
	"UTC     =     "+string(scans(is).time)+" / ", $
	"LST     =     "+string((startable(j).ra+scans(is).ha)*3600) $
						 +" / LST (sec)", $
	"ALT     = "+string(90.-scans(is).za)+"     / Altitude (deg)", $
	"BUNIT   = 'ADU     '           /", $
	"EXTNAME = 'FLUX'               /", $
	"HDUCLASS= 'ESO     '           / hdu classification", $
	"HDUDOC  = 'DICD    '           / hdu reference document", $
	"HDUVERS = 'DICD V6.0'          / hdu reference document version", $
	"HDUCLAS1= 'IMAGE   '           / hdu format classification", $
	"HDUCLAS2= 'DATA    '           / hdu type classification", $
	"ERRDATA = 'ERRS    '           / name of errs extension", $
	"QUALDATA= 'QUAL    '           / name of qual extension", $
	"HIERARCH ESO TEL ALT = "+string(float(90.-scans(is).za)) $
				 +" / Altitude (deg)", $
	"END     "]
writefits,fitsfile,flux,header
ext_header=["XTENSION= 'IMAGE'              / IMAGE extension", $
	"BITPIX  =                  -32 / Number of bits per data pixel", $
	"NAXIS   = 1                    / Number of data axes", $
	"NAXIS1  =               "+string(nc)+" /", $
	"PCOUNT  = 0                   / No Group Parameters", $
	"GCOUNT  = 1                   / One Data Group", $
	"EXTNAME =               'ERRS' /", $
	"CRPIX1  =      1.000000000E+00 /", $
	"CRVAL1  =     "+string(llambda(0))+" / nm", $
	"CDELT1  =     "+string(dlambda)+" / nm", $
	"CTYPE1  = 'LINEAR  '           /", $
	"BUNIT   = 'ADU     '           /", $
	"HDUCLASS= 'ESO     '           / hdu classification", $
	"HDUDOC  = 'DICD    '           / hdu reference document", $
	"HDUVERS = 'DICD V6.0'          / hdu reference document version", $
	"HDUCLAS1= 'IMAGE   '           / hdu format classification", $
	"HDUCLAS2= 'DATA    '           / hdu type classification", $
	"HDUCLAS3= 'RMSE    '           / hdu info classification", $
	"SCIDATA = 'FLUX    '           / name of data extension", $
	"QUALDATA= 'QUAL    '           / name of qual extension", $
	"END     "]
writefits,fitsfile,errs,ext_header,/append
ext_header=["XTENSION= 'IMAGE'              / IMAGE extension", $
	"BITPIX  =                  -32 / Number of bits per data pixel", $
	"NAXIS   =                    1 / Number of data axes", $
	"NAXIS1  =         "+string(nc)+" /", $
	"PCOUNT  = 0                    / No Group Parameters", $
	"GCOUNT  = 1                    / One Data Group", $
	"EXTNAME = 'QUAL'               /", $
	"CRPIX1  =      1.000000000E+00 /", $
	"CRVAL1  =     "+string(llambda(0))+" / nm", $
	"CDELT1  =     "+string(dlambda)+" / nm", $
	"CTYPE1  = 'LINEAR  '           /", $
	"HDUCLASS= 'ESO     '           / hdu classification", $
	"HDUDOC  = 'DICD    '           / hdu reference document", $
	"HDUVERS = 'DICD V6.0'          / hdu reference document version", $
	"HDUCLAS1= 'IMAGE   '           / hdu format classification", $
	"HDUCLAS2= 'DATA    '           / hdu type classification", $
	"HDUCLAS3= 'RMSE    '           / hdu info classification", $
	"SCIDATA = 'FLUX    '           / name of data extension", $
	"QUALDATA= 'ERRS    '           / name of qual extension", $
	"END     "]
writefits,fitsfile,qual,ext_header,/append
;
; Create links to TAPE1 and TAPE3
spawn,'which molecfit',reply
molecfit_bin=strmid(reply,0,strlen(reply)-8)
lblrtm=molecfit_bin+'lblrtm'
lnfl=molecfit_bin+'lnfl'
spawn,'ln -s /usr/local/src/molecfit/data/hitran/aer_v_3.2 TAPE1'
spawn,'ln -s '+!oyster_dir+'source/proprietary/tellrem/TAPE3 TAPE3'
;
; Create GDAS file from ftp://ftp.eso.org/pub/dfs/pipelines/skytools/molecfit/GDAS/
gdas_dir=!oyster_dir+'source/proprietary/tellrem/gdas/'
gdas_files=specname(file_search(gdas_dir+'C-70.4-24.6D????-??-??T??.gdas'))
obsdates=date2jd(strmid(gdas_files,12,10))
; Locate gdas file, recompute/reformat into GDAS format, save to tellrem/GDAS folder
obsdate=date2jd(date)
i=where(abs(obsdates-obsdate) eq min(abs(obsdates-obsdate)))
gdas_file=gdas_files(i(0))
status=dc_read_free(gdas_dir+gdas_file,press,hgt,temp_k,relhum,/col,ignore=['#'])
temp=temp_k-273.15
dp=dewpoint(temp,relhum)
y=strmid(date,0,4)
m=strmid(date,5,2)
d=strmid(date,8,2)
h=strmid(gdas_file,23,2)
GDAS_dir=!oyster_dir+'source/proprietary/tellrem/GDAS/'
GDAS_file='GDAS'+y+'_'+m+'_'+d+'_'+h
openw,unit,GDAS_dir+GDAS_file,/get_lun
printf,unit,' YR: '+y+'   MON: '+m+'   DAY: '+d+'   HOUR: '+h $
		   +'    AT POSITION: 290.6  66.4  LAT.:-24.63  LON.: -70.40'
printf,unit,''
printf,unit,' PRESS HGT(MSL) TEMP DEW PT  WND DIR  WND SPD'
printf,unit,' HPA       M      C     C       DEG     M/S'
printf,unit,' E = Estimated Surface Height'
printf,unit,''
wdr=0.0
wsp=0.0
for i=0,n_elements(press)-1 do $
	printf,unit,press(i),hgt(i),temp(i),dp(i),wdr,wsp, $
	format='(i8,2x,6(f9.3,2x))'
free_lun,unit
;
; Write tellrem.info file
openw,unit,'tellrem.info',/get_lun
printf,unit,GDAS_dir
printf,unit,!oyster_dir+'source/proprietary/tellrem/equ.atm'
printf,unit,lblrtm
printf,unit,pwd()
printf,unit,'gravity'
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro gravity_tellrem,target,extract=extract
;
; Run tellrem package (Rudolf et al. 2015) for telluric correction.
; The tellrem processes for all 4 telescopes are run concurrently in IDLbridge
; objects.
;
; The code is loaded from an IDL save files containing all compiled procedures.
; The file tellrem.cpr in oyster/source is produced as follows:
; The needed code in tellrem, mpfit, and idlastro is combined three separate
; code libraries: tellrem_lib.pro, mpfit_lib.pro, and idlastro_lib.pro by
; running, e.g., cat *.pro > tellrem_lib.pro. Then oyster is started, and the
; libraries are compiled. The command "save,filename='tellrem.cpr',/routines"
; write the compiled code which can be loaded by the IDLBridge objects.
;
; If this routine has changed, do the following after starting oyster:
; .run /home/chummel/oyster/source/mpfit/mpfit_lib.pro
; .run /home/chummel/oyster/source/idlastro/idlastro_lib.pro
; .run /home/chummel/oyster/source/proprietary/tellrem/tellrem_lib.pro
; save,filename='tellrem.cpr',/routines
; mv tellrem.cpr ~/oyster/source
;
; Run gravity_tellcorr,prepare=4 first.
;
common LocalGravityTellCorr,basedir,suffix,par_files,tel
;
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
prepare=4
;
; Reconstruct folders
spawn,'find . -name gravity.fits',files
if keyword_set(target) then begin
	index=where(strpos(files,target) ge 0,count)
	if count eq 0 then begin
		print,'Error: no files found for this target!'
		return
	endif
	files=files(index)
endif
n_files=n_elements(files)
if strlen(files(0)) eq 0 then begin
	print,'Error: no files gravity.fits found!'
	return
endif
basedir=strarr(n_files)
station=strarr(n_files)
for is=0,n_files-1 do begin
	words=nameparse(files(is),'/')
	if n_elements(words) eq 4 then begin
		basedir(is)=words(1)
		station(is)=words(2)
	endif
endfor
index=where(strlen(basedir) ne 0)
basedir=unique(basedir(index))
n_scans=n_elements(basedir)
;
if keyword_set(extract) then goto,EXTRACT
;
clean_bridge=objarr(prepare)
for j=0,prepare-1 do begin
	clean_bridge(j)=obj_new('IDL_IDLBridge')
	clean_bridge(j)->execute,"restore,'~/oyster/source/tellrem.cpr'"
endfor
;
; Spawn tellrem for all files
for is=0,n_scans-1 do begin
	cd,basedir(is),current=top_dir
	spawn,'ls',suffix
	if n_elements(suffix) ne prepare then begin
		print,'Error: number of folders nor equal number of telescopes!'
		return
	endif	       
;	Call tellrem (default tellrem parameters for NIR3 region)
	for ib=0,prepare-1 do begin
		cd,suffix(ib),current=old_dir
		spawn,'pwd',local_dir & local_dir=local_dir(0)
		clean_bridge(ib)->execute,"cd,'"+local_dir+"'"
		clean_bridge(ib)->execute,"tellrem",/nowait
		cd,old_dir
	endfor
	notdone = 1
	while notdone do begin
		done=0
		for j=0,prepare-1 do $
			done = done+clean_bridge[j]->Status()
		if done eq 0 then notdone=done
	endwhile
	cd,top_dir
endfor
print,''
print,'Cleaning up processes...'
for j=0,prepare-1 do obj_destroy,clean_bridge(j)
;
EXTRACT:
; Extract the telluric corrected spectra to pollux.fits
for is=0,n_scans-1 do begin
	cd,basedir(is),current=top_dir
	spawn,'ls',suffix
	if n_elements(suffix) ne prepare then begin
		print,'Error: number of folders nor equal number of telescopes!'
		return
	endif	       
	for ib=0,prepare-1 do begin
		cd,suffix(ib),current=old_dir
;		Read tellrem output
		restore,'tellrem_spectra.sav'
		lambda_aa=tellremspecs(0).wcln3
		flux=tellremspecs(0).cln3
		error=tellremspecs(0).ecln3
;		Read header from input file, then write corrected spectra
		fitsfile='gravity.fits'
		header=headfits(fitsfile)
		f=mrdfits(fitsfile)
		e=mrdfits(fitsfile,1,h_ext1)
		q=mrdfits(fitsfile,2,h_ext2)
;		Make sure corr. spectra start at same wavelength
		if fitshparse(header,'CRVAL1') ne fitshparse(h_ext1,'CRVAL1') $
			then begin
			print,'Error: starting value of new spectrum changed!'
			retall
		endif
		qual=q(0:n_elements(error)-1)
;		Update header keywords (naxis1) in new headers
		index=where(strpos(header,'NAXIS1') ge 0) & index=index(0)
		naxis1=header(index)
		j=strpos(naxis1,string(n_elements(f)))
		strput,naxis1,string(n_elements(flux)),j
		header(index)=naxis1
		index=where(strpos(h_ext1,'NAXIS1') ge 0) & index=index(0)
		naxis1=h_ext1(index)
		j=strpos(naxis1,string(n_elements(f)))
		strput,naxis1,string(n_elements(flux)),j
		h_ext1(index)=naxis1
		index=where(strpos(h_ext2,'NAXIS1') ge 0) & index=index(0)
		naxis1=h_ext2(index)
		j=strpos(naxis1,string(n_elements(f)))
		strput,naxis1,string(n_elements(flux)),j
		h_ext2(index)=naxis1
;		Write new file
		fitsfile='tellrem.fits'
		writefits,fitsfile,flux,header
		writefits,fitsfile,error,h_ext1,/append
		writefits,fitsfile,qual,h_ext2,/append
		cd,old_dir
	endfor
	cd,top_dir
endfor
;
print,'Finished running tellrem.'
;
end
;************************************************************************Block 8
pro get_tracks,files_astroreduced
;
for i=0,n_elements(files_astroreduced)-1 do begin
	d=mrdfits(file_astroreduced,4,h)
	if i eq 0 then tracks=d else tracks=[[tracks],[d]]
endfor
;
end
;-------------------------------------------------------------------------------
pro gravity_display_acq,f
;
; Display GRAVITY acquisition image for file "f". If not specified, look
; for all raw files in current directory and display the first found.
;
if n_elements(f) eq 0 then begin
	f=file_search('GRAVI*.fits')
endif
d=mrdfits(f(0),4,h)
tvscl,sqrt(d(*,*)>0)
;
end
;-------------------------------------------------------------------------------
pro gravity_north_angle,f
;
; Look for keyword QC.ACQ.FIELDi.NORTH_ANGLE (i=1-4) and if not found, compute
; from INS.SOBJ.X and Y and DROTOFFi (i=1-4).
; Code from gravi_pfits.c, gravi_pfits_get_fangle_acqcam
;
if n_elements(f) eq 0 then begin
	f=file_search('reflex_end_products/*/GRAVI*/*_ASTROREDUCED.fits')
endif
;
r=dblarr(4)
na=dblarr(4)
;
modified=0
;
for i=0,n_elements(f)-1 do begin
	print,specname(f(i))
	d=mrdfits(f(i),0,h,/silent)
	x=fitshparse(h,'INS.SOBJ.X')
	y=fitshparse(h,'INS.SOBJ.Y')
	r(0)=fitshparse(h,'INS.DROTOFF1')
	r(1)=fitshparse(h,'INS.DROTOFF2')
	r(2)=fitshparse(h,'INS.DROTOFF3')
	r(3)=fitshparse(h,'INS.DROTOFF4')
	for j=0,3 do begin
		na(j)=(-atan(x,y)*180./!pi-r(j)+270) mod 360
		if na(j) ge 180 then na(j)=na(j)-360
		if na(j) lt -180 then na(j)=na(j)+360
		js=string(j+1,format='(i1)')
		keyword='QC ACQ FIELD'+js+' NORTH_ANGLE'
		n_a=fitshparse(h,keyword,/quiet)
		if strlen(n_a) ne 0 then begin
			print,'Computed NA: ',string(na(j),format='(f10.5)')
			print,'Found in Hd: ',string(n_a,format='(f10.5)')
		endif else begin
			modified=1
			index=where(strpos(h,'ESO QC') ge 0,count)
			new_h=h(0:index(count-1))
			new_h=[new_h,'HIERARCH ESO '+keyword+' = ' $
			       +string(na(j),format='(f10.5)') $
			       +' / North angle']
			new_h=[new_h,h(index(count-1)+1:n_elements(h)-1)]
			h=new_h
		endelse
	endfor
	if modified then modfits,f(i),0,new_h
endfor
;
end
;-------------------------------------------------------------------------------
pro gravity_acq_scale,f,raw_dir=raw_dir,reduced_dir=reduced_dir,radius=radius
;
; Analyze the acquisition images to measure the separation of the two stars
; in pixels, and compare to the separation in arc seconds.
;
; The two images are cross-correlated and the Gaussians are fit to the peaks.
;
; Run this procedure in a folder with two directories: data_with_raw_calibs
; and reflex_end_products.
;
; See also: gravi_acqcam.c
;
; Radius: of circular mask placed on primary and secondary, default is 14 pixels
;
common LocalGravity_Acq_Scale,xy_sec1,xy_sec2
;
if not keyword_set(raw_dir) then raw_dir='data_with_raw_calibs'
if not keyword_set(reduced_dir) then reduced_dir='reflex_end_products/*'
if not keyword_set(radius) then radius=14
;
init=1
xy_sec1=0
xy_sec2=0
;
if n_elements(f) eq 0 then begin
	f=file_search(reduced_dir+'/'+'GRAVI*/*_ASTROREDUCED.fits')
endif
;
START:
;
; Remove all plotting windows
;
for i=0,n_elements(f)-1 do begin
	print,'Processing '+specname(f(i))
	d=mrdfits(f(i),0,h,/silent)
	rf=raw_dir+'/'+fitshparse(h,'ARCFILE')
	print,'Raw file',specname(rf)
	sf=raw_dir+'/'+fitshparse(h,'PRO.REC1.RAW2.NAME')
	print,'Sky file',specname(sf)
	modified=0
;
;	Get separation in arcsec from header
	rd=mrdfits(rf,0,rh)
	sobj_x=float(fitshparse(rh,'INS.SOBJ.X'))
	sobj_y=float(fitshparse(rh,'INS.SOBJ.Y'))
	sobj_offx=float(fitshparse(rh,'INS.SOBJ.OFFX'))
	sobj_offy=float(fitshparse(rh,'INS.SOBJ.OFFY'))
;	From  line 1315 in gravi_acqcam.c
	dx_in=sobj_x - sobj_offx
	dy_in = sobj_y - sobj_offy
	rho_in = sqrt(dx_in*dx_in + dy_in*dy_in)
; 
;	Prepare and analyze acquisition images
	images=mrdfits(rf,4,rh,/silent)
	sizes=size(images,/dim)
	n_img=sizes(2)	; Number of acquisition images recorded
	skyimg=mrdfits(sf,4,sh,/silent) & skyavg=total(skyimg,3)/n_img
	for j=0,n_img-1 do images(*,*,j)=images(*,*,j)-skyavg
	reduction_factor=4
	if n_elements(sizes) eq 3 then begin
		idims=nint(sizes(0:1)/reduction_factor)	
	endif else begin
		idims=[nint(sizes(0)/reduction_factor),sizes(1)/2]
		n_img=1
	endelse
	n_patches=4
;
;	Small patch
	unitvec=[1,1,1]
	patch_v=[1,0,-1]
	patch_i=transpose(patch_v#unitvec)
	patch_j=reverse(patch_v)#unitvec
;
;	Larger patch
	unitvec=[1,1,1,1,1]
	patch_v=[2,1,0,-1,-2]
	patch_i=transpose(patch_v#unitvec)
	patch_j=reverse(patch_v)#unitvec
;
for j=0,n_patches-1 do begin
	wdall
	patches=images(j*idims(0):(j+1)*idims(0)-1,0:idims(0)-1,*)
;	tp=total(patches,3)
;	print,'Maximum of integrated patch: ',max(tp)
;	tvscl,tp
;
	IF 0 THEN BEGIN
;	Remove bad pixels
	for k=0,n_img-1 do begin
		p=patches(*,*,k)
		m=medianve(p,e)
		t1=p-shift(p,0,1)
		t2=p-shift(p,0,-1)
		index=where(t1 gt e*40 or t2 lt -e*40,count)
		for l=0,count-1 do begin
			ij=whereindex(index(l),p)
			p(patch_i+ij(0),patch_j+ij(1))=m
		endfor
		patches(*,*,k)=p
	endfor
	tp=total(patches,3)
	print,'Maximum of cleaned patch: ',max(tp)
	window,xsize=idims(0),ysize=idims(1),/free
	tvscl,tp
;	Remove bright artefacts
	repeat begin
		print,max(tp)
		index=where(tp eq max(tp))
		ij=whereindex(index,tp)
		tp(patch_i+ij(0),patch_j+ij(1))=0
	endrep until max(tp) lt 160000
	window,xsize=idims(0),ysize=idims(1),/free
	tvscl,tp < 10000
	ENDIF
;
	patch=convol_fft(patches(*,*,0),/auto_correlation)
	for k=1,n_img-1 do begin
		patch_k=patches(*,*,k)
		patch=patch+convol_fft(patch_k,/auto_correlation)
	endfor
;	patch=median(patch,3)
	window,xsize=idims(0),ysize=idims(1),/free
	tvscl,patch
;
;	1st iteration, detect primary peak
	index=where(patch eq max(patch)) & index=index(0)
	xy=whereindex(index,patch)
	fita=[min(patch),max(patch),3,3,xy(0),xy(1),0]
	mask=notch2d(fltarr(idims),xy,radius)
	r=gauss2dfit(patch*mask,fita)
	patch=(patch-r)*abs(mask-1)
	fita_cen=fita(4:5)
;	print,'Center',fita_cen
	window,xsize=idims(0),ysize=idims(1),/free
	tvscl,sqrt(patch>0)
;
;	2nd iteration, detect one of the secondary peaks
	mask=notch2d(fltarr(idims),idims/2,80)
	index=where(patch eq max(patch*mask)) & index=index(0)
	xy=whereindex(index,patch)
	if total(xy_sec1) gt 0 then xy=xy_sec1
	fita=[0.,max(patch),3,3,xy(0),xy(1),0]
	mask=notch2d(fltarr(idims),xy,radius)
	r=gauss2dfit(patch*mask,fita)
	patch=(patch-r)*abs(mask-1)
	fita_sec1=fita(4:5)
;	print,'Secondary 1:',fita_sec1
	window,xsize=idims(0),ysize=idims(1),/free
	tvscl,sqrt(patch>0)
	answer='No'
	if total(xy_sec1) eq 0 then answer=$
	dialog_message('Secondary peak 1 correctly identified?',/question)
	if answer eq 'Yes' then xy_sec1=xy
;
;	3rd iteration, detect the other secondary peak
	mask=notch2d(fltarr(idims),idims/2,80)
	index=where(patch eq max(patch*mask)) & index=index(0)
	xy=whereindex(index,patch)
	if total(xy_sec2) gt 0 then xy=xy_sec2
	fita=[0.,max(patch),3,3,xy(0),xy(1),0]
	mask=notch2d(fltarr(idims),xy,radius)
	r=gauss2dfit(patch*mask,fita)
	patch=(patch-r)*abs(mask-1)
	fita_sec2=fita(4:5)
;	print,'Secondary 2:',fita_sec2
	window,xsize=idims(0),ysize=idims(1),/free
	tvscl,sqrt(patch>0)
	answer='No'
	if total(xy_sec2) eq 0 then answer=$
	dialog_message('Secondary peak 2 correctly identified?',/question)
	if answer eq 'Yes' then xy_sec2=xy
;
	if total(xy_sec1) gt 0 and total(xy_sec2) and init then begin
		init=0
		print,'All patches defined, now analyzing all files...'
		goto, START
	endif
;
;	Divide separation of cross-correlation peaks by 2
	sep=sqrt(total((fita_sec2-fita_sec1)^2))/2
	print,'Separation (pixels):',sep
;	From  line 1539 in gravi_acqcam.c
	pscale=rho_in/sep
	print,'Scale: ["/pix]',pscale
; 	Look for keyword QC.ACQ.FIELDi.SCALE and if not found, compute and add
	keyword='QC ACQ FIELD'+string(j+1,format='(i1)')+' SCALE'
	p_scale=float(fitshparse(h,keyword,/quiet))
	if p_scale ne 0 then begin
		print,'Computed SCALE:',float(pscale)
		print,'Found in Header:',p_scale
	endif else begin
		modified=1
		index=where(strpos(h,'ESO QC') ge 0,count)
		new_h=h(0:index(count-1))
		new_h=[new_h,'HIERARCH ESO '+keyword+' = ' $
		       +string(pscale,format='(f10.5)') $
		       +' / Scale']
		new_h=[new_h,h(index(count-1)+1:n_elements(h)-1)]
		h=new_h
	endelse
endfor
if modified then modfits,f(i),0,new_h
endfor
;
end
;-------------------------------------------------------------------------------
pro gravity_split,file
;
; Not functional currently!
;
oivis=0
oivis2=0
oit3=0
oiflux=0
;
read_oidata,file,oiarray, oitarget,oiwavelength,oivis, oivis2, oit3, oiflux
;
words=nameparse(file,'.')
;
if n_elements(oiwavelength) ne 2 then begin
	print,'This file does not contain split polarisation observations!'
	return
endif
;
n=n_elements(oiarray)
n_oivis=n_elements(oivis)
n_oivis2=n_elements(oivis2)
n_oiflux=n_elements(oiflux)
;
for i=0,1 do begin
	oiwavelength_i=oiwavelength(i)
	if n_oivis ne 0 then oivis_i=oivis(i*n_oivis/2:(i+1)*n_oivis/2-1)
	if n_oivis2 ne 0 then oivis2_i=oivis2(i*n_oivis2/2:(i+1)*n_oivis2/2-1)
	if n_oiflux ne 0 then oiflux_i=oiflux(i*n_oiflux/n:(i+1)*n_oiflux/n-1)
;	print,basename(file)+'_P'+string(i+1,format='(i1)')+'.fits'
; This doesn't work yet...
stop
	write_oidata,basename(file)+'_P'+string(i,format='(i1)')+'.fits', $
		oiarray,oitarget,oiwavelength_i,oivis_i,oiflux_i
endfor
;
end
;-------------------------------------------------------------------------------
pro gravity_rename_astroreduced,datetime
;
; Copy ASTROREDUCED files from reflex_end_products/YYYY-MM-DDTHH:MM:SS to dir.
; astroreduced, renaming the files to GRAVI.YYYY-MM-DDTHH:MM:SSS_ASTROREDUCED
;
if n_elements(datetime) eq 0 then begin
	print,'You must specify argument YYYY-MM-DDTHH:MM:SS'
	return
endif
;
f=file_search('reflex_end_products/'+datetime+'/GRAVI.*/*_ASTROREDUCED*.fits')
if strlen(f(0)) eq 0 then begin
	print,'No files found!'
	return
endif
;
spawn,'mkdir -p astroreduced'
;
for i=0,n_elements(f)-1 do begin
	newfile=specname(f(i))
	sdirs=nameparse(f(i),'/')
	words=nameparse(sdirs(2),'.')
	obsdt=words(1)
;
	words=nameparse(newfile,'_') & n=n_elements(words)
	if n_elements(words) eq 4 then $
	newfile=strjoin(words(0:n-2),'_')+'_'+obsdt+'_'+words(3)
	if n_elements(words) eq 5 then $
	newfile=strjoin(words(0:n-3),'_')+'_'+obsdt+'_'+words(3)+'_'+words(4)
	spawn,'cp -f '+f(i)+' astroreduced/'+newfile
endfor
;
print,'Copies of your files are now in astroreduced!'
;
end
;-------------------------------------------------------------------------------
pro gravity_oivis_check,filespec,keyword
;
; Check if OI_VIS extension has the column passed by keyword
; Use also as follows: gravity_oivis_check,file_search('*.fits'),'OI_VIS'
;
files=file_search(filespec)
;
for i=0,n_elements(files)-1 do begin
	ext=find_oiextn(files(i),'OI_VIS')
	if ext eq 0 then continue
	d=mrdfits(files(i),ext,h,/silent)
	if tag_exists(keyword,d) then begin
		print,files(i),': ',keyword,' was found'
	endif else begin
		print,files(i),': ',keyword,' not found'
	endelse
endfor
;
end
;************************************************************************Block 9
pro fluxplots,fits_file
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(fitsfile) eq 0 then $
	get_oifits,'SCI_BP_Cru_2017-02-20T06:41:17_SINGLE_SCI_VIS.fits' $
	else get_oifits,fits_file
gravity_compact
index=where(scans.star eq 'BP_Cru')
j=0 ; station index
l=genconfig.wavelength(*,0)*1e6
p=reform(scans(index).photometry(j,0,*))
;
!x.title='Wavelength ['+greek('mu')+'m]'
!p.multi=[0,1,2]
; Plot spectra
plot,l,p(*,0),psym=0,title='First and second (red) spectrum'
oplot,l,p(*,1),psym=0,color=tci(2)
; Plot ratio of the two spectra
s=p(*,0)/p(*,1)
plot,l,s,title='Ratio first/second spectrum'
r=poly_fit(l,s,1,sfit)
oplot,l,sfit,color=tci(3)
;
window,/free
!p.multi=[0,1,2]
; Plot both spectra, the second one scaled with the fit results
plot,l,p(*,0),psym=0,title='First spectrum and scaled second (green) spectrum'
oplot,l,p(*,1)*sfit,color=tci(3),psym=0
; Plot difference between spectra
plot,l,p(*,0)-p(*,1)*sfit,psym=0,title='Difference between spectra'
print,'S/N=',median(p(*,0))/stddev(p(*,0)-p(*,1)*sfit)
;
!p.multi=[0,2,2]
plot,l,reform(scans(index(1)).photometry(0,0,*)),ytitle='Flux AT1'
plot,l,reform(scans(index(1)).photometry(1,0,*)),ytitle='Flux AT2'
plot,l,reform(scans(index(1)).photometry(2,0,*)),ytitle='Flux AT3'
plot,l,reform(scans(index(1)).photometry(3,0,*)),ytitle='Flux AT4'
;
end
;-------------------------------------------------------------------------------
