/*+
 *  Name:
 *     EMS1FCERR

 *  Purpose:
 *     Get the error message associated with a facility error code.

 *  Language:
 *     Starlink ANSI C

 *  Invocation:
 *     ERRSTR = EMS1_FCERR( ERRNUM )

 *  Description:
 *     This C function is designed to be called from FORTRAN. It 
 *     uses the facility code files generated by the MESSGEN utility (see
 *     SUN/185) to return the error message associated with the status 
 *     code ERRNUM
 *     In the case of failure to obtain a valid facility error message,
 *     the following error idents may be displayed together with the numbers
 *     derived from the given error code.
 *     FACERR__BADARG  The given error code was not a valid facilty error code.
 *     FACERR__NOFAC   The required facilty error file was not found.
 *     FACERR__NOMSG   The required message number was not found in the file.

 *  Arguments:
 *     ERRNUM = INTEGER (Given)
 *        The error number as generated by the MESSGEN utility (VMS status
 *        code format).

 *  Authors:
 *     BKM: B.K. McIlwrath (STARLINK)
 *     AJC: A.J.Chipperfield (STARLINK)
 *     {enter_new_authors_here}

 *  History:
 *     12_AUG-1994 (BKM):
 *        Original version.
 *     20-SEP-1994 (AJC):
 *        Add special case SAI__OK to ems1_get_facility_err
 *        Reorganize error conditions and messages slightly
 *     21-JUN-1995 (AJC):
 *        Re-organise to use ems1_starf to look along EMS_PATH
 *        and, failing that, at ../help/fac_xxx_err relative to
 *        directories in PATH.
 *     02-AUG-1995 (BKM):
 *        Ensure that the file descriptor used to locate messages is closed.
 *        Rename get_facility_error to ems1_get_facility_error and make global
 *        (used by ICL).
 *     15-NOV-1996 (BKM):
 *        Correct bug - errstr being used in place of facility_name
 *     30-APR-1998 (AJC):
 *        Correct ident length and buffer length to allow 15 and 80 character
 *        strings respectively.
 *      1-JUN-1999 (AJC):
 *        Use changed name of ems1Starf
 *     21-MAR-2001 (AJC):
 *        Addition of trailing blanks not required now ems1Fcerr called from C.
 *     {enter_further_changes_here}

 *  Bugs:
 *     {note_any_bugs_here}

 *- */

/*  Macro Statements: */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "sae_par.h"
#include "ems.h"
#include "ems_par.h"

/* Facility system interface function declaration */
void ems1_get_facility_error
( 
unsigned int errcode,	/* error code in messgen format (given) */
char **facility_name,	/* facility name (returned) */
char **error_ident,	/* message identifier (returned) */
char **error_text	/* message text (returned */
);

/*  Function Definition: */
void ems1Fcerr( char *errstr, int *errnum_p ) {

/*  Local Variables: */
      char *facility_name;		/* facility name */
      char *message_ident;		/* message identifier */
      char *message_text;		/* message text */

      int length;                       /* Length of error text */
      int errlen=EMS__SZMSG;

/* Get string pointers. */
      ems1_get_facility_error( *errnum_p, &facility_name, &message_ident,
		      &message_text);

/*    Load message prefix. */
      if ((length = strlen(facility_name) + strlen(message_ident) + 3) 
        < errlen) {
         (void)strcpy( errstr, facility_name);
	 (void)strcat( errstr, "__");
         (void)strcat( errstr, message_ident);
      } else
         length = 0;

/*    Load message text */
      if (length != 0)
 	 if ((length = length + strlen(message_text) + 3) < errlen) {
	     (void)strcat( errstr, ", ");
             (void)strcat( errstr, message_text);
             length = strlen( errstr);
         }

      return; 
}

/*+ EMS1_GET_FACILITY_ERROR - interface to the facility error system */

void ems1_get_facility_error
( 
unsigned int errcode,	/* error code in messgen format (given) */
char **facility_name,	/* facility name (returned) */
char **error_ident,	/* message identifier (returned) */
char **error_text	/* message text (returned) */
)
{

/* This function translates the given 'errcode' to a facility number and a
 * message number. (As errcode is VMS compatable there is an severity code
 * available as well but this is currently unused on Unix).
 *
 * The facility code number (fac_code) is used to locate a file on the search
 * path defined by environment variable EMS_PATH (if defined).
 * If EMS_PATH is not defined, or does not produce a file, the file is looked
 * for at a pathname relative to each of the directories in the user's PATH.
 *
 * The error corresponding to the 'message number' is located and the three
 * strings facility name, message name and message text are returned. Suitable
 * defaults are used for any error condition.
 *
 * Author:
 *    B.K. McIlwrath (STARLINK)
 * History:
 *    Created: bkm, 18/08/1994
 *    Modified: bkm 02/08/95
 *	Close file descriptor used to search facility error files
 *
 */ 

#define MAXFAC 10		/* Cache size */
#define BUFSIZE 82

    static struct {
	int fac_code;
	char *file;
    } facilities[MAXFAC];	/* Facility location cache */
    static int cur_fac;
    static char facility[10], ident[16], text[BUFSIZE-1]; /* Must be static */

    int fac_code, mess_number, i;
    char *s, *s1;
    FILE *fp=NULL;
    char filename[24], buffer[BUFSIZE];
    char *foundfile;
    int pathlen;

/* Code start point */
    *facility_name = facility;
    *error_ident   = ident;
    *error_text    = text;

/* Set default facility name */
    strcpy(facility, "FACERR");

/* Check the given error code */
    if ( (errcode & 0x8000000) == 0) { /* Bit 27 unset - not a facility error */
       if ( errcode == SAI__OK ) {
           strcpy(facility, "SAI");
           strcpy(ident, "OK");
           sprintf(text, "application success status");
       } else {
	   strcpy(ident, "BADARG");
	   sprintf(text, 
	   "error %d (not a facility error code)",
	   errcode);
	return;
       }
       return;
    }

/* Get the facility identifier code and the message number */
    fac_code    = (errcode >> 16) & 0x7ff;
    mess_number = (errcode >> 3)  & 0xfff;
/*
 * Locate the facility definition file (fac_<facility_code>_err) as obtained
 * from the message source file using the messgen utility. 
 */
    for (i=0; i<cur_fac; i++)		/* Search facility cache */
	if (facilities[i].fac_code == fac_code) {
	    fp = fopen(facilities[i].file, "r");
	    break;
	}
    if (i == cur_fac) {
/* Attempt to find the file along EMS_PATH
 */
        fp = NULL;
	sprintf(filename, "fac_%d_err", fac_code);	/* File to find */
        if ( ems1Starf( "EMS_PATH", filename, "r", &foundfile, &pathlen ) ) {
	   fp = fopen(foundfile, "r");
        } else {
/*   if unsuccessful, try relative to PATH
 */
	   sprintf(filename, "../help/fac_%d_err", fac_code);
           if ( ems1Starf( "PATH", filename, "r", &foundfile, &pathlen ) ) 
           {
 	      fp = fopen(foundfile, "r");
           }
        }
	if (fp != NULL) {		/* Cache facility code and filename */
	    facilities[cur_fac].fac_code = fac_code;
	    facilities[cur_fac].file = (char *) malloc(strlen(foundfile) + 1);
	    strcpy(facilities[cur_fac].file, foundfile);
	    cur_fac++;
	}
    }
/*
 * Set defaults for message name and description
 */
    sprintf(text, "error %d (fac=%d,messid=%d)",
	    errcode, fac_code, mess_number);
    if (fp == NULL) {		/* Facility file not found */
	strcpy(ident, "NOFAC");
        strcat(text, " no facility message file");
	return;
    } else {
        strcpy(ident, "NOMSG");
    }
/*
 * Read definition file to locate specific error code
 */
    fgets(buffer, BUFSIZE, fp);
    if (strncmp(buffer, "FACILITY", 8) != 0) {
        strcpy(ident, "BADFIL");
        strcat(text, " bad message file format");
	cur_fac--;
	fclose(fp);
        return;
    }
    buffer[strlen(buffer)-1] = '\0';	/* remove newline */
    strcpy(facility, &buffer[9]);
    while (fgets(buffer, BUFSIZE, fp) != NULL) {
	if ((i = strtol(buffer, &s, 10)) != mess_number)
	    continue;
	buffer[strlen(buffer)-1] = '\0';
	s++;
	s1 = strchr(s, ',');
	*s1 = '\0';
	strcpy(ident, s);
	strcpy(text, ++s1);
	fclose(fp);
        return;
    }
    fclose(fp);
/*
 *  If we drop out here, the given message was not found.
 */
    strcat(text, " message number not found");
    return;
}

