;*******************************************************************************
; File: fitwidget.pro
;
; Description:
; ------------
; Container of IDL widget scripts for fitting in AMOEBA software. 
;
; Block directory:
; ----------------
;
; Block 0: genparms,starparms,binparms,posparms
;
; Block 1: ww_modelfit,ww_modelfitdestroyed,ww_modelfitparameters,ww_modellist
;	   ww_modelfitok
;
; Block 2: ww_fitinterferometry,ww_fitinterferometrydestroyed,
;	   ww_fitinterferometryok
;
; Block 3: ww_fitastrometry,ww_fitastrometrydestroyed,ww_fitastrometrycomp,
;          ww_ellipsefitoptions,ww_orbitfitoptions
;
; Block 4: ww_marquardtfit,ww_marquardtfitdestroyed,ww_marquardtfitoptions
;	   ww_setalamda,ww_settolerance,ww_setchifr
;
;************************************************************************Block 0
function genparms
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
return,[ $
	'RV' $
	]
return,[ $
	'RV', $
	'SM' $
	]
;
end
;-------------------------------------------------------------------------------
function starparms
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
num_wave=n_elements(gen_model.wavelengths)
mag_parms='Magnitude_'+ $
	strcompress(string(gen_model.wavelengths,format='(f5.2)'),/remove_all)
;
return,[ $
	'Mass', $
	'Diameter', $
	'Width', $
;	'Omega', $
	'Ratio', $
	'PA', $
	'Tilt', $
	'Teff', $
	mag_parms $
	]
;
end
;-------------------------------------------------------------------------------
function binparms
;
return,[ $
;	'MassRatio', $
	'Semimajoraxis', $
	'Eccentricity', $
	'Inclination', $
	'Periastron', $
	'Apsidalmotion', $
	'Ascendingnode', $
	'Period', $
	'Epoch' $
	]
;
end
;-------------------------------------------------------------------------------
function posparms
;
return,[ $
	'Rho', $
	'Theta' $
	]
;
end
;************************************************************************Block 1
pro ww_modelfit
;
; GUI for the model fits to combined data.
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
common WwFitWids,list1_wids,list2_wids
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; Check for existence of model
if n_elements(gen_model) eq 0 then begin
	print,'***Error(WW_MODELFIT): no model loaded!'
	return
endif
;
; Check for resolved flux component
index=where(star_model.type eq 0,n)
if n ne 0 then begin
	print,'Model has resolved flux component!'
	print,'Its value must be set manually, e.g.:'
	print,'star_model('+strcompress(string(n+1),/remove_all) $
		+').magnitudes= ', $
		strcompress(string(float(star_model(index(0)).magnitudes)), $
		/remove_all)
endif
;
; Return if widget is already displayed
if n_elements(modelfit_wid) eq 0 then modelfit_wid=0
if modelfit_wid ne 0 then begin
	print,'FitWidget is already displayed!'
	return
endif
;
; Create array to hold widget IDs of parameter lists
; list1 is for orbital parameters, list2 for relative comp. positions
if n_elements(list1_wids) eq 0 then list1_wids=lonarr(num_binary()+1)
if n_elements(list2_wids) eq 0 then list2_wids=lonarr(num_binary()+1)
;
; Initialize/reset options (disabled 12.12.2023 and moved to readmodel)
if n_elements(parameters) eq 0 then begin
	if init_modelfit() ne 0 then begin
		print,'No model loaded!'
		return
	endif
endif
;
; Create the widget
modelfit_wid=widget_base(kill_notify='ww_modelfitdestroyed', $
	resource_name='oyster', $
	/row,title='ModelFit',xoffset=!dxsize-183,yoffset=0)
column1_wid=widget_base(modelfit_wid,/column)
column2_wid=widget_base(modelfit_wid,/column)
;
; 	General model
	component=systemcomp()
	gen_wid=widget_base(column1_wid,/column,/frame)
;	label_wid=widget_label(gen_wid,value=component)
	button_wid=widget_button(gen_wid,value=component,event_func='ww_modellist')
	list_wid=widget_list(gen_wid, $
		value=genparms(),uvalue={c:component,p:genparms()}, $
		event_pro='ww_modelfitparameters', $
		/multiple,ysize=n_elements(genparms()))
;
for i=0,num_star()-1 do begin
;	Star models
	component=star_model(i).component
	label=star_model(i).component+' ('+star_model(i).wmc+')'
	star_wid=widget_base(column1_wid,/column,/frame)
;	label_wid=widget_label(star_wid,value=component)
	button_wid=widget_button(star_wid,value=label,event_func='ww_modellist')
	list_wid=widget_list(star_wid, $
		value=starparms(),uvalue={c:component,p:starparms()}, $
		event_pro='ww_modelfitparameters', $
		/multiple,ysize=n_elements(starparms()))
endfor
;
for i=0,num_binary()-1 do begin
;	Binary models
	component=binary_model(i).component
	bin_wid=widget_base(column2_wid,/column,/frame)
;	label_wid=widget_label(bin_wid,value=component)
	button_wid=widget_button(bin_wid,value=component, $
		uvalue=binary_model(i).method, $
		event_func='ww_modellist')
	list_wid=widget_list(bin_wid, $
		value=binparms(),uvalue={c:component,p:binparms()}, $
		event_pro='ww_modelfitparameters', $
		/multiple,ysize=n_elements(binparms()))
	list1_wids(i)=list_wid
endfor
;
button_wid=cw_bgroup(column1_wid,['Fit','ChiSq','Error'], $
	event_funct='ww_modelfitok',/return_name,/row)
button_wid=cw_bgroup(column1_wid,['Help','Opt'], $
	event_funct='ww_modelfitok',/return_name,/row)
;
widget_control,modelfit_wid,/realize
xmanager,'ww_modelfit',modelfit_wid,/no_block
;
;
end
;-------------------------------------------------------------------------------
pro ww_modelfitdestroyed,event
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
common WwFitWids,list1_wids,list2_wids
;
modelfit_wid=0
list1_wids(*)=0
r=init_modelfit()
;
end
;-------------------------------------------------------------------------------
pro ww_modelfitparameters,event
;
; Call back procedure which sets the fit parameter array.
;
common WwFitWids,list1_wids,list2_wids
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
widget_control,event.id,get_uvalue=componentparms
component=componentparms.c
index=where(parameters.component eq component,count)
if count eq 0 then index=where(strlen(parameters.component) eq 0)
index=index(0)
names=componentparms.p
name_index=widget_info(event.id,/list_select)
if name_index(0) ge 0 then names=names(name_index) else names=''
parameters(index).names=''
parameters(index).names(0:n_elements(names)-1)=names
if names(0) eq '' then component=''
parameters(index).component=component
;
index0=where(binary_model.component eq component,count0)
; list1 is for orbital parameters, list2 for (rho,theta) definitions
index1=where(event.id eq list1_wids,count1)
index2=where(event.id eq list2_wids,count2)
if count0 eq 1 then begin
	if count0 ne 0 then begin
;		Switch automatically the method depending on the selection
		if count1 ne 0 then binary_model(index0).method=1
		if count2 ne 0 then binary_model(index0).method=2
	endif
endif
;
; Clear the selection of the respective other list
index=where(event.id eq list1_wids,count) & index=index(0)
if count gt 0 then if list2_wids(index) ne 0 then $
	widget_control,list2_wids(index),set_value=posparms()
index=where(event.id eq list2_wids,count) & index=index(0)
if count gt 0 then if list1_wids(index) ne 0 then $
	widget_control,list1_wids(index),set_value=binparms()
;
end
;-------------------------------------------------------------------------------
function ww_modellist,event
;
widget_control,event.id,get_value=components,get_uvalue=method
words=nameparse(components)
list_modcomp,words(0),method
;
end
;-------------------------------------------------------------------------------
function ww_modelfitok,event
;
common ModelFit,parameters,ds_options
common LocalChaSel,chafiles
common LocalFitSel,oiffiles
common LocalPsnSel,psnfiles
common LocalVelSel,velfiles
common LocalMagSel,magfiles
;
case event.value of
	'Fit': 	begin
		print,'Beginning NLLSQ fitting...'
		fitmodel
		end
	'ChiSq':begin
		marquardtdata,y,ysig,ymod
		if total(ysig) eq 0 then return,-1
		chisq=total(((y-ymod)/ysig)^2)/n_elements(y)
		print,'Reduced Chi-Squared = ',chisq
		print,'Number of data points:',n_elements(y)
		if !d.window ne -1 or !d.name eq 'PS' then $
		histograph,(y-ymod)/ysig,min=-8,max=8,binsize=0.4, $
			xtitle=greek('sigma')+' deviation',chisq=chisq
		end
	'Error':begin
		print,'Estimating statistical errors...'
		fitmodel,/error
		end
	'Opt':	begin
		file=file_search('Z_h_steps.xdr')
		answer='Yes'
		if strlen(file) ne 0 then answer= $
		dialog_message('Z_h_steps exists. Overwrite?',/question)
		if answer ne 'Yes' then begin
			print,'OK, nothing to do.'
			return,0
		endif else file='Z_h_steps.xdr'
;		Check that data are loaded
		marquardtdata,y,ysig,ymod
		if total(ysig) eq 0 then begin
			print,'No data found!'
			return,-1
		endif
		print,'Computing optimal values for h (increments)...'
		set_h_steps,file
;
		print,'Reading data again, as they have been overwritten...'
		if ds_options.i then begin
		  if n_elements(chafiles) ne 0 then $
		  if strlen(chafiles(0)) ne 0 then load_interferometry,chafiles
		  if n_elements(oiffiles) ne 0 then $
		  if strlen(oiffiles(0)) ne 0 then load_interferometry,oiffiles
		endif
		if ds_options.a then load_astrometry,psnfiles
		if ds_options.s then load_spectroscopy,velfiles
		calcmodel
		print,'Click component labels to examine step sizes.'
		end
	'Help':	begin
		print,'________________________________________________'
		print,'GUI to fit a hierarchical stellar model. To fit'
		print,'Rho/Theta, select them in Fit/Interferometry!'
		print,'Select parameters to fit, then click on "Fit".'
		print,'To fit rho/theta to vis. data, select these in'
		print,'the FitInterferometry GUI but click "Fit" here!'
		print,'Compute statistical errors with button "Error".'
		print,'Opt computes optimal step sizes for Marquardt-'
		print,'Levenberg for the currently loaded data.'
		print,'To use the opt. steps, select Opt in FitControl.'
		print,'(Interferometric data have to be reloaded after'
		print,'running Opt.)'
		print,'______________________***_______________________'
		end
endcase
;
end
;************************************************************************Block 2
pro ww_fitinterferometry
;
; GUI for fits of binary positions to visibility data.
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
common WwFitWids,list1_wids,list2_wids
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common LocalFitNights,cbcf
;
; Return if widget is already displayed
if n_elements(ifit_wid) eq 0 then ifit_wid=0
if ifit_wid ne 0 then return
;
; Create array to hold widget IDs of parameter lists
if num_binary() eq 0 then begin
	print,'***Error(WW_FITINTERFEROMETRY): no model!'
	return
endif
;
; No longer needed?
if 0 then begin
num_bin_comps=0
for i=0,num_binary()-1 do $
	if binary_model(i).method eq 2 then num_bin_comps=num_bin_comps+1
if num_bin_comps eq 0 then begin
	print,'***Error: no binary component defined by method 2 (rho,theta)!'
	return
endif
if num_bin_comps lt num_binary() then begin
	print,'Warning: not all binary components defined by method 2!'
endif
;
; Check data
if n_elements(geoinfo) eq 0 then begin
	print,'***Error(WW_FITINTERFEROMETRY): no data!'
	return
endif
endif

if n_elements(list1_wids) eq 0 then list1_wids=lonarr(num_binary())
if n_elements(list2_wids) eq 0 then list2_wids=lonarr(num_binary())
;
; Initialize options
if init_fitinterferometry() ne 0 then return
;
; Check cbcf (CLEAN beam conversion factor)
if n_elements(cbcf) eq 0 then cbcf=7.0
;
; Create the widget
ifit_wid=widget_base(kill_notify='ww_fitinterferometrydestroyed', $
	resource_name='oyster', $
	/row,title='FitInterferometry',xoffset=!dxsize-500,yoffset=300)
column1_wid=widget_base(ifit_wid,/column)
;
for i=0,num_binary()-1 do begin
	component=binary_model(i).component
	bin_wid=widget_base(column1_wid,/column)
;	label_wid=widget_label(bin_wid,value=component)
	button_wid=widget_button(bin_wid,value=component,uvalue=2, $
		event_func='ww_modellist')
	list_wid=widget_list(bin_wid, $
		value=posparms(),uvalue={c:component,p:posparms()}, $
		event_pro='ww_modelfitparameters',/multiple, $
		ysize=n_elements(posparms()))
;		if binary_model(i).method ne 2 then $
;			widget_control,list_wid,sensitive=0
	list2_wids(i)=list_wid
endfor
;
column2_wid=widget_base(ifit_wid,/column)
;
text_wid=widget_text(column2_wid,value='CLEAN beam CF',xsize=14)
text_wid=widget_text(column2_wid,value=string(cbcf,format='(f4.1)'),/editable, $
	event_pro='ww_fitinterferometrycbcf',xsize=14)
;
column3_wid=widget_base(ifit_wid,/column)
;
button_wid=cw_bgroup(column3_wid,['Fit','Help'],/return_name, $
	event_funct='ww_fitinterferometryok',/column)
;
; Display widget
widget_control,ifit_wid,/realize
xmanager,'ww_fitinterferometry',ifit_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_fitinterferometrydestroyed,event
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
common WwFitWids,list1_wids,list2_wids
;
ifit_wid=0
list2_wids(*)=0
r=init_fitinterferometry()
;
end
;-------------------------------------------------------------------------------
pro ww_fitinterferometrycbcf,event
;
common LocalFitNights,cbcf
;
widget_control,event.id,get_value=command
s=stringparse(command(0))
cbcf=float(s(0))
print,'CLEAN CB conversion factor set to: '+string(cbcf,format='(f4.1)')
;
end
;-------------------------------------------------------------------------------
function ww_fitinterferometryok,event
;
common ModelFit,parameters,ds_options
;
case event.value of
	'Fit':	fitnights
	'Help':	begin
		print,'____________________________________________________'
		print,'GUI to fit rho/theta astrometry to visibilities, '
		print,'beginning with the values predicted by the orbit!'
		print,'Accounts for orbital motion during the night.'
		print,'The selected nights will be (re-)read from disk,'
		print,'and data for same night will be combined.'
		print,'The results will be written to fitnights.psn.'
		print,'CBCF is the factor by which the CLEAM beam will be'
		print,'reduced for the computation of the error ellipses.'
		print,'(Hit <Return> after entering a value)'
		print,'_______________________***__________________________'
		end
endcase
;
end
;************************************************************************Block 3
pro ww_fitastrometry
;
; Fits (rho,theta) positions derived from interferometric or speckle data.
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
;
; Return if widget is already displayed
if n_elements(afit_wid) eq 0 then afit_wid=0
if afit_wid ne 0 then return
;
; Initialize options
if init_fitastrometry() ne 0 then return
;
; Check if model is loaded
list_poscomps,components
if components(0) eq '' then return
;
; Create the widget
afit_wid=widget_base(/column,kill_notify='ww_fitastrometrydestroyed', $
	resource_name='oyster', $
	title='FitAstrometry',xoffset=!dxsize-500,yoffset=300)
;
; Component selection
opmenu_wid=widget_droplist(afit_wid,title='Component: ', $
	event_pro='ww_fitastrometrycomp',value=components,uvalue=components)
orbit_options.component=components(0)
ellipse_options.component=components(0)
;
row_wid=widget_base(afit_wid,/row)
;
; Ellipse menu
column1_wid=widget_base(row_wid,/column)
button_wid=widget_button(column1_wid,value='Ellipse', $
	event_pro='ww_fitellipse')
option=['All', $
	'Center', $
	'Major axis', $
	'Minor axis', $
	'Position angle']
option_wid=cw_bgroup(column1_wid,option,/nonexclusive, $
	event_funct='ww_ellipsefitoptions')
;
; Orbit menu
column2_wid=widget_base(row_wid,/column)
button_wid=widget_button(column2_wid,value='Orbit', $
	event_pro='ww_fitorbit')
option=['All', $
	'Semimajoraxis', $
	'Eccentricity', $
	'Inclination', $
	'Periastron', $
	'Ascendingnode', $
	'Period', $
	'Epoch']
option_wid=cw_bgroup(column2_wid,option,/nonexclusive, $
	event_funct='ww_orbitfitoptions')
;
button_wid=cw_bgroup(afit_wid,['Get orbit','Set model','Help'], $
	event_funct='ww_fitastrometryutil',/row,/return_name)
;
; Display widget
widget_control,afit_wid,/realize
xmanager,'ww_astrometry',afit_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro ww_fitastrometrydestroyed,event
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
;
afit_wid=0
;
end
;-------------------------------------------------------------------------------
pro ww_fitastrometrycomp,event
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
;
widget_control,event.id,get_uvalue=components
ellipse_options.component=components(event.index)
orbit_options.component=components(event.index)
;
end

;-------------------------------------------------------------------------------
function ww_ellipsefitoptions,event
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
;
case event.value of
	0: ellipse_options.all=event.select
	1: ellipse_options.c=event.select
	2: ellipse_options.a=event.select
	3: ellipse_options.b=event.select
	4: ellipse_options.p=event.select
endcase
end
;-------------------------------------------------------------------------------
function ww_orbitfitoptions,event
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
;
case event.value of
	0: orbit_options.all=event.select
	1: orbit_options.a=event.select
	2: orbit_options.e=event.select
	3: orbit_options.i=event.select
	4: orbit_options.w=event.select
	5: orbit_options.n=event.select
	6: orbit_options.p=event.select
	7: orbit_options.t=event.select
endcase
end
;-------------------------------------------------------------------------------
pro ww_fitellipse,event
;
fitellipse
;
end
;-------------------------------------------------------------------------------
pro ww_fitorbit,event
;
fitorbit
;
end
;-------------------------------------------------------------------------------
function ww_fitastrometryutil,event
;
case event.value of
	'Get orbit': setorbit
	'Set model': setmodel
	'Help':	begin
		print,'__________________________________________________'
		print,'Fit orbital elements to astrometry (rho,theta).'
		print,'Ellipse: fit ellipse; Orbit: fit orbit.'
		print,'Get orbit: get orbital elements from binary model.'
		print,'Set model: copy orbital elements to binary model.'
		print,'_______________________***________________________'
		end
endcase
;
end
;************************************************************************Block 4
pro ww_marquardtfit
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
common MarquardtFit,fit_options
;
; Return if widget is already displayed
if n_elements(control_wid) eq 0 then control_wid=0
if control_wid ne 0 then return
;
; Initialize options
if init_marquardtfit() ne 0 then return
;
; Create the widget
control_wid=widget_base(kill_notify='ww_marquardtfitdestroyed', $
	resource_name='oyster', $
	xoffset=!dxsize-500,yoffset=0,/column,title='FitControl')
;
; Fit options
label_wid=widget_label(control_wid,value='Fit options')
options=['One','Float','Opt.']
button_wid=cw_bgroup(control_wid,options,/nonexclusive,column=3, $
	event_funct='ww_marquardtfitoptions')
;
; Fit control parms (see also p. 684 of "Numerical Recipes in C", 2nd ed.)
alamda=['1.0','0.1','0.01','0.001']
opmenual_wid=widget_droplist(control_wid,title='Lambda:       ', $
	event_pro='ww_setalamda',value=alamda,uvalue=alamda)
widget_control,opmenual_wid,set_droplist_select=2
;
tolerance=['1e-4','1e-5','1e-6','1e-7','1e-8','1e-9','1e-10']
opmenuth_wid=widget_droplist(control_wid,title='Tolerance:    ', $
	event_pro='ww_settolerance',value=tolerance,uvalue=tolerance)
widget_control,opmenuth_wid,set_droplist_select=2
;
chifr=['0.1','0.01','0.001','0.0001']
opmenuch_wid=widget_droplist(control_wid,title='Convergence: ', $
	event_pro='ww_setchifr',value=chifr,uvalue=chifr)
widget_control,opmenuch_wid,set_droplist_select=2 ; must correspond to alloc
;
helpt=['Lambda','Tolerance','Convergence']
opmenuhp_wid=widget_droplist(control_wid,title='Help:   ', $
	event_pro='ww_getlmhelp',value=helpt,uvalue=helpt)
widget_control,opmenuhp_wid,set_droplist_select=1
;
widget_control,control_wid,/realize
xmanager,'ww_marquardtfit',control_wid,/no_block
;
end
;-------------------------------------------------------------------------------
function ww_marquardtfitoptions,event
;
common MarquardtFit,fit_options
;
case event.value of
	0: begin
	   fit_options.o=event.select
	   if fit_options.o then $
		print,'Only one ML iteration will be run.'
	   end
	1: begin
	   fit_options.f=event.select
	   if fit_options.f then begin
		print,'The calibration will be allowed to float.'
	   endif else begin
		print,'Float option turned off, must reload data!'
	   endelse
	   end
	2: begin
	   fit_options.h=event.select
	   if fit_options.h then $
		print,'Optimized steps will be read from Z_h_steps.xdr.'
	   end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_setalamda,event
;
common MarquardtFit,fit_options
;
widget_control,event.id,get_uvalue=alamda
fit_options.alamda=alamda(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_settolerance,event
;
common MarquardtFit,fit_options
;
widget_control,event.id,get_uvalue=tolerance
fit_options.tolerance=tolerance(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_setchifr,event
;
common MarquardtFit,fit_options
;
widget_control,event.id,get_uvalue=chifr
fit_options.chifr=chifr(event.index)
;
end
;-------------------------------------------------------------------------------
pro ww_getlmhelp,event
;
common MarquardtFit,fit_options
;
widget_control,event.id,get_uvalue=topic
;
case topic(event.index) of
	'Lambda':begin
		print,'__________________________________________________'
		print,'Hessian matrix control, large if far from minimum'
		print,'______________________***_________________________'
		end
	'Tolerance':begin
		print,'__________________________________________________'
		print,'Singular value decomposition: smallest e.v. ratio.'
		print,'______________________***_________________________'
		end
	'Convergence':begin
		print,'_________________________________________________'
		print,'Stop when ChiSq ratio drops below this value'
		print,'______________________***________________________'
		end
endcase
;
end
;-------------------------------------------------------------------------------
pro ww_marquardtfitdestroyed,event
;
common AmoebaWids,data_wid,modelfit_wid,ifit_wid,afit_wid,control_wid
;
control_wid=0
;
end
;-------------------------------------------------------------------------------
